 # Import pyCAPS module
import pyCAPS

# Import os module
import os
import shutil
import argparse

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'Nastran Aeroelastic Pytest Example',
                                 prog = 'nastran_Aeroelastic_PyTest',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = ["."+os.sep], nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument('-noAnalysis', action='store_true', default = False, help = "Don't run analysis code")
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

# Create project name
projectName = "NastranAeroWingTailBEM"

workDir = os.path.join(str(args.workDir[0]), projectName)

# Load CSM file
geometryScript = os.path.join("..","csmData","feaWingTailBEMAero.csm")
capsProblem = pyCAPS.Problem(problemName=workDir,
                           capsFile=geometryScript,
                           outLevel=args.outLevel)

wing_nRib  = int(capsProblem.geometry.cfgpmtr.wing.nrib)
wing_nSpar = int(capsProblem.geometry.cfgpmtr.wing.nspar)
vtail_nRib  = int(capsProblem.geometry.cfgpmtr.vtail.nrib)
vtail_nSpar = int(capsProblem.geometry.cfgpmtr.vtail.nspar)


# Load egadsTess aim
egads = capsProblem.analysis.create(aim = "egadsTessAIM")

# Set meshing parameters
egads.input.Edge_Point_Max = 5
egads.input.Edge_Point_Min = 5

# All quads in the grid
egads.input.Mesh_Elements = "Quad"

egads.input.Tess_Params = [.05,.5,15]

# Load nastran aim
capsProblem.analysis.create(aim = "nastranAIM",
                            name = "nastran")

capsProblem.analysis["nastran"].input["Mesh"].link(egads.output["Surface_Mesh"])

capsProblem.analysis["nastran"].input.Proj_Name = "nastranAero"

# Set analysis type
capsProblem.analysis["nastran"].input.Analysis_Type = "Aeroelastic"

# Set analysis
trim1 = { "analysisType" : "AeroelasticStatic",
          "trimSymmetry" : "ASYM",
          "analysisConstraint" : ["ribConstraint"],
          "analysisSupport" : ["ribSupport"],
          "controlConstraint" : ["Slat", "Flap", 
                                 "AlrnLeft", "AlrnRite",
                                 "ElvnLeft", "ElvnRite"],
          "magControlConstraint" : [0.0, 0.0,
                                    0.0, 0.0,
                                    0.0, 0.0],
          "machNumber"     : 0.5,
          "dynamicPressure": 50000,
          "density" : 1.0,
          "rigidVariable"  : ["ANGLEA"],
          "rigidConstraint": ["URDD3"],
          "magRigidConstraint" : [-50],
          }


capsProblem.analysis["nastran"].input.Analysis = {"Trim1": trim1}

# Set materials
unobtainium  = {"youngModulus" : 2.2E6 ,
                "poissonRatio" : .5,
                "density"      : 7850}

capsProblem.analysis["nastran"].input.Material = {"Unobtainium": unobtainium}

# Set property
shell  = {"propertyType"      : "Shell",
          "membraneThickness" : 0.2,
          "bendingInertiaRatio" : 1.0, # Default
          "shearMembraneRatio"  : 5.0/6.0} # Default }

shell2  = {"propertyType"      : "Shell",
           "material"          : "Unobtainium",
           "membraneThickness" : 0.002,
           "bendingInertiaRatio" : 1.0, # Default
           "shearMembraneRatio"  : 5.0/6.0} # Default }

Property = {}
Property["wingRib_Root"] = shell
Property["wingRib_Tip"] = shell
for irib in range(wing_nRib):
    Property[f"wingRib{irib+1}"] = shell

for irib in range(wing_nRib+1):
    for ispar in range(wing_nSpar):
        Property[f"wingSpar{ispar+1}_{irib}"] = shell
        
for irib in range(wing_nRib+1):
    for ispar in range(wing_nSpar+1):
        Property[f"wingSkin{ispar}_{irib}"] = shell2

Property["vtailRib_Root"] = shell
Property["vtailRib_Tip"] = shell
for irib in range(vtail_nRib):
    Property[f"vtailRib{irib+1}"] = shell

for irib in range(vtail_nRib+1):
    for ispar in range(vtail_nSpar):
        Property[f"vtailSpar{ispar+1}_{irib}"] = shell
        
for irib in range(vtail_nRib+1):
    for ispar in range(vtail_nSpar+1):
        Property[f"vtailSkin{ispar}_{irib}"] = shell2


capsProblem.analysis["nastran"].input.Property = Property


# Defined Connections
connection = {"dofDependent" : 123456,
              "connectionType" : "RigidBody"}

capsProblem.analysis["nastran"].input.Connect = {"Rib_Root": connection}


# Set constraints
constraint = {"groupName" : ["Rib_Root_Point"],
              "dofConstraint" : 12456}

capsProblem.analysis["nastran"].input.Constraint = {"ribConstraint": constraint}

# Set supports
support = {"groupName" : ["Rib_Root_Point"],
           "dofSupport": 3}

capsProblem.analysis["nastran"].input.Support = {"ribSupport": support}


# Aero
wingLeft = {"numChord"          : 10,
            "numSpanPerSection" : 10,
            "interpSpline" : [{"splineName" : ["leftWingNodes_Top","rootWingNodes_Top"], "usage":"DISP"},
                              {"splineName" : ["leftWingSkin_Top","leftWingEdges_Top","leftWingNodes_Top","rootWingEdges_Top","rootWingNodes_Top"], "usage":"FORCE", "nElem":20, "mElem":20, "spanLimited":True}]}

wingRite = {"numChord"          : 10,
            "numSpanPerSection" : 10,
            "interpSpline" : [{"splineName" : ["riteWingNodes_Top","rootWingNodes_Top"], "usage":"DISP"},
                              {"splineName" : ["riteWingSkin_Top","riteWingEdges_Top","riteWingNodes_Top","rootWingEdges_Top","rootWingNodes_Top"], "usage":"FORCE", "nElem":20, "mElem":20, "spanLimited":True}]}

vtailLeft = {"numChord"          : 10,
             "numSpanPerSection" : 10,
             "interpSpline" : [{"splineName" : ["leftVtailNodes_Top"], "usage":"DISP"},
                               {"splineName" : ["leftVtailSkin_Top","leftVtailEdges_Top","leftVtailNodes_Top"], "usage":"FORCE", "nElem":20, "mElem":20, "spanLimited":True}]}

vtailRite = {"numChord"          : 10,
             "numSpanPerSection" : 10,
             "interpSpline" : [{"splineName" : ["riteVtailNodes_Top"], "usage":"DISP"},
                               {"splineName" : ["riteVtailSkin_Top","riteVtailEdges_Top","riteVtailNodes_Top"], "usage":"FORCE", "nElem":20, "mElem":20, "spanLimited":True}]}

# Define the lifting surfaces
capsProblem.analysis["nastran"].input.VLM_Surface = {"leftWing": wingLeft, "riteWing": wingRite, 
                                                     "VtailLeft":vtailLeft, "VtailRite":vtailRite}

# Define the control surfaces
capsProblem.analysis["nastran"].input.VLM_Control= {"Slat":{}, "Flap":{}, "AlrnLeft":{}, "AlrnRite":{}, "ElvnLeft":{"leOrTe":1}, "ElvnRite":{"leOrTe":1}}

# Run AIM pre-analysis
capsProblem.analysis["nastran"].preAnalysis()

####### Run Nastran####################
print ("\n\nRunning Nastran......")

if args.noAnalysis == False:
    capsProblem.analysis["nastran"].system("nastran old=no notify=no batch=no scr=yes sdirectory=./ " + capsProblem.analysis["nastran"].input.Proj_Name +  ".dat"); # Run Nastran via system call
else:
    capsProblem.analysis["nastran"].system("test_bdf -e 0 " + capsProblem.analysis["nastran"].input.Proj_Name +  ".dat"); # Run pyNastran test_bdf via system call
    # Copy old results if no analysis available
    shutil.copy2(os.path.join("..","analysisData","nastran",projectName+".f06"), 
                 os.path.join(capsProblem.analysis["nastran"].analysisDir,capsProblem.analysis["nastran"].input.Proj_Name+".f06"))
    shutil.copy2(os.path.join("..","analysisData","nastran",projectName+".op2"), 
                 os.path.join(capsProblem.analysis["nastran"].analysisDir,capsProblem.analysis["nastran"].input.Proj_Name+".op2"))

print ("Done running Nastran!")

# Run AIM post-analysis
capsProblem.analysis["nastran"].postAnalysis()
