/*
 *************************************************************************
 *                                                                       *
 * udpMisDatHex -- udp for Missile Datcom HEX Airfoil                    *
 *                                                                       *
 *            Written by Marshall Galbraith @ MIT                        *
 *            Patterned after .c code written by John Dannenhoffer       *
 *                                                @ Syracuse University  *
 *                                                                       *
 *************************************************************************
 */

/*
 * Copyright (C) 2011/2025 Marshall Galbraith @ MIT
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

extern "C" {
#define NUMUDPARGS 6
#include "udpUtilities.h"
}

#define ZUPPER_ARG(UDP)                    (UDP)->arg[0]
#define ZUPPER_VAL(UDP) ((const double *) ((UDP)->arg[0].val))[0]
#define ZUPPER_DOT(UDP) ((const double *) ((UDP)->arg[0].dot))[0]
#define ZLOWER_ARG(UDP)                    (UDP)->arg[1]
#define ZLOWER_VAL(UDP) ((const double *) ((UDP)->arg[1].val))[0]
#define ZLOWER_DOT(UDP) ((const double *) ((UDP)->arg[1].dot))[0]
#define LMAXU_ARG( UDP)                    (UDP)->arg[2]
#define LMAXU_VAL( UDP) ((const double *) ((UDP)->arg[2].val))[0]
#define LMAXU_DOT( UDP) ((const double *) ((UDP)->arg[2].dot))[0]
#define LMAXL_ARG( UDP)                    (UDP)->arg[3]
#define LMAXL_VAL( UDP) ((const double *) ((UDP)->arg[3].val))[0]
#define LMAXL_DOT( UDP) ((const double *) ((UDP)->arg[3].dot))[0]
#define LFLATU_ARG(UDP)                    (UDP)->arg[4]
#define LFLATU_VAL(UDP) ((const double *) ((UDP)->arg[4].val))[0]
#define LFLATU_DOT(UDP) ((const double *) ((UDP)->arg[4].val))[0]
#define LFLATL_ARG(UDP)                    (UDP)->arg[5]
#define LFLATL_VAL(UDP) ((const double *) ((UDP)->arg[5].val))[0]
#define LFLATL_DOT(UDP) ((const double *) ((UDP)->arg[5].dot))[0]

#define ZUPPER_SURREAL(UDP) SurrealS<1>( ZUPPER_VAL(UDP), ZUPPER_DOT(UDP))
#define ZLOWER_SURREAL(UDP) SurrealS<1>( ZLOWER_VAL(UDP), ZLOWER_DOT(UDP))
#define LMAXU_SURREAL( UDP) SurrealS<1>( LMAXU_VAL( UDP), LMAXU_DOT( UDP))
#define LMAXL_SURREAL( UDP) SurrealS<1>( LMAXL_VAL( UDP), LMAXL_DOT( UDP))
#define LFLATU_SURREAL(UDP) SurrealS<1>( LFLATU_VAL(UDP), LFLATU_DOT(UDP))
#define LFLATL_SURREAL(UDP) SurrealS<1>( LFLATL_VAL(UDP), LFLATL_DOT(UDP))

/* data about possible arguments */
static const char  *argNames[NUMUDPARGS] = {"zupper",     "zlower",     "lmaxu",     "lmaxl",   "lflatu",    "lflatl",   };
static int          argTypes[NUMUDPARGS] = {ATTRREALSEN, ATTRREALSEN, ATTRREALSEN, ATTRREALSEN, ATTRREALSEN, ATTRREALSEN,};
static int          argIdefs[NUMUDPARGS] = {0,            0,          0,            0,          0,            0,         };
static double       argDdefs[NUMUDPARGS] = {0.025,       -1.,         0.5,         -1.,         0.,          -1.,        };


#include "egads_dot.h"

template<class T> T    ZUPPER(udp_T *udp);
template<> double      ZUPPER< double      >(udp_T *udp) { return ZUPPER_VAL(udp); }
template<> SurrealS<1> ZUPPER< SurrealS<1> >(udp_T *udp) { return ZUPPER_SURREAL(udp); }

template<class T> T    ZLOWER(udp_T *udp);
template<> double      ZLOWER< double      >(udp_T *udp) { return ZLOWER_VAL(udp); }
template<> SurrealS<1> ZLOWER< SurrealS<1> >(udp_T *udp) { return ZLOWER_SURREAL(udp); }

template<class T> T    LMAXU(udp_T *udp);
template<> double      LMAXU< double      >(udp_T *udp) { return LMAXU_VAL(udp); }
template<> SurrealS<1> LMAXU< SurrealS<1> >(udp_T *udp) { return LMAXU_SURREAL(udp); }

template<class T> T    LMAXL(udp_T *udp);
template<> double      LMAXL< double      >(udp_T *udp) { return LMAXL_VAL(udp); }
template<> SurrealS<1> LMAXL< SurrealS<1> >(udp_T *udp) { return LMAXL_SURREAL(udp); }

template<class T> T    LFLATU(udp_T *udp);
template<> double      LFLATU< double      >(udp_T *udp) { return LFLATU_VAL(udp); }
template<> SurrealS<1> LFLATU< SurrealS<1> >(udp_T *udp) { return LFLATU_SURREAL(udp); }

template<class T> T    LFLATL(udp_T *udp);
template<> double      LFLATL< double      >(udp_T *udp) { return LFLATL_VAL(udp); }
template<> SurrealS<1> LFLATL< SurrealS<1> >(udp_T *udp) { return LFLATL_SURREAL(udp); }

template class SurrealS<1>;
typedef SurrealS<1> SurrealS1;

#include <vector>

#define LENMSG 512

extern "C" {
/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"
}

template<class T>
void Coordinates(udp_T *udp, int &i, T X[7][3])
{
  T zlower = ZLOWER<T>(udp) > 0 ? ZLOWER<T>(udp) : ZUPPER<T>(udp);
  T lmaxl  = LMAXL<T>(udp)  > 0 ? LMAXL<T>(udp)  : LMAXU<T>(udp);
  T lflatl = LFLATL<T>(udp) > 0 ? LFLATL<T>(udp) : LFLATU<T>(udp);

  i = 0;

  /* TE */
  X[i][0] = 1;
  X[i][1] = 0;
  X[i][2] = 0;
  i++;

  /* Upper flat TE */
  if (LFLATU<T>(udp) > 0) {
    X[i][0] = LMAXU<T>(udp) + LFLATU<T>(udp);
    X[i][1] = ZUPPER<T>(udp);
    X[i][2] = 0;
    i++;
  }

  /* Upper flat LE */
  X[i][0] = LMAXU<T>(udp);
  X[i][1] = ZUPPER<T>(udp);
  X[i][2] = 0;
  i++;

  /* LE */
  X[i][0] = 0;
  X[i][1] = 0;
  X[i][2] = 0;
  i++;

  /* Lower flat LE */
  X[i][0] = lmaxl;
  X[i][1] = -zlower;
  X[i][2] = 0;
  i++;

  /* Lower flat TE */
  if (lflatl > 0) {
    X[i][0] = lmaxl + lflatl;
    X[i][1] = -zlower;
    X[i][2] = 0;
    i++;
  }

  /* TE */
  X[i][0] = 1;
  X[i][1] = 0;
  X[i][2] = 0;
}



/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */
extern "C"
int
udpExecute(ego context,                 /* (in)  context */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
  int    status = EGADS_SUCCESS;
  double data[9], tdata[2];
  int    nnode, sense[6];
  ego    eedges[6], elines[6], enodes[7], eloop, eplane, eface;
  double X[7][3];
  int    i, d;
  char   *message=NULL;
  udp_T  *udps = *Udps;
  udp_T  *udp = &udps[0];


  ROUTINE(udpExecute);

  /* --------------------------------------------------------------- */

#ifdef DEBUG
  printf("udpSplineDesPmtr.udpExecute\n");
#endif

  /* default return values */
  *ebody  = NULL;
  *nMesh  = 0;
  *string = NULL;

  MALLOC(message, char, LENMSG);
  message[0] = '\0';

  if (ZUPPER_ARG(udp).size != 1) {
    snprintf(message, LENMSG, "Expecting 1 'zupper' value. There are %d\n", ZUPPER_ARG(udp).size);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (ZUPPER_VAL(udp) < 0) {
    snprintf(message, LENMSG, "Expecting 'zupper' (%lf) positive.\n", ZUPPER_VAL(udp));
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (ZLOWER_ARG(udp).size != 1) {
    snprintf(message, LENMSG, "Expecting 1 'zlower' value. There are %d\n", ZLOWER_ARG(udp).size);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (LMAXU_ARG(udp).size != 1) {
    snprintf(message, LENMSG, "Expecting 1 'lmaxu' value. There are %d\n", LMAXU_ARG(udp).size);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (LMAXL_ARG(udp).size != 1) {
    snprintf(message, LENMSG, "Expecting 1 'lmaxl' value. There are %d\n", LMAXL_ARG(udp).size);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (LFLATU_ARG(udp).size != 1) {
    snprintf(message, LENMSG, "Expecting 1 'lflatu' value. There are %d\n", LFLATU_ARG(udp).size);
    status = EGADS_RANGERR;
    goto cleanup;
  }
  if (LFLATL_ARG(udp).size != 1) {
    snprintf(message, LENMSG, "Expecting 1 'lflatl' value. There are %d\n", LFLATL_ARG(udp).size);
    status = EGADS_RANGERR;
    goto cleanup;
  }


  /* Cache copy of arguments for future use.
   * This also increments numUdp and copies udps[numUdp] to udps[numUdp].
   * Caching should only be performed after checking for valid inputs.
   */
  status = cacheUdp(NULL);
  CHECK_STATUS(cacheUdp);
  udp = &udps[numUdp];

  Coordinates(udp, nnode, X);

  /* Nodes */
  for (i = 0; i < nnode; i++) {
    status = EG_makeTopology(context, NULL, NODE, 0,
                             X[i], 0, NULL, NULL, &(enodes[i]));
    CHECK_STATUS(EG_makeTopology);
  }
  enodes[nnode] = enodes[0];

  /* Lines */
  for (i = 0; i < nnode; i++) {
    for (d = 0; d < 3; d++) {
      data[d] = X[i][d];
      data[d+3] = X[i+1][d] - X[i][d];
    }

    status = EG_makeGeometry(context, CURVE, LINE, NULL, NULL, data, &elines[i]);
    CHECK_STATUS(EG_makeGeometry);
  }

  /* Edges */
  for (i = 0; i < nnode; i++) {

    tdata[0] = 0;
    tdata[1] = 0;

    for (d = 0; d < 3; d++) {
      tdata[1] += pow(X[i+1][d] - X[i][d],2);
    }
    tdata[1] = sqrt(tdata[1]);

    status = EG_makeTopology(context, elines[i], EDGE, TWONODE,
                             tdata, 2, &enodes[i], NULL, &eedges[i]);
    CHECK_STATUS(EG_makeTopology);
  }

  /* create Loop of the Edge */
  for (i = 0; i < nnode; i++) sense[i] = SFORWARD;
  status = EG_makeTopology(context, NULL, LOOP, CLOSED,
                           NULL, nnode, eedges, sense, &eloop);
  CHECK_STATUS(EG_makeTopology);

  /* create a plane for the loop */
  data[0] = 0.;
  data[1] = 0.;
  data[2] = 0.;
  data[3] = 1.; data[4] = 0.; data[5] = 0.;
  data[6] = 0.; data[7] = 1.; data[8] = 0.;

  status = EG_makeGeometry(context, SURFACE, PLANE, NULL, NULL, data, &eplane);
  CHECK_STATUS(EG_makeGeometry);

  /* create the face from the plane and the loop */
  status = EG_makeTopology(context, eplane, FACE, SFORWARD,
                           NULL, 1, &eloop, sense, &eface);
  CHECK_STATUS(EG_makeTopology);

  /* create the face body */
  status = EG_makeTopology(context, NULL, BODY, FACEBODY, NULL, 1, &eface, NULL, ebody);
  CHECK_STATUS(EG_makeTopology);

  /* cleanup in reverse order */
  status = EG_deleteObject(eface);
  CHECK_STATUS(EG_deleteObject);
  status = EG_deleteObject(eplane);
  CHECK_STATUS(EG_deleteObject);
  status = EG_deleteObject(eloop);
  CHECK_STATUS(EG_deleteObject);
  for (i = 0; i < nnode; i++) {
    status = EG_deleteObject(eedges[i]);
    CHECK_STATUS(EG_deleteObject);
    status = EG_deleteObject(elines[i]);
    CHECK_STATUS(EG_deleteObject);
  }
  for (i = 0; i < nnode; i++) {
    status = EG_deleteObject(enodes[i]);
    CHECK_STATUS(EG_deleteObject);
  }

  /* remember this model (body) */
  udp->ebody = *ebody;

#ifdef DEBUG
  printf("udpExecute -> *ebody=%lx\n", (long)(*ebody));
#endif

cleanup:
  if (strlen(message) > 0) {
    *string = message;
    printf("%s\n", message);
  } else if (status != EGADS_SUCCESS) {
    printf("ERROR: status = %d \n", status);
    FREE(message);
    *string = udpErrorStr(status);
  } else {
    FREE(message);
  }

  return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */
extern "C"
int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
               int    npnt,             /* (in)  number of points */
               int    entType,          /* (in)  OCSM entity type */
               int    entIndex,         /* (in)  OCSM entity index (bias-1) */
               double uvs[],            /* (in)  parametric coordinates for evaluation */
               double vels[])           /* (out) velocities */
{
  int    status = EGADS_SUCCESS;
  int    iudp = 0, judp, stride, ipnt, *senses;
  int    nchild, nnode, nedge;
  int    oclass, mtype;
  double trange[4];
  double point[18], point_dot[18];
  ego    eent, eref, eplane, eloop, *eedges, *echildren;
  ego    elines[6]={NULL,NULL,NULL,NULL,NULL,NULL}, enodes[7]={NULL,NULL,NULL,NULL,NULL,NULL,NULL};
  SurrealS<1> X[7][3], tdata[2], data[18];
  udp_T   *udp = NULL;

  int    i, d;
  char   *message=NULL;

  ROUTINE(udpSensitivity);

  /* --------------------------------------------------------------- */

  /* check that ebody matches one of the ebodys */
  iudp = 0;
  for (judp = 1; judp <= numUdp; judp++) {
    if (ebody == udps[judp].ebody) {
      iudp = judp;
      break;
    }
  }
  if (iudp <= 0) {
    return EGADS_NOTMODEL;
  }

  udp = &udps[iudp];

#ifdef DEBUG
  printf("udpSplineDesPmtr.udpSensitivity iudp = %d\n", iudp);

  printf("ZUPPER_VAL = %f\n", ZUPPER_VAL(udp) );
  printf("ZLOWER_VAL = %f\n", ZLOWER_VAL(udp) );
  printf("LMAXU_VAL  = %f\n", LMAXU_VAL(udp)  );
  printf("LMAXL_VAL  = %f\n", LMAXL_VAL(udp)  );
  printf("LFLATU_VAL = %f\n", LFLATU_VAL(udp)  );
  printf("LFLATL_VAL = %f\n", LFLATL_VAL(udp)  );
  printf("\n");
#endif

  MALLOC(message, char, LENMSG);
  message[0] = '\0';

  /* build the sensitivity if needed */
  if (udp->ndotchg > 0 || EG_hasGeometry_dot(ebody) != EGADS_SUCCESS) {

    /* get the loop from the body */
    status = EG_getTopology(ebody, &eplane, &oclass, &mtype, trange, &nchild, &echildren,
                            &senses);
    CHECK_STATUS(EG_getTopology);
    eloop = echildren[0];

    /* get the edges from the loop */
    status = EG_getTopology(eloop, &eref, &oclass, &mtype, trange, &nedge, &eedges,
                            &senses);
    CHECK_STATUS(EG_getTopology);

    /* get the nodes and the lines from the edges */
    for (i = 0; i < nedge; i++) {
      status = EG_getTopology(eedges[i], &elines[i], &oclass, &mtype, trange, &nchild, &echildren,
                              &senses);
      CHECK_STATUS(EG_getTopology);

      enodes[i  ] = echildren[0];
      enodes[i+1] = echildren[1];
    }

    Coordinates(udp, nnode, X);

    /* Nodes */
    for (i = 0; i < nnode; i++) {
      status = EG_setGeometry_dot(enodes[i], NODE, 0, NULL, X[i]);
      CHECK_STATUS(EG_setGeometry_dot);
    }

    /* Lines */
    for (i = 0; i < nnode; i++) {
      for (d = 0; d < 3; d++) {
        data[d] = X[i][d];
        data[d+3] = X[i+1][d] - X[i][d];
      }

      status = EG_setGeometry_dot(elines[i], CURVE, LINE, NULL, data);
      CHECK_STATUS(EG_makeGeometry);
    }

    /* Edges */
    for (i = 0; i < nnode; i++) {

      tdata[0] = 0;
      tdata[1] = 0;

      for (d = 0; d < 3; d++) {
        tdata[1] += pow(X[i+1][d] - X[i][d],2);
      }
      tdata[1] = sqrt(tdata[1]);

      status = EG_setRange_dot(eedges[i], EDGE, tdata);
      CHECK_STATUS(EG_setRange_dot);
    }

    /* plane data */
    data[0] = 0.;
    data[1] = 0.;
    data[2] = 0.;
    data[3] = 1.; data[4] = 0.; data[5] = 0.;
    data[6] = 0.; data[7] = 1.; data[8] = 0.;

    status = EG_setGeometry_dot(eplane, SURFACE, PLANE, NULL, data);
    CHECK_STATUS(EG_setGeometry_dot);
  }


  /* find the ego entity */
  if (entType == OCSM_NODE) {
    status = EG_getBodyTopos(ebody, NULL, NODE, &nchild, &echildren);
    CHECK_STATUS(EG_getBodyTopos);

    stride = 0;
    eent = echildren[entIndex-1];

    FREE(echildren);
  } else if (entType == OCSM_EDGE) {
    status = EG_getBodyTopos(ebody, NULL, EDGE, &nchild, &echildren);
    CHECK_STATUS(EG_getBodyTopos);

    stride = 1;
    eent = echildren[entIndex-1];

    FREE(echildren);
  } else if (entType == OCSM_FACE) {
    status = EG_getBodyTopos(ebody, NULL, FACE, &nchild, &echildren);
    CHECK_STATUS(EG_getBodyTopos);

    stride = 2;
    eent = echildren[entIndex-1];

    FREE(echildren);
  } else {
    printf("udpSensitivity: bad entType=%d\n", entType);
    status = EGADS_GEOMERR;
    goto cleanup;
  }


  /* get the velocities from the entity */
  for (ipnt = 0; ipnt < npnt; ipnt++) {
    status = EG_evaluate_dot(eent, &(uvs[stride*ipnt]), NULL, point, point_dot);
    CHECK_STATUS(EG_evaluate_dot);

    /* return the point velocity */
    vels[3*ipnt  ] = point_dot[0];
    vels[3*ipnt+1] = point_dot[1];
    vels[3*ipnt+2] = point_dot[2];
  }


cleanup:
  if (strlen(message) > 0) {
   printf("%s\n", message);
  } else if (status != EGADS_SUCCESS) {
   printf("ERROR: status = %d \n", status);
  }

  FREE(message);

  return status;

}
