/*
 ************************************************************************
 *                                                                      *
 * OpenCSM -- an open-source constructive solid modeler                 *
 *                                                                      *
 *            Written by John Dannenhoffer @ Syracuse University        *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2010/2025  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <float.h>
#include <math.h>
#include <assert.h>
#include <sys/stat.h>

#define FORCE_FINITE_DIFFS  0           /* =1 to force finite differences */
#define INTERP_VEL_SCHEME   7           /* =0 use eggMorph, =6 use egads coarse mapping, =7 use Laplace-Beltrami */
#define MORPH_GRID          1           /* =0 use EG_mapTessBody, =1 use EGG_morph if possible */
#define CLEAR_DOTS          0           /* =1 to clear dots from all Bodys after each Branch is executed */

#define REMOVE_SPURIOUS     0           /* =1 to remove spurious Nodes */

#define PRINT_SKETCH_TABLE  0           /* =1 to print sketch table when SKEND is executed */
#define PRINT_PARAMETERS    0           /* =1 to print Parameters at the beginning of each Branch */
#define PRINT_BODYS         0           /* =1 to print Bodys for each statement */
#define PRINT_LAST_EGO      0           /* =1 to print Body nbody, =2 to print Body nbody with attributes */
#define PRINT_PROGRESS      0           /* =1 to print progress at end of each Branch */
#define PRINT_GEOMSENS      0           /* =1 to print velocities foreach Node,Edge,Face, =2 to show how (use outLevel=2) */
#define PRINT_TESSSENS      0           /* =1 to print at each stage of tessellation sensitivity generation */
#define PRINT_BODY_INFO     0           /* =1 to print Body info for all Bodys on stack */
#define PRINT_MAPPING_INFO  0           /* =1 to print mapping differences during finite difference tess. sens. */
#define PRINT_CALLHISTORY   0           /* =1 to generate callHistory associated with sensitivities */
#define PRINT_DOT_STATUS    0           /* =1 to show "dot" status of each Storage and Body */
#define PRINT_EDATA         0           /* =1 to show @edata after every EVALUATE statement */
#define PRINT_NONMANIFOLD   0           /* =1 to show manifold Edges in ocsmPrintBodys */

                                        /* define to generate segfault when signalError is called */
//#define SEGFAULT_ON_SIGNAL  1

// the following all need "make grafic")
#define PLOT_SPLINES        0           /* =1 to plot splines */
#define PLOT_ARCLENGTHS     0           /* =1 to plot arclength distributions */
#define PLOT_TRIMCURVES     0           /* =1 to plot trim curves at end of ocsmBuild */
#define PLOT_UVTESS         0           /* =1 to plot tessellation in UV */
#define PLOT_EDGEVELS       0.0         /* >0 to plot tessellation sensitivities on Edges */
#define PLOT_FACEVELS       0.1         /* >0 to plot tessellation sensitivities on Faces */

#include "OpenCSM.h"
#include "udp.h"
#include "egg.h"
#include "egads_dot.h"
#include "egadsSplineVels.h"
#include "emp.h"

#if PRINT_PROGRESS
    #define PPRINT_INIT(MESG)  cpu_beg = clock(); SPRINT0(0, #MESG);
    #define PPRINT0(MESG)      cpu_end = clock(); SPRINT1(0, "...%10.3f sec: " #MESG, (double)(cpu_end-cpu_beg)/(double)(CLOCKS_PER_SEC)); cpu_beg=cpu_end;
    #define PPRINT1(MESG,A)    cpu_end = clock(); SPRINT2(0, "...%10.3f sec: " #MESG, (double)(cpu_end-cpu_beg)/(double)(CLOCKS_PER_SEC), A); cpu_beg=cpu_end;
    #define PPRINT2(MESG,A,B)  cpu_end = clock(); SPRINT3(0, "...%10.3f sec: " #MESG, (double)(cpu_end-cpu_beg)/(double)(CLOCKS_PER_SEC), A, B); cpu_beg=cpu_end;
#else
    #define PPRINT_INIT(MESG)
    #define PPRINT0(MESG)
    #define PPRINT1(MESG,A)
    #define PPRINT2(MESG,A,B)
#endif

#ifdef WIN32
    #include <windows.h>
    #include <direct.h>
    #define  snprintf    _snprintf
    #define  strcasecmp  _stricmp
    #define  SLASH       '\\'
    #define  DLL         HINSTANCE
    #define  getcwd      _getcwd
    #define  MKDIR(A)    mkdir(A)
#else
    #include <strings.h>
    #include <unistd.h>
    #include <dlfcn.h>
    #define  SLASH       '/'
    #define  DLL         void *
    #define  MKDIR(A)    mkdir(A, 0777)
#endif

#define CINT    const int
#define CDOUBLE const double
#define CCHAR   const char
#define STRNCPY(TO, FROM, LEN) strncpy(TO, FROM, LEN); TO[LEN-1] = '\0';
#define STRNCAT(TO, FROM, LEN) strncat(TO, FROM, LEN); TO[LEN-1] = '\0';
#define EPRINT  printf

#define   DTIME_NOM           0.00001     /* nominal finite difference step */

#ifdef GRAFIC
    #include "grafic.h"
#endif

#define OCSM_MAGIC        4433340       /* magic number */
#define TESS_PARAM_0      0.0250
#define TESS_PARAM_1      0.0075
#define TESS_PARAM_2      20.0

/*
 ************************************************************************
 *                                                                      *
 * Structures                                                           *
 *                                                                      *
 ************************************************************************
 */

/* "Rpn" contains information associated with Rpn (pseudo-code) */
typedef struct {
    int    type;                       /* type (see below) */
    char   text[MAX_STRVAL_LEN];       /* associated text */
} rpn_T;

/* "Stack" is used within the RPN evaluator */
typedef struct {
    double val;                        /* value */
    double dot;                        /* velocity */
    int    nan;                        /* =1 if NaN */
    char   str[MAX_STRVAL_LEN];        /* string value */
} stack_T;

/* "Patn" contains information associated with a patbeg/patend,
          ifthen/elseif/else/endif, catbeg/catend, or macbeg/macend pair */
typedef struct {
    int    itype;                      /* OCSM_PATBEG, OCSM_IFTHEN, OCSM_CATBEG, or OCSM_RECALL */
    int    ibeg;                       /* Branch number of patbeg, ifthen, catbeg, or recall */
    int    iend;                       /* Branch number of patend, endif,  catend */
    int    ncopy;                      /* total number of copies */
    int    icopy;                      /* current instance number (1:ncopy) */
    int    ipmtr;                      /* Parameter index of iterator */
} patn_T;

/* "Sket" is a Sketch */
typedef struct {
    int    type;                       /* 0 for old style, 1 for xy, 2 for yz, 3 for zx */
    int    irel;                       /* =1 if coordinates are relative to SKBEG */
    int    size;                       /* size of x[], y[], and z[] */
    int    solved;                     /* =1 if Sketch is solved */
    int    signal;                     /* =0 if no signal */
    int    ix;                         /* Parameter index associated with ::x[] */
    int    iy;                         /* Parameter index associated with ::y[] */
    int    iz;                         /* Parameter index associated with ::z[] */
    int    id;                         /* Parameter index associated with ::d[] */

    double xbase;                      /* X-coordinate in skbeg statement */
    double ybase;                      /* Y-coordinate in skbeg statement */
    double zbase;                      /* Z-coordinate in skbeg statement */
    double ubase;                      /* X-velocity   in skbeg statement */
    double vbase;                      /* Y-velocity   in skbeg statement */
    double wbase;                      /* Z-velocity   in skbeg statement */

    int    nseg;                       /* number of segments */
    int    itype[MAX_SKETCH_SIZE];     /* Branch type */
    int    ibrch[MAX_SKETCH_SIZE];     /* Branch index */
    double x[    MAX_SKETCH_SIZE];     /* X-coordinate at end of segment */
    double y[    MAX_SKETCH_SIZE];     /* Y-coordinate at end of segment */
    double z[    MAX_SKETCH_SIZE];     /* Z-coordinate at end of segment */
    double u[    MAX_SKETCH_SIZE];     /* X-velocity   at end of segment */
    double v[    MAX_SKETCH_SIZE];     /* Y-velocity   at end of segment */
    double w[    MAX_SKETCH_SIZE];     /* Z-velocity   at end of segment */

    int    nvar;                       /* number of variables (at least 2*nseg) */
    int    ipmtr[MAX_SKETCH_SIZE];     /* Parameter index */
    int    index[MAX_SKETCH_SIZE];     /* value index */

    int    ncon;                       /* number of constraints */
    char   ctype[MAX_SKETCH_SIZE];     /* constraint type (letter) */
    int    ipnt[ MAX_SKETCH_SIZE];     /* first  constraint index */
    int    ip1[  MAX_SKETCH_SIZE];     /* second constraint index */
    char   con[MAX_SKETCH_SIZE][MAX_EXPR_LEN];
} sket_T;

/* user data information for EG_*_vels spline sensitvity functions */
typedef struct {
    int           *isketch;            /* array  of Body numbers sketches */
    modl_T        *MODL;               /* pointer to MODL */
} egadsSpline_T;

/* red-black tree */
typedef struct {
    int    nnode;                      /* current number of Nodes */
    int    mnode;                      /* maximum number of Nodes */
    int    root;                       /* index of root Node */
    int    chunk;                      /* chunk size */
    int*   key1;                       /* array of primary    keys */
    int*   key2;                       /* array of secondary  keys */
    int*   key3;                       /* array of tertiary   keys */
    int*   key4;                       /* array of quaternary keys */
    int*   data;                       /* data associated with entry */
    int*   left;                       /* array of left children */
    int*   rite;                       /* array of rite children */
    int*   prnt;                       /* array of parents */
    int*   colr;                       /* array of colors */
} rbt_T;

/* parallelization structure for ocsmJacobian and ocsmAdjoint */
typedef struct {
    void      *mutex;                   /* the mutex or NULL for single thread */
    long      master;                   /* master thread ID */
    int       nthread;                  /* number of threads */
    modl_T    *MODL;                    /* pointer to base MODL */
    int       nglob;                    /* global number of points in tessellation */

    int       ibody;                    /* Body index (1:nbody) */
    int       idp;                      /* next DESPMTR to use */
    int       ndp;                      /* number of DESPMTRs */
    int       *ipmtr;                   /* array  of DESPMTRS */
    int       *irow;                    /* array of row numbers */
    int       *icol;                    /* array of column numbers */
    int       nobj;                     /* number of objective functions */
    double    *dOdX;                    /* array of d(obj)/d(xyz) -- input */
    double    *dOdD;                    /* array of d(obj)/d(DESPMTR) -- output */
    double    *dXdD;                    /* array of d(xyz)/d(DESPMTR) -- output */
    int       status;                   /* error return */
} empJA_T;

/*
 ************************************************************************
 *                                                                      *
 * Definitions (for structures above)                                   *
 *                                                                      *
 ************************************************************************
 */

#define           PARSE_NOP         0   /* no operation */
#define           PARSE_OP1         1   /* either add "+" or subtract "-" */
#define           PARSE_OP2         2   /* either multiply "*" or divide "/" */
#define           PARSE_OP3         3   /* exponentiation "^" */
#define           PARSE_OPENP       4   /* open  parenthesis "(" */
#define           PARSE_CLOSEP      5   /* close parenthesis ")" */
#define           PARSE_OPENB       6   /* open  bracket "[" */
#define           PARSE_CLOSEB      7   /* close bracket "]" */
#define           PARSE_COMMA       8   /* comma "," */
#define           PARSE_NAME        9   /* variable name */
#define           PARSE_ARRAY      10   /* array name */
#define           PARSE_FUNC       11   /* function name */
#define           PARSE_NUMBER     12   /* number */
#define           PARSE_STRING     13   /* string */
#define           PARSE_END        14   /* end of Rpn-code */

#define           RBT_BLACK         0
#define           RBT_RED           1

/*
 ************************************************************************
 *                                                                      *
 * Declarations for support routines defined below                      *
 *                                                                      *
 ************************************************************************
 */

static int addTraceToEdge(modl_T *modl, int ibody, int iedge);
static int addTraceToFace(modl_T *modl, int ibody, int iface);
static int addTraceToNode(modl_T *modl, int ibody, int inode);
static int buildApplied(  modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildBoolean(  modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildBoolConne(modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildBoolEleva(modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildBoolExtra(modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildBoolInter(modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildBoolJoin( modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildBoolSubtr(modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildBoolUnion(modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildGrown(    modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildPrimitive(modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[]);
static int buildSketch(   modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[],
                          int npatn, patn_T patn[], sket_T *sket);
static int buildSolver(   modl_T *modl, int ibrch, varg_T args[], int *nvar, int solvars[],
                          int *ncon, int solcons[]);
static int buildTransform(modl_T *modl, int ibrch, varg_T args[], int *nstack, int stack[]);
static int colorizeEdge(modl_T *modl, int ibody, int iedge);
static int colorizeFace(modl_T *modl, int ibody, int iface);
static int colorizeNode(modl_T *modl, int ibody, int inode);
static int compressFilename(char filename[]);
static void computeAdjoint(void *empStruct);
static void computeJacobian(void *empStruct);
static int computeMassProps(modl_T *modl);
static int computeMassPropsDot(modl_T *modl);
       int convertToBSplines(ego inbody, double mat[], int mincp, int mindeg, ego *ebody);
static int copyEdgeAttributes(modl_T *modl, int ibody, int nxsect, ego xsects[]);
static int copyUserEdgeAttributes(ego src, ego tgt);
static int createPerturbation(modl_T *modl);
       int createTessVels(modl_T *modl, int ibody);
static int createVelocityCache(modl_T *MODL, int jbody);
static int delPmtrByName(modl_T *modl, char name[]);
static int distBodyInSolid(modl_T *modl, int ibody1, int ibody2, double *dist, double pnt1[], double pnt2[]);
static int distBodyNearBody(modl_T *modl, int ibody1, int ibody2, double *dist, double pnt1[], double pnt2[]);
static int distBodyNearSolid(modl_T *modl, int ibody1, int ibody2, double *dist, double pnt1[], double pnt2[]);
static int distEdgeEdge(const ego edge1, double param1[], const ego edge2, double param2[], int mm);
static int distEdgeNode(const ego edge, double *t, const double xtz[], int mm);
static int distFaceEdge(const ego face, double param1[], const ego edge, double param2[], int mm);
static int distFaceFace(const ego face1, double uv1[], const ego face2, double uv2[], int mm);
static int distFaceNode(const ego face, double uv[], const double xyz[], int mm);
static int distObjObj(modl_T *modl, int ibody1, double pnt1[], int ibody2, double pnt2[], int mm);
static int distSetupPnt(ego etess, int ptype, int pindx, double pnt[]);
static int dumpEgadsFile(modl_T *modl, int ibody);
//$$$static int efaceJacobian(modl_T *MODL, int ibody, int iface, double dudue[], double dvdue[], double dudve[], double dvdve[]);
static int evalRpn(rpn_T *rpn, /*@null@*/modl_T *modl, double *val, double *dot, char str[]);
static int faceContains(ego eface, double xx, double yy, double zz);
static int finishBody(modl_T *modl, int ibody);
static int finishCopy(modl_T *modl, int src, /*@null@*/double matrix[], int ibody);
static int finiteDifference(modl_T *modl, int ibody, int seltype, int iselect, int npnt, /*@null@*/double uv[], double dxyz[]);
static int fixSketch(sket_T *sket, char vars_in[], char cons_mod[]);
static int fixSketchRank(sket_T *sket, int npnt, int segtyp[], int *jrank);
static int freeBody(modl_T *modl, int ibody);
static int getBodyTolerance(ego ebody, double *toler);
static int getEdgeHistory(modl_T *MODL, int ibody, int iedge, int *nhist, int *hist[]);
static int getShellFromBody(modl_T *modl, ego ebody, ego *eshell);
static int getToken(char *text, int nskip, char sep, int maxtok, char *token);
static int joinSheetBodys(modl_T *modl, int ibodyl, int ibodyr, int itype, double toler, ego *ebody);
static int joinWireBodys(modl_T *modl, int ibodyl, int ibodyr, double toler, ego *ebody);
static double largestDot(ego ebody);
static int makeClone(empJA_T *empJA, modl_T *srcModl, modl_T *tgtModl);
static int makeEdge(modl_T *modl, ego ebeg, ego eend, ego *eedge);
static int makeFace(modl_T *modl, ego eedges[], int fillstyle, int dirn, double toler, ego *eface);
static int makeNodeBody(modl_T *modl, double xyz[], ego *ebody);
static int markNodesAsKeep(modl_T *modl, ego ebody, int inodyl, int ibodyr, char label[]);
static int matches(char pattern[], const char string[]);
//$$$static int matchBodys(modl_T *modl, int ibody, int jbody, double toler);
static int matchEdges(modl_T *modl, int ibody, int iedge, int jbody, int jedge, double toler);
static int matchEdges2(modl_T *modl, ego eedge, int jbody, int jedge, double toler);
static int matchFaces(modl_T *modl, int ibody, int iface, int jbody, int jface, double toler);
static int matchLoop(ego eloops[]);
//$$$static int matchNodes(modl_T *modl, int ibody, int inode, int jbody, int jnode, double toler);
static int matchValue(varg_T arg, int itype, int nlist, CINT *tempIlist, CDOUBLE *tempRlist, CCHAR *tempClist);
static int matsol(double A[], double b[], int n, int m, double x[]);
//$$$static int mvcInterp(int nloop, CINT nper[], CDOUBLE uvframe[], CDOUBLE uv[], double weights[]);
static int newBody(modl_T *modl, int ibrch, int brtype, int ileft, int irite, /*@null@*/varg_T args[], int hasdots, int botype, int *ibody);
static int newSeqnum(modl_T *modl, int ibody);
static int parseName(modl_T *modl, char string[], char pname[], int *ipmtr, int *irow, int *icol);
static int printBodyAttrs(modl_T *modl, int ibody);
static int printBodyInfo(body_T *body);
static int printEgoList(ego current);
static int printFaceVelocities(modl_T *modl, int ibody, int iface);
static int printNurbs(ego ebody);
static int printPmtrs(modl_T *modl, FILE *fp);
static int rank(double mat[], int nrow, int ncol);
//$$$static int rbfWeights(int nbnd, double srad2, CDOUBLE uv[], double duv[], double weights[]);
static int rbtCompareKeys(int ikey1, int jkey1, int ikey2, int jkey2, int ikey3, int jkey3, int ikey4, int jkey4);
static int rbtCreate(int chunk, rbt_T **tree);
static int rbtDelete(rbt_T *tree);
static int rbtInsert(rbt_T *tree, int key1, int key2, int key3, int key4, int data);
static int rbtLookup(rbt_T *tree, int inode);
//$$$static int rbtMaximum(rbt_T *tree, int istart);
//$$$static int rbtMinimum(rbt_T *tree, int istart);
//$$$static int rbtNext(rbt_T *tree, int istart);
//$$$static int rbtPrev(rbt_T *tree, int istart);
static void rbtRotateLeft(rbt_T *tree, int inode);
static void rbtRotateRite(rbt_T *tree, int inode);
static int rbtSearch(rbt_T *tree, int key1, int key2, int key3, int key4);
static int recycleBody(modl_T *modl, int ibrch, int brtype, varg_T args[], int hasdots);
static int removeFaceAttributes(ego ebody);
static int removePerturbation(modl_T *modl);
       int removeVels(modl_T *modl,  int ibody);
static int reorderLoops(modl_T *modl, int nloop, ego eloops[], int startFrom);
static int reportBestMatch(modl_T *modl, int ibodyl, int ibodyr);
static int selectBody(ego emodel, char *order, int index);
static int setEgoAttribute(modl_T *modl, int ibrch, ego eobject);
static int setFaceAttribute(modl_T *modl, int ibody, int iface, int jbody, int jford, int npatn, patn_T *patn);
static int setupAtPmtrs(modl_T *modl, int havesel);
static int setupForFiniteDifferences(modl_T *modl);
static int setupUdprimFile(modl_T *modl, int itype, FILE *csm_file, char filename[], int *linenum, char str[]);
static int signalError(/*@null@*/void *modl, int status, char format[], ...);
static int signalError2(/*@null@*/void *modl, int status, char filename[], int linenum, char format[], ...);
static int solidBoolean(modl_T *modl, ego ebodyl, ego ebodyr, int type, double maxtol, ego *emodel);
static int solveSketch(modl_T *modl, sket_T *sket);
static int solveSketchLM(modl_T *modl, sket_T *sket);
static int solveSketchOrig(modl_T *modl, sket_T *sket);
static int sparseInit(int nrow, int ncol, double Amat[], int Imat[]);
static int sparseInsert(int irow, int icol, double A, int nrow, int ncol, double Amat[], int Imat[]);
static int sparseSolve(int nrow, int ncol, double Amat[], int Imat[], int nrhs, double Rhs[], double Soln[]);
static int splineVelocityOfBspline(/*@unused@*/void* usrData, /*@unused@*/const ego secs[], /*@unused@*/int isec, /*@unused@*/ego eedge, ego egeom, int *ivec[], double *rvec[], double *rvec_dot[]);
static int splineVelocityOfEdge(void* usrData, /*@unused@*/const ego secs[], int isec, ego eedge, CINT npnt, CDOUBLE ts[], CDOUBLE ts_dot[], double xyz[], double xyz_dot[], double dxdt_beg[], double dxdt_beg_dot[], double dxdt_end[], double dxdt_end_dot[]);
static int splineVelocityOfNode(void* usrData, /*@unused@*/const ego secs[], int isec, ego enode, /*@unused@*/ego eedge, double xyz[], double xyz_dot[]);
static int splineVelocityOfRange(void* usrData, /*@unused@*/const ego secs[], int isec, ego eedge, double trange[], double trange_dot[]);
static int storeCsystem(modl_T *modl, int ibrch, int ibody);
static int str2rpn(char str[], char prefix[], rpn_T *rpn);
static int str2val(char expr[], /*@null@*/modl_T *modl, double *val, double *dot, char str[]);
static int str2valNoSignal(char expr[], modl_T *modl, double *val, double *dot, char str[]);
static int str2vals(char expr[], modl_T *modl, int *nrow, int *ncol, double *vals[], double *dots[], char str[]);
static int solsvd(double A[], double b[], int mrow, int ncol, double W[], double x[]);
static int velocityForPrimitive(modl_T *modl, int ibody, int npnt, double xyz[], double xyz_dot[]);
       int velocityOfEdge(modl_T *modl, int ibody, int iedge, int npnt, /*@null@*/double t[], double dxyz[]);
       int velocityOfFace(modl_T *modl, int ibody, int iface, int npnt, /*@null@*/double uv[], double dxyz[]);
       int velocityOfNode(modl_T *modl, int ibody, int inode, double dxyz[]);
//$$$static double wendland(CDOUBLE uv1[], CDOUBLE uv2[], double srad2);
static int writeAsciiStl(modl_T *modl, int nstack, int stack[], char filename[]);
static int writeAsciiUgrid(modl_T *modl, int ibody, char filename[]);
static int writeBinaryStl(modl_T *modl, int nstack, int stack[], char filename[]);
static int writeObjFile(modl_T *MODL, int ibody, char attrName[], char filename[]);
static int writePlotFile(modl_T *modl, int ibody, char filename[]);
static int writeSensFile(modl_T *modl, int ibody, char filename[]);
static int writeTessFile(modl_T *modl, int ibody, char filename[]);
static int writeVtkFile(modl_T *MODL, int ibody, char attrName[], char filename[]);
static int xformEdgeToOriginal(modl_T *modl, int ibody, int iedge, double mat[], double *scale);
static int xformFaceToOriginal(modl_T *modl, int ibody, int iface, double mat[], double *scale);
static int xformNodeToOriginal(modl_T *modl, int ibody, int inode, double mat[], double *scale);
static int xformVelocity(modl_T *modl, int jbody, int nxyz, double xyz_pnt[], double dxyz[]);

#ifdef GRAFIC
static void plotFace(int*, void*, void*, void*, void*, void*,
                           void*, void*, void*, void*, void*, float*, char*, int);
static int  plotEdgeVels(char pltitl[], modl_T *MODL, int ibody, int iedge, double data[]);
static int  plotFaceVels1(char pltitl[], modl_T *MODL, int ibody, int iface, int nrank, double data[]);
static void plotFaceVels3(int*, void*, void*, void*, void*, void*,
                          void*, void*, void*, void*, void*, float*, char*, int);
#endif

/* non-prototyped EGADS routines */
extern int EG_getEdgeUVeval(ego eface, ego eedge, int sense, double t, double *uv);
extern int EG_getTessEFace(ego tess, int index, ego *faces, double *uvs);
extern int EG_sensitTopo(int iface, double r[], double dxyz[]);
extern int EG_setUserPointer(ego context, void *ptr);
extern int EG_spline1dFit(int endx, int imaxx, const double *xyz, /*@null@*/const double *kn, double tol, int *ivec, double **rdata);
extern int EG_spline1dFit_dot(int endx, int imaxx, const double *xyz, const double *xyz_dot, /*@null@*/const double *kn, /*@null@*/const double *kn_dot, double tol, int *ivec, double **rdata, double **rdata_dot);
extern int EG_spline1dTan(int imaxx, /*@null@*/const double *t1, const double *xyz, /*@null@*/const double *tn, /*@null@*/const double *kn, double tol, int *ivec, double **rdata);
extern int EG_spline1dTan_dot(int imaxx, /*@null@*/const double *t1, /*@null@*/const double *t1_dot, const double *xyz, const double *xyz_dot, /*@null@*/const double *tn, /*@null@*/const double *tn_dot, /*@null@*/const double *kn, /*@null@*/const double *kn_dot, double tol, int *ivec, double **rdata, double **rdata_dot);
extern int EG_getTessFrame(const ego tess, int index, const egBary **bary, int *nftri, const int **ftris);
extern int EG_relPosTs(ego geom, int n, /*@null@*/ const double *rel, double *ts, double *xyzs);

#if PRINT_CALLHISTORY > 0
   FILE *fp_callHistory;
   int  i_callHistory, indent_callHistory=0;
#endif

/*
 ************************************************************************
 *                                                                      *
 * Global variables and defines                                         *
 *                                                                      *
 ************************************************************************
 */

static int outLevel  = 1;     /* global since it needs to be settable
                                 before a MODL is created */

static void *auxPtr  = NULL;  /* auxiliary pointer */


/*
 ************************************************************************
 *                                                                      *
 *   ocsmVersion - return current version                               *
 *                                                                      *
 ************************************************************************
 */

int
ocsmVersion(int    *imajor,             /* (out) major version number */
            int    *iminor)             /* (out) minor version number */
{
    int       status = SUCCESS;         /* (out) return status */

    ROUTINE(ocsmVersion);

    /* --------------------------------------------------------------- */

    *imajor = OCSM_MAJOR_VERSION;
    *iminor = OCSM_MINOR_VERSION;

//cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSetOutLevel - set output level                                 *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSetOutLevel(int    ilevel)      /* (in)  output level: */
                                    /*       <0 do not change */
                                    /*       =0 errors only */
                                    /*       =1 nominal (default) */
                                    /*       =2 debug */
{
    int       old_outLevel = 0;     /* (out) previous outLevel */

    ROUTINE(ocsmSetOutLevel);

    /* --------------------------------------------------------------- */

    /* note that this does not change EGADS' outLevel (since we do not
       have access to the EGADS context -- in fact it might not be
       created yet) */

    old_outLevel = outLevel;       /* global variable */
    if (ilevel >= 0) {
        outLevel     = ilevel;
    }

//cleanup:
    return old_outLevel;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSetAuxPtr - set (global) auxiliary pointer                     *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSetAuxPtr(void   *newAuxPtr)        /* (in)  auxiliary pointer */
{
    int       status = SUCCESS;         /* (out) return status */

    ROUTINE(ocsmSetAuxPtr);

    /* --------------------------------------------------------------- */

    auxPtr = newAuxPtr;

//cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetAuxPtr - get (global) auxiliary pointer                     *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetAuxPtr(void   **oldAuxPtr)       /* (out) auxiliary pointer */
{
    int       status = SUCCESS;         /* (out) return status */

    ROUTINE(ocsmGetAuxPtr);

    /* --------------------------------------------------------------- */

    *oldAuxPtr = auxPtr;

//cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmLoad - create a MODL by reading a .csm file                    *
 *                                                                      *
 ************************************************************************
 */

int
ocsmLoad(char   filename[],             /* (in)  file to be read (with .csm) */
         void   **modl)                 /* (out) pointer to MODL */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL=NULL;

    /* Sketch contains linsegs, cirarcs, splines, and beziers */
    int       nskpt;

    /* number of open patterns */
    int       npatn = 0;

    int       filelen, filetype, linenum, numudc;
    int       i, j, k, inquote, narg, irow, nrow, icol, ncol, indx, count;
    int       ibrch, jbrch, ipmtr, jpmtr, icount, jcount, insolver, ival, nval;
    int       udp_num, *udp_types, *udp_idef, ifound;
    double    value, dot, bound, *udp_ddef;
    char      *templine=NULL, *nextline=NULL, *command=NULL, *str=NULL;
    char      *str1=NULL, *str2=NULL, *str3=NULL;
    char      *str4=NULL, *str5=NULL, *str6=NULL;
    char      *str7=NULL, *str8=NULL, *str9=NULL;
    char      defn[MAX_EXPR_LEN], tmpfilename[MAX_EXPR_LEN], pathname[MAX_EXPR_LEN];
    char      pmtrName[MAX_EXPR_LEN], **udp_names;
    void      *temp;

    FILE      *fp_csm=NULL;

    ROUTINE(ocsmLoad);

#define CHECK_STATUS2(X)                                                \
    if (status < SUCCESS) {                                             \
        (void) signalError2(MODL, status,                               \
                            filename, linenum,                          \
                            "error detected in %s during ocsmLoad()", #X); \
        goto cleanup;                                                   \
    }

    /* --------------------------------------------------------------- */

    SPRINT1(1, "--> enter ocsmLoad(filename=%s)", filename);

    linenum = 0;

    /* check for valid extension */
    filelen = STRLEN(filename);
    if (filelen == 0) {
        filetype = -1;
    } else if (filelen < 5) {
        SPRINT0(2, "    illegal filename");
        filetype = 0;
    } else if (filename[filelen-4] == '.' && filename[filelen-3] == 'c' &&
               filename[filelen-2] == 's' && filename[filelen-1] == 'm'   ) {
        SPRINT0(2, "    reading .csm file");
        filetype = 0;
    } else if (filename[filelen-4] == '.' && filename[filelen-3] == 'c' &&
               filename[filelen-2] == 'p' && filename[filelen-1] == 'c'   ) {
        SPRINT0(2, "    reading .cpc file");
        filetype = 1;
    } else if (filename[filelen-4] == '.' && filename[filelen-3] == 'u' &&
               filename[filelen-2] == 'd' && filename[filelen-1] == 'c'   ) {
        SPRINT0(2, "    reading .udc file");
        filetype = 2;
    } else if (filename[filelen-4] == '.' && filename[filelen-3] == 'v' &&
               filename[filelen-2] == 'f' && filename[filelen-1] == 'y'   ) {
        SPRINT0(2, "    reading .vfy file");
        filetype = 3;
    } else {
        *modl  = NULL;
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    }

    (void) compressFilename(filename);

    if (FORCE_FINITE_DIFFS == 1) {
        SPRINT0(0, "WARNING:: ***************************************");
        SPRINT0(0, "WARNING:: all sensitivities by finite differences");
        SPRINT0(0, "WARNING:: ***************************************");
    }

    if (MORPH_GRID != 1) {
        SPRINT1(0, "MORPH_GRID = %d", MORPH_GRID);
    }

    MALLOC(templine, char, MAX_LINE_LEN);
    MALLOC(nextline, char, MAX_LINE_LEN);
    MALLOC(command,  char, MAX_LINE_LEN);

    MALLOC(str,  char, MAX_STRVAL_LEN);

    MALLOC(str1, char, MAX_LINE_LEN);
    MALLOC(str2, char, MAX_LINE_LEN);
    MALLOC(str3, char, MAX_LINE_LEN);
    MALLOC(str4, char, MAX_LINE_LEN);
    MALLOC(str5, char, MAX_LINE_LEN);
    MALLOC(str6, char, MAX_LINE_LEN);
    MALLOC(str7, char, MAX_LINE_LEN);
    MALLOC(str8, char, MAX_LINE_LEN);
    MALLOC(str9, char, MAX_LINE_LEN);

    /* initializtions are only done if a .csm or .cpc file */
    if (filetype == -1 || filetype == 0 || filetype == 1) {

        /* make a new MODL and initialize it */
        MALLOC(MODL, modl_T, 1);

        MODL->magic      = OCSM_MAGIC;
        MODL->checked    = 0;
        MODL->embedded   = 0;
        MODL->ibrch      = 0;
        MODL->nextseq    = 1;
        MODL->ngroup     = 0;
        MODL->recycle    = 0;
        MODL->verify     = 0;
        MODL->cleanup    = 25;
        MODL->dumpEgads  = 0;
        MODL->loadEgads  = 0;
        MODL->hasMPs     = 0;
        MODL->printStack = 0;
        MODL->printAttr[0] = '\0';
        MODL->printPmtrs = 0;
        MODL->printStors = 0;
        MODL->tessAtEnd  = 1;
        MODL->erepAtEnd  = 0;
        MODL->bodyLoaded = 0;

        MODL->seltype = -1;
        MODL->selbody = -1;
        MODL->selsize =  0;
        MODL->sellist = NULL;

        MODL->level = 0;

        for (i = 0; i < 11; i++) {
            MODL->scope[i] = 0;
        }

        if (STRLEN(filename) == 0) {
            MODL->filename[0] = '\0';
        } else {
            /*@ignore@*/
            STRNCPY(MODL->filename, filename, MAX_FILENAME_LEN);
            /*@end@*/
        }

        MODL->tmpDirNum = -1;

        MODL->ninline = 0;
        MODL->minline = 0;
        MODL->sinline = NULL;

        MODL->nattr = 0;
        MODL->attr  = NULL;

        MODL->nstor = 0;
        MODL->stor  = NULL;

        MODL->nbrch = 0;
        MODL->mbrch = 0;
        MODL->brch  = NULL;

        MODL->npmtr = 0;
        MODL->mpmtr = 0;
        MODL->pmtr  = NULL;
        MODL->prefix[0] = '\0';

        MODL->nbody = 0;
        MODL->mbody = 0;
        MODL->body  = NULL;

        MODL->forceFDs   = 0;
        MODL->numdots    = 0;
        MODL->perturb    = NULL;
        MODL->basemodl   = NULL;
        MODL->matchSeq   = NULL;
        MODL->dtime      = 0;

        MODL->context     = NULL;
        MODL->userdata    = NULL;
        MODL->mesgCB      = NULL;
        MODL->bcstCB      = NULL;
        MODL->sizeCB      = NULL;

        MODL->eggname[0]  = '\0';
        MODL->eggGenerate = NULL;
        MODL->eggMorph    = NULL;
        MODL->eggInfo     = NULL;
        MODL->eggDump     = NULL;
        MODL->eggLoad     = NULL;
        MODL->eggFree     = NULL;

        for (i = 0; i < MAXPRIM; i++) {
            MODL->NumUdp[i] = 0;
            MODL->Udps[  i] = NULL;
        }

        MODL->nwarn   = 0;
        MODL->sigCode = 0;
        MODL->sigMesg = NULL;
        MALLOC(MODL->sigMesg, char, MAX_STR_LEN);
        MODL->sigMesg[0] = '\0';

        for (i = 0; i < 101; i++) {
            MODL->profile[i].ncall = 0;
            MODL->profile[i].time  = 0;
        }

    /* if reading a .udc file ensure that we are already
       reading a .csm or .cpc file */
    } else if (filetype == 2) {
        MODL = *modl;

        assert (MODL != NULL);
        assert (STRLEN(MODL->filename) > 0);

    /* if reading a .vfy file ensure that we are already
       reading a .csm or .cpc file */
    } else {
        MODL = *modl;

        assert (MODL != NULL);
        assert (STRLEN(MODL->filename) > 0);
    }

    /* return value */
    *modl = MODL;

    if (filetype == -1) {
        SPRINT0(1, "WARNING:: loading empty MODL");
        (MODL->nwarn)++;
        goto cleanup;
    }

    if (STRLEN(filename) > MAX_FILENAME_LEN) {
        status = signalError2(MODL, OCSM_ILLEGAL_ARGUMENT, filename, linenum,
                              "filename has more than %d characters", MAX_FILENAME_LEN);
        goto cleanup;
    }

    /* open the file */
    if (STRLEN(filename) > 0) {
        fp_csm = fopen(filename, "r");
        if (fp_csm == NULL) {
            if (strstr(filename, ".vfy") != NULL) {
                SPRINT1(0, "WARNING:: cannot open \"%s\", so verification is being skipped", filename);
                (MODL->nwarn)++;
                status = SUCCESS;
            } else {
                status = signalError2(MODL, OCSM_FILE_NOT_FOUND, filename, linenum,
                                      "file \"%s\" not found", filename);
                goto cleanup;
            }
            goto cleanup;
        }
    }

    /* initialize the number of active Sketch points, patterns, and active UDCs */
    nskpt  = 0;
    npatn  = 0;
    numudc = 0;

    insolver = 0;

    SPLINT_CHECK_FOR_NULL(MODL);

    /* read commands from .csm file until the end of file */
    /*   note that we should not use feof here since we will
         prematurely exit loop if last line does not end with CR/LF */
    while (fp_csm != NULL) {

        /* read the next line */
        temp = fgets(templine, MAX_LINE_LEN, fp_csm);
        linenum++;

        /* we have found the end of the current file (without an end statement) */
        if (temp == NULL) {

            /* in a UDC */
            if (filetype == 2) {
                SPLINT_CHECK_FOR_NULL(MODL->brch);

                /* if there is no END, add it now */
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_END, filename, linenum,
                                         NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                    CHECK_STATUS2(ocsmNewBrch);

                /* decrement the level */
                (MODL->level)--;

                SPRINT0(1, "\n<<< Reverting to parent file (EOF detected)\n");
            }

            /* close the file and exit */
            fclose(fp_csm);
            fp_csm = NULL;

            break;
        }

        /* ignore spaces at the beginning of the line */
        for (j = 0; j < STRLEN(templine); j++) {
            if (templine[0] == ' ') {
                for (k = 0; k < STRLEN(templine); k++) {
                    templine[k] = templine[k+1];
                }
            }
        }

        /* make sure that there are no | characters */
        for (j = 0; j < STRLEN(templine); j++) {
            if (templine[j] == '|') {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "the bar character (|) cannot be used");
                goto cleanup;
            }
        }

        /* process the next line:
           - ignore anything after a #
           - ignore anything after a \ and concatenate next line
           - remove any spaces between double quotes (") so that a user
             could include spaces within an expression as long as
             it is surrounded by double quotes */
        if (templine[0] != '#') {
            SPLINT_CHECK_FOR_NULL(fp_csm);

            inquote = 0;
            j       = 0;
            for (i = 0; i < STRLEN(templine); i++) {
                if (templine[i] == '#') {
                    nextline[j++] = '\0';
                    break;
                } else if (templine[i] == '\\') {
                    linenum++;
                    if (fgets(templine, MAX_LINE_LEN, fp_csm) == NULL) break;
                    i = -1;
                } else if (templine[i] == '"') {
                    inquote = 1 - inquote;
                } else if (templine[i] == ' '  || templine[i] == '\t' ||
                           templine[i] == '\r' || templine[i] == '\n'   ) {
                    if (inquote == 0 && j > 0) {
                        nextline[j++] = templine[i];
                    }
                } else {
                    nextline[j++] = templine[i];
                }

                if (j >= MAX_LINE_LEN-2) {
                    status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                          "input line is too long");
                    goto cleanup;
                }
            }
        } else {
            STRNCPY(nextline, templine, MAX_LINE_LEN);
            j = STRLEN(nextline);
        }
        nextline[j++] = '\0';

        /* strip white spaces from end of nextline */
        for (i = j-2; i >= 0; i--) {
            if (nextline[i] == ' '  || nextline[i] == '\t' ||
                nextline[i] == '\r' || nextline[i] == '\n'   ) {
                nextline[i] = '\0';
            } else {
                break;
            }
        }
        SPRINT4(1, "    nextline (%2d:%4d)[%4d]: %s", MODL->level, linenum, MODL->ibrch+1, nextline);
        if (STRLEN(nextline) <= 1) continue;

        /* get the command from the next input line */
        sscanf(nextline, "%2047s", command);

        /* input is: "# comment" */
        if (strncmp(command, "#", 1) == 0) {
            /* nothing to do */

        /* input is: "applycsys $csysName ibody=0" */
        } else if (strcmp(command, "applycsys") == 0 ||
                   strcmp(command, "APPLYCSYS") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "APPLYCSYS cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "APPLYCSYS cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s\n",
                          &(str1[1]), str2);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "APPLYCSYS requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_APPLYCSYS, filename, linenum,
                                 str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "arc xend yend zend dist $plane=xy" */
        } else if (strcmp(command, "arc") == 0 ||
                   strcmp(command, "ARC") == 0   ) {
            if (nskpt == 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_NOT_OPEN, filename, linenum,
                                      "ARC must be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "ARC cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str5, "$");
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2046s\n",
                          str1, str2, str3, str4, &(str5[1]));
            if (narg == 4) {
                STRNCPY(str5, "$xy", MAX_EXPR_LEN);
            } else if (narg == 5) {
                if        (strcmp(str5, "$xy") == 0 || strcmp(str5, "$XY") == 0) {
                    STRNCPY(str5, "$xy", MAX_EXPR_LEN);
                } else if (strcmp(str5, "$yz") == 0 || strcmp(str5, "$YZ") == 0) {
                    STRNCPY(str5, "$yz", MAX_EXPR_LEN);
                } else if (strcmp(str5, "$zx") == 0 || strcmp(str5, "$ZX") == 0) {
                    STRNCPY(str5, "$zx", MAX_EXPR_LEN);
                } else {
                    status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                          "PLANE must be XY, YZ, or ZX");
                    goto cleanup;
                }
            } else {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "ARC requires at least 4 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_ARC, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* increment the number of Sketch points */
            nskpt++; nskpt++;

        /* input is: "assert arg1 arg2 toler=0 verify=0" */
        } else if (strcmp(command, "assert") == 0 ||
                   strcmp(command, "ASSERT") == 0   ) {

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4);
            if (narg < 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "ASSERT requires at least 2 arguments");
                goto cleanup;
            }
            if (narg < 3) {
                strcpy(str3, "0");
            }
            if (narg < 4) {
                strcpy(str4, "0");
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_ASSERT, filename, linenum,
                                 str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "attribute $attrName attrList */
        } else if (strcmp(command, "attribute") == 0 ||
                   strcmp(command, "ATTRIBUTE") == 0   ) {

            /* determine which Branch should be attributed, or if it is
               a global Attribute (ibrch=0) */
            ibrch = 0;
            for (jbrch = MODL->ibrch; jbrch > 0; jbrch--) {
                SPLINT_CHECK_FOR_NULL(MODL->brch);

                if (MODL->brch[jbrch].type != OCSM_SET      &&
                    MODL->brch[jbrch].type != OCSM_DIMENSION  ) {
                    ibrch = jbrch;
                    break;
                }
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s\n",
                          str1, str2);
            if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "ATTRIBUTE requires 2 arguments");
                goto cleanup;
            }

            /* set the Branch's Attribute */
            if        (ibrch == 0) {
                status = ocsmSetAttr(MODL, ibrch, str1, str2);
                CHECK_STATUS2(ocsmSetAttr);
            } else {
                SPLINT_CHECK_FOR_NULL(MODL->brch);

                if (MODL->brch[ibrch].type == OCSM_ASSERT    ||
                    MODL->brch[ibrch].type == OCSM_CATBEG    ||
                    MODL->brch[ibrch].type == OCSM_CATEND    ||
                    MODL->brch[ibrch].type == OCSM_DIMENSION ||
                    MODL->brch[ibrch].type == OCSM_ELSE      ||
                    MODL->brch[ibrch].type == OCSM_ELSEIF    ||
                    MODL->brch[ibrch].type == OCSM_END       ||
                    MODL->brch[ibrch].type == OCSM_ENDIF     ||
                    MODL->brch[ibrch].type == OCSM_IFTHEN    ||
                    MODL->brch[ibrch].type == OCSM_INTERFACE ||
//                  MODL->brch[ibrch].type == OCSM_LBOUND    ||
                    MODL->brch[ibrch].type == OCSM_MACBEG    ||
                    MODL->brch[ibrch].type == OCSM_MACEND    ||
                    MODL->brch[ibrch].type == OCSM_MARK      ||
                    MODL->brch[ibrch].type == OCSM_PATBEG    ||
                    MODL->brch[ibrch].type == OCSM_PATBREAK  ||
                    MODL->brch[ibrch].type == OCSM_PATEND    ||
                    MODL->brch[ibrch].type == OCSM_PREFIX    ||
                    MODL->brch[ibrch].type == OCSM_PROJECT   ||
//                  MODL->brch[ibrch].type == OCSM_SELECT    ||
                    MODL->brch[ibrch].type == OCSM_SET       ||
                    MODL->brch[ibrch].type == OCSM_EVALUATE  ||
                    MODL->brch[ibrch].type == OCSM_GETATTR   ||
                    MODL->brch[ibrch].type == OCSM_SKBEG     ||
                    MODL->brch[ibrch].type == OCSM_SKCON     ||
                    MODL->brch[ibrch].type == OCSM_SKVAR     ||
                    MODL->brch[ibrch].type == OCSM_SOLBEG    ||
                    MODL->brch[ibrch].type == OCSM_SOLEND    ||
                    MODL->brch[ibrch].type == OCSM_STORE     ||
                    MODL->brch[ibrch].type == OCSM_MESSAGE   ||
                    MODL->brch[ibrch].type == OCSM_SPECIAL   ||
                    MODL->brch[ibrch].type == OCSM_THROW     ||
//                  MODL->brch[ibrch].type == OCSM_UBOUND    ||
                    MODL->brch[ibrch].type == OCSM_UDPARG      ) {
                    status = signalError2(MODL, OCSM_ILLEGAL_ATTRIBUTE, filename, linenum,
                                          "a \"%s\" Branch cannot be attributed", ocsmGetText(MODL->brch[ibrch].type));
                    goto cleanup;
                } else {
                    status = ocsmSetAttr(MODL, ibrch, str1, str2);
                    CHECK_STATUS2(ocsmSetAttr);
                }
            }

        /* input is: "bezier x y z" */
        } else if (strcmp(command, "bezier") == 0 ||
                   strcmp(command, "BEZIER") == 0   ) {
            if (nskpt == 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_NOT_OPEN, filename, linenum,
                                      "BEZIER must be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "BEZIER cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg != 3) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "BEZIRE requires 3 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_BEZIER, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* increment the number of Sketch points */
            nskpt++;

        /* input is: "blend begList=0 endList=0 reorder=0 oneFace=0 periodic=0 copyAttr=0" */
        } else if (strcmp(command, "blend") == 0 ||
                   strcmp(command, "BLEND") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "BLEND cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "BLEND cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4, str5, str6);
            if   (narg < 1) {
                STRNCPY(str1, "0", MAX_EXPR_LEN);
            }
            if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            }
            if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }
            if (narg < 4) {
                STRNCPY(str4, "0", MAX_EXPR_LEN);
            }
            if (narg < 5) {
                STRNCPY(str5, "0", MAX_EXPR_LEN);
            }
            if (narg < 6) {
                STRNCPY(str6, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_BLEND, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "box xbase ybase zbase dx dy dz" */
        } else if (strcmp(command, "box") == 0 ||
                   strcmp(command, "BOX") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "BOX cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "BOX cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4, str5, str6);
            if (narg != 6) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "BOX requires 6 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_BOX, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "catbeg sigCode" */
        } else if (strcmp(command, "catbeg") == 0 ||
                   strcmp(command, "CATBEG") == 0   ) {

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "CATBEG requires 1 argument");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_CATBEG, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "catend" */
        } else if (strcmp(command, "catend") == 0 ||
                   strcmp(command, "CATEND") == 0   ) {

            /* find the matching catbeg */
            i = 1;
            for (ibrch = MODL->nbrch; ibrch > 0; ibrch--) {
                SPLINT_CHECK_FOR_NULL(MODL->brch);

                if        (MODL->brch[ibrch].type == OCSM_CATEND) {
                    i++;
                } else if (MODL->brch[ibrch].type == OCSM_CATBEG) {
                    i--;
                    if (i == 0) {
                        break;
                    }
                }
            }
            if (ibrch <= 0) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "CATEND must follow a CATBEG");
                goto cleanup;
            }

            /* no arguments */

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_CATEND, filename, linenum,
                                 NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            npatn--;

        /* input is: "cfgpmtr $pmtrName value" */
        } else if (strcmp(command, "cfgpmtr") == 0 ||
                   strcmp(command, "CFGPMTR") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "CFGPMTR cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "CFGPMTR cannot be in SOLBEG/SOLEND");
                goto cleanup;
            } else if (MODL->scope[MODL->level] > 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "CFGPMTR not allowed except at top level");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s\n",
                          str1, str2);
            if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "CFGPMTR requires 2 arguments");
                goto cleanup;
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[0] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /* make sure that values is a semicolon-separated list of numbers */
            icount = 0;
            while (icount < STRLEN(str2)) {
                jcount = 0;

                while (icount < STRLEN(str2)) {
                    if (str2[icount] == ';') {
                        icount++;
                        break;
                    } else {
                        defn[jcount  ] =str2[icount];
                        defn[jcount+1] = '\0';
                        icount++;
                        jcount++;
                    }
                }

                /* check for a valid number */
                if (sscanf(defn, "%lf", &value) == 0) {
                    status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                          "values must only contain numbers");
                    goto cleanup;
                }
            }

            /* break str1 into pmtrName[irow,icol] */
            status = parseName(MODL, str1, pmtrName, &ipmtr, &irow, &icol);
            CHECK_STATUS2(parseName);

            /* if it does not exist, create it now */
            if (ipmtr == 0) {
                status = ocsmNewPmtr(MODL, pmtrName, OCSM_CFGPMTR, 1, 1);
                CHECK_STATUS2(ocsmNewPmtr);
                ipmtr = MODL->npmtr;
            } else {
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                /* if it exists and is UNKNOWN (because of a DIMENSION statemet),
                   convert to a CFGPMTR */
                if (MODL->pmtr[ipmtr].type == OCSM_UNKNOWN) {
                    MODL->pmtr[ipmtr].type = OCSM_CFGPMTR;

                    /* make sure that Parameter is CFGPMTR */
                } else if (MODL->pmtr[ipmtr].type == OCSM_LOCALVAR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_LOCALVAR, filename, linenum,
                                          "%s is an internal parameter", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_OUTPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_OUTPMTR, filename, linenum,
                                          "%s is an OUTPMTR", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_CONPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_CONPMTR, filename, linenum,
                                          "%s is a CONPMTR", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_DESPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_DESPMTR, filename, linenum,
                                          "%s is a CFGPMTR", str1);
                    goto cleanup;
                }
            }

            /* store the values for the whole Parameter */
            if ((irow ==    0 && icol ==    0) ||
                (irow == -999 && icol == -999)   ) {
                icount = 0;
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                    for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                        jcount = 0;

                        while (icount < STRLEN(str2)) {
                            if (str2[icount] == ';') {
                                icount++;
                                break;
                            } else {
                                defn[jcount  ] = str2[icount];
                                defn[jcount+1] = '\0';
                                icount++;
                                jcount++;
                            }
                        }

                        status = ocsmGetValu(MODL, ipmtr, irow, icol, &value, &dot);
                        CHECK_STATUS2(ocsmGetValu);

                        if (fabs(value+HUGEQ) > 1 && fabs(dot+HUGEQ) > 1) {
                            SPRINT0(1, "WARNING:: value not overwritten");
                            (MODL->nwarn)++;
                        } else {
                            if (strcmp(defn, "") != 0) {
                                status = ocsmSetValu(MODL, ipmtr, irow, icol, defn);
                                CHECK_STATUS2(ocsmSetValu);
                            }
                        }
                    }
                }

            /* store the values for all rows of the Parameter */
            } else if (irow == -999) {
                icount = 0;
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                    jcount = 0;

                    while (icount < STRLEN(str2)) {
                        if (str2[icount] == ';') {
                            icount++;
                            break;
                        } else {
                            defn[jcount  ] = str2[icount];
                            defn[jcount+1] = '\0';
                            icount++;
                            jcount++;
                        }
                    }

                    status = ocsmGetValu(MODL, ipmtr, irow, icol, &value, &dot);
                    CHECK_STATUS2(ocsmGetValu);

                    if (fabs(value+HUGEQ) > 1 && fabs(dot+HUGEQ) > 1) {
                        SPRINT0(1, "WARNING:: value not overwritten");
                        (MODL->nwarn)++;
                    } else {
                        if (strcmp(defn, "") != 0) {
                            status = ocsmSetValu(MODL, ipmtr, irow, icol, defn);
                            CHECK_STATUS2(ocsmSetValu);
                        }
                    }
                }

            /* store the values for all columns of the Parameter */
            } else if (icol == -999) {
                icount = 0;
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                    jcount = 0;

                    while (icount < STRLEN(str2)) {
                        if (str2[icount] == ';') {
                            icount++;
                            break;
                        } else {
                            defn[jcount  ] = str2[icount];
                            defn[jcount+1] = '\0';
                            icount++;
                            jcount++;
                        }
                    }

                    status = ocsmGetValu(MODL, ipmtr, irow, icol, &value, &dot);
                    CHECK_STATUS2(ocsmGetValu);

                    if (fabs(value+HUGEQ) > 1 && fabs(dot+HUGEQ) > 1) {
                        SPRINT0(1, "WARNING:: value not overwritten");
                        (MODL->nwarn)++;
                    } else {
                        if (strcmp(defn, "") != 0) {
                            status = ocsmSetValu(MODL, ipmtr, irow, icol, defn);
                            CHECK_STATUS2(ocsmSetValu);
                        }
                    }
                }

            /* store a single value into the Parameter */
            } else {
                if (irow <= 0) irow = 1;
                if (icol <= 0) icol = 1;

                icount = 0;
                jcount = 0;

                while (icount < STRLEN(str2)) {
                    if (str2[icount] == ';') {
                        icount++;
                        break;
                    } else {
                        defn[jcount  ] = str2[icount];
                        defn[jcount+1] = '\0';
                        icount++;
                        jcount++;
                    }
                }

                status = ocsmGetValu(MODL, ipmtr, irow, icol, &value, &dot);
                CHECK_STATUS2(ocsmGetValu);

                if (fabs(value+HUGEQ) > 1 && fabs(dot+HUGEQ) > 1) {
                    SPRINT0(1, "WARNING:: value not overwritten");
                    (MODL->nwarn)++;
                } else {
                    if (strcmp(defn, "") != 0) {
                        status = ocsmSetValu(MODL, ipmtr, irow, icol, defn);
                        CHECK_STATUS2(ocsmSetValu);
                    }
                }
            }

        /* input is: "chamfer radius edgeList=0 listStyle=0" */
        } else if (strcmp(command, "chamfer") == 0 ||
                   strcmp(command, "CHAMFER") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "CHAMFER cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "CHAMFER cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if        (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "CHAMFER requires at least 1 argument");
                goto cleanup;
            } else if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            } else if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_CHAMFER, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "cirarc xon yon zon xend yend zend" */
        } else if (strcmp(command, "cirarc") == 0 ||
                   strcmp(command, "CIRARC") == 0   ) {
            if (nskpt == 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_NOT_OPEN, filename, linenum,
                                      "CIRARC msut be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "CIRARC cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4, str5, str6);

            if (narg != 6) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "CIRARC requires 6 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_CIRARC, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* increment the number of Sketch points */
            nskpt++; nskpt++;

        /* input is: "combine toler=0" */
        } else if (strcmp(command, "combine") == 0 ||
                   strcmp(command, "COMBINE") == 0   ) {

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg < 1) {
                STRNCPY(str1, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPRIM, filename, linenum,
                                 "$$$/combine", "$toler", str1, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* if in a .cpc file, increment the number of UDCs if str1 starts with "$/" or "$$" */
            if (filetype == 1) {
                if (str1[1] == '$' || str1[1] == '$') {
                    numudc++;
                }
            }

        /* input is: "elevate toler=0" */
        } else if (strcmp(command, "elevate") == 0 ||
                   strcmp(command, "ELEVATE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "ELEVATE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "ELEVATE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg < 1) {
                STRNCPY(str1, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_ELEVATE, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "cone xvrtx yvrtx zvrtx xbase ybase zbase radius" */
        } else if (strcmp(command, "cone") == 0 ||
                   strcmp(command, "CONE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "CONE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "CONE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4, str5, str6, str7);
            if (narg != 7) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "CONE requires 7 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_CONE, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "connect faceList1 faceList2 edgeList1=0 edgeList2=0 toler=0" */
        } else if (strcmp(command, "connect") == 0 ||
                   strcmp(command, "CONNECT") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "CONNECT cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "CONNECT cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4, str5);
            if (narg < 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "CONNECT requires 2 arguments");
                goto cleanup;
            }
            if (narg < 3) {
                strcpy(str3, "0");
            }
            if (narg < 4) {
                strcpy(str4, "0");
            }
            if (narg < 5) {
                strcpy(str5, "0");
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_CONNECT, filename, linenum,
                                 str1, str2, str3, str4, str5, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "conpmtr $pmtrName values" */
        } else if (strcmp(command, "conpmtr") == 0 ||
                   strcmp(command, "CONPMTR") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "CONPMTR cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "CONPMTR cannot be in SOLBEG/SOLEND");
                goto cleanup;
            } else if (MODL->scope[MODL->level] > 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "CONPMTR not allowed except at top level");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s\n",
                          str1, str2);
            if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "CONPMTR requires 2 arguments");
                goto cleanup;
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[0] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /* check if pmtrName already exists, and if it does
               ensure that it is a CONPMTR parameter */
            ipmtr = -1;
            for (jpmtr = 1; jpmtr <= MODL->npmtr; jpmtr++) {
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                if (strcmp(MODL->pmtr[jpmtr].name, &(str1[1])) == 0) {

                    if (MODL->pmtr[jpmtr].type == OCSM_CONPMTR) {
                        ipmtr = jpmtr;
                        break;
                    } else if (MODL->pmtr[jpmtr].type == OCSM_LOCALVAR) {
                        status = signalError2(MODL, OCSM_PMTR_IS_LOCALVAR, filename, linenum,
                                              "%s is an internal parameter", str1);
                        goto cleanup;
                    } else if (MODL->pmtr[jpmtr].type == OCSM_OUTPMTR) {
                        status = signalError2(MODL, OCSM_PMTR_IS_OUTPMTR, filename, linenum,
                                              "%s is a OUTPMTR", str1);
                        goto cleanup;
                    } else if (MODL->pmtr[jpmtr].type == OCSM_DESPMTR ||
                               MODL->pmtr[jpmtr].type == OCSM_CFGPMTR   ) {
                        status = signalError2(MODL, OCSM_PMTR_IS_DESPMTR, filename, linenum,
                                              "%s is a CFGPMTR or CFGPMTR", str1);
                        goto cleanup;
                    }
                }
            }

            /* find the number of entries in the values argument */
            nval = 1;
            for (i = 1; i < STRLEN(str2)-1; i++) {
                if (str2[i] == ';') nval++;
            }

            /* create the Parameter (if it does not already exist) */
            if (ipmtr < 0) {
                status = ocsmNewPmtr(MODL, str1, OCSM_CONPMTR, 1, nval);
                CHECK_STATUS2(ocsmNewPmtr);

                ipmtr = MODL->npmtr;
            }

            /* store the values */
            i = 0;
            for (ival = 1; ival <= nval; ival++) {
                j = 0;
                while (i <= STRLEN(str2)) {
                    if (str2[i] == ';' || i == STRLEN(str2)) {
                        str3[j] = '\0';

                        status = str2val(str3, NULL, &value, &dot, str);
                        CHECK_STATUS2(str2val);
                        if (STRLEN(str) > 0) {
                            status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                                  "expression must evaluate to a number");
                            goto cleanup;
                        } else {
                            SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                            if (fabs(MODL->pmtr[ipmtr].value[ival-1]+HUGEQ) < EPS06) {
                                status = ocsmSetValu(MODL, ipmtr, 1, ival, str3);
                                CHECK_STATUS2(ocsmSetValu);
                            } else if (fabs(MODL->pmtr[ipmtr].value[ival-1]-value) < EPS06) {
                                /* value already defined */
                            } else {
                                /* already defined with a different value */
                                status = signalError2(MODL, OCSM_NAME_ALREADY_DEFINED, filename, linenum,
                                                      "%s is already defined with a different value", str1);
                                goto cleanup;
                            }
                        }

                        i++;
                        break;
                    } else {
                        str3[j++] = str2[i++];
                    }
                }
            }

        /* input is: "csystem $csysName csysList */
        } else if (strcmp(command, "csystem") == 0 ||
                   strcmp(command, "CSYSTEM") == 0   ) {

            SPLINT_CHECK_FOR_NULL(MODL->brch);

            /* determine which Branch should be attributed, or if it is
               a global Attribute (ibrch=0) */
            ibrch = 0;
            for (jbrch = MODL->ibrch; jbrch > 0; jbrch--) {
                if (MODL->brch[jbrch].type != OCSM_SET      &&
                    MODL->brch[jbrch].type != OCSM_DIMENSION  ) {
                    ibrch = jbrch;
                    break;
                }
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s\n",
                          str1, str2);
            if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "CSYSTEM requires 2 arguments");
                goto cleanup;
            }

            /* set the Branch's csystem */
            if (MODL->brch[ibrch].type == OCSM_ASSERT    ||
                MODL->brch[ibrch].type == OCSM_CATBEG    ||
                MODL->brch[ibrch].type == OCSM_CATEND    ||
                MODL->brch[ibrch].type == OCSM_DIMENSION ||
                MODL->brch[ibrch].type == OCSM_ELSE      ||
                MODL->brch[ibrch].type == OCSM_ELSEIF    ||
                MODL->brch[ibrch].type == OCSM_END       ||
                MODL->brch[ibrch].type == OCSM_ENDIF     ||
                MODL->brch[ibrch].type == OCSM_IFTHEN    ||
                MODL->brch[ibrch].type == OCSM_INTERFACE ||
//              MODL->brch[ibrch].type == OCSM_LBOUND    ||
                MODL->brch[ibrch].type == OCSM_MACBEG    ||
                MODL->brch[ibrch].type == OCSM_MACEND    ||
                MODL->brch[ibrch].type == OCSM_MARK      ||
                MODL->brch[ibrch].type == OCSM_PATBEG    ||
                MODL->brch[ibrch].type == OCSM_PATBREAK  ||
                MODL->brch[ibrch].type == OCSM_PATEND    ||
                MODL->brch[ibrch].type == OCSM_PROJECT   ||
//              MODL->brch[ibrch].type == OCSM_SELECT    ||
                MODL->brch[ibrch].type == OCSM_SET       ||
                MODL->brch[ibrch].type == OCSM_EVALUATE  ||
                MODL->brch[ibrch].type == OCSM_GETATTR   ||
                MODL->brch[ibrch].type == OCSM_SKBEG     ||
                MODL->brch[ibrch].type == OCSM_SKCON     ||
                MODL->brch[ibrch].type == OCSM_SKVAR     ||
                MODL->brch[ibrch].type == OCSM_SOLBEG    ||
                MODL->brch[ibrch].type == OCSM_SOLEND    ||
                MODL->brch[ibrch].type == OCSM_STORE     ||
                MODL->brch[ibrch].type == OCSM_MESSAGE   ||
                MODL->brch[ibrch].type == OCSM_SPECIAL   ||
                MODL->brch[ibrch].type == OCSM_THROW     ||
//              MODL->brch[ibrdh].type == OCSM_UBOUND    ||
                MODL->brch[ibrch].type == OCSM_UDPARG      ) {
                status = signalError2(MODL, OCSM_ILLEGAL_CSYSTEM, filename, linenum,
                                      "a \"%s\" Branch cannot get Csystem", ocsmGetText(MODL->brch[ibrch].type));
                goto cleanup;
            } else {
                status = ocsmSetCsys(MODL, ibrch, str1, str2);
                CHECK_STATUS2(ocsmSetCsys);
            }

        /* input is: "cylinder xbeg ybeg zbeg xend yend zend radius" */
        } else if (strcmp(command, "cylinder") == 0 ||
                   strcmp(command, "CYLINDER") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "CYLINDER cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "CYLINDER cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4, str5, str6, str7);
            if (narg != 7) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "CYLINDER requires 7 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_CYLINDER, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "despmtr $pmtrName values" */
        } else if (strcmp(command, "despmtr") == 0 ||
                   strcmp(command, "DESPMTR") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "DESPMTR cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "DESPMTR cannot be in SOLBEG/SOLEND");
                goto cleanup;
            } else if (MODL->scope[MODL->level] > 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "DESPMTR not allowed except at top level");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s\n",
                          str1, str2);
            if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "DESPMTR requires 2 arguments");
                goto cleanup;
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[0] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /* make sure that values is a semicolon-separated list of numbers */
            icount = 0;
            while (icount < STRLEN(str2)) {
                jcount = 0;

                while (icount < STRLEN(str2)) {
                    if (str2[icount] == ';') {
                        icount++;
                        break;
                    } else {
                        defn[jcount  ] =str2[icount];
                        defn[jcount+1] = '\0';
                        icount++;
                        jcount++;
                    }
                }

                /* check for a valid number */
                if (sscanf(defn, "%lf", &value) == 0) {
                    status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                          "values must only contain numbers");
                    goto cleanup;
                }
            }

            /* break str1 into pmtrName[irow,icol] */
            status = parseName(MODL, str1, pmtrName, &ipmtr, &irow, &icol);
            CHECK_STATUS2(parseName);

            /* if it does not exist, create it now */
            if (ipmtr == 0) {
                status = ocsmNewPmtr(MODL, pmtrName, OCSM_DESPMTR, 1, 1);
                CHECK_STATUS2(ocsmNewPmtr);
                ipmtr = MODL->npmtr;
            } else {
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                /* if it exists and is UNKNOWN (because of a DIMENSION statement),
                   convert to a DESPMTR */
                if (MODL->pmtr[ipmtr].type == OCSM_UNKNOWN) {
                    MODL->pmtr[ipmtr].type = OCSM_DESPMTR;

                /* make sure that Parameter is DESPMTR */
                } else if (MODL->pmtr[ipmtr].type == OCSM_LOCALVAR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_LOCALVAR, filename, linenum,
                                          "%s is an internal parameter", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_OUTPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_OUTPMTR, filename, linenum,
                                          "%s is an OUTPMTR", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_CONPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_CONPMTR, filename, linenum,
                                          "%s is a CONPMTR", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_CFGPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_DESPMTR, filename, linenum,     // OCSM_PMTR_IS_CFGPMTR is not defined
                                          "%s is a CFGPMTR", str1);
                    goto cleanup;
                }
            }

            /* store the values for the whole Parameter */
            if ((irow ==    0 && icol ==    0) ||
                (irow == -999 && icol == -999)   ) {
                icount = 0;
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                    for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                        jcount = 0;

                        while (icount < STRLEN(str2)) {
                            if (str2[icount] == ';') {
                                icount++;
                                break;
                            } else {
                                defn[jcount  ] = str2[icount];
                                defn[jcount+1] = '\0';
                                icount++;
                                jcount++;
                            }
                        }

                        status = ocsmGetValu(MODL, ipmtr, irow, icol, &value, &dot);
                        CHECK_STATUS2(ocsmGetValu);

                        if (fabs(value+HUGEQ) > 1 && fabs(dot+HUGEQ) > 1) {
                            SPRINT0(1, "WARNING:: value not overwritten");
                            (MODL->nwarn)++;
                        } else {
                            if (strcmp(defn, "") != 0) {
                                status = ocsmSetValu(MODL, ipmtr, irow, icol, defn);
                                CHECK_STATUS2(ocsmSetValu);
                            }
                        }
                    }
                }

            /* store the values for all rows of the Parameter */
            } else if (irow == -999) {
                icount = 0;
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                    jcount = 0;

                    while (icount < STRLEN(str2)) {
                        if (str2[icount] == ';') {
                            icount++;
                            break;
                        } else {
                            defn[jcount  ] = str2[icount];
                            defn[jcount+1] = '\0';
                            icount++;
                            jcount++;
                        }
                    }

                    status = ocsmGetValu(MODL, ipmtr, irow, icol, &value, &dot);
                    CHECK_STATUS2(ocsmGetValu);

                    if (fabs(value+HUGEQ) > 1 && fabs(dot+HUGEQ) > 1) {
                        SPRINT0(1, "WARNING:: value not overwritten");
                        (MODL->nwarn)++;
                    } else {
                        if (strcmp(defn, "") != 0) {
                            status = ocsmSetValu(MODL, ipmtr, irow, icol, defn);
                            CHECK_STATUS2(ocsmSetValu);
                        }
                    }
                }

            /* store the values for all columns of the Parameter */
            } else if (icol == -999) {
                icount = 0;
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                    jcount = 0;

                    while (icount < STRLEN(str2)) {
                        if (str2[icount] == ';') {
                            icount++;
                            break;
                        } else {
                            defn[jcount  ] = str2[icount];
                            defn[jcount+1] = '\0';
                            icount++;
                            jcount++;
                        }
                    }

                    status = ocsmGetValu(MODL, ipmtr, irow, icol, &value, &dot);
                    CHECK_STATUS2(ocsmGetValu);

                    if (fabs(value+HUGEQ) > 1 && fabs(dot+HUGEQ) > 1) {
                        SPRINT0(1, "WARNING:: value not overwritten");
                        (MODL->nwarn)++;
                    } else {
                        if (strcmp(defn, "") != 0) {
                            status = ocsmSetValu(MODL, ipmtr, irow, icol, defn);
                            CHECK_STATUS2(ocsmSetValu);
                        }
                    }
                }

            /* store a single value into the Parameter */
            } else {
                if (irow <= 0) irow = 1;
                if (icol <= 0) icol = 1;

                icount = 0;
                jcount = 0;

                while (icount < STRLEN(str2)) {
                    if (str2[icount] == ';') {
                        icount++;
                        break;
                    } else {
                        defn[jcount  ] = str2[icount];
                        defn[jcount+1] = '\0';
                        icount++;
                        jcount++;
                    }
                }

                status = ocsmGetValu(MODL, ipmtr, irow, icol, &value, &dot);
                CHECK_STATUS2(ocsmGetValu);

                if (fabs(value+HUGEQ) > 1 && fabs(dot+HUGEQ) > 1) {
                    SPRINT0(1, "WARNING:: value not overwritten");
                    (MODL->nwarn)++;
                } else {
                    if (strcmp(defn, "") != 0) {
                        status = ocsmSetValu(MODL, ipmtr, irow, icol, defn);
                        CHECK_STATUS2(ocsmSetValu);
                    }
                }
            }

        /* input is: "dimension $pmtrName nrow ncol=1" */
        } else if (strcmp(command, "dimension") == 0 ||
                   strcmp(command, "DIMENSION") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "DIMENSION cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "DIMENSION cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s\n",
                          &(str1[1]), str2, str3);
            if (narg < 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "DIMENSION requires at least 2 arguments");
                goto cleanup;
            } else if (narg == 2) {
                STRNCPY(str3, "1", MAX_EXPR_LEN);
            } else if (narg >= 4) {
                SPRINT0(1, "WARNING:: despmtr argument is obsolete and will be ignored");
                (MODL->nwarn)++;
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[1] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /*  create a Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_DIMENSION, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* create an UNKNOWN Parameter if one does not already exist and str2 and str3
               both can be evaluated to a number

               notes:  1. the type will get changed from UNKNOWN to DESPMTR by a DESPMTR statement
                       2. the type will get changed from UNKNOWN to CFGPMTR by a CFGPMTR statement
                       3. the type will get changed from UNKNOWN to OUTPMTR by a OUTPMTR statement
                       4. all UNKNOWNs will be deleted at the end of ocsmLoad
            */
            ipmtr = -1;

            if (strncmp(str1, "$:", 2) != 0 || STRLEN(MODL->prefix) == 0) {
                for (jpmtr = 1; jpmtr <= MODL->npmtr; jpmtr++) {
                    SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                    if (strcmp(&str1[1], MODL->pmtr[jpmtr].name) == 0) {
                        ipmtr = jpmtr;
                        break;
                    }
                }
            } else {
                for (jpmtr = 1; jpmtr <= MODL->npmtr; jpmtr++) {
                    SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                    i = STRLEN(MODL->prefix);
                    j = STRLEN(MODL->pmtr[jpmtr].name);
                    if (strncmp(&str1[1  ], MODL->prefix,           i) == 0 &&
                        strncmp(&str1[1+i], MODL->pmtr[jpmtr].name, j) == 0) {
                        ipmtr = jpmtr;
                        break;
                    }
                }
            }

            if (ipmtr < 0) {
                status = str2valNoSignal(str2, MODL, &value, &dot, str);
                if (status == SUCCESS && strlen(str) == 0) {
                    nrow = NINT(value);
                    if (nrow <= 0) {
                        status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                              "nrow is not positive");
                        goto cleanup;
                    }
                } else {
                    nrow = -1;
                }

                status = str2valNoSignal(str3, MODL, &value, &dot, str);
                if (status == SUCCESS && strlen(str) == 0) {
                    ncol = NINT(value);
                    if (ncol <= 0) {
                        status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                              "ncol is not positive");
                        goto cleanup;
                    }
                } else {
                    ncol = -1;
                }

                status = SUCCESS;

                if (nrow > 0 && ncol > 0) {
                    status = ocsmNewPmtr(MODL, &(str1[1]), OCSM_UNKNOWN, nrow, ncol);
                    CHECK_STATUS2(ocsmNewPmtr);
                }
            }

        /* input is: "dump $filename remove=0 toMark=0 withTess=0 $grpName=. putColors=0" */
        } else if (strcmp(command, "dump") == 0 ||
                   strcmp(command, "DUMP") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "DUMP cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "DUMP cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            strcpy(str5, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s %2047s %2046s %2047s\n",
                          &(str1[1]), str2, str3, str4, &(str5[1]), str6);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "DUMP requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            }
            if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }
            if (narg < 4) {
                STRNCPY(str4, "0", MAX_EXPR_LEN);
            }
            if (narg < 5) {
                STRNCPY(str5, "$.", MAX_EXPR_LEN);
            }
            if (narg < 6) {
                STRNCPY(str6, "0", MAX_EXPR_LEN);
            }

            /* convert filename if it starts with "$/" to relative pathname */
            if (strncmp(str1, "$$/", 3) == 0) {
                STRNCPY(pathname, filename, MAX_EXPR_LEN);
                i = STRLEN(pathname) - 1;
                while (pathname[i] != SLASH) {
                    pathname[i] = '\0';
                    i--;
                    if (i < 0) break;
                }
                STRNCPY(tmpfilename, &(str1[3]), MAX_EXPR_LEN);
                snprintf(str1, MAX_EXPR_LEN, "$%s%s", pathname, tmpfilename);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_DUMP, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "else" */
        } else if (strcmp(command, "else") == 0 ||
                   strcmp(command, "ELSE") == 0   ) {

            /* find the matching ifthen */
            i = 0;
            for (ibrch = MODL->nbrch; ibrch > 0; ibrch--) {
                SPLINT_CHECK_FOR_NULL(MODL->brch);

                if        (MODL->brch[ibrch].type == OCSM_ENDIF ) {
                    i++;
                } else if (MODL->brch[ibrch].type == OCSM_IFTHEN) {
                    i--;
                }
            }
            if (i >= 0) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "ELSE must follow IFTHEN");
                goto cleanup;
            }

            /* no arguments */

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_ELSE, filename, linenum,
                                 NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "elseif val1 $op1=ne val2=0 $op2=and val3=0 $op3=eq val4=0" */
        } else if (strcmp(command, "elseif") == 0 ||
                   strcmp(command, "ELSEIF") == 0   ) {

            /* find the matching ifthen */
            i = 0;
            for (ibrch = MODL->nbrch; ibrch > 0; ibrch--) {
                SPLINT_CHECK_FOR_NULL(MODL->brch);

                if        (MODL->brch[ibrch].type == OCSM_ENDIF ) {
                    i++;
                } else if (MODL->brch[ibrch].type == OCSM_IFTHEN) {
                    i--;
                }
            }
            if (i >= 0) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "ELSEIF must follow IFTHEN");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str2, "$");
            strcpy(str4, "$");
            strcpy(str6, "$");
            strcpy(str8, "$");
            narg = sscanf(nextline, "%*s %2047s %2046s %2047s %2046s %2047s %2046s %2047s\n",
                          str1, &(str2[1]), str3, &(str4[1]), str5, &(str6[1]), str7);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "ELSEIF requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                strcpy(str2, "$ne");
            }
            if (narg < 3) {
                strcpy(str3, "0");
            }
            if (narg < 4) {
                strcpy(str4, "$and");
            }
            if (narg < 5) {
                strcpy(str5, "0");
            }
            if (narg < 6) {
                strcpy(str6, "$eq");
            }
            if (narg < 7) {
                strcpy(str7, "0");
            }

            if (strcmp(str2, "$lt") != 0 && strcmp(str2, "$LT") != 0 &&
                strcmp(str2, "$le") != 0 && strcmp(str2, "$LE") != 0 &&
                strcmp(str2, "$eq") != 0 && strcmp(str2, "$EQ") != 0 &&
                strcmp(str2, "$ge") != 0 && strcmp(str2, "$GE") != 0 &&
                strcmp(str2, "$gt") != 0 && strcmp(str2, "$GT") != 0 &&
                strcmp(str2, "$ne") != 0 && strcmp(str2, "$NE") != 0   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "op1 must be LT, LE, EQ, GE, GT, or NE");
                goto cleanup;
            }
            if (strcmp(str4, "$or" ) != 0 && strcmp(str4, "$OR" ) != 0 &&
                strcmp(str4, "$and") != 0 && strcmp(str4, "$AND") != 0 &&
                strcmp(str4, "$xor") != 0 && strcmp(str4, "$XOR") != 0   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "op2 must be OR, AND, or XOR");
                goto cleanup;
            }
            if (strcmp(str6, "$lt") != 0 && strcmp(str6, "$LT") != 0 &&
                strcmp(str6, "$le") != 0 && strcmp(str6, "$LE") != 0 &&
                strcmp(str6, "$eq") != 0 && strcmp(str6, "$EQ") != 0 &&
                strcmp(str6, "$ge") != 0 && strcmp(str6, "$GE") != 0 &&
                strcmp(str6, "$gt") != 0 && strcmp(str6, "$GT") != 0 &&
                strcmp(str6, "$ne") != 0 && strcmp(str6, "$NE") != 0   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "op3 must be LT, LE, EQ, GE, GT, or NE");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_ELSEIF, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "end" */
        } else if (strcmp(command, "end") == 0 ||
                   strcmp(command, "END") == 0   ) {

            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "END cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "END cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* if in a .csm file, we are done */
            if (filetype == 0) {
                SPLINT_CHECK_FOR_NULL(fp_csm);

                fclose(fp_csm);
                fp_csm = NULL;

                break;

            /* if reading .cpc file and there are one or more open UDCs, create a new Branch */
            } else if (filetype == 1 && numudc > 0) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_END, filename, linenum,
                                     NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                CHECK_STATUS2(ocsmNewBrch);

                numudc--;

            /* if reading .cpc file and there are no open UDCs, we are done */
            } else if (filetype == 1) {
                SPLINT_CHECK_FOR_NULL(fp_csm);

                fclose(fp_csm);
                fp_csm = NULL;

                break;

            /* if reading .udc file, add the END and we are done */
            } else if (filetype == 2) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_END, filename, linenum,
                                     NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                CHECK_STATUS2(ocsmNewBrch);

                (MODL->level)--;

                SPLINT_CHECK_FOR_NULL(fp_csm);

                SPRINT0(1, "\n<<< Reverting to parent file (END detected)\n");

                fclose(fp_csm);
                fp_csm = NULL;

                break;

            /* if reading a .vfy file, then we are done */
            } else if (filetype == 3) {
                SPLINT_CHECK_FOR_NULL(fp_csm);

                fclose(fp_csm);
                fp_csm = NULL;

                break;

            /* otherwise, close file and revert to previous level */
            } else {
                SPLINT_CHECK_FOR_NULL(fp_csm);

                fclose(fp_csm);
                fp_csm = NULL;

                /* create a new Branch and decrement the level */
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_END, filename, linenum,
                                     NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                CHECK_STATUS2(ocsmNewBrch);

                (MODL->level)--;

                SPRINT1(1, "\n<<< Reverting to parent file (END detected in filetype=%d)\n", filetype);
                break;
            }

        /* input is: "endif" */
        } else if (strcmp(command, "endif") == 0 ||
                   strcmp(command, "ENDIF") == 0   ) {

            /* find the matching ifthen */
            i = 1;
            for (ibrch = MODL->nbrch; ibrch > 0; ibrch--) {
                SPLINT_CHECK_FOR_NULL(MODL->brch);

                if        (MODL->brch[ibrch].type == OCSM_ENDIF ) {
                    i++;
                } else if (MODL->brch[ibrch].type == OCSM_IFTHEN) {
                    i--;
                    if (i == 0) {
                        break;
                    }
                }
            }
            if (ibrch <= 0) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "ENDIF must follow IFTHEN");
                goto cleanup;
            }

            /* no arguments */

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_ENDIF, filename, linenum,
                                 NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            npatn--;

        /* input is: "evaluate $type ..." */
        } else if (strcmp(command, "evaluate") == 0 ||
                   strcmp(command, "EVALUATE") == 0   ) {

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s %2047s %2047s %2047s\n",
                          &(str1[1]), str2, str3, str4, str5, str6);

            /* $node ibody inode */
            if        (strcmp(str1, "$node") == 0 ||
                       strcmp(str1, "$NODE") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE NODE requires ibody inode");
                    goto cleanup;
                }

            /* $edge ibody iedge t */
            } else if (strcmp(str1, "$edge") == 0 ||
                       strcmp(str1, "$EDGE") == 0   ) {
                if (narg != 4) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE EDGE requires ibody iedge t");
                    goto cleanup;
                }

            /* $edgebbox ibody iedge */
            } else if (strcmp(str1, "$edgebbox") == 0 ||
                       strcmp(str1, "$EDGEBBOX") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE EDGEBBOX requires ibody iedge");
                    goto cleanup;
                }

            /* $edgerng ibody iedge */
            } else if (strcmp(str1, "$edgerng") == 0 ||
                       strcmp(str1, "$EDGERNG") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE EDGERNG requires ibody iedge");
                    goto cleanup;
                }

            /* $edgeinv ibody iedge x y z */
            } else if (strcmp(str1, "$edgeinv") == 0 ||
                       strcmp(str1, "$EDGEINV") == 0   ) {
                if (narg != 6) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE EDGEINV requires ibody iedge x y z");
                    goto cleanup;
                }

            /* $edgekt ibody iedge*/
            } else if (strcmp(str1, "$edgekt") == 0 ||
                       strcmp(str1, "$EDGEKT") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE EDGEKT requires ibody iedge");
                    goto cleanup;
                }

            /* $edgecp ibody iedge */
            } else if (strcmp(str1, "$edgecp") == 0 ||
                       strcmp(str1, "$EDGECP") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE EDGECP requires ibody iedge");
                    goto cleanup;
                }

            /* $edgetess ibody iedge */
            } else if (strcmp(str1, "$edgetess") == 0 ||
                       strcmp(str1, "$EDGETESS") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE EDGETESS requires ibody iedge");
                    goto cleanup;
                }

            /* $face ibody iface u v */
            } else if (strcmp(str1, "$face") == 0 ||
                       strcmp(str1, "$FACE") == 0   ) {
                if (narg != 5) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE FACE requires ibody iface u v");
                    goto cleanup;
                }

            /* $facebbox ibody iface */
            } else if (strcmp(str1, "$facebbox") == 0 ||
                       strcmp(str1, "$FACEBBOX") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE FACEBBOX requires ibody iface");
                    goto cleanup;
                }

            /* $facerng ibody iface */
            } else if (strcmp(str1, "$facerng") == 0 ||
                       strcmp(str1, "$FACERNG") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE FACERNG requires ibody iface");
                    goto cleanup;
                }

            /* $faceinv ibody iface x y z */
            } else if (strcmp(str1, "$faceinv") == 0 ||
                       strcmp(str1, "$FACEINV") == 0   ) {
                if (narg != 6) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE FACEINV requires ibody iface x y z");
                    goto cleanup;
                }

            /* $faceukt ibody iface */
            } else if (strcmp(str1, "$faceukt") == 0 ||
                       strcmp(str1, "$FACEUKT") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE FACEUKT requires ibody iface");
                    goto cleanup;
                }

            /* $facevkt ibody iface */
            } else if (strcmp(str1, "$facevkt") == 0 ||
                       strcmp(str1, "$FACEVKT") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE FACEVKT requires ibody iface");
                    goto cleanup;
                }

            /* $facecp ibody iface */
            } else if (strcmp(str1, "$facecp") == 0 ||
                       strcmp(str1, "$FACECP") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE FACECP requires ibody iface");
                    goto cleanup;
                }

            /* $facetess ibody iface */
            } else if (strcmp(str1, "$facetess") == 0 ||
                       strcmp(str1, "$FACETESS") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE FACETESS requires ibody iface");
                    goto cleanup;
                }

            /* $dist ibody1 ibody2 */
            } else if (strcmp(str1, "$dist") == 0 ||
                       strcmp(str1, "$DIST") == 0   ) {
                if (narg != 3) {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "EVALUATE DIST requires ibody1 ibody2");
                    goto cleanup;
                }

            } else {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "type must be NODE, EDGE, EDGEBBOX, EDGERNG, EDGEINV, EDGEKT, EDGECP, EDGETESS, FACE, FACEBBOX, FACERNG, FACEINV, FACEUKT, FACEVKT, FACECP, FACETESS, or DIST");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_EVALUATE, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "extract entList" */
        } else if (strcmp(command, "extract") == 0 ||
                   strcmp(command, "EXTRACT") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "EXTRACT cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "EXTRACT cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract argument */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "EXTRACT requires 1 argument");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_EXTRACT, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "extrude dx dy dz" */
        } else if (strcmp(command, "extrude") == 0 ||
                   strcmp(command, "EXTRUDE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "EXTRUDE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "EXTRUDE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg != 3) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "EXTRUDE requires 3 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_EXTRUDE, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "fillet radius edgeList=0 listStyle=0" */
        } else if (strcmp(command, "fillet") == 0 ||
                   strcmp(command, "FILLET") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "FILLET cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "FILLET cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if        (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "ASSERT requires at least 1 argument");
                goto cleanup;
            } else if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            } else if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_FILLET, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "getattr $pmtrName attrID global=0" */
        } else if (strcmp(command, "getattr") == 0 ||
                   strcmp(command, "GETATTR") == 0   ) {

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s\n",
                          &(str1[1]), str2, str3);
            if (narg < 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "GETATTR requires at least 2 arguments");
                goto cleanup;
            }

            if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[1] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_GETATTR, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "group nbody=0" */
        } else if (strcmp(command, "group") == 0 ||
                   strcmp(command, "GROUP") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "GROUP cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "GROUP cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg < 1) {
                STRNCPY(str1, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_GROUP, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "hollow thick=0 entList=0 listStyle=0" */
        } else if (strcmp(command, "hollow") == 0 ||
                   strcmp(command, "HOLLOW") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "HOLLOW cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "HOLLOW cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if        (narg < 1) {
                STRNCPY(str1, "0", MAX_EXPR_LEN);
                STRNCPY(str2, "0", MAX_EXPR_LEN);
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            } else if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            } else if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_HOLLOW, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "ifthen val1 $op1=ne val2=0 $op2=and val3=0 $op3=eq val4=0" */
        } else if (strcmp(command, "ifthen") == 0 ||
                   strcmp(command, "IFTHEN") == 0   ) {

            /* extract arguments */
            strcpy(str2, "$");
            strcpy(str4, "$");
            strcpy(str6, "$");
            strcpy(str8, "$");
            narg = sscanf(nextline, "%*s %2047s %2046s %2047s %2046s %2047s %2046s %2047s\n",
                          str1, &(str2[1]), str3, &(str4[1]), str5, &(str6[1]), str7);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "IFTHEN requires at least 1 argument");
                goto cleanup;
            }

            if (narg < 2) {
                strcpy(str2, "$ne");
            }
            if (narg < 3) {
                strcpy(str3, "0");
            }
            if (narg < 4) {
                strcpy(str4, "$and");
            }
            if (narg < 5) {
                strcpy(str5, "0");
            }
            if (narg < 6) {
                strcpy(str6, "$eq");
            }
            if (narg < 7) {
                strcpy(str7, "0");
            }

            if (strcmp(str2, "$lt") != 0 && strcmp(str2, "$LT") != 0 &&
                strcmp(str2, "$le") != 0 && strcmp(str2, "$LE") != 0 &&
                strcmp(str2, "$eq") != 0 && strcmp(str2, "$EQ") != 0 &&
                strcmp(str2, "$ge") != 0 && strcmp(str2, "$GE") != 0 &&
                strcmp(str2, "$gt") != 0 && strcmp(str2, "$GT") != 0 &&
                strcmp(str2, "$ne") != 0 && strcmp(str2, "$NE") != 0   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "op1 must be LT, LE, EQ, GE, GT, or NE");
                goto cleanup;
            }
            if (strcmp(str4, "$or" ) != 0 && strcmp(str4, "$OR" ) != 0 &&
                strcmp(str4, "$and") != 0 && strcmp(str4, "$AND") != 0 &&
                strcmp(str4, "$xor") != 0 && strcmp(str4, "$XOR") != 0   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "op2 must be OR, AND, or XOR");
                goto cleanup;
            }
            if (strcmp(str6, "$lt") != 0 && strcmp(str6, "$LT") != 0 &&
                strcmp(str6, "$le") != 0 && strcmp(str6, "$LE") != 0 &&
                strcmp(str6, "$eq") != 0 && strcmp(str6, "$EQ") != 0 &&
                strcmp(str6, "$ge") != 0 && strcmp(str6, "$GE") != 0 &&
                strcmp(str6, "$gt") != 0 && strcmp(str6, "$GT") != 0 &&
                strcmp(str6, "$ne") != 0 && strcmp(str6, "$NE") != 0   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "op3 must be LT, LE, EQ, GE, GT, or NE");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_IFTHEN, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "import $filename bodynumber=1 getcolors=0" */
        } else if (strcmp(command, "import") == 0 ||
                   strcmp(command, "IMPORT") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "IMPORT cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "IMPORT cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract argument */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s\n",
                          &(str1[1]), str2, str3);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "IMPORT requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                STRNCPY(str2, "1", MAX_EXPR_LEN);
            }
            if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }

            /* convert filename if it starts with "$/" or "$$/" to relative pathname */
            if (strncmp(str1, "$$/", 3) == 0) {
                STRNCPY(pathname, filename, MAX_EXPR_LEN);
                i = STRLEN(pathname) - 1;
                while (pathname[i] != SLASH) {
                    pathname[i] = '\0';
                    i--;
                    if (i < 0) break;
                }
                STRNCPY(tmpfilename, &(str1[3]), MAX_EXPR_LEN);
                snprintf(str1, MAX_EXPR_LEN, "$%s%s", pathname, tmpfilename);
            } else if (strncmp(str1, "$$$/", 4) == 0) {
                STRNCPY(pathname, filename, MAX_EXPR_LEN);
                i = STRLEN(pathname) - 1;
                while (pathname[i] != SLASH) {
                    pathname[i] = '\0';
                    i--;
                    if (i < 0) break;
                }
                STRNCPY(tmpfilename, &(str1[4]), MAX_EXPR_LEN);
                snprintf(str1, MAX_EXPR_LEN, "$%s%s", pathname, tmpfilename);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_IMPORT, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "interface $argName $argType default=0" */
        } else if (strcmp(command, "interface") == 0 ||
                   strcmp(command, "INTERFACE") == 0   ) {
            if (filetype == 0 && MODL->level <= 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "INTERFACE not allowed in .csm file");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");  strcpy(str2, "$");
            narg = sscanf(nextline, "%*s %2046s %2046s %2047s\n",
                          &(str1[1]), &(str2[1]), str3);
            if        (narg == 2) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            } else if (narg != 3) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "INTERFACE requires at least 2 arguments");
                goto cleanup;
            }

            /* ensure that valid argType is given */
            if (strcmp(str2, "$dim") == 0 || strcmp(str2, "$DIM") == 0) {
                SPRINT0(1, "WARNING:: INTERFACE DIM is obsolete. use DIMENSION instead");
                (MODL->nwarn)++;
            } else if ((strcmp(str2, "$in")  != 0) && (strcmp(str2, "$IN" ) != 0) &&
                       (strcmp(str2, "$out") != 0) && (strcmp(str2, "$OUT") != 0) &&
                       (strcmp(str2, "$all") != 0) && (strcmp(str2, "$ALL") != 0)   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "argType must be IN, OUT, DIM, or ALL");
                goto cleanup;
            }

            /* if an include-type UDC, reduce the scope back (note: it was
               increased by the UDPRIM statement) */
            if ((strcmp(str2, "$all") == 0) || (strcmp(str2, "$ALL") == 0)) {
                MODL->scope[MODL->level] = MODL->scope[MODL->level-1];
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_INTERFACE, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "intersect $order=none index=1 maxtol=0" */
        } else if (strcmp(command, "intersect") == 0 ||
                   strcmp(command, "INTERSECT") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "INTERSECT cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "INTERSECT cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s\n",
                          &(str1[1]), str2, str3);
            if (narg < 1) {
                STRNCPY(str1, "$none", MAX_EXPR_LEN);
            }
            if (narg < 2) {
                STRNCPY(str2, "1",     MAX_EXPR_LEN);
            }
            if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_INTERSECT, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "join toler=0 toMark=0" */
        } else if (strcmp(command, "join") == 0 ||
                   strcmp(command, "JOIN") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "JOIN cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "JOIN cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract argument */
            narg = sscanf(nextline, "%*s %2047s %2047s\n",
                          str1, str2);
            if        (narg < 1) {
                STRNCPY(str1, "0", MAX_EXPR_LEN);
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            } else if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_JOIN, filename, linenum,
                                 str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "lbound $pmtrName expression" */
        } else if (strcmp(command, "lbound") == 0 ||
                   strcmp(command, "LBOUND") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "LBOUND cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "LBOUND cannot be in SOLBEG/SOLEND");
                goto cleanup;
            } else if (MODL->scope[MODL->level] > 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "LBOUND not allowed except at top level");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s\n",
                          str1, str2);
            if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "LBOUND requires 2 arguments");
                goto cleanup;
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[0] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /* make sure that bounds is a semicolon-separated list of numbers */
            icount = 0;
            while (icount < STRLEN(str2)) {
                jcount = 0;

                while (icount < STRLEN(str2)) {
                    if (str2[icount] == ';') {
                        icount++;
                        break;
                    } else {
                        defn[jcount  ] =str2[icount];
                        defn[jcount+1] = '\0';
                        icount++;
                        jcount++;
                    }
                }

                /* check for a valid number */
                if (sscanf(defn, "%lf", &value) == 0) {
                    status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                          "expression must evaluate to a number");
                    goto cleanup;
                }
            }

            /* break str1 into pmtrName[irow,icol] */
            status = parseName(MODL, str1, pmtrName, &ipmtr, &irow, &icol);
            CHECK_STATUS2(parseName);

            /* if it does not exist, signal error */
            if (ipmtr == 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "\"%s\" does not exist", pmtrName);
                goto cleanup;
            } else {
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                /* make sure that Parameter is DESPMTR or CFGPMTR */
                if (MODL->pmtr[ipmtr].type == OCSM_LOCALVAR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_LOCALVAR, filename, linenum,
                                          "%s is an internal parameter", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_OUTPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_OUTPMTR, filename, linenum,
                                          "%s is an OUTPMTR", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_CONPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_CONPMTR, filename, linenum,
                                          "%s is a CONPMTR", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_UNKNOWN) {
                    status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                          "%s is an unknown type", str1);
                    goto cleanup;
                }
            }

            /* store the bounds for the whole Parameter */
            if ((irow ==    0 && icol ==    0) ||
                (irow == -999 && icol == -999)   ) {
                icount = 0;
                for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                    for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                        jcount = 0;

                        while (icount < STRLEN(str2)) {
                            if (str2[icount] == ';') {
                                icount++;
                                break;
                            } else {
                                defn[jcount  ] = str2[icount];
                                defn[jcount+1] = '\0';
                                icount++;
                                jcount++;
                            }
                        }

                        status = str2val(defn, MODL, &bound, &dot, str);
                        CHECK_STATUS2(str2val);
                        if (STRLEN(str) > 0) {
                            status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                                  "expression must evaluate to a number");
                            goto cleanup;
                        }

                        indx = icol-1 + (irow-1)*MODL->pmtr[ipmtr].ncol;
                        MODL->pmtr[ipmtr].lbnd[indx] = bound;
                    }
                }

            /* store the bounds for all rows of the Parameter */
            } else if (irow == -999) {
                icount = 0;
                for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                    jcount = 0;

                    while (icount < STRLEN(str2)) {
                        if (str2[icount] == ';') {
                            icount++;
                            break;
                        } else {
                            defn[jcount  ] = str2[icount];
                            defn[jcount+1] = '\0';
                            icount++;
                            jcount++;
                        }
                    }

                    status = str2val(defn, MODL, &bound, &dot, str);
                    CHECK_STATUS2(str2val);
                    if (STRLEN(str) > 0) {
                        status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                              "expression must evaluate to a number");
                        goto cleanup;
                    }

                    indx = icol-1 + (irow-1)*MODL->pmtr[ipmtr].ncol;
                    MODL->pmtr[ipmtr].lbnd[indx] = bound;
                }

            /* store the bounds for all columns of the Parameter */
            } else if (icol == -999) {
                icount = 0;
                for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                    jcount = 0;

                    while (icount < STRLEN(str2)) {
                        if (str2[icount] == ';') {
                            icount++;
                            break;
                        } else {
                            defn[jcount  ] = str2[icount];
                            defn[jcount+1] = '\0';
                            icount++;
                            jcount++;
                        }
                    }

                    status = str2val(defn, MODL, &bound, &dot, str);
                    CHECK_STATUS2(str2val);
                    if (STRLEN(str) > 0) {
                        status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                              "expression must evaluate to a number");
                        goto cleanup;
                    }

                    indx = icol-1 + (irow-1)*MODL->pmtr[ipmtr].ncol;
                    MODL->pmtr[ipmtr].lbnd[indx] = bound;
                }

            /* store a single bound into the Parameter */
            } else {
                if (irow <= 0) irow = 1;
                if (icol <= 0) icol = 1;

                icount = 0;
                jcount = 0;

                while (icount < STRLEN(str2)) {
                    if (str2[icount] == ';') {
                        icount++;
                        break;
                    } else {
                        defn[jcount  ] = str2[icount];
                        defn[jcount+1] = '\0';
                        icount++;
                        jcount++;
                    }
                }

                status = str2val(defn, MODL, &bound, &dot, str);
                CHECK_STATUS2(str2val);
                if (STRLEN(str) > 0) {
                    status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                          "expression must evaluate to a number");
                    goto cleanup;
                }

                indx = icol-1 + (irow-1)*MODL->pmtr[ipmtr].ncol;
                MODL->pmtr[ipmtr].lbnd[indx] = bound;
            }

        /* input is: "linseg x y z" */
        } else if (strcmp(command, "linseg") == 0 ||
                   strcmp(command, "LINSEG") == 0   ) {
            if (nskpt == 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_NOT_OPEN, filename, linenum,
                                      "LINSEG must be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "LINSEG cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg != 3) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "LINSEG requires 3 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_LINSEG, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* increment the number of Sketch points */
            nskpt++;

        /* input is: "loft smooth" */
        } else if (strcmp(command, "loft") == 0 ||
                   strcmp(command, "LOFT") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "LOFT cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "LOFT cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract argument */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "LOFT requires 1 argument");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_LOFT, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "macbeg imacro" */
        } else if (strcmp(command, "macbeg") == 0 ||
                   strcmp(command, "MACBEG") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "MACBEG cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "MACBEG cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract argument */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "MACBEG requires 1 argument");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_MACBEG, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "macend" */
        } else if (strcmp(command, "macend") == 0 ||
                   strcmp(command, "MACEND") == 0   ) {
            if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "MACEND cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* no arguments */

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_MACEND, filename, linenum,
                                 NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "mark" */
        } else if (strcmp(command, "mark") == 0 ||
                   strcmp(command, "MARK") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "MARK cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "MARK cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* no arguments */

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_MARK, filename, linenum,
                                 NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "message $text $schar=_ $fileName=. $openType=a" */
        } else if (strcmp(command, "message") == 0 ||
                   strcmp(command, "MESSAGE") == 0   ) {

            /* extract argument */
            strcpy(str1, "$");
            strcpy(str2, "$");
            strcpy(str3, "$");
            strcpy(str4, "$");
            narg = sscanf(nextline, "%*s %2046s %2046s %2046s %2046s\n",
                          &(str1[1]), &(str2[1]), &(str3[1]), &(str4[1]));
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "MESSAGE requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                strcpy(str2, "$_");
            }
            if (narg < 3) {
                strcpy(str3, "$.");
            }
            if (narg < 4) {
                strcpy(str4, "$a");
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_MESSAGE, filename, linenum,
                                 str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "mirror nx ny nz dist=0" */
        } else if (strcmp(command, "mirror") == 0 ||
                   strcmp(command, "MIRROR") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "MIRROR cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "MIRROR cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4);
            if (narg == 3) {
                STRNCPY(str4, "0", MAX_EXPR_LEN);
            } else if (narg != 4) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "MIRROR requires 4 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_MIRROR, filename, linenum,
                                 str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "name $branchName" */
        } else if (strcmp(command, "name") == 0 ||
                   strcmp(command, "NAME") == 0   ) {

            /* previous Branch will be named */
            ibrch = MODL->ibrch;
            if (ibrch < 1) {
                status = signalError2(MODL, OCSM_ILLEGAL_BRCH_INDEX, filename, linenum,
                                      "NAME must follow a Branch");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "NAME requires 1 argument");
                goto cleanup;
            }

            /* set the Branch's name */
            status = ocsmSetName(MODL, ibrch, str1);
            CHECK_STATUS2(ocsmSetName);

        /* input is: "outpmtr $pmtrName" */
        } else if (strcmp(command, "outpmtr") == 0 ||
                   strcmp(command, "OUTPMTR") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "OUTPMTR cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "OUTPMTR cannot be in SOLBEG/SOLEND");
                goto cleanup;
            } else if (MODL->scope[MODL->level] > 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "OUTPMTR not allowed except at top level");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "OUTPMTR requires 1 argument");
                goto cleanup;
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[0] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /* break str1 into pmtrName[irow,icol] */
            status = parseName(MODL, str1, pmtrName, &ipmtr, &irow, &icol);
            CHECK_STATUS2(parseName);

            /* if it does not exist, create it now */
            if (ipmtr == 0) {
                status = ocsmNewPmtr(MODL, pmtrName, OCSM_OUTPMTR, 1, 1);
                CHECK_STATUS2(ocsmNewPmtr);
                ipmtr = MODL->npmtr;
            } else {
                SPLINT_CHECK_FOR_NULL(MODL->pmtr);

                /* if it exists and is UNKNOWN (because of a DIMENSION statemet),
                   convert to a OUTPMTR */
                if (MODL->pmtr[ipmtr].type == OCSM_UNKNOWN) {
                    MODL->pmtr[ipmtr].type = OCSM_OUTPMTR;

                /* make sure that Parameter is OUTPMTR */
                } else if (MODL->pmtr[ipmtr].type == OCSM_LOCALVAR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_LOCALVAR, filename, linenum,
                                          "%s is an internal parameter", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_DESPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_DESPMTR, filename, linenum,
                                          "%s is a DESPMTR", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_CFGPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_DESPMTR, filename, linenum,
                                          "%s is a CFGPMTR", str1);
                    goto cleanup;
                } else if (MODL->pmtr[ipmtr].type == OCSM_CONPMTR) {
                    status = signalError2(MODL, OCSM_PMTR_IS_CONPMTR, filename, linenum,
                                          "%s is a CONPMTR", str1);
                    goto cleanup;
                }
            }

        /* input is: "patbeg $pmtrName ncopy" */
        } else if (strcmp(command, "patbeg") == 0 ||
                   strcmp(command, "PATBEG") == 0   ) {
            if (npatn >= MAX_NESTING) {
                status = signalError2(MODL, OCSM_NESTED_TOO_DEEPLY, filename, linenum,
                                      "npatn=%d >0 MAX_NESTING=%d", npatn, MAX_NESTING);
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "PATBEG cannot be in SOLBEG/SOLEND");
                goto cleanup;
            } else {
                npatn++;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s\n",
                          &(str1[1]), str2);
            if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "PATBEG requires 2 arguments");
                goto cleanup;
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[1] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_PATBEG, filename, linenum,
                                 str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "patbreak val1=1 $op1=gt val2=0 $op2=and val3=0 $op3=eq val4=0" */
        } else if (strcmp(command, "patbreak") == 0 ||
                   strcmp(command, "PATBREAK") == 0   ) {

            if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "PATBREAK cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* find the matching patbeg */
            i = 1;
            for (ibrch = MODL->nbrch; ibrch > 0; ibrch--) {
                SPLINT_CHECK_FOR_NULL(MODL->brch);

                if        (MODL->brch[ibrch].type == OCSM_PATEND) {
                    i++;
                } else if (MODL->brch[ibrch].type == OCSM_PATBEG) {
                    i--;
                    if (i == 0) {
                        break;
                    }
                }
            }
            if (ibrch <= 0) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "PATBREAK must follow a PATBEG");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str2, "$");
            strcpy(str4, "$");
            strcpy(str6, "$");
            strcpy(str8, "$");
            narg = sscanf(nextline, "%*s %2047s %2046s %2047s %2046s %2047s %2046s %2047s\n",
                          str1, &(str2[1]), str3, &(str4[1]), str5, &(str6[1]), str7);

            if (narg < 1) {
                strcpy(str1, "1");
            }
            if (narg < 2) {
                strcpy(str2, "$gt");
            }
            if (narg < 3) {
                strcpy(str3, "0");
            }
            if (narg < 4) {
                strcpy(str4, "$and");
            }
            if (narg < 5) {
                strcpy(str5, "0");
            }
            if (narg < 6) {
                strcpy(str6, "$eq");
            }
            if (narg < 7) {
                strcpy(str7, "0");
            }

            if (strcmp(str2, "$lt") != 0 && strcmp(str2, "$LT") != 0 &&
                strcmp(str2, "$le") != 0 && strcmp(str2, "$LE") != 0 &&
                strcmp(str2, "$eq") != 0 && strcmp(str2, "$EQ") != 0 &&
                strcmp(str2, "$ge") != 0 && strcmp(str2, "$GE") != 0 &&
                strcmp(str2, "$gt") != 0 && strcmp(str2, "$GT") != 0 &&
                strcmp(str2, "$ne") != 0 && strcmp(str2, "$NE") != 0   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "op1 must be LT, LE, EQ, GE, GT, or NE");
                goto cleanup;
            }
            if (strcmp(str4, "$or" ) != 0 && strcmp(str4, "$OR" ) != 0 &&
                strcmp(str4, "$and") != 0 && strcmp(str4, "$AND") != 0 &&
                strcmp(str4, "$xor") != 0 && strcmp(str4, "$XOR") != 0   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "op2 must be OR, AND, or XOR");
                goto cleanup;
            }
            if (strcmp(str6, "$lt") != 0 && strcmp(str6, "$LT") != 0 &&
                strcmp(str6, "$le") != 0 && strcmp(str6, "$LE") != 0 &&
                strcmp(str6, "$eq") != 0 && strcmp(str6, "$EQ") != 0 &&
                strcmp(str6, "$ge") != 0 && strcmp(str6, "$GE") != 0 &&
                strcmp(str6, "$gt") != 0 && strcmp(str6, "$GT") != 0 &&
                strcmp(str6, "$ne") != 0 && strcmp(str6, "$NE") != 0   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "op3 must be LT, LE, EQ, GE, GT, or NE");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_PATBREAK, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "patend" */
        } else if (strcmp(command, "patend") == 0 ||
                   strcmp(command, "PATEND") == 0   ) {
            if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "PATEND cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* find the matching patbeg */
            i = 1;
            for (ibrch = MODL->nbrch; ibrch > 0; ibrch--) {
                SPLINT_CHECK_FOR_NULL(MODL->brch);

                if        (MODL->brch[ibrch].type == OCSM_PATEND) {
                    i++;
                } else if (MODL->brch[ibrch].type == OCSM_PATBEG) {
                    i--;
                    if (i == 0) {
                        break;
                    }
                }
            }
            if (ibrch <= 0) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "PATEND must follow a PATBEG");
                goto cleanup;
            }

            /* no arguments */

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_PATEND, filename, linenum,
                                 NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            npatn--;

        /* input is: "point xloc yloc zloc" */
        } else if (strcmp(command, "point") == 0 ||
                   strcmp(command, "POINT") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "POINT cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "POINT cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg != 3) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "POINT requires 3 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_POINT, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "prefix $prefix=." */
        } else if (strcmp(command, "prefix") == 0 ||
                   strcmp(command, "PREFIX") == 0   ) {

            /* extract argument */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s\n",
                          &(str1[1]));
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "PREFIX requires 1 argument");
                goto cleanup;
            }

            /* set the prefix */
            if (strcmp(str1, "$.") == 0) {
                MODL->prefix[0] = '\0';
            } else {
                STRNCPY(MODL->prefix, &(str1[1]), MAX_NAME_LEN);
            }

            SPRINT1(1, "        prefix is now \"%s\"", MODL->prefix);

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_PREFIX, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "project x y z dx dy dz useEdges=0" */
        } else if (strcmp(command, "project") == 0 ||
                   strcmp(command, "PROJECT") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "PROJECT cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "PROJECT cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract argument and save*/
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4, str5, str6, str7);
            if (narg < 6) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "PROJECT requires at least 6 arguments");
                goto cleanup;
            }
            if (narg < 7) {
                strcpy(str7, "0");
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_PROJECT, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "recall imacro" */
        } else if (strcmp(command, "recall") == 0 ||
                   strcmp(command, "RECALL") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "RECALL cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "RECALL cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract argument and save*/
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "RECALL requires 1 argument");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_RECALL, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "reorder ishift iflip=0 reverse=0" */
        } else if (strcmp(command, "reorder") == 0 ||
                   strcmp(command, "REORDER") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "REORDER cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "REORDER cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg == 1) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            } else if (narg == 2) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            } else if (narg == 3) {
            } else {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "REORDER requires at least 1 argument");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_REORDER, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "restore $name index=0" */
        } else if (strcmp(command, "restore") == 0 ||
                   strcmp(command, "RESTORE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "RESTORE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "RESTORE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s\n",
                          &(str1[1]), str2);
            if (narg == 1) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            } else if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "RESTORE requires at least 1 argument");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_RESTORE, filename, linenum,
                                 str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "revolve xorig yorig zorig dxaxis dyaxis dzaxis angDeg" */
        } else if (strcmp(command, "revolve") == 0 ||
                   strcmp(command, "REVOLVE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "REVOLVE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "REVOLVE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4, str5, str6, str7);
            if (narg != 7) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "REVOLVE requires 7 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_REVOLVE, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "rotatex angDeg yaxis=0 zaxis=0" */
        } else if (strcmp(command, "rotatex") == 0 ||
                   strcmp(command, "ROTATEX") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "ROTATEX cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "ROTATEX cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "ROTATEX requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                strcpy(str2, "0");
            }
            if (narg < 3) {
                strcpy(str3, "0");
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_ROTATEX, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "rotatey angDeg zaxis=0 xaxis=0" */
        } else if (strcmp(command, "rotatey") == 0 ||
                   strcmp(command, "ROTATEY") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "ROTATEY cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "ROTATEY cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "ROTATEY requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                strcpy(str2, "0");
            }
            if (narg < 3) {
                strcpy(str3, "0");
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_ROTATEY, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "rotatez angDeg xaxis=0 yaxis=0" */
        } else if (strcmp(command, "rotatez") == 0 ||
                   strcmp(command, "ROTATEZ") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "ROTATEZ cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "ROTATEZ cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "ROTATEZ requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                strcpy(str2, "0");
            }
            if (narg < 3) {
                strcpy(str3, "0");
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_ROTATEZ, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "rule reorder=0 periodic=0 copyAttr=0" */
        } else if (strcmp(command, "rule") == 0 ||
                   strcmp(command, "RULE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "RULE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "RULE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg < 1) {
                STRNCPY(str1, "0", MAX_EXPR_LEN);
            }
            if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            }
            if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_RULE, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "scale fact xcent=0 ycent=0 zcent=0" */
        } else if (strcmp(command, "scale") == 0 ||
                   strcmp(command, "SCALE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "SCALE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SCALE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SCALE requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                strcpy(str2, "0");
            }
            if (narg < 3) {
                strcpy(str3, "0");
            }
            if (narg < 4) {
                strcpy(str4, "0");
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SCALE, filename, linenum,
                                 str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "select $type arg1 ..." */
        } else if (strcmp(command, "select") == 0 ||
                   strcmp(command, "SELECT") == 0   ) {

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          &(str1[1]), str2, str3, str4, str5, str6, str7, str8);

            /* create the new Branch */
            if        (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SELECT requires at least 1 argument");
                goto cleanup;
            } else if (strcmp(str1, "$body") == 0 || strcmp(str1, "$BODY") == 0) {
                /* body */
                if        (narg == 1) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                /* body attrName1 attrValue1 */
                } else if (str2[0] == '$' && narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, "$*", "$*", "$*", "$*", NULL, NULL);
                /* body attrName1 attrValue1 attrName2 attrValue2 */
                } else if (str2[0] == '$' && narg == 5) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, "$*", "$*", NULL, NULL);
                /* body attrName1 attrValue1 attrName2 attrValue2 attrName3 attrValue3 */
                } else if (str2[0] == '$' && narg == 7) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, str7,  NULL, NULL);
                /* body ibody */
                } else if (narg == 2) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT BODY requires 1, 2, 3, 5, or 7 arguments");
                    goto cleanup;
                }
            } else if (strcmp(str1, "$face") == 0 || strcmp(str1, "$FACE") == 0) {
                /* face */
                if (narg == 1) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                /* face attrName1 attrValue1 */
                } else if (str2[0] == '$' && narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, "$*", "$*", "$*", "$*", NULL, NULL);
                /* face attrName1 attrValue1 attrName2 attrValue2 */
                } else if (str2[0] == '$' && narg == 5) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, "$*", "$*", NULL, NULL);
                /* face attrName1 attrValue1 attrName2 attrValue2 attrName3 attrValue3 */
                } else if (str2[0] == '$' && narg == 7) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, str7,  NULL, NULL);
                /* face iface  */
                } else if (narg == 2) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                /* face ibody1 iford1 (0 for any)   --or--   face -1 ibody1   --or--   face -2 ibody1 */
                } else if (narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
                /* face ibody1 iford1 iseq=1 (0 for any) */
                } else if (narg == 3 || narg == 4) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);

                /* face xmin xmax ymin ymax zmin zmax */
                } else if (narg == 7) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, str7, NULL, NULL);

                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT FACE requires 1, 2, 3, 4, 5, or 7 arguments");
                    goto cleanup;
                }
            } else if (strcmp(str1, "$edge") == 0 || strcmp(str1, "$EDGE") == 0) {
                /* edge */
                if (narg == 1) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                /* edge attrName1 attrValue1 */
                } else if (str2[0] == '$' && narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, "$*", "$*", "$*", "$*", NULL, NULL);
                /* edge attrName1 attrValue1 attrName2 attrValue2 */
                } else if (str2[0] == '$' && narg == 5) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, "$*", "$*", NULL, NULL);
                /* edge attrName1 attrValue1 attrName2 attrValue2 attrName3 attrValue3 */
                } else if (str2[0] == '$' && narg == 7) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, str7,  NULL, NULL);
                /* edge iedge */
                } else if (narg == 2) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                /* edge -1 ibody1   --or--   edge -2 ibody1 */
                } else if (narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
                /* edge ibody1 iford1 ibody2 iford2 (0 for any) */
                } else if (narg == 5) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, NULL, NULL, NULL, NULL);
                /* edge ibody1 iford1 ibody2 iford2 iseq=1 (0 for any) */
                } else if (narg == 5 || narg == 6) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, NULL, NULL, NULL);
                /* edge x y z */
                } else if (narg == 4) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);

                /* edge xmin xmax ymin ymax zmin zmax */
                } else if (narg == 7) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, str7, NULL, NULL);

                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT EDGE requires 1, 2, 3, 4, 5, 6, or 7 arguments");
                    goto cleanup;
                }
            } else if (strcmp(str1, "$loop") == 0 || strcmp(str1, "$LOOP") == 0) {
                /* loop iface iloop */
                if (narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, NULL, NULL ,NULL, NULL, NULL, NULL);
                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT LOOP requires 2 arguments");
                    goto cleanup;
                }
            } else if (strcmp(str1, "$node") == 0 || strcmp(str1, "$NODE") == 0) {
                /* node */
                if (narg == 1) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                /* node attrName1 attrValue1 */
                } else if (str2[0] == '$' && narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, "$*", "$*", "$*", "$*", NULL, NULL);
                /* node attrName1 attrValue1 attrName2 attrValue2 */
                } else if (str2[0] == '$' && narg == 5) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, "$*", "$*", NULL, NULL);
                /* node attrName1 attrValue1 attrName2 attrValue2 attrName3 attrValue3 */
                } else if (str2[0] == '$' && narg == 7) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, str7,  NULL, NULL);
                /* node inode */
                } else if (narg == 2) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                /* node -1 ibody1   --or--   node -2 ibody1 */
                } else if (narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
                /* node x y z */
                } else if (narg == 4) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);

                /* node xmin xmax ymin ymax zmin zmax */
                } else if (narg == 7) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, str7, NULL, NULL);

                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT NODE requires 1, 2, 3, 4, 5, or 7 arguments");
                    goto cleanup;
                }
            } else if (strcmp(str1, "$add") == 0 || strcmp(str1, "$ADD") == 0) {
                /* add iface  -or-  add iedge  -or-  add inode */
                if        (narg == 2) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);

                /* add attrName1 attrValue1 */
                } else if (str2[0] == '$' && narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, "$*", "$*", "$*", "$*", NULL, NULL);
                /* add attrName1 attrValue1 attrName2 attrValue2 */
                } else if (str2[0] == '$' && narg == 5) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, "$*", "$*", NULL, NULL);
                /* add attrName1 attrValue1 attrName2 attrValue2 attrName3 attrValue3 */
                } else if (str2[0] == '$' && narg == 7) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
                /* add ibody1 iford1 */
                } else if (narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);

                /* add ibody1 iford1 iseq=1 */
                } else if (narg == 4) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);

                /* add ibody1 iford1 ibody2 iford2 */
                } else if (narg == 5) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, NULL, NULL, NULL, NULL);

                /* add ibody1 iford1 ibody2 iford2 iseq=1 */
                } else if (narg == 6) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, NULL, NULL, NULL);

                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT ADD has wrong number of arguments");
                    goto cleanup;
                }
            } else if (strcmp(str1, "$sub") == 0 || strcmp(str1, "$SUB") == 0) {
                /* sub ient */
                if        (narg == 2) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);

                /* sub attrName1 attrValue1 */
                } else if (str2[0] == '$' && narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, "$*", "$*", "$*", "$*", NULL, NULL);
                /* sub attrName1 attrValue1 attrName2 attrValue2 */
                } else if (str2[0] == '$' && narg == 5) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, "$*", "$*", NULL, NULL);
                /* sub attrName1 attrValue1 attrName2 attrValue2 attrName3 attrValue3 */
                } else if (str2[0] == '$' && narg == 7) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, str7,  NULL, NULL);
                /* sub ibody1 iford1 */
                } else if (narg == 3) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);

                /* sub ibody1 iford1 iseq=1 */
                } else if (narg == 4) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);

                /* sub ibody1 iford1 ibody2 iford2 */
                } else if (narg == 5) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, NULL, NULL, NULL, NULL);

                /* sub ibody1 iford1 ibody2 iford2 iseq=1 */
                } else if (narg == 6) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, str3, str4, str5, str6, NULL, NULL, NULL);

                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT SUB has wrong number of arguments");
                    goto cleanup;
                }

            /* not */
            } else if (strcmp(str1, "$not") == 0 || strcmp(str1, "$NOT") == 0) {
                if (narg == 1) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT NOT has wrong number of arguments");
                }

            /* flip */
            } else if (strcmp(str1, "$flip") == 0 || strcmp(str1, "$FLIP") == 0) {
                if (narg == 1) {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT FLIP has wrong number of arguments");
                }

            } else if (strcmp(str1, "$sort") == 0 || strcmp(str1, "$SORT") == 0) {
                /* sort $key */
                if (narg == 2 && str2[0] == '$') {
                    status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SELECT, filename, linenum,
                                         str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
                } else {
                    status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                          "SELECT SORT has wrong number of arguments");
                    goto cleanup;
                }
            } else {
                status = signalError2(MODL, OCSM_ILLEGAL_TYPE, filename, linenum,
                                      "type must be BODY, FACE, EDGE, NODE, ADD, SUB, NOT, FLIP, or SORT");
                goto cleanup;
            }
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "set $pmtrName exprs" */
        } else if (strcmp(command, "set") == 0 ||
                   strcmp(command, "SET") == 0   ) {
            if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SET cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s\n",
                          &(str1[1]), str2);
            if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SET requires 2 arguments");
                goto cleanup;
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[1] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SET, filename, linenum,
                                 str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "skbeg x y z relative=0" */
        } else if (strcmp(command, "skbeg") == 0 ||
                   strcmp(command, "SKBEG") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "SKBEG cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SKBEG cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4);
            if (narg < 3) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SKBEG requires at least 3 arguments");
                goto cleanup;
            }
            if (narg < 4) {
                STRNCPY(str4, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SKBEG, filename, linenum,
                                 str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* increment the number of Sketch points */
            nskpt++;

        /* input is: "skcon $type index1 index2=-1 $value=0" */
        } else if (strcmp(command, "skcon") == 0 ||
                   strcmp(command, "SKCON") == 0   ) {
            if (nskpt <= 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_NOT_OPEN, filename, linenum,
                                      "SKCON must be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SKCON cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* make sure that this follows a skvar or skcon statement */
            SPLINT_CHECK_FOR_NULL(MODL->brch);

            if (MODL->brch[MODL->ibrch].type != OCSM_SKVAR &&
                MODL->brch[MODL->ibrch].type != OCSM_SKCON   ) {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "only SKVAR or SKCON can preceed SKCON statement");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            strcpy(str4, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s %2046s\n",
                          &(str1[1]), str2, str3, &(str4[1]));
            if (narg < 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SKCON requires 2 arguments");
                goto cleanup;
            }
            if (narg < 3) {
                STRNCPY(str3, "-1", MAX_EXPR_LEN);
            }
            if (narg < 4) {
                STRNCPY(str4, "$0", MAX_EXPR_LEN);
            }

            if        (strcmp(str1, "$X") == 0) {
            } else if (strcmp(str1, "$Y") == 0) {
            } else if (strcmp(str1, "$P") == 0) {
            } else if (strcmp(str1, "$T") == 0) {
            } else if (strcmp(str1, "$A") == 0) {
            } else if (strcmp(str1, "$W") == 0) {
            } else if (strcmp(str1, "$D") == 0) {
            } else if (strcmp(str1, "$H") == 0) {
            } else if (strcmp(str1, "$V") == 0) {
            } else if (strcmp(str1, "$I") == 0) {
            } else if (strcmp(str1, "$Z") == 0) {
            } else if (strcmp(str1, "$L") == 0) {
            } else if (strcmp(str1, "$R") == 0) {
            } else if (strcmp(str1, "$S") == 0) {
            } else {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "type must be X, Y, P, T, A, W, D, H, V, I, Z, L, R, or S");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SKCON, filename, linenum,
                                 str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "skend wireonly=0" */
        } else if (strcmp(command, "skend") == 0 ||
                   strcmp(command, "SKEND") == 0   ) {
            if (nskpt < 1) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "SKEND must follow SKBEG");
                goto cleanup;
            } else if (nskpt > MAX_SKETCH_SIZE) {
                status = signalError2(MODL, OCSM_TOO_MANY_SKETCH_POINTS, filename, linenum,
                                      "more than %d sketch points", MAX_SKETCH_SIZE);
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SKEND cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg <= 0) {
                STRNCPY(str1, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SKEND, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* reset the number of Sketch points */
            nskpt = 0;

        /* input is: "skvar $type valList" */
        } else if (strcmp(command, "skvar") == 0 ||
                   strcmp(command, "SKVAR") == 0   ) {
            if (nskpt <= 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_NOT_OPEN, filename, linenum,
                                      "SKBAR must be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SKVAR cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* make sure that this follows a skbeg statement */
            SPLINT_CHECK_FOR_NULL(MODL->brch);

            if (MODL->brch[MODL->ibrch].type != OCSM_SKBEG) {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "only SKBEG can preceed SKVAR statement");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s\n",
                          &(str1[1]), str2);
            if (narg < 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SKVAR requires 2 arguments");
                goto cleanup;
            }

            if        (strcmp(str1, "$xy") == 0 || strcmp(str1, "$XY") == 0) {
                STRNCPY(str1, "$xy", MAX_EXPR_LEN);
            } else if (strcmp(str1, "$yz") == 0 || strcmp(str1, "$YZ") == 0) {
                STRNCPY(str1, "$yz", MAX_EXPR_LEN);
            } else if (strcmp(str1, "$zx") == 0 || strcmp(str1, "$ZX") == 0) {
                STRNCPY(str1, "$zx", MAX_EXPR_LEN);
            } else {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "PLANE must be XY, YZ, or ZX");
                goto cleanup;
            }

            /* make sure that str2 contains the correct number of semicolons */
            count = 0;
            for (i = 0; i < STRLEN(str2); i++) {
                if (str2[i] == ';') {
                    count++;
                }
            }

            if (count == 0 || count%3 != 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_VALUE, filename, linenum,
                                      "valList has %d semi-colons, but must contain triplets", count);
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SKVAR, filename, linenum,
                                 str1, str2, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "solbeg $varList" */
        } else if (strcmp(command, "solbeg") == 0 ||
                   strcmp(command, "SOLBEG") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "SOLBEG cannot be in SKBEG/SKEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s\n",
                          &(str1[1]));
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SOLBEG requires 1 argument");
                goto cleanup;
            }

            /* remember that we are in a solver block */
            insolver = 1;

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SOLBEG, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "solcon $expr" */
        } else if (strcmp(command, "solcon") == 0 ||
                   strcmp(command, "SOLCON") == 0   ) {
            if (insolver != 1) {
                status = signalError2(MODL, OCSM_SOLVER_IS_NOT_OPEN, filename, linenum,
                                      "SOLCON must be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s\n",
                          &(str1[1]));
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SOLCON requires 1 argument");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SOLCON, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "solend" */
        } else if (strcmp(command, "solend") == 0 ||
                   strcmp(command, "SOLEND") == 0   ) {
            if (insolver != 1) {
                status = signalError2(MODL, OCSM_SOLVER_IS_NOT_OPEN, filename, linenum,
                                      "SOLEND must be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* no arguments */

            /* close the solver */
            insolver = 0;

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SOLEND, filename, linenum,
                                 NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "special $option=. arg1=0 arg2=0 arg3=0 arg4=0 arg5=0 arg6=0 arg7=0 arg8=0" */
        } else if (strcmp(command, "special") == 0 ||
                   strcmp(command, "SPECIAL") == 0   ) {

            /* extract arguments */
            strcpy(str1, "$.");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          &(str1[1]), str2, str3, str4, str5, str6, str7, str8, str9);
            if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            }
            if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }
            if (narg < 4) {
                STRNCPY(str4, "0", MAX_EXPR_LEN);
            }
            if (narg < 5) {
                STRNCPY(str5, "0", MAX_EXPR_LEN);
            }
            if (narg < 6) {
                STRNCPY(str6, "0", MAX_EXPR_LEN);
            }
            if (narg < 7) {
                STRNCPY(str7, "0", MAX_EXPR_LEN);
            }
            if (narg < 8) {
                STRNCPY(str8, "0", MAX_EXPR_LEN);
            }
            if (narg < 9) {
                STRNCPY(str9, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SPECIAL, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, str7, str8, str9);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "sphere xcent ycent zcent radius" */
        } else if (strcmp(command, "sphere") == 0 ||
                   strcmp(command, "SPHERE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "SPHERE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SPHERE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4);
            if (narg != 4) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SPHERE requires 4 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SPHERE, filename, linenum,
                                 str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "spline x y z" */
        } else if (strcmp(command, "spline") == 0 ||
                   strcmp(command, "SPLINE") == 0   ) {
            if (nskpt == 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_NOT_OPEN, filename, linenum,
                                      "SPLINE must be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SPLINE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg != 3) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SPLINE requires 3 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SPLINE, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* increment the number of Sketch points */
            nskpt++;

        /* input is: "sslope dx dy dz" */
        } else if (strcmp(command, "sslope") == 0 ||
                   strcmp(command, "SSLOPE") == 0   ) {
            if (nskpt == 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_NOT_OPEN, filename, linenum,
                                      "SSLOPE must be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SSLOPE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg != 3) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "SSLOPE requires 3 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SSLOPE, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

            /* increment the number of Sketch points */
            nskpt++;

        /* input is: "store $name index=0 keep=0" */
        } else if (strcmp(command, "store") == 0 ||
                   strcmp(command, "STORE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "STORE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "STORE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s\n",
                          &(str1[1]), str2, str3);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "STORE requires at least 1 argument");
                goto cleanup;
            }
            if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            }
            if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_STORE, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "subtract $order=none index=1 maxtol=0 scribeAll=0" */
        } else if (strcmp(command, "subtract") == 0 ||
                   strcmp(command, "SUBTRACT") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "SUBTRACT cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SUBTRACT cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");
            narg = sscanf(nextline, "%*s %2046s %2047s %2047s %2047s\n",
                          &(str1[1]), str2, str3, str4);
            if (narg < 1) {
                STRNCPY(str1, "$none", MAX_EXPR_LEN);
            }
            if (narg < 2) {
                STRNCPY(str2, "1",     MAX_EXPR_LEN);
            }
            if (narg < 3) {
                STRNCPY(str3, "0",     MAX_EXPR_LEN);
            }
            if (narg < 4) {
                STRNCPY(str4, "0",     MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SUBTRACT, filename, linenum,
                                 str1, str2, str3, str4, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "sweep" */
        } else if (strcmp(command, "sweep") == 0 ||
                   strcmp(command, "SWEEP") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "SWEEP cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "SWEEP cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* no arguments */

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_SWEEP, filename, linenum,
                                 NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "throw sigCode" */
        } else if (strcmp(command, "throw") == 0 ||
                   strcmp(command, "THROW") == 0   ) {

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s\n",
                          str1);
            if (narg != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "THROW requires 1 argument");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_THROW, filename, linenum,
                                 str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "torus xcent ycent zcent dxaxis dyaxis dzaxis majorRad minorRad" */
        } else if (strcmp(command, "torus") == 0 ||
                   strcmp(command, "TORUS") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "TORUS cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "TORUS cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s %2047s %2047s %2047s %2047s %2047s\n",
                          str1, str2, str3, str4, str5, str6, str7, str8);
            if (narg != 8) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "TORUS requires 8 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_TORUS, filename, linenum,
                                 str1, str2, str3, str4, str5, str6, str7, str8, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "translate dx dy dz" */
        } else if (strcmp(command, "translate") == 0 ||
                   strcmp(command, "TRANSLATE") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "TRANSLATE cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "TRANSLATE cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg != 3) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "TRANSLATE requires 3 arguments");
                goto cleanup;
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_TRANSLATE, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "ubound $pmtrName expression" */
        } else if (strcmp(command, "ubound") == 0 ||
                   strcmp(command, "UBOUND") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "UBOUND cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "UBOUND cannot be in SOLBEG/SOLEND");
                goto cleanup;
            } else if (MODL->scope[MODL->level] > 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                      "UBOUND not allowed except at top level");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s\n",
                          str1, str2);
            if (narg != 2) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "UBOUND requires 2 arguments");
                goto cleanup;
            }

            /* do not allow pmtrName to start with '@' */
            if (str1[0] == '@') {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "pmtrName cannot start with an at-sign");
                goto cleanup;
            }

            /* make sure that values is a semicolon-separated list of numbers */
            icount = 0;
            while (icount < STRLEN(str2)) {
                jcount = 0;

                while (icount < STRLEN(str2)) {
                    if (str2[icount] == ';') {
                        icount++;
                        break;
                    } else {
                        defn[jcount  ] =str2[icount];
                        defn[jcount+1] = '\0';
                        icount++;
                        jcount++;
                    }
                }

                /* check for a valid number */
                if (sscanf(defn, "%lf", &value) == 0) {
                    status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                          "expression must evaluate to a number");
                    goto cleanup;
                }
            }

            /* break str1 into pmtrName[irow,icol] */
            status = parseName(MODL, str1, pmtrName, &ipmtr, &irow, &icol);
            CHECK_STATUS2(parseName);

            SPLINT_CHECK_FOR_NULL(MODL->pmtr);

            /* if it does not exist, signal error */
            if (ipmtr == 0) {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "\"%s\" does not exist", pmtrName);
                goto cleanup;

            /* make sure that Parameter is DESPMTR or CFGPMTR */
            } else if (MODL->pmtr[ipmtr].type == OCSM_LOCALVAR) {
                status = signalError2(MODL, OCSM_PMTR_IS_LOCALVAR, filename, linenum,
                                      "%s is an internal parameter", str1);
                goto cleanup;
            } else if (MODL->pmtr[ipmtr].type == OCSM_OUTPMTR) {
                status = signalError2(MODL, OCSM_PMTR_IS_OUTPMTR, filename, linenum,
                                      "%s is an OUTPMTR", str1);
                goto cleanup;
            } else if (MODL->pmtr[ipmtr].type == OCSM_CONPMTR) {
                status = signalError2(MODL, OCSM_PMTR_IS_CONPMTR, filename, linenum,
                                      "%s is a CONPMTR", str1);
                goto cleanup;
            } else if (MODL->pmtr[ipmtr].type == OCSM_UNKNOWN) {
                status = signalError2(MODL, OCSM_ILLEGAL_PMTR_NAME, filename, linenum,
                                      "%s is an unknown type", str1);
                goto cleanup;
            }

            /* store the bounds for the whole Parameter */
            if ((irow ==    0 && icol ==    0) ||
                (irow == -999 && icol == -999)   ) {
                icount = 0;
                for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                    for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                        jcount = 0;

                        while (icount < STRLEN(str2)) {
                            if (str2[icount] == ';') {
                                icount++;
                                break;
                            } else {
                                defn[jcount  ] = str2[icount];
                                defn[jcount+1] = '\0';
                                icount++;
                                jcount++;
                            }
                        }

                        status = str2val(defn, MODL, &bound, &dot, str);
                        CHECK_STATUS2(str2val);
                        if (STRLEN(str) > 0) {
                            status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                                  "expression must evaluate to a number");
                            goto cleanup;
                        }

                        indx = icol-1 + (irow-1)*MODL->pmtr[ipmtr].ncol;
                        MODL->pmtr[ipmtr].ubnd[indx] = bound;
                    }
                }

            /* store the bounds for all rows of the Parameter */
            } else if (irow == -999) {
                icount = 0;
                for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                    jcount = 0;

                    while (icount < STRLEN(str2)) {
                        if (str2[icount] == ';') {
                            icount++;
                            break;
                        } else {
                            defn[jcount  ] = str2[icount];
                            defn[jcount+1] = '\0';
                            icount++;
                            jcount++;
                        }
                    }

                    status = str2val(defn, MODL, &bound, &dot, str);
                    CHECK_STATUS2(str2val);
                    if (STRLEN(str) > 0) {
                        status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                              "expression must evaluate to a number");
                        goto cleanup;
                    }

                    indx = icol-1 + (irow-1)*MODL->pmtr[ipmtr].ncol;
                    MODL->pmtr[ipmtr].ubnd[indx] = bound;
                }

            /* store the bounds for all columns of the Parameter */
            } else if (icol == -999) {
                icount = 0;
                for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                    jcount = 0;

                    while (icount < STRLEN(str2)) {
                        if (str2[icount] == ';') {
                            icount++;
                            break;
                        } else {
                            defn[jcount  ] = str2[icount];
                            defn[jcount+1] = '\0';
                            icount++;
                            jcount++;
                        }
                    }

                    status = str2val(defn, MODL, &bound, &dot, str);
                    CHECK_STATUS2(str2val);
                    if (STRLEN(str) > 0) {
                        status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                              "expression must evaluate to a number");
                        goto cleanup;
                    }

                    indx = icol-1 + (irow-1)*MODL->pmtr[ipmtr].ncol;
                    MODL->pmtr[ipmtr].ubnd[indx] = bound;
                }

            /* store a single bound into the Parameter */
            } else {
                if (irow <= 0) irow = 1;
                if (icol <= 0) icol = 1;

                icount = 0;
                jcount = 0;

                while (icount < STRLEN(str2)) {
                    if (str2[icount] == ';') {
                        icount++;
                        break;
                    } else {
                        defn[jcount  ] = str2[icount];
                        defn[jcount+1] = '\0';
                        icount++;
                        jcount++;
                    }
                }

                status = str2val(defn, MODL, &bound, &dot, str);
                CHECK_STATUS2(str2val);
                if (STRLEN(str) > 0) {
                    status = signalError2(MODL, OCSM_WRONG_PMTR_TYPE, filename, linenum,
                                          "expression must evaluate to a number");
                    goto cleanup;
                }

                indx = icol-1 + (irow-1)*MODL->pmtr[ipmtr].ncol;
                MODL->pmtr[ipmtr].ubnd[indx] = bound;
            }

        /* input is: "udparg $primtype $argName1 argValue1 $argName2 argValue2 $argName3 argValue3 $argName4 argValue4" */
        } else if (strcmp(command, "udparg") == 0 ||
                   strcmp(command, "UDPARG") == 0   ) {

            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "UDPARG cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "UDPARG cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");  strcpy(str2, "$");  strcpy(str4, "$");  strcpy(str6, "$");  strcpy(str8, "$");
            narg = sscanf(nextline, "%*s %2046s %2046s %2047s %2046s %2047s %2046s %2047s %2046s %2047s\n",
                          &(str1[1]), &(str2[1]), str3, &(str4[1]), str5,
                                      &(str6[1]), str7, &(str8[1]), str9);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "UDPARG requires at least 1 argument");
                goto cleanup;
            } else if (narg%2 != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "UDPARG  requires 1, 3, 5, 7, or 9 arguments");
                goto cleanup;
            }

            if (narg < 3) str3[0] = '\0';
            if (narg < 5) str5[0] = '\0';
            if (narg < 7) str7[0] = '\0';
            if (narg < 9) str9[0] = '\0';

            /* get the arguments associated with the UDP/UDF */
            if (isalpha(str1[1]) != 0) {
                status = udp_initialize(&str1[1], MODL, &udp_num, &udp_names, &udp_types, &udp_idef, &udp_ddef);

                /* if we got an error, then we will not process any filename-like arguments */
                if (status != EGADS_SUCCESS) {

                /* adjust file spec or extract "inline" file */
                } else {
                    SPLINT_CHECK_FOR_NULL(fp_csm);

                    ifound = 0;
                    for (i = 0; i < udp_num; i++) {
                        if (strcasecmp(&str2[1], udp_names[i]) == 0 && udp_types[i] == ATTRFILE) {
                            status = setupUdprimFile(MODL, 1, fp_csm, filename, &linenum, str3);
                            CHECK_STATUS2(setupUdprimFile);

                            ifound = 1;
                            break;
                        }
                    }
                    if (ifound == 0) {
                        status = setupUdprimFile(MODL, 0, fp_csm, filename, &linenum, str3);
                        CHECK_STATUS2(setupUdprimFile);
                    }

                    ifound = 0;
                    for (i = 0; i < udp_num; i++) {
                        if (strcasecmp(&str4[1], udp_names[i]) == 0 && udp_types[i] == ATTRFILE) {
                            status = setupUdprimFile(MODL, 1, fp_csm, filename, &linenum, str5);
                            CHECK_STATUS2(setupUdprimFile);

                            ifound = 1;
                        break;
                        }
                    }
                    if (ifound == 0) {
                        status = setupUdprimFile(MODL, 0, fp_csm, filename, &linenum, str5);
                        CHECK_STATUS2(setupUdprimFile);
                    }

                    ifound = 0;
                    for (i = 0; i < udp_num; i++) {
                        if (strcasecmp(&str6[1], udp_names[i]) == 0 && udp_types[i] == ATTRFILE) {
                            status = setupUdprimFile(MODL, 1, fp_csm, filename, &linenum, str7);
                            CHECK_STATUS2(setupUdprimFile);

                            ifound = 1;
                            break;
                        }
                    }
                    if (ifound == 0) {
                        status = setupUdprimFile(MODL, 0, fp_csm, filename, &linenum, str7);
                        CHECK_STATUS2(setupUdprimFile);
                    }

                    ifound = 0;
                    for (i = 0; i < udp_num; i++) {
                        if (strcasecmp(&str8[1], udp_names[i]) == 0 && udp_types[i] == ATTRFILE) {
                            status = setupUdprimFile(MODL, 1, fp_csm, filename, &linenum, str9);
                            CHECK_STATUS2(setupUdprimFile);

                            ifound = 1;
                            break;
                        }
                    }
                    if (ifound == 0) {
                        status = setupUdprimFile(MODL, 0, fp_csm, filename, &linenum, str9);
                        CHECK_STATUS2(setupUdprimFile);
                    }
                }
            }

            /* create the new Branch */
            if        (narg < 3) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPARG, filename, linenum,
                                     str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            } else if (narg < 5) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPARG, filename, linenum,
                                     str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            } else if (narg < 7) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPARG, filename, linenum,
                                     str1, str2, str3, str4, str5, NULL, NULL, NULL, NULL);
            } else if (narg < 9) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPARG, filename, linenum,
                                     str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
            } else {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPARG, filename, linenum,
                                     str1, str2, str3, str4, str5, str6, str7, str8, str9);
            }
            CHECK_STATUS2(ocsmNewBrch);

        /* input is: "udprim $primtype $argName1 argValue1 $argName2 argValue2 $argName3 argValue3 $argName4 argValue4" */
        } else if (strcmp(command, "udprim") == 0 ||
                   strcmp(command, "UDPRIM") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "UDPRIM cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "UDPRIM cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            strcpy(str1, "$");  strcpy(str2, "$");  strcpy(str4, "$");  strcpy(str6, "$");  strcpy(str8, "$");
            narg = sscanf(nextline, "%*s %2046s %2046s %2047s %2046s %2047s %2046s %2047s %2046s %2047s\n",
                          &(str1[1]), &(str2[1]), str3, &(str4[1]), str5,
                                      &(str6[1]), str7, &(str8[1]), str9);
            if (narg < 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "UDPRIM requires at least 1 argument");
                goto cleanup;
            } else if (narg%2 != 1) {
                status = signalError2(MODL, OCSM_NOT_ENOUGH_ARGS, filename, linenum,
                                      "UDPRIM requires 1, 3, 5, 7, or 9 arguments");
                goto cleanup;
            }

            if (narg < 3) str3[0] = '\0';
            if (narg < 5) str5[0] = '\0';
            if (narg < 7) str7[0] = '\0';
            if (narg < 9) str9[0] = '\0';

            /* get the arguments associated with the UDP/UDF */
            if (isalpha(str1[1]) != 0) {
                status = udp_initialize(&str1[1], MODL, &udp_num, &udp_names, &udp_types, &udp_idef, &udp_ddef);

                /* if we got an error, then we will not process any filename-like arguments */
                if (status != EGADS_SUCCESS) {

                /* adjust file spec or extract "inline" file */
                } else {
                    SPLINT_CHECK_FOR_NULL(fp_csm);

                    ifound = 0;
                    for (i = 0; i < udp_num; i++) {
                        if (strcasecmp(&str2[1], udp_names[i]) == 0 && udp_types[i] == ATTRFILE) {
                            status = setupUdprimFile(MODL, 1, fp_csm, filename, &linenum, str3);
                            CHECK_STATUS2(setupUdprimFile);

                            ifound = 1;
                            break;
                        }
                    }
                    if (ifound == 0) {
                        status = setupUdprimFile(MODL, 0, fp_csm, filename, &linenum, str3);
                        CHECK_STATUS2(setupUdprimFile);
                    }

                    ifound = 0;
                    for (i = 0; i < udp_num; i++) {
                        if (strcasecmp(&str4[1], udp_names[i]) == 0 && udp_types[i] == ATTRFILE) {
                            status = setupUdprimFile(MODL, 1, fp_csm, filename, &linenum, str5);
                            CHECK_STATUS2(setupUdprimFile);

                            ifound = 1;
                            break;
                        }
                    }
                    if (ifound == 0) {
                        status = setupUdprimFile(MODL, 0, fp_csm, filename, &linenum, str5);
                        CHECK_STATUS2(setupUdprimFile);
                    }

                    ifound = 0;
                    for (i = 0; i < udp_num; i++) {
                        if (strcasecmp(&str6[1], udp_names[i]) == 0 && udp_types[i] == ATTRFILE) {
                            status = setupUdprimFile(MODL, 1, fp_csm, filename, &linenum, str7);
                            CHECK_STATUS2(setupUdprimFile);

                            ifound = 1;
                            break;
                        }
                    }
                    if (ifound == 0) {
                        status = setupUdprimFile(MODL, 0, fp_csm, filename, &linenum, str7);
                        CHECK_STATUS2(setupUdprimFile);
                    }

                    ifound = 0;
                    for (i = 0; i < udp_num; i++) {
                        if (strcasecmp(&str8[1], udp_names[i]) == 0 && udp_types[i] == ATTRFILE) {
                            status = setupUdprimFile(MODL, 1, fp_csm, filename, &linenum, str9);
                            CHECK_STATUS2(setupUdprimFile);

                            ifound = 1;
                            break;
                        }
                    }
                    if (ifound == 0) {
                        status = setupUdprimFile(MODL, 0, fp_csm, filename, &linenum, str9);
                        CHECK_STATUS2(setupUdprimFile);
                    }
                }
            }

            /* create the new Branch */
            if        (narg < 3) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPRIM, filename, linenum,
                                     str1, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
            } else if (narg < 5) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPRIM, filename, linenum,
                                     str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            } else if (narg < 7) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPRIM, filename, linenum,
                                     str1, str2, str3, str4, str5, NULL, NULL, NULL, NULL);
            } else if (narg < 9) {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPRIM, filename, linenum,
                                     str1, str2, str3, str4, str5, str6, str7, NULL, NULL);
            } else {
                status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UDPRIM, filename, linenum,
                                     str1, str2, str3, str4, str5, str6, str7, str8, str9);
            }
            CHECK_STATUS2(ocsmNewBrch);

            /* if in a .cpc file, increment the number of UDCs if str1 starts with "$" or "/" */
            if (filetype == 1) {
                if (str1[1] == '$' || str1[1] == '/') {
                    numudc++;
                }
            }

        /* input is: "union toMark=0 trimList=0 maxtol=0" */
        } else if (strcmp(command, "union") == 0 ||
                   strcmp(command, "UNION") == 0   ) {
            if (nskpt > 0) {
                status = signalError2(MODL, OCSM_SKETCH_IS_OPEN, filename, linenum,
                                      "UNION cannot be in SKBEG/SKEND");
                goto cleanup;
            } else if (insolver != 0) {
                status = signalError2(MODL, OCSM_SOLVER_IS_OPEN, filename, linenum,
                                      "UNION cannot be in SOLBEG/SOLEND");
                goto cleanup;
            }

            /* extract arguments */
            narg = sscanf(nextline, "%*s %2047s %2047s %2047s\n",
                          str1, str2, str3);
            if (narg < 1) {
                STRNCPY(str1, "0", MAX_EXPR_LEN);
            }
            if (narg < 2) {
                STRNCPY(str2, "0", MAX_EXPR_LEN);
            }
            if (narg < 3) {
                STRNCPY(str3, "0", MAX_EXPR_LEN);
            }

            /* create the new Branch */
            status = ocsmNewBrch(MODL, MODL->ibrch, OCSM_UNION, filename, linenum,
                                 str1, str2, str3, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS2(ocsmNewBrch);

        /* illegal command */
        } else {
            status = signalError2(MODL, OCSM_ILLEGAL_STATEMENT, filename, linenum,
                                  "unrecognized command \"%s\"", command);
            goto cleanup;
        }
    }

    /* remove all UNKNOWN Parameters, no matter its scope (since they were created by a DIMENSION
       statement in preparation for a DESPMTR, CFGPMTR, or OUTPMTR statement) */
    if (filetype == -1 || filetype == 0 || filetype == 1) {
        for (ipmtr = MODL->npmtr; ipmtr > 0; ipmtr--) {
            SPLINT_CHECK_FOR_NULL(MODL->pmtr);

            if (MODL->pmtr[ipmtr].type == OCSM_UNKNOWN) {
                MODL->pmtr[ipmtr].scope = MODL->scope[MODL->level];
                status = ocsmDelPmtr(MODL, ipmtr);
                CHECK_STATUS2(ocsmDelPmtr);
            }
        }
    }

cleanup:
#undef CHECK_STATUS2
    if (MODL != NULL) {
        if (status == 0) {
            status = MODL->sigCode;
        }

        /* close the .csm file */
        if (fp_csm != NULL) fclose(fp_csm);
    }

    FREE(str9);
    FREE(str8);
    FREE(str7);
    FREE(str6);
    FREE(str5);
    FREE(str4);
    FREE(str3);
    FREE(str2);
    FREE(str1);

    FREE(str );

    FREE(command );
    FREE(nextline);
    FREE(templine);

    return status;
}


/***********************************************************************/
/*                                                                     */
/*   ocsmLoadFromModel - create a MODL from an ego MODEL               */
/*                                                                     */
/***********************************************************************/

int
ocsmLoadFromModel(ego    emodel,        /* egads MODEL */
                  void   **modl)        /* (out) pointer to MODL */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL=NULL;

    int    i, oclass1, mtype1, oclass2, mtype2, nchild, *senses, ichild, nnode;
    int    ibody, okay, stat, npts;
    double data[4];
    ego    eref, *echilds, ebody;

    ROUTINE(ocsmLoadFromModel);

    /* --------------------------------------------------------------- */

    SPRINT0(1, "--> enter ocsmLoadFromModel()");

    /* make a new MODL and initialize it */
    MALLOC(MODL, modl_T, 1);

    MODL->magic      = OCSM_MAGIC;
    MODL->checked    = 0;
    MODL->embedded   = 0;
    MODL->ibrch      = 0;
    MODL->nextseq    = 1;
    MODL->ngroup     = 0;
    MODL->recycle    = 0;
    MODL->verify     = 0;
    MODL->cleanup    = 25;
    MODL->dumpEgads  = 0;
    MODL->loadEgads  = 0;
    MODL->hasMPs     = 0;
    MODL->printStack = 0;
    MODL->printAttr[0] = '\0';
    MODL->printPmtrs = 0;
    MODL->printStors = 0;
    MODL->tessAtEnd  = 1;
    MODL->erepAtEnd  = 0;
    MODL->bodyLoaded = 0;

    MODL->seltype = -1;
    MODL->selbody = -1;
    MODL->selsize =  0;
    MODL->sellist = NULL;

    MODL->level = 0;

    for (i = 0; i < 11; i++) {
        MODL->scope[i] = 0;
    }

    MODL->filename[0] = '\0';

    MODL->tmpDirNum = -1;

    MODL->ninline = 0;
    MODL->minline = 0;
    MODL->sinline = NULL;

    MODL->nattr = 0;
    MODL->attr  = NULL;

    MODL->nstor = 0;
    MODL->stor  = NULL;

    MODL->nbrch = 0;
    MODL->mbrch = 0;
    MODL->brch  = NULL;

    MODL->npmtr = 0;
    MODL->mpmtr = 0;
    MODL->pmtr  = NULL;
    MODL->prefix[0] ='\0';

    MODL->nbody = 0;
    MODL->mbody = 0;
    MODL->body  = NULL;

    MODL->forceFDs   = 0;
    MODL->numdots    = 0;
    MODL->perturb    = NULL;
    MODL->basemodl   = NULL;
    MODL->matchSeq   = NULL;
    MODL->dtime      = 0;

    status = EG_getContext(emodel, &(MODL->context));
    CHECK_STATUS(EG_getContext);

    MODL->userdata    = NULL;
    MODL->mesgCB      = NULL;
    MODL->bcstCB      = NULL;
    MODL->sizeCB      = NULL;

    MODL->eggname[0]  = '\0';
    MODL->eggGenerate = NULL;
    MODL->eggMorph    = NULL;
    MODL->eggInfo     = NULL;
    MODL->eggDump     = NULL;
    MODL->eggLoad     = NULL;
    MODL->eggFree     = NULL;

    for (i = 0; i < MAXPRIM; i++) {
        MODL->NumUdp[i] = 0;
        MODL->Udps[  i] = NULL;
    }

    MODL->nwarn   = 0;
    MODL->sigCode = 0;
    MODL->sigMesg = NULL;
    MALLOC(MODL->sigMesg, char, MAX_STR_LEN);
    MODL->sigMesg[0] = '\0';

    for (i = 0; i < 101; i++) {
        MODL->profile[i].ncall = 0;
        MODL->profile[i].time  = 0;
    }

    /* return value */
    *modl = MODL;

    /* get info from emodel */
    status = EG_getTopology(emodel, &eref, &oclass1, &mtype1,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass1 != MODEL) {
        status = signalError2(MODL, OCSM_UNSUPPORTED, "<emodel>", 0,
                              "emodel is not a MODEL");
        goto cleanup;
    }

    /* loop through the children of the MODEL */
    for (ichild = 0; ichild < MAX(nchild,mtype1); ichild++) {
        status = EG_getInfo(echilds[ichild], &oclass2, &mtype2, NULL, NULL, NULL);
        CHECK_STATUS(EG_getInfo);

        if (oclass2 == BODY) {
            if        (mtype2 == SOLIDBODY) {
                status = newBody(MODL, 0, 0, -1, -1, NULL, 0, OCSM_SOLID_BODY, &ibody);
                CHECK_STATUS(newBody);
            } else if (mtype2 == SHEETBODY) {
                status = newBody(MODL, 0, 0, -1, -1, NULL, 0, OCSM_SHEET_BODY, &ibody);
                CHECK_STATUS(newBody);
            } else {
                status = EG_getBodyTopos(echilds[ichild], NULL, NODE, &nnode, NULL);
                CHECK_STATUS(EG_getBodyTopos);

                if (nnode > 1) {
                    status = newBody(MODL, 0, 0, -1, -1, NULL, 0, OCSM_WIRE_BODY, &ibody);
                    CHECK_STATUS(newBody);
                } else {
                    status = newBody(MODL, 0, 0, -1, -1, NULL, 0, OCSM_NODE_BODY, &ibody);
                    CHECK_STATUS(newBody);
                }
            }

            SPLINT_CHECK_FOR_NULL(MODL->body);

            status = EG_copyObject(echilds[ichild], NULL, &ebody);
            CHECK_STATUS(EG_copyObject);

            MODL->body[ibody].ebody = ebody;

            status = EG_attributeAdd(ebody, "__removeAttributes__", ATTRSTRING,
                                     1, NULL, NULL, "y");
            CHECK_STATUS(EG_attributeAdd);

            status = finishBody(MODL, ibody);
            CHECK_STATUS(finishBody);

            MODL->body[ibody].onstack = 1;

        } else if (oclass2 == EBODY) {
            status = signalError2(MODL, OCSM_UNSUPPORTED, "<emodel>", 0,
                                  "emodel cannot contain an EBODY");
            goto cleanup;

        } else if (oclass2 == TESSELLATION) {
            SPLINT_CHECK_FOR_NULL(MODL->body);

            status = EG_statusTessBody(echilds[ichild], &ebody, &stat, &npts);
            CHECK_STATUS(EG_statusTessBody);

            okay = 0;
            for (ibody = 0; ibody < nchild; ibody++) {
                if (echilds[ibody] == ebody) {
                    status = EG_copyObject(echilds[ichild], MODL->body[ibody+1].ebody,
                                           &(MODL->body[ibody+1].etess));
                    CHECK_STATUS(EG_copyObject);
                    okay = 1;
                    break;
                }
            }

            if (okay == 0) {
                status = signalError2(MODL, OCSM_UNSUPPORTED, "<emodel>", 0,
                                      "TESSELLATION found that does not link to a BODY");
                goto cleanup;
            }
        }
    }

    SPLINT_CHECK_FOR_NULL(MODL->body);

    /* create default tessellations for any Body that does not already have a tessellation */
    for (ibody = 1; ibody <= MODL->nbody; ibody++) {
        if (MODL->body[ibody].etess == NULL) {
            status = ocsmTessellate(MODL, ibody);
            CHECK_STATUS(ocsmTessellate);
        }
    }


cleanup:
    return status;
}


/***********************************************************************/
/*                                                                     */
/*   ocsmLoadDict - load dictionary from dictname                      */
/*                                                                     */
/***********************************************************************/

int
ocsmLoadDict(void   *modl,              /* (in)  pointer to MODL */
             char   dictname[])         /* (in)  file that contains dictionary */
{
    int       status = SUCCESS;         /* (out) return status */

    int       jpmtr, ipmtr, nrow, irow, ncol, icol, index;
    double    *vals=NULL, *dots=NULL;
    char      pmtrName[MAX_EXPR_LEN], pmtrDefn[MAX_EXPR_LEN], str[MAX_EXPR_LEN];
    FILE      *dict_fp=NULL;

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmLoadDict);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* if there was a dictionary, load the definitions from it now */
    if (STRLEN(dictname) <= 0) goto cleanup;

    dict_fp = fopen(dictname, "r");
    if (dict_fp == NULL) {
        SPRINT1(0, "ERROR:: dictionary \"%s\" not found", dictname);
        status = OCSM_FILE_NOT_FOUND;
        goto cleanup;
    }

    SPRINT1(0, "--> Opening dictionary \"%s\"", dictname);

    while (1) {
        status = fscanf(dict_fp, "%s %s\n", pmtrName, pmtrDefn);
        if (status < 2) break;

        /* make sure we have a good Parameter name */
        SPRINT2(0, "    defining constant %-20s = %s", pmtrName, pmtrDefn);

        if (pmtrName[0] == '@') {
            SPRINT0(0, "ERROR:: constant name cannot start with @");
            status = OCSM_ILLEGAL_PMTR_NAME;
            goto cleanup;
        }

        /* parse the value(s) */
        status = str2vals(pmtrDefn, MODL, &nrow, &ncol, &vals, &dots, str);
        if (status < SUCCESS) {
            SPRINT1(0, "ERROR:: constant value(s) \"%s\"could not be parsed", pmtrDefn);
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        } else if (strlen(str) > 0) {
            SPRINT1(0, "ERROR:: constant value(s)\"%s\" must not be a string", pmtrDefn);
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }
        SPLINT_CHECK_FOR_NULL(vals);
        SPLINT_CHECK_FOR_NULL(dots);

        /* see if a CONPMTR already exists and if it matches the current definition */
        ipmtr = -1;
        for (jpmtr = 1; jpmtr <= MODL->npmtr; jpmtr++) {
            if (strcmp(MODL->pmtr[jpmtr].name, pmtrName) == 0) {
                if (MODL->pmtr[jpmtr].type != OCSM_CONPMTR ||
                    MODL->pmtr[jpmtr].nrow != nrow         ||
                    MODL->pmtr[jpmtr].ncol != ncol           ) {
                    SPRINT1(0, "ERROR:: \"%s\" is already defined and does not match file", pmtrName);
                    status = OCSM_NAME_ALREADY_DEFINED;
                    goto cleanup;
                }

                index = 0;
                for (irow = 1; irow <= nrow; irow++) {
                    for (icol = 1; icol <= ncol; icol++) {
                        if (fabs(MODL->pmtr[jpmtr].value[index]-vals[index]) > EPS06) {
                            SPRINT5(0, "ERROR:: for %s[%d,%d], new value (%10.5f) and old value (%10.5f) do not agree",
                                    pmtrName, irow, icol, MODL->pmtr[jpmtr].value[index], vals[index]);
                            status = OCSM_ILLEGAL_VALUE;
                            goto cleanup;
                        }
                        index++;
                    }
                }
                ipmtr = jpmtr;
                break;
            }
        }

        /* we found a CONPMTR that matches, so read next line of file */
        if (ipmtr > 0) {
            FREE(vals);
            FREE(dots);
            continue;
        }

        /* create a CONPMTR and set its values */
        status = ocsmNewPmtr(MODL, pmtrName, OCSM_CONPMTR, nrow, ncol);
        if (status != SUCCESS) {
            SPRINT1(0, "ERROR:: problem creating \"%s\"", pmtrName);
            status = OCSM_ILLEGAL_PMTR_NAME;
            goto cleanup;
        }

        for (index = 0; index < nrow*ncol; index++) {
            MODL->pmtr[MODL->npmtr].value[index] = vals[index];
            MODL->pmtr[MODL->npmtr].dot[  index] = dots[index];
        }

        FREE(vals);
        FREE(dots);
    }

    /* getting here means that we succeeded */
    status = SUCCESS;

cleanup:
    if (dict_fp != NULL) fclose(dict_fp);

    FREE(vals);
    FREE(dots);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *  ocsmAdjustUDCs - adjust UDCs to be colocated with the .csm file     *
 *                                                                      *
 ************************************************************************
 */

int
ocsmAdjustUDCs(void   *modl)            /* (in)  pointer to MODL */
{
    int       status = SUCCESS;         /* (out) return status */

    int       ibrch, jbrch, seen, i;
    char      csmFilename[MAX_FILENAME_LEN+2], udcFilename[MAX_FILENAME_LEN+2];
    char      ext[5], buf[MAX_LINE_LEN];
    FILE      *fp_src=NULL, *fp_tgt=NULL;
    modl_T    *MODL = (modl_T *)modl;

    ROUTINE(ocsmAdjustUDCs);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    csmFilename[0] = '\0';
    udcFilename[0] = '\0';

    /* copy all .udc files into the same directory as the .csm file */
    for (ibrch = 1;  ibrch <= MODL->nbrch; ibrch++) {
        STRNCPY(ext, &(MODL->brch[ibrch].filename[strlen(MODL->brch[ibrch].filename)-4]), 4);

        /* .csm file */
        if (strcmp(ext, ".csm") == 0) {
            if (strlen(csmFilename) == 0) {
                STRNCPY(csmFilename, MODL->brch[ibrch].filename, MAX_FILENAME_LEN);
            } else if (strcmp(csmFilename, MODL->brch[ibrch].filename) != 0) {
                SPRINT3(0, "ERROR:: \"%s\" and \"%s\" differ for ibrch=%d",
                        csmFilename, MODL->brch[ibrch].filename, ibrch);
                status = OCSM_NAME_ALREADY_DEFINED;
                goto cleanup;
            }

        /* .udc file */
        } else if (strcmp(ext, ".udc") == 0) {

            /* check if we have seen ths file before */
            seen = 0;
            for (jbrch = 1; jbrch < ibrch; jbrch++) {
                if (strcmp(MODL->brch[jbrch].filename, MODL->brch[ibrch].filename) == 0) {
                    seen = 1;
                    break;
                }
            }

            /* no further processing if we have seen this filename before */
            if (seen == 1) continue;

            /* make the name of the new UDC file as the part of the .csm
               filename up to the last slash, followed by the part of the
               Branch's .udc filename after the last slash */
            STRNCPY(udcFilename, csmFilename, MAX_FILENAME_LEN);

            seen = 0;
            for (i = strlen(udcFilename)-1; i >= 0; i--) {
                if (udcFilename[i] == SLASH) {
                    udcFilename[i+1] = '\0';
                    seen = 1;
                    break;
                }
            }
            if (seen == 0) {
                udcFilename[0] = '\0';
            }

            seen = 0;
            for (i = strlen(MODL->brch[ibrch].filename)-1; i >= 0; i--) {
                if (MODL->brch[ibrch].filename[i] == SLASH) {
                    STRNCAT(udcFilename, &(MODL->brch[ibrch].filename[i+1]), MAX_FILENAME_LEN);
                    seen = 1;
                    break;
                }
            }
            if (seen == 0) {
                STRNCAT(udcFilename, MODL->brch[ibrch].filename, MAX_FILENAME_LEN);
            }

            /* copy the file */
            if (strcmp(udcFilename, MODL->brch[ibrch].filename) == 0) {
                SPRINT2(0, "WARNING:: skipping copy of \"%s\" to \"%s\"", MODL->brch[ibrch].filename, udcFilename);
                continue;
            }

            fp_src = fopen(MODL->brch[ibrch].filename, "r");
            if (fp_src == NULL) {
                SPRINT2(0, "ERROR:: \"%s\" not found while processing ibrch=%d",
                        MODL->brch[ibrch].filename, ibrch);
                status = OCSM_FILE_NOT_FOUND;
                goto cleanup;
            }

            fp_tgt = fopen(udcFilename, "w");
            if (fp_tgt == NULL) {
                SPRINT2(0, "ERROR:: \"%s\" cannot be created while processing ibrch=%d",
                        udcFilename, ibrch);
                status = OCSM_FILE_NOT_FOUND;
                goto cleanup;
            }

            while (fgets(buf, MAX_LINE_LEN, fp_src) != NULL) {
                fputs(buf, fp_tgt);
            }

            fclose(fp_src);   fp_src = NULL;
            fclose(fp_tgt);   fp_tgt = NULL;

        /* unknown filetype */
        } else {
            SPRINT2(0, "ERROR:: \"%s\" at ibrch=%d is neither .csm nor .udc",
                    MODL->brch[ibrch].filename, ibrch);
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }
    }

    /* adjust the Branches to point to the new .udc locations */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {

        /* adjust filename entry */
        if (strstr(MODL->brch[ibrch].filename, ".udc") != NULL) {

            /* make the name of the new UDC file as the part of the .csm
               filename up to the last slash, followed by the part of the
               Branch's .udc filename after the last slash */
            STRNCPY(udcFilename, csmFilename, MAX_FILENAME_LEN);

            seen = 0;
            for (i = strlen(udcFilename)-1; i >= 0; i--) {
                if (udcFilename[i] == SLASH) {
                    udcFilename[i+1] = '\0';
                    seen = 1;
                    break;
                }
            }
            if (seen == 0) {
                udcFilename[0] = '\0';
            }

            seen = 0;
            for (i = strlen(MODL->brch[ibrch].filename)-1; i >= 0; i--) {
                if (MODL->brch[ibrch].filename[i] == SLASH) {
                    STRNCAT(udcFilename, &(MODL->brch[ibrch].filename[i+1]), MAX_FILENAME_LEN);
                    seen = 1;
                    break;
                }
            }
            if (seen == 0) {
                STRNCAT(udcFilename, MODL->brch[ibrch].filename, MAX_FILENAME_LEN);
            }

            strcpy(MODL->brch[ibrch].filename, udcFilename);
        }

        /* adjust primtype associated with UDPARG or UDPRIM */
        SPLINT_CHECK_FOR_NULL(buf);

        if (MODL->brch[ibrch].type == OCSM_UDPARG ||
            MODL->brch[ibrch].type == OCSM_UDPRIM   ) {

            /* original .udc in pwd */
            if        (strncmp(MODL->brch[ibrch].arg1, "$/",   2) == 0) {
                STRNCPY(buf, MODL->brch[ibrch].arg1, MAX_LINE_LEN);
                snprintf(MODL->brch[ibrch].arg1, MAX_LINE_LEN, "$$/%s", &(buf[2]));

            /* original .udc in system directory */
            } else if (strncmp(MODL->brch[ibrch].arg1, "$$$/", 4) == 0) {
                STRNCPY(buf, MODL->brch[ibrch].arg1, MAX_LINE_LEN);
                snprintf(MODL->brch[ibrch].arg1, MAX_LINE_LEN, "$$/%s", &(buf[4]));

            /* original .udc in ESP_UDC_PATH directory */
            } else if (strncmp(MODL->brch[ibrch].arg1, "$$$$/", 5) == 0) {
                STRNCPY(buf, MODL->brch[ibrch].arg1, MAX_LINE_LEN);
                snprintf(MODL->brch[ibrch].arg1, MAX_LINE_LEN, "$$$/%s", &(buf[5]));
            }
        }
    }

cleanup:
    if (fp_src != NULL) fclose(fp_src);
    if (fp_tgt != NULL) fclose(fp_tgt);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *  ocsmUpdateDespmtrs - update CFGPMTRs and DESPMTRs from filename     *
 *                                                                      *
 ************************************************************************
 */

int
ocsmUpdateDespmtrs(void   *modl,        /* (in)  pointer to MODL */
                   char   filename[])   /* (in)  file that contains DESPMTRs */
{
    int       status = SUCCESS;         /* (out) return status */

    int       ipmtr, irow, icol, i, j, nin, nvals, ndum, nrow, ncol, index;
    int       ibrch, nrow_new, ncol_new, index_new;
    double    vals_new, dots_new, *vals=NULL, *dots=NULL;
    double    *value_new=NULL, *lbnd_new=NULL, *ubnd_new=NULL, *dot_new=NULL;
    char      pname[MAX_NAME_LEN], pmtrname[MAX_NAME_LEN], str[MAX_EXPR_LEN];
    char      prefix[MAX_NAME_LEN], templine[MAX_LINE_LEN];
    char      tempname1[MAX_NAME_LEN];
    char      name[MAX_NAME_LEN], values[MAX_LINE_LEN], str_new[MAX_STRVAL_LEN];
    FILE      *fp=NULL;
    void      *temp;

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmUpdateDespmtrs);

    /* --------------------------------------------------------------- */

    /*
      valid statments are of the form:
         name value(s)
      where "name" can be in the form:
         name             (for the whole array)
         name[index]      (for an element of an array)
         name[irow,icol]  (for an element of an array)
         name[irow,:]     (for the irow'th row of an array)
         name[:,icol]     (for the icol'th column of an array)
         @prefix          (to set   the prefix --- values not needed)
         @.               (to clear the prefix --- values not needed)
      and where "value(s)" is a semi-colon-separated set of numbers
      if the name begins with a colon, the prefix is prepended
      if fewer values are prescribed than are needed, the last value
         is repeated
      if more values are prescribed than are needed, the extra values
         are discarded
      the hash symbol (#) introduces a comment and everything starting
         at the hash is ignored
      blank lines are ignored
      CFGPMTRs and DESPMTRs are resized if a CFGPMTR or DESPMTR that is
         used in a DIMENSION statement is changed, where rows are
         resized before columns
    */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* clear the prefix */
    MODL->prefix[0] = '\0';

    /* if there was a file, update the DESPMTRs mentioned in it now */
    if (STRLEN(filename) <= 0) goto cleanup;

    fp = fopen(filename, "r");
    if (fp == NULL) {
        SPRINT1(0, "ERROR:: DESPMTR file \"%s\" not found", filename);
        status = OCSM_FILE_NOT_FOUND;
        goto cleanup;
    }

    SPRINT1(0, "--> Opening DESPMTR file \"%s\"", filename);

    /* initially there is no prefix */
    prefix[0] = '\0';

    /* read until we gat to the end of file */
    while (fp != NULL) {

        /* read the next line */
        temp = fgets(templine, MAX_LINE_LEN, fp);

        /* check for end of file */
        if (temp == NULL) {
            fclose(fp);
            fp = NULL;

            break;
        }

        SPRINT1x(1, "    processing: %s", templine);

        /* ignore spaces at the beginning of the line */
        for (i = 0; i < STRLEN(templine); i++) {
            if (templine[0] == ' ') {
                for (j = 0; j < STRLEN(templine); j++) {
                    templine[j] = templine[j+1];
                }
            }
        }

        /* make sure that there are no | characters */
        for (j = 0; j < STRLEN(templine); j++) {
            if (templine[j] == '|') {
                status = signalError(MODL, OCSM_ILLEGAL_STATEMENT,
                                      "the bar character (|) cannot be used");
                goto cleanup;
            }
        }

        /* ignore anything after a # */
        for (i = 0; i < STRLEN(templine); i++) {
            if (templine[i] == '#') {
                templine[i] = '\0';
                break;
            }
        }

        /* read the name and value(s) if they exist */
        name[  0] = '\0';
        values[0] = '\0';
        nin = sscanf(templine, "%63s %2047s", name, values);

        /* skip blank lines */
        if (nin < 1) continue;

        /* adjust prefix if name starts with an at-sign */
        if (name[0] == '@') {
            if (name[1] == '.') {
                prefix[0] = '\0';
            } else {
                strncpy(prefix, &name[1], MAX_NAME_LEN);
            }
            continue;
        }

        /* make sure we have value(s) */
        if (nin < 2) {
            status = signalError(MODL, OCSM_ILLEGAL_STATEMENT,
                                 "value(s) are required");
            goto cleanup;
        }

        /* pull the name and indices out of name */
        status = parseName(modl, name, pname, &ipmtr, &irow, &icol);
        CHECK_STATUS(parseName);

        /* prepend the prefix */
        if (pname[0] != '$') {
            strncpy(pmtrname, &(pname[0]), MAX_NAME_LEN);
            if (strlen(prefix) > 0 && pname[0] == ':') {
                snprintf(pmtrname, MAX_NAME_LEN, "%s%s", prefix, &(pname[0]));
            }
        } else {
            strncpy(pmtrname, &(pname[1]), MAX_NAME_LEN);
            if (strlen(prefix) > 0 && pname[1] == ':') {
                snprintf(pmtrname, MAX_NAME_LEN, "%s%s", prefix, &(pname[1]));
            }
        }

        /* determine the Parameter index */
        status = ocsmFindPmtr(MODL, pmtrname, 0, 0, 0, &ipmtr);
        CHECK_STATUS(ocsmFindPmtr);

        /* make sure we are modifying a DESPMTR or CFGPMTR */
        if (MODL->pmtr[ipmtr].type != OCSM_DESPMTR &&
            MODL->pmtr[ipmtr].type != OCSM_CFGPMTR   ) {
            status = signalError(MODL, OCSM_WRONG_PMTR_TYPE,
                                 "not a DESPMTR or CFGPMTR");
            goto cleanup;
        }

        /* extract the values that were specified */
        status = str2vals(values, modl, &ndum, &nvals, &vals, &dots, str);
        CHECK_STATUS(str2vals);

        SPLINT_CHECK_FOR_NULL(vals);
        SPLINT_CHECK_FOR_NULL(dots);

        /* get the number of rows and column in ipmtr */
        nrow = MODL->pmtr[ipmtr].nrow;
        ncol = MODL->pmtr[ipmtr].ncol;

        /* if name is specified in the form "name", copy vals into
           the Parameter storage */
        if (irow == 0 && icol == 0) {
            for (irow = 1; irow <= nrow; irow++) {
                for (icol = 1; icol <= ncol; icol++) {
                    index = (icol-1) + (irow-1) * ncol;
                    if (index < nvals) {
                        if (MODL->pmtr[ipmtr].value[index] != vals[index]) {
                            MODL->pmtr[ipmtr].value[index] = vals[index];
                            SPRINT4(1, "       setting %s[%3d,%3d] to %12.6f",
                                    pmtrname, irow, icol, vals[index]);
                        }
                    } else {
                        if (MODL->pmtr[ipmtr].value[index] != vals[nvals-1]) {
                            MODL->pmtr[ipmtr].value[index] = vals[nvals-1];
                            SPRINT4(1, "       setting %s[%3d,%3d] to %12.6f",
                                    pmtrname, irow, icol, vals[nvals-1]);
                        }
                    }
                }
            }

        /* if name is specified in the form "name[index]", copy vals[0] into
           the Parameter storage */
        } else if (icol == 0) {
            if (irow <= 0 || irow > nrow*ncol) {
                status = signalError(MODL, OCSM_ILLEGAL_NUMBER,
                                     "index (%d) must not exceed %d", irow, nrow*ncol);
                goto cleanup;
            }

            index = irow - 1;
            if (MODL->pmtr[ipmtr].value[index] != vals[0]) {
                MODL->pmtr[ipmtr].value[index] = vals[0];
                SPRINT4(1, "       setting %s[%3d,%3d] to %12.6f",
                        pmtrname, irow, icol, vals[0]);
            }

        /* if name is specified in the form "name[irow,icol]", copy vals[0] into
           the Parameter storage */
        } else if (irow > 0 && icol > 0) {
            if (irow > nrow) {
                status = signalError(MODL, OCSM_ILLEGAL_NUMBER,
                                     "row (%d) must not exceed %d", irow, nrow);
                goto cleanup;
            } else if (icol > ncol) {
                status = signalError(MODL, OCSM_ILLEGAL_NUMBER,
                                     "column (%d) must not exceed %d", icol, ncol);
                goto cleanup;
            }

            index = (icol-1) + (irow-1) * ncol;
            if (MODL->pmtr[ipmtr].value[index] != vals[0]) {
                MODL->pmtr[ipmtr].value[index] = vals[0];
                SPRINT4(1, "       setting %s[%3d,%3d] to %12.6f",
                        pmtrname, irow, icol, vals[0]);
            }

        /* if name is specified in the form "name[:,icol]", copy vals into
           the icol'th column of the Parameter storage */
        } else if (irow == -999) {
            if (icol > ncol) {
                status = signalError(MODL, OCSM_ILLEGAL_NUMBER,
                                     "column (%d) must not exceed %d", icol, ncol);
                goto cleanup;
            }

            for (irow = 1; irow <= nrow; irow++) {
                index = (icol-1) + (irow-1) * ncol;
                if (irow-1 < nvals) {
                    if (MODL->pmtr[ipmtr].value[index] != vals[irow-1]) {
                        MODL->pmtr[ipmtr].value[index] = vals[irow-1];
                        SPRINT4(1, "       setting %s[%3d,%3d] to %12.6f",
                                pmtrname, irow, icol, vals[irow-1]);
                    }
                } else {
                    if (MODL->pmtr[ipmtr].value[index] != vals[nvals-1]) {
                        MODL->pmtr[ipmtr].value[index] = vals[nvals-1];
                        SPRINT4(1, "       setting %s[%3d,%3d] to %12.6f",
                                pmtrname, irow, icol, vals[nvals-1]);
                    }
                }
            }

        /* if name is specified in the form "name[irow,:]:, copy vals into
           the irow'th row of the Parameter storage */
        } else if (icol == -999) {
            if (irow > nrow) {
                status = signalError(MODL, OCSM_ILLEGAL_NUMBER,
                                     "row (%d) must not exceed %d", irow, nrow);
                goto cleanup;
            }

            for (icol = 1; icol <= ncol; icol++) {
                index = (icol-1) + (irow-1) * ncol;
                if (icol-1 < nvals) {
                    if (MODL->pmtr[ipmtr].value[index] != vals[icol-1]) {
                        MODL->pmtr[ipmtr].value[index] = vals[icol-1];
                        SPRINT4(1, "       setting %s[%3d,%3d] to %12.6f",
                                pmtrname, irow, icol, vals[icol-1]);
                    }
                } else {
                    if (MODL->pmtr[ipmtr].value[index] != vals[nvals-1]) {
                        MODL->pmtr[ipmtr].value[index] = vals[nvals-1];
                        SPRINT4(1, "       setting %s[%3d,%3d] to %12.6f",
                                pmtrname, irow, icol, vals[nvals-1]);
                    }
                }
            }

        /* invalid name */
        } else {
            status = -999;
            goto cleanup;
        }

        FREE(vals);
        FREE(dots);

        /* look through the DIMENSION statements associated with DESPMTRs and
           CFGPMTRs and update their shape if they refer to the Parameter
           that was just changed */
        for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
            if (MODL->brch[ibrch].type == OCSM_PREFIX) {
                if (MODL->brch[ibrch].arg1[0] == '.') {
                    strncpy(MODL->prefix, "", MAX_NAME_LEN);
                } else {
                    strncpy(MODL->prefix, &(MODL->brch[ibrch].arg1[1]), MAX_NAME_LEN);
                }
                continue;

            } else if (MODL->brch[ibrch].type != OCSM_DIMENSION) {
                continue;
            }

            strncpy(tempname1, &(MODL->brch[ibrch].arg1[1]), MAX_NAME_LEN);
            if (strlen(prefix) > 0 && MODL->brch[ibrch].arg1[1] == ':') {
                snprintf(tempname1, MAX_NAME_LEN, "%s%s", prefix, &(MODL->brch[ibrch].arg1[1]));
            }

            status = ocsmFindPmtr(MODL, tempname1, 0, 0, 0, &ipmtr);
            CHECK_STATUS(ocsmFindPmtr);

            if (MODL->pmtr[ipmtr].type != OCSM_DESPMTR &&
                MODL->pmtr[ipmtr].type != OCSM_CFGPMTR   ) continue;

            /* try to resize nrow first */
            nrow = MODL->pmtr[ipmtr].nrow;
            ncol = MODL->pmtr[ipmtr].ncol;

            status = str2val(MODL->brch[ibrch].arg2, MODL, &vals_new, &dots_new, str_new);
            CHECK_STATUS(str2val);

            nrow_new = NINT(vals_new);
            ncol_new = ncol;

            if (nrow_new != nrow) {
                SPRINT3(1, "          resizing %s to have %3d rows (from %3d)",
                        MODL->pmtr[ipmtr].name, nrow_new, nrow);

                /* make a new temporary array that is the correct size */
                MALLOC(value_new, double, nrow_new*ncol_new);
                MALLOC(lbnd_new,  double, nrow_new*ncol_new);
                MALLOC(ubnd_new,  double, nrow_new*ncol_new);
                MALLOC(dot_new,   double, nrow_new*ncol_new);

                /* copy the values into the temporary arrays */
                for (irow = 1; irow <= nrow_new; irow++) {
                    for (icol = 1; icol <= ncol; icol++) {
                        if (irow <= nrow) {
                            index     = (icol-1) + (irow-1) * ncol;
                            index_new = (icol-1) + (irow-1) * ncol_new;
                        } else {
                            index     = (icol-1) + (nrow-1) * ncol;
                            index_new = (icol-1) + (irow-1) * ncol_new;
                        }

                        value_new[index_new] = MODL->pmtr[ipmtr].value[index];
                        lbnd_new[ index_new] = MODL->pmtr[ipmtr].lbnd[ index];
                        ubnd_new[ index_new] = MODL->pmtr[ipmtr].ubnd[ index];
                        dot_new[  index_new] = 0;
                    }
                }

                /* put the temporary arrays into the Parameter storage */
                FREE(MODL->pmtr[ipmtr].value);
                FREE(MODL->pmtr[ipmtr].lbnd );
                FREE(MODL->pmtr[ipmtr].ubnd );
                FREE(MODL->pmtr[ipmtr].dot  );

                MODL->pmtr[ipmtr].nrow  = nrow_new;
                MODL->pmtr[ipmtr].ncol  = ncol_new;
                MODL->pmtr[ipmtr].value = value_new;
                MODL->pmtr[ipmtr].lbnd  = lbnd_new;
                MODL->pmtr[ipmtr].ubnd  = ubnd_new;
                MODL->pmtr[ipmtr].dot   = dot_new;

                value_new = NULL;
                lbnd_new  = NULL;
                ubnd_new  = NULL;
                dot_new   = NULL;
            }

            /* try to resize ncol second */
            nrow = MODL->pmtr[ipmtr].nrow;
            ncol = MODL->pmtr[ipmtr].ncol;

            status = str2val(MODL->brch[ibrch].arg3, MODL, &vals_new, &dots_new, str_new);
            CHECK_STATUS(str2val);

            nrow_new = nrow;
            ncol_new = NINT(vals_new);

            if (ncol_new != ncol) {
                SPRINT3(1, "          resizing %s to have %3d columns (from %3d)",
                        MODL->pmtr[ipmtr].name, ncol_new, ncol);

                /* make a new temporary array that is the correct size */
                MALLOC(value_new, double, nrow_new*ncol_new);
                MALLOC(lbnd_new,  double, nrow_new*ncol_new);
                MALLOC(ubnd_new,  double, nrow_new*ncol_new);
                MALLOC(dot_new,   double, nrow_new*ncol_new);

                /* copy the values into the temporary arrays */
                for (irow = 1; irow <= nrow; irow++) {
                    for (icol = 1; icol <= ncol_new; icol++) {
                        if (icol <= ncol) {
                            index     = (icol-1) + (irow-1) * ncol;
                            index_new = (icol-1) + (irow-1) * ncol_new;
                        } else {
                            index     = (ncol-1) + (irow-1) * ncol;
                            index_new = (icol-1) + (irow-1) * ncol_new;
                        }

                        value_new[index_new] = MODL->pmtr[ipmtr].value[index];
                        lbnd_new[ index_new] = MODL->pmtr[ipmtr].lbnd[ index];
                        ubnd_new[ index_new] = MODL->pmtr[ipmtr].ubnd[ index];
                        dot_new[  index_new] = 0;
                    }
                }

                /* put the temporary arrays into the Parameter storage */
                FREE(MODL->pmtr[ipmtr].value);
                FREE(MODL->pmtr[ipmtr].lbnd );
                FREE(MODL->pmtr[ipmtr].ubnd );
                FREE(MODL->pmtr[ipmtr].dot  );

                MODL->pmtr[ipmtr].nrow  = nrow_new;
                MODL->pmtr[ipmtr].ncol  = ncol_new;
                MODL->pmtr[ipmtr].value = value_new;
                MODL->pmtr[ipmtr].lbnd  = lbnd_new;
                MODL->pmtr[ipmtr].ubnd  = ubnd_new;
                MODL->pmtr[ipmtr].dot   = dot_new;

                value_new = NULL;
                lbnd_new  = NULL;
                ubnd_new  = NULL;
                dot_new   = NULL;
            }
        }

        /* reset the prefix */
        MODL->prefix[0] = '\0';
    }

    SPRINT0(1, " ");

    /* getting here means that we succeeded */
    status = SUCCESS;

cleanup:
    if (fp != NULL) fclose(fp);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetFilelist - get a list of all .csm, .cpc, and .udc file      *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetFilelist(void   *modl,           /* (in)  pointer to MODL */
                char   *filelist[])     /* (out) bar-separated list of files */
                                        /*       must be freed by user */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int        ibrch, lenFilelist=0;
    char       tmpFilename[MAX_FILENAME_LEN+2], *myFilelist=NULL;
    void       *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(ocsmGetFilelist);

    /* --------------------------------------------------------------- */

    /* initialize filelist */
    lenFilelist = MAX_FILENAME_LEN + 1000;
    MALLOC(myFilelist, char, lenFilelist);

    /* start with .csm or .cpc file */
    STRNCPY(myFilelist, MODL->filename, MAX_FILENAME_LEN);
    strcat(myFilelist, "|");

    /* look through all Branches and add its filename if not already in filelist */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        strcpy(tmpFilename, MODL->brch[ibrch].filename);
        strcat(tmpFilename, "|");

        if (strstr(myFilelist, tmpFilename) == NULL) {
            if (STRLEN(myFilelist)+STRLEN(MODL->brch[ibrch].filename)+2 > lenFilelist) {
                lenFilelist += 1000;
                RALLOC(myFilelist, char, lenFilelist);
            }

            strcat(myFilelist, tmpFilename);
        }
    }

    /* return the filelist (which must be freed by user) */
    *filelist = myFilelist;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetFiletree - get file tree info                               *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetFiletree(void   *modl,           /* (in)  pointer to MODL */
                char   *info[])         /* (out) info about the file tree */
                                        /*       must be freed by user */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       mchar, indent, ibrch, i;
    char      prefix[MAX_LINE_LEN+1], line[MAX_LINE_LEN+1];
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(ocsmGetFiletree);

    /* --------------------------------------------------------------- */

    /* initialize output buffer */
    mchar = 1000;
    MALLOC(*info, char, mchar);
    *info[0] = '\0';

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    snprintf(line, MAX_LINE_LEN, "Files used in previous build:\n");
    if (strlen(*info)+strlen(line) >= mchar) {
        mchar += strlen(line) + 1000;
        RALLOC(*info, char, mchar);
    }
    strcat(*info, line);

    /* initialize list */
    indent = 0;

    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        if (MODL->brch[ibrch].ileft == -2 &&
            MODL->brch[ibrch].irite == -2 &&
            MODL->brch[ibrch].ichld == -2   ) continue;

        if (ibrch == 1) {
            indent = 0;
        } else if (MODL->brch[ibrch].type == OCSM_END) {
            indent--;
            continue;
        } else if (MODL->brch[ibrch-1].type == OCSM_UDPRIM && strncmp(MODL->brch[ibrch-1].arg1, "$$", 2) == 0) {
            indent++;
        } else {
            continue;
        }

        prefix[0] = '\0';
        for (i = 0; i < indent; i++) {
            strncat(prefix, "    ", MAX_LINE_LEN);
        }

        snprintf(line, MAX_LINE_LEN, "%s[[%s:%d]]\n", prefix, MODL->brch[ibrch].filename, 1);
        if (strlen(*info)+strlen(line) >= mchar) {
            mchar += strlen(line) + 1000;
            RALLOC(*info, char, mchar);
        }
        strcat(*info, line);
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSave - save a MODL to a file                                   *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSave(void   *modl,                  /* (in)  pointer to MODL */
         char   filename[])             /* (in)  file to be written (with extension) */
                                        /*       .csm -> write outer .csm file */
                                        /*       .cpc -> write checkpointed .csm file */
                                        /*       .udc -> write a .udc file */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       filelen, filetype, i, j;
    int       ipmtr, jpmtr, ibrch, jbrch, kbrch=-1, iattr, irow, icol, index, indent, nindent=0, n_inline;
    char      udcname[MAX_EXPR_LEN], templine[MAX_LINE_LEN];
    FILE      *csm_file, *fp_inline;

    ROUTINE(ocsmSave);

    /* --------------------------------------------------------------- */

    SPRINT1(1, "--> enter ocsmSave(filename=%s)", filename);

    csm_file  = NULL;
    fp_inline = NULL;

    /* check for a valid extension */
    filelen = STRLEN(filename);
    if (filelen < 5) {
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    } else if (filename[filelen-4] == '.' && filename[filelen-3] == 'c' &&
               filename[filelen-2] == 's' && filename[filelen-1] == 'm'   ) {
        SPRINT0(2, "    writing .csm file");
        filetype = 0;
    } else if (filename[filelen-4] == '.' && filename[filelen-3] == 'c' &&
               filename[filelen-2] == 'p' && filename[filelen-1] == 'c'   ) {
        SPRINT0(2, "    writing .cpc file");
        filetype = 1;
    } else if (filename[filelen-4] == '.' && filename[filelen-3] == 'u' &&
               filename[filelen-2] == 'd' && filename[filelen-1] == 'c'   ) {
        SPRINT0(2, "    writing .udc file");
        filetype = 2;

        /* get udc name (part after last $ or / ) */
        j          = 0;
        udcname[j] = '\0';
        for (i = 0; i < STRLEN(filename)-4; i++) {
            if (filename[i] == '$' || filename[i] == '/') {
                j          = 0;
                udcname[j] = '\0';
            } else {
                udcname[j  ] = filename[i];
                udcname[j+1] = '\0';
                j++;
            }
        }

        /* find the UDPRIM statement that refers to udcname */
        kbrch = -1;
        for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
            if (       MODL->brch[ibrch].type           == OCSM_UDPRIM &&
                strstr(MODL->brch[ibrch].arg1, udcname) != NULL        &&
                (MODL->brch[ibrch].arg1[1] == '/' || MODL->brch[ibrch].arg1[1] == '$')) {
                kbrch = ibrch;
                break;
            }
        }
    } else {
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    }

    /* open file */
    if (STRLEN(filename) == 0) {
        SPRINT0(0, "ERROR:: filename not specified");
        status = OCSM_FILE_NOT_FOUND;
        goto cleanup;
    }

    csm_file = fopen(filename, "w");
    if (csm_file == NULL) {
        SPRINT1(0, "ERROR:: cannot open \"%s\"", filename);
        status = OCSM_FILE_NOT_FOUND;
        goto cleanup;
    }

    /* write header */
    fprintf(csm_file, "# %s written by ocsmSave (v%d.%02d)\n",
            filename, OCSM_MAJOR_VERSION, OCSM_MINOR_VERSION);

    /* write the constant, output, and design Parameters */
    if (filetype == 0 || filetype == 1) {
        fprintf(csm_file, "\n# Constant, Design, and Output Parameters:\n");
        for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
            if (MODL->pmtr[ipmtr].type  == OCSM_CONPMTR) {

                fprintf(csm_file, "CONPMTR   %s   %11.5f\n",
                        MODL->pmtr[ipmtr].name, MODL->pmtr[ipmtr].value[0]);
            } else if (MODL->pmtr[ipmtr].type  == OCSM_OUTPMTR) {

                fprintf(csm_file, "OUTPMTR   %s\n",
                        MODL->pmtr[ipmtr].name);
            } else if (MODL->pmtr[ipmtr].type  == OCSM_DESPMTR) {

                /* see if there is a DIMENSION statement associated with this DESPMTR */
                ibrch = 0;
                for (jbrch = 1; jbrch <= MODL->nbrch; jbrch++) {
                    if (MODL->brch[jbrch].type == OCSM_DIMENSION &&
                        strcmp(MODL->pmtr[ipmtr].name, &(MODL->brch[jbrch].arg1[1])) == 0) {
                        ibrch = jbrch;
                        break;
                    }
                }

                if (ibrch > 0) {
                    fprintf(csm_file, "DIMENSION %s   %s   %s\n",
                            &(MODL->brch[ibrch].arg1[1]),
                            MODL->brch[ibrch].arg2,
                            MODL->brch[ibrch].arg3);

                    index = 0;
                    for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                        fprintf(csm_file, "DESPMTR   %s[%d,:]   \"", MODL->pmtr[ipmtr].name, irow);
                        for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                            fprintf(csm_file, " %11.5f;", MODL->pmtr[ipmtr].value[index]);
                            index++;
                        }
                        fprintf(csm_file, "\"\n");
                    }
                } else if (MODL->pmtr[ipmtr].nrow > 1 || MODL->pmtr[ipmtr].ncol > 1) {
                    fprintf(csm_file, "DIMENSION %s   %d   %d\n",
                            MODL->pmtr[ipmtr].name,
                            MODL->pmtr[ipmtr].nrow,
                            MODL->pmtr[ipmtr].ncol);

                    index = 0;
                    for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                        fprintf(csm_file, "DESPMTR   %s[%d,:]   \"", MODL->pmtr[ipmtr].name, irow);
                        for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                            fprintf(csm_file, " %11.5f;", MODL->pmtr[ipmtr].value[index]);
                            index++;
                        }
                        fprintf(csm_file, "\"\n");
                    }

                } else {
                    fprintf(csm_file, "DESPMTR   %s   %11.5f\n",
                            MODL->pmtr[ipmtr].name, MODL->pmtr[ipmtr].value[0]);
                }

                index = 0;
                for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                    for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                        if (MODL->pmtr[ipmtr].lbnd[index] > -HUGEQ) {
                            fprintf(csm_file, "LBOUND      %s[%d,%d]   %f\n",
                                    MODL->pmtr[ipmtr].name,
                                    irow, icol,
                                    MODL->pmtr[ipmtr].lbnd[index]);
                        }
                        if (MODL->pmtr[ipmtr].ubnd[index] < +HUGEQ) {
                            fprintf(csm_file, "UBOUND      %s[%d,%d]   %f\n",
                                    MODL->pmtr[ipmtr].name,
                                    irow, icol,
                                    MODL->pmtr[ipmtr].ubnd[index]);
                        }

                        index++;
                    }
                }
            } else if (MODL->pmtr[ipmtr].type == OCSM_CFGPMTR) {
                fprintf(csm_file, "CFGPMTR   %s   %11.5f\n",
                            MODL->pmtr[ipmtr].name, MODL->pmtr[ipmtr].value[0]);
            }
        }
    }

    /* write the global Attributes */
    if (filetype == 0 || filetype == 1) {
        fprintf(csm_file, "\n# Global Attributes:\n");
        for (iattr = 0; iattr < MODL->nattr; iattr++) {
            fprintf(csm_file, "ATTRIBUTE %s   %s\n",
                    MODL->attr[iattr].name,
                    MODL->attr[iattr].defn);
        }
    }

    /* make sure indentation is up to date */
    status = ocsmCheck(MODL);
    CHECK_STATUS(ocsmCheck);

    /* write the Branches */
    fprintf(csm_file, "\n# Branches:\n");
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {

        /* special treatment if writing a .udc file and there is a matching UDPRIM */
        if (filetype == 2 && kbrch > 0) {

            /* advance to first Branch following UDPRIM statement */
            if (ibrch <= kbrch) {
                nindent = MODL->brch[ibrch].indent + 1;
                continue;

            /* we are done if an END statement */
            } else if (MODL->brch[ibrch].type == OCSM_END) {
                break;
            }

        /* special treatment if writing a .udc file and there is no matching UDPRIM */
        } else if (filetype == 2 && ibrch == 1) {
            fprintf(csm_file, "INTERFACE . all\n");

        }

        for (indent = nindent; indent < MODL->brch[ibrch].indent; indent++) {
            fprintf(csm_file, "   ");
        }

        if        (MODL->brch[ibrch].type == OCSM_APPLYCSYS) {
            fprintf(csm_file, "APPLYCSYS %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2);
        } else if (MODL->brch[ibrch].type == OCSM_ARC) {
            fprintf(csm_file, "ARC       %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                  &(MODL->brch[ibrch].arg5[1]));
        } else if (MODL->brch[ibrch].type == OCSM_ASSERT) {
            fprintf(csm_file, "ASSERT    %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4);
        } else if (MODL->brch[ibrch].type == OCSM_BEZIER) {
            fprintf(csm_file, "BEZIER    %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_BLEND) {
            fprintf(csm_file, "BLEND     %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5,
                    MODL->brch[ibrch].arg6);
        } else if (MODL->brch[ibrch].type == OCSM_BOX) {
            fprintf(csm_file, "BOX       %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5,
                    MODL->brch[ibrch].arg6);
        } else if (MODL->brch[ibrch].type == OCSM_CATBEG) {
            fprintf(csm_file, "CATBEG    %s\n",
                    MODL->brch[ibrch].arg1);
        } else if (MODL->brch[ibrch].type == OCSM_CATEND) {
            fprintf(csm_file, "CATEND\n");
        } else if (MODL->brch[ibrch].type == OCSM_CHAMFER) {
            fprintf(csm_file, "CHAMFER   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_CIRARC) {
            fprintf(csm_file, "CIRARC    %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5,
                    MODL->brch[ibrch].arg6);
        } else if (MODL->brch[ibrch].type == OCSM_ELEVATE) {
            fprintf(csm_file, "ELEVATE   %s\n",
                    MODL->brch[ibrch].arg1);
        } else if (MODL->brch[ibrch].type == OCSM_CONE) {
            fprintf(csm_file, "CONE      %s   %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5,
                    MODL->brch[ibrch].arg6,
                    MODL->brch[ibrch].arg7);
        } else if (MODL->brch[ibrch].type == OCSM_CONNECT) {
            fprintf(csm_file, "CONNECT   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5);
        } else if (MODL->brch[ibrch].type == OCSM_CYLINDER) {
            fprintf(csm_file, "CYLINDER  %s   %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5,
                    MODL->brch[ibrch].arg6,
                    MODL->brch[ibrch].arg7);
        } else if (MODL->brch[ibrch].type == OCSM_DIMENSION) {
            ipmtr = 0;
            for (jpmtr = 1; jpmtr <= MODL->npmtr; jpmtr++) {
                if (strcmp(MODL->pmtr[jpmtr].name, &(MODL->brch[ibrch].arg1[1])) == 0) {
                    if (MODL->pmtr[jpmtr].type == OCSM_DESPMTR ||
                        MODL->pmtr[jpmtr].type == OCSM_CFGPMTR   ) {
                        ipmtr = jpmtr;
                        break;
                    }
                }
            }

            if (ipmtr == 0) {
                fprintf(csm_file, "DIMENSION %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            }
        } else if (MODL->brch[ibrch].type == OCSM_DUMP) {
            fprintf(csm_file, "DUMP      %s   %s   %s   %s   %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                  &(MODL->brch[ibrch].arg5[1]),
                    MODL->brch[ibrch].arg6);
        } else if (MODL->brch[ibrch].type == OCSM_ELSE) {
            fprintf(csm_file, "ELSE\n");
        } else if (MODL->brch[ibrch].type == OCSM_ELSEIF) {
            fprintf(csm_file, "ELSEIF    %s   %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                  &(MODL->brch[ibrch].arg2[1]),
                    MODL->brch[ibrch].arg3,
                  &(MODL->brch[ibrch].arg4[1]),
                    MODL->brch[ibrch].arg5,
                  &(MODL->brch[ibrch].arg6[1]),
                    MODL->brch[ibrch].arg7);
        } else if (MODL->brch[ibrch].type == OCSM_END) {
            fprintf(csm_file, "END\n");
        } else if (MODL->brch[ibrch].type == OCSM_ENDIF) {
            fprintf(csm_file, "ENDIF\n");
        } else if (MODL->brch[ibrch].type == OCSM_EVALUATE) {
            if        (strcmp(MODL->brch[ibrch].arg1, "$node") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$NODE") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$edge") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$EDGE") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3,
                        MODL->brch[ibrch].arg4);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$edgebbox") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$EDGEBBOX") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$edgerng") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$EDGERNG") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$edgeinv") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$EDGEINV") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s   %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3,
                        MODL->brch[ibrch].arg4,
                        MODL->brch[ibrch].arg5,
                        MODL->brch[ibrch].arg6);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$edgekt") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$EDGEKT") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$edgecp") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$EDGECP") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$edgetess") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$EDGETESS") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$face") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$FACE") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3,
                        MODL->brch[ibrch].arg4,
                        MODL->brch[ibrch].arg5);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$facebbox") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$FACEBBOX") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$facerng") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$FACERNG") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$faceinv") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$FACEINV") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s   %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3,
                        MODL->brch[ibrch].arg4,
                        MODL->brch[ibrch].arg5,
                        MODL->brch[ibrch].arg6);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$faceukt") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$FACEUKT") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$facevkt") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$FACEVKT") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$facecp") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$FACECP") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$facetess") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$FACETESS") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$dist") == 0 ||
                       strcmp(MODL->brch[ibrch].arg1, "$DIST") == 0   ) {
                fprintf(csm_file, "EVALUATE  %s   %s   %s\n",
                      &(MODL->brch[ibrch].arg1[1]),
                        MODL->brch[ibrch].arg2,
                        MODL->brch[ibrch].arg3);
            }
        } else if (MODL->brch[ibrch].type == OCSM_EXTRACT) {
            fprintf(csm_file, "EXTRACT   %s\n",
                    MODL->brch[ibrch].arg1);
        } else if (MODL->brch[ibrch].type == OCSM_EXTRUDE) {
            fprintf(csm_file, "EXTRUDE   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_FILLET) {
            fprintf(csm_file, "FILLET    %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_GETATTR) {
            fprintf(csm_file, "GETATTR   %s %s %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_GROUP) {
            fprintf(csm_file, "GROUP     %s\n",
                MODL->brch[ibrch].arg1);
        } else if (MODL->brch[ibrch].type == OCSM_HOLLOW) {
            fprintf(csm_file, "HOLLOW    %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_IFTHEN) {
            fprintf(csm_file, "IFTHEN    %s   %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                  &(MODL->brch[ibrch].arg2[1]),
                    MODL->brch[ibrch].arg3,
                  &(MODL->brch[ibrch].arg4[1]),
                    MODL->brch[ibrch].arg5,
                  &(MODL->brch[ibrch].arg6[1]),
                    MODL->brch[ibrch].arg7);
        } else if (MODL->brch[ibrch].type == OCSM_IMPORT) {
            fprintf(csm_file, "IMPORT    %s   %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_INTERFACE) {
            fprintf(csm_file, "INTERFACE %s   %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                  &(MODL->brch[ibrch].arg2[1]),
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_INTERSECT) {
            fprintf(csm_file, "INTERSECT %s   %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3    );
        } else if (MODL->brch[ibrch].type == OCSM_JOIN) {
            fprintf(csm_file, "JOIN     %s    %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2);
        } else if (MODL->brch[ibrch].type == OCSM_LINSEG) {
            fprintf(csm_file, "LINSEG    %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_LOFT) {
            fprintf(csm_file, "LOFT      %s\n",
                    MODL->brch[ibrch].arg1);
        } else if (MODL->brch[ibrch].type == OCSM_MACBEG) {
            fprintf(csm_file, "MACBEG    %s\n",
                    MODL->brch[ibrch].arg1);
        } else if (MODL->brch[ibrch].type == OCSM_MACEND) {
            fprintf(csm_file, "MACEND\n");
        } else if (MODL->brch[ibrch].type == OCSM_MARK) {
            fprintf(csm_file, "MARK\n");
        } else if (MODL->brch[ibrch].type == OCSM_MESSAGE) {
            fprintf(csm_file, "MESSAGE   %s   %s   %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                  &(MODL->brch[ibrch].arg2[1]),
                  &(MODL->brch[ibrch].arg3[1]),
                  &(MODL->brch[ibrch].arg4[1]));
        } else if (MODL->brch[ibrch].type == OCSM_MIRROR) {
            fprintf(csm_file, "MIRROR    %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4);
        } else if (MODL->brch[ibrch].type == OCSM_PATBEG) {
            fprintf(csm_file, "PATBEG    %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2    );
        } else if (MODL->brch[ibrch].type == OCSM_PATBREAK) {
            fprintf(csm_file, "PATBREAK  %s   %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                  &(MODL->brch[ibrch].arg2[1]),
                    MODL->brch[ibrch].arg3,
                  &(MODL->brch[ibrch].arg4[1]),
                    MODL->brch[ibrch].arg5,
                  &(MODL->brch[ibrch].arg6[1]),
                    MODL->brch[ibrch].arg7);
        } else if (MODL->brch[ibrch].type == OCSM_PATEND) {
            fprintf(csm_file, "PATEND\n");
        } else if (MODL->brch[ibrch].type == OCSM_POINT) {
            fprintf(csm_file, "POINT     %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_PREFIX) {
            fprintf(csm_file, "PREFIX    %s\n",
                  &(MODL->brch[ibrch].arg1[1]));
        } else if (MODL->brch[ibrch].type == OCSM_PROJECT) {
            fprintf(csm_file, "PROJECT   %s   %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5,
                    MODL->brch[ibrch].arg6,
                    MODL->brch[ibrch].arg7);
        } else if (MODL->brch[ibrch].type == OCSM_RECALL) {
            fprintf(csm_file, "RECALL    %s\n",
                    MODL->brch[ibrch].arg1);
        } else if (MODL->brch[ibrch].type == OCSM_REORDER) {
            fprintf(csm_file, "REORDER   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_RESTORE) {
            fprintf(csm_file, "RESTORE   %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2    );
        } else if (MODL->brch[ibrch].type == OCSM_REVOLVE) {
            fprintf(csm_file, "REVOLVE   %s   %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5,
                    MODL->brch[ibrch].arg6,
                    MODL->brch[ibrch].arg7);
        } else if (MODL->brch[ibrch].type == OCSM_ROTATEX) {
            fprintf(csm_file, "ROTATEX   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_ROTATEY) {
            fprintf(csm_file, "ROTATEY   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_ROTATEZ) {
            fprintf(csm_file, "ROTATEZ   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_RULE) {
            fprintf(csm_file, "RULE      %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_SCALE) {
            fprintf(csm_file, "SCALE     %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4);
        } else if (MODL->brch[ibrch].type == OCSM_SELECT) {
            fprintf(csm_file, "SELECT    %s",
                  &(MODL->brch[ibrch].arg1[1]));
            if (MODL->brch[ibrch].narg >= 2) {
                fprintf(csm_file, "   %s",
                    MODL->brch[ibrch].arg2);
            }
            if (MODL->brch[ibrch].narg >= 3) {
                fprintf(csm_file, "   %s",
                    MODL->brch[ibrch].arg3);
            }
            if (MODL->brch[ibrch].narg >= 4) {
                fprintf(csm_file, "   %s",
                    MODL->brch[ibrch].arg4);
            }
            if (MODL->brch[ibrch].narg >= 5) {
                fprintf(csm_file, "   %s",
                    MODL->brch[ibrch].arg5);
            }
            if (MODL->brch[ibrch].narg >= 6) {
                fprintf(csm_file, "   %s",
                    MODL->brch[ibrch].arg6);
            }
            if (MODL->brch[ibrch].narg >= 7) {
                fprintf(csm_file, "   %s",
                    MODL->brch[ibrch].arg7);
            }
            if (MODL->brch[ibrch].narg >= 8) {
                fprintf(csm_file, "   %s",
                    MODL->brch[ibrch].arg8);
            }
            fprintf(csm_file, "\n");
        } else if (MODL->brch[ibrch].type == OCSM_SET) {
            fprintf(csm_file, "SET       %s %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2);
        } else if (MODL->brch[ibrch].type == OCSM_SKBEG) {
            fprintf(csm_file, "SKBEG     %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4);
        } else if (MODL->brch[ibrch].type == OCSM_SKCON) {
            fprintf(csm_file, "SKCON     %s   %s   %s  %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                  &(MODL->brch[ibrch].arg4[1]));
        } else if (MODL->brch[ibrch].type == OCSM_SKEND) {
            fprintf(csm_file, "SKEND     %s\n",
                    MODL->brch[ibrch].arg1);
        } else if (MODL->brch[ibrch].type == OCSM_SKVAR) {
            fprintf(csm_file, "SKVAR     %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2);
        } else if (MODL->brch[ibrch].type == OCSM_SOLBEG) {
            fprintf(csm_file, "SOLBEG    %s\n",
                  &(MODL->brch[ibrch].arg1[1]));
        } else if (MODL->brch[ibrch].type == OCSM_SOLCON) {
            fprintf(csm_file, "SOLCON    %s\n",
                  &(MODL->brch[ibrch].arg1[1]));
        } else if (MODL->brch[ibrch].type == OCSM_SOLEND) {
            fprintf(csm_file, "SOLEND\n");
        } else if (MODL->brch[ibrch].type == OCSM_SPECIAL) {
            fprintf(csm_file, "SPECIAL   %s   %s   %s   %s   %s   %s   %s   %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5,
                    MODL->brch[ibrch].arg6,
                    MODL->brch[ibrch].arg7,
                    MODL->brch[ibrch].arg8,
                    MODL->brch[ibrch].arg9);
        } else if (MODL->brch[ibrch].type == OCSM_SPHERE) {
            fprintf(csm_file, "SPHERE    %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4);
        } else if (MODL->brch[ibrch].type == OCSM_SPLINE) {
            fprintf(csm_file, "SPLINE    %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_SSLOPE) {
            fprintf(csm_file, "SSLOPE    %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_STORE) {
            fprintf(csm_file, "STORE     %s   %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3    );
        } else if (MODL->brch[ibrch].type == OCSM_SUBTRACT) {
            fprintf(csm_file, "SUBTRACT  %s   %s   %s   %s\n",
                  &(MODL->brch[ibrch].arg1[1]),
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4    );
        } else if (MODL->brch[ibrch].type == OCSM_SWEEP) {
            fprintf(csm_file, "SWEEP\n");
        } else if (MODL->brch[ibrch].type == OCSM_THROW) {
            fprintf(csm_file, "THROW     %s\n",
                    MODL->brch[ibrch].arg1);
        } else if (MODL->brch[ibrch].type == OCSM_TORUS) {
            fprintf(csm_file, "TORUS     %s   %s   %s   %s   %s   %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3,
                    MODL->brch[ibrch].arg4,
                    MODL->brch[ibrch].arg5,
                    MODL->brch[ibrch].arg6,
                    MODL->brch[ibrch].arg7,
                    MODL->brch[ibrch].arg8);
        } else if (MODL->brch[ibrch].type == OCSM_TRANSLATE) {
            fprintf(csm_file, "TRANSLATE %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        } else if (MODL->brch[ibrch].type == OCSM_UDPARG ||
                   MODL->brch[ibrch].type == OCSM_UDPRIM   ) {
            fp_inline = NULL;           // no inline file by default
            n_inline  = -1;

            if (MODL->brch[ibrch].type == OCSM_UDPARG) {
                fprintf(csm_file, "UDPARG    %s",
                        &(MODL->brch[ibrch].arg1[1]));
            } else {
                fprintf(csm_file, "UDPRIM    %s",
                        &(MODL->brch[ibrch].arg1[1]));
            }

            if (MODL->brch[ibrch].narg >= 3) {
                if        (strncmp(MODL->brch[ibrch].arg3, "$tmp_OpenCSM_", 13) == 0) {
                    fprintf(csm_file, "   %s   <<",
                            &(MODL->brch[ibrch].arg2[1]));
                    fp_inline = fopen(&MODL->brch[ibrch].arg3[1], "r");
                } else if (strncmp(MODL->brch[ibrch].arg3, "$<<inline/", 10) == 0) {
                    fprintf(csm_file, "   %s   <<",
                            &(MODL->brch[ibrch].arg2[1]));
                    n_inline = strtol(&(MODL->brch[ibrch].arg3[10]), NULL, 10);
                } else {
                    fprintf(csm_file, "   %s   %s",
                            &(MODL->brch[ibrch].arg2[1]),
                              MODL->brch[ibrch].arg3);
                }
            }
            if (MODL->brch[ibrch].narg >= 5) {
                if        (strncmp(MODL->brch[ibrch].arg5, "$tmp_OpenCSM_", 13) == 0) {
                    fprintf(csm_file, "   %s   <<",
                            &(MODL->brch[ibrch].arg4[1]));
                    fp_inline = fopen(&MODL->brch[ibrch].arg5[1], "r");
                } else if (strncmp(MODL->brch[ibrch].arg5, "$<<inline/", 10) == 0) {
                    fprintf(csm_file, "   %s   <<",
                            &(MODL->brch[ibrch].arg4[1]));
                    n_inline = strtol(&(MODL->brch[ibrch].arg5[10]), NULL, 10);
                } else {
                    fprintf(csm_file, "   %s   %s",
                            &(MODL->brch[ibrch].arg4[1]),
                              MODL->brch[ibrch].arg5);
                }
            }
            if (MODL->brch[ibrch].narg >= 7) {
                if        (strncmp(MODL->brch[ibrch].arg7, "$tmp_OpenCSM_", 13) == 0) {
                    fprintf(csm_file, "   %s   <<",
                            &(MODL->brch[ibrch].arg6[1]));
                    fp_inline = fopen(&MODL->brch[ibrch].arg7[1], "r");
                } else if (strncmp(MODL->brch[ibrch].arg7, "$<<inline/", 10) == 0) {
                    fprintf(csm_file, "   %s   <<",
                            &(MODL->brch[ibrch].arg6[1]));
                    n_inline = strtol(&(MODL->brch[ibrch].arg7[10]), NULL, 10);
                } else {
                    fprintf(csm_file, "   %s   %s",
                            &(MODL->brch[ibrch].arg6[1]),
                              MODL->brch[ibrch].arg7);
                }
            }
            if (MODL->brch[ibrch].narg >= 9) {
                if        (strncmp(MODL->brch[ibrch].arg9, "$tmp_OpenCSM_", 13) == 0) {
                    fprintf(csm_file, "   %s   <<",
                            &(MODL->brch[ibrch].arg8[1]));
                    fp_inline = fopen(&MODL->brch[ibrch].arg9[1], "r");
                } else if (strncmp(MODL->brch[ibrch].arg9, "$<<inline/", 10) == 0) {
                    fprintf(csm_file, "   %s   <<",
                            &(MODL->brch[ibrch].arg8[1]));
                    n_inline = strtol(&(MODL->brch[ibrch].arg9[10]), NULL, 10);
                } else {
                    fprintf(csm_file, "   %s   %s",
                            &(MODL->brch[ibrch].arg8[1]),
                              MODL->brch[ibrch].arg9);
                }
            }
            fprintf(csm_file, "\n");

            /* write an inline file (if there is one) */
            if (fp_inline != NULL) {
                while (1) {
                    if (fgets(templine, MAX_LINE_LEN, fp_inline) == NULL) break;

                    fprintf(csm_file, "%s", templine);
                }

                fclose(fp_inline);
                fp_inline = NULL;
                fprintf(csm_file, ">>\n");

            } else if (n_inline >= 0) {
                fprintf(csm_file, "%s", &(MODL->sinline[n_inline+3]));
                fprintf(csm_file, ">>\n");
            }

            /* skip Branches within this enclosed .udc (if not writing .cpc file) */
            if (filetype != 1 && MODL->brch[ibrch].type == OCSM_UDPRIM) {
                if (strncmp(MODL->brch[ibrch].arg1, "$/",    2) == 0 ||
                    strncmp(MODL->brch[ibrch].arg1, "$$/",   3) == 0 ||
                    strncmp(MODL->brch[ibrch].arg1, "$$$/",  4) == 0 ||
                    strncmp(MODL->brch[ibrch].arg1, "$$$$/", 5) == 0   ) {
                    jbrch = ibrch;
                    while (MODL->brch[ibrch].type   != OCSM_END                ||
                           MODL->brch[ibrch].indent != MODL->brch[jbrch].indent  ) {
                        ibrch++;
                        if (ibrch > MODL->nbrch) {
                            status = signalError(MODL, OCSM_INTERNAL_ERROR,
                                                 "could not find matching END statement");
                            goto cleanup;
                        }
                    }
                }
            }
        } else if (MODL->brch[ibrch].type == OCSM_UNION) {
            fprintf(csm_file, "UNION     %s   %s   %s\n",
                    MODL->brch[ibrch].arg1,
                    MODL->brch[ibrch].arg2,
                    MODL->brch[ibrch].arg3);
        }

        /* write the name of the Branch (if not the default name) */
        if (strncmp(MODL->brch[ibrch].name, "Brch_", 5) != 0) {
            fprintf(csm_file, "NAME      %s\n",
                    MODL->brch[ibrch].name);
        }

        /* write the Attributes for the Branch */
        for (iattr = 0; iattr < MODL->brch[ibrch].nattr; iattr++) {
            if        (MODL->brch[ibrch].attr[iattr].type != ATTRCSYS) {
                fprintf(csm_file, "ATTRIBUTE %s   %s\n",
                        MODL->brch[ibrch].attr[iattr].name,
                        MODL->brch[ibrch].attr[iattr].defn);
            } else {
                fprintf(csm_file, "CSYSTEM   %s   %s\n",
                        MODL->brch[ibrch].attr[iattr].name,
                        MODL->brch[ibrch].attr[iattr].defn);
            }
        }
    }

    /* close the file */
    fprintf(csm_file, "\nEND\n");
    fclose(csm_file);
    csm_file = NULL;

cleanup:
    if (csm_file  != NULL) fclose(csm_file );
    if (fp_inline != NULL) fclose(fp_inline);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *  ocsmSaveDespmtrs - save CFGPMTRs and DESPMTRs to filename           *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSaveDespmtrs(void   *modl,          /* (in)  pointer to MODL */
                 char   filename[])     /* (in)  file to write */
{
    int       status = SUCCESS;         /* (out) return status */

    int       ipmtr, irow, icol, index;
    FILE      *fp=NULL;

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmSaveDespmtrs);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* create the file */
    fp = fopen(filename, "w");
    if (fp == NULL) {
        SPRINT1(0, "ERROR:: DESPMTR file \"%s\" could not be created", filename);
        status = OCSM_FILE_NOT_FOUND;
        goto cleanup;
    }

    /* loop through the CFGPMTRS */
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (MODL->pmtr[ipmtr].type != OCSM_CFGPMTR) continue;

        index = 0;
        for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
            for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                fprintf(fp, "%s[%d,%d]  %15.8e\n",
                        MODL->pmtr[ipmtr].name, irow, icol,
                        MODL->pmtr[ipmtr].value[index]);
                index++;
            }
        }
    }

    /* loop through the DESPMTRS */
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (MODL->pmtr[ipmtr].type != OCSM_DESPMTR) continue;

        index = 0;
        for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
            for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                fprintf(fp, "%s[%d,%d]  %15.8e\n",
                        MODL->pmtr[ipmtr].name, irow, icol,
                        MODL->pmtr[ipmtr].value[index]);
                index++;
            }
        }
    }

cleanup:
    if (fp != NULL) fclose(fp);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmCopy - copy a MODL                                             *
 *                                                                      *
 ************************************************************************
 */

int
ocsmCopy(void   *srcModl,               /* (in)  pointer to source MODL */
         void   **newModl)              /* (out) pointer to new    MODL */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *SRC_MODL = (modl_T*)srcModl;
    modl_T    *NEW_MODL = NULL;

    int       ibrch, iattr, istor, ipmtr, irow, icol, index, i, j, itmp;
    int       iprim, iarg, isize;
    char      filename_src[MAX_FILENAME_LEN], filename_new[MAX_FILENAME_LEN];
    char      templine[MAX_LINE_LEN];
    FILE      *fp_src, *fp_new;
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(ocsmCopy);

    /* --------------------------------------------------------------- */

    /* default return value */
    *newModl = NULL;

    /* check magic number */
    if (SRC_MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (SRC_MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* make a new MODL and initialize it */
    MALLOC(NEW_MODL, modl_T, 1);

    NEW_MODL->magic      = OCSM_MAGIC;
    NEW_MODL->checked    = 0;
    NEW_MODL->embedded   = SRC_MODL->embedded;
    NEW_MODL->ibrch      = 0;
    NEW_MODL->nextseq    = 1;
    NEW_MODL->ngroup     = SRC_MODL->ngroup;
    NEW_MODL->recycle    = 0;
    NEW_MODL->verify     = SRC_MODL->verify;
    NEW_MODL->cleanup    = SRC_MODL->cleanup;
    NEW_MODL->dumpEgads  = SRC_MODL->dumpEgads;
    NEW_MODL->loadEgads  = SRC_MODL->loadEgads;
    NEW_MODL->hasMPs     = SRC_MODL->hasMPs;
    NEW_MODL->printStack = SRC_MODL->printStack;
    STRNCPY(NEW_MODL->printAttr, SRC_MODL->printAttr, MAX_STRVAL_LEN-1);
    NEW_MODL->printPmtrs = SRC_MODL->printPmtrs;
    NEW_MODL->printStors = SRC_MODL->printStors;
    NEW_MODL->tessAtEnd  = SRC_MODL->tessAtEnd;
    NEW_MODL->erepAtEnd  = SRC_MODL->erepAtEnd;
    NEW_MODL->bodyLoaded = SRC_MODL->bodyLoaded;

    NEW_MODL->seltype = -1;
    NEW_MODL->selbody = -1;
    NEW_MODL->selsize =  0;
    NEW_MODL->sellist = NULL;

    NEW_MODL->level = SRC_MODL->level;
    for (i = 0; i < 11; i++) {
        NEW_MODL->scope[i] = SRC_MODL->scope[i];
    }

    strcpy(NEW_MODL->filename, SRC_MODL->filename);

    NEW_MODL->ninline = SRC_MODL->ninline;
    NEW_MODL->minline = SRC_MODL->minline;
    NEW_MODL->sinline = NULL;

    if (NEW_MODL->minline > 0) {
        MALLOC(NEW_MODL->sinline, char, NEW_MODL->minline);
        memcpy(NEW_MODL->sinline, SRC_MODL->sinline, (NEW_MODL->minline)*sizeof(char));
    }

    NEW_MODL->nattr = 0;
    NEW_MODL->attr  = NULL;

    NEW_MODL->nstor = 0;
    NEW_MODL->stor  = NULL;

    NEW_MODL->nbrch = 0;
    NEW_MODL->mbrch = 0;
    NEW_MODL->brch  = NULL;

    NEW_MODL->npmtr = 0;
    NEW_MODL->mpmtr = 0;
    NEW_MODL->pmtr  = NULL;
    STRNCPY(NEW_MODL->prefix, SRC_MODL->prefix, MAX_NAME_LEN);

    NEW_MODL->nbody = 0;
    NEW_MODL->mbody = 0;
    NEW_MODL->body  = NULL;

    NEW_MODL->forceFDs   = SRC_MODL->forceFDs;
    NEW_MODL->numdots    = SRC_MODL->numdots;
    NEW_MODL->perturb    = NULL;
    NEW_MODL->basemodl   = NULL;
    NEW_MODL->matchSeq   = NULL;
    NEW_MODL->dtime      = 0;

    NEW_MODL->context     = SRC_MODL->context;
    NEW_MODL->userdata    = SRC_MODL->userdata;
    NEW_MODL->mesgCB      = SRC_MODL->mesgCB;
    NEW_MODL->bcstCB      = SRC_MODL->bcstCB;
    NEW_MODL->sizeCB      = SRC_MODL->sizeCB;

    strcpy(NEW_MODL->eggname, SRC_MODL->eggname);
    NEW_MODL->eggGenerate = SRC_MODL->eggGenerate;
    NEW_MODL->eggMorph    = SRC_MODL->eggMorph;
    NEW_MODL->eggDump     = SRC_MODL->eggDump;
    NEW_MODL->eggLoad     = SRC_MODL->eggLoad;
    NEW_MODL->eggFree     = SRC_MODL->eggFree;

    for (iprim = 0; iprim < MAXPRIM; iprim++) {
        NEW_MODL->NumUdp[iprim] = 0;
        NEW_MODL->Udps[  iprim] = NULL;
    }

    for (iprim = 0; iprim < MAXPRIM; iprim++) {
        if (SRC_MODL->Udps[iprim] == NULL) break;

        NEW_MODL->NumUdp[iprim] = SRC_MODL->NumUdp[iprim];

        NEW_MODL->Udps[iprim] = NULL;
        MALLOC(NEW_MODL->Udps[iprim], udp_T, SRC_MODL->NumUdp[iprim]+1);

        for (i = 0; i <= NEW_MODL->NumUdp[iprim]; i++) {
            NEW_MODL->Udps[iprim][i].ebody    = SRC_MODL->Udps[iprim][i].ebody;
            NEW_MODL->Udps[iprim][i].narg     = SRC_MODL->Udps[iprim][i].narg;
            NEW_MODL->Udps[iprim][i].arg      = NULL;
            NEW_MODL->Udps[iprim][i].ndotchg  = 0;
            NEW_MODL->Udps[iprim][i].bodyList = NULL;
            NEW_MODL->Udps[iprim][i].data     = NULL;

            MALLOC(NEW_MODL->Udps[iprim][i].arg, udparg_T, NEW_MODL->Udps[iprim][i].narg);
            for (iarg = 0; iarg < NEW_MODL->Udps[iprim][i].narg; iarg++) {
                NEW_MODL->Udps[iprim][i].arg[iarg].type = SRC_MODL->Udps[iprim][i].arg[iarg].type;
                NEW_MODL->Udps[iprim][i].arg[iarg].val  = NULL;
                NEW_MODL->Udps[iprim][i].arg[iarg].dot  = NULL;
                NEW_MODL->Udps[iprim][i].arg[iarg].size = SRC_MODL->Udps[iprim][i].arg[iarg].size;
                NEW_MODL->Udps[iprim][i].arg[iarg].nrow = SRC_MODL->Udps[iprim][i].arg[iarg].nrow;
                NEW_MODL->Udps[iprim][i].arg[iarg].ncol = SRC_MODL->Udps[iprim][i].arg[iarg].ncol;

                isize = NEW_MODL->Udps[iprim][i].arg[iarg].size;

                if        (NEW_MODL->Udps[iprim][i].arg[iarg].type == +ATTRSTRING) {
                    MALLOC(NEW_MODL->Udps[iprim][i].arg[iarg].val, char, isize+1);

                    memcpy(NEW_MODL->Udps[iprim][i].arg[iarg].val, SRC_MODL->Udps[iprim][i].arg[iarg].val, (isize+1)*sizeof(char  ));
                } else if (NEW_MODL->Udps[iprim][i].arg[iarg].type == +ATTRFILE) {
                    MALLOC(NEW_MODL->Udps[iprim][i].arg[iarg].val, char, isize+1);

                    memcpy(NEW_MODL->Udps[iprim][i].arg[iarg].val, SRC_MODL->Udps[iprim][i].arg[iarg].val, (isize+1)*sizeof(char  ));
                } else if (NEW_MODL->Udps[iprim][i].arg[iarg].type == +ATTRINT ||
                           NEW_MODL->Udps[iprim][i].arg[iarg].type == -ATTRINT   ) {
                    MALLOC(NEW_MODL->Udps[iprim][i].arg[iarg].val, int, isize);

                    memcpy(NEW_MODL->Udps[iprim][i].arg[iarg].val, SRC_MODL->Udps[iprim][i].arg[iarg].val, isize*sizeof(int   ));
                } else if (NEW_MODL->Udps[iprim][i].arg[iarg].type == +ATTRREAL ||
                           NEW_MODL->Udps[iprim][i].arg[iarg].type == -ATTRREAL   ) {
                    MALLOC(NEW_MODL->Udps[iprim][i].arg[iarg].val, double, isize);

                    memcpy(NEW_MODL->Udps[iprim][i].arg[iarg].val, SRC_MODL->Udps[iprim][i].arg[iarg].val, isize*sizeof(double));
                } else if (NEW_MODL->Udps[iprim][i].arg[iarg].type == +ATTRREALSEN ||
                           NEW_MODL->Udps[iprim][i].arg[iarg].type == -ATTRREALSEN   ) {
                    MALLOC(NEW_MODL->Udps[iprim][i].arg[iarg].val, double, isize);
                    MALLOC(NEW_MODL->Udps[iprim][i].arg[iarg].dot, double, isize);

                    memcpy(NEW_MODL->Udps[iprim][i].arg[iarg].val, SRC_MODL->Udps[iprim][i].arg[iarg].val, isize*sizeof(double));
                    memcpy(NEW_MODL->Udps[iprim][i].arg[iarg].dot, SRC_MODL->Udps[iprim][i].arg[iarg].dot, isize*sizeof(double));
                } else if (NEW_MODL->Udps[iprim][i].arg[iarg].type == +ATTRREBUILD) {
                } else if (NEW_MODL->Udps[iprim][i].arg[iarg].type == +ATTRRECYCLE) {
                } else if (NEW_MODL->Udps[iprim][i].arg[iarg].type == 0) {
                    MALLOC(NEW_MODL->Udps[iprim][i].arg[iarg].val, double, 1);

                    memcpy(NEW_MODL->Udps[iprim][i].arg[iarg].val, SRC_MODL->Udps[iprim][i].arg[iarg].val, isize*sizeof(double));
                }
            }

            if (SRC_MODL->Udps[iprim][i].bodyList != NULL) {
                for (j = 0; j < 9999; j++) {
                    if (SRC_MODL->Udps[iprim][i].bodyList[j] == 0) break;
                }
                MALLOC(NEW_MODL->Udps[iprim][i].bodyList, int, j+1);
                for (j = 0; j < 9999; j++) {
                    NEW_MODL->Udps[iprim][i].bodyList[j] = SRC_MODL->Udps[iprim][i].bodyList[j];
                    if (SRC_MODL->Udps[iprim][i].bodyList[j] == 0) break;
                }
            }
        }
    }

    NEW_MODL->nwarn   = 0;
    NEW_MODL->sigCode = 0;
    NEW_MODL->sigMesg = NULL;
    MALLOC(NEW_MODL->sigMesg, char, MAX_STR_LEN);
    NEW_MODL->sigMesg[0] = '\0';

    for (i = 0; i < 101; i++) {
        NEW_MODL->profile[i].ncall = SRC_MODL->profile[i].ncall;
        NEW_MODL->profile[i].time  = SRC_MODL->profile[i].time;
    }

    /* return value */
    *newModl = NEW_MODL;

    /* copy the global Parameters */
    for (iattr = 0; iattr < SRC_MODL->nattr; iattr++) {
        status = ocsmSetAttr(NEW_MODL, 0,
                             SRC_MODL->attr[iattr].name,
                             SRC_MODL->attr[iattr].defn);
    }

    /* copy the Storge locations */
    NEW_MODL->nstor = SRC_MODL->nstor;

    MALLOC(NEW_MODL->stor, stor_T, NEW_MODL->nstor);

    for (istor = 0; istor < NEW_MODL->nstor; istor++) {
        STRNCPY(NEW_MODL->stor[istor].name, SRC_MODL->stor[istor].name, MAX_NAME_LEN);

        NEW_MODL->stor[istor].index = SRC_MODL->stor[istor].index;
        NEW_MODL->stor[istor].nbody = SRC_MODL->stor[istor].nbody;
        NEW_MODL->stor[istor].ibody = NULL;
        NEW_MODL->stor[istor].ebody = NULL;

        MALLOC(NEW_MODL->stor[istor].ibody, int, NEW_MODL->stor[istor].nbody);
        MALLOC(NEW_MODL->stor[istor].ebody, ego, NEW_MODL->stor[istor].nbody);

        for (i = 0; i < NEW_MODL->stor[istor].nbody; i++) {
            NEW_MODL->stor[istor].ibody[i] = SRC_MODL->stor[istor].ibody[i];
            NEW_MODL->stor[istor].ebody[i] = SRC_MODL->stor[istor].ebody[i];
        }
    }

    /* copy the Parameter table (done before Branch table so that
       new Parameters do not get created by a OCSM_SET, OCSM_GETATTR, or OCSM_PATBEG) */

    /* make sure new Parameter is created with the correct scope */
    for (ipmtr = 1; ipmtr <= SRC_MODL->npmtr; ipmtr++) {
        for (j = 0; j < 11; j++) {
            if (SRC_MODL->scope[j] == SRC_MODL->pmtr[ipmtr].scope) {
                NEW_MODL->level = j;
                break;
            }
        }
        if        (strcmp(SRC_MODL->pmtr[ipmtr].name, "@stack") == 0) {
            status = ocsmNewPmtr(NEW_MODL,
                                 SRC_MODL->pmtr[ipmtr].name,
                                 SRC_MODL->pmtr[ipmtr].type,
                                 1,
                                 MAX_STACK_SIZE);
            /* this was commented out beacuse EVALUATE FACECP (and others) can
               have more than 23 entries.  so we resort to acutually using the
               size of the source instead (which we probably should have done
               all along) */
//$$$        } else if (strcmp(SRC_MODL->pmtr[ipmtr].name, "@edata") == 0) {
//$$$            status = ocsmNewPmtr(NEW_MODL,
//$$$                                 SRC_MODL->pmtr[ipmtr].name,
//$$$                                 SRC_MODL->pmtr[ipmtr].type,
//$$$                                 1,
//$$$                                 23);
        } else {
            status = ocsmNewPmtr(NEW_MODL,
                                 SRC_MODL->pmtr[ipmtr].name,
                                 SRC_MODL->pmtr[ipmtr].type,
                                 SRC_MODL->pmtr[ipmtr].nrow,
                                 SRC_MODL->pmtr[ipmtr].ncol);
        }
        CHECK_STATUS(ocsmNewPmtr);
        SPLINT_CHECK_FOR_NULL(NEW_MODL->pmtr);
        NEW_MODL->level = SRC_MODL->level;

        if (SRC_MODL->pmtr[ipmtr].value != NULL) {
            index = 0;
            for (irow = 1; irow <= SRC_MODL->pmtr[ipmtr].nrow; irow++) {
                for (icol = 1; icol <= SRC_MODL->pmtr[ipmtr].ncol; icol++) {
                    NEW_MODL->pmtr[ipmtr].value[index] = SRC_MODL->pmtr[ipmtr].value[index];
                    NEW_MODL->pmtr[ipmtr].dot[  index] = SRC_MODL->pmtr[ipmtr].dot[  index];

                    if (SRC_MODL->pmtr[ipmtr].type == OCSM_DESPMTR) {
                        NEW_MODL->pmtr[ipmtr].lbnd[ index] = SRC_MODL->pmtr[ipmtr].lbnd[ index];
                        NEW_MODL->pmtr[ipmtr].ubnd[ index] = SRC_MODL->pmtr[ipmtr].ubnd[ index];
                    }

                    index++;
                }
            }
        }
        if (SRC_MODL->pmtr[ipmtr].str != NULL) {
            strcpy(NEW_MODL->pmtr[ipmtr].str, SRC_MODL->pmtr[ipmtr].str);
        }

        NEW_MODL->pmtr[ipmtr].mprop = SRC_MODL->pmtr[ipmtr].mprop;
    }

    /* copy the Branch table.  Note that this cannot be done with
       ocsmNewBrch since we do not want UDCs to make recursive calls
       to ocsmLoad */
    NEW_MODL->mbrch = SRC_MODL->mbrch;
    NEW_MODL->nbrch = SRC_MODL->nbrch;

    RALLOC(NEW_MODL->brch, brch_T, NEW_MODL->mbrch+1);

    for (ibrch = 1; ibrch <= SRC_MODL->nbrch; ibrch++) {
        NEW_MODL->brch[ibrch].name     = NULL;
        NEW_MODL->brch[ibrch].type     = SRC_MODL->brch[ibrch].type;
        NEW_MODL->brch[ibrch].bclass   = SRC_MODL->brch[ibrch].bclass;
        NEW_MODL->brch[ibrch].level    = SRC_MODL->brch[ibrch].level;
        NEW_MODL->brch[ibrch].indent   = SRC_MODL->brch[ibrch].indent;
        NEW_MODL->brch[ibrch].filename = NULL;
        NEW_MODL->brch[ibrch].linenum  = SRC_MODL->brch[ibrch].linenum;
        NEW_MODL->brch[ibrch].actv     = SRC_MODL->brch[ibrch].actv;
        NEW_MODL->brch[ibrch].dirty    = SRC_MODL->brch[ibrch].dirty;
        NEW_MODL->brch[ibrch].nattr    = 0;
        NEW_MODL->brch[ibrch].attr     = NULL;
        NEW_MODL->brch[ibrch].ileft    = SRC_MODL->brch[ibrch].ileft;
        NEW_MODL->brch[ibrch].irite    = SRC_MODL->brch[ibrch].irite;
        NEW_MODL->brch[ibrch].ichld    = SRC_MODL->brch[ibrch].ichld;
        NEW_MODL->brch[ibrch].nmprp    = 0;
        NEW_MODL->brch[ibrch].mprp     = NULL;
        NEW_MODL->brch[ibrch].narg     = SRC_MODL->brch[ibrch].narg;
        NEW_MODL->brch[ibrch].arg1     = NULL;
        NEW_MODL->brch[ibrch].arg2     = NULL;
        NEW_MODL->brch[ibrch].arg3     = NULL;
        NEW_MODL->brch[ibrch].arg4     = NULL;
        NEW_MODL->brch[ibrch].arg5     = NULL;
        NEW_MODL->brch[ibrch].arg6     = NULL;
        NEW_MODL->brch[ibrch].arg7     = NULL;
        NEW_MODL->brch[ibrch].arg8     = NULL;
        NEW_MODL->brch[ibrch].arg9     = NULL;

        MALLOC(NEW_MODL->brch[ibrch].name, char, STRLEN(SRC_MODL->brch[ibrch].name)+1);
        STRNCPY(NEW_MODL->brch[ibrch].name, SRC_MODL->brch[ibrch].name, STRLEN(SRC_MODL->brch[ibrch].name)+1);

        MALLOC(NEW_MODL->brch[ibrch].filename, char, MAX_FILENAME_LEN);
        STRNCPY(NEW_MODL->brch[ibrch].filename, SRC_MODL->brch[ibrch].filename, MAX_FILENAME_LEN);

        if (SRC_MODL->brch[ibrch].arg1 != NULL) {
            MALLOC(NEW_MODL->brch[ibrch].arg1, char, (int)STRLEN(SRC_MODL->brch[ibrch].arg1)+1);
            strcpy(NEW_MODL->brch[ibrch].arg1,                   SRC_MODL->brch[ibrch].arg1   );
        }

        if (SRC_MODL->brch[ibrch].arg2 != NULL) {
            MALLOC(NEW_MODL->brch[ibrch].arg2, char, (int)STRLEN(SRC_MODL->brch[ibrch].arg2)+1);
            strcpy(NEW_MODL->brch[ibrch].arg2,                   SRC_MODL->brch[ibrch].arg2   );
        }

        if (SRC_MODL->brch[ibrch].arg3 != NULL) {
            MALLOC(NEW_MODL->brch[ibrch].arg3, char, (int)STRLEN(SRC_MODL->brch[ibrch].arg3)+1);
            strcpy(NEW_MODL->brch[ibrch].arg3,                   SRC_MODL->brch[ibrch].arg3   );
        }

        if (SRC_MODL->brch[ibrch].arg4 != NULL) {
            MALLOC(NEW_MODL->brch[ibrch].arg4, char, (int)STRLEN(SRC_MODL->brch[ibrch].arg4)+1);
            strcpy(NEW_MODL->brch[ibrch].arg4,                   SRC_MODL->brch[ibrch].arg4   );
        }

        if (SRC_MODL->brch[ibrch].arg5 != NULL) {
            MALLOC(NEW_MODL->brch[ibrch].arg5, char, (int)STRLEN(SRC_MODL->brch[ibrch].arg5)+1);
            strcpy(NEW_MODL->brch[ibrch].arg5,                   SRC_MODL->brch[ibrch].arg5   );
        }

        if (SRC_MODL->brch[ibrch].arg6 != NULL) {
            MALLOC(NEW_MODL->brch[ibrch].arg6, char, (int)STRLEN(SRC_MODL->brch[ibrch].arg6)+1);
            strcpy(NEW_MODL->brch[ibrch].arg6,                   SRC_MODL->brch[ibrch].arg6   );
        }

        if (SRC_MODL->brch[ibrch].arg7 != NULL) {
            MALLOC(NEW_MODL->brch[ibrch].arg7, char, (int)STRLEN(SRC_MODL->brch[ibrch].arg7)+1);
            strcpy(NEW_MODL->brch[ibrch].arg7,                   SRC_MODL->brch[ibrch].arg7   );
        }

        if (SRC_MODL->brch[ibrch].arg8 != NULL) {
            MALLOC(NEW_MODL->brch[ibrch].arg8, char, (int)STRLEN(SRC_MODL->brch[ibrch].arg8)+1);
            strcpy(NEW_MODL->brch[ibrch].arg8,                   SRC_MODL->brch[ibrch].arg8   );
        }

        if (SRC_MODL->brch[ibrch].arg9 != NULL) {
            MALLOC(NEW_MODL->brch[ibrch].arg9, char, (int)STRLEN(SRC_MODL->brch[ibrch].arg9)+1);
            strcpy(NEW_MODL->brch[ibrch].arg9,                   SRC_MODL->brch[ibrch].arg9   );
        }

        /* copy the Attributes */
        for (iattr = 0; iattr < SRC_MODL->brch[ibrch].nattr; iattr++) {
            if (SRC_MODL->brch[ibrch].attr[iattr].type != ATTRCSYS) {
                status = ocsmSetAttr(NEW_MODL, ibrch,
                                     SRC_MODL->brch[ibrch].attr[iattr].name,
                                     SRC_MODL->brch[ibrch].attr[iattr].defn);
            } else {
                status = ocsmSetCsys(NEW_MODL, ibrch,
                                     SRC_MODL->brch[ibrch].attr[iattr].name,
                                     SRC_MODL->brch[ibrch].attr[iattr].defn);
            }
        }
    }

    /* restore the sequence number */
    NEW_MODL->nextseq = SRC_MODL->nextseq;

    if (SRC_MODL->tmpDirNum < 0) {
        NEW_MODL->tmpDirNum = SRC_MODL->tmpDirNum;
    } else {
        snprintf(filename_src, MAX_FILENAME_LEN, "tmp_OpenCSM_%02d/00", SRC_MODL->tmpDirNum);

        /* find a directory for the new tmp_OpenCSM files */
        for (itmp = 0; itmp < 999; itmp++) {
            snprintf(filename_new, MAX_FILENAME_LEN, "tmp_OpenCSM_%02d/00", itmp);

            fp_new = fopen(filename_new, "r");
            if (fp_new != NULL) {
                fclose(fp_new);
                continue;
            }

            /* we found an unused directory, so make it now */
            snprintf(filename_new, MAX_FILENAME_LEN, "tmp_OpenCSM_%02d", itmp);
            if (MKDIR(filename_new) != 0) {
                SPRINT1(0, "ERROR:: could not make \"%s\"", filename_new);
                status = OCSM_FILE_NOT_FOUND;
                goto cleanup;
            }

            NEW_MODL->tmpDirNum = itmp;
            break;
        }

        if (NEW_MODL->tmpDirNum < 0) {
            SPRINT0(0, "ERROR:: could not find possible tmpdir");
            status = OCSM_FILE_NOT_FOUND;
            goto cleanup;
        }

        /* change from using old tmp_OpenCSM files to new ones */
        for (itmp = 0; itmp < 999; itmp++) {

            /* try to copy the files (starting at 00) */
            snprintf(filename_src, MAX_FILENAME_LEN, "tmp_OpenCSM_%02d/%02d", SRC_MODL->tmpDirNum, itmp);
            fp_src = fopen(filename_src, "r");
            if (fp_src == NULL) break;

            snprintf(filename_new, MAX_FILENAME_LEN, "tmp_OpenCSM_%02d/%02d", NEW_MODL->tmpDirNum, itmp);
            fp_new = fopen(filename_new, "w");
            if (fp_new == NULL) {
                SPRINT1(0, "ERROR:: file \"%s\" could nt be created", filename_new);
                status = OCSM_FILE_NOT_FOUND;
                goto cleanup;
            }

            snprintf(filename_src, MAX_FILENAME_LEN, "$tmp_OpenCSM_%02d/%02d", SRC_MODL->tmpDirNum, itmp);
            snprintf(filename_new, MAX_FILENAME_LEN, "$tmp_OpenCSM_%02d/%02d", NEW_MODL->tmpDirNum, itmp);

            /* copy the file */
            while (1) {
                if (fgets(templine, MAX_LINE_LEN, fp_src) == NULL) break;
                fprintf(fp_new, "%s", templine);
            }
            fclose(fp_src);
            fclose(fp_new);

            /* change all mentions in UDPARG and UDPRIM Branches of old filename to new filename */
            for (ibrch = 1; ibrch <= SRC_MODL->nbrch; ibrch++) {
                if (SRC_MODL->brch[ibrch].type == OCSM_UDPARG ||
                    SRC_MODL->brch[ibrch].type == OCSM_UDPRIM   ) {

                    if (SRC_MODL->brch[ibrch].narg >= 3 && strcmp(SRC_MODL->brch[ibrch].arg3, filename_src) == 0) {
                        strcpy(NEW_MODL->brch[ibrch].arg3, filename_new);
                    }
                    if (SRC_MODL->brch[ibrch].narg >= 5 && strcmp(SRC_MODL->brch[ibrch].arg5, filename_src) == 0) {
                        strcpy(NEW_MODL->brch[ibrch].arg5, filename_new);
                    }
                    if (SRC_MODL->brch[ibrch].narg >= 7 && strcmp(SRC_MODL->brch[ibrch].arg7, filename_src) == 0) {
                        strcpy(NEW_MODL->brch[ibrch].arg7, filename_new);
                    }
                    if (SRC_MODL->brch[ibrch].narg >= 9 && strcmp(SRC_MODL->brch[ibrch].arg9, filename_src) == 0) {
                        strcpy(NEW_MODL->brch[ibrch].arg9, filename_new);
                    }
                }
            }
        }
    }

    /* do NOT copy the Body table */

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmFree - free all storage associated with a MODL                 *
 *                                                                      *
 ************************************************************************
 */

int
ocsmFree(
/*@null@*/void   *modl)                 /* (in)  pointer to MODL (or NULL) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       itmp, ibrch, iattr, iface, istor, ipmtr, ibody;
    char      tmpdirname[MAX_EXPR_LEN];

    ROUTINE(ocsmFree);

    /* --------------------------------------------------------------- */

    /* if modl is NULL, then we are cleaning up the udp cache */
    if (MODL == NULL) {

        /* clean the udp cache */
        udp_cleanupAll();

        goto cleanup;
    }

    /* check magic number */
    if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* clean up tmp files */
    if (MODL->tmpDirNum >= 0) {
        for (itmp = 0; itmp < 999; itmp++) {
            snprintf(tmpdirname, MAX_EXPR_LEN, "tmp_OpenCSM_%02d/%02d", MODL->tmpDirNum, itmp);
            if (remove(tmpdirname) != 0) {
                break;
            }
        }

        snprintf(tmpdirname, MAX_EXPR_LEN, "tmp_OpenCSM_%02d", MODL->tmpDirNum);
        if (rmdir(tmpdirname) == 0) {
            SPRINT1(1, "--> temporary directory \"%s\" removed", tmpdirname);
            MODL->tmpDirNum = -1;
        } else {
            SPRINT1(0, "ERROR:: could not remove \"%s\"\n", tmpdirname);
            status = OCSM_FILE_NOT_FOUND;
            goto cleanup;
        }
    }

    /* if this has a perturbed Body, free it first */
    status = removePerturbation(MODL);
    CHECK_STATUS(removePerturbation);

    /* remove any velocity information */
    status = removeVels(MODL, 0);
    CHECK_STATUS(removeVels);

    /* free up the Body table */
    for (ibody = 1; ibody <= MODL->mbody; ibody++) {
        if (MODL->body[ibody].etess != NULL) {
            status = EG_deleteObject(MODL->body[ibody].etess);
            if (status < EGADS_SUCCESS) {
                SPRINT1(0, "WARNING:: problem removing MODL->body[%d].etess", ibody);
            }

            MODL->body[ibody].etess = NULL;
        }

        if (MODL->body[ibody].eetess != NULL) {
            status = EG_deleteObject(MODL->body[ibody].eetess);
            CHECK_STATUS(EG_deleteObject);

            MODL->body[ibody].eetess = NULL;
        }

        if (MODL->body[ibody].eebody != NULL) {
            status = EG_deleteObject(MODL->body[ibody].eebody);
            CHECK_STATUS(EG_deleteObject);

            MODL->body[ibody].eebody = NULL;
        }

        if (MODL->body[ibody].ebody != NULL) {
            /* ignore "dereference with active objects" message and/or any errors
               that may occur since this happens for Bodys that are restored */
            (void) EG_setOutLevel(MODL->context, 0);
            (void) EG_deleteObject(MODL->body[ibody].ebody);
            (void) EG_setOutLevel(MODL->context, outLevel);

            MODL->body[ibody].ebody = NULL;
        }

        status = removeVels(MODL, ibody);
        CHECK_STATUS(removeVels);

        if (MODL->body[ibody].face != NULL) {
            for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
                if (MODL->body[ibody].face[iface].eggdata != NULL) {
                    status = MODL->eggFree(MODL->body[ibody].face[iface].eggdata);
                    CHECK_STATUS(eggFree);
                }
            }
        }

        status = freeBody(MODL, ibody);
        CHECK_STATUS(freeBody);
    }

    FREE(MODL->body);
    MODL->nbody = 0;

    /* free up the UDP tables */
    status = udp_free(MODL);
    CHECK_STATUS(udp_free);

    /* free up the Parameter table */
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        FREE(MODL->pmtr[ipmtr].name );
        FREE(MODL->pmtr[ipmtr].value);
        FREE(MODL->pmtr[ipmtr].dot  );
        FREE(MODL->pmtr[ipmtr].lbnd );
        FREE(MODL->pmtr[ipmtr].ubnd );
        FREE(MODL->pmtr[ipmtr].str  );
    }

    FREE(MODL->pmtr);
    MODL->npmtr = 0;

    /* free up the Branch table */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        for (iattr = 0; iattr < MODL->brch[ibrch].nattr; iattr++) {
            FREE(MODL->brch[ibrch].attr[iattr].name);
            FREE(MODL->brch[ibrch].attr[iattr].defn);
        }

        FREE(MODL->brch[ibrch].name    );
        FREE(MODL->brch[ibrch].filename);
        FREE(MODL->brch[ibrch].attr    );
        FREE(MODL->brch[ibrch].mprp    );
        FREE(MODL->brch[ibrch].arg1    );
        FREE(MODL->brch[ibrch].arg2    );
        FREE(MODL->brch[ibrch].arg3    );
        FREE(MODL->brch[ibrch].arg4    );
        FREE(MODL->brch[ibrch].arg5    );
        FREE(MODL->brch[ibrch].arg6    );
        FREE(MODL->brch[ibrch].arg7    );
        FREE(MODL->brch[ibrch].arg8    );
        FREE(MODL->brch[ibrch].arg9    );
    }

    FREE(MODL->brch);
    MODL->nbrch = 0;

    /* free up the selection list */
    MODL->selsize = 0;
    FREE(MODL->sellist);

    /* free up the global Attributes */
    for (iattr = 0; iattr < MODL->nattr; iattr++) {
        FREE(MODL->attr[iattr].name);
        FREE(MODL->attr[iattr].defn);
    }

    FREE(MODL->attr);
    MODL->nattr = 0;

    /* free up the Storage table */
    for (istor = 0; istor < MODL->nstor; istor++) {
        FREE(MODL->stor[istor].ibody);
        FREE(MODL->stor[istor].ebody);
    }

    FREE(MODL->stor);
    MODL->nstor = 0;

    /* free up the inline file stream */
    FREE(MODL->sinline);

    /* free up the message buffer */
    FREE(MODL->sigMesg);

    /* set the magic number to zero in case someone tries to use
       the address to  this MODL again */
    MODL->magic = 0;

    /* free up the MODL structure */
    FREE(MODL);

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmInfo - get info about a MODL                                   *
 *                                                                      *
 ************************************************************************
 */

int
ocsmInfo(void   *modl,                  /* (in)  pointer to MODL */
         int    *nbrch,                 /* (out) number of Branches */
         int    *npmtr,                 /* (out) number of Parameters */
         int    *nbody)                 /* (out) number of Bodys */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmInfo);

    /* --------------------------------------------------------------- */

    /* default return values */
    *nbrch = -1;
    *npmtr = -1;

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* return values */
    *nbrch = MODL->nbrch;
    *npmtr = MODL->npmtr;
    *nbody = MODL->nbody;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmCheck - check that Branches are properly ordered               *
 *                                                                      *
 ************************************************************************
 */

int
ocsmCheck(void   *modl)                 /* (in)  pointer to MODL */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       ibrch, ipass, indent, linenum;
    int       inest[MAX_NESTING], jnest[MAX_NESTING], nnest;
    char      *filename;

    ROUTINE(ocsmCheck);

    /* --------------------------------------------------------------- */

    SPRINT0(1, "--> enter ocsmCheck()");

    /* initially there are no errors */
    ipass = 0;

    /* clear the signals */
    MODL->nwarn      = 0;
    MODL->sigCode    = 0;
    MODL->sigMesg[0] = '\0';

    /* check magic number */
    if (MODL == NULL) {
        status = signalError(MODL, OCSM_NOT_MODL_STRUCTURE,
                             "MODL=NULL");
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = signalError(MODL, OCSM_NOT_MODL_STRUCTURE,
                             "bad magic number");
        goto cleanup;
    }

    /* if we have already checked, just return */
    if (MODL->checked == 1) {
        goto cleanup;
    }

    /* make sure Blocks of Branches are properly nested */
    nnest = 0;

    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        MODL->ibrch = ibrch;

        filename = MODL->brch[ibrch].filename;
        linenum  = MODL->brch[ibrch].linenum;

        if        (MODL->brch[ibrch].type == OCSM_PATBEG) {
            if (nnest >= MAX_NESTING) {
                status = signalError(MODL, OCSM_NESTED_TOO_DEEPLY,
                                     "can only be nested %d levels deep", MAX_NESTING);
                goto cleanup;
            }

            inest[nnest] = OCSM_PATBEG;
            jnest[nnest] = ibrch;
            nnest++;
        } else if (MODL->brch[ibrch].type == OCSM_PATEND) {
            if (nnest <= 0) {
                status = signalError(MODL, OCSM_IMPROPER_NESTING,
                                     "PATEND without matching PATBEG");
                goto cleanup;
            } else if (inest[nnest-1] != OCSM_PATBEG) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "PATEND is improperly nested\n (follows \"%s\") at [[%s:%d]]",
                                      ocsmGetText(inest[nnest-1]),
                                      MODL->brch[jnest[nnest-1]].filename,
                                      MODL->brch[jnest[nnest-1]].linenum);
                goto cleanup;
            }

            nnest--;
        } else if (MODL->brch[ibrch].type == OCSM_IFTHEN) {
            if (nnest >= MAX_NESTING) {
                status = signalError(MODL, OCSM_NESTED_TOO_DEEPLY,
                                     "can only be nested %d levels deep", MAX_NESTING);
                goto cleanup;
            }

            inest[nnest] = OCSM_IFTHEN;
            jnest[nnest] = ibrch;
            nnest++;
        } else if (MODL->brch[ibrch].type == OCSM_ELSEIF) {
            if (nnest <= 0) {
                status = signalError(MODL, OCSM_IMPROPER_NESTING,
                                     "ELSEIF without matching IFTHEN");
                goto cleanup;
            } else if (inest[nnest-1] != OCSM_IFTHEN) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "ELSEIF is improperly nested\n (follows \"%s\") at [[%s:%d]]",
                                      ocsmGetText(inest[nnest-1]),
                                      MODL->brch[jnest[nnest-1]].filename,
                                      MODL->brch[jnest[nnest-1]].linenum);
                goto cleanup;
            }
        } else if (MODL->brch[ibrch].type == OCSM_ELSE) {
            if (nnest <= 0) {
                status = signalError(MODL, OCSM_IMPROPER_NESTING,
                                     "ELSE without matching IFTHEN");
                goto cleanup;
            } else if (inest[nnest-1] != OCSM_IFTHEN) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "ELSE is improperly nested\n (follows \"%s\") at [[%s:%d]]",
                                      ocsmGetText(inest[nnest-1]),
                                      MODL->brch[jnest[nnest-1]].filename,
                                      MODL->brch[jnest[nnest-1]].linenum);
                goto cleanup;
            }

            inest[nnest-1] = OCSM_ELSE;
            jnest[nnest-1] = ibrch;
        } else if (MODL->brch[ibrch].type == OCSM_ENDIF) {
            if (nnest <= 0) {
                status = signalError(MODL, OCSM_IMPROPER_NESTING,
                                     "ENDIF without matching IFTHEN");
                goto cleanup;
            } else if (inest[nnest-1] != OCSM_IFTHEN &&
                       inest[nnest-1] != OCSM_ELSE     ) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "ENDIF is improperly nested\n (follows \"%s\") at [[%s:%d]]",
                                      ocsmGetText(inest[nnest-1]),
                                      MODL->brch[jnest[nnest-1]].filename,
                                      MODL->brch[jnest[nnest-1]].linenum);
                goto cleanup;
            }

            nnest--;
        } else if (MODL->brch[ibrch].type == OCSM_CATBEG) {
            if (nnest >= MAX_NESTING) {
                status = signalError(MODL, OCSM_NESTED_TOO_DEEPLY,
                                     "can only be nested %d levels deep", MAX_NESTING);
                goto cleanup;
            }

            inest[nnest] = OCSM_CATBEG;
            jnest[nnest] = ibrch;
            nnest++;
        } else if (MODL->brch[ibrch].type == OCSM_CATEND) {
            if (nnest <= 0) {
                status = signalError(MODL, OCSM_IMPROPER_NESTING,
                                     "CATEND without matching CATBEG");
                goto cleanup;
            } else if (inest[nnest-1] != OCSM_CATBEG) {
                status = signalError2(MODL, OCSM_IMPROPER_NESTING, filename, linenum,
                                      "CATEND is improperly nested\n (follows \"%s\") at [[%s:%d]]",
                                      ocsmGetText(inest[nnest-1]),
                                      MODL->brch[jnest[nnest-1]].filename,
                                      MODL->brch[jnest[nnest-1]].linenum);
                goto cleanup;
            }

            nnest--;
        }
    }

    if (nnest > 0) {
        status = signalError2(MODL, OCSM_NESTING_NOT_CLOSED, filename, linenum,
                              "%d block(s) are still open\n (last one is \"%s\" at [[%s:%d]]",
                              nnest, ocsmGetText(inest[nnest-1]),
                              MODL->brch[jnest[nnest-1]].filename,
                              MODL->brch[jnest[nnest-1]].linenum);
        goto cleanup;
    }

    /* initialize all indentation and parent/child relationships */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        MODL->brch[ibrch].indent = -1;
        MODL->brch[ibrch].ileft  = -2;
        MODL->brch[ibrch].irite  = -2;
        MODL->brch[ibrch].ichld  = -2;
    }

    /* set up the indentation */
    indent = 0;

    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {

        /* decrease indentation at the end of a block */
        if        (MODL->brch[ibrch].type == OCSM_MACEND   ||
                   MODL->brch[ibrch].type == OCSM_PATBREAK ||
                   MODL->brch[ibrch].type == OCSM_PATEND   ||
                   MODL->brch[ibrch].type == OCSM_ELSEIF   ||
                   MODL->brch[ibrch].type == OCSM_ELSE     ||
                   MODL->brch[ibrch].type == OCSM_ENDIF    ||
                   MODL->brch[ibrch].type == OCSM_CATEND   ||
                   MODL->brch[ibrch].type == OCSM_SKEND    ||
                   MODL->brch[ibrch].type == OCSM_SOLEND   ||
                   MODL->brch[ibrch].type == OCSM_END        ) {
            indent--;
        }

        if (indent < 0) {
            SPRINT0(1, "WARNING:: fixing indentation");
            (MODL->nwarn)++;
            indent = 0;
        }

        /* set the indentation */
        MODL->brch[ibrch].indent = indent;

        /* increase indentation at the beginning of a block */
        if        (MODL->brch[ibrch].type == OCSM_MACBEG   ||
                   MODL->brch[ibrch].type == OCSM_PATBEG   ||
                   MODL->brch[ibrch].type == OCSM_PATBREAK ||
                   MODL->brch[ibrch].type == OCSM_IFTHEN   ||
                   MODL->brch[ibrch].type == OCSM_ELSEIF   ||
                   MODL->brch[ibrch].type == OCSM_ELSE     ||
                   MODL->brch[ibrch].type == OCSM_CATBEG   ||
                   MODL->brch[ibrch].type == OCSM_SKBEG    ||
                   MODL->brch[ibrch].type == OCSM_SOLBEG     ) {
            indent++;
        } else if (MODL->brch[ibrch].type    == OCSM_UDPRIM &&
                   strncmp(MODL->brch[ibrch].arg1, "$/", 2) == 0) {
            indent++;
        } else if (MODL->brch[ibrch].type    == OCSM_UDPRIM &&
                   strncmp(MODL->brch[ibrch].arg1, "$$/", 3) == 0) {
            indent++;
        } else if (MODL->brch[ibrch].type    == OCSM_UDPRIM &&
                   strncmp(MODL->brch[ibrch].arg1, "$$$/", 4) == 0) {
            indent++;
        } else if (MODL->brch[ibrch].type    == OCSM_UDPRIM &&
                   strncmp(MODL->brch[ibrch].arg1, "$$$$/", 5) == 0) {
            indent++;
        }
    }

    /* warn if improper nesting was found */
    if (indent == 1 && MODL->brch[MODL->nbrch].type != OCSM_END) {
        /* okay */
    } else if (indent != 0) {
        SPRINT1(1, "WARNING:: indent=%d, but should be 0", indent);
        (MODL->nwarn)++;
    }

    /* MODL has passed all checks */
    MODL->checked = 1;

cleanup:
    if (MODL->checked == 1) {
        SPRINT0(1, "--> checks passed");
    } else {
        SPRINT0(1, "--> checks failed");

        /* mark the Branch after the last one that passed as the Branch with the error */
        if (ipass < MODL->nbrch) {
            MODL->brch[ipass+1].ileft = -3;
        } else {
            status = signalError(MODL, OCSM_ILLEGAL_BRCH_INDEX, "");
        }
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmRegMesgCB - register callback function for exporting mesgs     *
 *                                                                      *
 ************************************************************************
 */

int
ocsmRegMesgCB(void   *modl,             /* (in)  pointer to MODL */
              void   (*callback)(char*))
                                        /* (in)  handle of callback function */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* store pointer to callback */
    MODL->mesgCB = callback;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmRegBcstCB - register callback function for broadcasting mesgs  *
 *                                                                      *
 ************************************************************************
 */

int
ocsmRegBcstCB(void   *modl,             /* (in)  pointer to MODL */
              void   (*callback)(char*))
                                        /* (in)  handle of callback function */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* store pointer to callback */
    MODL->bcstCB = callback;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmRegSizeCB - register callback function for DESPMTR size changes*
 *                                                                      *
 ************************************************************************
 */

int
ocsmRegSizeCB(void   *modl,             /* (in)  pointer to MODL */
              void   (*callback)(void*, int, int, int))
                                        /* (in)  handle of callback function */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* store pointer to callback */
    MODL->sizeCB = callback;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmBuild - build Bodys by executing the MODL up to a given Branch *
 *                                                                      *
 ************************************************************************
 */

int
ocsmBuild(void   *modl,                 /* (in)  pointer to MODL */
          int    buildTo,               /* (in)  last Branch to execute (or 0 for all, or -1 for no recycling) */
          int    *builtTo,              /* (out) last Branch executed successfully */
          int    *nbody,                /* (in)  number of entries allocated in body[] */
                                        /* (out) number of Bodys on the stack if body!=NULL*/
/*@null@*/int    body[])                /* (out) array  of Bodys on the stack (LIFO)
                                                 (at least *nbody long) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    /* stack contains Body indices, Sketch indices (if negative), or 0 for MARKs */
    int       nstack, nstackSave, *stack=NULL, nbodySave;

    /* Sketch contains linsegs, cirarcs, splines, and beziers */
    sket_T    *sket=NULL;

    /* storage loaction information */
    int        imacro, *macros=NULL;

    /* pattern information */
    int        npatn, npatn_sig, bool1, bool2, bool3;
    patn_T     *patn=NULL;
    double     newValue;

    /* solver information */
    int        nsolvar, nsolcon;
    int        *solvars=NULL, *solcons=NULL;

    /* select/sort information */
    int        nobjs, ient, ipass, nswap, iswap, count;
    double     rswap, *props=NULL, bbox[6];
    ego        *eobjs;

    int        status2, ibrch, jbrch, ibrchSave=0, type, ibrchl, i, j, iface, iedge, iloop, inode, nbodyMax;
    int        nshell, nbodys, nfaces, nloops, nedges, nnodes, ibody1, ibody2, ibeg, iend;
    int        iattr, ipmtr, jpmtr, istor, jstor, kstor, icount, nmacro, jstack, verify, icatch;
    int        ibody, jbody, jface, jedge, jnode, ibodyl, irow, nrow, icol, ncol, indx, itype, nlist, ilist, jlist;
    int        *iblist=NULL, nblist, iseq, ileft, irite, attrType, attrLen, nefaces;
    int        tempSize, *tempList=NULL, inlist;

    varg_T     args[10];
    double     dihedral, toler, value, dot, *values=NULL, *dots=NULL;
    double     dist, pnt1[3], pnt2[3], data1[18], data2[18];
    char       pname[MAX_EXPR_LEN], pmtrName[MAX_EXPR_LEN], thisArg[MAX_LINE_LEN], str[MAX_STRVAL_LEN], temp[MAX_STRVAL_LEN];
    ego        *eshells, *ebodys, *efaces, *eloops, *eedges, *enodes, etemp2;

    CINT       *tempIlist;
    CDOUBLE    *tempRlist;
    CCHAR      *tempClist, *aname;

    int        imajor, iminor, iarg, ival, hasdots, istack, igroup, ngroup, nattr, nego1, nego2;
    int        ibest, npnt, ipnt, ntri, itri, AT_iedge, AT_iface, AT_xcg, AT_ycg, AT_zcg;
    int        match1, match2, match3, periodic, iter, oclass, mtype, nchild, *senses, iprim;
    int        udp_num, *udp_types, *udp_idef, iflag;
    CINT       *ptype, *pindx, *tris, *tric;
    double     tbest, xbest, ybest, zbest, dbest, dtest;
    double     u0, v0, x0, y0, z0, u1, v1, x1, y1, z1, u2, v2, x2, y2, z2, s0, s1, tt, mat[9], rhs[3], res[3], sval[3], s0s1tt[3];
    double     xyz_[3], uv_[3], uv_dot[2], data[18], uvrange[4], duvt[3], dmin, *udp_ddef;
    CDOUBLE    *xyz, *uv;
    char       dumpfile[MAX_EXPR_LEN], *extension, **udp_names;
    void       *realloc_temp=NULL;              /* used by RALLOC macro */

    CCHAR      *OC_ver;
    FILE       *fp;
    ego        ebodyl, ebody, emodel=NULL, *etemp=NULL, enode, eedge, eface, eobj, eref, *echilds;
    clock_t    old_time, new_time;

    ROUTINE(ocsmBuild);

#define CATCH_STATUS(X)                                                 \
    if (status < SUCCESS) {                                             \
        MODL->sigCode = status;                                         \
        EPRINT( "ERROR:: build terminated early due to BAD STATUS = %d (%s) from %s (called from %s:%d)\n", status, ocsmGetText(status), #X, routine, __LINE__); \
        goto finalize;                                                  \
    }

    /* --------------------------------------------------------------- */

#if PRINT_CALLHISTORY > 0
    fp_callHistory = fopen("callHistory", "w");
    indent_callHistory = 0;
#endif

    SPRINT1(1, "--> enter ocsmBuild(buildTo=%d)", buildTo);

    for (iarg = 1; iarg < 10; iarg++) {
        args[iarg].val = NULL;
        args[iarg].dot = NULL;
    }

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    MODL->cleanup   = 25;

    /* clear the signals */
    MODL->nwarn      = 0;
    MODL->sigCode    = 0;
    MODL->sigMesg[0] = '\0';

    MODL->ibrch      = 0;

    MALLOC(stack,   int,    MAX_STACK_SIZE);
    MALLOC(macros,  int,    MAX_NUM_MACROS+1);
    MALLOC(patn,    patn_T, MAX_NESTING);
    MALLOC(sket,    sket_T, 1);
    MALLOC(solvars, int,    MAX_SOLVER_SIZE);
    MALLOC(solcons, int,    MAX_SOLVER_SIZE);

    *builtTo = 0;
    nbodyMax = *nbody;

    /* reset the profile info */
    for (i = 0; i < 101; i++) {
        MODL->profile[i].ncall = 0;
        MODL->profile[i].time  = 0;
    }

    /* if MODL is not checked already, do it now (since checking a MODL
       has side-effects that are needed during build process) */
    if (MODL->checked != 1) {
        status = ocsmCheck(MODL);
        CHECK_STATUS(ocsmCheck);
    }

    /* defer all active Branches within macro definitions */
    nmacro = 0;
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        if (MODL->brch[ibrch].actv == OCSM_ACTIVE && nmacro > 0) {
            MODL->brch[ibrch].actv = OCSM_DEFERRED;
        }
        if        (MODL->brch[ibrch].type == OCSM_MACBEG) {
            nmacro++;
        } else if (MODL->brch[ibrch].type == OCSM_MACEND) {
            nmacro--;
        }
    }

    /* initialize all parent/child flags for Branches */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        MODL->brch[ibrch].ileft = -2;
        MODL->brch[ibrch].irite = -2;
        MODL->brch[ibrch].ichld = -2;
    }

    EG_revision(&imajor, &iminor, &OC_ver);
    if (strstr(OC_ver, "Interim") == NULL) {
        SPRINT3(1, "\n    EGADS version %2d.%02d (with %s)\n", imajor, iminor, &(OC_ver[5]));
    } else {
        SPRINT3(1, "\n    Interim EGADS version %2d.%02d (with %s)\n", imajor, iminor, &(OC_ver)[20]);
    }

    /* create a new EGADS context (if not already set) */
    if (MODL->context == NULL) {
        status = EG_open(&(MODL->context));
        CHECK_STATUS(EG_open);
    }

    /* update the thread used by EGADS */
    status = EG_updateThread(MODL->context);
    CHECK_STATUS(EG_updateThread);

    status = EG_setOutLevel(MODL->context, outLevel);
    CHECK_STATUS(EG_setOutLevel);

    /* store the MODL pointer in the context */
    status = EG_setUserPointer(MODL->context, (void*)MODL);
    CHECK_STATUS(EG_setUserPointer);

    /* set the global level */
    MODL->level = 0;
    MODL->scope[MODL->level] = 0;

    /* reset the number of Groups */
    MODL->ngroup = 0;

    /* reset the prefix */
    MODL->prefix[0] = '\0';

    /* reset the last Body loaded from a Body_*.egads file */
    MODL->bodyLoaded = 0;

    /* reset the storage */
    for (istor = 0; istor < MODL->nstor; istor++) {
        FREE(MODL->stor[istor].ibody);
        FREE(MODL->stor[istor].ebody);
    }
    FREE(MODL->stor);
    MODL->nstor = 0;

    /* remove internal Parameters that may be left over from a failure
          in a Sketch that was being solved */
    status = delPmtrByName(MODL, "::d");
    if (status != OCSM_NAME_NOT_FOUND) {
        CHECK_STATUS(delPmtrByName);
    }

    status = delPmtrByName(MODL, "::z");
    if (status != OCSM_NAME_NOT_FOUND) {
        CHECK_STATUS(delPmtrByName);
    }

    status = delPmtrByName(MODL, "::y");
    if (status != OCSM_NAME_NOT_FOUND) {
        CHECK_STATUS(delPmtrByName);
    }

    status = delPmtrByName(MODL, "::x");
    if (status != OCSM_NAME_NOT_FOUND) {
        CHECK_STATUS(delPmtrByName);
    }

    /* errors if any DESPMTR Parameters are out of bounds.  this can
       only happen based upon values in the .csm file */
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (MODL->pmtr[ipmtr].type != OCSM_DESPMTR) continue;

        indx = 0;
        for (irow = 0; irow < MODL->pmtr[ipmtr].nrow; irow++) {
            for (icol = 0; icol < MODL->pmtr[ipmtr].ncol; icol++) {
                if (MODL->pmtr[ipmtr].value[indx] < MODL->pmtr[ipmtr].lbnd[indx]) {
                    *builtTo = -99999;
                    status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                         "value of %s (%f) is lower  than lower bound (%f)",
                                         MODL->pmtr[ipmtr].name,
                                         MODL->pmtr[ipmtr].value[indx],
                                         MODL->pmtr[ipmtr].lbnd[ indx]);
                    goto cleanup;        // cannot be caught
                }

                if (MODL->pmtr[ipmtr].value[indx] > MODL->pmtr[ipmtr].ubnd[indx]) {
                    *builtTo = -99999;
                    status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                         "value of %s (%f) is higher than upper bound (%f)",
                                         MODL->pmtr[ipmtr].name,
                                         MODL->pmtr[ipmtr].value[indx],
                                         MODL->pmtr[ipmtr].ubnd[ indx]);
                    goto cleanup;        // cannot be caught
                }

                indx++;
            }
        }
    }

    /* count number of non-zero dots associated with DESPMTRs */
    MODL->numdots = 0;
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (MODL->pmtr[ipmtr].type  != OCSM_DESPMTR) continue;

        indx = 0;
        for (irow = 0; irow < MODL->pmtr[ipmtr].nrow; irow++) {
            for (icol = 0; icol < MODL->pmtr[ipmtr].ncol; icol++) {
                if (MODL->pmtr[ipmtr].dot[indx] != 0) {
                    (MODL->numdots)++;
                }

                indx++;
            }
        }
    }

    /* reinitailize values of all OUTPMTR Parameters */
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (MODL->pmtr[ipmtr].type  != OCSM_OUTPMTR) continue;

        if (MODL->pmtr[ipmtr].str != NULL) {
            MODL->pmtr[ipmtr].str[0] = '\0';
        } else {
            indx = 0;
            for (irow = 0; irow < MODL->pmtr[ipmtr].nrow; irow++) {
                for (icol = 0; icol < MODL->pmtr[ipmtr].ncol; icol++) {
                    MODL->pmtr[ipmtr].value[indx] = -HUGEQ;
                    indx++;
                }
            }
        }
    }

    /* convert any internal Parameters (other than @edata and @stack) that are a matrix or
       has a string value into a scalar (so that rebuilding will start with
       exactly the same assumptions as the first build) */
    for (ipmtr = MODL->npmtr; ipmtr > 0; ipmtr--) {
        if (MODL->pmtr[ipmtr].type != OCSM_LOCALVAR &&
            MODL->pmtr[ipmtr].type != OCSM_OUTPMTR    ) continue;

        if (strcmp(MODL->pmtr[ipmtr].name, "@edata") == 0 ||
            strcmp(MODL->pmtr[ipmtr].name, "@stack") == 0   ) continue;

        if        (MODL->pmtr[ipmtr].nrow >  1 || MODL->pmtr[ipmtr].ncol >  1) {
            if (strcmp(MODL->pmtr[ipmtr].name, "@sellist") != 0) {
                SPRINT1(2, "INFO:: %s is a vector and is being converted to scalar", MODL->pmtr[ipmtr].name);
            }
            MODL->pmtr[ipmtr].nrow = 1;
            MODL->pmtr[ipmtr].ncol = 1;
        } else if (MODL->pmtr[ipmtr].nrow == 0 || MODL->pmtr[ipmtr].ncol == 0) {
            SPRINT1(2, "INFO:: %s is a string and is being converted to scalar", MODL->pmtr[ipmtr].name);
            FREE(MODL->pmtr[ipmtr].str);

            MALLOC(MODL->pmtr[ipmtr].value, double, 1);
            MALLOC(MODL->pmtr[ipmtr].dot,   double, 1);

            MODL->pmtr[ipmtr].value[0] = -HUGEQ;
            MODL->pmtr[ipmtr].dot[  0] = 0;

            if (MODL->pmtr[ipmtr].type == OCSM_DESPMTR ||
                MODL->pmtr[ipmtr].type == OCSM_CFGPMTR   ) {
                MALLOC(MODL->pmtr[ipmtr].lbnd,  double, 1);
                MALLOC(MODL->pmtr[ipmtr].ubnd,  double, 1);

                MODL->pmtr[ipmtr].lbnd[ 0] = -HUGEQ;
                MODL->pmtr[ipmtr].ubnd[ 0] = +HUGEQ;
            }

            MODL->pmtr[ipmtr].nrow = 1;
            MODL->pmtr[ipmtr].ncol = 1;
        }
    }

    /* remove the tessellation velocity cache if it exists */
    status = removeVels(MODL, 0);
    CHECK_STATUS(removeVels);

    /* initialize the hasdots flag for all Bodys */
    for (ibody = 1; ibody <= MODL->nbody; ibody++) {
        MODL->body[ibody].hasdots = 0;
        MODL->body[ibody].hasdxyz = NULL;
    }

    /* remove dots from all Bodys (possibly left over from previous build) */
    for (ibody = 1; ibody <= MODL->nbody; ibody++) {
        if (MODL->body[ibody].ebody != NULL) {
            status = EG_setGeometry_dot(MODL->body[ibody].ebody, 0, 0, NULL, NULL, NULL);
            CHECK_STATUS(EG_setGeometry_dot);

            SPRINT2(2, "removing dot from ibody=%d (line %d)", ibody, __LINE__);
        }
    }

    /* remove any EBodys that might exist */
    for (ibody = 1; ibody <= MODL->nbody; ibody++) {
        if (MODL->body[ibody].eetess != NULL) {
            status = EG_deleteObject(MODL->body[ibody].eetess);
            CHECK_STATUS(EG_deleteObject);

            MODL->body[ibody].eetess = NULL;
        }

        if (MODL->body[ibody].eebody != NULL) {
            status = EG_deleteObject(MODL->body[ibody].eebody);
            CHECK_STATUS(EG_deleteObject);

            MODL->body[ibody].eebody = NULL;
        }
    }

    /* if buildTo was set to -1, clear all udp storage */
    if (buildTo == -1) {
        for (ibody = 1; ibody <= MODL->nbody; ibody++) {
            if (MODL->body[ibody].brtype != OCSM_UDPRIM) continue;

            for (iprim = 0; iprim < MAXPRIM; iprim++) {
                for (i = 1; i <= MODL->NumUdp[iprim]; i++) {
                    if (MODL->Udps[iprim][i].ebody == MODL->body[ibody].ebody) {
                        MODL->Udps[iprim][i].ebody = NULL;  // mark for deletion

                        /* note that this guard is needed because it is possible
                           that a UDPRIM that failed will cause the system to think
                           that this is a UDPRIM when in fact it is something else */
                        if (MODL->body[ibody].arg[1].str != NULL) {
                            status = udp_clean(MODL->body[ibody].arg[1].str, MODL);
                            if (status == EGADS_NOTFOUND) status = EGADS_SUCCESS;
                            CHECK_STATUS(udp_clean);
                        }
                    }
                }
            }
        }

        /* also mark all Branches as being dirty */
        for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
            MODL->brch[ibrch].dirty = 1;
        }

        buildTo = 0;
    }

    /* find the first Body that has to be regenerated, because its
       Branch is dirty */
    MODL->recycle = 0;
    for (ibody = 1; ibody <= MODL->nbody; ibody++) {
        ibrch = MODL->body[ibody].ibrch;
        if (MODL->nbrch == 0 || ibrch == 0 || ibrch > MODL->nbrch || MODL->brch[ibrch].dirty > 0) {

            /* free up all Bodys starting at ibody */
            for (jbody = ibody; jbody <= MODL->nbody; jbody++) {

                status = removeVels(MODL, jbody);
                CHECK_STATUS(removeVels);

                for (jface = 1; jface <= MODL->body[jbody].nface; jface++) {
                    if (MODL->body[jbody].face[jface].eggdata != NULL) {
                        status = MODL->eggFree(MODL->body[jbody].face[jface].eggdata);
                        CHECK_STATUS(eggFree);
                    }
                }

                status = freeBody(MODL, jbody);
                CHECK_STATUS(freeBody);

                if (MODL->body[jbody].etess != NULL) {
                    status = EG_deleteObject(MODL->body[jbody].etess);
                    CHECK_STATUS(EG_deleteObject);

                    MODL->body[jbody].etess = NULL;
                }

                if (MODL->body[jbody].ebody != NULL) {
                    status = EG_deleteObject(MODL->body[jbody].ebody);
                    if (status == EGADS_EMPTY) status = SUCCESS;
                    CHECK_STATUS(EG_deleteObject);

                    MODL->body[jbody].ebody = NULL;
                }
            }

            /* remove any .ichld pointers to any Bodys at
               ibody or beyond */
            for (jbody = 1; jbody <= MODL->nbody; jbody++) {
                if (MODL->body[jbody].ichld >= ibody) {
                    MODL->body[jbody].ichld = 0;
                }
            }

            break;
        }
        MODL->recycle = ibody;
    }
    MODL->nbody = 0;

    /* any Body after ibody should be freed */
    for (jbody = MODL->recycle+1; jbody <= MODL->mbody; jbody++) {
        status = freeBody(MODL, jbody);
        CHECK_STATUS(freeBody);
    }

    /* initialize the stack */
    nstack = 0;

    /* initialize the number of active Sketch points (and constraints) */
    sket->type   = 0;
    sket->size   = 0;
    sket->solved = 1;
    sket->signal = 0;
    sket->nseg   = 0;
    sket->nvar   = 0;
    sket->ncon   = 0;

    /* initiailze the number of nested patterns and the number of
       patterns when a signal was detected */
    npatn     = 0;
    npatn_sig = -1;

    /* initialize the number of solver constraints */
    nsolvar = 0;
    nsolcon = 0;

    /* initialize the storage locations */
    for (imacro = 1; imacro <= MAX_NUM_MACROS; imacro++) {
        macros[imacro] = -1;
    }

    /* initialize the arguments */
    for (iarg = 1; iarg < 10; iarg++) {
        args[iarg].nval = -1;
        args[iarg].nrow =  0;
        args[iarg].ncol =  0;
        args[iarg].val  = NULL;
        args[iarg].dot  = NULL;
    }

    /* if buildTo was set to zero, reset it (locally) to be greater
       then MODL->nbrch */
    if (buildTo == 0) {
        buildTo = MODL->nbrch + 1;
    }

    /* initially we do not need finite differences and we have not
       yet calculated the mass properties */
    if (MODL->forceFDs == 0) {
        MODL->needFDs = FORCE_FINITE_DIFFS;
    } else {
        MODL->needFDs = 1;
    }
    if (MODL->needFDs != 0) {
        SPRINT0(1, "INFO:: analytic sensitivities unavailable at initialization");
    }
    MODL->hasMPs  = 0;

    /* set up the at-Parameters (so that @version gets set) */
    status = setupAtPmtrs(MODL, 0);
    CHECK_STATUS(setupAtPmtrs);

    /* loop through and process all the Branches (up to buildTo) */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        nstackSave = nstack;
        nbodySave  = MODL->nbody;

        type = MODL->brch[ibrch].type;

        MODL->ibrch = ibrch;

        if (MODL->sigCode == 0) {
            ibrchSave = ibrch;
        }

        /* print all Pmtrs */
        if (PRINT_PARAMETERS == 1) {
            SPRINT2(1, "MODL->brch[%3d].type=%s", ibrch, ocsmGetText(MODL->brch[ibrch].type));
            SPRINT0(1, "    name                 type       scope nrow ncol");
            for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
                SPRINT2x(1, "%3d %-20s ", ipmtr,
                         MODL->pmtr[ipmtr].name);
                SPRINT2x(1, "%-10s %5d ",
                         ocsmGetText(MODL->pmtr[ipmtr].type),
                         MODL->pmtr[ipmtr].scope);
                SPRINT2x(1, "%4d %4d",
                         MODL->pmtr[ipmtr].nrow,
                         MODL->pmtr[ipmtr].ncol);
                if (MODL->pmtr[ipmtr].nrow == 0) {
                    SPRINT1(1, " %s", MODL->pmtr[ipmtr].str);
                } else {
                    for (i = 0; i < MODL->pmtr[ipmtr].nrow*MODL->pmtr[ipmtr].ncol; i++) {
                        SPRINT1x(1, " %12.6f", MODL->pmtr[ipmtr].value[i]);
                    }
                    SPRINT0(1, " ");
                }
            }
        }

        /* exit if we exceeded buildTo */
        if (ibrch > buildTo) {
            break;
        }

        /* if Branch is deferred, mark it as active but skip it for now */
        if (MODL->brch[ibrch].actv == OCSM_DEFERRED) {
            MODL->brch[ibrch].actv =  OCSM_ACTIVE;
            SPRINT1(1, "    deferring [%4d]:", ibrch);
            continue;
        }

        /* skip this Branch if it is not active */
        if (MODL->brch[ibrch].actv != OCSM_ACTIVE) {
            SPRINT1(1, "    skipping  [%4d]:", ibrch);
            continue;
        }

        /* if there is an uncaught signal, determine what to do with
           it.  note: checking if CATBEG matches the signal is done during
           the execution of the CATBEG */
        if (MODL->sigCode != 0) {

            /* if this is the first time we have seen the signal, remember the
               number of entries in the pattern table */
            if (npatn_sig == -1) {
                npatn_sig = npatn;
            }

            /* if this is the beginning of a block, add it to the pattern list */
            if        (type == OCSM_PATBEG ||
                       type == OCSM_IFTHEN ||
                       type == OCSM_RECALL   ) {
                if (npatn < MAX_NESTING) {
                    patn[npatn].itype = type;
                    patn[npatn].ibeg  = ibrch;
                    npatn++;
                    continue;
                } else {
                    status = signalError(MODL, OCSM_NESTED_TOO_DEEPLY,
                                         "npatn=%d >0 MAX_NESTING=%d", npatn, MAX_NESTING);
                    goto cleanup;        // cannot be caught
                }

            /* if this is the beginning of a UDC, add it to the pattern list (and
               increment the level) */
            } else if (type == OCSM_UDPRIM) {
                if (isalpha(MODL->brch[ibrch].arg1[1]) != 0) {
                    continue;
                } else if (npatn < MAX_NESTING) {
                    patn[npatn].itype = type;
                    patn[npatn].ibeg  = ibrch;
                    npatn++;

                    (MODL->level)++;
                    continue;
                } else {
                    status = signalError(MODL, OCSM_NESTED_TOO_DEEPLY,
                                         "npatn=%d >0 MAX_NESTING=%d", npatn, MAX_NESTING);
                    goto cleanup;        // cannot be caught
                }

            /* if this is the end of a UDC, remove it and the variables at the
               local scope from the pattern list */
            } else if (type == OCSM_END) {
                if (MODL->level > 0 && MODL->scope[MODL->level] > MODL->scope[MODL->level-1]) {
                    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
                        if (MODL->pmtr[ipmtr].scope >= MODL->scope[MODL->level]) {
                            FREE(MODL->pmtr[ipmtr].name );
                            FREE(MODL->pmtr[ipmtr].value);
                            FREE(MODL->pmtr[ipmtr].dot  );
                            FREE(MODL->pmtr[ipmtr].lbnd );
                            FREE(MODL->pmtr[ipmtr].ubnd );
                            FREE(MODL->pmtr[ipmtr].str  );

                            for (jpmtr = ipmtr+1; jpmtr <= MODL->npmtr; jpmtr++) {

                                /* copy jpmtr over jpmtr-1 */
                                MODL->pmtr[jpmtr-1].name  = MODL->pmtr[jpmtr].name;
                                MODL->pmtr[jpmtr-1].type  = MODL->pmtr[jpmtr].type;
                                MODL->pmtr[jpmtr-1].scope = MODL->pmtr[jpmtr].scope;
                                MODL->pmtr[jpmtr-1].mprop = MODL->pmtr[jpmtr].mprop;
                                MODL->pmtr[jpmtr-1].nrow  = MODL->pmtr[jpmtr].nrow;
                                MODL->pmtr[jpmtr-1].ncol  = MODL->pmtr[jpmtr].ncol;
                                MODL->pmtr[jpmtr-1].value = MODL->pmtr[jpmtr].value;
                                MODL->pmtr[jpmtr-1].dot   = MODL->pmtr[jpmtr].dot;
                                MODL->pmtr[jpmtr-1].lbnd  = MODL->pmtr[jpmtr].lbnd;
                                MODL->pmtr[jpmtr-1].ubnd  = MODL->pmtr[jpmtr].ubnd;
                                MODL->pmtr[jpmtr-1].str   = MODL->pmtr[jpmtr].str;
                            }

                            ipmtr--;            /* revisit imptr in next trip through loop */

                            MODL->pmtr[MODL->npmtr].name  = NULL;
                            MODL->pmtr[MODL->npmtr].value = NULL;
                            MODL->pmtr[MODL->npmtr].dot   = NULL;
                            MODL->pmtr[MODL->npmtr].lbnd  = NULL;
                            MODL->pmtr[MODL->npmtr].ubnd  = NULL;
                            MODL->pmtr[MODL->npmtr].str   = NULL;
                            (MODL->npmtr)--;
                        }
                    }
                }

                /* decrement the level */
                (MODL->level)--;

                npatn--;
                npatn_sig = MIN(npatn_sig, npatn);
                continue;

            /* if this is the end of a block, remove it from the pattern list */
            } else if (type == OCSM_PATEND ||
                       type == OCSM_ENDIF  ||
                       type == OCSM_MACEND   ) {
                npatn--;
                npatn_sig = MIN(npatn_sig, npatn);
                continue;

            /* if this is a CATBEG but we are in a block that was entered
               after the signal was created, skip the CATBEG */
            } else if (type == OCSM_CATBEG && npatn > npatn_sig) {
                continue;

            /* otherwise if this is not a CATBEG, go to the next Branch */
            } else if (type != OCSM_CATBEG) {
                continue;
            }
        }

        /* if this is an executable Sketch statement and we need
           to solve the Sketch, do it before interpreting the
           arguments and velocities */
        if (type == OCSM_LINSEG || type == OCSM_CIRARC ||
            type == OCSM_ARC    || type == OCSM_SPLINE ||
            type == OCSM_SSLOPE || type == OCSM_BEZIER   ) {
            status = solveSketch(MODL, sket);

            if (status == OCSM_UNDERCONSTRAINED ||
                status == OCSM_OVERCONSTRAINED  ||
                status == OCSM_NOT_CONVERGED      ) {
                sket->signal = status;
                status = SUCCESS;
            } else {
                CATCH_STATUS(solveSketch);
            }
        }

        SPRINT1x(3, "----------%s", ocsmGetText(type));
        if (MODL->brch[ibrch].narg >= 1) {SPRINT1x(3, "  %s", MODL->brch[ibrch].arg1);}
        if (MODL->brch[ibrch].narg >= 2) {SPRINT1x(3, "  %s", MODL->brch[ibrch].arg2);}
        if (MODL->brch[ibrch].narg >= 3) {SPRINT1x(3, "  %s", MODL->brch[ibrch].arg3);}
        if (MODL->brch[ibrch].narg >= 4) {SPRINT1x(3, "  %s", MODL->brch[ibrch].arg4);}
        if (MODL->brch[ibrch].narg >= 5) {SPRINT1x(3, "  %s", MODL->brch[ibrch].arg5);}
        if (MODL->brch[ibrch].narg >= 6) {SPRINT1x(3, "  %s", MODL->brch[ibrch].arg6);}
        if (MODL->brch[ibrch].narg >= 7) {SPRINT1x(3, "  %s", MODL->brch[ibrch].arg7);}
        if (MODL->brch[ibrch].narg >= 8) {SPRINT1x(3, "  %s", MODL->brch[ibrch].arg8);}
        if (MODL->brch[ibrch].narg >= 9) {SPRINT1x(3, "  %s", MODL->brch[ibrch].arg9);}
        SPRINT0(3, "----------");

        /* get the values for the arguments and velocities */
        for (iarg = 1; iarg < 10; iarg++) {
            args[iarg].nval = -1;
            args[iarg].nrow =  0;
            args[iarg].ncol =  0;

            if (iarg > MODL->brch[ibrch].narg) continue;

                                  thisArg[0] = '\0';
            if (iarg == 1) strcpy(thisArg, MODL->brch[ibrch].arg1);
            if (iarg == 2) strcpy(thisArg, MODL->brch[ibrch].arg2);
            if (iarg == 3) strcpy(thisArg, MODL->brch[ibrch].arg3);
            if (iarg == 4) strcpy(thisArg, MODL->brch[ibrch].arg4);
            if (iarg == 5) strcpy(thisArg, MODL->brch[ibrch].arg5);
            if (iarg == 6) strcpy(thisArg, MODL->brch[ibrch].arg6);
            if (iarg == 7) strcpy(thisArg, MODL->brch[ibrch].arg7);
            if (iarg == 8) strcpy(thisArg, MODL->brch[ibrch].arg8);
            if (iarg == 9) strcpy(thisArg, MODL->brch[ibrch].arg9);

            if        (iarg == 1 && type == OCSM_SET) {      // allows implicit string
                STRNCPY(str, &(thisArg[1]), MAX_STRVAL_LEN);
            } else if (iarg == 1 && type == OCSM_PATBEG) {   // allows implicit string
                STRNCPY(str, &(thisArg[1]), MAX_STRVAL_LEN);
            } else if (iarg == 1 && type == OCSM_GETATTR) {  // allows implicit string
                STRNCPY(str, &(thisArg[1]), MAX_STRVAL_LEN);
            } else if (iarg == 4 && type == OCSM_SKCON) {    // delays parse of constraint
                STRNCPY(str, &(thisArg[1]), MAX_STRVAL_LEN);
            } else if (iarg == 1 && type == OCSM_SOLCON) {   // delays parse of constraint
                STRNCPY(str, &(thisArg[1]), MAX_STRVAL_LEN);
            } else if (iarg >  4 && type == OCSM_IFTHEN) {   // allows for lazy evaluations
                break;
            } else if (iarg >  4 && type == OCSM_ELSEIF) {   // allows for lazy evaluations
                break;
            } else if (iarg >  4 && type == OCSM_PATBREAK){  // allows for lazy evaluations
                break;
            } else if (npatn > 0 && patn[npatn-1].itype == OCSM_IFTHEN && patn[npatn-1].ncopy == 1 &&
                                    type == OCSM_ELSEIF) {   // do not eval if IFTHEN succeeded
                break;
            } else {
                status = str2vals(thisArg, MODL, &(args[iarg].nrow), &(args[iarg].ncol), &values, &dots, str);

                if (status < SUCCESS) {
                    SPRINT4(0, "ERROR:: problem evaluating argument %d (%s) for Branch %d (%s)",
                           iarg, thisArg, ibrch, ocsmGetText(MODL->brch[ibrch].type));

                    FREE(values);
                    FREE(dots  );

                    /* if there was an evaluation error during the start (or middle)
                       of a block of statements, jump to after the block */
                    if        (MODL->brch[ibrch].type == OCSM_PATBEG) {
                        icount = 1;
                        for (jbrch = ibrch+1; jbrch <= MODL->nbrch; jbrch++) {
                            if (MODL->brch[jbrch].type == OCSM_PATBEG) icount++;
                            if (MODL->brch[jbrch].type == OCSM_PATEND) icount--;

                            if (icount == 0) {
                                ibrch = jbrch;
                                break;
                            }
                        }
                    } else if (MODL->brch[ibrch].type == OCSM_PATBREAK) {
                        npatn--;

                        icount = 1;
                        for (jbrch = ibrch+1; jbrch <= MODL->nbrch; jbrch++) {
                            if (MODL->brch[jbrch].type == OCSM_PATBEG) icount++;
                            if (MODL->brch[jbrch].type == OCSM_PATEND) icount--;

                            if (icount == 0) {
                                ibrch = jbrch;
                                break;
                            }
                        }
                    } else if (MODL->brch[ibrch].type == OCSM_IFTHEN) {
                        icount = 1;
                        for (jbrch = ibrch+1; jbrch <= MODL->nbrch; jbrch++) {
                            if (MODL->brch[jbrch].type == OCSM_IFTHEN) icount++;
                            if (MODL->brch[jbrch].type == OCSM_ENDIF ) icount--;

                            if (icount == 0) {
                                ibrch = jbrch;
                                break;
                            }
                        }
                    } else if (MODL->brch[ibrch].type == OCSM_ELSEIF) {
                        npatn--;

                        icount = 1;
                        for (jbrch = ibrch+1; jbrch <= MODL->nbrch; jbrch++) {
                            if (MODL->brch[jbrch].type == OCSM_IFTHEN) icount++;
                            if (MODL->brch[jbrch].type == OCSM_ENDIF ) icount--;

                            if (icount == 0) {
                                ibrch = jbrch;
                                break;
                            }
                        }
                    } else if (MODL->brch[ibrch].type == OCSM_SKBEG) {
                        for (jbrch = ibrch+1; jbrch <= MODL->nbrch; jbrch++) {
                            if (MODL->brch[jbrch].type == OCSM_SKEND) {
                                ibrch = jbrch;
                                break;
                            }
                        }
                    } else if (MODL->brch[ibrch].type == OCSM_SOLBEG) {
                        icount = 1;
                        for (jbrch = ibrch+1; jbrch <= MODL->nbrch; jbrch++) {
                            if (MODL->brch[jbrch].type == OCSM_SOLBEG) icount++;
                            if (MODL->brch[jbrch].type == OCSM_SOLEND) icount--;

                            if (icount == 0) {
                                ibrch = jbrch;
                                break;
                            }
                        }
                    }
                    goto next_branch;
                }
            }

            if (STRLEN(str) > 0) {
                FREE(args[iarg].str);
                MALLOC(args[iarg].str, char, MAX_STRVAL_LEN);

                args[iarg].nval = 0;
                STRNCPY(args[iarg].str, str, MAX_STRVAL_LEN);
            } else {
                args[iarg].nval = (args[iarg].nrow) * (args[iarg].ncol);

                FREE(  args[iarg].val);   /* also free's .str since they are unioned */
                MALLOC(args[iarg].val, double, args[iarg].nval);

                FREE(  args[iarg].dot);
                MALLOC(args[iarg].dot, double, args[iarg].nval);

                SPLINT_CHECK_FOR_NULL(values);
                SPLINT_CHECK_FOR_NULL(dots  );

                for (i = 0; i < args[iarg].nval; i++) {
                    args[iarg].val[i] = values[i];
                    args[iarg].dot[i] = dots[  i];
                }
            }

            FREE(values);
            FREE(dots  );
        }

        /* if this is a UDPARG or UDPRIM statment for a UDF or UDF and if it
           has an +ATTRREAL or -ATTRREAL argument that has dots, we will need to
           compute sensitivities via finite differences */
        if (MODL->needFDs == 0 && (type == OCSM_UDPARG || type == OCSM_UDPRIM)) {
            if (isalpha(args[1].str[0]) != 0) {

                status = udp_initialize(args[1].str, MODL, &udp_num, &udp_names, &udp_types, &udp_idef, &udp_ddef);

                if (status == EGADS_NOLOAD) {
                    status = EGADS_SUCCESS;
                } else if (status == EGADS_NULLOBJ) {
                    status = signalError(MODL, OCSM_UDP_ERROR1,
                                         "UDP/UDF \"%s\" could not be loaded", args[1].str);
                    goto cleanup;        // cannot be caught
                }
                CHECK_STATUS(udp_initialize);

                for (iarg = 2; iarg < MODL->brch[ibrch].narg; iarg+=2) {

                    /* set iflag is the argument has any non-zero dots */
                    iflag = 0;
                    for (i = 0; i < args[iarg+1].nval; i++) {
                        if (fabs(args[iarg+1].dot[i]) > EPS12) {
                            iflag++;
                        }
                    }
                    if (iflag == 0) continue;

                    /* see if the corresponding UDP argument is an ATTRINT or ATTRREAL */
                    for (i = 0; i < udp_num; i++) {
                        if (strcasecmp(args[iarg].str, udp_names[i]) == 0) {
                            if (udp_types[i] == +ATTRREAL || udp_types[i] == -ATTRREAL) {
                                if (MODL->needFDs == 0) {
                                    MODL->needFDs = 1;
                                    SPRINT1(1, "--> analytic sensitivities unavailable because \"%s\" has non-zero dots", args[iarg].str);
                                }
                                break;
                            }
                        }
                    }
                }
            }
        }

        /* make sure that there is enough room on the stack */
        if (nstack >= MAX_STACK_SIZE-1) {
            status = signalError(MODL, OCSM_TOO_MANY_BODYS_ON_STACK,
                                 "too many Bodys on Stack");
            goto cleanup;        // cannot be caught
        }

        MODL->brch[ibrch].ileft = -1;
        MODL->brch[ibrch].irite = -1;
        MODL->brch[ibrch].ichld = -1;

        /* if there is a BcstCallback defined, broadcast message now */
        if (MODL->bcstCB != NULL) {
            snprintf(temp, MAX_STRVAL_LEN, "updateSolveBtn|1|building %s (%d/%d)|",
                     ocsmGetText(MODL->brch[ibrch].type), ibrch, MODL->nbrch);
            MODL->bcstCB(temp);
        }

        /* execute Branch ibrch */
        old_time = clock();

        if        (MODL->brch[ibrch].bclass == OCSM_PRIMITIVE) {
            status = buildPrimitive(MODL, ibrch, args, &nstack, stack, npatn, patn);
            if (MODL->sigCode != SUCCESS) goto next_branch;
            CATCH_STATUS(buildPrimitive);

            if ( MODL->brch[ibrch].type == OCSM_UDPRIM) {
                if (MODL->brch[ibrch].arg1[1] == '/' ||
                    MODL->brch[ibrch].arg1[1] == '$'   ) {
                    if (npatn < MAX_NESTING) {
                        patn[npatn].itype = OCSM_UDPRIM;
                        patn[npatn].ibeg  = ibrch;
                        npatn++;
                    } else {
                        status = signalError(MODL, OCSM_NESTED_TOO_DEEPLY,
                                             "npatn=%d >0 MAX_NESTING=%d", npatn, MAX_NESTING);
                        goto cleanup;        // cannot be caught
                    }
                }
            }
        } else if (MODL->brch[ibrch].bclass == OCSM_GROWN) {
            status = buildGrown(MODL, ibrch, args, &nstack, stack, npatn, patn);
            if (MODL->sigCode != SUCCESS) goto next_branch;
            CATCH_STATUS(buildGrown);
        } else if (MODL->brch[ibrch].bclass == OCSM_APPLIED) {
            status = buildApplied(MODL, ibrch, args, &nstack, stack, npatn, patn);
            if (MODL->sigCode != SUCCESS) goto next_branch;
            CATCH_STATUS(buildApplied);
        } else if (MODL->brch[ibrch].bclass == OCSM_BOOLEAN) {
            status = buildBoolean(MODL, ibrch, args, &nstack, stack, npatn, patn);
            if (MODL->sigCode != SUCCESS) goto next_branch;
            CATCH_STATUS(buildBoolean);
        } else if (MODL->brch[ibrch].bclass == OCSM_TRANSFORM) {
            status = buildTransform(MODL, ibrch, args, &nstack, stack);
            if (MODL->sigCode != SUCCESS) goto next_branch;
            CATCH_STATUS(buildTransform);
        } else if (MODL->brch[ibrch].bclass == OCSM_SKETCH) {
            status = buildSketch(MODL, ibrch, args, &nstack, stack, npatn, patn, sket);
            if (MODL->sigCode == OCSM_UNDERCONSTRAINED ||
                MODL->sigCode == OCSM_OVERCONSTRAINED  ||
                MODL->sigCode == OCSM_NOT_CONVERGED      ) {
                // do nothing
            } else if (MODL->sigCode != SUCCESS) {
                goto next_branch;
            }
//            CATCH_STATUS(buildSketch);

            if (MODL->brch[ibrch].type != OCSM_SKBEG &&
                MODL->brch[ibrch].type != OCSM_SKVAR &&
                MODL->brch[ibrch].type != OCSM_SKCON   ) {
                MODL->brch[ibrch  ].ileft = ibrch - 1;
                MODL->brch[ibrch-1].ichld = ibrch;
            }
        } else if (MODL->brch[ibrch].bclass == OCSM_SOLVER) {
            status = buildSolver(MODL, ibrch, args, &nsolvar, solvars, &nsolcon, solcons);
            if (MODL->sigCode != SUCCESS) goto next_branch;
            CATCH_STATUS(buildSolver);

        /* execute: "dimension $pmtrName nrow ncol=1" */
        } else if (type == OCSM_DIMENSION) {
            SPRINT4(1, "    executing [%4d] dimension:     %s  %11.5f  %11.5f",
                    ibrch, args[1].str, args[2].val[0], args[3].val[0]);

            status = ocsmNewPmtr(MODL, args[1].str, OCSM_UNKNOWN,
                                 NINT(args[2].val[0]), NINT(args[3].val[0]));
            CHECK_STATUS(ocsmNewPmtr);

        /* execute: "set $pmtrName exprs" */
        } else if (type == OCSM_SET) {
            SPRINT3(1, "    executing [%4d] set:            %s  %s",
                    ibrch, args[1].str, MODL->brch[ibrch].arg2);

            /* if $pmtrName is an implicit string, evaluate it */
            if (args[1].str[0] != '!') {
                STRNCPY(pname, args[1].str, MAX_EXPR_LEN);
            } else {
                status = str2val(args[1].str, MODL, &value, &dot, pname);
                if (MODL->sigCode != SUCCESS) goto next_branch;
                CHECK_STATUS(str2val);
            }

            /* break pname into pmtrName[irow,icol] */
            status = parseName(MODL, pname, pmtrName, &jpmtr, &irow, &icol);
            CATCH_STATUS(parseName);

            /* if it does not exist, create it now */
            if (jpmtr == 0) {
                status = ocsmNewPmtr(MODL, pmtrName, OCSM_LOCALVAR, 1, 1);
                if (status == OCSM_ILLEGAL_PMTR_NAME) {
                    status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                         "\"%s\" is an illegal Parameter name", pmtrName);
                    goto cleanup;    // cannot be caught
                }
                CHECK_STATUS(ocsmNewPmtr);

                jpmtr = MODL->npmtr;

            /* make sure that Parameter is INTERNAL or OUTPMTR */
            } else if (MODL->pmtr[jpmtr].type == OCSM_DESPMTR) {
                status = signalError(MODL, OCSM_PMTR_IS_DESPMTR,
                                     "\"%s\" cannot be SET because it is a DESPMTR", pmtrName);
                goto cleanup;        // cannot be caught
            } else if (MODL->pmtr[jpmtr].type == OCSM_CFGPMTR) {
                status = signalError(MODL, OCSM_PMTR_IS_DESPMTR,
                                     "\"%s\" cannot be SET because it is a CFGPMTR", pmtrName);
                goto cleanup;        // cannot be caught
            } else if (MODL->pmtr[jpmtr].type == OCSM_CONPMTR) {
                status = signalError(MODL, OCSM_PMTR_IS_CONPMTR,
                                     "\"%s\" cannot be SET because it is a CONPMTR", pmtrName);
                goto cleanup;        // cannot be caught

            /* convert UNKNOWN to LOCALVAR */
            } else if (MODL->pmtr[jpmtr].type == OCSM_UNKNOWN) {
                MODL->pmtr[jpmtr].type = OCSM_LOCALVAR;
            }

            /* string-value mode (expression evaluates to string) */
            if (args[2].nval == 0) {
                if (irow != 0 || icol != 0) {
                    status = OCSM_WRONG_PMTR_TYPE;
                    CATCH_STATUS(set);
                }

                /* remove previous values and velocities and make
                   room for string */
                if (MODL->pmtr[jpmtr].str == NULL) {
                    FREE(MODL->pmtr[jpmtr].value);
                    FREE(MODL->pmtr[jpmtr].dot  );
                    FREE(MODL->pmtr[jpmtr].lbnd );
                    FREE(MODL->pmtr[jpmtr].ubnd );

                    MALLOC(MODL->pmtr[jpmtr].str, char, MAX_STRVAL_LEN);

                    MODL->pmtr[jpmtr].nrow = 0;
                    MODL->pmtr[jpmtr].ncol = 0;
                }

                /* store the string value */
                status = ocsmSetValu(MODL, jpmtr, 1, 1, MODL->brch[ibrch].arg2);
                CATCH_STATUS(ocsmSetValu);

                status = ocsmGetValuS(MODL, jpmtr, thisArg);
                CATCH_STATUS(ocsmGetValuS);

                SPRINT2(1, "                          %s = $%s",
                        pmtrName, thisArg);

            /* single-set mode (non-string) */
            } else if (args[2].nval == 1) {

                /* if currently a string value, remove it and make
                   room for single value */
                if (MODL->pmtr[jpmtr].str != NULL) {
                    FREE(MODL->pmtr[jpmtr].str);

                    MALLOC(MODL->pmtr[jpmtr].value, double, 1);
                    MALLOC(MODL->pmtr[jpmtr].dot,   double, 1);

                    MODL->pmtr[jpmtr].value[0] = -HUGEQ;
                    MODL->pmtr[jpmtr].dot[  0] = 0;

                    MODL->pmtr[jpmtr].nrow = 1;
                    MODL->pmtr[jpmtr].ncol = 1;
                }

                /* store the value(s) */
                if (irow == 0 && icol == 0) {
                    if (MODL->pmtr[jpmtr].nrow > 1 || MODL->pmtr[jpmtr].ncol > 1) {
                        for (irow = 1; irow <= MODL->pmtr[jpmtr].nrow; irow++) {
                            for (icol = 1; icol <= MODL->pmtr[jpmtr].ncol; icol++) {
                                status = ocsmSetValuD(MODL, jpmtr, irow, icol, args[2].val[0]);
                                CHECK_STATUS(ocsmSetValuD);

                                status = ocsmSetVelD( MODL, jpmtr, irow, icol, args[2].dot[0]);
                                CHECK_STATUS(ocsmSetVelD);

                                SPRINT5(1, "                          %s[%d,%d] = %11.5f %11.5f",
                                        pmtrName, irow, icol, args[2].val[0], args[2].dot[0]);
                            }
                        }
                    } else {
                        status = ocsmSetValuD(MODL, jpmtr, 1, 1, args[2].val[0]);
                        CHECK_STATUS(ocsmSetValuD);

                        status = ocsmSetVelD( MODL, jpmtr, 1, 1, args[2].dot[0]);
                        CHECK_STATUS(ocsmSetVelD);

                        SPRINT3(1, "                          %s = %11.5f %11.5f",
                                pmtrName, args[2].val[0], args[2].dot[0]);
                    }
                } else if (icol == 0) {
                    if (irow < 1 || irow > MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol) {
                        status = signalError(MODL, OCSM_ILLEGAL_PMTR_INDEX,
                                             "index must be between 1 and %d", MODL->pmtr[jpmtr].nrow);
                        goto next_branch;
                    } else {
                        status = ocsmSetValuD(MODL, jpmtr, irow, icol, args[2].val[0]);
                        CATCH_STATUS(ocsmSetValuD);

                        status = ocsmSetVelD( MODL, jpmtr, irow, icol, args[2].dot[0]);
                        CATCH_STATUS(ocsmSetVelD);

                        SPRINT3(1, "                          %s = %11.5f %11.5f",
                                pmtrName, args[2].val[0], args[2].dot[0]);
                    }
                } else if (irow < 1 || irow > MODL->pmtr[jpmtr].nrow) {
                    status = signalError(MODL, OCSM_ILLEGAL_PMTR_INDEX,
                                         "row index must be between 1 and %d", MODL->pmtr[jpmtr].nrow);
                    goto next_branch;
                } else if (icol < 1 || icol > MODL->pmtr[jpmtr].ncol) {
                    status = signalError(MODL, OCSM_ILLEGAL_PMTR_INDEX,
                                         "column index must be between 1 and %d", MODL->pmtr[jpmtr].ncol);
                    goto next_branch;
                } else {
                    status = ocsmSetValuD(MODL, jpmtr, irow, icol, args[2].val[0]);
                    CATCH_STATUS(ocsmSetValuD);

                    status = ocsmSetVelD( MODL, jpmtr, irow, icol, args[2].dot[0]);
                    CATCH_STATUS(ocsmSetVelD);

                    SPRINT3(1, "                          %s = %11.5f %11.5f",
                            pmtrName, args[2].val[0], args[2].dot[0]);
                }

            /* multi-set mode */
            } else {

                /* we cannot be storing to a string value */
                if (MODL->pmtr[jpmtr].str != NULL) {
                    status = OCSM_WRONG_PMTR_TYPE;
                    CATCH_STATUS(set);
                }

                /* store the values */
                icount = 0;
                value  = 0;
                dot    = 0;
                for (irow = 1; irow <= MODL->pmtr[jpmtr].nrow; irow++) {
                    for (icol = 1; icol <= MODL->pmtr[jpmtr].ncol; icol++) {

                        /* next value (if it exists; otherwise it is extended) */
                        if (icount < args[2].nval) {
                            value = args[2].val[icount];
                            dot   = args[2].dot[icount];
                        }

                        status = ocsmSetValuD(MODL, jpmtr, irow, icol, value);
                        CATCH_STATUS(ocsmSetValuD);

                        status = ocsmSetVelD(MODL, jpmtr, irow, icol, dot);
                        CATCH_STATUS(ocsmSetVelD);

                        if (icount < args[2].nval) {
                            if (value != -HUGEQ) {
                                SPRINT5(1, "                          %s[%d,%d] = %11.5f %11.5f",
                                        pmtrName, irow, icol, value, dot);
                            } else {
                                SPRINT3(1, "                          %s[%d,%d] = undefined",
                                        pmtrName, irow, icol);
                            }
                        } else {
                            if (value != -HUGEQ) {
                                SPRINT5(1, "                          %s[%d,%d] = %11.5f %11.5f (extended)",
                                        pmtrName, irow, icol, value, dot);
                            } else {
                                SPRINT3(1, "                          %s[%d,%d] = undefined (extended)",
                                        pmtrName, irow, icol);
                            }
                        }

                        icount++;
                    }
                }
            }

            /* if the SET statment has Attributes, create the global Attributes now */
            if (MODL->brch[ibrch].nattr > 0) {
                for (iattr = 0; iattr < MODL->brch[ibrch].nattr; iattr++) {
                    status = ocsmSetAttr(MODL, 0, MODL->brch[ibrch].attr[iattr].name,
                                         MODL->brch[ibrch].attr[iattr].defn);
                    CHECK_STATUS(ocsmSetAttr);
                }
            }

        /* execute: "evaluate $type ..." */
        } else if (type == OCSM_EVALUATE) {

            jpmtr = 0;
            for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
                if (strcmp(MODL->pmtr[ipmtr].name, "@edata") == 0      &&
                    MODL->pmtr[ipmtr].scope == MODL->scope[MODL->level]  ) {
                    jpmtr = ipmtr;
                    break;
                }
            }

            if (jpmtr == 0) {
                status = signalError(MODL, OCSM_INTERNAL_ERROR,
                                     "@data could not be found");
                goto cleanup;        // cannot be caught
            }

            /* "evaluate $node ibody inode" */
            if        (strcmp(args[1].str, "node") == 0 ||
                       strcmp(args[1].str, "NODE") == 0   ) {
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                inode = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (inode < 1 || inode > MODL->body[ibody].nnode) {
                    status = signalError(MODL, OCSM_NODE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Node");
                    goto next_branch;
                }

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 3;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = MODL->body[ibody].node[inode].x;
                MODL->pmtr[jpmtr].value[ 1] = MODL->body[ibody].node[inode].y;
                MODL->pmtr[jpmtr].value[ 2] = MODL->body[ibody].node[inode].z;

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;
                MODL->pmtr[jpmtr].dot[   2] = 0;

                if (MODL->numdots > 0) {
                    MALLOC(dots, double, 3);

                    status = velocityOfNode(MODL, ibody, inode, dots);
                    CHECK_STATUS(velocityOfNode);

                    MODL->pmtr[jpmtr].dot[0] = dots[0];
                    MODL->pmtr[jpmtr].dot[1] = dots[1];
                    MODL->pmtr[jpmtr].dot[2] = dots[2];

                    FREE(dots);
                }

            /* "evaluate $edge ibody iedge t"    or
               "evaluate $edge ibody iedge $beg" or
               "evaluate $edge ibody iedge $end"    */
            } else if (strcmp(args[1].str, "edge") == 0 ||
                       strcmp(args[1].str, "EDGE") == 0   ) {
                if (args[4].nval == 0) {
                    SPRINT5(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f  %s",
                            ibrch, args[1].str, args[2].val[0],
                                                args[3].val[0],
                                                args[4].str);
                } else {
                    SPRINT5(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f  %11.5f",
                            ibrch, args[1].str, args[2].val[0],
                                                args[3].val[0],
                                                args[4].val[0]);
                }

                ibody = NINT(args[2].val[0]);
                iedge = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iedge < 1 || iedge > MODL->body[ibody].nedge) {
                    status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Edge");
                    goto next_branch;
                } else if (args[4].nval == 0               &&
                           strcmp(args[4].str, "beg") != 0 &&
                           strcmp(args[4].str, "end") != 0   ) {
                    status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                         "EVALUATE EDGE expects a number or \"beg\" or \"end\"");
                    goto next_branch;
                }

                status = EG_getRange(MODL->body[ibody].edge[iedge].eedge, uvrange, &periodic);
                CHECK_STATUS(EG_getRange);

                if        (strcmp(args[4].str, "beg") == 0) {
                    uv_[0] = uvrange[0];
                } else if (strcmp(args[4].str, "end") == 0) {
                    uv_[0] = uvrange[1];
                } else if (args[4].val[0] < uvrange[0]) {
                    uv_[0] = uvrange[0];
                } else if (args[4].val[0] > uvrange[1]) {
                    uv_[0] = uvrange[1];
                } else {
                    uv_[0] = args[4].val[0];
                }

                status = EG_evaluate(MODL->body[ibody].edge[iedge].eedge, uv_, data);
                CHECK_STATUS(EG_evaluate);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 10;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = uv_[0];
                MODL->pmtr[jpmtr].value[ 1] = data[0];
                MODL->pmtr[jpmtr].value[ 2] = data[1];
                MODL->pmtr[jpmtr].value[ 3] = data[2];
                MODL->pmtr[jpmtr].value[ 4] = data[3];
                MODL->pmtr[jpmtr].value[ 5] = data[4];
                MODL->pmtr[jpmtr].value[ 6] = data[5];
                MODL->pmtr[jpmtr].value[ 7] = data[6];
                MODL->pmtr[jpmtr].value[ 8] = data[7];
                MODL->pmtr[jpmtr].value[ 9] = data[8];

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;
                MODL->pmtr[jpmtr].dot[   2] = 0;
                MODL->pmtr[jpmtr].dot[   3] = 0;
                MODL->pmtr[jpmtr].dot[   4] = 0;
                MODL->pmtr[jpmtr].dot[   5] = 0;
                MODL->pmtr[jpmtr].dot[   6] = 0;
                MODL->pmtr[jpmtr].dot[   7] = 0;
                MODL->pmtr[jpmtr].dot[   8] = 0;
                MODL->pmtr[jpmtr].dot[   9] = 0;

                if (MODL->numdots > 0) {
                    MALLOC(dots, double, 3);

                    if        (strcmp(args[4].str, "beg") == 0) {
                        ibeg = MODL->body[ibody].edge[iedge].ibeg;

                        status = velocityOfNode(MODL, ibody, ibeg, dots);
                        CHECK_STATUS(velocityOfNode);

                        MODL->pmtr[jpmtr].dot[0] = (data[3] * dots[0] + data[4] * dots[1] + data[5] * dots[2])
                                                 / (data[3] * data[3] + data[4] * data[4] + data[5] * data[5]);
                        MODL->pmtr[jpmtr].dot[1] = dots[0];
                        MODL->pmtr[jpmtr].dot[2] = dots[1];
                        MODL->pmtr[jpmtr].dot[3] = dots[2];

                    } else if (strcmp(args[4].str, "end") == 0) {
                        iend = MODL->body[ibody].edge[iedge].iend;

                        status = velocityOfNode(MODL, ibody, iend, dots);
                        CHECK_STATUS(velocityOfNode);

                        MODL->pmtr[jpmtr].dot[0] = (data[3] * dots[0] + data[4] * dots[1] + data[5] * dots[2])
                                                 / (data[3] * data[3] + data[4] * data[4] + data[5] * data[5]);
                        MODL->pmtr[jpmtr].dot[1] = dots[0];
                        MODL->pmtr[jpmtr].dot[2] = dots[1];
                        MODL->pmtr[jpmtr].dot[3] = dots[2];

                    } else {
                        status = ocsmGetVel(MODL, ibody, OCSM_EDGE, iedge, 1, uv_, dots);
                        CHECK_STATUS(ocsmGetVel);

                        uv_dot[0] = args[4].dot[0];

                        MODL->pmtr[jpmtr].dot[1] = dots[0] + data[3] * uv_dot[0];
                        MODL->pmtr[jpmtr].dot[2] = dots[1] + data[4] * uv_dot[0];
                        MODL->pmtr[jpmtr].dot[3] = dots[2] + data[5] * uv_dot[0];
                    }

                    FREE(dots);
                }

            /* "evaluate $edgebbox ibody iedge" */
            } else if (strcmp(args[1].str, "edgebbox") == 0 ||
                       strcmp(args[1].str, "EDGEBBOX") == 0   ) {
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iedge = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iedge < 1 || iedge > MODL->body[ibody].nedge) {
                    status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Edge");
                    goto next_branch;
                }

                if (MODL->needFDs == 0) {
                    MODL->needFDs = 1;
                    SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE EDGEBBOX");
                }

                status = EG_getBoundingBox(MODL->body[ibody].edge[iedge].eedge, bbox);
                CHECK_STATUS(EG_getBoundingBox);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 6;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = bbox[0];
                MODL->pmtr[jpmtr].value[ 1] = bbox[1];
                MODL->pmtr[jpmtr].value[ 2] = bbox[2];
                MODL->pmtr[jpmtr].value[ 3] = bbox[3];
                MODL->pmtr[jpmtr].value[ 4] = bbox[4];
                MODL->pmtr[jpmtr].value[ 5] = bbox[5];

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;
                MODL->pmtr[jpmtr].dot[   2] = 0;
                MODL->pmtr[jpmtr].dot[   3] = 0;
                MODL->pmtr[jpmtr].dot[   4] = 0;
                MODL->pmtr[jpmtr].dot[   5] = 0;

            /* "evaluate $edgerng ibody iedge" */
            } else if (strcmp(args[1].str, "edgerng") == 0 ||
                       strcmp(args[1].str, "EDGERNG") == 0   ) {
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iedge = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iedge < 1 || iedge > MODL->body[ibody].nedge) {
                    status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Edge");
                    goto next_branch;
                }

                status = EG_getRange(MODL->body[ibody].edge[iedge].eedge, uvrange, &periodic);
                CHECK_STATUS(EG_getRange);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 2;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = uvrange[0];
                MODL->pmtr[jpmtr].value[ 1] = uvrange[1];

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;

                if (MODL->numdots > 0) {
                    MALLOC(dots, double, 3);

                    status = EG_evaluate(MODL->body[ibody].edge[iedge].eedge, &uvrange[0], data);
                    CHECK_STATUS(EG_evaluate);

                    ibeg = MODL->body[ibody].edge[iedge].ibeg;

                    status = velocityOfNode(MODL, ibody, ibeg, dots);
                    CHECK_STATUS(velocityOfNode);

                    MODL->pmtr[jpmtr].dot[0] = (data[3] * dots[0] + data[4] * dots[1] + data[5] * dots[2])                                           / (data[3] * data[3] + data[4] * data[4] + data[5] * data[5]);

                    status = EG_evaluate(MODL->body[ibody].edge[iedge].eedge, &uvrange[1], data);
                    CHECK_STATUS(EG_evaluate);

                    iend = MODL->body[ibody].edge[iedge].iend;

                    status = velocityOfNode(MODL, ibody, iend, dots);
                    CHECK_STATUS(velocityOfNode);

                    MODL->pmtr[jpmtr].dot[1] = (data[3] * dots[0] + data[4] * dots[1] + data[5] * dots[2])                                           / (data[3] * data[3] + data[4] * data[4] + data[5] * data[5]);

                    FREE(dots);
                }

            /* "evaluate $edgeinv ibody iedge x y z" */
            } else if (strcmp(args[1].str, "edgeinv") == 0 ||
                       strcmp(args[1].str, "EDGEINV") == 0   ) {
                SPRINT7(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f  %11.5f  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0],
                                            args[4].val[0],
                                            args[5].val[0],
                                            args[6].val[0]);

                ibody = NINT(args[2].val[0]);
                iedge = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iedge < 1 || iedge > MODL->body[ibody].nedge) {
                    status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Edge");
                    goto next_branch;
                }

                if (MODL->needFDs == 0) {
                    MODL->needFDs = 1;
                    SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE EDGEINV");
                }

                xyz_[0] = args[4].val[0];
                xyz_[1] = args[5].val[0];
                xyz_[2] = args[6].val[0];

                status = EG_invEvaluate(MODL->body[ibody].edge[iedge].eedge, xyz_, uv_, data);
                CHECK_STATUS(EG_invEvaluate);

                status = EG_getRange(MODL->body[ibody].edge[iedge].eedge, uvrange, &periodic);
                CHECK_STATUS(EG_getRange);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 5;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = uv_[ 0];
                MODL->pmtr[jpmtr].value[ 1] = data[0];
                MODL->pmtr[jpmtr].value[ 2] = data[1];
                MODL->pmtr[jpmtr].value[ 3] = data[2];
                MODL->pmtr[jpmtr].value[ 4] = (uv_[0] - uvrange[0]) / (uvrange[1] - uvrange[0]);

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;
                MODL->pmtr[jpmtr].dot[   2] = 0;
                MODL->pmtr[jpmtr].dot[   3] = 0;
                MODL->pmtr[jpmtr].dot[   4] = 0;

            /* "evaluate $edgekt ibody iedge" */
            } else if (strcmp(args[1].str, "edgekt") == 0 ||
                       strcmp(args[1].str, "EDGEKT") == 0   ) {
                int *header;
                double *rdata, *rdata2, *rdata_dot=NULL;
                ego ecurve;
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iedge = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iedge < 1 || iedge > MODL->body[ibody].nedge) {
                    status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Edge");
                    goto next_branch;
                }

                status = EG_getTopology(MODL->body[ibody].edge[iedge].eedge, &ecurve, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getGeometry(ecurve, &oclass, &mtype, &eref, &header, &rdata);
                CHECK_STATUS(EG_getGeometry);

                if (oclass != CURVE || mtype != BSPLINE) {
                    status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                         "Edge not associated with a Bspline curve");

                    if (header != NULL) EG_free(header);
                    if (rdata  != NULL) EG_free(rdata );

                    goto next_branch;
                }

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = header[3];
                MODL->pmtr[jpmtr].ncol = 1;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                for (i = 0; i < header[3]; i++) {
                    MODL->pmtr[jpmtr].value[i] = rdata[i];
                    MODL->pmtr[jpmtr].dot[  i] = 0;
                }

                /* return sensitivity if it exists on the geometry */
                if (MODL->numdots > 0) {

                    /* do a dummy evaluation so the the dots get filled */
                    status = EG_hasGeometry_dot(MODL->body[ibody].edge[iedge].eedge);
                    if (status == EGADS_NOTFOUND) {
                        uv_[0] = rdata[0];

                        MALLOC(dots, double, 3);

                        MODL->body[ibody].hasdots = 1;
                        status = velocityOfEdge(MODL, ibody, iedge, 1, uv_, dots);
                        CHECK_STATUS(velocityOfEdge);
                        MODL->body[ibody].hasdots = 0;

                        FREE(dots);
                    }

                    status = EG_hasGeometry_dot(MODL->body[ibody].edge[iedge].eedge);
                    if (status == EGADS_SUCCESS) {
                        status = EG_getGeometry_dot(ecurve, &rdata2, &rdata_dot);
                        CHECK_STATUS(EG_getGeometry_dot);

                        EG_free(rdata2);

                        SPLINT_CHECK_FOR_NULL(rdata_dot);

                        for (i = 0; i < header[3]; i++) {
                            MODL->pmtr[jpmtr].dot[i] = rdata_dot[i];
                        }
                    } else if (MODL->needFDs == 0) {
                        MODL->needFDs = 1;
                        SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE EDGEKT");
                    }
                }

                if (header    != NULL) EG_free(header   );
                if (rdata     != NULL) EG_free(rdata    );
                if (rdata_dot != NULL) EG_free(rdata_dot);

            /* "evaluate $edgecp ibody iedge" */
            } else if (strcmp(args[1].str, "edgecp") == 0 ||
                       strcmp(args[1].str, "EDGECP") == 0   ) {
                int *header;
                double *rdata, *rdata2, *rdata_dot=NULL;
                ego ecurve;
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iedge = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iedge < 1 || iedge > MODL->body[ibody].nedge) {
                    status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Edge");
                    goto next_branch;
                }

                status = EG_getTopology(MODL->body[ibody].edge[iedge].eedge, &ecurve, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getGeometry(ecurve, &oclass, &mtype, &eref, &header, &rdata);
                CHECK_STATUS(EG_getGeometry);

                if (oclass != CURVE || mtype != BSPLINE) {
                    status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                         "Edge not associated with a Bspline curve");

                    if (header != NULL) EG_free(header);
                    if (rdata  != NULL) EG_free(rdata );

                    goto next_branch;
                }

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = header[2];
                MODL->pmtr[jpmtr].ncol = 3;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                for (i = 0; i < header[2]*3; i++) {
                    MODL->pmtr[jpmtr].value[i] = rdata[header[3]+i];
                    MODL->pmtr[jpmtr].dot[  i] = 0;
                }

                /* return sensitivity if it exists on the geometry */
                if (MODL->numdots > 0) {

                    /* do a dummy evaluation so the the dots get filled */
                    status = EG_hasGeometry_dot(MODL->body[ibody].edge[iedge].eedge);
                    if (status == EGADS_NOTFOUND) {
                        uv_[0] = rdata[0];

                        MALLOC(dots, double, 3);

                        MODL->body[ibody].hasdots = 1;
                        status = velocityOfEdge(MODL, ibody, iedge, 1, uv_, dots);
                        CHECK_STATUS(velocityOfEdge);
                        MODL->body[ibody].hasdots = 0;

                        FREE(dots);
                    }

                    status = EG_hasGeometry_dot(MODL->body[ibody].edge[iedge].eedge);
                    if (status == EGADS_SUCCESS) {
                        status = EG_getGeometry_dot(ecurve, &rdata2, &rdata_dot);
                        CHECK_STATUS(EG_getGeometry_dot);

                        EG_free(rdata2);

                        SPLINT_CHECK_FOR_NULL(rdata_dot);

                        for (i = 0; i < header[2]*3; i++) {
                            MODL->pmtr[jpmtr].dot[i] = rdata_dot[header[3]+i];
                        }
                    } else if (MODL->needFDs == 0) {
                        MODL->needFDs = 1;
                        SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE EDGECP");
                    }
                }

                if (header    != NULL) EG_free(header   );
                if (rdata     != NULL) EG_free(rdata    );
                if (rdata_dot != NULL) EG_free(rdata_dot);

            /* "evaluate $edgetess ibody iedge" */
            } else if (strcmp(args[1].str, "edgetess") == 0 ||
                       strcmp(args[1].str, "EDGETESS") == 0   ) {
                CDOUBLE *t;

                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iedge = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iedge < 1 || iedge > MODL->body[ibody].nedge) {
                    status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Edge");
                    goto next_branch;
                }

                /* make sure we have a tessellation */
                status = ocsmTessellate(MODL, ibody);
                CHECK_STATUS(ocsmTessellate);

                status = EG_getTessEdge(MODL->body[ibody].etess, iedge,
                                        &npnt, &xyz, &t);
                CHECK_STATUS(EG_getTessEdge);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 1;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[0] = npnt;
                MODL->pmtr[jpmtr].dot[  0] = 0;

            /* "evaluate $face ibody iface u v" */
            } else if (strcmp(args[1].str, "face") == 0 ||
                       strcmp(args[1].str, "FACE") == 0   ) {
                SPRINT6(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0],
                                            args[4].val[0],
                                            args[5].val[0]);

                ibody = NINT(args[2].val[0]);
                iface = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iface < 1 || iface > MODL->body[ibody].nface) {
                    status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Face");
                    goto next_branch;
                }

                status = EG_getRange(MODL->body[ibody].face[iface].eface, uvrange, &periodic);
                CHECK_STATUS(EG_getRange);

                if        (args[4].val[0] < uvrange[0]) {
                    uv_[0] = uvrange[0];
                } else if (args[4].val[0] > uvrange[1]) {
                    uv_[0] = uvrange[1];
                } else {
                    uv_[0] = args[4].val[0];
                }

                if        (args[5].val[0] < uvrange[2]) {
                    uv_[1] = uvrange[2];
                } else if (args[5].val[0] > uvrange[3]) {
                    uv_[1] = uvrange[3];
                } else {
                    uv_[1] = args[5].val[0];
                }

                status = EG_getTopology(MODL->body[ibody].face[iface].eface, &eref, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_evaluate(MODL->body[ibody].face[iface].eface, uv_, data);
                CHECK_STATUS(EG_evaluate);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 23;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = uv_[  0];
                MODL->pmtr[jpmtr].value[ 1] = uv_[  1];
                MODL->pmtr[jpmtr].value[ 2] = data[ 0];
                MODL->pmtr[jpmtr].value[ 3] = data[ 1];
                MODL->pmtr[jpmtr].value[ 4] = data[ 2];
                MODL->pmtr[jpmtr].value[ 5] = data[ 3];
                MODL->pmtr[jpmtr].value[ 6] = data[ 4];
                MODL->pmtr[jpmtr].value[ 7] = data[ 5];
                MODL->pmtr[jpmtr].value[ 8] = data[ 6];
                MODL->pmtr[jpmtr].value[ 9] = data[ 7];
                MODL->pmtr[jpmtr].value[10] = data[ 8];
                MODL->pmtr[jpmtr].value[11] = data[ 9];
                MODL->pmtr[jpmtr].value[12] = data[10];
                MODL->pmtr[jpmtr].value[13] = data[11];
                MODL->pmtr[jpmtr].value[14] = data[12];
                MODL->pmtr[jpmtr].value[15] = data[13];
                MODL->pmtr[jpmtr].value[16] = data[14];
                MODL->pmtr[jpmtr].value[17] = data[15];
                MODL->pmtr[jpmtr].value[18] = data[16];
                MODL->pmtr[jpmtr].value[19] = data[17];
                MODL->pmtr[jpmtr].value[20] = (data[4] * data[8] - data[5] * data[7]) * mtype;
                MODL->pmtr[jpmtr].value[21] = (data[5] * data[6] - data[3] * data[8]) * mtype;
                MODL->pmtr[jpmtr].value[22] = (data[3] * data[7] - data[4] * data[6]) * mtype;

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;
                MODL->pmtr[jpmtr].dot[   2] = 0;
                MODL->pmtr[jpmtr].dot[   3] = 0;
                MODL->pmtr[jpmtr].dot[   4] = 0;
                MODL->pmtr[jpmtr].dot[   5] = 0;
                MODL->pmtr[jpmtr].dot[   6] = 0;
                MODL->pmtr[jpmtr].dot[   7] = 0;
                MODL->pmtr[jpmtr].dot[   8] = 0;
                MODL->pmtr[jpmtr].dot[   9] = 0;
                MODL->pmtr[jpmtr].dot[  10] = 0;
                MODL->pmtr[jpmtr].dot[  11] = 0;
                MODL->pmtr[jpmtr].dot[  12] = 0;
                MODL->pmtr[jpmtr].dot[  13] = 0;
                MODL->pmtr[jpmtr].dot[  14] = 0;
                MODL->pmtr[jpmtr].dot[  15] = 0;
                MODL->pmtr[jpmtr].dot[  16] = 0;
                MODL->pmtr[jpmtr].dot[  17] = 0;
                MODL->pmtr[jpmtr].dot[  18] = 0;
                MODL->pmtr[jpmtr].dot[  19] = 0;
                MODL->pmtr[jpmtr].dot[  20] = 0;
                MODL->pmtr[jpmtr].dot[  21] = 0;
                MODL->pmtr[jpmtr].dot[  22] = 0;

                if (MODL->numdots > 0) {
                    MALLOC(dots, double, 3);

                    status = ocsmGetVel(MODL, ibody, OCSM_FACE, iface, 1, uv_, dots);
                    CHECK_STATUS(ocsmGetVel);

                    uv_dot[0] = args[4].dot[0];
                    uv_dot[1] = args[5].dot[0];

                    MODL->pmtr[jpmtr].dot[2] = dots[0] + data[3] * uv_dot[0] + data[6] * uv_dot[1];
                    MODL->pmtr[jpmtr].dot[3] = dots[1] + data[4] * uv_dot[0] + data[7] * uv_dot[1];
                    MODL->pmtr[jpmtr].dot[4] = dots[2] + data[5] * uv_dot[0] + data[8] * uv_dot[1];

                    FREE(dots);
                }

            /* "evaluate $facebbox ibody iface" */
            } else if (strcmp(args[1].str, "facebbox") == 0 ||
                       strcmp(args[1].str, "FACEBBOX") == 0   ) {
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iface = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iface < 1 || iface > MODL->body[ibody].nface) {
                    status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Face");
                    goto next_branch;
                }

                if (MODL->needFDs == 0) {
                    MODL->needFDs = 1;
                    SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE FACEBBOX");
                }

                status = EG_getBoundingBox(MODL->body[ibody].face[iface].eface, bbox);
                CHECK_STATUS(EG_getBoundingBox);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 6;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = bbox[0];
                MODL->pmtr[jpmtr].value[ 1] = bbox[1];
                MODL->pmtr[jpmtr].value[ 2] = bbox[2];
                MODL->pmtr[jpmtr].value[ 3] = bbox[3];
                MODL->pmtr[jpmtr].value[ 4] = bbox[4];
                MODL->pmtr[jpmtr].value[ 5] = bbox[5];

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;
                MODL->pmtr[jpmtr].dot[   2] = 0;
                MODL->pmtr[jpmtr].dot[   3] = 0;
                MODL->pmtr[jpmtr].dot[   4] = 0;
                MODL->pmtr[jpmtr].dot[   5] = 0;

            /* "evaluate $facerng ibody iface" */
            } else if (strcmp(args[1].str, "facerng") == 0 ||
                       strcmp(args[1].str, "FACERNG") == 0   ) {
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iface = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iface < 1 || iface > MODL->body[ibody].nface) {
                    status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Face");
                    goto next_branch;
                }

                if (MODL->needFDs == 0) {
                    MODL->needFDs = 1;
                    SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE FACERNG");
                }

                status = EG_getRange(MODL->body[ibody].face[iface].eface, uvrange, &periodic);
                CHECK_STATUS(EG_getRange);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 4;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = uvrange[0];
                MODL->pmtr[jpmtr].value[ 1] = uvrange[1];
                MODL->pmtr[jpmtr].value[ 2] = uvrange[2];
                MODL->pmtr[jpmtr].value[ 3] = uvrange[3];

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;
                MODL->pmtr[jpmtr].dot[   2] = 0;
                MODL->pmtr[jpmtr].dot[   3] = 0;

            /* "evaluate $faceinv ibody iface x y z" */
            } else if (strcmp(args[1].str, "faceinv") == 0 ||
                       strcmp(args[1].str, "FACEINV") == 0   ) {
                SPRINT7(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f  %11.5f  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0],
                                            args[4].val[0],
                                            args[5].val[0],
                                            args[6].val[0]);

                ibody = NINT(args[2].val[0]);
                iface = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iface < 1 || iface > MODL->body[ibody].nface) {
                    status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Face");
                    goto next_branch;
                }

                if (MODL->needFDs == 0) {
                    MODL->needFDs = 1;
                    SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE FACEINV");
                }

                xyz_[0] = args[4].val[0];
                xyz_[1] = args[5].val[0];
                xyz_[2] = args[6].val[0];

                status = EG_invEvaluate(MODL->body[ibody].face[iface].eface, xyz_, uv_, data);
                CHECK_STATUS(EG_invEvaluate);

                status = EG_getRange(MODL->body[ibody].face[iface].eface, uvrange, &periodic);
                CHECK_STATUS(EG_getRange);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 7;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = uv_[0];
                MODL->pmtr[jpmtr].value[ 1] = uv_[1];
                MODL->pmtr[jpmtr].value[ 2] = data[0];
                MODL->pmtr[jpmtr].value[ 3] = data[1];
                MODL->pmtr[jpmtr].value[ 4] = data[2];
                MODL->pmtr[jpmtr].value[ 5] = (uv_[0] - uvrange[0]) / (uvrange[1] - uvrange[0]);
                MODL->pmtr[jpmtr].value[ 6] = (uv_[1] - uvrange[2]) / (uvrange[3] - uvrange[2]);

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;
                MODL->pmtr[jpmtr].dot[   2] = 0;
                MODL->pmtr[jpmtr].dot[   3] = 0;
                MODL->pmtr[jpmtr].dot[   4] = 0;
                MODL->pmtr[jpmtr].dot[   5] = 0;
                MODL->pmtr[jpmtr].dot[   6] = 0;

            /* "evaluate $faceukt ibody iface" */
            } else if (strcmp(args[1].str, "faceukt") == 0 ||
                       strcmp(args[1].str, "FACEUKT") == 0   ) {
                int *header;
                double *rdata, *rdata2, *rdata_dot=NULL;
                ego esurface;
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iface = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iface < 1 || iface > MODL->body[ibody].nface) {
                    status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Face");
                    goto next_branch;
                }

                status = EG_getTopology(MODL->body[ibody].face[iface].eface, &esurface, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getGeometry(esurface, &oclass, &mtype, &eref, &header, &rdata);
                CHECK_STATUS(EG_getGeometry);

                if (oclass != SURFACE || mtype != BSPLINE) {
                    status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                         "Face not associated with a Bspline surface");

                    if (header != NULL) EG_free(header);
                    if (rdata  != NULL) EG_free(rdata );

                    goto next_branch;
                }

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = header[3];
                MODL->pmtr[jpmtr].ncol = 1;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                for (i = 0; i < header[3]; i++) {
                    MODL->pmtr[jpmtr].value[i] = rdata[i];
                    MODL->pmtr[jpmtr].dot[  i] = 0;
                }

                /* return sensitivity if it exists on the geometry */
                if (MODL->numdots > 0) {

                    /* do a dummy evaluation so the the dots get filled */
                    status = EG_hasGeometry_dot(MODL->body[ibody].face[iface].eface);
                    if (status == EGADS_NOTFOUND) {
                        uv_[0] = rdata[0];
                        uv_[1] = rdata[header[3]];

                        MALLOC(dots, double, 3);

                        MODL->body[ibody].hasdots = 1;
                        status = velocityOfFace(MODL, ibody, iface, 1, uv_, dots);
                        CHECK_STATUS(velocityOfFace);
                        MODL->body[ibody].hasdots = 0;

                        FREE(dots);
                    }

                    status = EG_hasGeometry_dot(MODL->body[ibody].face[iface].eface);
                    if (status == EGADS_SUCCESS) {
                        status = EG_getGeometry_dot(esurface, &rdata2, &rdata_dot);
                        CHECK_STATUS(EG_getGeometry_dot);

                        EG_free(rdata2);

                        SPLINT_CHECK_FOR_NULL(rdata_dot);

                        for (i = 0; i < header[3]; i++) {
                            MODL->pmtr[jpmtr].dot[i] = rdata_dot[i];
                        }
                    } else if (MODL->needFDs == 0) {
                        MODL->needFDs = 1;
                        SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE FACEUKT");
                    }
                }

                if (header    != NULL) EG_free(header   );
                if (rdata     != NULL) EG_free(rdata    );
                if (rdata_dot != NULL) EG_free(rdata_dot);

            /* "evaluate $facevkt ibody iface" */
            } else if (strcmp(args[1].str, "facevkt") == 0 ||
                       strcmp(args[1].str, "FACEVKT") == 0   ) {
                int *header;
                double *rdata, *rdata2, *rdata_dot=NULL;
                ego esurface;
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iface = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iface < 1 || iface > MODL->body[ibody].nface) {
                    status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Face");
                    goto next_branch;
                }

                status = EG_getTopology(MODL->body[ibody].face[iface].eface, &esurface, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getGeometry(esurface, &oclass, &mtype, &eref, &header, &rdata);
                CHECK_STATUS(EG_getGeometry);

                if (oclass != SURFACE || mtype != BSPLINE) {
                    status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                         "Face not associated with a Bspline surface");

                    if (header != NULL) EG_free(header);
                    if (rdata  != NULL) EG_free(rdata );

                    goto next_branch;
                }

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = header[6];
                MODL->pmtr[jpmtr].ncol = 1;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                for (i = 0; i < header[6]; i++) {
                    MODL->pmtr[jpmtr].value[i] = rdata[header[3]+i];
                    MODL->pmtr[jpmtr].dot[  i] = 0;
                }

                /* return sensitivity if it exists on the geometry */
                if (MODL->numdots > 0) {

                    /* do a dummy evaluation so the the dots get filled */
                    status = EG_hasGeometry_dot(MODL->body[ibody].face[iface].eface);
                    if (status == EGADS_NOTFOUND) {
                        uv_[0] = rdata[0];
                        uv_[1] = rdata[header[3]];

                        MALLOC(dots, double, 3);

                        MODL->body[ibody].hasdots = 1;
                        status = velocityOfFace(MODL, ibody, iface, 1, uv_, dots);
                        CHECK_STATUS(velocityOfFace);
                        MODL->body[ibody].hasdots = 0;

                        FREE(dots);
                    }

                    status = EG_hasGeometry_dot(MODL->body[ibody].face[iface].eface);
                    if (status == EGADS_SUCCESS) {
                        status = EG_getGeometry_dot(esurface, &rdata2, &rdata_dot);
                        CHECK_STATUS(EG_getGeometry_dot);

                        EG_free(rdata2);

                        SPLINT_CHECK_FOR_NULL(rdata_dot);

                        for (i = 0; i < header[6]; i++) {
                            MODL->pmtr[jpmtr].dot[i] = rdata_dot[header[3]+i];
                        }
                    } else if (MODL->needFDs == 0) {
                        MODL->needFDs = 1;
                        SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE FACEVKT");
                    }
                }

                if (header    != NULL) EG_free(header   );
                if (rdata     != NULL) EG_free(rdata    );
                if (rdata_dot != NULL) EG_free(rdata_dot);

            /* "evaluate $facecp ibody iface" */
            } else if (strcmp(args[1].str, "facecp") == 0 ||
                       strcmp(args[1].str, "FACECP") == 0   ) {
                int *header;
                double *rdata, *rdata2, *rdata_dot=NULL;
                ego esurface;
                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iface = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iface < 1 || iface > MODL->body[ibody].nface) {
                    status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Face");
                    goto next_branch;
                }

                status = EG_getTopology(MODL->body[ibody].face[iface].eface, &esurface, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getGeometry(esurface, &oclass, &mtype, &eref, &header, &rdata);
                CHECK_STATUS(EG_getGeometry);

                if (oclass != SURFACE || mtype != BSPLINE) {
                    status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                         "Face not associated with a Bspline surface");

                    if (header != NULL) EG_free(header);
                    if (rdata  != NULL) EG_free(rdata );

                    goto next_branch;
                }

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = header[2] * header[5];
                MODL->pmtr[jpmtr].ncol = 3;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                for (i = 0; i < header[2]*header[5]*3; i++) {
                    MODL->pmtr[jpmtr].value[i] = rdata[header[3]+header[6]+i];
                    MODL->pmtr[jpmtr].dot[  i] = 0;
                }

                /* return sensitivity if it exists on the geometry */
                if (MODL->numdots > 0) {

                    /* do a dummy evaluation so the the dots get filled */
                    status = EG_hasGeometry_dot(MODL->body[ibody].face[iface].eface);
                    if (status == EGADS_NOTFOUND) {
                        uv_[0] = rdata[0];
                        uv_[1] = rdata[header[3]];

                        MALLOC(dots, double, 3);

                        MODL->body[ibody].hasdots = 1;
                        status = velocityOfFace(MODL, ibody, iface, 1, uv_, dots);
                        CHECK_STATUS(velocityOfFace);
                        MODL->body[ibody].hasdots = 0;

                        FREE(dots);
                    }

                    status = EG_hasGeometry_dot(MODL->body[ibody].face[iface].eface);
                    if (status == EGADS_SUCCESS) {
                        status = EG_getGeometry_dot(esurface, &rdata2, &rdata_dot);
                        CHECK_STATUS(EG_getGeometry_dot);

                        EG_free(rdata2);

                        SPLINT_CHECK_FOR_NULL(rdata_dot);

                        for (i = 0; i < header[2]*header[5]*3; i++) {
                            MODL->pmtr[jpmtr].dot[i] = rdata_dot[header[3]+header[6]+i];
                        }
                    } else if (MODL->needFDs == 0) {
                        MODL->needFDs = 1;
                        SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE FACECP");
                    }
                }

                if (header    != NULL) EG_free(header   );
                if (rdata     != NULL) EG_free(rdata    );
                if (rdata_dot != NULL) EG_free(rdata_dot);

            /* "evaluate $facetess ibody iface" */
            } else if (strcmp(args[1].str, "facetess") == 0 ||
                       strcmp(args[1].str, "FACETESS") == 0   ) {

                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody = NINT(args[2].val[0]);
                iface = NINT(args[3].val[0]);

                if (ibody < 1 || ibody > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body");
                    goto next_branch;
                } else if (iface < 1 || iface > MODL->body[ibody].nface) {
                    status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                         "EVALUATE specified nonexistant Face");
                    goto next_branch;
                }

                /* make sure we have a tessellation */
                status = ocsmTessellate(MODL, ibody);
                CHECK_STATUS(ocsmTessellate);

                status = EG_getTessFace(MODL->body[ibody].etess, iface,
                                        &npnt, &xyz, &uv, &ptype, &pindx,
                                        &ntri, &tris, &tric);
                CHECK_STATUS(EG_getTessFace);

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow = 1;
                MODL->pmtr[jpmtr].ncol = 2;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[0] = npnt;
                MODL->pmtr[jpmtr].value[1] = ntri;

                MODL->pmtr[jpmtr].dot[  0] = 0;
                MODL->pmtr[jpmtr].dot[  1] = 0;

            /* "evaluate $dist ibody1 ibody2" */
            } else if (strcmp(args[1].str, "dist") == 0 ||
                       strcmp(args[1].str, "DIST") == 0   ) {

                SPRINT4(1, "    executing [%4d] evaluate:       %s  %11.5f  %11.5f",
                        ibrch, args[1].str, args[2].val[0],
                                            args[3].val[0]);

                ibody1 = NINT(args[2].val[0]);
                ibody2 = NINT(args[3].val[0]);

                if (abs(ibody1) < 1 || abs(ibody1) > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body1");
                    goto next_branch;
                } else if (ibody2 < 1 || ibody2 > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "EVALUATE specified nonexistant Body2");
                    goto next_branch;
                }

                if (MODL->needFDs == 0) {
                    MODL->needFDs = 1;
                    SPRINT0(1, "INFO:: analytic sensitivities unavailable because of EVALUATE DIST");
                }

                /* get the clearance */
                status = ocsmClearance(MODL, ibody1, ibody2, &dist, pnt1, pnt2);
                CHECK_STATUS(ocsmClearance);

                if        (NINT(pnt1[0]) > 0) {
                    status = EG_evaluate(MODL->body[abs(ibody1)].face[+NINT(pnt1[0])].eface, &(pnt1[1]), data1);
                    CHECK_STATUS(EG_evaluate);
                } else if (NINT(pnt1[0]) < 0) {
                    status = EG_evaluate(MODL->body[abs(ibody1)].edge[-NINT(pnt1[0])].eedge, &(pnt1[1]), data1);
                    CHECK_STATUS(EG_evaluate);
                } else if (NINT(pnt1[1]) > 0) {
                    status = EG_evaluate(MODL->body[abs(ibody1)].node[+NINT(pnt1[1])].enode, &(pnt1[1]), data1);
                    CHECK_STATUS(EG_evaluate);
                } else{
                    status = EG_getBoundingBox(MODL->body[abs(ibody1)].ebody, data1);
                    CHECK_STATUS(EG_getBoundingBox);

                    data1[0] = (data1[0] + data1[3]) / 2;
                    data1[1] = (data1[1] + data1[4]) / 2;
                    data1[2] = (data1[2] + data1[5]) / 2;
                }

                if        (NINT(pnt2[0]) > 0) {
                    status = EG_evaluate(MODL->body[abs(ibody2)].face[+NINT(pnt2[0])].eface, &(pnt2[1]), data2);
                    CHECK_STATUS(EG_evaluate);
                } else if (NINT(pnt2[0]) < 0) {
                    status = EG_evaluate(MODL->body[abs(ibody2)].edge[-NINT(pnt2[0])].eedge, &(pnt2[1]), data2);
                    CHECK_STATUS(EG_evaluate);
                } else if (NINT(pnt2[1]) > 0) {
                    status = EG_evaluate(MODL->body[abs(ibody2)].node[+NINT(pnt2[1])].enode, &(pnt2[1]), data2);
                    CHECK_STATUS(EG_evaluate);
                } else {
                    status = EG_getBoundingBox(MODL->body[ibody2].ebody, data2);
                    CHECK_STATUS(EG_getBoundingBox);

                    data2[0] = (data2[0] + data2[3]) / 2;
                    data2[1] = (data2[1] + data2[4]) / 2;
                    data2[2] = (data2[2] + data2[5]) / 2;
                }

                /* resize and store data */
                MODL->pmtr[jpmtr].nrow =  1;
                MODL->pmtr[jpmtr].ncol = 13;

                RALLOC(MODL->pmtr[jpmtr].value, double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);
                RALLOC(MODL->pmtr[jpmtr].dot,   double, MODL->pmtr[jpmtr].nrow*MODL->pmtr[jpmtr].ncol);

                MODL->pmtr[jpmtr].value[ 0] = dist;
                MODL->pmtr[jpmtr].value[ 1] = NINT(pnt1[0]);
                MODL->pmtr[jpmtr].value[ 2] = pnt1[1];
                MODL->pmtr[jpmtr].value[ 3] = pnt1[2];
                MODL->pmtr[jpmtr].value[ 4] = data1[0];
                MODL->pmtr[jpmtr].value[ 5] = data1[1];
                MODL->pmtr[jpmtr].value[ 6] = data1[2];
                MODL->pmtr[jpmtr].value[ 7] = NINT(pnt2[0]);
                MODL->pmtr[jpmtr].value[ 8] = pnt2[1];
                MODL->pmtr[jpmtr].value[ 9] = pnt2[2];
                MODL->pmtr[jpmtr].value[10] = data2[0];
                MODL->pmtr[jpmtr].value[11] = data2[1];
                MODL->pmtr[jpmtr].value[12] = data2[2];

                MODL->pmtr[jpmtr].dot[   0] = 0;
                MODL->pmtr[jpmtr].dot[   1] = 0;
                MODL->pmtr[jpmtr].dot[   2] = 0;
                MODL->pmtr[jpmtr].dot[   3] = 0;
                MODL->pmtr[jpmtr].dot[   4] = 0;
                MODL->pmtr[jpmtr].dot[   5] = 0;
                MODL->pmtr[jpmtr].dot[   6] = 0;
                MODL->pmtr[jpmtr].dot[   7] = 0;
                MODL->pmtr[jpmtr].dot[   8] = 0;
                MODL->pmtr[jpmtr].dot[   9] = 0;
                MODL->pmtr[jpmtr].dot[  10] = 0;
                MODL->pmtr[jpmtr].dot[  11] = 0;
                MODL->pmtr[jpmtr].dot[  12] = 0;
            }

            if (PRINT_EDATA == 1) {
                for (i = 1; i <= MODL->pmtr[ipmtr].ncol; i++) {
                    SPRINT3(1, "        @edata[%3d]=%12.6f %12.6f", i,
                            MODL->pmtr[jpmtr].value[i-1],
                            MODL->pmtr[jpmtr].dot[  i-1]);
                }
            }

        /* execute: "getattr $pmtrName attrID global=0" */
        } else if (type == OCSM_GETATTR) {
            SPRINT4(1, "    executing [%4d] getattr:        %s  %s  %s",
                    ibrch, args[1].str, MODL->brch[ibrch].arg2, MODL->brch[ibrch].arg3);

            /* if $pmtrName is an implicit string, evaluate it */
            if (args[1].str[0] != '!') {
                STRNCPY(pname, args[1].str, MAX_EXPR_LEN);
            } else {
                status = str2val(args[1].str, MODL, &value, &dot, pname);
                if (MODL->sigCode != SUCCESS) goto next_branch;
                CHECK_STATUS(str2val);
            }

            SPLINT_CHECK_FOR_NULL(pname);

            if (strlen(pname) == 0) {
                status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                     "\"%s\" does not evalaute to a string", args[1].str);
                goto cleanup;      // cannot be caught
            } else if (strstr(pname, "[") != NULL) {
                status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                     "$pmtrName cannot contain a subscript");
                goto cleanup;      // cannot be caught
            } else if (pname[0] == '@') {
                status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                     "$pmtrName cannot start with @-sign");
                goto cleanup;      // cannot be caught
            }

            /* check to see if pname already exists */
            status = ocsmFindPmtr(MODL, pname, 0, 0, 0, &jpmtr);
            CHECK_STATUS(ocsmFindPmtr);

            /* if it already exists, make sure it is a local variable or outpmtr */
            if (jpmtr > 0) {
                if (MODL->pmtr[jpmtr].type == OCSM_DESPMTR) {
                    status = signalError(MODL, OCSM_PMTR_IS_DESPMTR,
                                         "\"%s\" cannot be SET because it is a DESPMTR", pname);
                    goto cleanup;        // cannot be caught
                } else if (MODL->pmtr[jpmtr].type == OCSM_CFGPMTR) {
                    status = signalError(MODL, OCSM_PMTR_IS_DESPMTR,
                                         "\"%s\" cannot be SET because it is a CFGPMTR", pname);
                    goto cleanup;        // cannot be caught
                } else if (MODL->pmtr[jpmtr].type == OCSM_CONPMTR) {
                    status = signalError(MODL, OCSM_PMTR_IS_CONPMTR,
                                         "\"%s\" cannot be SET because it is a CONPMTR", pname);
                    goto cleanup;        // cannot be caught
                }

                /* if an unknown type, make a local variable */
                if (MODL->pmtr[jpmtr].type == OCSM_UNKNOWN) {
                    MODL->pmtr[jpmtr].type =  OCSM_LOCALVAR;
                }

            /* if it does not exist, make it now */
            } else {
                status = ocsmFindPmtr(MODL, pname, OCSM_LOCALVAR, 1, 1, &jpmtr);
                if (status != SUCCESS) {
                    status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                         "\"%s\" is not a legal Parameter name", pname);
                    goto cleanup;        // cannot be caught;
                }
            }

            status = str2val(MODL->brch[ibrch].arg2, MODL, &value, &dot, str);
            CATCH_STATUS(str2val);

            /* get the ego associated with the selected entity */
            if        (NINT(args[3].val[0]) > 0) {
                eobj = NULL;
            } else if (MODL->seltype == 0) {
                eobj = MODL->body[MODL->selbody].node[MODL->sellist[0]].enode;
            } else if (MODL->seltype == 1) {
                eobj = MODL->body[MODL->selbody].edge[MODL->sellist[0]].eedge;
            } else if (MODL->seltype == 2) {
                eobj = MODL->body[MODL->selbody].face[MODL->sellist[0]].eface;
            } else if (MODL->selbody >= 1) {
                eobj = MODL->body[MODL->selbody].ebody;
            } else {
                status = signalError(MODL, OCSM_INTERNAL_ERROR,
                                     "nothing is selected");
                goto next_branch;
            }

            /* get the number of Attributes for this selected object */
            if (eobj != NULL) {
                status = EG_attributeNum(eobj, &nattr);
                CATCH_STATUS(EG_attributeNum);
            } else {
                nattr = MODL->nattr;
            }

            /* if attrID is an integer, return the name of the attrID'th attribute */
            if (STRLEN(str) == 0) {
                iattr = NINT(value);

                /* ensure that iattr is in range */
                if (iattr < 1 || iattr > nattr) {
                    status = signalError(MODL, OCSM_ILLEGAL_ATTRIBUTE,
                                         "iattr is out of range");
                    goto next_branch;
                }

                /* if jpmtr has rows and columns, convert to string */
                if (MODL->pmtr[jpmtr].nrow > 0 ||
                    MODL->pmtr[jpmtr].ncol > 0   ) {
                    FREE(MODL->pmtr[jpmtr].value);
                    FREE(MODL->pmtr[jpmtr].dot  );
                    FREE(MODL->pmtr[jpmtr].lbnd );
                    FREE(MODL->pmtr[jpmtr].ubnd );

                    MALLOC(MODL->pmtr[jpmtr].str, char, MAX_STRVAL_LEN);

                    MODL->pmtr[jpmtr].nrow = 0;
                    MODL->pmtr[jpmtr].ncol = 0;
                }

                /* store the name in jpmtr */
                if (eobj != NULL) {
                    status = EG_attributeGet(eobj, iattr, &aname, &itype, &nlist,
                                             &tempIlist, &tempRlist, &tempClist);
                    CATCH_STATUS(EG_attributeGet);

                    STRNCPY(MODL->pmtr[jpmtr].str, aname, MAX_STRVAL_LEN);
                } else {
                    STRNCPY(MODL->pmtr[jpmtr].str, MODL->attr[iattr-1].name, MAX_STRVAL_LEN);
                }

                SPRINT2(1, "                          %s = $%s",
                        pname, MODL->pmtr[jpmtr].str);

            /* if attrID is $_nattr_, return the number of Attributes */
            } else if (strcmp(str, "_nattr_") == 0) {

                /* if jpmtr is a string, convert to a number */
                if (MODL->pmtr[jpmtr].nrow == 0 &&
                    MODL->pmtr[jpmtr].ncol == 0   ) {
                    FREE(MODL->pmtr[jpmtr].str);

                    MALLOC(MODL->pmtr[jpmtr].value, double, 1);
                    MALLOC(MODL->pmtr[jpmtr].dot,   double, 1);

                    MODL->pmtr[jpmtr].value[0] = -HUGEQ;
                    MODL->pmtr[jpmtr].dot[  0] = 0;

                    MODL->pmtr[jpmtr].nrow = 1;
                    MODL->pmtr[jpmtr].ncol = 1;
                }

                /* store the result */
                if (MODL->pmtr[jpmtr].type == OCSM_UNKNOWN) {
                    MODL->pmtr[jpmtr].type =  OCSM_LOCALVAR;
                }

                status = ocsmSetValuD(MODL, jpmtr, 1, 1, (double)nattr);
                CATCH_STATUS(ocsmSetValuD);

                SPRINT2(1, "                          %s = %11.5f",
                        pname, (double)nattr);

            /* otherwise, look up the value of the given Attribute */
            } else {
                if (eobj != NULL) {
                    status = EG_attributeRet(eobj, str, &itype, &nlist,
                                             &tempIlist, &tempRlist, &tempClist);
                    if (status != SUCCESS) {
                        status = signalError(MODL, OCSM_ILLEGAL_ATTRIBUTE,
                                             "Attribute \"%s\" not found", str);
                        goto next_branch;
                    }
                } else {
                    itype = -999;
                    for (iattr = 0; iattr < MODL->nattr; iattr++) {
                        if (strcmp(str, MODL->attr[iattr].name) == 0) {
                            itype     = ATTRSTRING;
                            nlist     = STRLEN(MODL->attr[iattr].defn);
                            tempClist = MODL->attr[iattr].defn;
                            break;
                        }
                    }
                    if (itype == -999) {
                        status = signalError(MODL, OCSM_ILLEGAL_ATTRIBUTE,
                                             "Attribute \"%s\" not found", str);
                        goto next_branch;
                    }

                    if (MODL->attr[iattr].defn[0] == '$') {
                        itype     = ATTRSTRING;
                        nlist     = STRLEN(MODL->attr[iattr].defn) - 1;
                        tempClist = &(MODL->attr[iattr].defn[1]);
                    } else {
                        status = str2vals(MODL->attr[iattr].defn, MODL, &nrow, &ncol, &values, &dots, str);
                        CATCH_STATUS(str2val);

                        itype     = ATTRREAL;
                        nlist     = nrow * ncol;
                        tempRlist = values;
                    }
                }

                if (itype != ATTRINT  && itype != ATTRREAL  &&
                    itype != ATTRCSYS && itype != ATTRSTRING  ) {
                    status = signalError(MODL, OCSM_ILLEGAL_ATTRIBUTE,
                                         "Attribute \"%s\" is wrong type (%d)", str, itype);
                    goto next_branch;
                }

                /* adjust type/size of jpmtr */
                if (itype == ATTRINT || itype == ATTRREAL || itype == ATTRCSYS) {
                    FREE(MODL->pmtr[jpmtr].value);
                    FREE(MODL->pmtr[jpmtr].dot  );
                    FREE(MODL->pmtr[jpmtr].lbnd );
                    FREE(MODL->pmtr[jpmtr].ubnd );
                    FREE(MODL->pmtr[jpmtr].str  );

                    MALLOC(MODL->pmtr[jpmtr].value, double, nlist);
                    MALLOC(MODL->pmtr[jpmtr].dot,   double, nlist);

                    for (i = 0; i < nlist; i++) {
                        MODL->pmtr[jpmtr].value[i] = -HUGEQ;
                        MODL->pmtr[jpmtr].dot[  i] = 0;
                    }

                    MODL->pmtr[jpmtr].nrow = 1;
                    MODL->pmtr[jpmtr].ncol = nlist;
                } else if (itype == ATTRSTRING) {
                    FREE(MODL->pmtr[jpmtr].value);
                    FREE(MODL->pmtr[jpmtr].dot  );
                    FREE(MODL->pmtr[jpmtr].lbnd );
                    FREE(MODL->pmtr[jpmtr].ubnd );
                    FREE(MODL->pmtr[jpmtr].str  );

                    MALLOC(MODL->pmtr[jpmtr].str, char, MAX_STRVAL_LEN);

                    MODL->pmtr[jpmtr].nrow = 0;
                    MODL->pmtr[jpmtr].ncol = 0;
                }

                /* store the values in jpmtr */
                if (MODL->pmtr[jpmtr].type == OCSM_UNKNOWN) {
                    MODL->pmtr[jpmtr].type =  OCSM_LOCALVAR;
                }

                if (itype == ATTRINT) {
                    SPLINT_CHECK_FOR_NULL(tempIlist);

                    for (ilist = 0; ilist < nlist; ilist++) {
                        status = ocsmSetValuD(MODL, jpmtr, 1, ilist+1, (double)tempIlist[ilist]);
                        CATCH_STATUS(ocsmSetValuD);

                        SPRINT3(1, "                          %s[1,%2d] = %11.5f",
                                pname, ilist+1, (double)tempIlist[ilist]);
                    }
                } else if (itype == ATTRREAL || itype == ATTRCSYS) {
                    SPLINT_CHECK_FOR_NULL(tempRlist);
                    for (ilist = 0; ilist < nlist; ilist++) {
                        status = ocsmSetValuD(MODL, jpmtr, 1, ilist+1, tempRlist[ilist]);
                        CATCH_STATUS(ocsmSetValuD);

                        SPRINT3(1, "                          %s[1,%2d] = %11.5f",
                                pname, ilist+1, tempRlist[ilist]);
                    }
                } else {
                    SPLINT_CHECK_FOR_NULL(tempClist);

                    STRNCPY(MODL->pmtr[jpmtr].str, tempClist, MAX_STRVAL_LEN);

                    SPRINT2(1, "                          %s = $%s",
                            pname, tempClist);

                }
            }

            FREE(values);
            FREE(dots  );

        /* execute: "udparg $primtype $argName1 argValue1 $argName2 argValue2 $argName3 argValue3 $argName4 argValue4" */
        } else if (type == OCSM_UDPARG) {
            SPRINT2x(1, "    executing [%4d] udparg:     %s",
                    ibrch, args[1].str);

            hasdots = 0;

            for (iarg = 3; iarg <= MODL->brch[ibrch].narg; iarg+=2) {
                if (args[iarg].nval == 0) {
                    SPRINT2x(1, " %s %s",
                             args[iarg-1].str, args[iarg].str);
                } else {
                    SPRINT2x(1, " %s %11.5f",
                             args[iarg-1].str, args[iarg].val[0]);

                    for (ival = 0; ival < args[iarg].nval; ival++) {
                        if (args[iarg].dot[ival] != 0) hasdots = 1;
                    }

                    /* set hasdots to 2 if any arguments changed (so that we know to
                       rebuild the UDPRIM) */
                    ibody = MODL->nbody + 1;
                    if ((ibody <  MODL->recycle)                                           ||
                        (ibody == MODL->recycle && MODL->body[ibody].brtype != OCSM_UDPARG)  ) {
                        if (MODL->body[ibody].arg[iarg].nval == args[iarg].nval) {
                            for (ival = 0; ival < args[iarg].nval; ival++) {
                                if (MODL->body[ibody].arg[iarg].val[ival] != args[iarg].val[ival]) {
                                    hasdots = 2;
                                }
                            }
                        } else {
                            hasdots = 2;
                        }
                    }
                }
            }
            SPRINT0(1, " ");

            /* free a previous Body if it exists */
            if (MODL->nbody+1 <= MODL->recycle) {
                if (MODL->body[MODL->nbody+1].arg[1].val != NULL) {
                    status = freeBody(MODL, MODL->nbody+1);
                    CATCH_STATUS(freeBody);
                }
            }

            /* create a Body */
            status = newBody(MODL, ibrch, OCSM_UDPARG, -1, -1,
                             args, hasdots, OCSM_NULL_BODY, &ibody);
            CATCH_STATUS(newBody);

            /* finish the Body (UDPARG) */
            status = finishBody(MODL, ibody);
            if (MODL->sigCode != SUCCESS) goto next_branch;
            CATCH_STATUS(finishBody);

            /* do not push the Body onto the stack */
            SPRINT1(1, "                          Body   %4d created  (NULL Body)",
                    ibody);

        /* execute: "select $type arg1..." */
        } else if (type == OCSM_SELECT) {
            SPRINT1x(1, "    executing [%4d] select: ", ibrch);
            for (iarg = 1; iarg <= MODL->brch[ibrch].narg; iarg++) {
                if (args[iarg].nval == 0) {
                    SPRINT1x(1, "  %s",       args[iarg].str);
                } else if (fabs(NINT(args[iarg].val[0])-args[iarg].val[0]) > EPS12) {
                    SPRINT1x(1, "  %11.5f",   args[iarg].val[0]);
                } else {
                    SPRINT1x(1, "  %5d", NINT(args[iarg].val[0]));
                }
            }
            SPRINT0(1, " ");

            /* make sure we get new mass properties */
            MODL->hasMPs = 0;

            /* "select body ..." */
            if        (strcmp(args[1].str, "body") == 0 || strcmp(args[1].str, "BODY") == 0) {

                /* "body" */
                if        (MODL->brch[ibrch].narg == 1) {
                    MODL->seltype = -1;
                    for (MODL->selbody = MODL->nbody; MODL->selbody > 0; (MODL->selbody)--) {
                        if (MODL->body[MODL->selbody].botype != OCSM_NULL_BODY) break;
                    }

                /* "body attrName1 attrValue1 attrName2=$* attrValue2=$* attrName3=$* attrValue3=$*" */
                } else if (MODL->brch[ibrch].narg == 7 && args[2].nval == 0 &&
                                                          args[4].nval == 0 &&
                                                          args[6].nval == 0   ) {
                    ibody = -1;
                    for (jbody = 1; jbody <= MODL->nbody; jbody++) {
                        if (MODL->body[jbody].botype == OCSM_NULL_BODY) continue;
                        if (MODL->body[jbody].ebody == NULL) continue;

                        status = EG_attributeNum(MODL->body[jbody].ebody, &nattr);
                        CHECK_STATUS(EG_attributeNum);

                        match1 = 0;
                        match2 = 0;
                        match3 = 0;

                        for (iattr = 1; iattr <= nattr; iattr++) {
                            status = EG_attributeGet(MODL->body[jbody].ebody,
                                                     iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                            CHECK_STATUS(EG_attributeGet);

                            if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                            }

                            if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                            }

                            if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                            }
                        }

                        if (match1 == 1 && match2 == 1 && match3 == 1) {
                            if (ibody == -1) {
                                ibody = jbody;
                            } else {
                                status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                                     "more than one Body matches SELECT");
                                goto next_branch;
                            }
                        }
                    }

                    if (ibody > 0) {
                        MODL->seltype = -1;
                        MODL->selbody = ibody;
                        MODL->selsize = 0;

                        RALLOC(MODL->sellist, int, 1);
                    } else {
                        status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                             "SELECT specifies nonexistant Body");
                        goto next_branch;
                    }

                /* "body ibody" */
                } else if (MODL->brch[ibrch].narg == 2) {
                    if (NINT(args[2].val[0]) >= 1 && NINT(args[2].val[0]) <= MODL->nbody) {
                        MODL->seltype = -1;
                        MODL->selbody = NINT(args[2].val[0]);
                        MODL->selsize = 0;

                        RALLOC(MODL->sellist, int, 1);
                    } else if (NINT(args[2].val[0]) < 0 && nstack >= -NINT(args[2].val[0])) {
                        MODL->seltype = -1;
                        MODL->selbody = stack[nstack+NINT(args[2].val[0])];
                        MODL->selsize = 0;

                        RALLOC(MODL->sellist, int, 1);

                        if (MODL->selbody == 0) {
                            status = setupAtPmtrs(MODL, -1);
                            goto next_branch;
                        }
                    } else {
                        status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                             "SELECT specifies nonexistant Body");
                        goto next_branch;
                    }
                }

            /* "select face ..." */
            } else if (strcmp(args[1].str, "face") == 0 || strcmp(args[1].str, "FACE") == 0) {
                if (MODL->selbody <= 0) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "SELECT FACE specified nonexistant Body");
                    goto next_branch;

                } else if (MODL->body[MODL->selbody].nface < 1) {
                    SPRINT1(0, "WARNING:: no Faces associated with Body %d", MODL->selbody);
                    (MODL->nwarn)++;

                    MODL->seltype = -1;
                    MODL->selsize =  0;
                    RALLOC(MODL->sellist, int, 1);

                    goto next_branch;

                /* "face" */
                } else if (MODL->brch[ibrch].narg == 1) {
                    MODL->seltype = 2;
                    MODL->selsize = MODL->body[MODL->selbody].nface;

                    RALLOC(MODL->sellist, int, MODL->selsize);
                    for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                        MODL->sellist[iface-1] = iface;
                    }

                /* "face iface" */
                } else if (MODL->brch[ibrch].narg == 2) {
                    if (NINT(args[2].val[0]) >= 1 && NINT(args[2].val[0]) <= MODL->body[MODL->selbody].nface) {
                        MODL->seltype = 2;
                        MODL->selsize = 1;

                        RALLOC(MODL->sellist, int, MODL->selsize);
                        MODL->sellist[0] = NINT(args[2].val[0]);
                    } else {
                        status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                             "SELECT specified nonexistant Face");
                        goto next_branch;
                    }

                /* "face -1 ibody1" */
                } else if (MODL->brch[ibrch].narg == 3 && NINT(args[2].val[0]) == -1) {
                    ibodyl = NINT(args[3].val[0]);

                    if (ibodyl < 1 || ibodyl > MODL->nbody) {
                        status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                             "SELECT specified nonexistant Body");
                        goto next_branch;
                    }

                    MODL->seltype = 2;
                    MODL->selsize = 0;

                    /* find number of Faces that matches */
                    for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                        for (jface = 1; jface <= MODL->body[ibodyl].nface; jface++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].face[iface].eface,
                                                     MODL->body[ibodyl       ].face[jface].eface);
                            if (status == EGADS_SUCCESS) {
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                    if (MODL->selsize <= 0) {
                        status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                             "SELECT did not find any matching Faces");
                        goto next_branch;
                    }

                    RALLOC(MODL->sellist, int, MODL->selsize);

                    /* add Faces that match to the sellist */
                    MODL->selsize = 0;
                    for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                        for (jface = 1; jface <= MODL->body[ibodyl].nface; jface++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].face[iface].eface,
                                                     MODL->body[ibodyl       ].face[jface].eface);
                            if (status == EGADS_SUCCESS) {
                                MODL->sellist[MODL->selsize] = iface;
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                /* "face -2 ibody1" */
                } else if (MODL->brch[ibrch].narg == 3 && NINT(args[2].val[0]) == -2) {
                    ibodyl = NINT(args[3].val[0]);

                    if (ibodyl < 1 || ibodyl > MODL->nbody) {
                        status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                             "SELECT specified nonexistant Body");
                        goto next_branch;
                    } else if (MODL->body[ibodyl].botype != OCSM_SOLID_BODY) {
                        status = signalError(MODL, OCSM_WRONG_TYPES_ON_STACK,
                                             "SELECT FACE -2 requires that ibody1 be a SolidBody");
                        goto next_branch;
                    }

                    /* if selbody is a SolidBody, convert it to a SheetBody */
                    if (MODL->body[MODL->selbody].botype == OCSM_SOLID_BODY) {
                        status = EG_getTopology(MODL->body[MODL->selbody].ebody, &eref, &oclass, &mtype,
                                                data, &nshell, &eshells, &senses);
                        CHECK_STATUS(EG_getTopology);

                        status = EG_makeTopology(MODL->context, NULL, BODY, SHEETBODY,
                                                 NULL, nshell, eshells, NULL, &etemp2);
                        CHECK_STATUS(EG_topologyCreate);

                    } else {
                        etemp2 = MODL->body[MODL->selbody].ebody;
                    }

                    /* intersect selbody with ibody1 */
                    status = EG_generalBoolean(etemp2, MODL->body[ibodyl].ebody, INTERSECTION, 0.0, &emodel);

                    /* extract the Faces in the Body from the Model returned from EG_generalBoolean */
                    if (status == EGADS_SUCCESS) {
                        SPLINT_CHECK_FOR_NULL(emodel);

                        status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                                                data, &nbodys, &ebodys, &senses);
                        CHECK_STATUS(EG_getTopology);

                        status = EG_getBodyTopos(ebodys[0], NULL, FACE, &nfaces, &efaces);
                        CHECK_STATUS(EG_getBodyTopos);
                    } else {
                        EPRINT("WE SHOULD NOT GET HERE %d: status=%d\n", __LINE__, status);
                        exit(0);
                    }

                    MODL->seltype = 2;
                    MODL->selsize = 0;

                    /* find number of Faces that matches */
                    for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                        for (jface = 0; jface < nfaces; jface++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].face[iface].eface, efaces[jface]);
                            if (status == EGADS_SUCCESS) {
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                    if (MODL->selsize <= 0) {
                        EG_free(efaces);
                        status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                             "SELECT did not find any matching Faces");
                        goto next_branch;
                    }

                    RALLOC(MODL->sellist, int, MODL->selsize);

                    /* add Faces that match to the sellist */
                    MODL->selsize = 0;
                    for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                        for (jface = 0; jface < nfaces; jface++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].face[iface].eface, efaces[jface]);
                            if (status == EGADS_SUCCESS) {
                                MODL->sellist[MODL->selsize] = iface;
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                    EG_free(efaces);

                    if (emodel != NULL) {
                        status = EG_deleteObject(emodel);
                        CHECK_STATUS(EG_deleteObject);
                    }

                /* "face ibody1 iford1 iseq=1" (0 for any) */
                } else if (MODL->brch[ibrch].narg == 3 || MODL->brch[ibrch].narg == 4) {
                    MODL->seltype = 2;
                    MODL->selsize = 0;

                    if (MODL->brch[ibrch].narg == 4) {
                        iseq = NINT(args[4].val[0]);
                    } else if (NINT(args[2].val[0]) == 0 || NINT(args[3].val[0]) == 0) {
                        iseq = 0;
                    } else {
                        iseq = 1;
                    }

                    for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                        status = EG_attributeRet(MODL->body[MODL->selbody].face[iface].eface,
                                                 "_faceID", &itype, &nlist,
                                                 &tempIlist, &tempRlist, &tempClist);
                        CATCH_STATUS(EG_attributeRet);

                        if ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[0]) &&
                            (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[1]) &&
                            (                iseq == 0 ||                 iseq == tempIlist[2])   ) {
                            MODL->selsize++;
                        }
                    }

                    if (MODL->selsize > 0) {
                        RALLOC(MODL->sellist, int, MODL->selsize);

                        i = 0;
                        for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                            status = EG_attributeRet(MODL->body[MODL->selbody].face[iface].eface,
                                                     "_faceID", &itype, &nlist,
                                                     &tempIlist, &tempRlist, &tempClist);
                            CATCH_STATUS(EG_attributeRet);

                            if ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[0]) &&
                                (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[1]) &&
                                (                iseq == 0 ||                 iseq == tempIlist[2])    ) {
                                MODL->sellist[i] = iface;

                                i++;
                            }
                        }
                    } else {
                        status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                             "SELECT specified nonexistant Face");
                        goto next_branch;
                    }

                /* "face xmin xmax ymin ymax zmin zmax" */
                } else if (MODL->brch[ibrch].narg == 7 && args[2].nval == 1 && args[3].nval == 1 &&
                                                          args[4].nval == 1 && args[5].nval == 1 &&
                                                          args[6].nval == 1 && args[7].nval == 1   ) {
                    MODL->seltype = 2;
                    MODL->selsize = 0;

                    if (args[2].val[0] == args[3].val[0] &&
                        args[4].val[0] == args[5].val[0] &&
                        args[6].val[0] == args[7].val[0]   ) {

                        MODL->sellist[0] = -1;
                        MODL->selsize    = 1;

                        RALLOC(MODL->sellist, int, MAX(MODL->selsize, 1));

                        dbest = HUGEQ;
                        for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                            status = EG_getBoundingBox(MODL->body[MODL->selbody].face[iface].eface, bbox);
                            CATCH_STATUS(EG_getBoundingBox);

                            dtest = (2 * args[2].val[0] - bbox[0] - bbox[3]) * (2 * args[2].val[0] - bbox[0] - bbox[3])
                                  + (2 * args[4].val[0] - bbox[1] - bbox[4]) * (2 * args[4].val[0] - bbox[1] - bbox[4])
                                  + (2 * args[6].val[0] - bbox[2] - bbox[5]) * (2 * args[6].val[0] - bbox[2] - bbox[5]);
                            if (dtest < dbest) {
                                dbest            = dtest;
                                MODL->sellist[0] = iface;
                            }
                        }
                    } else {
                        for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                            status = EG_getBoundingBox(MODL->body[MODL->selbody].face[iface].eface, bbox);
                            CATCH_STATUS(EG_getBoundingBox);

                            icount = 0;
                            if (args[3].val[0] >= args[2].val[0]) {
                                if (bbox[0] >= args[2].val[0]-EPS12 && bbox[3] <= args[3].val[0]+EPS12) {
                                    icount++;
                                }
                            } else {
                                if (bbox[0] <= args[2].val[0]+EPS12 && bbox[3] >= args[3].val[0]-EPS12) {
                                    icount++;
                                }
                            }
                            if (args[5].val[0] >= args[4].val[0]) {
                                if (bbox[1] >= args[4].val[0]-EPS12 && bbox[4] <= args[5].val[0]+EPS12) {
                                    icount++;
                                }
                            } else {
                                if (bbox[1] <= args[4].val[0]+EPS12 && bbox[4] >= args[5].val[0]-EPS12) {
                                    icount++;
                                }
                            }
                            if (args[7].val[0] >= args[6].val[0]) {
                                if (bbox[2] >= args[6].val[0]-EPS12 && bbox[5] <= args[7].val[0]+EPS12) {
                                    icount++;
                                }
                            } else {
                                if (bbox[2] <= args[6].val[0]+EPS12 && bbox[5] >= args[7].val[0]-EPS12) {
                                    icount++;
                                }
                            }
                            if (icount == 3) {
                                MODL->selsize++;
                            }
                        }

                        RALLOC(MODL->sellist, int, MAX(MODL->selsize, 1));
                        if (MODL->selsize == 0) {
                            MODL->sellist[0] = -1;

                            status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                                 "SELECT specified nonexistant Face");
                            goto next_branch;
                        } else {
                            MODL->selsize = 0;

                            for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                                status = EG_getBoundingBox(MODL->body[MODL->selbody].face[iface].eface, bbox);
                                CATCH_STATUS(EG_getBoundingBox);

                                icount = 0;
                                if (args[3].val[0] >= args[2].val[0]) {
                                    if (bbox[0] >= args[2].val[0]-EPS12 && bbox[3] <= args[3].val[0]+EPS12) {
                                        icount++;
                                    }
                                } else {
                                    if (bbox[0] <= args[2].val[0]+EPS12 && bbox[3] >= args[3].val[0]-EPS12) {
                                        icount++;
                                    }
                                }
                                if (args[5].val[0] >= args[4].val[0]) {
                                    if (bbox[1] >= args[4].val[0]-EPS12 && bbox[4] <= args[5].val[0]+EPS12) {
                                        icount++;
                                    }
                                } else {
                                    if (bbox[1] <= args[4].val[0]+EPS12 && bbox[4] >= args[5].val[0]-EPS12) {
                                        icount++;
                                    }
                                }
                                if (args[7].val[0] >= args[6].val[0]) {
                                    if (bbox[2] >= args[6].val[0]-EPS12 && bbox[5] <= args[7].val[0]+EPS12) {
                                        icount++;
                                    }
                                } else {
                                    if (bbox[2] <= args[6].val[0]+EPS12 && bbox[5] >= args[7].val[0]-EPS12) {
                                        icount++;
                                    }
                                }
                                if (icount == 3) {
                                    MODL->sellist[MODL->selsize] = iface;
                                    MODL->selsize++;
                                }
                            }
                        }
                    }

                /* "face attrName1 attrValue1 attrName2=$* attrValue2=$* attrName3=$* attrValue3=$*" */
                } else if (MODL->brch[ibrch].narg == 7 && args[2].nval == 0 &&
                                                          args[4].nval == 0 &&
                                                          args[6].nval == 0   ) {
                    MODL->seltype = 2;
                    MODL->selsize = 0;

                    RALLOC(MODL->sellist, int, 1);

                    for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                        status = EG_attributeNum(MODL->body[MODL->selbody].face[iface].eface, &nattr);
                        CHECK_STATUS(EG_attributeNum);

                        match1 = 0;
                        match2 = 0;
                        match3 = 0;

                        for (iattr = 1; iattr <= nattr; iattr++) {
                            status = EG_attributeGet(MODL->body[MODL->selbody].face[iface].eface,
                                                     iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                            CHECK_STATUS(EG_attributeGet);

                            if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                            }

                            if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                            }

                            if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                            }
                        }

                        if (match1 == 1 && match2 == 1 && match3 == 1) {
                            (MODL->selsize)++;
                            RALLOC(MODL->sellist, int, MODL->selsize);

                            MODL->sellist[MODL->selsize-1] = iface;
                        }
                    }

                    if (MODL->selsize == 0) {
                        status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                             "SELECT specified nonexistant Face");
                        goto next_branch;
                    }
                } else {
                    status = OCSM_NOT_ENOUGH_ARGS;
                    CATCH_STATUS(select_face);
                }

            /* "select edge ..." */
            } else if (strcmp(args[1].str, "edge") == 0 || strcmp(args[1].str, "EDGE") == 0) {
                if (MODL->selbody <= 0) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "SELECT EDGE specified nonexistant Body");
                    goto next_branch;

                } else if (MODL->body[MODL->selbody].nedge < 1) {
                    SPRINT1(0, "WARNING:: no Edges associated with Body %d", MODL->selbody);
                    (MODL->nwarn)++;

                    MODL->seltype = -1;
                    MODL->selsize =  0;
                    RALLOC(MODL->sellist, int, 1);

                    goto next_branch;

                /* "edge" */
                } else if (MODL->brch[ibrch].narg == 1) {
                    MODL->seltype = 1;
                    MODL->selsize = MODL->body[MODL->selbody].nedge;

                    RALLOC(MODL->sellist, int, MODL->selsize);
                    MODL->selsize = 0;
                    for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                        if (MODL->body[MODL->selbody].edge[iedge].itype == DEGENERATE) continue;

                        MODL->sellist[MODL->selsize] = iedge;
                        MODL->selsize++;
                    }

                /* "edge iedge" */
                } else if (MODL->brch[ibrch].narg == 2) {
                    if (NINT(args[2].val[0]) >= 1 && NINT(args[2].val[0]) <= MODL->body[MODL->selbody].nedge) {
                        MODL->seltype = 1;
                        MODL->selsize = 1;

                        RALLOC(MODL->sellist, int, MODL->selsize);
                        MODL->sellist[0] = NINT(args[2].val[0]);
                    } else {
                        status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                             "SELECT specified nonexistant Edge");
                        goto next_branch;
                    }


                /* "edge -1 ibody1" */
                } else if (MODL->brch[ibrch].narg == 3 && NINT(args[2].val[0]) == -1) {
                    ibodyl = NINT(args[3].val[0]);

                    if (ibodyl < 1 || ibodyl > MODL->nbody) {
                        status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                             "SELECT specified nonexistant Body");
                        goto next_branch;
                    }

                    MODL->seltype = 1;
                    MODL->selsize = 0;

                    /* find number of Edges that matches */
                    for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                        for (jedge = 1; jedge <= MODL->body[ibodyl].nedge; jedge++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].edge[iedge].eedge,
                                                     MODL->body[ibodyl       ].edge[jedge].eedge);
                            if (status == EGADS_SUCCESS) {
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                    if (MODL->selsize <= 0) {
                        status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                             "SELECT did not find any matching Edges");
                        goto next_branch;
                    }

                    RALLOC(MODL->sellist, int, MODL->selsize);

                    /* add Edges that match to the sellist */
                    MODL->selsize = 0;
                    for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                        for (jedge = 1; jedge <= MODL->body[ibodyl].nedge; jedge++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].edge[iedge].eedge,
                                                     MODL->body[ibodyl       ].edge[jedge].eedge);
                            if (status == EGADS_SUCCESS) {
                                MODL->sellist[MODL->selsize] = iedge;
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                /* "edge -2 ibody1" */
                } else if (MODL->brch[ibrch].narg == 3 && NINT(args[2].val[0]) == -2) {
                    ibodyl = NINT(args[3].val[0]);

                    if (ibodyl < 1 || ibodyl > MODL->nbody) {
                        status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                             "SELECT specified nonexistant Body");
                        goto next_branch;
                    } else if (MODL->body[ibodyl].botype != OCSM_SOLID_BODY) {
                        status = signalError(MODL, OCSM_WRONG_TYPES_ON_STACK,
                                             "SELECT EDGE -2 requires that ibody1 be a SolidBody");
                        goto next_branch;
                    }

                    /* if selbody is a SolidBody, convert it to a SheetBody */
                    if (MODL->body[MODL->selbody].botype == OCSM_SOLID_BODY) {
                        status = EG_getTopology(MODL->body[MODL->selbody].ebody, &eref, &oclass, &mtype,
                                                data, &nshell, &eshells, &senses);
                        CHECK_STATUS(EG_getTopology);

                        status = EG_makeTopology(MODL->context, NULL, BODY, SHEETBODY,
                                                 NULL, nshell, eshells, NULL, &etemp2);
                        CHECK_STATUS(EG_topologyCreate);

                    } else {
                        etemp2 = MODL->body[MODL->selbody].ebody;
                    }

                    /* intersect selbody with ibody1 */
                    status = EG_generalBoolean(etemp2, MODL->body[ibodyl].ebody, INTERSECTION, 0.0, &emodel);

                    /* extract the Edges in the Body from the Model returned from EG_generalBoolean */
                    if (status == EGADS_SUCCESS) {
                        SPLINT_CHECK_FOR_NULL(emodel);

                        status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                                                data, &nbodys, &ebodys, &senses);
                        CHECK_STATUS(EG_getTopology);

                        status = EG_getBodyTopos(ebodys[0], NULL, EDGE, &nedges, &eedges);
                        CHECK_STATUS(EG_getBodyTopos);
                    } else {
                        EPRINT("WE SHOULD NOT GET HERE %d: status=%d\n", __LINE__, status);
                        exit(0);
                    }

                    MODL->seltype = 1;
                    MODL->selsize = 0;

                    /* find number of Edges that matches */
                    for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                        for (jedge = 0; jedge < nedges; jedge++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].edge[iedge].eedge, eedges[jedge]);
                            if (status == EGADS_SUCCESS) {
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                    if (MODL->selsize <= 0) {
                        EG_free(eedges);
                        status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                             "SELECT did not find any matching Edges");
                        goto next_branch;
                    }

                    RALLOC(MODL->sellist, int, MODL->selsize);

                    /* add Edges that match to the sellist */
                    MODL->selsize = 0;
                    for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                        for (jedge = 0; jedge < nedges; jedge++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].edge[iedge].eedge, eedges[jedge]);
                            if (status == EGADS_SUCCESS) {
                                MODL->sellist[MODL->selsize] = iedge;
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                    EG_free(eedges);

                    if (emodel != NULL) {
                        status = EG_deleteObject(emodel);
                        CHECK_STATUS(EG_deleteObject);
                    }

                /* "edge ibody1 iford1 ibody2 iford2 iseq=1" (0 for any) */
                } else if (MODL->brch[ibrch].narg == 5 || MODL->brch[ibrch].narg == 6) {
                    MODL->seltype = 1;
                    MODL->selsize = 0;

                    if (MODL->brch[ibrch].narg == 6) {
                        iseq = NINT(args[6].val[0]);
                    } else if (NINT(args[2].val[0]) == 0 || NINT(args[3].val[0]) == 0 ||
                               NINT(args[4].val[0]) == 0 || NINT(args[5].val[0]) == 0   ) {
                        iseq = 0;
                    } else {
                        iseq = 1;
                    }

                    if (NINT(args[2].val[0]) == 0 && NINT(args[3].val[0]) == 0 &&
                        NINT(args[4].val[0]) != 0 && NINT(args[5].val[0]) != 0   ) {
                        SPRINT0(1, "WARNING:: to SELECT non-manifold Edge, add \"SELECT SUB $_nface 2\" statement");
                        (MODL->nwarn)++;
                    }

                    for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                        if (MODL->body[MODL->selbody].edge[iedge].itype == DEGENERATE) continue;

                        status = EG_attributeRet(MODL->body[MODL->selbody].edge[iedge].eedge,
                                                 "_edgeID", &itype, &nlist,
                                                 &tempIlist, &tempRlist, &tempClist);
                        CATCH_STATUS(EG_attributeRet);

                        if ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[0]) &&
                            (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[1]) &&
                            (NINT(args[4].val[0]) == 0 || NINT(args[4].val[0]) == tempIlist[2]) &&
                            (NINT(args[5].val[0]) == 0 || NINT(args[5].val[0]) == tempIlist[3]) &&
                            (                iseq == 0 ||                 iseq == tempIlist[4])   ) {
                            MODL->selsize++;
                        }
                    }

                    if (MODL->selsize > 0) {
                        RALLOC(MODL->sellist, int, MODL->selsize);
                        MODL->selsize = 0;

                        for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                            if (MODL->body[MODL->selbody].edge[iedge].itype == DEGENERATE) continue;

                            status = EG_attributeRet(MODL->body[MODL->selbody].edge[iedge].eedge,
                                                     "_edgeID", &itype, &nlist,
                                                     &tempIlist, &tempRlist, &tempClist);
                            CATCH_STATUS(EG_attributeRet);

                            if ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[0]) &&
                                (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[1]) &&
                                (NINT(args[4].val[0]) == 0 || NINT(args[4].val[0]) == tempIlist[2]) &&
                                (NINT(args[5].val[0]) == 0 || NINT(args[5].val[0]) == tempIlist[3]) &&
                                (                iseq == 0 ||                 iseq == tempIlist[4])   ) {
                                MODL->sellist[MODL->selsize] = iedge;
                                MODL->selsize++;
                            }
                        }
                    } else {
                        status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                             "SELECT specified nonexistant Edge");
                        goto next_branch;
                    }

                /* "edge xmin xmax ymin ymax zmin zmax" */
                } else if (MODL->brch[ibrch].narg == 7 && args[2].nval == 1 && args[3].nval == 1 &&
                                                          args[4].nval == 1 && args[5].nval == 1 &&
                                                          args[6].nval == 1 && args[7].nval == 1   ) {
                    MODL->seltype = 1;
                    MODL->selsize = 0;

                    if (args[2].val[0] == args[3].val[0] &&
                        args[4].val[0] == args[5].val[0] &&
                        args[6].val[0] == args[7].val[0]   ) {

                        MODL->sellist[0] = -1;
                        MODL->selsize    = 1;

                        RALLOC(MODL->sellist, int, MAX(MODL->selsize, 1));

                        dbest = HUGEQ;
                        for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                            status = EG_getBoundingBox(MODL->body[MODL->selbody].edge[iedge].eedge, bbox);
                            CATCH_STATUS(EG_getBoundingBox);

                            dtest = (2 * args[2].val[0] - bbox[0] - bbox[3]) * (2 * args[2].val[0] - bbox[0] - bbox[3])
                                  + (2 * args[4].val[0] - bbox[1] - bbox[4]) * (2 * args[4].val[0] - bbox[1] - bbox[4])
                                  + (2 * args[6].val[0] - bbox[2] - bbox[5]) * (2 * args[6].val[0] - bbox[2] - bbox[5]);
                            if (dtest < dbest) {
                                dbest            = dtest;
                                MODL->sellist[0] = iedge;
                            }
                        }

                        if (MODL->sellist[0] == -1) {
                            EPRINT("WE SHOULD NOT GET HERE %d: status=%d\n", __LINE__, status);
                            exit(0);
                        }

                    } else {
                        for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                            if (MODL->body[MODL->selbody].edge[iedge].itype == DEGENERATE) continue;

                            status = EG_getBoundingBox(MODL->body[MODL->selbody].edge[iedge].eedge, bbox);
                            CATCH_STATUS(EG_getBoundingBox);

                            icount = 0;
                            if (args[3].val[0] >= args[2].val[0]) {
                                if (bbox[0] >= args[2].val[0]-EPS12 && bbox[3] <= args[3].val[0]+EPS12) {
                                    icount++;
                                }
                            } else {
                                if (bbox[0] <= args[2].val[0]+EPS12 && bbox[3] >= args[3].val[0]-EPS12) {
                                    icount++;
                                }
                            }
                            if (args[5].val[0] >= args[4].val[0]) {
                                if (bbox[1] >= args[4].val[0]-EPS12 && bbox[4] <= args[5].val[0]+EPS12) {
                                    icount++;
                                }
                            } else {
                                if (bbox[1] <= args[4].val[0]+EPS12 && bbox[4] >= args[5].val[0]-EPS12) {
                                    icount++;
                                }
                            }
                            if (args[7].val[0] >= args[6].val[0]) {
                                if (bbox[2] >= args[6].val[0]-EPS12 && bbox[5] <= args[7].val[0]+EPS12) {
                                    icount++;
                                }
                            } else {
                                if (bbox[2] <= args[6].val[0]+EPS12 && bbox[5] >= args[7].val[0]-EPS12) {
                                    icount++;
                                }
                            }
                            if (icount == 3) {
                                MODL->selsize++;
                            }
                        }

                        RALLOC(MODL->sellist, int, MAX(MODL->selsize, 1));
                        if (MODL->selsize == 0) {
                            MODL->sellist[0] = -1;

                            status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                                 "SELECT specified nonexistant Edge");
                            goto next_branch;
                        } else {
                            MODL->selsize = 0;

                            for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                                if (MODL->body[MODL->selbody].edge[iedge].itype == DEGENERATE) continue;

                                status = EG_getBoundingBox(MODL->body[MODL->selbody].edge[iedge].eedge, bbox);
                                CATCH_STATUS(EG_getBoundingBox);

                                icount = 0;
                                if (args[3].val[0] >= args[2].val[0]) {
                                    if (bbox[0] >= args[2].val[0]-EPS12 && bbox[3] <= args[3].val[0]+EPS12) {
                                        icount++;
                                    }
                                } else {
                                    if (bbox[0] <= args[2].val[0]+EPS12 && bbox[3] >= args[3].val[0]-EPS12) {
                                        icount++;
                                    }
                                }
                                if (args[5].val[0] >= args[4].val[0]) {
                                    if (bbox[1] >= args[4].val[0]-EPS12 && bbox[4] <= args[5].val[0]+EPS12) {
                                        icount++;
                                    }
                                } else {
                                    if (bbox[1] <= args[4].val[0]+EPS12 && bbox[4] >= args[5].val[0]-EPS12) {
                                        icount++;
                                    }
                                }
                                if (args[7].val[0] >= args[6].val[0]) {
                                    if (bbox[2] >= args[6].val[0]-EPS12 && bbox[5] <= args[7].val[0]+EPS12) {
                                        icount++;
                                    }
                                } else {
                                    if (bbox[2] <= args[6].val[0]+EPS12 && bbox[5] >= args[7].val[0]-EPS12) {
                                        icount++;
                                    }
                                }
                                if (icount == 3) {
                                    MODL->sellist[MODL->selsize] = iedge;
                                    MODL->selsize++;
                                }
                            }
                        }
                    }

                /* "edge attrName1 attrValue1 attrName2=$* attrValue2=$* attrName3=$* attrValue3=$*" */
                } else if (MODL->brch[ibrch].narg == 7 && args[2].nval == 0 &&
                                                          args[4].nval == 0 &&
                                                          args[6].nval == 0   ) {
                    MODL->seltype = 1;
                    MODL->selsize = 0;

                    RALLOC(MODL->sellist, int, 1);

                    for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                        if (MODL->body[MODL->selbody].edge[iedge].itype == DEGENERATE) continue;

                        status = EG_attributeNum(MODL->body[MODL->selbody].edge[iedge].eedge, &nattr);
                        CHECK_STATUS(EG_attributeNum);

                        match1 = 0;
                        match2 = 0;
                        match3 = 0;

                        for (iattr = 1; iattr <= nattr; iattr++) {
                            status = EG_attributeGet(MODL->body[MODL->selbody].edge[iedge].eedge,
                                                     iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                            CHECK_STATUS(EG_attributeGet);

                            if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                            }

                            if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                            }

                            if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                            }
                        }

                        if (match1 == 1 && match2 == 1 && match3 == 1) {
                            (MODL->selsize)++;
                            RALLOC(MODL->sellist, int, MODL->selsize);

                            MODL->sellist[MODL->selsize-1] = iedge;
                        }
                    }

                    if (MODL->selsize == 0) {
                        status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                             "SELECT specified nonexistant Edge");
                        goto next_branch;
                    }

                /* "edge x y z" */
                } else if (MODL->brch[ibrch].narg == 4 && args[2].nval == 1 &&
                                                          args[3].nval == 1 &&
                                                          args[4].nval == 1   ) {
                    MODL->seltype = 1;
                    MODL->selsize = 1;

                    RALLOC(MODL->sellist, int, MODL->selsize);
                    MODL->sellist[0] = -1;

                    dmin = HUGEQ;
                    for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                        if (MODL->body[MODL->selbody].edge[iedge].itype == DEGENERATE) continue;

                        status = EG_getRange(MODL->body[MODL->selbody].edge[iedge].eedge, uvrange, &periodic);
                        CATCH_STATUS(EG_getRange);

                        uvrange[0] = (uvrange[0] + uvrange[1]) / 2;
                        status = EG_evaluate(MODL->body[MODL->selbody].edge[iedge].eedge, uvrange, data);
                        CATCH_STATUS(EG_evaluate);

                        dtest = sqrt((data[0]-args[2].val[0]) * (data[0]-args[2].val[0])
                                    +(data[1]-args[3].val[0]) * (data[1]-args[3].val[0])
                                    +(data[2]-args[4].val[0]) * (data[2]-args[4].val[0]));
                        if (dtest < dmin) {
                            dmin = dtest;
                            MODL->sellist[0] = iedge;
                        }
                    }

                } else {
                    status = OCSM_NOT_ENOUGH_ARGS;
                    CATCH_STATUS(select_edge);
                }

            /* "select loop ..." */
            } else if (strcmp(args[1].str, "loop") == 0 || strcmp(args[1].str, "LOOP") == 0) {

                if (MODL->selbody <= 0) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "SELECT LOOP specified nonexistant Body");
                    goto next_branch;

                } else if (MODL->body[MODL->selbody].nface < 1) {
                    SPRINT1(0, "WARNING:: no Faces associated with Body %d", MODL->selbody);
                    (MODL->nwarn)++;

                    MODL->seltype = -1;
                    MODL->selsize =  0;
                    RALLOC(MODL->sellist, int, 1);

                    goto next_branch;

                /* "loop iface iloop" */
                } else if (MODL->brch[ibrch].narg == 3 && args[2].nval == 1 &&
                                                          args[3].nval == 1   ) {
                    iface = NINT(args[2].val[0]);
                    iloop = NINT(args[3].val[0]);

                    if (iface >= 1 && iface <= MODL->body[MODL->selbody].nface) {
                        status = EG_getTopology(MODL->body[MODL->selbody].face[iface].eface, &eref,
                                                &oclass, &mtype, data, &nloops, &eloops, &senses);
                        CHECK_STATUS(EG_getTopology);

                        if (iloop >= 1 && iloop <= nloops) {
                            status = EG_getTopology(eloops[iloop-1], &eref,
                                                    &oclass, &mtype, data, &nedges, &eedges, &senses);
                            CHECK_STATUS(EG_getTopology);

                            MODL->seltype = 1;
                            MODL->selsize = nedges;

                            RALLOC(MODL->sellist, int, MODL->selsize);

                            for (i = 0; i < nedges; i++) {
                                iedge = status = EG_indexBodyTopo(MODL->body[MODL->selbody].ebody, eedges[i]);
                                CHECK_STATUS(EG_indexBodyTopo);

                                MODL->sellist[i] = iedge;
                            }
                        } else {
                            status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                                 "SELECT specified nonexistant Loop for Face %d", iface);
                            goto next_branch;
                        }
                    } else {
                        status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                             "SELECT specified nonexistant Face");
                        goto next_branch;
                    }
                } else {
                    status = OCSM_NOT_ENOUGH_ARGS;
                    CATCH_STATUS(select_edge);
                }

            /* "select node ..." */
            } else if (strcmp(args[1].str, "node") == 0 || strcmp(args[1].str, "NODE") == 0) {
                if (MODL->selbody <= 0) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "SELECT NODE specified nonexistant Body");
                    goto next_branch;

                } else if (MODL->body[MODL->selbody].nnode < 1) {
                    SPRINT1(0, "WARNING:: no Nodes associated with Body %d", MODL->selbody);
                    (MODL->nwarn)++;

                    MODL->seltype = -1;
                    MODL->selsize =  0;
                    RALLOC(MODL->sellist, int, 1);

                    goto next_branch;

                /* "node" */
                } else if (MODL->brch[ibrch].narg == 1) {
                    MODL->seltype = 0;
                    MODL->selsize = MODL->body[MODL->selbody].nnode;

                    RALLOC(MODL->sellist, int, MODL->selsize);
                    for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                        MODL->sellist[inode-1] = inode;
                    }

                /* "node inode" */
                } else if (MODL->brch[ibrch].narg == 2) {
                    if (NINT(args[2].val[0]) >= 1 && NINT(args[2].val[0]) <= MODL->body[MODL->selbody].nnode) {
                        MODL->seltype = 0;
                        MODL->selsize = 1;

                        RALLOC(MODL->sellist, int, MODL->selsize);
                        MODL->sellist[0] = NINT(args[2].val[0]);
                    } else {
                        status = signalError(MODL, OCSM_NODE_NOT_FOUND,
                                             "SELECT specified nonexistant Node");
                        goto next_branch;
                    }

                /* "node -1 ibody1" */
                } else if (MODL->brch[ibrch].narg == 3 && NINT(args[2].val[0]) == -1) {
                    ibodyl = NINT(args[3].val[0]);

                    if (ibodyl < 1 || ibodyl > MODL->nbody) {
                        status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                             "SELECT specified nonexistant Body");
                        goto next_branch;
                    }

                    MODL->seltype = 0;
                    MODL->selsize = 0;

                    /* find number of Nodes that matches */
                    for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                        for (jnode = 1; jnode <= MODL->body[ibodyl].nnode; jnode++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].node[inode].enode,
                                                     MODL->body[ibodyl       ].node[jnode].enode);
                            if (status == EGADS_SUCCESS) {
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                    if (MODL->selsize <= 0) {
                        status = signalError(MODL, OCSM_NODE_NOT_FOUND,
                                             "SELECT did not find any matching Nodes");
                        goto next_branch;
                    }

                    RALLOC(MODL->sellist, int, MODL->selsize);

                    /* add Nodes that match to the sellist */
                    MODL->selsize = 0;
                    for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                        for (jnode = 1; jnode <= MODL->body[ibodyl].nnode; jnode++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].node[inode].enode,
                                                     MODL->body[ibodyl       ].node[jnode].enode);
                            if (status == EGADS_SUCCESS) {
                                MODL->sellist[MODL->selsize] = inode;
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                /* "node -2 ibody1" */
                } else if (MODL->brch[ibrch].narg == 3 && NINT(args[2].val[0]) == -2) {
                    ibodyl = NINT(args[3].val[0]);

                    if (ibodyl < 1 || ibodyl > MODL->nbody) {
                        status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                             "SELECT specified nonexistant Body");
                        goto next_branch;
                    } else if (MODL->body[ibodyl].botype != OCSM_SOLID_BODY) {
                        status = signalError(MODL, OCSM_WRONG_TYPES_ON_STACK,
                                             "SELECT NODE -2 requires that ibody1 be a SolidBody");
                        goto next_branch;
                    }

                    /* if selbody is a SolidBody, convert it to a SheetBody */
                    if (MODL->body[MODL->selbody].botype == OCSM_SOLID_BODY) {
                        status = EG_getTopology(MODL->body[MODL->selbody].ebody, &eref, &oclass, &mtype,
                                                data, &nshell, &eshells, &senses);
                        CHECK_STATUS(EG_getTopology);

                        status = EG_makeTopology(MODL->context, NULL, BODY, SHEETBODY,
                                                 NULL, nshell, eshells, NULL, &etemp2);
                        CHECK_STATUS(EG_topologyCreate);

                    } else {
                        etemp2 = MODL->body[MODL->selbody].ebody;
                    }

                    /* intersect selbody with ibody1 */
                    status = EG_generalBoolean(etemp2, MODL->body[ibodyl].ebody, INTERSECTION, 0.0, &emodel);

                    /* extract the Nodes in the Body from the Model returned from EG_generalBoolean */
                    if (status == EGADS_SUCCESS) {
                        SPLINT_CHECK_FOR_NULL(emodel);

                        status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                                                data, &nbodys, &ebodys, &senses);
                        CHECK_STATUS(EG_getTopology);

                        status = EG_getBodyTopos(ebodys[0], NULL, NODE, &nnodes, &enodes);
                        CHECK_STATUS(EG_getBodyTopos);
                    } else {
                        EPRINT("WE SHOULD NOT GET HERE %d: status=%d\n", __LINE__, status);
                        exit(0);
                    }

                    MODL->seltype = 0;
                    MODL->selsize = 0;

                    /* find number of Nodes that matches */
                    for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                        for (jnode = 0; jnode < nnodes; jnode++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].node[inode].enode, enodes[jnode]);
                            if (status == EGADS_SUCCESS) {
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                    if (MODL->selsize <= 0) {
                        EG_free(enodes);
                        status = signalError(MODL, OCSM_NODE_NOT_FOUND,
                                             "SELECT did not find any matching Nodes");
                        goto next_branch;
                    }

                    RALLOC(MODL->sellist, int, MODL->selsize);

                    /* add Nodes that match to the sellist */
                    MODL->selsize = 0;
                    for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                        for (jnode = 0; jnode < nnodes; jnode++) {
                            status = EG_isEquivalent(MODL->body[MODL->selbody].node[inode].enode, enodes[jnode]);
                            if (status == EGADS_SUCCESS) {
                                MODL->sellist[MODL->selsize] = inode;
                                MODL->selsize++;
                                break;
                            }
                        }
                    }

                    EG_free(enodes);

                    if (emodel != NULL) {
                        status = EG_deleteObject(emodel);
                        CHECK_STATUS(EG_deleteObject);
                    }

                /* "node x y z" */
                } else if (MODL->brch[ibrch].narg == 4) {
                    MODL->seltype = 0;
                    MODL->selsize = 1;

                    RALLOC(MODL->sellist, int, MODL->selsize);
                    MODL->sellist[0] = 1;
                    dbest = SQR(MODL->body[MODL->selbody].node[1].x-args[2].val[0])
                          + SQR(MODL->body[MODL->selbody].node[1].y-args[3].val[0])
                          + SQR(MODL->body[MODL->selbody].node[1].z-args[4].val[0]);

                    for (inode = 2;  inode <= MODL->body[MODL->selbody].nnode; inode++) {
                         dtest = SQR(MODL->body[MODL->selbody].node[inode].x-args[2].val[0])
                               + SQR(MODL->body[MODL->selbody].node[inode].y-args[3].val[0])
                               + SQR(MODL->body[MODL->selbody].node[inode].z-args[4].val[0]);
                         if (dtest < dbest) {
                             MODL->sellist[0] = inode;
                             dbest = dtest;
                         }
                    }

                /* "node xmin xmax ymin ymax zmin zmax" */
                } else if (MODL->brch[ibrch].narg == 7 && args[2].nval == 1 && args[3].nval == 1 &&
                                                          args[4].nval == 1 && args[5].nval == 1 &&
                                                          args[6].nval == 1 && args[7].nval == 1   ) {
                    MODL->seltype = 0;
                    MODL->selsize = 0;

                    if (args[2].val[0] != args[3].val[0] ||
                        args[4].val[0] != args[5].val[0] ||
                        args[6].val[0] != args[7].val[0]   ) {

                        for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                            status = EG_getBoundingBox(MODL->body[MODL->selbody].node[inode].enode, bbox);
                            CATCH_STATUS(EG_getBoundingBox);

                            if (bbox[0] >= args[2].val[0] && bbox[3] <= args[3].val[0] &&
                                bbox[1] >= args[4].val[0] && bbox[4] <= args[5].val[0] &&
                                bbox[2] >= args[6].val[0] && bbox[5] <= args[7].val[0]   ) {
                                MODL->selsize++;
                            }
                        }

                        RALLOC(MODL->sellist, int, MAX(MODL->selsize, 1));
                        if (MODL->selsize == 0) {
                            MODL->sellist[0] = -1;

                            status = signalError(MODL, OCSM_NODE_NOT_FOUND,
                                                 "SELECT specified nonexistant Node");
                            goto next_branch;
                        } else {
                            MODL->selsize = 0;

                            for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                                status = EG_getBoundingBox(MODL->body[MODL->selbody].node[inode].enode, bbox);
                                CATCH_STATUS(EG_getBoundingBox);

                                if (bbox[0] >= args[2].val[0] && bbox[3] <= args[3].val[0] &&
                                    bbox[1] >= args[4].val[0] && bbox[4] <= args[5].val[0] &&
                                    bbox[2] >= args[6].val[0] && bbox[5] <= args[7].val[0]   ) {
                                    MODL->sellist[MODL->selsize] = inode;
                                    MODL->selsize++;
                                }
                            }
                        }
                    } else {
                        MODL->sellist[0] = -1;
                        MODL->selsize    = 1;

                        RALLOC(MODL->sellist, int, MAX(MODL->selsize, 1));

                        dbest = HUGEQ;
                        for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                            status = EG_getBoundingBox(MODL->body[MODL->selbody].node[inode].enode, bbox);
                            CATCH_STATUS(EG_getBoundingBox);

                            dtest = (2 * args[2].val[0] - bbox[0] - bbox[3]) * (2 * args[2].val[0] - bbox[0] - bbox[3])
                                  + (2 * args[4].val[0] - bbox[1] - bbox[4]) * (2 * args[4].val[0] - bbox[1] - bbox[4])
                                  + (2 * args[6].val[0] - bbox[2] - bbox[5]) * (2 * args[6].val[0] - bbox[2] - bbox[5]);
                            if (dtest < dbest) {
                                dbest            = dtest;
                                MODL->sellist[0] = inode;
                            }
                        }
                    }

                /* "node attrName1 attrValue1 attrName2=$* attrValue2=$* attrName3=$* attrValue3=$*" */
                } else if (MODL->brch[ibrch].narg == 7 && args[2].nval == 0 &&
                                                          args[4].nval == 0 &&
                                                          args[6].nval == 0   ) {
                    MODL->seltype = 0;
                    MODL->selsize = 0;

                    RALLOC(MODL->sellist, int, 1);

                    for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                        status = EG_attributeNum(MODL->body[MODL->selbody].node[inode].enode, &nattr);
                        CHECK_STATUS(EG_attributeNum);

                        match1 = 0;
                        match2 = 0;
                        match3 = 0;

                        for (iattr = 1; iattr <= nattr; iattr++) {
                            status = EG_attributeGet(MODL->body[MODL->selbody].node[inode].enode,
                                                     iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                            CHECK_STATUS(EG_attributeGet);

                            if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                            }

                            if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                            }

                            if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                            }
                        }

                        if (match1 == 1 && match2 == 1 && match3 == 1) {
                            (MODL->selsize)++;
                            RALLOC(MODL->sellist, int, MODL->selsize);

                            MODL->sellist[MODL->selsize-1] = inode;
                        }
                    }

                    if (MODL->selsize == 0) {
                        status = signalError(MODL, OCSM_NODE_NOT_FOUND,
                                             "SELECT specified nonexistant Node");
                        goto next_branch;
                    }
                } else {
                    status = OCSM_NOT_ENOUGH_ARGS;
                    CATCH_STATUS(select_node);
                }

            /* "select add ..." */
            } else if (strcmp(args[1].str, "add") == 0 || strcmp(args[1].str, "ADD") == 0) {
                if (MODL->selbody <= 0) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "SELECT ADD specified nonexistant Body");
                    goto next_branch;

                } else if (MODL->seltype != 0 && MODL->seltype != 1 && MODL->seltype != 2) {
                    status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                         "SELECT ADD must follow SELECT NODE, EDGE, or FACE");
                    goto next_branch;

                /* "add iface" and seltype==2 */
                } else if (MODL->brch[ibrch].narg == 2 && MODL->seltype == 2) {
                    iface = NINT(args[2].val[0]);

                    if (iface >= 1 && iface <= MODL->body[MODL->selbody].nface) {
                        (MODL->selsize)++;
                        RALLOC(MODL->sellist, int, MODL->selsize);

                        MODL->sellist[MODL->selsize-1] = iface;
                    } else {
                        status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                             "Face not found");
                        goto next_branch;
                    }

                /* "add iedge" and seltype==1 */
                } else if (MODL->brch[ibrch].narg == 2 && MODL->seltype == 1) {
                    iedge = NINT(args[2].val[0]);

                    if (iedge >= 1 && iedge <= MODL->body[MODL->selbody].nedge) {
                        (MODL->selsize)++;
                        RALLOC(MODL->sellist, int, MODL->selsize);

                        MODL->sellist[MODL->selsize-1] = iedge;
                    } else {
                        status = signalError(MODL, OCSM_EDGE_NOT_FOUND,
                                             "Edge not found");
                        goto next_branch;
                    }

                /* "add inode" and seltype==0 */
                } else if (MODL->brch[ibrch].narg == 2 && MODL->seltype == 0) {
                    inode = NINT(args[2].val[0]);

                    if (inode >= 1 && inode <= MODL->body[MODL->selbody].nnode) {
                        (MODL->selsize)++;
                        RALLOC(MODL->sellist, int, MODL->selsize);

                        MODL->sellist[MODL->selsize-1] = inode;
                    } else {
                        status = signalError(MODL, OCSM_NODE_NOT_FOUND,
                                             "Node not found");
                        goto next_branch;
                    }

                /* "add attrName1 attrValue1 attrName2=$* attrValue2=$* attrName3=$* attrValue3=$*" */
                } else if (MODL->brch[ibrch].narg == 7 && args[2].nval == 0 &&
                                                          args[4].nval == 0 &&
                                                          args[6].nval == 0   ) {
                    if (MODL->seltype == 2) {
                        for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                            match1 = 0;
                            for (ilist = 0; ilist < MODL->selsize; ilist++) {
                                if (MODL->sellist[ilist] == iface) {
                                    match1 = 1;
                                    break;
                                }
                            }
                            if (match1 == 1) continue;

                            status = EG_attributeNum(MODL->body[MODL->selbody].face[iface].eface, &nattr);
                            CHECK_STATUS(EG_attributeNum);

                            match1 = 0;
                            match2 = 0;
                            match3 = 0;

                            for (iattr = 1; iattr <= nattr; iattr++) {
                                status = EG_attributeGet(MODL->body[MODL->selbody].face[iface].eface,
                                                         iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                                CHECK_STATUS(EG_attributeGet);

                                if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                    match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                    match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                    match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                                }
                            }

                            if (match1 == 1 && match2 == 1 && match3 == 1) {
                                (MODL->selsize)++;
                                RALLOC(MODL->sellist, int, MODL->selsize);

                                MODL->sellist[MODL->selsize-1] = iface;
                            }
                        }
                    } else if (MODL->seltype == 1) {
                        for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                            if (MODL->body[MODL->selbody].edge[iedge].itype == DEGENERATE) continue;

                            match1 = 0;
                            for (ilist = 0; ilist < MODL->selsize; ilist++) {
                                if (MODL->sellist[ilist] == iedge) {
                                    match1 = 1;
                                    break;
                                }
                            }
                            if (match1 == 1) continue;

                            status = EG_attributeNum(MODL->body[MODL->selbody].edge[iedge].eedge, &nattr);
                            CHECK_STATUS(EG_attributeNum);

                            match1 = 0;
                            match2 = 0;
                            match3 = 0;

                            for (iattr = 1; iattr <= nattr; iattr++) {
                                status = EG_attributeGet(MODL->body[MODL->selbody].edge[iedge].eedge,
                                                         iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                                CHECK_STATUS(EG_attributeGet);

                                if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                    match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                    match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                    match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                                }
                            }

                            if (match1 == 1 && match2 == 1 && match3 == 1) {
                                (MODL->selsize)++;
                                RALLOC(MODL->sellist, int, MODL->selsize);

                                MODL->sellist[MODL->selsize-1] = iedge;
                            }
                        }
                    } else if (MODL->seltype == 0) {
                        for (inode = 1; inode <= MODL->body[MODL->selbody].nnode; inode++) {
                            match1 = 0;
                            for (ilist = 0; ilist < MODL->selsize; ilist++) {
                                if (MODL->sellist[ilist] == inode) {
                                    match1 = 1;
                                    break;
                                }
                            }
                            if (match1 == 1) continue;

                            status = EG_attributeNum(MODL->body[MODL->selbody].node[inode].enode, &nattr);
                            CHECK_STATUS(EG_attributeNum);

                            match1 = 0;
                            match2 = 0;
                            match3 = 0;

                            for (iattr = 1; iattr <= nattr; iattr++) {
                                status = EG_attributeGet(MODL->body[MODL->selbody].node[inode].enode,
                                                         iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                                CHECK_STATUS(EG_attributeGet);

                                if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                    match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                    match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                    match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                                }
                            }

                            if (match1 == 1 && match2 == 1 && match3 == 1) {
                                (MODL->selsize)++;
                                RALLOC(MODL->sellist, int, MODL->selsize);

                                MODL->sellist[MODL->selsize-1] = inode;
                            }
                        }
                    }

                /* "add ibody1 iford1 iseq=1" and seltype==2 */
                } else if ( MODL->seltype == 2                                         &&
                           (MODL->brch[ibrch].narg == 3 || MODL->brch[ibrch].narg == 4)  ) {
                    if (MODL->brch[ibrch].narg == 3) {
                        iseq = 1;
                    } else {
                        iseq = NINT(args[4].val[0]);
                    }

                    for (iface = 1; iface <= MODL->body[MODL->selbody].nface; iface++) {
                        match1 = 0;
                        for (ilist = 0; ilist < MODL->selsize; ilist++) {
                            if (MODL->sellist[ilist] == iface) {
                                match1 = 1;
                                break;
                            }
                        }
                        if (match1 == 1) continue;

                        status = EG_attributeRet(MODL->body[MODL->selbody].face[iface].eface,
                                                 "_faceID", &itype, &nlist,
                                                 &tempIlist, &tempRlist, &tempClist);
                        CATCH_STATUS(EG_attributeRet);

                        if ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[0]) &&
                            (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[1]) &&
                                                                          iseq == tempIlist[2]   ) {
                            (MODL->selsize)++;
                            RALLOC(MODL->sellist, int, MODL->selsize);

                            MODL->sellist[MODL->selsize-1] = iface;
                        }
                    }

                /* "add ibody1 iford1 ibody2 iford2 iseq=1" amd seltype==1 */
                } else if ( MODL->seltype == 1                                         &&
                           (MODL->brch[ibrch].narg == 5 || MODL->brch[ibrch].narg == 6)  ) {
                    if (MODL->brch[ibrch].narg == 5) {
                        iseq = 1;
                    } else {
                        iseq = NINT(args[6].val[0]);
                    }

                    for (iedge = 1; iedge <= MODL->body[MODL->selbody].nedge; iedge++) {
                        if (MODL->body[MODL->selbody].edge[iedge].itype == DEGENERATE) continue;

                        match1 = 0;
                        for (ilist = 0; ilist < MODL->selsize; ilist++) {
                            if (MODL->sellist[ilist] == iedge) {
                                match1 = 1;
                                break;
                            }
                        }
                        if (match1 == 1) continue;

                        status = EG_attributeRet(MODL->body[MODL->selbody].edge[iedge].eedge,
                                                 "_edgeID", &itype, &nlist,
                                                 &tempIlist, &tempRlist, &tempClist);
                        CATCH_STATUS(EG_attributeRet);

                        if        ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[0]) &&
                                   (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[1]) &&
                                   (NINT(args[4].val[0]) == 0 || NINT(args[4].val[0]) == tempIlist[2]) &&
                                   (NINT(args[5].val[0]) == 0 || NINT(args[5].val[0]) == tempIlist[3]) &&
                                                                                 iseq == tempIlist[4]   ) {
                            (MODL->selsize)++;
                            RALLOC(MODL->sellist, int, MODL->selsize);

                            MODL->sellist[MODL->selsize-1] = iedge;
                        } else if ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[2]) &&
                                   (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[3]) &&
                                   (NINT(args[4].val[0]) == 0 || NINT(args[4].val[0]) == tempIlist[0]) &&
                                   (NINT(args[5].val[0]) == 0 || NINT(args[5].val[0]) == tempIlist[1]) &&
                                                                                 iseq == tempIlist[4]   ) {
                            (MODL->selsize)++;
                            RALLOC(MODL->sellist, int, MODL->selsize);

                            MODL->sellist[MODL->selsize-1] = iedge;
                        }
                    }

                } else {
                    status = OCSM_NOT_ENOUGH_ARGS;
                    CATCH_STATUS(select_add);
                }

            /* "select sub ..." */
            } else if (strcmp(args[1].str, "sub") == 0 || strcmp(args[1].str, "SUB") == 0) {
                if (MODL->selbody <= 0) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "SELECT SUB specified nonexistant Body");
                    goto next_branch;

                } else if (MODL->seltype != 0 && MODL->seltype != 1 && MODL->seltype != 2) {
                    status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                         "SELECT SUB must follow SELECT node, edge, or face");
                    goto next_branch;

                /* "sub ient" */
                } else if (MODL->brch[ibrch].narg == 2) {
                    ient   = NINT(args[2].val[0]);
                    match1 = 0;
                    for (ilist = 0; ilist < MODL->selsize; ilist++) {
                        if (MODL->sellist[ilist] == ient) {
                            for (jlist = ilist; jlist < MODL->selsize-1; jlist++) {
                                MODL->sellist[jlist] = MODL->sellist[jlist+1];
                            }

                            (MODL->selsize)--;
                            RALLOC(MODL->sellist, int, MODL->selsize);

                            match1++;
                            break;
                        }
                    }

                    if (match1 == 0) {
                        status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                             "SELECT SUB must specify a value in @sellist");
                        goto next_branch;
                    }

                /* "sub attrName1 attrValue1 attrName2=$* attrValue2=$* attrName3=$* attrValue3=$*" */
                } else if (MODL->brch[ibrch].narg == 7 && args[2].nval == 0 &&
                                                          args[4].nval == 0 &&
                                                          args[6].nval == 0   ) {
                    if (MODL->seltype == 2) {
                        for (ilist = 0; ilist < MODL->selsize; ilist++) {
                            iface = MODL->sellist[ilist];

                            status = EG_attributeNum(MODL->body[MODL->selbody].face[iface].eface, &nattr);
                            CHECK_STATUS(EG_attributeNum);

                            match1 = 0;
                            match2 = 0;
                            match3 = 0;

                            for (iattr = 1; iattr <= nattr; iattr++) {
                                status = EG_attributeGet(MODL->body[MODL->selbody].face[iface].eface,
                                                         iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                                CHECK_STATUS(EG_attributeGet);

                                if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                    match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                    match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                    match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                                }
                            }

                            if (match1 == 1 && match2 == 1 && match3 == 1) {
                                for (jlist = ilist; jlist < MODL->selsize-1; jlist++) {
                                    MODL->sellist[jlist] = MODL->sellist[jlist+1];
                                }
                                ilist--;

                                (MODL->selsize)--;
                                RALLOC(MODL->sellist, int, MODL->selsize);
                            }
                        }
                    } else if (MODL->seltype == 1) {
                        for (ilist = 0; ilist < MODL->selsize; ilist++) {
                            iedge = MODL->sellist[ilist];

                            status = EG_attributeNum(MODL->body[MODL->selbody].edge[iedge].eedge, &nattr);
                            CHECK_STATUS(EG_attributeNum);

                            match1 = 0;
                            match2 = 0;
                            match3 = 0;

                            for (iattr = 1; iattr <= nattr; iattr++) {
                                status = EG_attributeGet(MODL->body[MODL->selbody].edge[iedge].eedge,
                                                         iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                                CHECK_STATUS(EG_attributeGet);

                                if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                    match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                    match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                    match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                                }
                            }

                            if (match1 == 1 && match2 == 1 && match3 == 1) {
                                for (jlist = ilist; jlist < MODL->selsize-1; jlist++) {
                                    MODL->sellist[jlist] = MODL->sellist[jlist+1];
                                }
                                ilist--;

                                (MODL->selsize)--;
                                RALLOC(MODL->sellist, int, MODL->selsize);
                            }
                        }
                    } else if (MODL->seltype == 0) {
                        for (ilist = 0; ilist < MODL->selsize; ilist++) {
                            inode = MODL->sellist[ilist];

                            status = EG_attributeNum(MODL->body[MODL->selbody].node[inode].enode, &nattr);
                            CHECK_STATUS(EG_attributeNum);

                            match1 = 0;
                            match2 = 0;
                            match3 = 0;

                            for (iattr = 1; iattr <= nattr; iattr++) {
                                status = EG_attributeGet(MODL->body[MODL->selbody].node[inode].enode,
                                                         iattr, &aname, &itype, &nlist, &tempIlist, &tempRlist, &tempClist);
                                CHECK_STATUS(EG_attributeGet);

                                if (match1 == 0 && matches((char*)args[2].str, aname) == 1) {
                                    match1 = matchValue(args[3], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match2 == 0 && matches((char*)args[4].str, aname) == 1) {
                                    match2 = matchValue(args[5], itype, nlist, tempIlist, tempRlist, tempClist);
                                }

                                if (match3 == 0 && matches((char*)args[6].str, aname) == 1) {
                                    match3 = matchValue(args[7], itype, nlist, tempIlist, tempRlist, tempClist);
                                }
                            }

                            if (match1 == 1 && match2 == 1 && match3 == 1) {
                                for (jlist = ilist; jlist < MODL->selsize-1; jlist++) {
                                    MODL->sellist[jlist] = MODL->sellist[jlist+1];
                                }
                                ilist--;

                                (MODL->selsize)--;
                                RALLOC(MODL->sellist, int, MODL->selsize);
                            }
                        }
                    }

                /* "sub ibody1 iford1 iseq=1" and seltype==2 */
                } else if ( MODL->seltype == 2                                         &&
                           (MODL->brch[ibrch].narg == 3 || MODL->brch[ibrch].narg == 4)  ) {
                    if (MODL->brch[ibrch].narg == 3) {
                        iseq = 1;
                    } else {
                        iseq = NINT(args[4].val[0]);
                    }

                    for (ilist = 0; ilist < MODL->selsize; ilist++) {
                        iface = MODL->sellist[ilist];

                        status = EG_attributeRet(MODL->body[MODL->selbody].face[iface].eface,
                                                 "_faceID", &itype, &nlist,
                                                 &tempIlist, &tempRlist, &tempClist);
                        CHECK_STATUS(EG_attributeRet);

                        if ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[0]) &&
                            (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[1]) &&
                                                                          iseq == tempIlist[2]   ) {
                            for (jlist = ilist; jlist < MODL->selsize-1; jlist++) {
                                MODL->sellist[jlist] = MODL->sellist[jlist+1];
                            }
                            ilist--;

                            (MODL->selsize)--;
                            RALLOC(MODL->sellist, int, MODL->selsize);
                        }
                    }

                /* "sub ibody1 iford1 ibody2 iford2 iseq=1" and seltype==1 */
                } else if (MODL->seltype == 1                                         &&
                          (MODL->brch[ibrch].narg == 5 || MODL->brch[ibrch].narg == 6)  ) {
                    if (MODL->brch[ibrch].narg == 5) {
                        iseq = 1;
                    } else {
                        iseq = NINT(args[6].val[0]);
                    }

                    for (ilist = 0; ilist < MODL->selsize; ilist++) {
                        iedge = MODL->sellist[ilist];

                        status = EG_attributeRet(MODL->body[MODL->selbody].edge[iedge].eedge,
                                                 "_edgeID", &itype, &nlist,
                                                 &tempIlist, &tempRlist, &tempClist);
                        CHECK_STATUS(EG_attributeRet);

                        if        ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[0]) &&
                                   (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[1]) &&
                                   (NINT(args[4].val[0]) == 0 || NINT(args[4].val[0]) == tempIlist[2]) &&
                                   (NINT(args[5].val[0]) == 0 || NINT(args[5].val[0]) == tempIlist[3]) &&
                                                   iseq == tempIlist[4]   ) {
                            for (jlist = ilist; jlist < MODL->selsize-1; jlist++) {
                                MODL->sellist[jlist] = MODL->sellist[jlist+1];
                            }

                            (MODL->selsize)--;
                            RALLOC(MODL->sellist, int, MODL->selsize);
                        } else if ((NINT(args[2].val[0]) == 0 || NINT(args[2].val[0]) == tempIlist[2]) &&
                                   (NINT(args[3].val[0]) == 0 || NINT(args[3].val[0]) == tempIlist[3]) &&
                                   (NINT(args[4].val[0]) == 0 || NINT(args[4].val[0]) == tempIlist[0]) &&
                                   (NINT(args[5].val[0]) == 0 || NINT(args[5].val[0]) == tempIlist[1]) &&
                                                   iseq == tempIlist[4]   ) {
                            for (jlist = ilist; jlist < MODL->selsize-1; jlist++) {
                                MODL->sellist[jlist] = MODL->sellist[jlist+1];
                            }
                            ilist--;

                            (MODL->selsize)--;
                            RALLOC(MODL->sellist, int, MODL->selsize);
                        }
                    }

                } else {
                    status = OCSM_NOT_ENOUGH_ARGS;
                    CATCH_STATUS(select_node);
                }

            /* "select flip" */
            } else if (strcmp(args[1].str, "flip") == 0 || strcmp(args[1].str, "FLIP") == 0) {
                jlist = MODL->selsize-1;
                for (ilist = 0; ilist < MODL->selsize; ilist++) {
                    if (ilist >= jlist) break;

                    iswap                = MODL->sellist[ilist];
                    MODL->sellist[ilist] = MODL->sellist[jlist];
                    MODL->sellist[jlist] = iswap;

                    jlist--;
                }

            /* "select not" */
            } else if (strcmp(args[1].str, "not") == 0 || strcmp(args[1].str, "NOT") == 0) {
                if        (MODL->seltype == 0) {
                    tempSize = MODL->selsize;
                    MALLOC(tempList, int, tempSize);
                    for (i = 0; i < tempSize; i++) {
                        tempList[i] = MODL->sellist[i];
                    }

                    MODL->selsize = MODL->body[MODL->selbody].nnode - tempSize;
                    RALLOC(MODL->sellist, int, MODL->selsize);

                    MODL->selsize = 0;
                    for (j = 1; j <= MODL->body[MODL->selbody].nnode; j++) {
                        inlist = 0;
                        for (i = 0; i < tempSize; i++) {
                            if (tempList[i] == j) {
                                inlist = 1;
                                break;
                            }
                        }
                        if (inlist == 0) {
                            MODL->sellist[MODL->selsize] = j;
                            MODL->selsize++;
                        }
                    }

                    FREE(tempList);

                } else if (MODL->seltype == 1) {
                    tempSize = MODL->selsize;
                    MALLOC(tempList, int, tempSize);
                    for (i = 0; i < tempSize; i++) {
                        tempList[i] = MODL->sellist[i];
                    }

                    MODL->selsize = MODL->body[MODL->selbody].nedge - tempSize;
                    RALLOC(MODL->sellist, int, MODL->selsize);

                    MODL->selsize = 0;
                    for (j = 1; j <= MODL->body[MODL->selbody].nedge; j++) {
                        inlist = 0;
                        for (i = 0; i < tempSize; i++) {
                            if (tempList[i] == j) {
                                inlist = 1;
                                break;
                            }
                        }
                        if (inlist == 0) {
                            MODL->sellist[MODL->selsize] = j;
                            MODL->selsize++;
                        }
                    }

                    FREE(tempList);

                } else if (MODL->seltype == 2) {
                    tempSize = MODL->selsize;
                    MALLOC(tempList, int, tempSize);
                    for (i = 0; i < tempSize; i++) {
                        tempList[i] = MODL->sellist[i];
                    }

                    MODL->selsize = MODL->body[MODL->selbody].nface - tempSize;
                    RALLOC(MODL->sellist, int, MODL->selsize);

                    MODL->selsize = 0;
                    for (j = 1; j <= MODL->body[MODL->selbody].nface; j++) {
                        inlist = 0;
                        for (i = 0; i < tempSize; i++) {
                            if (tempList[i] == j) {
                                inlist = 1;
                                break;
                            }
                        }
                        if (inlist == 0) {
                            MODL->sellist[MODL->selsize] = j;
                            MODL->selsize++;
                        }
                    }

                    FREE(tempList);

                } else {
                    status = signalError(MODL, OCSM_NO_SELECTION,
                                         "NOT can only be applied if Nodes, Edges, or Faces are selected");
                    goto next_branch;
                }

            /* "select sort $key" */
            } else if (strcmp(args[1].str, "sort") == 0 || strcmp(args[1].str, "SORT") == 0) {
                if (MODL->selsize <= 0 || MODL->seltype < 0 || MODL->seltype > 2) {
                    status = signalError(MODL, OCSM_NO_SELECTION,
                                         "there is nothing selected to sort");
                    goto next_branch;
                }

                /* make a list of the properties */
                MALLOC(props, double, MODL->selsize);

                if        (MODL->seltype == 0) {
                    status = EG_getBodyTopos(MODL->body[MODL->selbody].ebody, NULL, NODE, &nobjs, &eobjs);
                } else if (MODL->seltype == 1) {
                    status = EG_getBodyTopos(MODL->body[MODL->selbody].ebody, NULL, EDGE, &nobjs, &eobjs);
                } else if (MODL->seltype == 2) {
                    status = EG_getBodyTopos(MODL->body[MODL->selbody].ebody, NULL, FACE, &nobjs, &eobjs);
                }
                CATCH_STATUS(select);
                SPLINT_CHECK_FOR_NULL(eobjs);

                for (ilist = 0; ilist < MODL->selsize; ilist++) {
                    ient = MODL->sellist[ilist] - 1;

                    if        (strcmp(args[2].str, "xmin") == 0 || strcmp(args[2].str, "XMIN") == 0) {
                        status = EG_getBoundingBox(eobjs[ient], data);
                        if (status != SUCCESS) EG_free(eobjs);
                        CATCH_STATUS(select);

                        props[ilist] = data[0];
                    } else if (strcmp(args[2].str, "ymin") == 0 || strcmp(args[2].str, "YMIN") == 0) {
                        status = EG_getBoundingBox(eobjs[ient], data);
                        if (status != SUCCESS) EG_free(eobjs);
                        CATCH_STATUS(select);

                        props[ilist] = data[1];
                    } else if (strcmp(args[2].str, "zmin") == 0 || strcmp(args[2].str, "ZMIN") == 0) {
                        status = EG_getBoundingBox(eobjs[ient], data);
                        if (status != SUCCESS) EG_free(eobjs);
                        CATCH_STATUS(select);

                        props[ilist] = data[2];
                    } else if (strcmp(args[2].str, "xmax") == 0 || strcmp(args[2].str, "XMAX") == 0) {
                        status = EG_getBoundingBox(eobjs[ient], data);
                        if (status != SUCCESS) EG_free(eobjs);
                        CATCH_STATUS(select);

                        props[ilist] = data[3];
                    } else if (strcmp(args[2].str, "ymax") == 0 || strcmp(args[2].str, "YMAX") == 0) {
                        status = EG_getBoundingBox(eobjs[ient], data);
                        if (status != SUCCESS) EG_free(eobjs);
                        CATCH_STATUS(select);

                        props[ilist] = data[4];
                    } else if (strcmp(args[2].str, "zmax") == 0 || strcmp(args[2].str, "ZMAX") == 0) {
                        status = EG_getBoundingBox(eobjs[ient], data);
                        if (status != SUCCESS) EG_free(eobjs);
                        CATCH_STATUS(select);

                        props[ilist] = data[5];
                    } else if (strcmp(args[2].str, "area") == 0 || strcmp(args[2].str, "AREA") == 0) {
                        if (MODL->seltype == 2) {
                            status = EG_getMassProperties(eobjs[ient], data);
                            if (status != SUCCESS) EG_free(eobjs);
                            CATCH_STATUS(select);

                            props[ilist] = data[1];
                        } else {
                            status = OCSM_ILLEGAL_ARGUMENT;
                            if (status != SUCCESS) EG_free(eobjs);
                            CATCH_STATUS(select);
                        }
                    } else if (strcmp(args[2].str, "length") == 0 || strcmp(args[2].str, "LENGTH") == 0) {
                        if (MODL->seltype == 1) {
                            status = EG_getMassProperties(eobjs[ient], data);
                            if (status != SUCCESS) EG_free(eobjs);
                            CATCH_STATUS(select);

                            props[ilist] = data[1];
                        } else {
                            status = OCSM_ILLEGAL_ARGUMENT;
                            if (status != SUCCESS) EG_free(eobjs);
                            CATCH_STATUS(select);
                        }
                    } else if (strcmp(args[2].str, "xcg") == 0 || strcmp(args[2].str, "XCG") == 0) {
                        if (MODL->seltype != 0) {
                            status = EG_getMassProperties(eobjs[ient], data);
                            if (status != SUCCESS) EG_free(eobjs);
                            CATCH_STATUS(select);

                            props[ilist] = data[2];
                        } else {
                            props[ilist] = MODL->body[MODL->selbody].node[ient+1].x;
                        }
                    } else if (strcmp(args[2].str, "ycg") == 0 || strcmp(args[2].str, "YCG") == 0) {
                        if (MODL->seltype != 0) {
                            status = EG_getMassProperties(eobjs[ient], data);
                            if (status != SUCCESS) EG_free(eobjs);
                            CATCH_STATUS(select);

                            props[ilist] = data[3];
                        } else {
                            props[ilist] = MODL->body[MODL->selbody].node[ient+1].y;
                        }
                    } else if (strcmp(args[2].str, "zcg") == 0 || strcmp(args[2].str, "ZCG") == 0) {
                        if (MODL->seltype != 0) {
                            status = EG_getMassProperties(eobjs[ient], data);
                            if (status != SUCCESS) EG_free(eobjs);
                            CATCH_STATUS(select);

                            props[ilist] = data[4];
                        } else {
                            props[ilist] = MODL->body[MODL->selbody].node[ient+1].z;
                        }
                    } else {
                        status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                             "illegal SORT key");
                        goto next_branch;
                    }
                }

                EG_free(eobjs);

                /* bubble sort */
                for (ipass = 0; ipass < MODL->selsize; ipass++) {
                    nswap = 0;
                    for (ilist = 0; ilist < MODL->selsize-1; ilist++) {
                        if (props[ilist+1] < props[ilist]-EPS06) {
                            iswap                  = MODL->sellist[ilist  ];
                            MODL->sellist[ilist  ] = MODL->sellist[ilist+1];
                            MODL->sellist[ilist+1] = iswap;

                            rswap          = props[ilist  ];
                            props[ilist  ] = props[ilist+1];
                            props[ilist+1] = rswap;

                            nswap++;
                        }
                    }
                    if (nswap == 0) break;
                }

                /* cleanup */
                FREE(props);

            } else {
                status = OCSM_ILLEGAL_ARGUMENT;
                CATCH_STATUS(select);
            }

            /* clear any signals that might have been thrown by str2val */
            MODL->sigCode    =   0;
            MODL->sigMesg[0] = '\0';

            /* update @-parameters (SELECT) */
            status = setupAtPmtrs(MODL, 1);
            CATCH_STATUS(setupAtPmtrs);

            /* put Attributes for OCSM_SELECT on the selected Body/Faces/Edges/Nodes */
            if (MODL->brch[ibrch].nattr > 0) {
                if        (MODL->seltype == 2) {
                    SPLINT_CHECK_FOR_NULL(MODL->sellist);

                    for (ilist = 0; ilist < MODL->selsize; ilist++) {
                        iface = MODL->sellist[ilist];
                        eface = MODL->body[MODL->selbody].face[iface].eface;

                        SPRINT1(1, "                     putting Branch Attributes on Face %d", iface);

                        status = setEgoAttribute(MODL, ibrch, eface);
                        CATCH_STATUS(setEgoAttribute);

                        status = colorizeFace(MODL, MODL->selbody, iface);
                        CATCH_STATUS(colorizeFace);
                    }
                } else if (MODL->seltype == 1) {
                    SPLINT_CHECK_FOR_NULL(MODL->sellist);

                    for (ilist = 0; ilist < MODL->selsize; ilist++) {
                        iedge = MODL->sellist[ilist];
                        eedge = MODL->body[MODL->selbody].edge[iedge].eedge;

                        SPRINT1(1, "                     putting Branch Attributes on Edge %d", iedge);

                        status = setEgoAttribute(MODL, ibrch, eedge);
                        CATCH_STATUS(setEgoAttribute);

                        status = colorizeEdge(MODL, MODL->selbody, iedge);
                        CATCH_STATUS(colorizeEdge);
                    }
                } else if (MODL->seltype == 0) {
                    SPLINT_CHECK_FOR_NULL(MODL->sellist);

                    for (ilist = 0; ilist < MODL->selsize; ilist++) {
                        inode = MODL->sellist[ilist];
                        enode = MODL->body[MODL->selbody].node[inode].enode;

                        SPRINT1(1, "                     putting Branch Attributes on Node %d", inode);

                        status = setEgoAttribute(MODL, ibrch, enode);
                        CATCH_STATUS(setEgoAttribute);

                        status = colorizeNode(MODL, MODL->selbody, inode);
                        CATCH_STATUS(colorizeNode);
                    }
                } else {
                    SPRINT0(1, "                     putting Branch Attributes on Body");

                    status = setEgoAttribute(MODL, ibrch, MODL->body[MODL->selbody].ebody);
                    CATCH_STATUS(setEgoAttribute);

                    status = storeCsystem(MODL, ibrch, MODL->selbody);
                    CHECK_STATUS(storeCsystem);
                }
            }

        /* execute: "interface $argName $argType default=0" */
        } else if (type == OCSM_INTERFACE) {
            if (args[3].nval == 0) {
                SPRINT4(1, "              [%4d] interface:  %s  %s  %s",
                        ibrch, args[1].str, args[2].str, args[3].str);
            } else {
                SPRINT4(1, "              [%4d] interface:  %s  %s  %11.5f",
                        ibrch, args[1].str, args[2].str, args[3].val[0]);
            }

            /* make sure that the local variable name is not the same
               as a constant or output Parameter */
            for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
                if        (strcmp(MODL->pmtr[ipmtr].name, args[1].str) == 0 &&
                           MODL->pmtr[ipmtr].type == OCSM_CONPMTR            ) {
                    status = signalError(MODL, OCSM_PMTR_IS_CONPMTR,
                                         "INTERFACE variable \"%s\" cannot match CONPMTR", args[1].str);
                    goto next_branch;
                } else if (strcmp(MODL->pmtr[ipmtr].name, args[1].str) == 0 &&
                           MODL->pmtr[ipmtr].type == OCSM_OUTPMTR             ) {
                    status = signalError(MODL, OCSM_PMTR_IS_OUTPMTR,
                                         "INTERFACE variable \"%s\" cannot match OUTPMTR", args[1].str);
                    goto next_branch;
                }
            }

            /* this is skipped because it was processed as part of the UDPRIM
               statement */

        /* execute: "prefix $prefix=." */
        } else if (type == OCSM_PREFIX) {
            SPRINT2(1, "    executing [%4d] prefix:     %s",
                    ibrch, args[1].str);

            /* clear the prefix if the argument is a dot (.) */
            if (strcmp(args[1].str, ".") == 0) {
                MODL->prefix[0] = '\0';

            /* otherwise set the prefix */
            } else {
                MODL->prefix[0] = '\0';

                /* make sure the prefix starts with a letter or colon */
                if (isalpha(args[1].str[0]) == 0 && args[1].str[0] != ':') {
                    status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                         "PREFIX must start with letter or colon");
                    goto next_branch;
                }

                /* make sure the prefix only contains letters, digits, colons,
                   at-signs, and underscores */
                for (i = 1; i < STRLEN(args[1].str); i++) {
                    if (isalpha(args[1].str[i]) == 0 && isdigit(args[1].str[i]) == 0 &&
                        args[1].str[i] != ':'        && args[1].str[i] != '@'        &&
                        args[1].str[i] != '_'                                          ) {
                        status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                             "PREFIX must only contain letters, digits, colons, at-signs, and underscores");
                        goto next_branch;
                    }
                }

                /* prefix is formed well, so store it */
                STRNCPY(MODL->prefix, args[1].str, MAX_NAME_LEN);
            }

            SPRINT1(1, "        prefix is now \"%s\"", MODL->prefix);

        /* execute: "project x y z dx dy dz useEdges=0" */
        } else if (type == OCSM_PROJECT) {
            SPRINT8(1, "    executing [%4d] project:    %11.5f  %11.5f  %11.5f  %11.5f  %11.5f  %11.5f  %11.5f",
                    ibrch, args[1].val[0], args[2].val[0], args[3].val[0],
                           args[4].val[0], args[5].val[0], args[6].val[0], args[7].val[0]);

            /* get the Bodys on the top of the stack */
            if (nstack < 1) {
                status = signalError(MODL, OCSM_INSUFFICIENT_BODYS_ON_STACK,
                                     "PROJECT expects a Body on stack");
                goto next_branch;
            } else {
                ibodyl = stack[nstack-1];
            }

            /* make sure that the current Body has a tessellation */
            status = ocsmTessellate(MODL, ibodyl);
            CHECK_STATUS(ocsmTessellate);

            /* loop through all Triangles in all Faces to find closest projection */
            if (NINT(args[7].val[0]) == 0) {
                ibest = 0;
                tbest = HUGEQ;
                xbest = HUGEQ;
                ybest = HUGEQ;
                zbest = HUGEQ;

                for (iface = 1; iface <= MODL->body[ibodyl].nface; iface++) {
                    status = EG_getTessFace(MODL->body[ibodyl].etess, iface,
                                            &npnt, &xyz, &uv, &ptype, &pindx,
                                            &ntri, &tris, &tric);
                    CATCH_STATUS(EG_getFaceTess);

                    for (itri = 0; itri < ntri; itri++) {
                        u0 = uv[ 2*tris[3*itri  ]-2];
                        v0 = uv[ 2*tris[3*itri  ]-1];
                        x0 = xyz[3*tris[3*itri  ]-3];
                        y0 = xyz[3*tris[3*itri  ]-2];
                        z0 = xyz[3*tris[3*itri  ]-1];
                        u1 = uv[ 2*tris[3*itri+1]-2];
                        v1 = uv[ 2*tris[3*itri+1]-1];
                        x1 = xyz[3*tris[3*itri+1]-3];
                        y1 = xyz[3*tris[3*itri+1]-2];
                        z1 = xyz[3*tris[3*itri+1]-1];
                        u2 = uv[ 2*tris[3*itri+2]-2];
                        v2 = uv[ 2*tris[3*itri+2]-1];
                        x2 = xyz[3*tris[3*itri+2]-3];
                        y2 = xyz[3*tris[3*itri+2]-2];
                        z2 = xyz[3*tris[3*itri+2]-1];

                        mat[0] = x0 - x2;  mat[1] = x1 - x2;  mat[2] = -args[4].val[0];  rhs[0] = args[1].val[0] - x2;
                        mat[3] = y0 - y2;  mat[4] = y1 - y2;  mat[5] = -args[5].val[0];  rhs[1] = args[2].val[0] - y2;
                        mat[6] = z0 - z2;  mat[7] = z1 - z2;  mat[8] = -args[6].val[0];  rhs[2] = args[3].val[0] - z2;

                        status = matsol(mat, rhs, 3, 1, s0s1tt);
                        if (status != SUCCESS) continue;

                        s0 = s0s1tt[0];
                        s1 = s0s1tt[1];
                        tt = s0s1tt[2];

                        if (tt < -EPS06 || s0 < -EPS06 || s1 < -EPS06 || (1-s0-s1) < -EPS06) continue;

                        if (tt < tbest) {
                            ibest  = iface;
                            tbest  = tt;
                            uv_[0] = s0 * (u0 - u2) + s1 * (u1 - u2) + u2;
                            uv_[1] = s0 * (v0 - v2) + s1 * (v1 - v2) + v2;
                            uv_[2] = tt;

                            /* newton iteration to put it on the surface (should usually converge
                               in fewer than 5 iterations) */
                            for (iter = 0; iter < 30; iter++) {
                                status = EG_evaluate(MODL->body[ibodyl].face[iface].eface, uv_, data);
                                CHECK_STATUS(EG_evaluate);

                                /* actually -1.0*residual */
                                rhs[0] = data[0] - (args[1].val[0] + uv_[2] * args[4].val[0]);
                                rhs[1] = data[1] - (args[2].val[0] + uv_[2] * args[5].val[0]);
                                rhs[2] = data[2] - (args[3].val[0] + uv_[2] * args[6].val[0]);

                                if (fabs(rhs[0]) < EPS06 && fabs(rhs[1]) < EPS06 && fabs(rhs[2]) < EPS06) break;

                                mat[0] = -data[3];   mat[1] = -data[6];   mat[2] = args[4].val[0];
                                mat[3] = -data[4];   mat[4] = -data[7];   mat[5] = args[5].val[0];
                                mat[6] = -data[5];   mat[7] = -data[8];   mat[8] = args[6].val[0];

                                status = matsol(mat, rhs, 3, 1, duvt);
                                CHECK_STATUS(matsol);

                                uv_[0] += duvt[0];
                                uv_[1] += duvt[1];
                                uv_[2] += duvt[2];
                            }

                            xbest = data[0];
                            ybest = data[1];
                            zbest = data[2];
                        }
                    }
                }

            /* loop through all segments in all Edges to find closest projection */
            } else {
                ibest = 0;
                tbest = HUGEQ;
                xbest = HUGEQ;
                ybest = HUGEQ;
                zbest = HUGEQ;

                for (iedge = 1; iedge <= MODL->body[ibodyl].nedge; iedge++) {
                    if (MODL->body[ibodyl].edge[iedge].itype == DEGENERATE) continue;

                    status = EG_getTessEdge(MODL->body[ibodyl].etess, iedge,
                                            &npnt, &xyz, &uv);
                    CATCH_STATUS(EG_getTessEdge);

                    for (ipnt = 1; ipnt < npnt; ipnt++) {
                        u0 = uv[   ipnt-1];
                        x0 = xyz[3*ipnt-3];
                        y0 = xyz[3*ipnt-2];
                        z0 = xyz[3*ipnt-1];
                        u1 = uv[   ipnt  ];
                        x1 = xyz[3*ipnt  ];
                        y1 = xyz[3*ipnt+1];
                        z1 = xyz[3*ipnt+2];

                        mat[0] = x0 - x1;  mat[1] = -args[4].val[0];  rhs[0] = args[1].val[0] - x0;
                        mat[2] = y0 - y1;  mat[3] = -args[5].val[0];  rhs[1] = args[2].val[0] - y0;
                        mat[4] = z0 - z1;  mat[5] = -args[6].val[0];  rhs[2] = args[3].val[0] - z0;

                        status = solsvd(mat, rhs, 3, 2, sval, s0s1tt);
                        if (status != SUCCESS) continue;

                        s0 = s0s1tt[0];
                        tt = s0s1tt[1];

                        if (tt < 0 || s0 < 0 || s0 > 1) continue;

                        if (tt < tbest) {
                            ibest  = iedge;
                            tbest  = tt;
                            uv_[0] = (1 - tt) * u0 + tt * u1;
                            uv_[1] = tt;

                            /* newton iteration to put it on the curve (should usually converge
                               in fewer than 5 iterations) */
                            for (iter = 0; iter < 30; iter++) {
                                status = EG_evaluate(MODL->body[ibodyl].edge[iedge].eedge, uv_, data);
                                CHECK_STATUS(EG_evaluate);

                                res[0] = data[0] - (args[1].val[0] + uv_[1] * args[4].val[0]);
                                res[1] = data[1] - (args[2].val[0] + uv_[1] * args[5].val[0]);
                                res[2] = data[2] - (args[3].val[0] + uv_[1] * args[6].val[0]);

                                if (fabs(res[0]) < EPS06 && fabs(res[1]) < EPS06 && fabs(res[2]) < EPS06) break;

                                mat[0] =  data[3]        * data[3]        + data[4]        * data[4]        + data[5]        * data[5];
                                mat[1] = -data[3]        * args[4].val[0] - data[4]        * args[5].val[0] - data[5]        * args[6].val[0];
                                mat[2] =  mat[1];
                                mat[3] =  args[4].val[0] * args[4].val[0] + args[5].val[0] * args[5].val[0] + args[6].val[0] * args[6].val[0];

                                rhs[0] = -data[3]        * res[0]         - data[4]        * res[1]         - data[5]        * res[2];
                                rhs[1] =  args[4].val[0] * res[0]         + args[5].val[0] * res[1]         + args[6].val[0] * res[2];

                                status = matsol(mat, rhs, 2, 1, duvt);
                                CHECK_STATUS(matsol);

                                uv_[0] += duvt[0];
                                uv_[1] += duvt[1];
                            }

                            xbest = data[0];
                            ybest = data[1];
                            zbest = data[2];
                        }
                    }
                }
            }

            /* if any hits, store the results of the best in the @-parameters */
            if (ibest > 0) {
                if (NINT(args[7].val[0]) == 0) {
                    status = ocsmFindPmtr(MODL, "@iface", OCSM_LOCALVAR, 1, 1, &AT_iface);
                    CATCH_STATUS(ocsmFindPmtr);

                    status = ocsmSetValuD(MODL, AT_iface, 1, 1, (double)ibest);
                    CATCH_STATUS(ocsmSetValuD);
                } else {
                    status = ocsmFindPmtr(MODL, "@iedge", OCSM_LOCALVAR, 1, 1, &AT_iedge);
                    CATCH_STATUS(ocsmFindPmtr);

                    status = ocsmSetValuD(MODL, AT_iedge, 1, 1, (double)ibest);
                    CATCH_STATUS(ocsmSetValuD);
                }

                status = ocsmFindPmtr(MODL, "@xcg", OCSM_LOCALVAR, 1, 1, &AT_xcg);
                CATCH_STATUS(ocsmFindPmtr);

                status = ocsmSetValuD(MODL, AT_xcg, 1, 1, xbest);
                CATCH_STATUS(ocsmSetValuD);

                status = ocsmFindPmtr(MODL, "@ycg", OCSM_LOCALVAR, 1, 1, &AT_ycg);
                CATCH_STATUS(ocsmFindPmtr);

                status = ocsmSetValuD(MODL, AT_ycg, 1, 1, ybest);
                CATCH_STATUS(ocsmSetValuD);

                status = ocsmFindPmtr(MODL, "@zcg", OCSM_LOCALVAR, 1, 1, &AT_zcg);
                CATCH_STATUS(ocsmFindPmtr);

                status = ocsmSetValuD(MODL, AT_zcg, 1, 1, zbest);
                CATCH_STATUS(ocsmSetValuD);

                /* do not recompute mass properties (to overwrite the values that we just set) */
                MODL->hasMPs = 1;

            /* otherwise, return an error */
            } else {
                status = signalError(MODL, OCSM_FACE_NOT_FOUND,
                                     "no Face was found");
                goto next_branch;
            }

        /* execute: "macbeg imacro" */
        } else if (type == OCSM_MACBEG) {
            SPRINT2(1, "    executing [%4d] macbeg:     %11.5f",
                    ibrch, args[1].val[0]);

            /* determine the storage location */
            imacro = NINT(args[1].val[0]);

            if (imacro < 1 || imacro > MAX_NUM_MACROS) {
                status = OCSM_ILLEGAL_MACRO_INDEX;
                CATCH_STATUS(macbeg);
            }

            /* an error if storage location is not empty */
            if (macros[imacro] > 0) {
                status = OCSM_STORAGE_ALREADY_USED;
                CATCH_STATUS(macbeg);
            }


            macros[imacro] = ibrch;

            SPRINT2(1, "                          Storing Branch %4d in storage %d", ibrch, imacro);

        /* execute: "macend" */
        } else if (type == OCSM_MACEND) {
            SPRINT1(1, "    executing [%4d] macend:",
                    ibrch);

            npatn--;
            ibrch = patn[npatn].iend;

        /* execute: "recall imacro" */
        } else if (type == OCSM_RECALL) {
            SPRINT2(1, "    executing [%4d] recall:     %11.5f",
                    ibrch, args[1].val[0]);

            /* determine the storage location */
            imacro = NINT(args[1].val[0]);

            if (imacro < 1 || imacro > MAX_NUM_MACROS) {
                status = OCSM_ILLEGAL_MACRO_INDEX;
                CATCH_STATUS(recall);
            } else if (macros[imacro] <= 0) {
                status = OCSM_NOTHING_PREVIOUSLY_STORED;
                CATCH_STATUS(recall);
            }

            if (npatn < MAX_NESTING) {
                patn[npatn].itype = OCSM_RECALL;
                patn[npatn].ibeg  = macros[imacro];
                patn[npatn].iend  = ibrch;
                patn[npatn].ncopy =  1;
                patn[npatn].icopy = +1;
                patn[npatn].ipmtr = -1;
                npatn++;
            } else {
                status = signalError(MODL, OCSM_NESTED_TOO_DEEPLY,
                                     "npatn=%d >0 MAX_NESTING=%d", npatn, MAX_NESTING);
                goto cleanup;        // cannot be caught
            }

            /* start executing just after the macbeg */
            ibrch = macros[imacro];

            SPRINT1(1, "                          Entering storage %d", imacro);

        /* execute: "store $name index=0 keep=0" */
        } else if (type == OCSM_STORE) {
            SPRINT4(1, "    executing [%4d] store:      %s  %11.5f  %11.5f",
                    ibrch, args[1].str, args[2].val[0], args[3].val[0]);

            /* if $name is . (dot), pop one Body off the stack */
            if (strcmp(args[1].str, ".") == 0) {
                if (args[2].val[0] != 0 || args[3].val[0] != 0) {
                    SPRINT0(1, "WARNING:: index and keep ignored since $name=.");
                    (MODL->nwarn)++;
                }

                nstack--;
                SPRINT1(1, "                          Body   %4d removed from stack", stack[nstack]);

                if (stack[nstack] == 0) {
                    status = setupAtPmtrs(MODL, -1);

                    goto next_branch;
                }

            /* if $name is .. (dot-dot), pop all Bodys off stack back to the Mark
               (or until stack is empty) */
            } else  if (strcmp(args[1].str, "..") == 0) {
                if (args[2].val[0] != 0 || args[3].val[0] != 0) {
                    SPRINT0(1, "WARNING:: index and keep ignored since $name=..");
                    (MODL->nwarn)++;
                }

                while (nstack > 0) {
                    if (stack[nstack-1] == 0) {
                        nstack--;
                        break;
                    } else {
                        nstack--;
                        SPRINT1(1, "                          Body   %4d removed from stack", stack[nstack]);
                    }
                }

            /* if $name is ... (ellipsis), pop all Bodys off stack */
            } else if (strcmp(args[1].str, "...") == 0) {
                if (args[2].val[0] != 0 || args[3].val[0] != 0) {
                    SPRINT0(1, "WARNING:: index and keep ignored since $name=...");
                    (MODL->nwarn)++;
                }

                while (nstack > 0) {
                    nstack--;
                    SPRINT1(1, "                          Body   %4d removed from stack", stack[nstack]);
                }

            } else {
                /* find the Body on the top of the stack */
                if (nstack < 1) {
                    status = signalError(MODL, OCSM_INSUFFICIENT_BODYS_ON_STACK,
                                         "STORE expects a Body on the stack");
                    goto next_branch;
                } else {
                    ibodyl = stack[nstack-1];
                }

                /* if ibodyl is 0, it is a mark, so return an error */
                if (ibodyl == 0) {
                    status = signalError(MODL, OCSM_INSUFFICIENT_BODYS_ON_STACK,
                                         "STORE cannot be applied to a MARK");
                    goto next_branch;
                } else {
                    igroup = MODL->body[ibodyl].igroup;
                }

                /* if index is negative, find first unused index */
                if (NINT(args[2].val[0]) < 0) {
                    if (MODL->stor == NULL) {
                        kstor = 0;
                    } else {
                        for (kstor = 0; kstor <= MODL->nstor; kstor++) {
                            icount = 0;
                            for (jstor = 0; jstor < MODL->nstor; jstor++) {
                                if (strcmp(args[1].str, MODL->stor[jstor].name) == 0 &&
                                    kstor == MODL->stor[jstor].index    ) {
                                    icount++;
                                    break;
                                }
                            }
                            if (icount == 0) {
                                SPRINT2(1, "INFO:: storage %s (%d) begin used",
                                        args[1].str, kstor);
                                break;
                            }
                        }
                    }
                } else {
                    kstor = NINT(args[2].val[0]);
                }

                /* see if name/index is already used */
                istor = -1;
                for (jstor = 0; jstor < MODL->nstor; jstor++) {
                    SPLINT_CHECK_FOR_NULL(MODL->stor);

                    if (strcmp(args[1].str, MODL->stor[jstor].name) == 0 &&
                        kstor ==  MODL->stor[jstor].index                  ) {
                        istor = jstor;
                        SPRINT2(1, "WARNING:: storage %s (%d) is being overwritten",
                                MODL->stor[jstor].name, MODL->stor[jstor].index);
                        (MODL->nwarn)++;
                        break;
                    }
                }

                /* make a list to keep track of the Bodys that should be stored */
                MALLOC(iblist, int, nstack);

                nblist = 0;
                while (nstack > 0) {
                    ibodyl = stack[--nstack];
                    if (ibodyl <= 0) {
                        if (nstack < MAX_STACK_SIZE) {
                            nstack++;
                        } else {
                            status = signalError(MODL, OCSM_TOO_MANY_BODYS_ON_STACK,
                                                 "Too many Bodys on Stack");
                            goto cleanup;        // cannot be caught
                        }
                        break;
                    }

                    if (MODL->body[ibodyl].igroup != igroup) {
                        if (nstack < MAX_STACK_SIZE) {
                            nstack++;
                        } else {
                            status = signalError(MODL, OCSM_TOO_MANY_BODYS_ON_STACK,
                                                 "Too many Bodys on Stack");
                            goto cleanup;        // cannot be caught
                        }
                        break;
                    }

                    iblist[nblist++] = ibodyl;
                }

                /* if storage does not exist, create it now */
                if (istor < 0) {
                    istor = MODL->nstor;

                    (MODL->nstor)++;
                    RALLOC(MODL->stor, stor_T, MODL->nstor);

                    STRNCPY(MODL->stor[istor].name, args[1].str, MAX_NAME_LEN);
                    MODL->stor[istor].index = kstor;
                    MODL->stor[istor].nbody = nblist;
                    MODL->stor[istor].ibody = NULL;
                    MODL->stor[istor].ebody = NULL;

                    MALLOC(MODL->stor[istor].ibody, int, MODL->stor[istor].nbody);
                    MALLOC(MODL->stor[istor].ebody, ego, MODL->stor[istor].nbody);

                /* if it already exists, update the info */
                } else {
                    SPLINT_CHECK_FOR_NULL(MODL->stor);

                    MODL->stor[istor].nbody = nblist;

                    RALLOC(MODL->stor[istor].ibody, int, MODL->stor[istor].nbody);
                    RALLOC(MODL->stor[istor].ebody, ego, MODL->stor[istor].nbody);
                }

                /* if the keep flag is set, restore the stack */
                if (NINT(args[3].val[0]) == 1) {
                    if (nstack+nblist < MAX_STACK_SIZE) {
                        nstack += nblist;
                    } else {
                        status = signalError(MODL, OCSM_TOO_MANY_BODYS_ON_STACK,
                                             "Too many Bodys on Stack");
                        goto cleanup;        // cannot be caught
                    }
                }

                /* save the Body info */
                SPLINT_CHECK_FOR_NULL(MODL->stor);

                for (i = MODL->stor[istor].nbody-1; i >= 0; i--) {
                    ibodyl = iblist[i];

                    MODL->stor[istor].ibody[i] = ibodyl;
                    MODL->stor[istor].ebody[i] = MODL->body[ibodyl].ebody;

                    SPRINT1(1, "                          Body   %4d stored", ibodyl);
                }

                FREE(iblist);
            }

        /* execute: "patbeg $pmtrName ncopy" */
        } else if (type == OCSM_PATBEG) {
            SPRINT3(1, "    executing [%4d] patbeg:         %s  %11.5f",
                    ibrch, args[1].str, args[2].val[0]);

            /* if $pmtrName is an implicit string, evaluate it */
            if (args[1].str[0] != '!') {
                STRNCPY(pname, args[1].str, MAX_EXPR_LEN);
            } else {
                status = str2val(args[1].str, MODL, &value, &dot, pname);
                if (MODL->sigCode != SUCCESS) goto next_branch;
                CHECK_STATUS(str2val);
            }

            SPLINT_CHECK_FOR_NULL(pname);

            if (strlen(pname) == 0) {
                status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                     "\"%s\" does not evalaute to a string", args[1].str);
                goto cleanup;      // cannot be caught
            } else if (strstr(pname, "[") != NULL) {
                status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                     "$pmtrName cannot contain a subscript");
                goto cleanup;      // cannot be caught
            } else if (pname[0] == '@') {
                status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                     "$pmtrName cannot start with @-sign");
                goto cleanup;      // cannot be caught
            }

            /* check to see if pname already exists */
            status = ocsmFindPmtr(MODL, pname, 0, 0, 0, &jpmtr);
            CHECK_STATUS(ocsmFindPmtr);

            /* if it already exists, make sure it is a scalar local variable or outpmtr */
            if (jpmtr > 0) {
                if (MODL->pmtr[jpmtr].type == OCSM_DESPMTR) {
                    status = signalError(MODL, OCSM_PMTR_IS_DESPMTR,
                                         "\"%s\" cannot be SET because it is a DESPMTR", pname);
                    goto cleanup;        // cannot be caught
                } else if (MODL->pmtr[jpmtr].type == OCSM_CFGPMTR) {
                    status = signalError(MODL, OCSM_PMTR_IS_DESPMTR,
                                         "\"%s\" cannot be SET because it is a CFGPMTR", pname);
                    goto cleanup;        // cannot be caught
                } else if (MODL->pmtr[jpmtr].type == OCSM_CONPMTR) {
                    status = signalError(MODL, OCSM_PMTR_IS_CONPMTR,
                                         "\"%s\" cannot be SET because it is a CONPMTR", pname);
                    goto cleanup;        // cannot be caught
                } else if (MODL->pmtr[jpmtr].nrow > 1 || MODL->pmtr[jpmtr].ncol > 1) {
                    status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                         "$pmtrName cannot be the name of an array");
                    goto cleanup;        // cannot be caught
                }

                /* if an unknown type, make a local variable */
                if (MODL->pmtr[jpmtr].type == OCSM_UNKNOWN) {
                    MODL->pmtr[jpmtr].type =  OCSM_LOCALVAR;
                }

            /* if it does not exist, make it now */
            } else {
                status = ocsmFindPmtr(MODL, pname, OCSM_LOCALVAR, 1, 1, &jpmtr);
                if (status != SUCCESS) {
                    status = signalError(MODL, OCSM_ILLEGAL_PMTR_NAME,
                                         "\"%s\" is not a legal Parameter name", pname);
                    goto cleanup;        // cannot be caught;
                }
            }

            /* initialize info about the pattern */
            if (npatn < MAX_NESTING) {
                patn[npatn].itype = OCSM_PATBEG;
                patn[npatn].ibeg  = ibrch;
                patn[npatn].iend  = -1;
                patn[npatn].ncopy = NINT(args[2].val[0]);
                patn[npatn].icopy =  1;
                patn[npatn].ipmtr = jpmtr;
            } else {
                status = signalError(MODL, OCSM_NESTED_TOO_DEEPLY,
                                     "npatn=%d >0 MAX_NESTING=%d", npatn, MAX_NESTING);
                goto cleanup;        // cannot be caught
            }

            /* store info about this pattern */
            /* find matching patend */
            icount = 1;
            for (ibrchl = ibrch+1; ibrchl <= MODL->nbrch; ibrchl++) {
                if        (MODL->brch[ibrchl].type == OCSM_PATBEG) {
                    icount++;
                } else if (MODL->brch[ibrchl].type == OCSM_PATEND) {
                    icount--;

                    if (icount == 0) {
                        patn[npatn].iend = ibrchl;
                        break;
                    }
                }
            }

            /* if no copies are required, jump to Branch after patend */
            if (patn[npatn].ncopy < 1) {
                if (patn[npatn].iend > patn[npatn].ibeg) {
                    ibrch = patn[npatn].iend;
                } else {
                    break;    /* used if no matching patend */
                }

            /* otherwise increment the number of patterns and continue */
            } else {
                irow = icol = 1;

                status = ocsmSetValuD(MODL, patn[npatn].ipmtr, irow, icol, 1.0);
                CATCH_STATUS(ocsmSetValuD);

                SPRINT2(1, "                          pattern counter: %s = %3d",
                        MODL->pmtr[patn[npatn].ipmtr].name, 1);

                npatn++;
            }

        /* execute: "patbreak val1=1 $op1=gt val2=0 $op2=and val3=0 $op3=eq val4=0" */
        } else if (type == OCSM_PATBREAK) {

            /* evaluate val1 $op1 val2 */
            bool1 = 0;
            if        (strcmp(args[2].str, "lt") == 0 ||
                       strcmp(args[2].str, "LT") == 0   ) {
                if (args[1].val[0] <  args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "le") == 0 ||
                       strcmp(args[2].str, "LE") == 0   ) {
                if (args[1].val[0] <= args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "eq") == 0 ||
                       strcmp(args[2].str, "EQ") == 0   ) {
                if (args[1].val[0] == args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "ge") == 0 ||
                       strcmp(args[2].str, "GE") == 0   ) {
                if (args[1].val[0] >= args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "gt") == 0 ||
                       strcmp(args[2].str, "GT") == 0   ) {
                if (args[1].val[0] >  args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "ne") == 0 ||
                       strcmp(args[2].str, "NE") == 0   ) {
                if (args[1].val[0] != args[3].val[0]) bool1 = 1;
            }

            /* set up for lazy evaluations */
            if        (strcmp(args[4].str, "and") == 0 ||
                       strcmp(args[4].str, "AND") == 0   ) {
                if (bool1 == 0) {
                    bool2  =  0;
                } else {
                    bool2  = -1;
                }
            } else if (strcmp(args[4].str, "or") == 0 ||
                       strcmp(args[4].str, "OR") == 0   ) {
                if (bool1 == 0) {
                    bool2  = -1;
                } else {
                    bool2  =  1;
                }
            } else if (strcmp(args[4].str, "xor") == 0 ||
                       strcmp(args[4].str, "XOR") == 0   ) {
                bool2  = -1;
            } else {
                bool2 = 0;
            }

            /* no need to evaluate last 3 arguments */
            if (bool2 >= 0) {
                SPRINT5(1, "    executing [%4d] patbreak:       %11.5f  %s  %11.5f  %s  ...",
                        ibrch, args[1].val[0], args[2].str, args[3].val[0], args[4].str);

            /* evaluate last 3 arguments */
            } else {
                for (iarg = 5; iarg <= 7; iarg++) {
                    if (iarg == 5) strcpy(thisArg, MODL->brch[ibrch].arg5);
                    if (iarg == 6) strcpy(thisArg, MODL->brch[ibrch].arg6);
                    if (iarg == 7) strcpy(thisArg, MODL->brch[ibrch].arg7);

                    status = str2vals(thisArg, MODL, &(args[iarg].nrow), &(args[iarg].ncol), &values, &dots, str);
                    if (status < SUCCESS) {
                        SPRINT4(0, "ERROR:: problem evaluating argument %d (%s) for Branch %d (%s)",
                                iarg, thisArg, ibrch, ocsmGetText(MODL->brch[ibrch].type));
                        CATCH_STATUS(str2vals);
                    }

                    if (STRLEN(str) > 0) {
                        FREE(  args[iarg].str);
                        MALLOC(args[iarg].str, char, MAX_STRVAL_LEN);

                        args[iarg].nval = 0;
                        STRNCPY(args[iarg].str, str, MAX_STRVAL_LEN);
                    } else {
                        args[iarg].nval = (args[iarg].nrow) * (args[iarg].ncol);

                        FREE(  args[iarg].val);   /* also free's .str since they are unioned */
                        MALLOC(args[iarg].val, double, args[iarg].nval);

                        FREE(  args[iarg].dot);
                        MALLOC(args[iarg].dot, double, args[iarg].nval);

                        SPLINT_CHECK_FOR_NULL(values);
                        SPLINT_CHECK_FOR_NULL(dots  );

                        for (i = 0; i < args[iarg].nval; i++) {
                            args[iarg].val[i] = values[i];
                            args[iarg].dot[i] = dots[  i];
                        }
                    }

                    FREE(values);
                    FREE(dots  );
                }

                SPRINT8(1, "    executing [%4d] patbreak:       %11.5f  %s  %11.5f  %s  %11.5f  %s  %11.5f",
                        ibrch, args[1].val[0], args[2].str, args[3].val[0], args[4].str,
                        args[5].val[0], args[6].str, args[7].val[0]);

                /* evaluate val3 $op3 val4 */
                bool2 = 0;
                if        ((strcmp(args[4].str, "or") == 0 ||
                            strcmp(args[4].str, "OR") == 0   ) && bool1 == 1) {
                    bool2 = 1;
                } else if ((strcmp(args[4].str, "and") == 0 ||
                            strcmp(args[4].str, "AND") == 0   ) && bool1 == 0) {
                    bool2 = 0;
                } else if (strcmp(args[6].str, "lt") == 0 ||
                           strcmp(args[6].str, "LT") == 0   ) {
                    if (args[5].val[0] <  args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "le") == 0 ||
                           strcmp(args[6].str, "LE") == 0   ) {
                    if (args[5].val[0] <= args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "eq") == 0 ||
                           strcmp(args[6].str, "EQ") == 0   ) {
                    if (args[5].val[0] == args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "ge") == 0 ||
                           strcmp(args[6].str, "GE") == 0   ) {
                    if (args[5].val[0] >= args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "gt") == 0 ||
                           strcmp(args[6].str, "GT") == 0   ) {
                    if (args[5].val[0] >  args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "ne") == 0 ||
                           strcmp(args[6].str, "NE") == 0   ) {
                    if (args[5].val[0] != args[7].val[0]) bool2 = 1;
                }
            }

            /* evaluate (val1 $op1 val2) $op2 (val3 $op3 val4) */
            bool3 = 0;
            if        (strcmp(args[4].str, "or") == 0 ||
                       strcmp(args[4].str, "OR") == 0   ) {
                bool3 = (bool1 == 1) || (bool2 == 1);
            } else if (strcmp(args[4].str, "and") == 0 ||
                       strcmp(args[4].str, "AND") == 0   ) {
                bool3 = (bool1 == 1) && (bool2 == 1);
            } else if (strcmp(args[4].str, "xor") == 0 ||
                       strcmp(args[4].str, "XOR") == 0   ) {
                bool3 = (bool1 == 1) != (bool2 == 1);
            }

            /* break out of this pattern if the test pass */
            if (bool3 != 0) {
                npatn--;

                while (ibrch < MODL->nbrch && MODL->brch[ibrch].type != OCSM_PATEND) {
                    ibrch++;

                    if        (MODL->brch[ibrch].type == OCSM_IFTHEN ||
                               MODL->brch[ibrch].type == OCSM_MACBEG   ) {
                        npatn++;
                    } else if (MODL->brch[ibrch].type == OCSM_ENDIF ||
                               MODL->brch[ibrch].type == OCSM_MACEND  ) {
                        npatn--;
                    }
                }
            }

        /* execute: "patend" */
        } else if (type == OCSM_PATEND) {
            SPRINT1(1, "    executing [%4d] patend:",
                    ibrch);

            /* increment the iterator */
            (patn[npatn-1].icopy)++;

            /* go back for another copy */
            if (patn[npatn-1].icopy <= patn[npatn-1].ncopy) {
                newValue = patn[npatn-1].icopy;

                irow = icol = 1;

                status = ocsmSetValuD(MODL, patn[npatn-1].ipmtr, irow, icol, newValue);
                CATCH_STATUS(ocsmSetValuD);

                ibrch = patn[npatn-1].ibeg;

                SPRINT2(1, "                          pattern counter: %s = %3d",
                        MODL->pmtr[patn[npatn-1].ipmtr].name, patn[npatn-1].icopy);

            /* otherwise, we are finished with the pattern */
            } else {
                npatn--;
            }

        /* execute: "ifthen val1 $op1=ne val2=0 $op2=and val3=0 $op3=eq val4=0" */
        } else if (type == OCSM_IFTHEN) {

            /* evaluate val1 $op1 val2 */
            bool1 = 0;
            if        (strcmp(args[2].str, "lt") == 0 ||
                       strcmp(args[2].str, "LT") == 0   ) {
                if (args[1].val[0] <  args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "le") == 0 ||
                       strcmp(args[2].str, "LE") == 0   ) {
                if (args[1].val[0] <= args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "eq") == 0 ||
                       strcmp(args[2].str, "EQ") == 0   ) {
                if (args[1].val[0] == args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "ge") == 0 ||
                       strcmp(args[2].str, "GE") == 0   ) {
                if (args[1].val[0] >= args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "gt") == 0 ||
                       strcmp(args[2].str, "GT") == 0   ) {
                if (args[1].val[0] >  args[3].val[0]) bool1 = 1;
            } else if (strcmp(args[2].str, "ne") == 0 ||
                       strcmp(args[2].str, "NE") == 0   ) {
                if (args[1].val[0] != args[3].val[0]) bool1 = 1;
            }

            /* set up for lazy evaluations */
            if        (strcmp(args[4].str, "and") == 0 ||
                       strcmp(args[4].str, "AND") == 0   ) {
                if (bool1 == 0) {
                    bool2  =  0;
                } else {
                    bool2  = -1;
                }
            } else if (strcmp(args[4].str, "or") == 0 ||
                       strcmp(args[4].str, "OR") == 0   ) {
                if (bool1 == 0) {
                    bool2  = -1;
                } else {
                    bool2  =  1;
                }
            } else if (strcmp(args[4].str, "xor") == 0 ||
                       strcmp(args[4].str, "XOR") == 0   ) {
                bool2  = -1;
            } else {
                bool2 = 0;
            }

            /* no need to evaluate last 3 arguments */
            if (bool2 >= 0) {
                SPRINT5(1, "    executing [%4d] ifthen:         %11.5f  %s  %11.5f  %s  ...",
                        ibrch, args[1].val[0], args[2].str, args[3].val[0], args[4].str);

            /* evaluate last 3 arguments */
            } else {
                for (iarg = 5; iarg <= 7; iarg++) {
                    if (iarg == 5) strcpy(thisArg, MODL->brch[ibrch].arg5);
                    if (iarg == 6) strcpy(thisArg, MODL->brch[ibrch].arg6);
                    if (iarg == 7) strcpy(thisArg, MODL->brch[ibrch].arg7);

                    status = str2vals(thisArg, MODL, &(args[iarg].nrow), &(args[iarg].ncol), &values, &dots, str);
                    if (status < SUCCESS) {
                        SPRINT4(0, "ERROR:: problem evaluating argument %d (%s) for Branch %d (%s)",
                                iarg, thisArg, ibrch, ocsmGetText(MODL->brch[ibrch].type));
                        CATCH_STATUS(str2vals);
                    }

                    if (STRLEN(str) > 0) {
                        FREE(  args[iarg].str);
                        MALLOC(args[iarg].str, char, MAX_STRVAL_LEN);

                        args[iarg].nval = 0;
                        STRNCPY(args[iarg].str, str, MAX_STRVAL_LEN);
                    } else {
                        args[iarg].nval = (args[iarg].nrow) * (args[iarg].ncol);

                        FREE(  args[iarg].val);   /* also free's .str since they are unioned */
                        MALLOC(args[iarg].val, double, args[iarg].nval);

                        FREE(  args[iarg].dot);
                        MALLOC(args[iarg].dot, double, args[iarg].nval);

                        SPLINT_CHECK_FOR_NULL(values);
                        SPLINT_CHECK_FOR_NULL(dots  );

                        for (i = 0; i < args[iarg].nval; i++) {
                            args[iarg].val[i] = values[i];
                            args[iarg].dot[i] = dots[  i];
                        }
                    }

                    FREE(values);
                    FREE(dots  );
                }

                SPRINT8(1, "    executing [%4d] ifthen:         %11.5f  %s  %11.5f  %s  %11.5f  %s  %11.5f",
                        ibrch, args[1].val[0], args[2].str, args[3].val[0], args[4].str,
                        args[5].val[0], args[6].str, args[7].val[0]);

                /* evaluate val3 $op3 val4 */
                bool2 = 0;
                if        ((strcmp(args[4].str, "or") == 0 ||
                            strcmp(args[4].str, "OR") == 0   ) && bool1 == 1) {
                    bool2 = 1;
                } else if ((strcmp(args[4].str, "and") == 0 ||
                            strcmp(args[4].str, "AND") == 0   ) && bool1 == 0) {
                    bool2 = 0;
                } else if (strcmp(args[6].str, "lt") == 0 ||
                           strcmp(args[6].str, "LT") == 0   ) {
                    if (args[5].val[0] <  args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "le") == 0 ||
                           strcmp(args[6].str, "LE") == 0   ) {
                    if (args[5].val[0] <= args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "eq") == 0 ||
                           strcmp(args[6].str, "EQ") == 0   ) {
                    if (args[5].val[0] == args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "ge") == 0 ||
                           strcmp(args[6].str, "GE") == 0   ) {
                    if (args[5].val[0] >= args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "gt") == 0 ||
                           strcmp(args[6].str, "GT") == 0   ) {
                    if (args[5].val[0] >  args[7].val[0]) bool2 = 1;
                } else if (strcmp(args[6].str, "ne") == 0 ||
                           strcmp(args[6].str, "NE") == 0   ) {
                    if (args[5].val[0] != args[7].val[0]) bool2 = 1;
                }
            }

            /* evaluate (val1 $op1 val2) $op2 (val3 $op3 val4) */
            bool3 = 0;
            if        (strcmp(args[4].str, "or") == 0 ||
                       strcmp(args[4].str, "OR") == 0   ) {
                bool3 = (bool1 == 1) || (bool2 == 1);
            } else if (strcmp(args[4].str, "and") == 0 ||
                       strcmp(args[4].str, "AND") == 0   ) {
                bool3 = (bool1 == 1) && (bool2 == 1);
            } else if (strcmp(args[4].str, "xor") == 0 ||
                       strcmp(args[4].str, "XOR") == 0   ) {
                bool3 = (bool1 == 1) != (bool2 == 1);
            }

            /* remember that we are in an IFTHEN structure */
            if (npatn < MAX_NESTING) {
                patn[npatn].itype = OCSM_IFTHEN;
                patn[npatn].ibeg  = ibrch;
                patn[npatn].iend  = -1;
                patn[npatn].ncopy = bool3;
                patn[npatn].icopy = -1;
                patn[npatn].ipmtr = -1;
                npatn++;
            } else {
                status = signalError(MODL, OCSM_NESTED_TOO_DEEPLY,
                                     "npatn=%d >0 MAX_NESTING=%d", npatn, MAX_NESTING);
                goto cleanup;        // cannot be caught
            }

            /* if expression evaluated to false, skip to next matching
               ELSEIF, ELSE, or ENDIF */
            if (bool3 == 0) {
                ibrch++;
                icount = 0;   /* tells number of IFTHEN blocks that are within
                                 the one that evaluated to false */
                while (ibrch <= MODL->nbrch) {
                    if        (MODL->brch[ibrch].type == OCSM_IFTHEN) {
                        icount++;
                    } else if (MODL->brch[ibrch].type == OCSM_ELSEIF ||
                               MODL->brch[ibrch].type == OCSM_ELSE     ) {
                        if (icount == 0) {
                            ibrch--;
                            break;
                        }
                    } else if (MODL->brch[ibrch].type == OCSM_ENDIF) {
                        if (icount == 0) {
                            ibrch--;
                            break;
                        } else {
                            icount--;
                        }
                    }
                    ibrch++;
                }
            }

        /* execute: "elseif val1 $op1=ne val2=0 $op2=and val3=0 $op3=eq val4=0" */
        } else if (type == OCSM_ELSEIF) {

            /* if we got here by executing the previous IFTHEN block, skip to
               just before the next matching ENDIF Branch */
            if (patn[npatn-1].ncopy == 1) {
                icount = 0;   /* tells number of IFTHEN blocks that are within
                                 the one that evaluated to false */
                while (ibrch <= MODL->nbrch) {
                    if        (MODL->brch[ibrch].type == OCSM_IFTHEN) {
                        icount++;
                    } else if (MODL->brch[ibrch].type == OCSM_ENDIF) {
                        if (icount == 0) {
                            ibrch--;
                            break;
                        } else {
                            icount--;
                        }
                    }
                    ibrch++;
                }
            } else {

                /* evaluate val1 $op1 val2 */
                bool1 = 0;
                if        (strcmp(args[2].str, "lt") == 0 ||
                           strcmp(args[2].str, "LT") == 0   ) {
                    if (args[1].val[0] <  args[3].val[0]) bool1 = 1;
                } else if (strcmp(args[2].str, "le") == 0 ||
                           strcmp(args[2].str, "LE") == 0   ) {
                    if (args[1].val[0] <= args[3].val[0]) bool1 = 1;
                } else if (strcmp(args[2].str, "eq") == 0 ||
                           strcmp(args[2].str, "EQ") == 0   ) {
                    if (args[1].val[0] == args[3].val[0]) bool1 = 1;
                } else if (strcmp(args[2].str, "ge") == 0 ||
                           strcmp(args[2].str, "GE") == 0   ) {
                    if (args[1].val[0] >= args[3].val[0]) bool1 = 1;
                } else if (strcmp(args[2].str, "gt") == 0 ||
                           strcmp(args[2].str, "GT") == 0   ) {
                    if (args[1].val[0] >  args[3].val[0]) bool1 = 1;
                } else if (strcmp(args[2].str, "ne") == 0 ||
                           strcmp(args[2].str, "NE") == 0   ) {
                    if (args[1].val[0] != args[3].val[0]) bool1 = 1;
                }

                /* set up for lazy evaluations */
                if        (strcmp(args[4].str, "and") == 0 ||
                           strcmp(args[4].str, "AND") == 0   ) {
                    if (bool1 == 0) {
                        bool2  =  0;
                    } else {
                        bool2  = -1;
                    }
                } else if (strcmp(args[4].str, "or") == 0 ||
                           strcmp(args[4].str, "OR") == 0   ) {
                    if (bool1 == 0) {
                        bool2  = -1;
                    } else {
                        bool2  =  1;
                    }
                } else if (strcmp(args[4].str, "xor") == 0 ||
                           strcmp(args[4].str, "XOR") == 0   ) {
                    bool2  = -1;
                } else {
                    bool2 = 0;
                }

                /* no need to evaluate last 3 arguments */
                if (bool2 >= 0) {
                    SPRINT4(1, "    executing [%4d] elseif:         %11.5f  %s  %11.5f  ...",
                            ibrch, args[1].val[0], args[2].str, args[3].val[0]);

                /* evaluate last 3 arguments */
                } else {
                    for (iarg = 5; iarg <= 7; iarg++) {
                        if (iarg == 5) strcpy(thisArg, MODL->brch[ibrch].arg5);
                        if (iarg == 6) strcpy(thisArg, MODL->brch[ibrch].arg6);
                        if (iarg == 7) strcpy(thisArg, MODL->brch[ibrch].arg7);

                        status = str2vals(thisArg, MODL, &(args[iarg].nrow), &(args[iarg].ncol), &values, &dots, str);
                        if (status < SUCCESS) {
                            SPRINT4(0, "ERROR:: problem evaluating argument %d (%s) for Branch %d (%s)",
                                    iarg, thisArg, ibrch, ocsmGetText(MODL->brch[ibrch].type));
                            CATCH_STATUS(str2vals);
                        }

                        if (STRLEN(str) > 0) {
                            FREE(  args[iarg].str);
                            MALLOC(args[iarg].str, char, MAX_STRVAL_LEN);

                            args[iarg].nval = 0;
                            STRNCPY(args[iarg].str, str, MAX_STRVAL_LEN);
                        } else {
                            args[iarg].nval = (args[iarg].nrow) * (args[iarg].ncol);

                            FREE(  args[iarg].val);   /* also free's .str since they are unioned */
                            MALLOC(args[iarg].val, double, args[iarg].nval);

                            FREE(  args[iarg].dot);
                            MALLOC(args[iarg].dot, double, args[iarg].nval);

                            SPLINT_CHECK_FOR_NULL(values);
                            SPLINT_CHECK_FOR_NULL(dots  );

                            for (i = 0; i < args[iarg].nval; i++) {
                                args[iarg].val[i] = values[i];
                                args[iarg].dot[i] = dots[  i];
                            }
                        }

                        FREE(values);
                        FREE(dots  );
                    }

                    SPRINT8(1, "    executing [%4d] elseif:         %11.5f  %s  %11.5f  %s  %11.5f  %s  %11.5f",
                            ibrch, args[1].val[0], args[2].str, args[3].val[0], args[4].str,
                            args[5].val[0], args[6].str, args[7].val[0]);

                    /* evaluate val3 $op3 val4 */
                    bool2 = 0;
                    if        ((strcmp(args[4].str, "or") == 0 ||
                                strcmp(args[4].str, "OR") == 0   ) && bool1 == 1) {
                        bool2 = 1;
                    } else if ((strcmp(args[4].str, "and") == 0 ||
                                strcmp(args[4].str, "AND") == 0   ) && bool1 == 0) {
                        bool2 = 0;
                    } else if (strcmp(args[6].str, "lt") == 0 ||
                               strcmp(args[6].str, "LT") == 0   ) {
                        if (args[5].val[0] <  args[7].val[0]) bool2 = 1;
                    } else if (strcmp(args[6].str, "le") == 0 ||
                               strcmp(args[6].str, "LE") == 0   ) {
                        if (args[5].val[0] <= args[7].val[0]) bool2 = 1;
                    } else if (strcmp(args[6].str, "eq") == 0 ||
                               strcmp(args[6].str, "EQ") == 0   ) {
                        if (args[5].val[0] == args[7].val[0]) bool2 = 1;
                    } else if (strcmp(args[6].str, "ge") == 0 ||
                               strcmp(args[6].str, "GE") == 0   ) {
                        if (args[5].val[0] >= args[7].val[0]) bool2 = 1;
                    } else if (strcmp(args[6].str, "gt") == 0 ||
                               strcmp(args[6].str, "GT") == 0   ) {
                        if (args[5].val[0] >  args[7].val[0]) bool2 = 1;
                    } else if (strcmp(args[6].str, "ne") == 0 ||
                               strcmp(args[6].str, "NE") == 0   ) {
                        if (args[5].val[0] != args[7].val[0]) bool2 = 1;
                    }
                }

                /* evaluate (val1 $op1 val2) $op2 (val3 $op3 val4) */
                bool3 = 0;
                if        (strcmp(args[4].str, "or") == 0 ||
                           strcmp(args[4].str, "OR") == 0   ) {
                    bool3 = (bool1 ==1) || (bool2 == 1);
                } else if (strcmp(args[4].str, "and") == 0 ||
                           strcmp(args[4].str, "AND") == 0   ) {
                    bool3 = (bool1 == 1) && (bool2 == 1);
                } else if (strcmp(args[4].str, "xor") == 0 ||
                           strcmp(args[4].str, "XOR") == 0   ) {
                    bool3 = (bool1 == 1) != (bool2 == 1);
                }

                patn[npatn-1].ncopy = bool3;

                /* if expression evaluated to false, skip to next matching
                   elseif, else, or endif */
                if (bool3 == 0) {
                    ibrch++;
                    icount = 0;
                    while (ibrch <= MODL->nbrch) {
                        if        (MODL->brch[ibrch].type == OCSM_IFTHEN) {
                            icount++;
                        } else if (MODL->brch[ibrch].type == OCSM_ELSEIF ||
                                   MODL->brch[ibrch].type == OCSM_ELSE     ) {
                            if (icount == 0) {
                                ibrch--;
                                break;
                            }
                        } else if (MODL->brch[ibrch].type == OCSM_ENDIF) {
                            if (icount == 0) {
                                ibrch--;
                                break;
                            } else {
                                icount--;
                            }
                        }
                        ibrch++;
                    }
                }
            }

        /* execute: "else" */
        } else if (type == OCSM_ELSE) {

            /* if we got here by executing the previous ifthen block, skip to
               just before the next matching endif Branch */
            if (patn[npatn-1].ncopy == 1) {
                icount = 0;   /* tells number of IFTHEN blocks that are within
                                 the one that evaluated to false */
                while (ibrch <= MODL->nbrch) {
                    if        (MODL->brch[ibrch].type == OCSM_IFTHEN) {
                        icount++;
                    } else if (MODL->brch[ibrch].type == OCSM_ENDIF) {
                        if (icount == 0) {
                            ibrch--;
                            break;
                        } else {
                            icount--;
                        }
                    }
                    ibrch++;
                }
            } else {
                SPRINT1(1, "    executing [%4d] else:",
                        ibrch);
            }

        /* execute: "endif" */
        } else if (type == OCSM_ENDIF) {

            /* remove the patn associated with the ifthen statement */
            npatn--;

        /* execute: "throw sigCode" */
        } else if (type == OCSM_THROW) {
            SPRINT2(1, "    executing [%4d] throw:          %11.5f",
                    ibrch, args[1].val[0]);

            status = signalError(MODL, NINT(args[1].val[0]),
                                 "signal %d thrown by user", NINT(args[1].val[0]));
            goto next_branch;

        /* execute: "catbeg sigCode" */
        } else if (type == OCSM_CATBEG) {
            SPRINT2(1, "    executing [%4d] catbeg:         %s",
                    ibrch, MODL->brch[ibrch].arg1);

            if (MODL->brch[ibrch].arg1[0] != '$') {
                icatch = NINT(args[1].val[0]);
            } else if (strcmp(MODL->brch[ibrch].arg1, "$all"                        ) == 0) {
                icatch = 0;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$body_not_found"             ) == 0) {
                icatch = OCSM_BODY_NOT_FOUND;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$colinear_sketch_points"     ) == 0) {
                icatch = OCSM_COLINEAR_SKETCH_POINTS;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$created_too_many_bodys"     ) == 0) {
                icatch = OCSM_CREATED_TOO_MANY_BODYS;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$did_not_create_body"        ) == 0) {
                icatch = OCSM_DID_NOT_CREATE_BODY;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$edge_not_found"             ) == 0) {
                icatch = OCSM_EDGE_NOT_FOUND;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$error_in_bodys_on_stack"    ) == 0) {
                icatch = OCSM_ERROR_IN_BODYS_ON_STACK;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$face_not_found"             ) == 0) {
                icatch = OCSM_FACE_NOT_FOUND;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$file_not_found"             ) == 0) {
                icatch = OCSM_FILE_NOT_FOUND;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$func_arg_out_of_bounds"     ) == 0) {
                icatch = OCSM_FUNC_ARG_OUT_OF_BOUNDS;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$illegal_argument"           ) == 0) {
                icatch = OCSM_ILLEGAL_ARGUMENT;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$illegal_attribute"          ) == 0) {
                icatch = OCSM_ILLEGAL_ATTRIBUTE;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$illegal_csystem"            ) == 0) {
                icatch = OCSM_ILLEGAL_CSYSTEM;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$illegal_pmtr_index"         ) == 0) {
                icatch = OCSM_ILLEGAL_PMTR_INDEX;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$illegal_pmtr_name"          ) == 0) {
                icatch = OCSM_ILLEGAL_PMTR_NAME;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$illegal_func_name"          ) == 0) {
                icatch = OCSM_ILLEGAL_FUNC_NAME;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$illegal_value"              ) == 0) {
                icatch = OCSM_ILLEGAL_VALUE;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$insufficient_bodys_on_stack") == 0) {
                icatch = OCSM_INSUFFICIENT_BODYS_ON_STACK;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$name_not_found"             ) == 0) {
                icatch = OCSM_NAME_NOT_FOUND;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$node_not_found"             ) == 0) {
                icatch = OCSM_NODE_NOT_FOUND;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$non_coplanar_sketch_points" ) == 0) {
                icatch = OCSM_NON_COPLANAR_SKETCH_POINTS;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$no_selection"               ) == 0) {
                icatch = OCSM_NO_SELECTION;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$underconstrained"           ) == 0) {
                icatch = OCSM_UNDERCONSTRAINED;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$overconstrained"            ) == 0) {
                icatch = OCSM_OVERCONSTRAINED;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$not_converged"              ) == 0) {
                icatch = OCSM_NOT_CONVERGED;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$self_intersecting"          ) == 0) {
                icatch = OCSM_SELF_INTERSECTING;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$wrong_types_on_stack"       ) == 0) {
                icatch = OCSM_WRONG_TYPES_ON_STACK;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$assert_failed"              ) == 0) {
                icatch = OCSM_ASSERT_FAILED;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$udp_error1"                 ) == 0) {
                icatch = OCSM_UDP_ERROR1;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$udp_error2"                 ) == 0) {
                icatch = OCSM_UDP_ERROR2;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$udp_error3"                 ) == 0) {
                icatch = OCSM_UDP_ERROR3;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$udp_error4"                 ) == 0) {
                icatch = OCSM_UDP_ERROR4;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$udp_error5"                 ) == 0) {
                icatch = OCSM_UDP_ERROR5;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$udp_error6"                 ) == 0) {
                icatch = OCSM_UDP_ERROR6;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$udp_error7"                 ) == 0) {
                icatch = OCSM_UDP_ERROR7;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$udp_error8"                 ) == 0) {
                icatch = OCSM_UDP_ERROR8;
            } else if (strcmp(MODL->brch[ibrch].arg1, "$udp_error9"                 ) == 0) {
                icatch = OCSM_UDP_ERROR9;
            } else {
                status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                     "unknown signal type (%s)", MODL->brch[ibrch].arg1);
                goto cleanup;        // cannot be caught
            }

            /* if we are catching any signal and one is set, catch it and reset the signal */
            if (icatch == 0 && MODL->sigCode != 0) {
                SPRINT2(1, "    --> catching signal %d (%s)", MODL->sigCode, ocsmGetText(MODL->sigCode));

                status = ocsmFindPmtr(MODL, "@signal", OCSM_LOCALVAR, 1, 1, &ipmtr);
                CHECK_STATUS(ocsmFindPmtr);

                status = ocsmSetValuD(MODL, ipmtr, 1, 1, (double)(MODL->sigCode));
                CHECK_STATUS(ocsmSetValuD);

                MODL->sigCode    = 0;
                MODL->sigMesg[0] = '\0';
                npatn_sig        = -1;

            /* if there is no uncaught signal, or the uncaught signal
               does not match this catbeg, skip to the matching catend */
            } else if (icatch == 0 || MODL->sigCode == 0 || MODL->sigCode != icatch) {
                icount = 1;
                while (ibrch < MODL->nbrch) {
                    ibrch++;

                    if        (MODL->brch[ibrch].type == OCSM_CATBEG) {
                        icount++;
                    } else if (MODL->brch[ibrch].type == OCSM_CATEND) {
                        icount--;

                        if (icount == 0) {
                            break;
                        }
                    }
                }

            /* otherwise catch this signal by resetting the signal */
            } else {
                SPRINT2(1, "    --> catching signal %d (%s)", MODL->sigCode, ocsmGetText(MODL->sigCode));
                MODL->sigCode    = 0;
                MODL->sigMesg[0] = '\0';
                npatn_sig        = -1;
            }

        /* execute: "catend" */
        } else if (type == OCSM_CATEND) {

        /* execute: "mark" */
        } else if (type == OCSM_MARK) {
            SPRINT1(1, "    executing [%4d] mark:",
                    ibrch);

            /* push a Mark onto the stack */
            stack[nstack++] = 0;

            SPRINT0(1, "                          Mark        created");

        /* execute: "group nbody=0" */
        } else if (type == OCSM_GROUP) {
            SPRINT2(1, "    executing [%4d] group:         %11.5f",
                    ibrch, args[1].val[0]);

            /* make sure that there is a Body on the stack */
            if (nstack < 1) {
                status = signalError(MODL, OCSM_INSUFFICIENT_BODYS_ON_STACK,
                                     "GROUP expects a Body on the stack");
                goto next_branch;
            } else if (stack[nstack-1] <= 0) {
                status = signalError(MODL, OCSM_INSUFFICIENT_BODYS_ON_STACK,
                                     "GROUP expects a Body on the stack");
                goto next_branch;
            }

            /* the number of Bodys specified */
            ngroup = NINT(args[1].val[0]);

            /* if nbody is negative, break up the current group */
            if (ngroup < 0) {

                /* change the group number for all Bodys on stack which are in the same group
                   as the Body on te top of the stack */
                for (istack = nstack-2; istack >= 0; istack--) {
                    if (stack[istack] > 0) {
                        if (MODL->body[stack[istack]].igroup == MODL->body[stack[nstack-1]].igroup) {
                            (MODL->ngroup)++;
                            MODL->body[stack[istack]].igroup = MODL->ngroup;
                        }
                    }
                }

            } else {
                istack = nstack - 1;
                igroup = MODL->body[stack[istack]].igroup;

                /* if ngroup is not given, potentially group all Bodys */
                if (ngroup == 0) {
                    ngroup = MODL->nbody + 1;
                }

                /* ngroup is the number of Bodys that we need to add */
                ngroup--;

                /* loop back through the stack */
                while (istack > 0) {
                    if (ngroup <= 0) break;

                    istack--;

                    /* Sketch found -- error */
                    if (stack[istack] < 0) {
                        status = signalError(MODL, OCSM_WRONG_TYPES_ON_STACK,
                                             "GROUP expects Bodys on the stack");
                        goto next_branch;

                    /* Body found, so change its Group */
                    } else if (stack[istack] > 0) {
                        MODL->body[stack[istack]].igroup = igroup;
                        SPRINT2(1, "                          Body   %4d added to Group %4d",
                                stack[istack], igroup);

                        /* decrement number of Bodys+1 that we still need to add to the Group */
                        ngroup--;

                    /* Mark found, so remove it from the stack */
                    } else {
                        while (istack < nstack) {
                            stack[istack] = stack[istack+1];
                            istack++;
                        }

                        nstack--;
                        break;
                    }
                }
            }

            /* if the group statement has Attributes, apply to all Bodys in the Group */
            if (stack[nstack-1] > 0 && MODL->brch[ibrch].nattr > 0) {
                for (ibody = 1; ibody <= MODL->nbody; ibody++) {
                    if (MODL->body[ibody].igroup != MODL->body[stack[nstack-1]].igroup) continue;
                    ebody = MODL->body[ibody].ebody;

                    status = setEgoAttribute(MODL, ibrch, ebody);
                    CATCH_STATUS(setEgoAttribute);
                }
            }

        /* execute: "dump $filename remove=0 toMark=0 withTess=0 $grpName=. putColors=0" */
        } else if (type == OCSM_DUMP) {
            SPRINT7(1, "    executing [%4d] dump:       %s  %11.5f  %11.5f  %11.5f  %s  %11.5f",
                    ibrch, args[1].str, args[2].val[0], args[3].val[0], args[4].val[0], args[5].str, args[6].val[0]);

            /* make sure that there is a Body on the stack */
            if (nstack < 1) {
                status = signalError(MODL, OCSM_INSUFFICIENT_BODYS_ON_STACK,
                                     "DUMP expects a Body on the stack");
                goto next_branch;
            }

            /* check that an extension (filetype) was given */
            extension = NULL;
            for (i = STRLEN(args[1].str)-1; i > 1; i--) {
                if (args[1].str[i] == '.') {
                    extension = &(args[1].str[i]);
                    break;
                }
            }
            if (extension == NULL) {
                status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                     "no filetype specified");
                goto next_branch;

            /* if the filetype is .stl or .STL, write a stereo-lithography file */
            } else if (strcmp(extension, ".stl") == 0 || strcmp(extension, ".STL") == 0) {

                if (NINT(args[3].val[0]) == 1) {
                    status = writeAsciiStl(MODL, nstack, stack, args[1].str);
                    if (status == OCSM_FILE_NOT_FOUND) {
                        status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                             "DUMP could not open \"%s\" for writing", args[1].str);
                        goto next_branch;
                    }
                    CHECK_STATUS(writeAsciiStl);

                    SPRINT0(1, "WARNING:: multiple Bodys in STL file is non-standard");
                    (MODL->nwarn)++;

                    for (istack = 0; istack < nstack; istack++) {
                        if (stack[istack] > 0) {
                            SPRINT1(1, "                          Body   %4d dumped", stack[istack]);
                        }
                    }
                } else {
                    status = writeAsciiStl(MODL, 1, &(stack[nstack-1]), args[1].str);
                    if (status == OCSM_FILE_NOT_FOUND) {
                        status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                             "DUMP could not open \"%s\" for writing", args[1].str);
                        goto next_branch;
                    }
                    CHECK_STATUS(writeAsciiStl);

                    SPRINT1(1, "                          Body   %4d dumped", stack[nstack-1]);

                    if (NINT(args[2].val[0]) == 1) {
                        SPRINT1(1, "                          Body   %4d removed from stack", stack[nstack-1]);
                        nstack--;
                    }
                }

            /* if the filetype is .bstl or .BSTL, write a binary stereo-lithography file */
            } else if (strcmp(extension, ".bstl") == 0 || strcmp(extension, ".BSTL") == 0) {

                if (NINT(args[3].val[0]) == 1) {
                    status = writeBinaryStl(MODL, nstack, stack, args[1].str);
                    if (status == OCSM_FILE_NOT_FOUND) {
                        status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                             "DUMP could not open \"%s\" for writing", args[1].str);
                        goto next_branch;
                    }
                    CHECK_STATUS(writeBinaryStl);

                    SPRINT0(1, "WARNING:: multiple Bodys in STL file is non-standard");
                    (MODL->nwarn)++;

                    for (istack = 0; istack < nstack; istack++) {
                        if (stack[istack] > 0) {
                            SPRINT1(1, "                          Body   %4d dumped", stack[istack]);
                        }
                    }
                } else {
                    status = writeBinaryStl(MODL, 1, &(stack[nstack-1]), args[1].str);
                    if (status == OCSM_FILE_NOT_FOUND) {
                        status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                             "DUMP could not open \"%s\" for writing", args[1].str);
                        goto next_branch;
                    }
                    CHECK_STATUS(writeBinaryStl);

                    SPRINT1(1, "                          Body   %4d dumped", stack[nstack-1]);

                    if (NINT(args[2].val[0]) == 1) {
                        SPRINT1(1, "                          Body   %4d removed from stack", stack[nstack-1]);
                        nstack--;
                    }
                }

            /* if the filetype is .ugrid or .UGRID, write a AFLR3 input file */
            } else if (strcmp(extension, ".ugrid") == 0 || strcmp(extension, ".UGRID") == 0) {

                if (NINT(args[3].val[0]) == 1) {
                    status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                         "only one Body can be written to .ugrid file");
                    goto next_branch;
                }

                SPRINT1(1, "                          Body   %4d dumped", stack[nstack-1]);
                if (NINT(args[2].val[0]) == 1) {
                    SPRINT1(1, "                          Body   %4d removed from stack", stack[nstack-1]);
                    nstack--;
                }

                /* write the file */
                status = writeAsciiUgrid(MODL, stack[nstack-1], args[1].str);
                if (status == OCSM_FILE_NOT_FOUND) {
                    status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                         "DUMP could not open \"%s\" for writing", args[1].str);
                    goto next_branch;
                }
                CHECK_STATUS(writeAsciiUgrid);

            /* if the filetype is .obj, write a WaveFront .obj file */
            } else if (strcmp(extension, ".obj") == 0) {
                ibody = stack[nstack-1];

                if (MODL->body[ibody].etess == NULL) {
                    status = ocsmTessellate(MODL, ibody);
                    CHECK_STATUS(ocsmTessellate);
                }

                status = writeObjFile(MODL, ibody, args[5].str, args[1].str);
                if (status == OCSM_FILE_NOT_FOUND) {
                    status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                         "DUMP could not open \"%s\" for writing", args[1].str);
                    goto next_branch;
                }
                CHECK_STATUS(writeObjFile);

            /* if the filetype is .vtk, write a .vtk file */
            } else if (strcmp(extension, ".vtk") == 0) {
                ibody = stack[nstack-1];

                if (MODL->body[ibody].etess == NULL) {
                    status = ocsmTessellate(MODL, ibody);
                    CHECK_STATUS(ocsmTessellate);
                }

                status = writeVtkFile(MODL, ibody, args[5].str, args[1].str);
                if (status == OCSM_FILE_NOT_FOUND) {
                    status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                         "DUMP could not open \"%s\" for writing", args[1].str);
                    goto next_branch;
                }
                CHECK_STATUS(writeVtkFile);

            /* if the filetype is .plot, write points into plotfile */
            } else if (strcmp(extension, ".plot") == 0 || strcmp(extension, ".PLOT") == 0) {
                ibody = stack[nstack-1];

                if (MODL->body[ibody].etess == NULL) {
                    status = ocsmTessellate(MODL, ibody);
                    CHECK_STATUS(ocsmTessellate);
                }

                status = writePlotFile(MODL, ibody, args[1].str);
                if (status == OCSM_FILE_NOT_FOUND) {
                    status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                         "DUMP could not open \"%s\" for writing", args[1].str);
                    goto next_branch;
                }
                CHECK_STATUS(writePlotFile);

            /* if the filetype is .tess, write tessellation file */
            } else if (strcmp(extension, ".tess") == 0 || strcmp(extension, ".TESS") == 0) {
                ibody = stack[nstack-1];

                if (MODL->body[ibody].etess == NULL) {
                    status = ocsmTessellate(MODL, ibody);
                    CHECK_STATUS(ocsmTessellate);
                }

                status = writeTessFile(MODL, ibody, args[1].str);
                if (status == OCSM_FILE_NOT_FOUND) {
                    status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                         "DUMP could not open \"%s\" for writing", args[1].str);
                    goto next_branch;
                }
                CHECK_STATUS(writeTessFile);

            /* if the filetype is .sens, write sensitivity file */
            } else if (strcmp(extension, ".sens") == 0 || strcmp(extension, ".SENS") == 0) {
                ibody = stack[nstack-1];

                /* only dump a .sens file if not in process of computing sensitivities
                   via finite differences, since trying to dump the .sens file will
                   result in an infinite loop of recursive calls */
                if (MODL->basemodl == NULL) {

                    if (NINT(args[3].val[0]) != 0) {
                        SPRINT0(1, "WARNING:: only last Body on stack dumped into .sens file");
                        (MODL->nwarn)++;
                    }

                    if (MODL->body[ibody].etess == NULL) {
                        status = ocsmTessellate(MODL, ibody);
                        CHECK_STATUS(ocsmTessellate);
                    }

                    /* propagate the .hasdots to all children -- take care of internal Xsects, such as a RULE or BLEND */
                    for (jbody = 1; jbody <= MODL->nbody; jbody++) {
                        if (MODL->body[jbody].ichld > 0 && MODL->body[jbody].hasdots > 0) {
                            MODL->body[MODL->body[jbody].ichld].hasdots = MODL->body[jbody].hasdots;
                        }
                    }

                    /* propagate the hasdots to all children -- take care of multiple RESTOREs */
                    for (jbody = 1; jbody <= MODL->nbody; jbody++) {
                        ileft = MODL->body[jbody].ileft;
                        irite = MODL->body[jbody].irite;

                        if (ileft > 0) {
                            if (MODL->body[ileft].hasdots > 0) {
                                MODL->body[jbody].hasdots = 1;
                            }
                        }
                        if (irite > 0) {
                            if (MODL->body[irite].hasdots > 0) {
                                MODL->body[jbody].hasdots = 1;
                            }
                        }
                    }

                    status = writeSensFile(MODL, ibody, args[1].str);
                    if (status == OCSM_FILE_NOT_FOUND) {
                        status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                             "DUMP could not open \"%s\" for writing", args[1].str);
                        goto next_branch;
                    }
                    CHECK_STATUS(writeSensFile);
                }

            /* if the filetype is .egg, write a EGG restart file */
            } else if (strcmp(extension, ".egg") == 0 || strcmp(extension, ".EGG") == 0) {
                ibody = stack[nstack-1];

                if (MODL->eggDump == NULL) {
                    SPRINT0(1, "WARNING:: eggDump does not exist.  File not dumped");
                    (MODL->nwarn)++;
                } else {
                    SPRINT1(1, "                          Body   %4d dumped", ibody);
                    if (NINT(args[2].val[0]) == 1) {
                        SPRINT1(1, "                          Body   %4d removed from stack", ibody);
                        nstack--;
                    }

                    /* write the file */
                    fp = fopen(args[1].str, "w");

                    if (fp != NULL) {
                        fprintf(fp, "%8d %8d %8d\n",
                                MODL->body[ibody].nnode,
                                MODL->body[ibody].nedge,
                                MODL->body[ibody].nface);

                        for (iedge = 1; iedge <= MODL->body[ibody].nedge; iedge++) {
                            if (MODL->body[ibody].edge[iedge].itype == DEGENERATE) continue;

                            status = EG_getTessEdge(MODL->body[ibody].etess, iedge,
                                                    &npnt, &xyz, &uv);
                            CHECK_STATUS(EG_getTessEdge);

                            fprintf(fp, "%8d\n", npnt);
                        }

                        for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
                            status = MODL->eggDump(MODL->body[ibody].face[iface].eggdata, fp);
                            CHECK_STATUS(eggDump);
                        }

                        fclose(fp);
                    }
                }

            /* otherwise assume it is a type that EGADS knows how to handle */
            } else {
                nego1 = 0;
                nego2 = 0;

                /* make a model from the Body (and Tessellation) on the top of the stack */
                if (NINT(args[3].val[0]) != 1) {
                    MALLOC(etemp, ego, 2);

                    status = EG_copyObject(MODL->body[stack[nstack-1]].ebody, NULL, &(etemp[nego1]));
                    CATCH_STATUS(EG_copyObject);
                    nego1++;
                    nego2++;

                    if ((NINT(args[6].val[0]) == 1                                            ) &&
                        (strcmp(extension, ".stp" ) == 0 || strcmp(extension, ".STP" ) == 0 ||
                         strcmp(extension, ".step") == 0 || strcmp(extension, ".STEP") == 0 ||
                         strcmp(extension, ".igs" ) == 0 || strcmp(extension, ".IGS" ) == 0 ||
                         strcmp(extension, ".iges") == 0 || strcmp(extension, ".IGES") == 0   )   ) {

                        status = EG_getBodyTopos(etemp[nego1-1], NULL, FACE, &nfaces, &efaces);
                        CHECK_STATUS(EG_getBodyTopos);

                        for (iface = 0; iface < nfaces; iface++) {
                            status = EG_attributeRet(efaces[iface], "_color",
                                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
                            if (status == EGADS_SUCCESS) {
                                status = EG_attributeAdd(efaces[iface], "Color",
                                                         attrType, attrLen, tempIlist, tempRlist, tempClist);
                                CHECK_STATUS(EG_attributeAdd);

                                SPRINT1(2, "        transferring color for iface=%d", iface+1);
                            }
                        }

                        EG_free(efaces);    efaces = NULL;

                        status = EG_getBodyTopos(etemp[nego1-1], NULL, EDGE, &nedges, &eedges);
                        CHECK_STATUS(EG_getBodyTopos);

                        for (iedge = 0; iedge < nedges; iedge++) {
                            status = EG_attributeRet(eedges[iedge], "_color",
                                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
                            if (status == EGADS_SUCCESS) {
                                status = EG_attributeAdd(eedges[iedge], "Color",
                                                         attrType, attrLen, tempIlist, tempRlist, tempClist);
                                CHECK_STATUS(EG_attributeAdd);

                                SPRINT1(2, "        transferring color for iedge=%d", iedge+1);
                            }
                        }

                        EG_free(eedges);    eedges = NULL;
                    }

                    if ((NINT(args[4].val[0]) != 0                                           ) &&
                        (strcmp(extension, ".egads") == 0 || strcmp(extension, ".EGADS") == 0)   ) {
                        if (MODL->body[stack[nstack-1]].etess == NULL) {
                            status = ocsmTessellate(MODL, stack[nstack-1]);
                            CATCH_STATUS(ocsmTessellate);
                        }

                        status = EG_copyObject(MODL->body[stack[nstack-1]].etess,
                                               etemp[0], &(etemp[nego2]));
                        CATCH_STATUS(EG_copyObject);

                        nego2++;
                    }

                    SPRINT1(1, "                          Body   %4d dumped", stack[nstack-1]);
                    if (NINT(args[2].val[0]) == 1) {
                        SPRINT1(1, "                          Body   %4d removed from stack", stack[nstack-1]);
                        nstack--;
                    }

                /* make a model from all the Bodys (and Tessellations) back to the Mark */
                } else {
                    MALLOC(etemp, ego, 2*(nstack+1));

                    for (jstack = nstack-1; jstack >= 0; jstack--) {
                        if (stack[jstack] == 0) {
                            break;
                        } else if (stack[jstack] > 0) {
                            status = EG_copyObject(MODL->body[stack[jstack]].ebody, NULL, &(etemp[nego1]));
                            CATCH_STATUS(EG_copyObject);
                            nego1++;
                            nego2++;

                            if ((NINT(args[6].val[0]) == 1                                            ) &&
                                (strcmp(extension, ".stp" ) == 0 || strcmp(extension, ".STP" ) == 0 ||
                                 strcmp(extension, ".step") == 0 || strcmp(extension, ".STEP") == 0 ||
                                 strcmp(extension, ".igs" ) == 0 || strcmp(extension, ".IGS" ) == 0 ||
                                 strcmp(extension, ".iges") == 0 || strcmp(extension, ".IGES") == 0   )   ) {

                                status = EG_getBodyTopos(etemp[nego1-1], NULL, FACE, &nfaces, &efaces);
                                CHECK_STATUS(EG_getBodyTopos);

                                for (iface = 0; iface < nfaces; iface++) {
                                    status = EG_attributeRet(efaces[iface], "_color",
                                                             &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
                                    if (status == EGADS_SUCCESS) {
                                        status = EG_attributeAdd(efaces[iface], "Color",
                                                                 attrType, attrLen, tempIlist, tempRlist, tempClist);
                                        CHECK_STATUS(EG_attributeAdd);

                                        SPRINT1(2, "        transferring color for iface=%d", iface+1);
                                    }
                                }

                                EG_free(efaces);    efaces = NULL;

                                status = EG_getBodyTopos(etemp[nego1-1], NULL, EDGE, &nedges, &eedges);
                                CHECK_STATUS(EG_getBodyTopos);

                                for (iedge = 0; iedge < nedges; iedge++) {
                                    status = EG_attributeRet(eedges[iedge], "_color",
                                                             &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
                                    if (status == EGADS_SUCCESS) {
                                        status = EG_attributeAdd(eedges[iedge], "Color",
                                                                 attrType, attrLen, tempIlist, tempRlist, tempClist);
                                        CHECK_STATUS(EG_attributeAdd);

                                        SPRINT1(2, "        transferring color for iedge=%d", iedge+1);
                                    }
                                }

                                EG_free(eedges);    eedges = NULL;
                            }

                            SPRINT1(1, "                          Body   %4d dumped", stack[jstack]);
                        }
                    }

                    if (NINT(args[4].val[0]) != 0 && (strcmp(extension, ".egads") == 0 || strcmp(extension, ".EGADS") == 0)) {
                        for (jstack = nstack-1; jstack >= 0; jstack--) {
                            if (stack[jstack] == 0) {
                                break;
                            } else if (stack[jstack] > 0) {
                                if (MODL->body[stack[jstack]].etess == NULL) {
                                    status = ocsmTessellate(MODL, stack[jstack]);
                                    CATCH_STATUS(ocsmTessellate);
                                }

                                status = EG_copyObject(MODL->body[stack[jstack]].etess,
                                                       etemp[nstack-1-jstack], &(etemp[nego2]));
                                CATCH_STATUS(EG_copyObject);
                                nego2++;
                            }
                        }
                    }

                    for (jstack = nstack-1; jstack >= 0; jstack--) {
                        if (stack[jstack] == 0) {
                            if (NINT(args[2].val[0]) == 1) {
                                SPRINT0(1, "                          Mark        removed from stack");
                                nstack--;
                            }
                            break;
                        } else if (stack[jstack] > 0) {
                            if (NINT(args[2].val[0]) == 1) {
                                SPRINT1(1, "                          Body   %4d removed from stack", stack[jstack]);
                                nstack--;
                            }
                        }
                    }
                }

                status = EG_makeTopology(MODL->context, NULL, MODEL, nego2,
                                         NULL, nego1, etemp, NULL, &emodel);
                CATCH_STATUS(EG_makeTopology);

                FREE(etemp);

                SPLINT_CHECK_FOR_NULL(emodel);

                /* add the global Attributes to the Model */
                for (iattr = 0; iattr < MODL->nattr; iattr++) {
                    status = EG_attributeAdd(emodel, MODL->attr[iattr].name, ATTRSTRING,
                                             STRLEN(MODL->attr[iattr].defn), NULL, NULL, MODL->attr[iattr].defn);
                    CATCH_STATUS(EG_attributeAdd);
                }

                /* add the DESPMTRs, CFGPMTRs, and OUTPMTRs as Attributes on the Model */
                for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
                    if (MODL->pmtr[ipmtr].type == OCSM_DESPMTR)  {
                        snprintf(pmtrName, MAX_EXPR_LEN, "_despmtr_%s", MODL->pmtr[ipmtr].name);

                        status = EG_attributeAdd(emodel, pmtrName, ATTRREAL,
                                                 MODL->pmtr[ipmtr].nrow*MODL->pmtr[ipmtr].ncol,
                                                 NULL, MODL->pmtr[ipmtr].value, NULL);
                        CATCH_STATUS(EG_attributeAdd);
                    } else if (MODL->pmtr[ipmtr].type == OCSM_CFGPMTR)  {
                        snprintf(pmtrName, MAX_EXPR_LEN, "_cfgpmtr_%s", MODL->pmtr[ipmtr].name);

                        status = EG_attributeAdd(emodel, pmtrName, ATTRREAL,
                                                 MODL->pmtr[ipmtr].nrow*MODL->pmtr[ipmtr].ncol,
                                                 NULL, MODL->pmtr[ipmtr].value, NULL);
                        CATCH_STATUS(EG_attributeAdd);
                    } else if (MODL->pmtr[ipmtr].type == OCSM_OUTPMTR) {
                        snprintf(pmtrName, MAX_EXPR_LEN, "_outpmtr_%s", MODL->pmtr[ipmtr].name);

                        if (MODL->pmtr[ipmtr].value != NULL) {
                            status = EG_attributeAdd(emodel, pmtrName, ATTRREAL,
                                                     MODL->pmtr[ipmtr].nrow*MODL->pmtr[ipmtr].ncol,
                                                     NULL, MODL->pmtr[ipmtr].value, NULL);
                        } else {
                            status = EG_attributeAdd(emodel, pmtrName, ATTRSTRING,
                                                     MODL->pmtr[ipmtr].nrow,
                                                     NULL, NULL, MODL->pmtr[ipmtr].str);
                        }
                        CATCH_STATUS(EG_attributeAdd);
                    }
                }

                if (outLevel >= 3) {
                    SPRINT0(3, "emodel\n");
                    ocsmPrintEgo(emodel);
                }

                /* strip the dollarsign off the filename */
                STRNCPY(dumpfile, args[1].str, MAX_EXPR_LEN);

                /* if file exists, delete it now */
                status = remove(dumpfile);
                if (status == 0) {
                    SPRINT1(1, "WARNING:: file \"%s\" is being overwritten", dumpfile);
                    (MODL->nwarn)++;
                }

                status = EG_saveModel(emodel, dumpfile);
                if (status == EGADS_WRITERR) {
                    status = signalError(MODL, EGADS_WRITERR,
                                         "DUMP could not open \"%s\" for writing", dumpfile);
                    goto next_branch;
                }
                CATCH_STATUS(EG_saveModel);

                status = EG_deleteObject(emodel);
                CATCH_STATUS(EG_deleteObject);
            }

        /* execute: "end" */
        } else if (type == OCSM_END) {
            SPRINT1(1, "    executing [%4d] end:", ibrch);

            /* do nothing if we are at the top level */
            if (MODL->level <= 0) break;

            count = 0;
            SPRINT0x(1, "<");
            for (i = 0; i < MODL->level; i++) {
                SPRINT0x(1, "~~~~");
            }
            for (i = ibrch; i >= 0; i--) {
                if (MODL->brch[i].type == OCSM_END) {
                    count++;
                } else if ( MODL->brch[i].type == OCSM_UDPRIM &&
                           (MODL->brch[i].arg1[1] == '$' ||
                            MODL->brch[i].arg1[1] == '/'   )    ) {
                    count--;
                    if (count == 0) {
                        SPRINT2(1, " exit  %s (from line %d)", &(MODL->brch[i].arg1[1]), i);
                        break;
                    }
                }
            }

            /* remove from the pattern list */
            npatn--;

            /* find the UDPRIM Branch associated with this END */
            for (jbrch = ibrch-1; jbrch > 0; jbrch--) {
                if (MODL->brch[jbrch].indent == MODL->brch[ibrch].indent) {
                    if (MODL->brch[jbrch].type == OCSM_UDPRIM) {
                        break;
                    } else {
                        status = signalError(MODL, OCSM_INTERNAL_ERROR,
                                             "expecting UDPRIM, but got %s", ocsmGetText(MODL->brch[jbrch].type));
                        goto cleanup;        // cannot be caught
                    }
                }
            }

            /* jbrch is first Branch after the UDPRIM statement that called this .udc */
            jbrch++;
            while (jbrch < ibrch) {
                if (MODL->brch[jbrch].type != OCSM_INTERFACE) break;

                if (strcmp(MODL->brch[jbrch].arg2, "$out") == 0 ||
                    strcmp(MODL->brch[jbrch].arg2, "$OUT") == 0   ) {

                    /* get the value of the local variable */
                    status = str2vals(&(MODL->brch[jbrch].arg1[1]), MODL, &nrow, &ncol, &values, &dots, str);
                    CATCH_STATUS(str2vals);

                    /* find or create an @@-parameter up one level and store the value/dot */
                    (MODL->level)--;
                    snprintf(pmtrName, MAX_EXPR_LEN, "@@%s", &(MODL->brch[jbrch].arg1[1]));

                    if (STRLEN(str) == 0) {
                        SPLINT_CHECK_FOR_NULL(values);
                        SPLINT_CHECK_FOR_NULL(dots  );

                        status = ocsmFindPmtr(MODL, pmtrName, OCSM_LOCALVAR, nrow, ncol, &ipmtr);

                        /* we may need to increase the size of this @@-parameter */
                        if (status == OCSM_NAME_ALREADY_DEFINED) {
                            status = ocsmDelPmtr(MODL, ipmtr);
                            CHECK_STATUS(ocsmDelPmtr);

                            status = ocsmFindPmtr(MODL, pmtrName, OCSM_LOCALVAR, nrow, ncol, &ipmtr);
                        }
                        CATCH_STATUS(ocsmFindPmtr);

                        i = 0;
                        for (irow = 1; irow <= nrow; irow++) {
                            for (icol = 1; icol <= ncol; icol++) {
                                status = ocsmSetValuD(MODL, ipmtr, irow, icol, values[i]);
                                CATCH_STATUS(ocsmSetValuD);

                                status = ocsmSetVelD(MODL, ipmtr, irow, icol, dots[i]);
                                CATCH_STATUS(ocsmSetVelD);

                                i++;
                            }
                        }
                    } else {
                        status = ocsmFindPmtr(MODL, pmtrName, OCSM_LOCALVAR, 0, 0, &ipmtr);
                        CATCH_STATUS(ocsmFindPmtr);

                        STRNCPY(temp, "$", MAX_STRVAL_LEN  );
                        STRNCAT(temp, str, MAX_STRVAL_LEN-1);
                        status = ocsmSetValu(MODL, ipmtr, 1, 1, temp);
                        CATCH_STATUS(ocsmSetValu);
                    }


                    FREE(values);
                    FREE(dots  );

                    (MODL->level)++;
                }

                jbrch++;
            }

            /* remove all the Parameters at MODL->scope if a new scope */
            if (MODL->scope[MODL->level] > MODL->scope[MODL->level-1]) {
                for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
                    if (MODL->pmtr[ipmtr].scope >= MODL->scope[MODL->level]) {
                        FREE(MODL->pmtr[ipmtr].name );
                        FREE(MODL->pmtr[ipmtr].value);
                        FREE(MODL->pmtr[ipmtr].dot  );
                        FREE(MODL->pmtr[ipmtr].lbnd );
                        FREE(MODL->pmtr[ipmtr].ubnd );
                        FREE(MODL->pmtr[ipmtr].str  );

                        for (jpmtr = ipmtr+1; jpmtr <= MODL->npmtr; jpmtr++) {

                            /* copy jpmtr over jpmtr-1 */
                            MODL->pmtr[jpmtr-1].name  = MODL->pmtr[jpmtr].name;
                            MODL->pmtr[jpmtr-1].type  = MODL->pmtr[jpmtr].type;
                            MODL->pmtr[jpmtr-1].scope = MODL->pmtr[jpmtr].scope;
                            MODL->pmtr[jpmtr-1].mprop = MODL->pmtr[jpmtr].mprop;
                            MODL->pmtr[jpmtr-1].nrow  = MODL->pmtr[jpmtr].nrow;
                            MODL->pmtr[jpmtr-1].ncol  = MODL->pmtr[jpmtr].ncol;
                            MODL->pmtr[jpmtr-1].value = MODL->pmtr[jpmtr].value;
                            MODL->pmtr[jpmtr-1].dot   = MODL->pmtr[jpmtr].dot;
                            MODL->pmtr[jpmtr-1].lbnd  = MODL->pmtr[jpmtr].lbnd;
                            MODL->pmtr[jpmtr-1].ubnd  = MODL->pmtr[jpmtr].ubnd;
                            MODL->pmtr[jpmtr-1].str   = MODL->pmtr[jpmtr].str;
                        }

                        ipmtr--;            /* revisit imptr in next trip through loop */

                        MODL->pmtr[MODL->npmtr].name  = NULL;
                        MODL->pmtr[MODL->npmtr].value = NULL;
                        MODL->pmtr[MODL->npmtr].dot   = NULL;
                        MODL->pmtr[MODL->npmtr].lbnd  = NULL;
                        MODL->pmtr[MODL->npmtr].ubnd  = NULL;
                        MODL->pmtr[MODL->npmtr].str   = NULL;
                        (MODL->npmtr)--;
                    }
                }

            }

            /* decrement the level */
            (MODL->level)--;

        /* execute: "assert arg1 arg2 toler=0 verify=0" */
        } else if (type == OCSM_ASSERT) {

            /* determine if verification is required */
            verify = NINT(args[4].val[0]);

            if (verify <= MODL->verify) {
                SPRINT5(1, "    executing [%4d] assert:     %s  %s  %s  %s",
                        ibrch, MODL->brch[ibrch].arg1,
                               MODL->brch[ibrch].arg2,
                               MODL->brch[ibrch].arg3,
                               MODL->brch[ibrch].arg4);

                /* set up tolerance */
                if (args[3].val[0] == 0) {
                    toler = 1.0e-6;
                } else if (args[3].val[0] < 0) {
                    toler = fabs(args[1].val[0] * args[3].val[0]);
                } else {
                    toler = args[3].val[0];
                }

                /* return error if arg1 != arg2 */
                if (fabs(args[1].val[0]-args[2].val[0]) > toler) {
                    SPRINT1(0, "    arg1 = %20.8f", args[1].val[0]);
                    SPRINT1(0, "    arg2 = %20.8f", args[2].val[0]);
                    if (args[3].val[0] < 0) {
                        SPRINT1(0, " pct err = %20.8f", 100*(args[1].val[0]-args[2].val[0])/args[1].val[0]);
                    } else {
                        SPRINT1(0, " abs err = %20.8f",     (args[1].val[0]-args[2].val[0])               );
                    }
                    status = signalError(MODL, OCSM_ASSERT_FAILED,
                                         "ASSERT failed (%12.5e disagrees with %12.5e)", args[1].val[0], args[2].val[0]);
                    goto next_branch;
                }
            } else {
                SPRINT5(1, "    ignoring  [%4d] assert:     %s  %s  %s  %s",
                        ibrch, MODL->brch[ibrch].arg1,
                               MODL->brch[ibrch].arg2,
                               MODL->brch[ibrch].arg3,
                               MODL->brch[ibrch].arg4);
            }

        /* execute: "message $text $schar=_ $fileName=. $openType=a" */
        } else if (type == OCSM_MESSAGE) {
            SPRINT5(1, "    executing [%4d] message:    %s  %s  %s  %s",
                    ibrch, &(MODL->brch[ibrch].arg1[1]),
                           &(MODL->brch[ibrch].arg2[1]),
                           &(MODL->brch[ibrch].arg3[1]),
                           &(MODL->brch[ibrch].arg4[1]));

            if (args[1].nval != 0) {
                status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                     "MESSAGE text is expected to be a string");
                goto next_branch;
            } else if (args[2].nval != 0) {
                status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                     "MESSAGE schar is expected to be a string");
                goto next_branch;
            } else if (strlen(args[2].str) != 1) {
                status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                     "MESSAGE schar is expected to be a single character");
                goto next_branch;
            } else if (args[3].nval != 0) {
                status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                     "MESSAGE fileName is expected to be a string");
                goto next_branch;
            } else if (args[4].nval != 0) {
                status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                     "MESSAGE openType is expected to be a string");
                goto next_branch;
            }

            /* reuse the previous string */
            STRNCPY(thisArg, args[1].str, MAX_LINE_LEN);

            for (i = 0; i < strlen(thisArg); i++) {
                if (thisArg[i] == args[2].str[0]) {
                    thisArg[i] = ' ';
                }
            }

            /* return message to GUI */
            if (strcmp(args[3].str, ".") == 0) {
                SPRINT1(0, "\nMESSAGE:: %s\n", thisArg);

                /* send this message to the message callback routine (if defined) */
                if (MODL->mesgCB != NULL) {
                    MODL->mesgCB(thisArg);
                }

            /* write message to a file */
            } else {
                if        (strcmp(args[4].str, "a") == 0 || strcmp(args[4].str, "A") == 0) {
                    fp = fopen(args[3].str, "a");
                } else if (strcmp(args[4].str, "n") == 0 || strcmp(args[4].str, "N") == 0) {
                    fp = fopen(args[3].str, "w");
                } else {
                    status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                         "MESSAGE openType should be \"n\" or \"a\"");
                    goto next_branch;
                }

                if (fp == NULL) {
                    status = signalError(MODL, OCSM_FILE_NOT_FOUND,
                                         "MESSAGE could not open file");
                    goto next_branch;
                }

                fprintf(fp, "%s\n", thisArg);
                fclose(fp);
            }

        /* execute: "special $arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 arg9" */
        } else if (type == OCSM_SPECIAL) {
            SPRINT10(1, "    executing [%4d] special:    %s  %s  %s  %s  %s  %s  %s  %s  %s",
                    ibrch, MODL->brch[ibrch].arg1,
                           MODL->brch[ibrch].arg2,
                           MODL->brch[ibrch].arg3,
                           MODL->brch[ibrch].arg4,
                           MODL->brch[ibrch].arg5,
                           MODL->brch[ibrch].arg6,
                           MODL->brch[ibrch].arg7,
                           MODL->brch[ibrch].arg8,
                           MODL->brch[ibrch].arg9);

            /* "$printCPs" */
            if (strcmp(MODL->brch[ibrch].arg1, "$printCPs") == 0) {

                status = printNurbs(MODL->body[MODL->nbody].ebody);
                CHECK_STATUS(printNurbs);

            /* "$clearance ibody1 ibody2" */
            } else if (strcmp(MODL->brch[ibrch].arg1, "$clearance") == 0) {
                int    sense;
                double trange[2];
                ego    enodes2[2], ecurve, eloop;

                ibody1 = NINT(args[2].val[0]);
                ibody2 = NINT(args[3].val[0]);

                status = ocsmClearance(MODL, ibody1, ibody2, &dist, pnt1, pnt2);
                CHECK_STATUS(ocsmClearance);

                SPRINT1(1, "clearance -> dist=%12.6f", dist);

                status = ocsmFindPmtr(MODL, "@sensitivity", OCSM_LOCALVAR, 1, 1, &ipmtr);
                CHECK_STATUS(ocsmFindPmtr);

                status = ocsmSetValuD(MODL, ipmtr, 1, 1, dist);
                CHECK_STATUS(ocsmSetValuD);

                if (fabs(pnt1[0]) > EPS12 && fabs(pnt2[0]) > EPS12) {
                    status = EG_evaluate(MODL->body[abs(ibody1)].face[NINT(pnt1[0])].eface, &pnt1[1], data1);
                    CHECK_STATUS(EG_evaluate);

                    status = EG_evaluate(MODL->body[abs(ibody2)].face[NINT(pnt2[0])].eface, &pnt2[1], data2);
                    CHECK_STATUS(EG_evaluate);

                    SPRINT7(1, "      ibody1=%3d, pnt1=%3d %12.6f %12.6f, xyz1=%12.6f %12.6f %12.6f",
                            ibody1, NINT(pnt1[0]), pnt1[1], pnt1[2], data1[0], data1[1], data1[2]);
                    SPRINT7(1, "      ibody2=%3d, pnt2=%3d %12.6f %12.6f, xyz2=%12.6f %12.6f %12.6f",
                            ibody2, NINT(pnt2[0]), pnt2[1], pnt2[2], data2[0], data2[1], data2[2]);

                    hasdots = 0;

                    /* recycle old Body(s) if not dirty */
                    status = recycleBody(MODL, ibrch, type, args, hasdots);
                    CHECK_STATUS(recycleBody);

                    if (status > 0) {
                        for (jbody = 0; jbody < status; jbody++) {
                            stack[nstack++] = MODL->nbody - jbody;
                        }
                        status = SUCCESS;
                        goto next_branch;
                    }

                    ibody1 = abs(ibody1);

                    /* make a WireBody between the two "closest" points */
                    data1[3] = data2[0] - data1[0];
                    data1[4] = data2[1] - data1[1];
                    data1[5] = data2[2] - data1[2];

                    trange[0] = 0;
                    trange[1] = sqrt(data1[3]*data1[3] + data1[4]*data1[4] + data1[5]*data1[5]);

                    if (trange[1] > EPS06) {
                        status = EG_makeTopology(MODL->context, NULL, NODE, 0, data1, 0, NULL, NULL, &enodes2[0]);
                        CHECK_STATUS(EG_makeTopology);

                        status = EG_makeTopology(MODL->context, NULL, NODE, 0, data2, 0, NULL, NULL, &enodes2[1]);
                        CHECK_STATUS(EG_makeTopology);

                        status = EG_makeGeometry(MODL->context, CURVE, LINE, NULL, NULL, data1, &ecurve);
                        CHECK_STATUS(EG_makeGeometry);

                        status = EG_makeTopology(MODL->context, ecurve, EDGE, TWONODE, trange, 2, enodes2, NULL, &eedge);
                        CHECK_STATUS(EG_makeTopology);

                        sense = SFORWARD;

                        status = EG_makeTopology(MODL->context, NULL, LOOP, OPEN, NULL, 1, &eedge, &sense, &eloop);
                        CHECK_STATUS(EG_makeTopology);

                        status = EG_makeTopology(MODL->context, NULL, BODY, WIREBODY, NULL, 1, &eloop, NULL, &ebody);
                        CHECK_STATUS(EG_makeTopology);

                        status = newBody(MODL, ibrch, type, abs(ibody1), ibody2,
                                     args, hasdots, OCSM_WIRE_BODY, &ibody);
                        CHECK_STATUS(newBody);
                    } else {
                        status = makeNodeBody(MODL, data1, &ebody);
                        CHECK_STATUS(makeNodeBody);

                        status = newBody(MODL, ibrch, type, abs(ibody1), ibody2,
                                     args, hasdots, OCSM_NODE_BODY, &ibody);
                        CHECK_STATUS(newBody);
                    }

                    MODL->body[ibody].ebody = ebody;

                    /* update @-parameters and finish Body (SPECIAL) */
                    status = setupAtPmtrs(MODL, 0);
                    CHECK_STATUS(setupAtPmtrs);

                    status = finishBody(MODL, ibody);
                    if (MODL->sigCode != SUCCESS) goto cleanup;        // cannot be caught
                    CHECK_STATUS(finishBody);

                    /* push the Body onto the stack */
                    stack[nstack++] = ibody;

                    status = getBodyTolerance(MODL->body[ibody].ebody, &toler);
                    CHECK_STATUS(getBodyTolerance);

                    SPRINT5(1, "                          Body   %4d created  (toler=%11.4e, nnode=%4d, nedge=%4d, nface=%4d)",
                            ibody, toler, MODL->body[ibody].nnode, MODL->body[ibody].nedge, MODL->body[ibody].nface);
                } else {
                    SPRINT0(1, "                       no Body created");
                }

            /* "$tracePmtrs pattern" */
            } else if (strcmp(MODL->brch[ibrch].arg1, "$tracePmtrs") == 0) {
                char *info=NULL;

                if (args[2].nval != 0) {
                    status = signalError(MODL, OCSM_ILLEGAL_ARGUMENT,
                                         "pattern to be a string");
                    goto next_branch;
                }

                status = ocsmTracePmtrs(MODL, args[2].str, &info);
                CHECK_STATUS(ocsmTracePmtrs);

                FREE(info);

            /* "$provides dim1 ...: */
            } else if (strcmp(MODL->brch[ibrch].arg1, "$provides") == 0) {

                /* do nothing --- just  marker */

            /* "$edgeDist ibody1 ibody2 toler" */
            } else if (strcmp(MODL->brch[ibrch].arg1, "$edgeDist") == 0) {
                int    iedge1, iedge2;
                double trange1[2], t1, t2, dmax;

                ibody1 = NINT(args[2].val[0]);
                ibody2 = NINT(args[3].val[0]);
                toler  =      args[4].val[0];

                if        (ibody1 < 1 || ibody1 > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "ibody1 (%d) is out of range", ibody1);
                    goto next_branch;
                } else if (ibody2 < 1 || ibody2 > MODL->nbody) {
                    status = signalError(MODL, OCSM_BODY_NOT_FOUND,
                                         "ibody2 (%d) is out of range", ibody2);
                    goto next_branch;
                } else if (toler <= 0) {
                    status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                         "toler (%12.6e) musst be positive", toler);
                    goto next_branch;
                }

                /* find the maximum distance (discretely) from 101 points on
                   ibody1:iedge1 to any point on ibody2:iedge2 */
                for (iedge1 = 1; iedge1 <= MODL->body[ibody1].nedge; iedge1++) {
                    if (MODL->body[ibody1].edge[iedge1].itype == DEGENERATE) continue;

                    status = EG_getRange(MODL->body[ibody1].edge[iedge1].eedge, trange1, &periodic);
                    CHECK_STATUS(EG_getRange);

                    for (iedge2 = 1; iedge2 <= MODL->body[ibody2].nedge; iedge2++) {
                        if (MODL->body[ibody2].edge[iedge2].itype == DEGENERATE) continue;

                        dmax = 0;

                        for (ipnt = 0; ipnt < 101; ipnt++) {
                            t1 = trange1[0] + (trange1[1] - trange1[0]) * (double)ipnt / 100.;

                            status = EG_evaluate(MODL->body[ibody1].edge[iedge1].eedge, &t1, data1);
                            CHECK_STATUS(EG_evaluate);

                            status = EG_invEvaluate(MODL->body[ibody2]. edge[iedge2].eedge, data1, &t2, data2);
                            CHECK_STATUS(EG_invEvaluate);

                            dist = (data2[0] - data1[0]) * (data2[0] - data1[0])
                                +  (data2[1] - data1[1]) * (data2[1] - data1[1])
                                +  (data2[2] - data1[2]) * (data2[2] - data1[2]);
                            if (dist > dmax) dmax = dist;
                        }

                        if (sqrt(dmax) < toler) {
                            SPRINT5(1, "Edge1=%5d:%-5d  Edge2=%5d:%-5d  dmax=%12.6e",
                                    ibody1, iedge1, ibody2, iedge2, sqrt(dmax));
                        }
                    }
                }

            } else if (strcmp(MODL->brch[ibrch].arg1, "$egoLifetime") == 0) {
                /*@-shadow@*/
                int    senses;
                double data[6], trange[2], params[3];
                ego    ecurve1, enodes[2], eedge1, eloop1, ebody0, ebody1, ebody2[2], ebody3[2], emodel1;
                /*@+shadow@*/

                printEgoList(MODL->context);      // nego=0

                /* make a WireBody (ebody0) */
                data[0] = data[1] = data[2] = 0;
                data[3] = data[4] = data[5] = 1;
                status = EG_makeGeometry(MODL->context, CURVE, LINE, NULL, NULL, data, &ecurve1);
                SPRINT2(1, "EG_makeGeometry -> status=%d, eurve=%llx", status, (long long)ecurve1);
                printEgoList(MODL->context);      // nego=2 (+2)

                status = EG_makeTopology(MODL->context, NULL, NODE, 0, &data[0], 0, NULL, NULL, &enodes[0]);
                SPRINT2(1, "EG_makeTopology(enodes[0]) -> status=%d, enodes[0]=%llx", status, (long long)enodes[0]);
                printEgoList(MODL->context);      // nego=4 (+2)

                status = EG_makeTopology(MODL->context, NULL, NODE, 0, &data[3], 0, NULL, NULL, &enodes[1]);
                SPRINT2(1, "EG_makeTopology(enodes[1]) -> status=%d, enodes[1]=%llx", status, (long long)enodes[1]);
                printEgoList(MODL->context);      // nego=6 (+2)

                trange[0] = 0;
                trange[1] = sqrt(3);
                status = EG_makeTopology(MODL->context, ecurve1, EDGE, TWONODE, trange, 2, enodes, NULL, &eedge1);
                SPRINT2(1, "EG_makeTopology(eedge1) -> status=%d, eedge1=%llx", status, (long long)eedge1);
                printEgoList(MODL->context);      // nego=11 (+5)

                senses = SFORWARD;
                status = EG_makeTopology(MODL->context, NULL, LOOP, OPEN, NULL, 1, &eedge1, &senses, &eloop1);
                SPRINT2(1, "EG_makeTopology(eloop1) -> status=%d, eloop1=%llx", status, (long long)eloop1);
                printEgoList(MODL->context);      // nego=14 (+3)

                /* note that this makes copies of the unattached egos */
                status = EG_makeTopology(MODL->context, NULL, BODY, WIREBODY, NULL, 1, &eloop1, NULL, &ebody0);
                SPRINT2(1, "EG_makeTopology(ebody0) -> status=%d, ebody0=%llx", status, (long long)ebody0);
                printEgoList(MODL->context);      // nego=26 (+12)

                /* make a SolidBody (ebody1) */
                status = EG_makeSolidBody(MODL->context, BOX, data, &ebody1);
                SPRINT2(1, "EG_makeSolidBody -> status=%d, ebody1=%llx", status, (long long)ebody1);
                printEgoList(MODL->context);      // nego=158 (+132)

                /* copy ebody1 to ebody2[0] */
                status = EG_copyObject(ebody1, NULL, &ebody2[0]);
                SPRINT2(1, "EG_copyObject -> status=%d, ebody2[0]=%llx", status, (long long)ebody2[0]);
                printEgoList(MODL->context);      // nego=290 (+132)

                /* tessellate ebody2[0] (ebody2[1]) */
                params[0] = 0.1;
                params[1] = 15;
                params[2] = 0.1;
                status = EG_makeTessBody(ebody2[0], params, &ebody2[1]);
                SPRINT2(1, "EG_makeTessBody -> status=%d, ebody2[1]=%llx", status, (long long)ebody2[1]);
                printEgoList(MODL->context);      // nego=293 (+3)

                /* copy ebody1 to ebody3[0] */
                status = EG_copyObject(ebody1, NULL, &ebody3[0]);
                SPRINT2(1, "EG_copyObject -> status=%d, ebody3[0]=%llx", status, (long long)ebody3[0]);
                printEgoList(MODL->context);      // nego=525 (+132)

                /* tessellate ebody3[0] (ebody3[1]) */
                status = EG_makeTessBody(ebody3[0], params, &ebody3[1]);
                SPRINT2(1, "EG_makeTessBody -> status=%d, ebody3[1]=%llx", status, (long long)ebody3[1]);
                printEgoList(MODL->context);      // nego=428 (+3)

                /* make a Model (emodel1) that contains ebody2[0] and ebody2[1].  note that this does
                   not create any new Bodys/Tessellations, but rather just points to them */
                status = EG_makeTopology(MODL->context, NULL, MODEL, 2, NULL,
                                         1, ebody2, NULL, &emodel1);
                SPRINT2(1, "EG_makeTopology -> status=%d, emodel1=%llx", status, (long long)emodel1);
                printEgoList(MODL->context);      // nego=430 (+2)

                /* delete the unattached ecurve1, enodes, eedge1, eloop1 that were
                   used to make ebody0 */
                status = EG_deleteObject(MODL->context);
                SPRINT1(1, "EG_deleteObject(context) -> status=%d", status);
                printEgoList(MODL->context);      // nego=416 (-14)

                /* try to delete ebody2[1].  this fails because ebody2[1] is still used by emodel1 */
                status = EG_deleteObject(ebody2[1]);
                SPRINT1(1, "EG_deleteObject(ebody2[1]) -> status=%d", status);
                printEgoList(MODL->context);      // nego=416

                /* try to delete ebody2[0]. this fails because ebody2[0] is still used by emodel1 */
                status = EG_deleteObject(ebody2[0]);
                SPRINT1(1, "EG_deleteObject(ebody2[0]) -> status=%d", status);
                printEgoList(MODL->context);      // nego=416

                /* delete emodel1 (and ebody2[0] and ebody2[1], which it contains) */
                status = EG_deleteObject(emodel1);
                SPRINT1(1, "EG_deleteObject(emodel1) -> status=%d", status);
                printEgoList(MODL->context);      // nego=279 (-137)

                /* delete ebody1 */
                status = EG_deleteObject(ebody1);
                SPRINT1(1, "EG_deleteObject(ebody1) -> status=%d", status);
                printEgoList(MODL->context);      // nego=147 (-132)

                /* try to delete ebody3[0].  this fails because it is still ysed by ebody3[1] */
                status = EG_deleteObject(ebody3[0]);
                SPRINT1(1, "EG_deleteObject(ebody3[0]) -> status=%d", status);
                printEgoList(MODL->context);      // nego=147

                /* delete ebody3[1] and then ebody3[0] */
                status = EG_deleteObject(ebody3[1]);
                SPRINT1(1, "EG_deleteObject(ebody3[1]) -> status=%d", status);
                printEgoList(MODL->context);      // nego=144 (-3)

                status = EG_deleteObject(ebody3[0]);
                SPRINT1(1, "EG_deleteObject(ebody3[0]) -> status=%d", status);
                printEgoList(MODL->context);      // nego=12 (-132)

                /* deleting the context does not do anythig since there are
                   no unattached egos */
                status = EG_deleteObject(MODL->context);
                SPRINT1(1, "EG_deleteObject(context) -> status=%d", status);
                printEgoList(MODL->context);      // nego=12

                /* closing the context deletes the remaining ebody0 */
                status = EG_close(MODL->context);
                SPRINT1(1, "EG_close -> status=%d", status);
                                                  // nego=0

                /* make a new context (for future commands) */
                status = EG_open(&(MODL->context));
                CHECK_STATUS(EG_open);

            /* "$supellTest N R" */
            } else if (strcmp(MODL->brch[ibrch].arg1, "$supellTest") == 0) {
#ifdef GRAFIC
                int     io_kbd=5, io_scr=6, indgr=1+4+8+16+64;
                int     nplot=0, nline=0, ilin[9], isym[9], nper[9];
                int     iedge, npnt, i;
                float   xplot[10000], yplot[10000];
                CDOUBLE *t, *xyz, *dxyz;
                char    pltitl[80];

                grinit_(&io_kbd, &io_scr, "supellTest", STRLEN("supellTest"));

                status = ocsmTessellate(MODL, MODL->nbody);
                CHECK_STATUS(ocsmTessellate);

                for (iedge = 1; iedge <= MODL->body[MODL->nbody].nedge; iedge++) {
                    status = EG_getTessEdge(MODL->body[MODL->nbody].etess, iedge,
                                            &npnt, &xyz, &t);
                    CHECK_STATUS(EG_getTessEdge);

                    status = ocsmGetTessVel(MODL, MODL->nbody, OCSM_EDGE, iedge, &dxyz);
                    CHECK_STATUS(ocsmGetTessVel);

                    nplot = 0;
                    nline = 0;

                    for (i = 0; i < npnt; i++) {
                        xplot[nplot] = t[i];
                        yplot[nplot] = sqrt(xyz[3*i]*xyz[3*i] + xyz[3*i+1]*xyz[3*i+1]) - args[3].val[0];
                        nplot++;
                    }
                    ilin[nline] = +GR_SOLID;
                    isym[nline] = -GR_CIRCLE;
                    nper[nline] = npnt;
                    nline++;

                    snprintf(pltitl, 79, "~t~delta R~Edge %d, numpnt=%d", iedge, NINT(args[2].val[0]));
                    grline_(ilin, isym, &nline, pltitl, &indgr, xplot, yplot, nper, STRLEN(pltitl));

                    nplot = 0;
                    nline = 0;

                    for (i = 0; i < npnt; i++) {
                        xplot[nplot] = t[i];
                        yplot[nplot] = sqrt(dxyz[3*i]*dxyz[3*i] + dxyz[3*i+1]*dxyz[3*i+1]) - 1;
                        nplot++;
                    }
                    ilin[nline] = +GR_SOLID;
                    isym[nline] = -GR_CIRCLE;
                    nper[nline] = npnt;
                    nline++;

                    snprintf(pltitl, 79, "~t~delta mag(Rdot)~Edge %d, numpnts=%d", iedge, NINT(args[2].val[0]));
                    grline_(ilin, isym, &nline, pltitl, &indgr, xplot, yplot, nper, STRLEN(pltitl));

                    nplot = 0;
                    nline = 0;

                    for (i = 0; i < npnt; i++) {
                        xplot[nplot] = i;
                        yplot[nplot] = atan2(xyz[3*i+1], xyz[3*i]);
                        nplot++;
                    }
                    ilin[nline] = +GR_DASHED;
                    isym[nline] = -GR_CIRCLE;
                    nper[nline] = npnt;
                    nline++;

                    for (i = 0; i < npnt; i++) {
                        xplot[nplot] = i;
                        yplot[nplot] = atan2(dxyz[3*i+1], dxyz[3*i]);
                        nplot++;
                    }
                    ilin[nline] = +GR_SOLID;
                    isym[nline] = -GR_PLUS;
                    nper[nline] = npnt;
                    nline++;

                    snprintf(pltitl, 79, "~i~ang(R, Rdot)~Edge %d, numpnts=%d", iedge, NINT(args[2].val[0]));
                    grline_(ilin, isym, &nline, pltitl, &indgr, xplot, yplot, nper, STRLEN(pltitl));
                }
#endif

            /* default - just make copy of Body on stack */
            } else {

                /* pop a Body from the stack */
                if (nstack < 1) {
                    status = signalError(MODL, OCSM_INSUFFICIENT_BODYS_ON_STACK,
                                         "SPECIAL expects a Body on the stack");
                    goto next_branch;
                } else if (stack[nstack-1] <= 0) {
                    status = signalError(MODL, OCSM_INSUFFICIENT_BODYS_ON_STACK,
                                         "SPECIAL expects a Body on the stack");
                    goto next_branch;
                } else {
                    ibodyl = stack[--nstack];
                }

                hasdots = 0;

                /* recycle old Body(s) if not dirty */
                status = recycleBody(MODL, ibrch, type, args, hasdots);
                CHECK_STATUS(recycleBody);

                if (status > 0) {
                    for (jbody = 0; jbody < status; jbody++) {
                        stack[nstack++] = MODL->nbody - jbody;
                    }
                    status = SUCCESS;
                    goto next_branch;
                }

                ebodyl = MODL->body[ibodyl].ebody;

                /* create a Body */
                status = newBody(MODL, ibrch, type, ibodyl, -1,
                                 args, hasdots, MODL->body[ibodyl].botype, &ibody);
                CHECK_STATUS(newBody);

                /* default behavior is to ignore the arguments and simply copy the Body */
                status = EG_copyObject(ebodyl, NULL, &ebody);
                CHECK_STATUS(EG_copyObject);

                MODL->body[ibody].ebody = ebody;

                /* update @-parameters and finish Body (SPECIAL) */
                status = setupAtPmtrs(MODL, 0);
                CHECK_STATUS(setupAtPmtrs);

                status = finishBody(MODL, ibody);
                if (MODL->sigCode != SUCCESS) goto cleanup;        // cannot be caught
                CHECK_STATUS(finishBody);

                /* push the Body onto the stack */
                stack[nstack++] = ibody;

                status = getBodyTolerance(MODL->body[ibody].ebody, &toler);
                CHECK_STATUS(getBodyTolerance);

                SPRINT5(1, "                          Body   %4d created  (toler=%11.4e, nnode=%4d, nedge=%4d, nface=%4d)",
                        ibody, toler, MODL->body[ibody].nnode, MODL->body[ibody].nedge, MODL->body[ibody].nface);
            }
        }

        /* keep track of profile info */
        new_time = clock();

        if (type >= OCSM_DIMENSION && type <= OCSM_MESSAGE) {
            MODL->profile[type-100].ncall += 1;
            MODL->profile[type-100].time  += (new_time - old_time);
        }

        /* record the required CPU time to create this Body */
        if (MODL->nbody > nbodySave) {
            MODL->body[MODL->nbody].CPU = (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC);
        }

        /* record that we successfully executed this Branch */
        *builtTo = ibrch;

next_branch:
        if (STRLEN(MODL->printAttr) > 0) {
            status = printBodyAttrs(MODL, MODL->nbody);
            CHECK_STATUS(printBodyAttrs);
        }

        if (MODL->printPmtrs == 1) {
            status = printPmtrs(MODL, stdout);
            CHECK_STATUS(printPmtrs);
        }

        /* restore the stack if there is an error */
        if (MODL->sigCode == OCSM_UNDERCONSTRAINED ||
            MODL->sigCode == OCSM_OVERCONSTRAINED  ||
            MODL->sigCode == OCSM_NOT_CONVERGED      ) {
            // do nothing
        } else if (MODL->sigCode != 0) {
            nstack      = nstackSave;
            MODL->nbody = nbodySave;
        }

        /* clean up all unattached egos */
        if (MODL->loadEgads < 0 || (MODL->loadEgads == 0 && MODL->nbody >= MODL->recycle)) {
            (MODL->cleanup)--;
            if (MODL->cleanup <= 0) {
                MODL->cleanup = 25;

                (void) EG_setOutLevel(MODL->context, 0);
                status = EG_deleteObject(MODL->context);
                (void) EG_setOutLevel(MODL->context, outLevel);
                CHECK_STATUS(EG_deleteObject);
            }
        }

        /* set up @stack */
        for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
            if (strcmp(MODL->pmtr[ipmtr].name, "@stack") == 0      &&
                MODL->pmtr[ipmtr].scope == MODL->scope[MODL->level]  ) {
                if (nstack < 1) {
                    MODL->pmtr[ipmtr].value[0] = -1;
                    MODL->pmtr[ipmtr].ncol     = 1;
                } else {
                    for (i = 0; i < nstack; i++) {
                        MODL->pmtr[ipmtr].value[i] = stack[i];
                    }
                    MODL->pmtr[ipmtr].ncol = nstack;
                }
            }
        }

        /* print the current stack */
        if (MODL->printStack != 0) {
            SPRINT1(1, "Stack at end of \"%s\" statement:", ocsmGetText(type));
            for (i = 0; i < nstack; i++) {
                ibodyl = stack[i];

                if (ibodyl == 0) {
                    SPRINT1(1, "    stack[%3d] = Mark", i);
                } else if (ibodyl < 0) {
                    SPRINT2(1, "    stack[%3d] = Sketch %d", i, ibodyl);
                } else {
                    SPRINT8(1, "    stack[%3d] = Body %5d %-20s %-20s ileft=%3d, irite = %3d, ichld=%3d, igroup=%3d",
                            i, ibodyl,
                            ocsmGetText(MODL->body[ibodyl].brtype),
                            ocsmGetText(MODL->body[ibodyl].botype),
                            MODL->body[ibodyl].ileft,
                            MODL->body[ibodyl].irite,
                            MODL->body[ibodyl].ichld,
                            MODL->body[ibodyl].igroup);
                }
            }
        }

        if (PRINT_BODYS == 1) {
            int    nnode, nedge, nface;
            char   onstack;

            SPRINT1(1, "Bodys at end of \"%s\" statement:", ocsmGetText(type));
            SPRINT0(1, "   ibody                                     igroup ileft irite nnode nedge nface");
            for (ibodyl = 1; ibodyl <= MODL->nbody; ibodyl++) {
                if (MODL->body[ibodyl].ebody == NULL) {
                    nnode  = -1;
                    nedge  = -1;
                    nface  = -1;
                    oclass = -1;
                    mtype  = -1;
                } else {
                    EG_getBodyTopos(MODL->body[ibodyl].ebody, NULL, NODE, &nnode, NULL);
                    EG_getBodyTopos(MODL->body[ibodyl].ebody, NULL, EDGE, &nedge, NULL);
                    EG_getBodyTopos(MODL->body[ibodyl].ebody, NULL, FACE, &nface, NULL);
                    EG_getTopology( MODL->body[ibodyl].ebody, &eref, &oclass, &mtype, data, &nchild, &echilds, &senses);
                }

                onstack = ' ';
                for (jstack = 0; jstack < nstack; jstack++) {
                    if (stack[jstack] == ibodyl) {
                        onstack = '*';
                        break;
                    }
                }

                SPRINT10(1, "   %5d%c %-20s %-14s %5d %5d %5d %5d %5d %5d",
                         ibodyl, onstack,
                         ocsmGetText(MODL->body[ibodyl].brtype),
                         ocsmGetText(MODL->body[ibodyl].botype),
                         MODL->body[ibodyl].igroup,
                         MODL->body[ibodyl].ileft,
                         MODL->body[ibodyl].irite,
                         nnode, nedge, nface);
            }
        }

        if (PRINT_LAST_EGO > 0 && MODL->nbody > 0) {
            int oldOutLevel = outLevel;
            SPRINT1(1, "Body %d", MODL->nbody);
            outLevel = PRINT_LAST_EGO;
            ocsmPrintEgo(MODL->body[MODL->nbody].ebody);
            outLevel = oldOutLevel;
        }

        /* clear "dots" on all Bodys */
        if (CLEAR_DOTS == 1) {
            for (istor = 0; istor <  MODL->nstor; istor++) {
                SPLINT_CHECK_FOR_NULL(MODL->stor);

                for (i = 0; i < MODL->stor[istor].nbody; i++) {
                    status = EG_setGeometry_dot(MODL->stor[istor].ebody[i], 0, 0, NULL, NULL, NULL);
                    CHECK_STATUS(EG_setGeometry_dot);

                    SPRINT3(2, "removing dot from istor=%d:%d (line %d)", istor, i, __LINE__);
                }
            }
            for (ibody = 1; ibody <= MODL->nbody; ibody++) {
                if (MODL->body[ibody].ebody != NULL) {
                    status = EG_setGeometry_dot(MODL->body[ibody].ebody, 0, 0, NULL, NULL, NULL);
                    CHECK_STATUS(EG_setGeometry_dot);

                    SPRINT2(2, "removing dot from ibody=%d (line %d)", ibody, __LINE__);
                }
            }
        }

        /* print "dot" status of all Storages and Bodys */
        if (PRINT_DOT_STATUS == 1) {
            for (istor = 0; istor <  MODL->nstor; istor++) {
                SPLINT_CHECK_FOR_NULL(MODL->stor);

                for (i = 0; i < MODL->stor[istor].nbody; i++) {
                    SPRINT4(1, "stor[%3d].ebody[%3d]=%llx (hasdot=%3d)",
                            istor, i, (long long)MODL->stor[istor].ebody[i],
                            EG_hasGeometry_dot(MODL->stor[istor].ebody[i]));
                }
            }
            for (ibody = 1; ibody <= MODL->nbody; ibody++) {
                if (MODL->body[ibody].ebody != NULL) {
                    SPRINT5(1, "body[%3d].ebody=%llx, brtype=%-10s, botype=%-10s, (largestDot=%10.5f)",
                            ibody, (long long)MODL->body[ibody].ebody,
                            ocsmGetText(MODL->body[ibody].brtype),
                            ocsmGetText(MODL->body[ibody].botype),
                            largestDot(MODL->body[ibody].ebody));
                }
            }
        }
    }

finalize:
    SPRINT0(1, "    finalizing:");

    /* mark all Branches up to builtTo as as not dirty and those after
       builtTo as dirty */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        if (ibrch <= *builtTo) {
            MODL->brch[ibrch].dirty = 0;
        } else {
            MODL->brch[ibrch].dirty = 1;
        }
    }

    /* (optionally) print info about the contents of the Storages */
    if (MODL->printStors == 1) {
        SPLINT_CHECK_FOR_NULL(MODL->stor);

        SPRINT0(1, "    istor name                           index ibody(s)");
        for (istor = 0l; istor < MODL->nstor; istor++) {
            SPRINT3x(1, "    %5d %-30s %5d", istor, MODL->stor[istor].name, MODL->stor[istor].index);
            for (i = 0; i < MODL->stor[istor].nbody; i++) {
                SPRINT2x(1, " %5d (%s)", MODL->stor[istor].ibody[i], ocsmGetText(MODL->body[MODL->stor[istor].ibody[i]].botype));
            }
            SPRINT0(1, " ");
        }
    }

    /* verify that the level is back to 0 */
    if (MODL->scope[MODL->level] != 0) {
        status = signalError(MODL, OCSM_INTERNAL_ERROR,
                             "MODL->scope=%d (but should be 0)", MODL->level);
    }

    /* special processing if there was an uncaught error signal */
    if (MODL->sigCode < 0) {

        /* close the Sketch */
        sket->type   = 0;
        sket->size   = 0;
        sket->solved = 1;
        sket->nseg   = 0;
        sket->nvar   = 0;
        sket->ncon   = 0;

        /* return negative of builtTo */
        (*builtTo) = -ibrchSave;

    /* special processing if there was an uncaught user-thrown signal */
    } else if (MODL->sigCode > 0) {
        SPRINT1(1, "WARNING:: there was an uncaught user-thrown signal %d",
                MODL->sigCode);
        (MODL->nwarn)++;

        /* close the Sketch */
        sket->type   = 0;
        sket->size   = 0;
        sket->solved = 1;
        sket->nseg   = 0;
        sket->nvar   = 0;
        sket->ncon   = 0;

        /* return negative of builtTo */
        (*builtTo) = -ibrchSave;
    }

    /* make sure the Sketch is not open */
    if (sket->nseg > 0) {
        SET_STATUS(OCSM_SKETCH_IS_OPEN, end);
    }

    /* free up the storage associated with the arguments */
    for (iarg = 1; iarg < 10; iarg++) {
        FREE(args[iarg].val);   /* also free's .str since they are unioned */
        FREE(args[iarg].dot);
    }

    /* make sure that user-supplied body[] is big enough */
    if (body != NULL && nstack > nbodyMax) {
        status = OCSM_TOO_MANY_BODYS_ON_STACK;
        goto cleanup;
    }

    /* pre-mark all the Bodys as not being on the stack (fixed below) */
    for (ibody = 1; ibody <= MODL->nbody; ibody++) {
        MODL->body[ibody].onstack = 0;
    }

    /* report a warning if there is a mark on the stack */
    for (i = 0; i < nstack; i++) {
        ibody = stack[i];
        if (ibody == 0) {
            SPRINT0(1, "WARNING:: mark being removed from the stack");
            (MODL->nwarn)++;

            for (j = i+1; j < nstack; j++) {
                stack[j-1] = stack[j];
            }
            nstack--;

            i--;  /* repeat in case first moved stack entry was also a mark */
        }
    }

    /* mark and (optionally) tessellate the stack's Bodys */
    for (i = 0; i < nstack; i++) {
        ibody = stack[i];
        MODL->body[ibody].onstack = 1;

        if (MODL->tessAtEnd == 1) {
            status = ocsmTessellate(MODL, ibody);
            CHECK_STATUS(ocsmTessellate);
        }
    }

    /* if any Body on the stack has _erepAttr and _erepAngle Attributes, create the EBody */
    if (MODL->tessAtEnd == 1 && MODL->erepAtEnd == 1) {
        for (ibody = 1; ibody <= MODL->nbody; ibody++) {
            if (MODL->body[ibody].onstack != 1) continue;

            status = EG_attributeRet(MODL->body[ibody].ebody, "_erepAngle", &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status != SUCCESS || attrType != ATTRREAL) continue;
            SPLINT_CHECK_FOR_NULL(tempRlist);
            dihedral = tempRlist[0];

            status = EG_attributeRet(MODL->body[ibody].ebody, "_erepAttr", &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == SUCCESS) {
                if (attrType == ATTRSTRING) {
                    status = EG_initEBody(MODL->body[ibody].etess, dihedral, &(MODL->body[ibody].eebody));
                    CHECK_STATUS(EG_initEBody);

                    status = EG_makeAttrEFaces(MODL->body[ibody].eebody, tempClist, &nefaces, NULL);
                    if (status != SUCCESS) {
                        status = EG_deleteObject(MODL->body[ibody].eebody);
                        CHECK_STATUS(EG_deleteObject);

                        MODL->body[ibody].eebody = NULL;

                        SPRINT1(1, "WARNING:: could not make an EBody of ibody=%d", ibody);
                        (MODL->nwarn)++;

                        continue;
                    }

                    status = EG_finishEBody(MODL->body[ibody].eebody);
                    CHECK_STATUS(EG_finishEBody);

                    /* tessellate the EBody */
                    status = EG_attributeRet(MODL->body[ibody].ebody, "__hasTess__", &attrType, &attrLen,
                                             &tempIlist, &tempRlist, &tempClist);
                    if (status != EGADS_SUCCESS || attrType != ATTRINT || attrLen != 1) {
                        status = EG_attributeRet(MODL->body[ibody].ebody, "_tParams", &attrType, &attrLen,
                                                 &tempIlist, &tempRlist, &tempClist);
                        CHECK_STATUS(EG_attributeRet);

                        SPRINT4(1, "--> Tessellating EBody %6d     (%12.5e %12.5e %7.3f)",
                                ibody, tempRlist[0], tempRlist[1], tempRlist[2]);

                        status = EG_makeTessBody(MODL->body[ibody].eebody, (double*)tempRlist, &(MODL->body[ibody].eetess));
                        CHECK_STATUS(EG_makeTessBody);

                        status = EG_attributeAdd(MODL->body[ibody].eetess, ".tessType", ATTRSTRING,
                                                 4, NULL, NULL, "Tris");
                        CHECK_STATUS(EG_attributeAdd);
                    }

                    /* update the at-Parameters */
                    status = setupAtPmtrs(MODL, 0);
                    CHECK_STATUS(setupAtPmtrs);
                }
            }
        }
    }

    /* return Bodys on the stack (LIFO) */
    if (body != NULL) {
        *nbody = nstack;
    }

    nbodyMax = nstack;
    for (i = 0; i < nbodyMax; i++) {
        if (body != NULL) {
            body[i] = stack[--nstack];

            if (MODL->body[body[i]].botype != OCSM_NODE_BODY) {
                status = getBodyTolerance(MODL->body[body[i]].ebody, &toler);
                CHECK_STATUS(getBodyTolerance);

                SPRINT5(1, "    Body %5d   nnode=%-6d   nedge=%-6d   nface=%-6d   toler=%11.4e", body[i],
                        MODL->body[body[i]].nnode,
                        MODL->body[body[i]].nedge,
                        MODL->body[body[i]].nface, toler);

                if (outLevel >= 3) {
                    status = printBodyAttrs(MODL, i);
                    CHECK_STATUS(printBodyAttrs);
                }
            } else {
                SPRINT5(1, "    Body %5d   nnode=%-6d   nedge=%-6d   nface=%-6d   toler=%11.4e", body[i],
                        1, 0, 0, 0.0);
            }
        }
    }

    /* print info for all Bodys in stack */
    if (PRINT_BODY_INFO == 1) {
        for (ibody = 1; ibody <= MODL->nbody; ibody++) {
            if (MODL->body[ibody].onstack > 0) {
                SPRINT1(1, "ibody=%d", ibody);
                status = printBodyInfo(&(MODL->body[ibody]));
                CHECK_STATUS(printBodyInfo);
            }
        }
    }

    /* since we do not know what goes on inside a UDP/UDF, set .hassens=1 on all
       Bodys associated with UDPRIMs if numdots>0 (to make sure that their cache's are removed) */
    if (MODL->numdots > 0) {
        for (ibody = 1; ibody <= MODL->nbody; ibody++) {
            if (MODL->body[ibody].brtype == OCSM_UDPRIM) {
                MODL->body[ibody].hassens = 1;
            }
        }
    }

    /* propagate the .hasdots to all children -- take care of internal Xsects, such as a RULE or BLEND */
    for (ibody = 1; ibody <= MODL->nbody; ibody++) {
        if (MODL->body[ibody].ichld > 0 && MODL->body[ibody].hasdots > 0) {
            MODL->body[MODL->body[ibody].ichld].hasdots = MODL->body[ibody].hasdots;
        }
    }

    /* propagate the hasdots to all children -- take care of multiple RESTOREs */
    for (ibody = 1; ibody <= MODL->nbody; ibody++) {
        ileft = MODL->body[ibody].ileft;
        irite = MODL->body[ibody].irite;

        if (ileft > 0) {
            if (MODL->body[ileft].hasdots > 0) {
                MODL->body[ibody].hasdots = 1;
            }
        }
        if (irite > 0) {
            if (MODL->body[irite].hasdots > 0) {
                MODL->body[ibody].hasdots = 1;
            }
        }
    }

    /* if loadEgads was set, unset it (since we only want to run this once)
       and do not let Bodys be recycled */
    if (MODL->loadEgads != 0) {
        MODL->loadEgads = 0;

        for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
            MODL->brch[ibrch].dirty = 1;
        }
    }

    /* plot arc-length distribution for all Edges in all Bodys on stack */
#ifdef GRAFIC
    if (PLOT_ARCLENGTHS == 1) {
        int    periodic, iedge, ipnt;
        int    io_kbd=5, io_scr=6, ilin[3], isym[3], nper[3], nline, indgr;
        float  tplot[101], splot[101];
        double trange[4], tt, data[18], xlast, ylast, zlast;
        char   pltitl[80];

        SPRINT0(0, "plotting arc-length distributions using GRAFIC");

        snprintf(pltitl, 79, "arc-length distribution");
        grinit_(&io_kbd, &io_scr, pltitl, STRLEN(pltitl));

        for (ibody = 1; ibody <= MODL->nbody; ibody++) {
            if (MODL->body[ibody].onstack == 0) continue;

            for (iedge = 1; iedge <= MODL->body[ibody].nedge; iedge++) {
                if (MODL->body[ibody].edge[iedge].itype == DEGENERATE) continue;

                EG_getRange(MODL->body[ibody].edge[iedge].eedge, trange, &periodic);

                EG_evaluate(MODL->body[ibody].edge[iedge].eedge, trange, data);

                tplot[0] = trange[0];
                splot[0] = 0;

                for (ipnt = 1; ipnt < 101; ipnt++) {
                    xlast = data[0];
                    ylast = data[1];
                    zlast = data[2];

                    tt = trange[0] + (trange[1] - trange[0]) * (double)(ipnt) / 101.;

                    EG_evaluate(MODL->body[ibody].edge[iedge].eedge, &tt, data);

                    tplot[ipnt] = tt;
                    splot[ipnt] = splot[ipnt-1] + sqrt(SQR(data[0]-xlast) + SQR(data[1]-ylast) + SQR(data[2]-zlast));
                }

                indgr   = 1  + 4 + 8 + 16 + 64;
                ilin[0] = +GR_SOLID;
                isym[0] = -GR_STAR;
                nper[0] = 101;
                nline   = 1;

                snprintf(pltitl, 79, "~t~s~Body %d, Edge %d", ibody, iedge);
                grline_(ilin, isym, &nline, pltitl,
                        &indgr, tplot, splot, nper, STRLEN(pltitl));
            }
        }
    }
#endif

    /* plot UV tessellation for all Faces for all Bodys on stack */
#ifdef GRAFIC
    if (PLOT_UVTESS == 1) {
        int     ibody, iface, ip0, ip1, ip2;
        int     io_kbd=5, io_scr=6, indgr=1+4+16+64;
        int     ndata, nline, *nper=NULL, *ilin=NULL, *isym=NULL;
        float   *uplot=NULL, *vplot=NULL;
        char    pltitl[80];

        SPRINT0(0, "Plotting tessellation in UV using GRAFIC");

        grinit_(&io_kbd, &io_scr, "UV tessellation", STRLEN("UV tessellation"));

        for (ibody = 1; ibody <= MODL->nbody; ibody++) {
            if (MODL->body[ibody].onstack == 0) continue;

            for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
                status = EG_getTessFace(MODL->body[ibody].etess, iface,
                                        &npnt, &xyz, &uv, &ptype, &pindx,
                                        &ntri, &tris, &tric);
                if (status != EGADS_SUCCESS) {
                    SPRINT2(0, "ERROR:: no tessellation for ibody=%d, iface=%d", ibody, iface);
                    continue;
                } else if (ntri == 0) {
                    SPRINT2(0, "ERROR:: no tessellation for ibody=%d, iface=%d", ibody, iface);
                    continue;
                }

                MALLOC(uplot, float, 4*ntri);
                MALLOC(vplot, float, 4*ntri);
                MALLOC(nper,  int,     ntri);
                MALLOC(ilin,  int,     ntri);
                MALLOC(isym,  int,     ntri);

                snprintf(pltitl, 79, "~u~v~ ibody=%d, iface=%d", ibody, iface);

                ndata = 0;
                nline = 0;

                for (itri = 0; itri < ntri; itri++) {
                    ip0 = tris[3*itri  ] - 1;
                    ip1 = tris[3*itri+1] - 1;
                    ip2 = tris[3*itri+2] - 1;

                    uplot[ndata] = uv[2*ip0  ];
                    vplot[ndata] = uv[2*ip0+1];
                    ndata++;

                    uplot[ndata] = uv[2*ip1  ];
                    vplot[ndata] = uv[2*ip1+1];
                    ndata++;

                    uplot[ndata] = uv[2*ip2  ];
                    vplot[ndata] = uv[2*ip2+1];
                    ndata++;

                    uplot[ndata] = uv[2*ip0  ];
                    vplot[ndata] = uv[2*ip0+1];
                    ndata++;

                    nper[nline] = 4;
                    ilin[nline] = GR_SOLID;
                    isym[nline] = 0;
                    nline++;
                }

                grline_(ilin, isym, &nline, pltitl, &indgr, uplot, vplot, nper, STRLEN(pltitl));

                FREE(uplot);
                FREE(vplot);
                FREE(nper );
                FREE(ilin );
                FREE(isym );
            }
        }
    }
#endif

    /* plot trim curves for all Faces all Bodys on stack */
#ifdef GRAFIC
    if (PLOT_TRIMCURVES == 1) {
        int     ibody, iface, oclass, mtype, nloop, *senses, nedge, iedge, periodic, i;
        double  data[18], trange[4], t;
        int     io_kbd=5, io_scr=6, indgr=1+4+16+64;
        int     ndata, nline, *nper=NULL, *ilin=NULL, *isym=NULL;
        float   *uplot=NULL, *vplot=NULL;
        char    pltitl[80];
        ego     eref, *eloops, *eedges;

        SPRINT0(0, "Plotting trim curves using GRAFIC");

        grinit_(&io_kbd, &io_scr, "Trim curves", STRLEN("Trim curves"));

        for (ibody = 1; ibody <= MODL->nbody; ibody++) {
            if (MODL->body[ibody].onstack == 0) continue;

            for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
                status = EG_getTopology(MODL->body[ibody].face[iface].eface,
                                        &eref, &oclass, &mtype, data, &nloop, &eloops, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getInfo(eref, &oclass, &mtype, NULL, NULL, NULL);
                CHECK_STATUS(EG_getInfo);

                if (mtype == PLANE) continue;

                status = EG_getTopology(eloops[0], &eref, &oclass, &mtype, data, &nedge, &eedges, &senses);
                CHECK_STATUS(EG_getTopology);

                MALLOC(uplot, float, 103*nedge);
                MALLOC(vplot, float, 103*nedge);
                MALLOC(nper,  int,     3*nedge);
                MALLOC(ilin,  int,     3*nedge);
                MALLOC(isym,  int,     3*nedge);

                snprintf(pltitl, 79, "~u~v~ ibody=%d, iface=%d", ibody, iface);
                ndata = 0;
                nline = 0;

                for (iedge = 0; iedge < nedge; iedge++) {
                    status = EG_getRange(eedges[iedge], trange, &periodic);
                    CHECK_STATUS(EG_getRange);

                    for (i = 0; i < 101; i++) {
                        t = trange[0] + (trange[1] - trange[0]) * i * 0.01;

                        status = EG_evaluate(eedges[iedge+nedge], &t, data);
                        CHECK_STATUS(EG_evaluate);

                        /* circle at beginning */
                        if (i == 0) {
                            uplot[ndata] = data[0];
                            vplot[ndata] = data[1];
                            ndata++;

                            nper[nline] = 1;
                            ilin[nline] = 0;
                            isym[nline] = GR_CIRCLE;
                            nline++;
                        }

                        uplot[ndata] = data[0];
                        vplot[ndata] = data[1];
                        ndata++;
                    }

                    nper[nline] = 101;
                    ilin[nline] = GR_SOLID;
                    isym[nline] = GR_PLUS;
                    nline++;

                    /* triangle at end */
                    uplot[ndata] = uplot[ndata-1];
                    vplot[ndata] = vplot[ndata-1];
                    ndata++;

                    nper[nline] = 1;
                    ilin[nline] = 0;
                    isym[nline] = GR_TRIANGLE;
                    nline++;
                }

                grline_(ilin, isym, &nline, pltitl, &indgr, uplot, vplot, nper, STRLEN(pltitl));

                FREE(uplot);
                FREE(vplot);
                FREE(nper );
                FREE(ilin );
                FREE(isym );
            }
        }
    }
#endif

    /* compute distance between Face evaluations and tessellation points at boundaries of Faces */
    if (0) {
        double xyz_out[18], tol=1e-2;

        SPRINT0(0, "Checking distances at boundaries of Faces");

        for (ibody = 1; ibody <= MODL->nbody; ibody++) {
            if (MODL->body[ibody].onstack != 1) continue;

            for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
                status = EG_getTessFace(MODL->body[ibody].etess, iface,
                                        &npnt, &xyz, &uv, &ptype, &pindx,
                                        &ntri, &tris, &tric);
                CHECK_STATUS(EG_getTessFace);

                for (ipnt = 0; ipnt < npnt; ipnt++) {
                    if (ptype[ipnt] >= 0) {
                        status = EG_evaluate(MODL->body[ibody].face[iface].eface, &(uv[2*ipnt]), xyz_out);
                        CHECK_STATUS(EG_evaluate);

                        dist = sqrt((xyz[3*ipnt  ]-xyz_out[0]) * (xyz[3*ipnt  ]-xyz_out[0])
                                   +(xyz[3*ipnt+1]-xyz_out[1]) * (xyz[3*ipnt+1]-xyz_out[1])
                                   +(xyz[3*ipnt+2]-xyz_out[2]) * (xyz[3*ipnt+2]-xyz_out[2]));
                        if (dist > tol) {
                            SPRINT7(0, "ibody=%2d, iface=%3d, ipnt=%4d, dist=%12.4e (%12.5f %12.5f %12.5f)",
                                   ibody, iface, ipnt, dist, xyz[3*ipnt], xyz[3*ipnt+1], xyz[3*ipnt+2]);
                        }
                    }
                }
            }
        }
    }

    /* if this MODL is a perturbation (that is, has a basemodl), restore the MODL pointer in the context */
    if (MODL->basemodl != NULL) {
        status = EG_setUserPointer(MODL->context, (void*)(MODL->basemodl));
        CHECK_STATUS(EG_setUserPointer);
    }

    /* if there is a pending signal and outLevel==0, print message now */
    if (outLevel == 0 && strlen(MODL->sigMesg) > 0) {
        SPRINT1(0, "%s", MODL->sigMesg);
    }

cleanup:
    if (modl != NULL) {
        (void) EG_setOutLevel(MODL->context, 0);
        status2 = EG_deleteObject(MODL->context);
        (void) EG_setOutLevel(MODL->context, outLevel);
        if (status2 < EGADS_SUCCESS) {
            EPRINT( "ERROR:: BAD STATUS = %d from %s (called from %s:%d)\n", status2, "EG_deleteObject", routine, __LINE__);
        }
    }

    for (iarg = 1; iarg < 10; iarg++) {
        FREE(args[iarg].val);   /* also free's .str since they are unioned */
        FREE(args[iarg].dot);
    }

    fflush(0);

    FREE(props  );
    FREE(sket   );
    FREE(solcons);
    FREE(solvars);
    FREE(patn   );
    FREE(macros );
    FREE(stack  );
    FREE(iblist );
    FREE(values );
    FREE(dots   );
    FREE(tempList);

#undef CATCH_STATUS
    /* if MODL->sigCode is not success, return it */
    if (modl != NULL) {
        if (MODL->sigCode != SUCCESS) {
            status = MODL->sigCode;
        }
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmPrintProfile - print profile from last ocsmBuild()             *
 *                                                                      *
 ************************************************************************
 */

int
ocsmPrintProfile(void   *modl,          /* (in)  pointer to MODL */
                 char   filename[])     /* (in)  file to which output is appended (or "" for stdout) */
{
    int       status = SUCCESS;         /* (out) return status */

    int       i, total_call;
    FILE      *fp=NULL;
    clock_t   total_time;

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmPrintProfile);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    if (strlen(filename) == 0) {
        fp = stdout;
    } else {
        fp = fopen(filename, "a");
        if (fp == NULL) {
            status = OCSM_FILE_NOT_FOUND;
            goto cleanup;
        }
    }

    /* print the profile info */
    fprintf(fp, "==> Profile information\n");
    total_time = 0;
    total_call = 0;
    for (i = 0; i < 101; i++) {
        total_call += MODL->profile[i].ncall;
        total_time += MODL->profile[i].time;
    }
    fprintf(fp, "    Branch type           ncall  time (sec)    pct\n");
    for (i = 0; i < 101; i++) {
        if (MODL->profile[i].ncall > 0) {
            fprintf(fp, "    %-20s  %5d  %10.3f  %5.1f\n", ocsmGetText(i+100), MODL->profile[i].ncall,
                    (double)(MODL->profile[i].time) / (double)(CLOCKS_PER_SEC),
                    (double)(MODL->profile[i].time) / (double)(total_time) * 100);
        }
    }
    fprintf(fp, "    Total                 %5d  %10.3f\n", total_call, (double)(total_time)/(double)(CLOCKS_PER_SEC));

cleanup:
    if (fp != NULL && strlen(filename) > 0) fclose(fp);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmTessellate - tessellate one or more Bodys                      *
 *                                                                      *
 ************************************************************************
 */

int
ocsmTessellate(void   *modl,            /* (in)  pointer to MODL */
               int    ibody)            /* (in)  Body index (1:nbody) or 0 for all on stack */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       jbody, attrType, attrLen, iface, iedge, oclass, mtype, mtype_face, nlup, ilup;
    int       npnt_edge, npnt_face, ntri_face, nedg, ii, ipnt, itri, nchild, i, periodic;
    int       npnt_egg, nbnd_egg, ntri_egg, nquad, state, npts, count, nloop, iloop, nedge;
    int       nbnd, lup[20], *senses, *tris_new=NULL;
    CINT      *tempIlist, *pindx, *ptype, *tris, *tric, *p_egg, *tris_egg;
    double    params[3], bbox[6], size, uvlims[4], data[18], data2[18], trange[4];
    double    uv1[2], uv2[2], xyz_out[3];
    double    *uv_new=NULL, *xyz_new=NULL;
    CDOUBLE   *tempRlist, *xyz_edge, *t_edge, *xyz_face, *uv_face, *uv_egg;
    CCHAR     *tempClist;
    void      *eggdata, *eggdata_new;
    modl_T    *BASE;
    ego       eref, *elups, *eedgs, *echilds, *eloops, *eedges, ebody, newBody, newTess;
    ego       esurface;

    ROUTINE(ocsmTessellate);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check for valid Body */
    if (ibody < 0 || ibody > MODL->nbody) {
        status = OCSM_ILLEGAL_BODY_INDEX;
        goto cleanup;
    }

    BASE = MODL->basemodl;

    /* loop through all Bodys */
    for (jbody = 1; jbody <= MODL->nbody; jbody++) {

        /* if ibody is specified and not this Body, skip jbody */
        if (ibody > 0 && jbody != ibody) {
            continue;

        /* if no Bodys specified but jbody is not on the stack, skip jbody */
        } else if (ibody == 0 && MODL->body[jbody].onstack == 0) {
            continue;
        }

        /* if the Body is a Node, return now */
        if (MODL->body[jbody].botype == OCSM_NODE_BODY) {
            continue;
        }

        /* if there is already a tessellation, return now */
        if (MODL->body[jbody].etess != NULL) {
            continue;
        }

        /* keep track of the number of Faces that are quadded */
        nquad = 0;

        /* set up the tessellation parameters.  if there are .tParams
           on this Body, use them */
        status = EG_attributeRet(MODL->body[jbody].ebody,
                                 ".tParams", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == SUCCESS && attrLen == 3) {
            params[0] = tempRlist[0];
            params[1] = tempRlist[1];
            params[2] = tempRlist[2];

        /* otherwise, use the defaults based upon the size of the bounding box */
        } else {
            status = EG_getBoundingBox(MODL->body[jbody].ebody, bbox);
            CHECK_STATUS(EG_getBoundingBox);

            size = sqrt(SQR(bbox[3]-bbox[0]) + SQR(bbox[4]-bbox[1]) + SQR(bbox[5]-bbox[2]));

            params[0] = TESS_PARAM_0 * size;
            params[1] = TESS_PARAM_1 * size;
            params[2] = TESS_PARAM_2;

            status = EG_attributeRet(MODL->body[jbody].ebody, "_tessFactor",
                                     &attrType, &attrLen, &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS && attrType == ATTRREAL) {
                if (tempRlist[0] > 0.01) {
                    params[0] *= tempRlist[0];
                    params[1] *= tempRlist[0];
                } else {
                    SPRINT1(1, "WARNING:: _tessFactor (%12.5f) must be greater than 0.01 to be applied", tempRlist[0]);
                    (MODL->nwarn)++;
                }
            } else {
                status = EGADS_SUCCESS;
            }
        }

        /* print tessellation parameters */
        SPRINT4(1, "--> Tessellating Body %6d     (%12.5e %12.5e %7.3f)",
                jbody, params[0], params[1], params[2]);

        /* this is a base (not perturbed) MODL */
        if (BASE == NULL) {

            /* start by using EGADS' tessellator so that the Node and Edges
               get properly tessellated */
            status = EG_makeTessBody(MODL->body[jbody].ebody, params,
                                     &(MODL->body[jbody].etess));
            CHECK_STATUS(EG_makeTessBody);

            status = EG_attributeAdd(MODL->body[jbody].etess, ".tessType", ATTRSTRING,
                                     4, NULL, NULL, "Tris");
            CHECK_STATUS(EG_attributeAdd);

            /* overwrite the _tParams Attribute so that we know how the Body
               was tessellated */
            status = EG_attributeAdd(MODL->body[jbody].ebody, "_tParams", ATTRREAL,
                                     3, NULL, params, NULL);
            CHECK_STATUS(EG_attributeAdd);

            /* .tessType=="Mixed" indicates that templates were used and there
               is at least one Face that is all quads.   in this case, .mixed will be set
               (for all Faces) */

            /* make new-style Quads (whole Body) if Attribute is set on the Body */
            status = EG_attributeRet(MODL->body[jbody].ebody,
                                     "_makeQuads", &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == SUCCESS) {

                /* new-style quadding, where .tessType=="Quad" and .mixed is created */
                status = EG_quadTess(MODL->body[jbody].etess, &newTess);
                if (status == SUCCESS) {
                    status = EG_deleteObject(MODL->body[jbody].etess);
                    CHECK_STATUS(EG_deleteObject);

                    MODL->body[jbody].etess = newTess;
                } else {
                    status = SUCCESS;
                }

             /* make old-style Quads (on a Face) if Attribute is set on the Face.
                this does not have a .tessType and cannot be dumped to an .egads file */
            } else {
                params[0] = 0;
                params[1] = 0;
                params[2] = 0;
                for (iface = 1; iface <= MODL->body[jbody].nface; iface++) {
                    status = EG_attributeRet(MODL->body[jbody].face[iface].eface,
                                             "_makeQuads", &attrType, &attrLen,
                                             &tempIlist, &tempRlist, &tempClist);
                    if (status == SUCCESS) {
                        status = EG_makeQuads(MODL->body[jbody].etess, params, iface);
                        if (status == SUCCESS) {
                            nquad++;
                        }
                    } else {
                        status = SUCCESS;
                    }
                }
                if (nquad > 0 && nquad < MODL->body[jbody].nface) {
                    SPRINT3(1, "WARNING:: only %d of %d Faces were quadded for Body %d",
                            nquad, MODL->body[jbody].nface, jbody);
                    (MODL->nwarn)++;
                }
            }

            /* find the minimum number of points in any Face tessellation */
            count = 0;
            for (iface = 1; iface <= MODL->body[jbody].nface; iface++) {
                status = EG_getTessFace(MODL->body[jbody].etess, iface,
                                        &npnt_face, &xyz_face, &uv_face, &ptype, &pindx,
                                        &ntri_face, &tris, &tric);
                if (status == EGADS_NODATA) {               /* can happen if .tParams[0]<0 */
                    status = EGADS_SUCCESS;
                    continue;
                }
                CHECK_STATUS(EG_getTessface);

                if (npnt_face <= 0) count++;
            }

            /* there is an external grid generator */
            if (STRLEN(MODL->eggname) > 0 && count > 0) {
                SPRINT3(1, "WARNING:: external tessellator skipped for Body %d because %d of %d Faces were not tessellated",
                        jbody, count, MODL->body[jbody].nface);
                (MODL->nwarn)++;

            } else if (STRLEN(MODL->eggname) > 0) {

                /* open the current tessellation for editing */
                status = EG_openTessBody(MODL->body[jbody].etess);
                CHECK_STATUS(EG_openTessBody);

                /* loop through each face and make an alternative tessellation */
                for (iface = 1; iface <= MODL->body[jbody].nface; iface++) {

                    /* get the EGADS tessellation */
                    status = EG_getTessFace(MODL->body[jbody].etess, iface,
                                            &npnt_face, &xyz_face, &uv_face, &ptype, &pindx,
                                            &ntri_face, &tris, &tric);
                    CHECK_STATUS(EG_getTessface);

                    /* set up the Loop information */
                    status = EG_getTopology(MODL->body[jbody].face[iface].eface, &eref,
                                            &oclass, &mtype_face, uvlims, &nlup, &elups, &senses);
                    CHECK_STATUS(EG_getTopology);

                    nbnd = 0;
                    for (ilup = 0; ilup < nlup; ilup++) {
                        lup[ilup] = 0;

                        status = EG_getTopology(elups[ilup], &eref,
                                                &oclass, &mtype, uvlims, &nedg, &eedgs, &senses);
                        CHECK_STATUS(EG_getTopology);

                        for (ii = 0; ii < nedg; ii++) {
                            iedge = EG_indexBodyTopo(MODL->body[jbody].ebody, eedgs[ii]);

                            status = EG_getTopology(eedgs[ii], &eref, &oclass, &mtype,
                                                    data, &nchild, &echilds, &senses);
                            CHECK_STATUS(EG_getTopology);

                            /* degenerate Edge has 2 Points, but only one of them
                               is in the Loop */
                            if (mtype == DEGENERATE) continue;

                            status = EG_getTessEdge(MODL->body[jbody].etess, iedge,
                                                    &npnt_edge, &xyz_edge, &t_edge);
                            CHECK_STATUS(eg_getTessEdge);

                            lup[ilup] += (npnt_edge - 1);
                        }
                        nbnd += lup[ilup];
                    }

                    lup[nlup] = 0;

                    /* if all the points are Boundary Nodes, there are no Nodes
                       that are internal to the Face that need to be generated */
                    if (nbnd == npnt_face) continue;

                    /* get enough space to hold the Boundary points */
                    MALLOC(uv_new, double, 2*npnt_face);

                    /* set up boundary points */
                    for (ipnt = 0; ipnt < nbnd; ipnt++) {
                        uv_new[2*ipnt  ] = uv_face[2*ipnt  ];
                        uv_new[2*ipnt+1] = uv_face[2*ipnt+1];
                    }

                    /* generate the new grid */
                    status = MODL->eggGenerate(uv_new, lup, &eggdata);
                    /* CHECK_STATUS below */

                    FREE(uv_new);

                    /* if a successful tessellation, put the tessellation info into
                       EGADS etess object */
                    if (status == SUCCESS) {
                        status = MODL->eggInfo(eggdata, &npnt_egg, &nbnd_egg, &uv_egg, &p_egg,
                                               &ntri_egg, &tris_egg);
                        CHECK_STATUS(eggInfo);

                        SPRINT5(1, "WARNING:: using eggGenerate for Face %3d:%-5d  (npnt=%5d, nbnd=%5d, ntri=%5d)",
                                jbody, iface, npnt_egg, nbnd_egg, ntri_egg);
                        (MODL->nwarn)++;

                        MALLOC(uv_new,   double, 2*npnt_egg);
                        MALLOC(xyz_new,  double, 3*npnt_egg);
                        MALLOC(tris_new, int,    3*ntri_egg);

                        for (ipnt = 0; ipnt < nbnd_egg; ipnt++) {
                            uv_new[ 2*ipnt  ] = uv_face[ 2*ipnt  ];
                            uv_new[ 2*ipnt+1] = uv_face[ 2*ipnt+1];
                            xyz_new[3*ipnt  ] = xyz_face[3*ipnt  ];
                            xyz_new[3*ipnt+1] = xyz_face[3*ipnt+1];
                            xyz_new[3*ipnt+2] = xyz_face[3*ipnt+2];
                        }

                        for (ipnt = nbnd_egg; ipnt < npnt_egg; ipnt++) {
                            uv_new[2*ipnt  ] = uv_egg[2*ipnt  ];
                            uv_new[2*ipnt+1] = uv_egg[2*ipnt+1];

                            status = EG_evaluate(MODL->body[jbody].face[iface].eface,
                                                 &(uv_new[2*ipnt]), data);
                            CHECK_STATUS(EG_evaluate);

                            xyz_new[3*ipnt  ] = data[0];
                            xyz_new[3*ipnt+1] = data[1];
                            xyz_new[3*ipnt+2] = data[2];
                        }

                        if (mtype_face == SFORWARD) {
                            for (itri = 0; itri < ntri_egg; itri++) {
                                tris_new[3*itri  ] = tris_egg[3*itri  ] + 1;
                                tris_new[3*itri+1] = tris_egg[3*itri+1] + 1;
                                tris_new[3*itri+2] = tris_egg[3*itri+2] + 1;
                            }
                        } else {
                            for (itri = 0; itri < ntri_egg; itri++) {
                                tris_new[3*itri  ] = tris_egg[3*itri+2] + 1;
                                tris_new[3*itri+1] = tris_egg[3*itri+1] + 1;
                                tris_new[3*itri+2] = tris_egg[3*itri  ] + 1;
                            }
                        }

                        status = EG_setTessFace(MODL->body[jbody].etess, iface,
                                                npnt_egg, xyz_new, uv_new,
                                                ntri_egg, tris_new);
                        CHECK_STATUS(EG_setTessFace);

                        FREE(tris_new);
                        FREE(uv_new  );
                        FREE(xyz_new );

                        /* remember the new egg data */
                        MODL->body[jbody].face[iface].eggdata = (void*) eggdata;

                        /* unsuccessful external tessellator */
                    } else {
                        SPRINT2(1, "WARNING:: external tessellator skipped for Face %d:%d", jbody, iface);
                        SPRINT0(1, "          reverting to EGADS");
                        (MODL->nwarn)++;

                        status = MODL->eggFree(eggdata);
                        CHECK_STATUS(eggFree);
                    }
                }

                /* cheking the status of the etess closes it */
                status = EG_statusTessBody(MODL->body[jbody].etess, &ebody, &state, &npts);
                CHECK_STATUS(EG_statusTessBody);
            }

            /* print out bounding box info if a Face contains no Triangles */
            for (iface = 1; iface <= MODL->body[jbody].nface; iface++) {
                status = EG_getTessFace(MODL->body[jbody].etess, iface,
                                        &npnt_face, &xyz_face, &uv_face, &ptype, &pindx,
                                        &ntri_face, &tris, &tric);
                if (status == EGADS_NODATA) {               /* can happen if .tParams[0]<0 */
                    status = EGADS_SUCCESS;
                    continue;
                }
                CHECK_STATUS(EG_getTessFace);

                if (ntri_face <= 0) {
                    status = EG_getBoundingBox(MODL->body[jbody].face[iface].eface, bbox);
                    CHECK_STATUS(EG_getBoundingBox);

                    SPRINT4(1, "WARNING:: Face %3d:%-4d has no triangles,  %12.4f <= x <= %12.4f", jbody, iface, bbox[0], bbox[3]);
                    SPRINT2(1, "                                           %12.4f <= y <= %12.4f",               bbox[1], bbox[4]);
                    SPRINT2(1, "                                           %12.4f <= z <= %12.4f",               bbox[2], bbox[5]);

                    status = EG_getTopology(MODL->body[jbody].face[iface].eface, &esurface, &oclass, &mtype,
                                            data, &nloop, &eloops, &senses);
                    CHECK_STATUS(EG_getTopology);

                    status = EG_getInfo(esurface, &oclass, &mtype_face, NULL, NULL, NULL);
                    CHECK_STATUS(EG_getInfo);

                    for (iloop = 0; iloop < nloop; iloop++) {
                        SPRINT1(1, "          Loop %d", iloop+1);

                        status = EG_getTopology(eloops[iloop], &eref, &oclass, &mtype, data, &nedge, &eedges, &senses);
                        CHECK_STATUS(EG_getTopology);

                        if (mtype_face != PLANE) {
                            for (i = 0; i < nedge; i++) {
                                iedge = status = EG_indexBodyTopo(MODL->body[jbody].ebody, eedges[i]);
                                CHECK_STATUS(EG_indexBodyTopo);

                                status = EG_getRange(eedges[i], trange, &periodic);
                                CHECK_STATUS(EG_getRange);

                                status = EG_evaluate(eedges[i+nedge], &trange[0], data );
                                CHECK_STATUS(EG_evaluate);

                                status = EG_evaluate(eedges[i+nedge], &trange[1], data2);
                                CHECK_STATUS(EG_evaluate);

                                if (senses[i] == SFORWARD) {
                                    SPRINT5(1, "               Edge %4d: beg=%12.5f %12.5f, end=%12.5f %12.5f",
                                            iedge, data[0], data[1], data2[0], data2[1]);
                                } else {
                                    SPRINT5(1, "               Edge %4d: end=%12.5f %12.5f, beg=%12.5f %12.5f",
                                            iedge, data2[0], data2[1], data[0], data[1]);
                                }
                            }

                            SPRINT0(1, "          Use \"make grafic\" to see the Loops on this Face");
                        } else {
                            for (i = 0; i < nedge; i++) {
                                iedge = status = EG_indexBodyTopo(MODL->body[jbody].ebody, eedges[i]);
                                CHECK_STATUS(EG_indexBodyTopo);

                                status = EG_getRange(eedges[i], trange, &periodic);
                                CHECK_STATUS(EG_getRange);

                                status = EG_evaluate(eedges[i], &trange[0], data);
                                CHECK_STATUS(EG_evaluate);

                                status = EG_invEvaluate(esurface, data, uv1, xyz_out);
                                CHECK_STATUS(EG_invEvaluate);

                                status = EG_evaluate(eedges[i], &trange[1], data);
                                CHECK_STATUS(EG_evaluate);

                                status = EG_invEvaluate(esurface, data, uv2, xyz_out);
                                CHECK_STATUS(EG_invEvaluate);

                                if (senses[i] == SFORWARD) {
                                    SPRINT5(1, "               Edge %4d: beg=%12.5f %12.5f, end=%12.5f %12.5f",
                                            iedge, uv1[0], uv1[1], uv2[0], uv2[1]);
                                } else {
                                    SPRINT5(1, "               Edge %4d: end=%12.5f %12.5f, beg=%12.5f %12.5f",
                                            iedge, uv2[0], uv2[1], uv1[0], uv1[1]);
                                }
                            }
                        }
                    }

                    (MODL->nwarn)++;

#ifdef GRAFIC
                    if (mtype_face != PLANE) {
                        int io_kbd=5, io_scr=6, indgr=1+2+4+16+64, isymb=0;
                        char  pltitl[80];

                        grinit_(&io_kbd, &io_scr, "plotTile", STRLEN("plotTile"));

                        snprintf(pltitl, 80, "~u~v~Face %d:%d", jbody, iface);
                        grctrl_(plotFace, &indgr, pltitl,
                                (void*)(MODL),
                                (void*)(&jbody),
                                (void*)(&iface),
                                (void*)(&isymb),
                                (void*)NULL,
                                (void*)NULL,
                                (void*)NULL,
                                (void*)NULL,
                                (void*)NULL,
                                (void*)NULL,
                                STRLEN(pltitl));
                    } else {
                        SPRINT0(0, "planar Face is not shown with GRAFIC");
                    }
#endif
                }
            }

        /* this is a perturbed MODL */
        } else {

            /* map BASE jbody's tessellation onto the current MODL jbody */
            if (BASE->body[jbody].etess != NULL) {
                if (MODL->body[jbody].etess == NULL) {

                    /* check if the body needs to be mapped */
                    status = EG_attributeRet(MODL->body[jbody].ebody,
                                             "__mapBody__", &attrType, &attrLen,
                                             &tempIlist, &tempRlist, &tempClist);
                    if (status != EGADS_SUCCESS) {
                        newBody = NULL;
                        status = EG_mapBody(BASE->body[jbody].ebody, MODL->body[jbody].ebody,
                                            "_faceID", &newBody);
                        if (status == SUCCESS && newBody != NULL) {
                            if (PRINT_MAPPING_INFO == 1) {
                                status = EG_attributeRet(newBody, ".nMap", &attrType, &attrLen,
                                                         &tempIlist, &tempRlist, &tempClist);
                                if (status == SUCCESS && attrType == ATTRINT) {
                                    int    nnode1, nnode2, inode, jnode, nnode;
                                    double data1[18], maxdiff=0;
                                    ego    *enodes1, *enodes2, *enodes;

                                    status = EG_getBodyTopos(BASE->body[jbody].ebody, NULL, NODE, &nnode1, &enodes1);
                                    CHECK_STATUS(EG_getBodyTopos);
                                    status = EG_getBodyTopos(MODL->body[jbody].ebody, NULL, NODE, &nnode2, &enodes2);
                                    CHECK_STATUS(EG_getBodyTopos);

                                    for (inode = 0; inode < nnode1; inode++) {
                                        jnode = tempIlist[inode] - 1;
                                        status = EG_getTopology(enodes1[inode], &eref, &oclass, &mtype, data1, &nnode, &enodes, &senses);
                                        CHECK_STATUS(EG_getTopology);
                                        status = EG_getTopology(enodes2[jnode], &eref, &oclass, &mtype, data2, &nnode, &enodes, &senses);
                                        CHECK_STATUS(EG_getTopology);

                                        SPRINT5(1, "Node mapping:: BASE:%3d MODL:%3d diff: %12.6f %12.6f %12.6f",
                                                inode+1, jnode+1, data1[0]-data2[0], data1[1]-data2[1], data1[2]-data2[2]);
                                        maxdiff = MAX(maxdiff, fabs(data1[0]-data2[0]));
                                        maxdiff = MAX(maxdiff, fabs(data1[1]-data2[1]));
                                        maxdiff = MAX(maxdiff, fabs(data1[2]-data2[2]));
                                    }
                                    SPRINT1(1, "Node maxdiff=%12.6f", maxdiff);

                                    EG_free(enodes1);
                                    EG_free(enodes2);
                                }

                                status = EG_attributeRet(newBody, ".eMap", &attrType, &attrLen,
                                                         &tempIlist, &tempRlist, &tempClist);
                                if (status == SUCCESS && attrType == ATTRINT) {
                                    int    nedge1, nedge2, jedge;
                                    double bbox1[6], bbox2[6], maxdiff=0;
                                    ego    *eedges1, *eedges2;

                                    status = EG_getBodyTopos(BASE->body[jbody].ebody, NULL, EDGE, &nedge1, &eedges1);
                                    CHECK_STATUS(EG_getBodyTopos);
                                    status = EG_getBodyTopos(MODL->body[jbody].ebody, NULL, EDGE, &nedge2, &eedges2);
                                    CHECK_STATUS(EG_getBodyTopos);

                                    for (iedge = 0; iedge < nedge1; iedge++) {
                                        jedge = tempIlist[iedge] - 1;
                                        status = EG_getBoundingBox(eedges1[iedge], bbox1);
                                        CHECK_STATUS(EG_getBoundingBox);
                                        status = EG_getBoundingBox(eedges2[jedge], bbox2);
                                        CHECK_STATUS(EG_getTopology);

                                        SPRINT8(1, "Edge mapping:: BASE:%3d MODL:%3d diff: %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f",
                                               iedge+1, jedge+1, bbox1[0]-bbox2[0], bbox1[1]-bbox2[1], bbox1[2]-bbox2[2],
                                                                 bbox1[3]-bbox2[3], bbox1[4]-bbox2[4], bbox1[5]-bbox2[5]);
                                        maxdiff = MAX(maxdiff, fabs(bbox1[0]-bbox2[0]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[1]-bbox2[1]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[2]-bbox2[2]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[3]-bbox2[3]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[4]-bbox2[4]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[5]-bbox2[5]));
                                    }
                                    SPRINT1(1, "Edge maxdiff=%12.6f", maxdiff);

                                    EG_free(eedges1);
                                    EG_free(eedges2);
                                }

                                status = EG_attributeRet(newBody, ".fMap", &attrType, &attrLen,
                                                         &tempIlist, &tempRlist, &tempClist);
                                if (status == SUCCESS && attrType == ATTRINT) {
                                    int    nface1, nface2, jface;
                                    double bbox1[6], bbox2[6], maxdiff=0;
                                    ego    *efaces1, *efaces2;

                                    status = EG_getBodyTopos(BASE->body[jbody].ebody, NULL, FACE, &nface1, &efaces1);
                                    CHECK_STATUS(EG_getBodyTopos);
                                    status = EG_getBodyTopos(MODL->body[jbody].ebody, NULL, FACE, &nface2, &efaces2);
                                    CHECK_STATUS(EG_getBodyTopos);

                                    for (iface = 0; iface < nface1; iface++) {
                                        jface = tempIlist[iface] - 1;
                                        status = EG_getBoundingBox(efaces1[iface], bbox1);
                                        CHECK_STATUS(EG_getBoundingBox);
                                        status = EG_getBoundingBox(efaces2[jface], bbox2);
                                        CHECK_STATUS(EG_getTopology);

                                        SPRINT8(1, "Face mapping:: BASE:%3d MODL:%3d diff: %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f",
                                               iface+1, jface+1, bbox1[0]-bbox2[0], bbox1[1]-bbox2[1], bbox1[2]-bbox2[2],
                                                                 bbox1[3]-bbox2[3], bbox1[4]-bbox2[4], bbox1[5]-bbox2[5]);
                                        maxdiff = MAX(maxdiff, fabs(bbox1[0]-bbox2[0]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[1]-bbox2[1]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[2]-bbox2[2]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[3]-bbox2[3]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[4]-bbox2[4]));
                                        maxdiff = MAX(maxdiff, fabs(bbox1[5]-bbox2[5]));
                                    }
                                    SPRINT1(1, "Face maxdiff=%12.6f", maxdiff);

                                    EG_free(efaces1);
                                    EG_free(efaces2);
                                }
                            }

//$$$                        // &&&&& do we need to clean up udp cache here?

                            EG_deleteObject(MODL->body[jbody].ebody);
                            MODL->body[jbody].ebody = newBody;

                            status = finishBody(MODL,jbody);
                            CHECK_STATUS(finishBody);
                        }

                        /* tag that the body has been mapped */
                        status = EG_attributeAdd(MODL->body[jbody].ebody, "__mapBody__", ATTRINT,
                                                 1, &jbody, NULL, NULL);
                        CHECK_STATUS(EG_attributeAdd);
                    }

                    status = EG_mapTessBody(BASE->body[jbody].etess, MODL->body[jbody].ebody,
                                            &(MODL->body[jbody].etess));
                    CHECK_STATUS(EG_mapTessBody);
                }
            }

            /* if there is a registered external grid generator, edit the tessellation
               that was made above */
            if (STRLEN(BASE->eggname) > 0 && MORPH_GRID == 1) {

                /* open the current tessellation for editing */
                status = EG_openTessBody(MODL->body[jbody].etess);
                CHECK_STATUS(EG_openTessBody);

                for (iface = 1; iface <= MODL->body[jbody].nface; iface++) {
                    eggdata = BASE->body[jbody].face[iface].eggdata;

                    /* call the external grid generator to put the
                       morphed tessellation onto the Faces of the ptrb Body */
                    if (eggdata != NULL) {
                        status = BASE->eggInfo(eggdata, &npnt_egg, &nbnd_egg, &uv_egg, &p_egg,
                                               &ntri_egg, &tris_egg);
                        CHECK_STATUS(eggInfo);

                        SPRINT5(1, "WARNING:: using eggMorph for Face %3d:%-5d  (npnt=%5d, nbnd=%5d, ntri=%5d)",
                                jbody, iface, npnt_egg, nbnd_egg, ntri_egg);
                        (MODL->nwarn)++;

                        status = EG_getTessFace(MODL->body[jbody].etess, iface,
                                                &npnt_face, &xyz_face, &uv_face, &ptype, &pindx,
                                                &ntri_face, &tris, &tric);
                        CHECK_STATUS(EG_getTessFace);

                        MALLOC(xyz_new, double, 3*npnt_egg);
                        MALLOC(uv_new,  double, 2*npnt_egg);

                        for (ipnt = 0; ipnt < nbnd_egg; ipnt++) {
                            uv_new[ 2*ipnt  ] = uv_face[ 2*ipnt ];
                            uv_new[ 2*ipnt+1] = uv_face[ 2*ipnt+1];
                            xyz_new[3*ipnt  ] = xyz_face[3*ipnt  ];
                            xyz_new[3*ipnt+1] = xyz_face[3*ipnt+1];
                            xyz_new[3*ipnt+2] = xyz_face[3*ipnt+2];
                        }

                        status = BASE->eggMorph(eggdata, uv_new, &eggdata_new);
                        CHECK_STATUS(eggMorph);

                        status = BASE->eggFree(eggdata_new);
                        CHECK_STATUS(eggFree);

                        for (ipnt = nbnd_egg; ipnt < npnt_egg; ipnt++) {
                            status = EG_evaluate(MODL->body[jbody].face[iface].eface,
                                                 &(uv_new[2*ipnt]), data);
                            CHECK_STATUS(EG_evaluate);

                            xyz_new[3*ipnt  ] = data[0];
                            xyz_new[3*ipnt+1] = data[1];
                            xyz_new[3*ipnt+2] = data[2];
                        }

                        status = EG_setTessFace(MODL->body[jbody].etess, iface,
                                                npnt_face, xyz_new, uv_new,
                                                ntri_face, tris);
                        CHECK_STATUS(EG_setTessFace);

                        FREE(xyz_new);
                        FREE(uv_new);
                    }
                }

                /* cheking the status of the etess closes it */
                status = EG_statusTessBody(MODL->body[jbody].etess, &ebody, &state, &npts);
                CHECK_STATUS(EG_statusTessBody);

                SPRINT1(1, "--> EG_statusTessBody -> status=%d", status);
            }
        }

        /* if there were no quads, create the global IDs */
        if (nquad == 0) {
            MODL->body[jbody].npnts = MODL->body[jbody].nnode;
            MODL->body[jbody].ntris = 0;

            for (iedge = 1; iedge <= MODL->body[jbody].nedge; iedge++) {
                if (MODL->body[jbody].edge[iedge].itype == DEGENERATE) continue;

                MODL->body[jbody].edge[iedge].globid = MODL->body[jbody].npnts - 1;

                status = EG_getTessEdge(MODL->body[jbody].etess, iedge,
                                        &npnt_edge, &xyz_edge, &t_edge);
                CHECK_STATUS(EG_getTessEdge);

                MODL->body[jbody].npnts += (npnt_edge-2);
            }

            for (iface = 1; iface <= MODL->body[jbody].nface; iface++) {
                status = EG_getTessFace(MODL->body[jbody].etess, iface,
                                        &npnt_face, &xyz_face, &uv_face, &ptype, &pindx,
                                        &ntri_face, &tris, &tric);
                if (status == EGADS_NODATA) {               /* can happen if .tParams[0]<0 */
                    status = EGADS_SUCCESS;
                    continue;
                }
                CHECK_STATUS(EG_getTessFace);

                for (ipnt = 0; ipnt < npnt_face; ipnt++) {
                    if (ptype[ipnt] < 0) {
                        MODL->body[jbody].face[iface].globid = MODL->body[jbody].npnts - ipnt + 1;
                        MODL->body[jbody].npnts += (npnt_face - ipnt);
                        break;
                    }
                }

                MODL->body[jbody].ntris += ntri_face;
            }

            /* otherwise (there are quads), so initialize global IDs to zero */
        } else {
            MODL->body[jbody].npnts = 0;
            MODL->body[jbody].ntris = 0;

            for (iedge = 1; iedge <= MODL->body[jbody].nedge; iedge++) {
                if (MODL->body[jbody].edge[iedge].itype == DEGENERATE) continue;

                MODL->body[jbody].edge[iedge].globid = 0;
            }
            for (iface = 1; iface <= MODL->body[jbody].nface; iface++) {
                MODL->body[jbody].face[iface].globid = 0;
            }
        }
    }

cleanup:
    FREE(uv_new);
    FREE(xyz_new);
    FREE(tris_new);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmBodyDetails - get information about one Body                   *
 *                                                                      *
 ************************************************************************
 */

int
ocsmBodyDetails(void   *modl,           /* (in)  pointer to MODL */
                char   filename[],      /* (in)  name of file from which Body was created */
                int    linenum,         /* (in)  line in filename from which Body was created */
                char   *info[])         /* (out) info about the Body (freeable) */
{
    int       status = SUCCESS;         /* (out) return status */

    int       ibody, jbody, ibrch, istor, i, attrType, attrLen, mchar;
    CINT      *tempIlist;
    CDOUBLE   *tempRlist;
    char      line[MAX_LINE_LEN+1], onstack, name[MAX_NAME_LEN];
    CCHAR     *tempClist;
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmBodyDetails);

    /* --------------------------------------------------------------- */

    /* initialize output buffer */
    mchar = 1000;
    MALLOC(*info, char, mchar);
    *info[0] = '\0';

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* find the Body that matches filename:linenum */
    ibody = 0;
    for (jbody = 1; jbody <= MODL->nbody; jbody++) {
        ibrch = MODL->body[jbody].ibrch;

        if (strcmp(MODL->brch[ibrch].filename, filename) == 0      &&
                   MODL->brch[ibrch].linenum             == linenum  ) {
            ibody = jbody;
            break;
        }
    }

    /* if no Body created by given line, list all Bodys */
    if (ibody == 0) {
        snprintf(line, MAX_LINE_LEN, "Listing of all Bodys (* if on stack):\n");

        if (strlen(*info)+strlen(line) >= mchar) {
            mchar += strlen(line) + 1000;
            RALLOC(*info, char, mchar);
        }
        strcat(*info, line);

        for (ibody = 1; ibody <= MODL->nbody; ibody++) {
            if (MODL->body[ibody].onstack == 0) {
                onstack = ' ';
            } else {
                onstack = '*';
            }

            status = EG_attributeRet(MODL->body[ibody].ebody, "_name", &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS && attrType == ATTRSTRING) {
                snprintf(name, 19, "%s", tempClist);
            } else {
                status = SUCCESS;
                snprintf(name, 19, "Body %d", ibody);
            }

            ibrch = MODL->body[ibody].ibrch;

            snprintf(line, MAX_LINE_LEN, "%c %-20s %-10s (%-10s @ [[%s:%d]])\n", onstack,
                     name, ocsmGetText(MODL->body[ibody].botype), ocsmGetText(MODL->body[ibody].brtype),
                     MODL->brch[ibrch].filename, MODL->brch[ibrch].linenum);

            if (strlen(*info)+strlen(line) >= mchar) {
                mchar += strlen(line) + 1000;
                RALLOC(*info, char, mchar);
            }
            strcat(*info, line);

            /* add storage info if it exists */
            for (istor = 0; istor < MODL->nstor; istor++) {
                for (i = 0; i < MODL->stor[istor].nbody; i++) {
                    if (ibody == MODL->stor[istor].ibody[i]) {
                        snprintf(line, MAX_LINE_LEN, "        stored in %-20s %d\n",
                                 MODL->stor[istor].name, MODL->stor[istor].index);

                        if (strlen(*info)+strlen(line) >= mchar) {
                            mchar += strlen(line) + 1000;
                            RALLOC(*info, char, mchar);
                        }
                        strcat(*info, line);
                    }
                }
            }
        }

    /* otherwise, return info for a specific Body */
    } else {
        status = EG_attributeRet(MODL->body[ibody].ebody, "_name", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS && attrType == ATTRSTRING) {
            snprintf(name, 19, "%s", tempClist);
        } else {
            status = SUCCESS;
            snprintf(name, 19, "Body %d", ibody);
        }

        snprintf(line, MAX_LINE_LEN, "%-20s          (%-10s @ [[%s:%d]])\n", name,
                 ocsmGetText(MODL->body[ibody].brtype), filename, linenum);

        if (strlen(*info)+strlen(line) >= mchar) {
            mchar += strlen(line) + 1000;
            RALLOC(*info, char, mchar);
        }
        strcat(*info, line);

        /* find all the Bodys that are used by this Body */
        for (jbody = 1; jbody <= MODL->nbody; jbody++) {
            if (jbody == ibody) continue;

            if (MODL->body[ibody].ileft == jbody ||
                MODL->body[ibody].irite == jbody ||
                MODL->body[jbody].ichld == ibody   ) {
                status = EG_attributeRet(MODL->body[jbody].ebody, "_name", &attrType, &attrLen,
                                         &tempIlist, &tempRlist, &tempClist);
                if (status == EGADS_SUCCESS && attrType == ATTRSTRING) {
                    snprintf(name, 19, "%s", tempClist);
                } else {
                    status = SUCCESS;
                    snprintf(name, 19, "Body %d", jbody);
                }

                ibrch = MODL->body[jbody].ibrch;

                snprintf(line, MAX_LINE_LEN, "uses:    %-20s (%-10s @ [[%s:%d]])\n", name,
                         ocsmGetText(MODL->brch[ibrch].type),
                         MODL->brch[ibrch].filename, MODL->brch[ibrch].linenum);

                if (strlen(*info)+strlen(line) >= mchar) {
                    mchar += strlen(line) + 1000;
                    RALLOC(*info, char, mchar);
                }
                strcat(*info, line);
            }
        }

        /* find all the Bodys that us this Body */
        for (jbody = 1; jbody <= MODL->nbody; jbody++) {
            if (jbody == ibody) continue;

            if (MODL->body[jbody].ileft == ibody ||
                MODL->body[jbody].irite == ibody ||
                MODL->body[ibody].ichld == jbody   ) {
                status = EG_attributeRet(MODL->body[jbody].ebody, "_name", &attrType, &attrLen,
                                         &tempIlist, &tempRlist, &tempClist);
                if (status == EGADS_SUCCESS && attrType == ATTRSTRING) {
                    snprintf(name, 19, "%s", tempClist);
                } else {
                    status = SUCCESS;
                    snprintf(name, 19, "Body %d", jbody);
                }

                ibrch = MODL->body[jbody].ibrch;

                snprintf(line, MAX_LINE_LEN, "used by: %-20s (%-10s @ [[%s:%d]])\n", name,
                         ocsmGetText(MODL->brch[ibrch].type),
                         MODL->brch[ibrch].filename, MODL->brch[ibrch].linenum);

                if (strlen(*info)+strlen(line) >= mchar) {
                    mchar += strlen(line) + 1000;
                    RALLOC(*info, char, mchar);
                }
                strcat(*info, line);
            }
        }
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmPerturb - create a perturbed MODL                              *
 *                                                                      *
 ************************************************************************
 */

int
ocsmPerturb(void   *modl,               /* (in)  pointer to MODL */
            int    npmtrs,              /* (in)  number of perturbed Parameters (or 0 to remove)*/
  /*@null@*/int    ipmtrs[],            /* (in)  array of Parameter indices (1:npmtr) */
  /*@null@*/int    irows[],             /* (in)  array of row       indices (1:nrow) */
  /*@null@*/int    icols[],             /* (in)  array of column    indices (1:ncol) */
  /*@null@*/double values[])            /* (in)  array of perturbed values */
{
    int       status = SUCCESS;         /* (out) return status */

    int       ii, ibody, builtTo, nbody_ptrb;
    modl_T    *MODL = (modl_T*)modl;
    modl_T    *PTRB = NULL;

    void      *modl_ptrb=NULL;

    ROUTINE(ocsmPerturb);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* if npmtrs is non-positive, remove the perturbation */
    if (npmtrs <= 0) {
        status = removePerturbation(MODL);
        CHECK_STATUS(removePerturbation);
        goto cleanup;
    }

    /* if there is no tessellation for any Body on the stack, return error */
    for (ibody = 1; ibody <= MODL->nbody; ibody++) {
        if (MODL->body[ibody].onstack == 0) continue;

        if (MODL->body[ibody].etess == NULL && MODL->body[ibody].botype != OCSM_NODE_BODY) {
            status = OCSM_NEED_TESSELLATION;
            goto cleanup;
        }
    }

    /* make a copy and set its driving parameters to "base + dtime * dot" */
    status = ocsmCopy(MODL, &modl_ptrb);
    CHECK_STATUS(ocsmCopy);

    PTRB = modl_ptrb;

    SPLINT_CHECK_FOR_NULL(PTRB);

    MODL->perturb  = PTRB;
    PTRB->basemodl = MODL;

    /* update the Parameters */
    if (npmtrs > 0) {
        SPLINT_CHECK_FOR_NULL(ipmtrs);
        SPLINT_CHECK_FOR_NULL(irows );
        SPLINT_CHECK_FOR_NULL(icols );
        SPLINT_CHECK_FOR_NULL(values);

        for (ii = 0; ii < npmtrs; ii++) {
            status = ocsmSetValuD(PTRB, ipmtrs[ii], irows[ii], icols[ii], values[ii]);
            CHECK_STATUS(ocsmSetValuD);
        }
    }

    /* recheck (this is needed to make sure that the indentation is updated) */
    status = ocsmCheck(PTRB);
    CHECK_STATUS(ocsmCheck);

    /* rebuild the perturbed copy */
    nbody_ptrb = 0;
    SPRINT0(1, ">>> building perturbation");
    status = ocsmBuild(PTRB, 0, &builtTo, &nbody_ptrb, NULL);
    CHECK_STATUS(ocsmBuild);

    /* check that base and ptrb have same number of Bodys */
    if (MODL->nbody > PTRB->nbody) {
        SPRINT0(1, "WARNING:: Base and perturbed models have different nbody");
        SPRINT1(1, "          MODL->nbody = %d", MODL->nbody);
        SPRINT1(1, "          PTRB->nbody = %d", PTRB->nbody);
        (MODL->nwarn)++;
    }

    /* check that base and ptrb Bodys match topologically */
    for (ibody = 1; ibody <= PTRB->nbody; ibody++) {
        if (MODL->body[ibody].onstack != 1) continue;

        status = EG_mapBody2(MODL->body[ibody].ebody, "_faceID", "_edgeID", PTRB->body[ibody].ebody);
        if (status != EGADS_SUCCESS) {
            SPRINT2(0, "WARNING:: EG_mapBody2(ibody=%d) -> status=%d\n", ibody, status);
            (MODL->nwarn)++;
            status = OCSM_DID_NOT_CREATE_BODY;
            goto cleanup;
        }
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmUpdateTess - update a tessellation from a file                 *
 *                                                                      *
 ************************************************************************
 */

int
ocsmUpdateTess(void   *modl,            /* (in)  pointer to MODL */
               int    ibody,            /* (in)  Body index (1:nbody) */
               char   filename[])       /* (in)  name of .tess file */
{
    int       status = SUCCESS;         /* (out) return status */

    int       nnode, inode, nedge, iedge, nface, iface, npnt, ipnt, ntri, itri;
    int       ptype, pindx, state, npts;
    int       *tris=NULL, tric0, tric1, tric2;
    double    x, y, z, *xyz=NULL, *t=NULL, *uv=NULL;
    ego       newTess;
    FILE      *fp=NULL;

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmUpdateTess);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid Body index was given */
    if (ibody < 1 || ibody > MODL->nbody) {
        status = OCSM_BODY_NOT_FOUND;
        goto cleanup;
    }

    /* check that filename can be opened */
    fp = fopen(filename, "r");
    if (fp == NULL) {
        status = OCSM_FILE_NOT_FOUND;
        goto cleanup;
    }

    /* make sure that the number of Nodes, Edges, and Faces in file
       match the Body */
    fscanf(fp, "%d %d %d", &nnode, &nedge, &nface);

    if        (nnode != MODL->body[ibody].nnode) {
        SPRINT0(0, "ERROR:: nnode mismatch");
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    } else if (nedge != MODL->body[ibody].nedge) {
        SPRINT0(0, "ERROR:: nedge mismatch");
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    } else if (nface != MODL->body[ibody].nface) {
        SPRINT0(0, "ERROR:: nface mismatch");
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    }

    /* make sure there is a tessellation object */
    if (MODL->body[ibody].etess == NULL) {
        status = OCSM_NEED_TESSELLATION;
        goto cleanup;
    }

    /* open the tessellation object for editing */
    status = EG_initTessBody(MODL->body[ibody].ebody, &newTess);
    CHECK_STATUS(EG_initTessBody);


    /* make sure that the Node locations match */
    for (inode = 1; inode <= MODL->body[ibody].nnode; inode++) {
        fscanf(fp, "%lf %lf %lf", &x, &y, &z);

        if (fabs(MODL->body[ibody].node[inode].x-x) > EPS06 ||
            fabs(MODL->body[ibody].node[inode].y-y) > EPS06 ||
            fabs(MODL->body[ibody].node[inode].z-z) > EPS06   ) {
            SPRINT1(0, "ERROR:: Node %d mismatch", inode);
            status = OCSM_ILLEGAL_VALUE;
            goto cleanup;
        }
    }

    /* read and update the tessellation associated with each Edge */
    for (iedge = 1; iedge <= MODL->body[ibody].nedge; iedge++) {
        if (MODL->body[ibody].edge[iedge].itype == DEGENERATE) continue;

        fscanf(fp, "%d", &npnt);

        MALLOC(xyz, double, 3*npnt);
        MALLOC(t,   double,   npnt);

        for (ipnt = 0; ipnt < npnt; ipnt++) {
            fscanf(fp, "%lf %lf %lf %lf",
                   &xyz[3*ipnt], &xyz[3*ipnt+1], &xyz[3*ipnt+2], &t[ipnt]);
        }

        status = EG_setTessEdge(newTess, iedge, npnt, xyz, t);
        CHECK_STATUS(EG_setTessEdge);

        FREE(xyz);
        FREE(t  );
    }

    /* read and update the tessellation associated with each Face */
    for (iface = 1; iface <= MODL->body[ibody].nface; iface++) {
        fscanf(fp, "%d %d", &npnt, &ntri);

        MALLOC(xyz,  double, 3*npnt);
        MALLOC(uv,   double, 2*npnt);
        MALLOC(tris, int,    3*ntri);

        for (ipnt = 0; ipnt < npnt; ipnt++) {
            fscanf(fp, "%lf %lf %lf %lf %lf %d %d",
                   &xyz[3*ipnt], &xyz[3*ipnt+1], &xyz[3*ipnt+2],
                   &uv[ 2*ipnt], &uv[ 2*ipnt+1], &ptype, &pindx);
        }

        for (itri = 0; itri < ntri; itri++) {
            fscanf(fp, "%d %d %d %d %d %d",
                   &tris[3*itri], &tris[3*itri+1], &tris[3*itri+2],
                   &tric0, &tric1, &tric2);
        }

        status = EG_setTessFace(newTess, iface, npnt, xyz, uv, ntri, tris);
        CHECK_STATUS(EG_setTessFace);

        FREE(xyz );
        FREE(uv  );
        FREE(tris);
    }

    /* checking the status closes the tessellation */
    status = EG_statusTessBody(newTess, &MODL->body[ibody].ebody, &state, &npts);
    CHECK_STATUS(EG_statusTessBody);

    /* get rid of old tessellation and attach new tessellation */
    status = EG_deleteObject(MODL->body[ibody].etess);
    CHECK_STATUS(EG_deleteObject);

    MODL->body[ibody].etess = newTess;

cleanup:
    FREE(xyz );
    FREE(t   );
    FREE(uv  );
    FREE(tris);

    if (fp != NULL) fclose(fp);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmClearance - compute clearance between Bodys                    *
 *                                                                      *
 ************************************************************************
 */

int
ocsmClearance(void   *modl,             /* (in)  pointer to MODL */
              int    ibody1,            /* (in)  <0  Body index of enclosing   Body */
                                        /*       >0  Body index of neighboring Body */
              int    ibody2,            /* (in)  Body index of second Body */
              double *dist,             /* (out) >0 distance between Bodys */
                                        /*       <0 penetration distance */
              double pnt1[],            /* (out) closest point on ibody1 */
                                        /*       +iface, u,     v */
                                        /*       -iedge, t,     0 */
                                        /*       0,      inode, 0 */
              double pnt2[])            /* (out) closest point on ibody2 */
                                        /*       +iface, u,     v */
                                        /*       -iedge, t,     0 */
                                        /*       0,      inode, 0 */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmClearance);

    /* --------------------------------------------------------------- */

    /* default returns */
    *dist = -HUGEQ;                     // very bad result

    pnt1[0] = 0;
    pnt2[0] = 0;

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* SolidBody ibody1 and ibody2 are assumed to be neighbors */
    if (ibody1 > 0 && MODL->body[ibody1].botype == OCSM_SOLID_BODY) {
        status = distBodyNearSolid(MODL, ibody1, ibody2, dist, pnt1, pnt2);
        CHECK_STATUS(distBodyNearSolid);

    /* non-SolidBody ibody1 and ibody2 are assumed to be neighbors */
    } else if (ibody1 > 0) {
        status = distBodyNearBody(MODL, ibody1, ibody2, dist, pnt1, pnt2);
        CHECK_STATUS(distBodyNearBody);

    /* SolidBody ibody1 is assumed to enclose ibody2 */
    } else if (MODL->body[-ibody1].botype == OCSM_SOLID_BODY) {
        status = distBodyInSolid(MODL, -ibody1, ibody2, dist, pnt1, pnt2);
        CHECK_STATUS(distBodyInSolid);

    /* error */
    } else {
        status = OCSM_WRONG_TYPES_ON_STACK;
        goto cleanup;
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmShowTblOfContents - show matrix of "SPECIAL provide" statements *
 *                                                                      *
 ************************************************************************
 */

int
ocsmShowTblOfContents(void   *modl,     /* (in)  pointer to MODL */
                      int    irow,      /* (in)  dimension to be associated with rows (1-8) */
                      int    icol,      /* (in)  dimension to be associated with columns (1-8) */
                      char   *info[])   /* (out) info containing matrix *freeable) */
{
    int       status = SUCCESS;         /* (out) return status */

    int       mchar, ibrch, jrow, krow, mrow=0, nrow=0, jcol, kcol, mcol=0, ncol=0;
    int       *mat=NULL;
    char      line[MAX_LINE_LEN+1], row[MAX_NAME_LEN+1], col[MAX_NAME_LEN+1];
    char      **cols=NULL, **rows=NULL, swap[MAX_NAME_LEN];
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmShowTblOfContents);

    /* --------------------------------------------------------------- */

    /* initialize output buffer */
    mchar = 1000;
    MALLOC(*info, char, mchar);
    *info[0] = '\0';

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (irow < 1 || irow > 8) {
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    } else if (icol < 1 || icol > 8) {
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    }

    MALLOC(rows, char*, 1);
    MALLOC(cols, char*, 1);

    snprintf(line, MAX_LINE_LEN, "Table of contents:\n");
    strcat(*info, line);

    /* loop through the Branches, making a list of all the mentioned row and columns */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        if (MODL->brch[ibrch].type != OCSM_SPECIAL ||
            strcmp(MODL->brch[ibrch].arg1, "$provides") != 0) continue;

        /* make a new row name if it does not already exist */
        if        (irow == 1) {
            strncpy(row, &MODL->brch[ibrch].arg2[1], MAX_NAME_LEN);
        } else if (irow == 2) {
            strncpy(row, &MODL->brch[ibrch].arg3[1], MAX_NAME_LEN);
        } else if (irow == 3) {
            strncpy(row, &MODL->brch[ibrch].arg4[1], MAX_NAME_LEN);
        } else if (irow == 4) {
            strncpy(row, &MODL->brch[ibrch].arg5[1], MAX_NAME_LEN);
        } else if (irow == 5) {
            strncpy(row, &MODL->brch[ibrch].arg6[1], MAX_NAME_LEN);
        } else if (irow == 6) {
            strncpy(row, &MODL->brch[ibrch].arg7[1], MAX_NAME_LEN);
        } else if (irow == 7) {
            strncpy(row, &MODL->brch[ibrch].arg8[1], MAX_NAME_LEN);
        } else {
            strncpy(row, &MODL->brch[ibrch].arg9[1], MAX_NAME_LEN);
        }

        krow = -1;
        for (jrow = 0; jrow < nrow; jrow++) {
            if (strcmp(row, rows[jrow]) == 0) {
                krow = jrow;
                break;
            }
        }
        if (krow == -1) {
            RALLOC(rows, char*, nrow+1);
            rows[nrow] = NULL;

            MALLOC(rows[nrow], char, MAX_NAME_LEN+1);
            strncpy(rows[nrow], row, MAX_NAME_LEN);

            if (strlen(row) > mrow) mrow = strlen(row);

            nrow++;
        }

        /* make a new column name if it does not already exist */
        if        (icol == 1) {
            strncpy(col, &MODL->brch[ibrch].arg2[1], MAX_NAME_LEN);
        } else if (icol == 2) {
            strncpy(col, &MODL->brch[ibrch].arg3[1], MAX_NAME_LEN);
        } else if (icol == 3) {
            strncpy(col, &MODL->brch[ibrch].arg4[1], MAX_NAME_LEN);
        } else if (icol == 4) {
            strncpy(col, &MODL->brch[ibrch].arg5[1], MAX_NAME_LEN);
        } else if (icol == 5) {
            strncpy(col, &MODL->brch[ibrch].arg6[1], MAX_NAME_LEN);
        } else if (icol == 6) {
            strncpy(col, &MODL->brch[ibrch].arg7[1], MAX_NAME_LEN);
        } else if (icol == 7) {
            strncpy(col, &MODL->brch[ibrch].arg8[1], MAX_NAME_LEN);
        } else {
            strncpy(col, &MODL->brch[ibrch].arg9[1], MAX_NAME_LEN);
        }

        kcol = -1;
        for (jcol = 0; jcol < ncol; jcol++) {
            if (strcmp(col, cols[jcol]) == 0) {
                kcol = jcol;
                break;
            }
        }
        if (kcol == -1) {
            RALLOC(cols, char*, ncol+1);
            cols[ncol] = NULL;

            MALLOC(cols[ncol], char, MAX_NAME_LEN+1);
            strncpy(cols[ncol], col, MAX_NAME_LEN);

            if (strlen(col) > mcol) mcol = strlen(col);

            ncol++;
        }
    }

    /* alphabetize rows and cols */
    for (krow = 0; krow < nrow; krow++) {
        for (jrow = 1; jrow < nrow; jrow++) {
            if (strcmp(rows[jrow-1], rows[jrow]) > 0) {
                strcpy(swap,         rows[jrow  ]);
                strcpy(rows[jrow  ], rows[jrow-1]);
                strcpy(rows[jrow-1], swap        );
            }
        }
    }

    for (kcol = 0; kcol < ncol; kcol++) {
        for (jcol = 1; jcol < ncol; jcol++) {
            if (strcmp(cols[jcol-1], cols[jcol]) > 0) {
                strcpy(swap,         cols[jcol  ]);
                strcpy(cols[jcol  ], cols[jcol-1]);
                strcpy(cols[jcol-1], swap        );
            }
        }
    }

    /* make and fill a matrix that will tell is the row.column is provided */
    MALLOC(mat, int, nrow*ncol);

    for (krow = 0; krow < nrow; krow++) {
        for (kcol = 0; kcol < ncol; kcol++) {
            mat[krow*ncol+kcol] = 0;
        }
    }


    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        if (MODL->brch[ibrch].type != OCSM_SPECIAL ||
            strcmp(MODL->brch[ibrch].arg1, "$provides") != 0) continue;

        krow = -1;
        kcol = -1;

        if      (irow == 1) {
            for (jrow = 0; jrow < nrow; jrow++) {
                if (strcmp(rows[jrow], &MODL->brch[ibrch].arg2[1]) == 0) {
                    krow = jrow;
                    break;
                }
            }
        } else if (irow == 2) {
            for (jrow = 0; jrow < nrow; jrow++) {
                if (strcmp(rows[jrow], &MODL->brch[ibrch].arg3[1]) == 0) {
                    krow = jrow;
                    break;
                }
            }
        } else if (irow == 3) {
            for (jrow = 0; jrow < nrow; jrow++) {
                if (strcmp(rows[jrow], &MODL->brch[ibrch].arg4[1]) == 0) {
                    krow = jrow;
                    break;
                }
            }
        } else if (irow == 4) {
            for (jrow = 0; jrow < nrow; jrow++) {
                if (strcmp(rows[jrow], &MODL->brch[ibrch].arg5[1]) == 0) {
                    krow = jrow;
                    break;
                }
            }
        } else if (irow == 5) {
            for (jrow = 0; jrow < nrow; jrow++) {
                if (strcmp(rows[jrow], &MODL->brch[ibrch].arg6[1]) == 0) {
                    krow = jrow;
                    break;
                }
            }
        } else if (irow == 6) {
            for (jrow = 0; jrow < nrow; jrow++) {
                if (strcmp(rows[jrow], &MODL->brch[ibrch].arg7[1]) == 0) {
                    krow = jrow;
                    break;
                }
            }
        } else if (irow == 7) {
            for (jrow = 0; jrow < nrow; jrow++) {
                if (strcmp(rows[jrow], &MODL->brch[ibrch].arg8[1]) == 0) {
                    krow = jrow;
                    break;
                }
            }
        } else {
            for (jrow = 0; jrow < nrow; jrow++) {
                if (strcmp(rows[jrow], &MODL->brch[ibrch].arg9[1]) == 0) {
                    krow = jrow;
                    break;
                }
            }
        }

        if      (icol == 1) {
            for (jcol = 0; jcol < ncol; jcol++) {
                if (strcmp(cols[jcol], &MODL->brch[ibrch].arg2[1]) == 0) {
                    kcol = jcol;
                    break;
                }
            }
        } else if (icol == 2) {
            for (jcol = 0; jcol < ncol; jcol++) {
                if (strcmp(cols[jcol], &MODL->brch[ibrch].arg3[1]) == 0) {
                    kcol = jcol;
                    break;
                }
            }
        } else if (icol == 3) {
            for (jcol = 0; jcol < ncol; jcol++) {
                if (strcmp(cols[jcol], &MODL->brch[ibrch].arg4[1]) == 0) {
                    kcol = jcol;
                    break;
                }
            }
        } else if (icol == 4) {
            for (jcol = 0; jcol < ncol; jcol++) {
                if (strcmp(cols[jcol], &MODL->brch[ibrch].arg5[1]) == 0) {
                    kcol = jcol;
                    break;
                }
            }
        } else if (icol == 5) {
            for (jcol = 0; jcol < ncol; jcol++) {
                if (strcmp(cols[jcol], &MODL->brch[ibrch].arg6[1]) == 0) {
                    kcol = jcol;
                    break;
                }
            }
        } else if (icol == 6) {
            for (jcol = 0; jcol < ncol; jcol++) {
                if (strcmp(cols[jcol], &MODL->brch[ibrch].arg7[1]) == 0) {
                    kcol = jcol;
                    break;
                }
            }
        } else if (icol == 7) {
            for (jcol = 0; jcol < ncol; jcol++) {
                if (strcmp(cols[jcol], &MODL->brch[ibrch].arg8[1]) == 0) {
                    kcol = jcol;
                    break;
                }
            }
        } else {
            for (jcol = 0; jcol < ncol; jcol++) {
                if (strcmp(cols[jcol], &MODL->brch[ibrch].arg9[1]) == 0) {
                    kcol = jcol;
                    break;
                }
            }
        }

        mat[krow*ncol+kcol]++;
    }

    /* make sure *info is large enough */
    mchar += (nrow+1) * (ncol+1) * (mcol+3);
    RALLOC(*info, char, mchar);

    /* now print the matrix header */
    snprintf(line, MAX_LINE_LEN, "%*s  ", -mrow, " ");
    strcat(*info, line);

    for (kcol = 0; kcol < ncol; kcol++) {
        snprintf(line, MAX_LINE_LEN, "%*s  ", -mcol, cols[kcol]);
        strcat(*info, line);
    }
    strcat(*info, "\n");

    /* matrix body */
    for (krow = 0; krow < nrow; krow++) {
        snprintf(line, MAX_LINE_LEN, "%*s  ", -mrow, rows[krow]);
        strcat(*info, line);

        for (kcol = 0; kcol < ncol; kcol++) {
            if (mat[krow*ncol+kcol] > 0) {
                snprintf(line, MAX_LINE_LEN, "%*s  ", -mcol, "X");
            } else {
                snprintf(line, MAX_LINE_LEN, "%*s  ", -mcol, " ");
            }
            strcat(*info, line);
        }
        strcat(*info, "\n");
    }

    assert(strlen(*info) < mchar);

cleanup:
    for (jrow = 0; jrow < nrow; jrow++) {
        FREE(rows[jrow]);
    }
    FREE(rows);

    for (jcol = 0; jcol < ncol; jcol++) {
        FREE(cols[jcol]);
    }
    FREE(cols);

    FREE(mat);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmNewBrch - create a new Branch                                  *
 *                                                                      *
 ************************************************************************
 */

int
ocsmNewBrch(void   *modl,               /* (in)  pointer to MODL */
            int    iafter,              /* (in)  Branch index (0-nbrch) after which to add */
            int    type,                /* (in)  Branch type (see below) */
            char   filename[],          /* (in)  filename where Branch is defined */
            int    linenum,             /* (in)  linenumber where Branch is defined (bias-1) */
  /*@null@*/char   arg1[],              /* (in)  Argument 1 (or NULL) */
  /*@null@*/char   arg2[],              /* (in)  Argument 2 (or NULL) */
  /*@null@*/char   arg3[],              /* (in)  Argument 3 (or NULL) */
  /*@null@*/char   arg4[],              /* (in)  Argument 4 (or NULL) */
  /*@null@*/char   arg5[],              /* (in)  Argument 5 (or NULL) */
  /*@null@*/char   arg6[],              /* (in)  Argument 6 (or NULL) */
  /*@null@*/char   arg7[],              /* (in)  Argument 7 (or NULL) */
  /*@null@*/char   arg8[],              /* (in)  Argument 8 (or NULL) */
  /*@null@*/char   arg9[])              /* (in)  Argument 9 (or NULL) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       bclass, narg, ibrch, jbrch, impstr, i;
    char      temparg[MAX_LINE_LEN], *esp_root, *homepath;
#ifdef WIN32
    char      *homedrive;
#endif
    char      udcfilename[MAX_EXPR_LEN], pathname[MAX_EXPR_LEN];
    void      *dum;
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(ocsmNewBrch);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* make sure valid ibrch was given */
    if (iafter < 0 || iafter > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* if this has a perturbed Body, free it first */
    status = removePerturbation(MODL);
    CHECK_STATUS(removePerturbation);

    /* remove any velocity information */
    status = removeVels(MODL, 0);
    CHECK_STATUS(removeVels);

    /* check that valid type is given (and determine bclass) */

    /* note: impstr = 0x100    if arg1 is an implicit string
                      0x080    if arg2 is an implicit string
                      0x040    if arg3 is an implicit string
                      0x020    if arg4 is an implicit string
                      0x010    if arg5 is an implicit string
                      0x008    if arg6 is an implicit string
                      0x004    if arg7 is an implicit string
                      0x002    if arg8 is an implicit string
                      0x0o1    if arg9 is an implicit string */

    if        (type == OCSM_APPLYCSYS) {     // $csysName ibody=0
        bclass = OCSM_TRANSFORM;
        narg   = 2;
        impstr = 0x100;
    } else if (type == OCSM_ARC) {           // xend yend zend dist $plane=xy
        bclass = OCSM_SKETCH;
        narg   = 5;
        impstr = 0x010;
    } else if (type == OCSM_ASSERT) {        // arg1 arg2 toler=0 verify=0
        bclass = OCSM_UTILITY;
        narg   = 4;
        impstr = 0x000;
    } else if (type == OCSM_BEZIER) {        // x y z
        bclass = OCSM_SKETCH;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_BLEND) {         // begList=0 endList=0 reorder=0 oneFace=0 periodic=0 copyAttr=0
        bclass = OCSM_GROWN;
        narg   = 6;
        impstr = 0x000;
    } else if (type == OCSM_BOX) {           // xbase ybase zbase dx dy dz
        bclass = OCSM_PRIMITIVE;
        narg   = 6;
        impstr = 0x000;
    } else if (type == OCSM_CATBEG) {        // sigCode
        bclass = OCSM_UTILITY;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_CATEND) {
        bclass = OCSM_UTILITY;
        narg   = 0;
        impstr = 0x000;
    } else if (type == OCSM_CHAMFER) {       // radius edgeList=0 listStyle=0
        bclass = OCSM_APPLIED;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_CIRARC) {        // xon yon zon xend yend zend
        bclass = OCSM_SKETCH;
        narg   = 6;
        impstr = 0x000;
    } else if (type == OCSM_ELEVATE) {       // toler=0
        bclass = OCSM_BOOLEAN;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_CONE) {          // xvrtx yvrtx zvrtx xbase ybase zbase radius
        bclass = OCSM_PRIMITIVE;
        narg   = 7;
        impstr = 0x000;
    } else if (type == OCSM_CONNECT) {       // faceList1 faceList2 edgelist1=0 edgelist2=0 toler=0
        bclass = OCSM_BOOLEAN;
        narg   = 5;
        impstr = 0x000;
    } else if (type == OCSM_CYLINDER) {      // xbeg ybeg zbeg xend yend zend radius
        bclass = OCSM_PRIMITIVE;
        narg   = 7;
        impstr = 0x000;
    } else if (type == OCSM_DIMENSION) {     // $pmtrName nrow ncol
        bclass = OCSM_UTILITY;
        narg   = 3;
        impstr = 0x100;
    } else if (type == OCSM_DUMP) {          // $filename remove=0 toMark=0 withTess=0 $grpName
        bclass = OCSM_UTILITY;
        narg   = 6;
        impstr = 0x110;
    } else if (type == OCSM_ELSE) {
        bclass = OCSM_UTILITY;
        narg   = 0;
        impstr = 0x000;
    } else if (type == OCSM_ELSEIF) {        // val1 $op1=ne val2=0 $op2=and val3=0 $op3=eq val40
        bclass = OCSM_UTILITY;
        narg   = 7;
        impstr = 0x0aa;
    } else if (type == OCSM_END) {
        bclass = OCSM_UTILITY;
        narg   = 0;
        impstr = 0x000;
    } else if (type == OCSM_ENDIF) {
        bclass = OCSM_UTILITY;
        narg   = 0;
        impstr = 0x000;
    } else if (type == OCSM_EVALUATE) {      // $type arg1 ...
        bclass = OCSM_UTILITY;
        narg   = 0;
        if        (arg1 == NULL) {
        } else if (strcmp(arg1, "$node"    ) == 0 || strcmp(arg1, "$NODE"    ) == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$edge"    ) == 0 || strcmp(arg1, "$EDGE"    ) == 0) {
            narg = 4;
        } else if (strcmp(arg1, "$edgebbox") == 0 || strcmp(arg1, "$EDGEBBOX") == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$edgerng" ) == 0 || strcmp(arg1, "$EDGERNG" ) == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$edgeinv" ) == 0 || strcmp(arg1, "$EDGEINV" ) == 0) {
            narg = 6;
        } else if (strcmp(arg1, "$edgekt"  ) == 0 || strcmp(arg1, "$EDGEKT"  ) == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$edgecp"  ) == 0 || strcmp(arg1, "$EDGECP"  ) == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$edgetess") == 0 || strcmp(arg1, "$EDGETESS") == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$face"    ) == 0 || strcmp(arg1, "$FACE"    ) == 0) {
            narg = 5;
        } else if (strcmp(arg1, "$facebbox") == 0 || strcmp(arg1, "$FACEBBOX") == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$facerng" ) == 0 || strcmp(arg1, "$FACERNG" ) == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$faceinv" ) == 0 || strcmp(arg1, "$FACEINV" ) == 0) {
            narg = 6;
        } else if (strcmp(arg1, "$faceukt" ) == 0 || strcmp(arg1, "$FACEUKT" ) == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$facevkt" ) == 0 || strcmp(arg1, "$FACEVKT" ) == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$facecp"  ) == 0 || strcmp(arg1, "$FACECP"  ) == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$facetess") == 0 || strcmp(arg1, "$FACETESS") == 0) {
            narg = 3;
        } else if (strcmp(arg1, "$dist"    ) == 0 || strcmp(arg1, "$DIST"    ) == 0) {
            narg = 3;
        } else {
            narg = 1;
        }
        impstr = 0x100;
    } else if (type == OCSM_EXTRACT) {       // entList
        bclass = OCSM_BOOLEAN;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_EXTRUDE) {       // dx dy dz
        bclass = OCSM_GROWN;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_FILLET) {        // radius edgeList=0 listStyle=0
        bclass = OCSM_APPLIED;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_GETATTR) {       // $pmtrName attrID global=0
        bclass = OCSM_UTILITY;
        narg   = 3;
        impstr = 0x100;
    } else if (type == OCSM_GROUP) {         // nbody=0
        bclass = OCSM_UTILITY;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_HOLLOW) {        // thick=0 entList=0 listStyle=0
        bclass = OCSM_APPLIED;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_IFTHEN) {        // val1 $op1=ne val2=0 $op2=and val3=0 $op3=eq val4=0
        bclass = OCSM_UTILITY;
        narg   = 7;
        impstr = 0x0aa;
    } else if (type == OCSM_IMPORT) {        // $filename bodynumber=1
        bclass = OCSM_PRIMITIVE;
        narg   = 3;
        impstr = 0x100;
    } else if (type == OCSM_INTERFACE) {     // $argName $argType default=0
        bclass = OCSM_UTILITY;
        narg   = 3;
        impstr = 0x180;
    } else if (type == OCSM_INTERSECT) {     // $order=none index=1 maxtol=0
        bclass = OCSM_BOOLEAN;
        narg   = 3;
        impstr = 0x100;
    } else if (type == OCSM_JOIN) {          // toler=0 toMark=0
        bclass = OCSM_BOOLEAN;
        narg   = 2;
        impstr = 0x000;
    } else if (type == OCSM_LINSEG) {        // x y z
        bclass = OCSM_SKETCH;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_LOFT) {          // smooth
        bclass = OCSM_GROWN;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_MACBEG) {        // imacro
        bclass = OCSM_UTILITY;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_MACEND) {
        bclass = OCSM_UTILITY;
        narg   = 0;
        impstr = 0x000;
    } else if (type == OCSM_MARK) {
        bclass = OCSM_UTILITY;
        narg   = 0;
        impstr = 0x000;
    } else if (type == OCSM_MESSAGE) {       // $text $schar= $fileName=. $openType=a
        bclass = OCSM_UTILITY;
        narg   = 4;
        impstr = 0x1e0;
    } else if (type == OCSM_MIRROR) {        // nx ny nz dist=0
        bclass = OCSM_TRANSFORM;
        narg   = 4;
        impstr = 0x000;
    } else if (type == OCSM_PATBEG) {        // $pmtrName ncopy
        bclass = OCSM_UTILITY;
        narg   = 2;
        impstr = 0x100;
    } else if (type == OCSM_PATBREAK) {      // val1 $op1=ne val2=0 $op2=and val3=0 $op3=eq val4=0
        bclass = OCSM_UTILITY;
        narg   = 7;
        impstr = 0x0aa;
    } else if (type == OCSM_PATEND) {
        bclass = OCSM_UTILITY;
        narg   = 0;
        impstr = 0x000;
    } else if (type == OCSM_POINT) {         // xloc yloc zloc
        bclass = OCSM_PRIMITIVE;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_PREFIX) {        // $prefix=.
        bclass = OCSM_UTILITY;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_PROJECT) {       // x y z dx dy dz useEdges=0
        bclass = OCSM_UTILITY;
        narg   = 7;
        impstr = 0x000;
    } else if (type == OCSM_RECALL) {        // imacro
        bclass = OCSM_UTILITY;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_REORDER) {       // ishift iflip=0 reverse=0
        bclass = OCSM_TRANSFORM;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_RESTORE) {       // $name index=0
        bclass = OCSM_PRIMITIVE;
        narg   = 2;
        impstr = 0x100;
    } else if (type == OCSM_REVOLVE) {       // xorig yorig zorig dxaxis dyaxis dzaxis angDeg
        bclass = OCSM_GROWN;
        narg   = 7;
        impstr = 0x000;
    } else if (type == OCSM_ROTATEX) {       // angDeg yaxis=0 zaxis=0
        bclass = OCSM_TRANSFORM;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_ROTATEY) {       // angDeg zaxis=0 xaxis=0
        bclass = OCSM_TRANSFORM;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_ROTATEZ) {       // angDeg xaxis=0 yaxis=0
        bclass = OCSM_TRANSFORM;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_RULE) {          // reorder=0 periodic=0 copyAttr=0
        bclass = OCSM_GROWN;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_SCALE) {         // fact xcent=0 ycent=0 zcent=0
        bclass = OCSM_TRANSFORM;
        narg   = 4;
        impstr = 0x000;
    } else if (type == OCSM_SELECT) {        // $type arg1 ...
        bclass = OCSM_UTILITY;
        narg   = 1;
        if (arg2 != NULL && arg2[0] != '\0') narg = 2;
        if (arg3 != NULL && arg3[0] != '\0') narg = 3;
        if (arg4 != NULL && arg4[0] != '\0') narg = 4;
        if (arg5 != NULL && arg5[0] != '\0') narg = 5;
        if (arg6 != NULL && arg6[0] != '\0') narg = 6;
        if (arg7 != NULL && arg7[0] != '\0') narg = 7;
        if (arg8 != NULL && arg8[0] != '\0') narg = 8;
        impstr = 0x100;
    } else if (type == OCSM_SET) {           // $pmtrName exprs
        bclass = OCSM_UTILITY;
        narg   = 2;
        impstr = 0x100;
    } else if (type == OCSM_SKBEG) {         // x y z relative=0
        bclass = OCSM_SKETCH;
        narg   = 4;
        impstr = 0x000;
    } else if (type == OCSM_SKCON) {         // $type index1 index2=1 $value=0
        bclass = OCSM_SKETCH;
        narg   = 4;
        impstr = 0x120;
    } else if (type == OCSM_SKEND) {         // wireonly=0
        bclass = OCSM_SKETCH;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_SKVAR) {         // $type valList
        bclass = OCSM_SKETCH;
        narg   = 2;
        impstr = 0x100;
    } else if (type == OCSM_SOLBEG) {        // $varList
        bclass = OCSM_SOLVER;
        narg   = 1;
        impstr = 0x100;
    } else if (type == OCSM_SOLCON) {        // $expr
        bclass = OCSM_SOLVER;
        narg   = 1;
        impstr = 0x100;
    } else if (type == OCSM_SOLEND) {
        bclass = OCSM_SOLVER;
        narg   = 0;
        impstr = 0x000;
    } else if (type == OCSM_SPECIAL) {       // $option=. arg1=0 arg2=0 arg3=0 arg4=0 arg5=0 arg6=0 arg7=0 arg8=0
        bclass = OCSM_UTILITY;
        narg   = 9;
        impstr = 0x000;
    } else if (type == OCSM_SPHERE) {        // xcent ycent zcent radius
        bclass = OCSM_PRIMITIVE;
        narg   = 4;
        impstr = 0x000;
    } else if (type == OCSM_SPLINE) {        // x y z
        bclass = OCSM_SKETCH;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_SSLOPE) {        // dx dy dz
        bclass = OCSM_SKETCH;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_STORE) {         // $name index=0 keep=0
        bclass = OCSM_UTILITY;
        narg   = 3;
        impstr = 0x100;
    } else if (type == OCSM_SUBTRACT) {      // $order=none index=1 maxtol=0 scribeAll=0
        bclass = OCSM_BOOLEAN;
        narg   = 4;
        impstr = 0x100;
    } else if (type == OCSM_SWEEP) {
        bclass = OCSM_GROWN;
        narg   = 0;
        impstr = 0x000;
    } else if (type == OCSM_THROW) {         // sigCode
        bclass = OCSM_UTILITY;
        narg   = 1;
        impstr = 0x000;
    } else if (type == OCSM_TORUS) {         // xcent ycent zcent dxaxis dyaxis dzaxis majorRad minorRad
        bclass = OCSM_PRIMITIVE;
        narg   = 8;
        impstr = 0x000;
    } else if (type == OCSM_TRANSLATE) {     // dx dy dz
        bclass = OCSM_TRANSFORM;
        narg   = 3;
        impstr = 0x000;
    } else if (type == OCSM_UDPARG) {        // $primtype $argName1 argvalue1 $argName2 argvalue2 $argName3 argvalue3 $argName4 argvalue4
        bclass = OCSM_UTILITY;
        narg   = 1;
        if (arg2 != NULL && arg2[0] != '\0' &&
            arg3 != NULL && arg3[0] != '\0'   ) narg = 3;
        if (arg4 != NULL && arg4[0] != '\0' &&
            arg5 != NULL && arg5[0] != '\0'   ) narg = 5;
        if (arg6 != NULL && arg6[0] != '\0' &&
            arg7 != NULL && arg7[0] != '\0'   ) narg = 7;
        if (arg8 != NULL && arg8[0] != '\0' &&
            arg9 != NULL && arg9[0] != '\0'   ) narg = 9;
        impstr = 0x1aa;
    } else if (type == OCSM_UDPRIM) {        // $primtype $argName1 argvalue1 $argName2 argvalue2 $argName3 argvalue3 $argName4 argvalue4
        bclass = OCSM_PRIMITIVE;
        narg  = 1;
        if (arg2 != NULL && arg2[0] != '\0' &&
            arg3 != NULL && arg3[0] != '\0'   ) narg = 3;
        if (arg4 != NULL && arg4[0] != '\0' &&
            arg5 != NULL && arg5[0] != '\0'   ) narg = 5;
        if (arg6 != NULL && arg6[0] != '\0' &&
            arg7 != NULL && arg7[0] != '\0'   ) narg = 7;
        if (arg8 != NULL && arg8[0] != '\0' &&
            arg9 != NULL && arg9[0] != '\0'   ) narg = 9;
        impstr = 0x1aa;
    } else if (type == OCSM_UNION) {         // toMark=0 trimList=0 maxtol=0
        bclass = OCSM_BOOLEAN;
        narg   = 3;
        impstr = 0x000;
    } else {
        status = OCSM_ILLEGAL_TYPE;
        goto cleanup;
    }

    /* mark MODL as not being checked */
    MODL->checked = 0;

    /* increment the number of Branches */
    (MODL->nbrch)++;
    ibrch = iafter + 1;

    MODL->ibrch = iafter + 1;

    /* extend the Branch list (if needed) */
    if (MODL->nbrch > MODL->mbrch) {
        MODL->mbrch += 25;
        RALLOC(MODL->brch, brch_T, MODL->mbrch+1);
    }

    /* copy the Branches up to make room for the new Branch */
#ifndef __clang_analyzer__
    if (ibrch < MODL->nbrch) {
        for (jbrch = MODL->nbrch; jbrch > ibrch; jbrch--) {
            MODL->brch[jbrch].name     = MODL->brch[jbrch-1].name;
            MODL->brch[jbrch].type     = MODL->brch[jbrch-1].type;
            MODL->brch[jbrch].bclass   = MODL->brch[jbrch-1].bclass;
            MODL->brch[jbrch].level    = MODL->brch[jbrch-1].level;
            MODL->brch[jbrch].indent   = MODL->brch[jbrch-1].indent;
            MODL->brch[jbrch].filename = MODL->brch[jbrch-1].filename;
            MODL->brch[jbrch].linenum  = MODL->brch[jbrch-1].linenum;
            MODL->brch[jbrch].actv     = MODL->brch[jbrch-1].actv;
            MODL->brch[jbrch].dirty    = 1;
            MODL->brch[jbrch].nattr    = MODL->brch[jbrch-1].nattr;
            MODL->brch[jbrch].attr     = MODL->brch[jbrch-1].attr;
            MODL->brch[jbrch].ileft    = MODL->brch[jbrch-1].ileft;
            MODL->brch[jbrch].irite    = MODL->brch[jbrch-1].irite;
            MODL->brch[jbrch].ichld    = MODL->brch[jbrch-1].ichld;
            MODL->brch[jbrch].nmprp    = MODL->brch[jbrch-1].nmprp;
            MODL->brch[jbrch].mprp     = MODL->brch[jbrch-1].mprp;
            MODL->brch[jbrch].narg     = MODL->brch[jbrch-1].narg;
            MODL->brch[jbrch].arg1     = MODL->brch[jbrch-1].arg1;
            MODL->brch[jbrch].arg2     = MODL->brch[jbrch-1].arg2;
            MODL->brch[jbrch].arg3     = MODL->brch[jbrch-1].arg3;
            MODL->brch[jbrch].arg4     = MODL->brch[jbrch-1].arg4;
            MODL->brch[jbrch].arg5     = MODL->brch[jbrch-1].arg5;
            MODL->brch[jbrch].arg6     = MODL->brch[jbrch-1].arg6;
            MODL->brch[jbrch].arg7     = MODL->brch[jbrch-1].arg7;
            MODL->brch[jbrch].arg8     = MODL->brch[jbrch-1].arg8;
            MODL->brch[jbrch].arg9     = MODL->brch[jbrch-1].arg9;
        }
    }
#endif

    /* initialize the new Branch */
    MODL->brch[ibrch].name     = NULL;
    MODL->brch[ibrch].type     = type;
    MODL->brch[ibrch].bclass   = bclass;
    MODL->brch[ibrch].level    = MODL->level;
    MODL->brch[ibrch].indent   = 0;
    MODL->brch[ibrch].filename = NULL;
    MODL->brch[ibrch].linenum  = linenum;
    MODL->brch[ibrch].actv     =  OCSM_ACTIVE;
    MODL->brch[ibrch].dirty    = 1;
    MODL->brch[ibrch].nattr    = 0;
    MODL->brch[ibrch].attr     = NULL;
    MODL->brch[ibrch].ileft    = -2;
    MODL->brch[ibrch].irite    = -2;
    MODL->brch[ibrch].ichld    = -2;
    MODL->brch[ibrch].nmprp    =  0;
    MODL->brch[ibrch].mprp     =  NULL;
    MODL->brch[ibrch].narg     =  narg;
    MODL->brch[ibrch].arg1     =  NULL;
    MODL->brch[ibrch].arg2     =  NULL;
    MODL->brch[ibrch].arg3     =  NULL;
    MODL->brch[ibrch].arg4     =  NULL;
    MODL->brch[ibrch].arg5     =  NULL;
    MODL->brch[ibrch].arg6     =  NULL;
    MODL->brch[ibrch].arg7     =  NULL;
    MODL->brch[ibrch].arg8     =  NULL;
    MODL->brch[ibrch].arg9     =  NULL;

    /* default name for the Branch */
    MALLOC(  MODL->brch[ibrch].name, char, 12);
    snprintf(MODL->brch[ibrch].name,       12, "Brch_%06d", MODL->nextseq);
    (MODL->nextseq)++;

    /* save the filename */
    MALLOC(MODL->brch[ibrch].filename, char, MAX_FILENAME_LEN);
    strcpy(MODL->brch[ibrch].filename, filename);

    /* save the Arguments */
    if (narg >= 1) {
        if (arg1 != NULL && STRLEN(arg1) > 0) {
            if ((impstr & 0x100) != 0 && arg1[0] != '$' && arg1[0] != '!') {
                SPRINT0(1, "WARNING:: converting arg1 to an implicit string");
                snprintf(temparg, MAX_LINE_LEN, "$%s", arg1);
            } else {
                snprintf(temparg, MAX_LINE_LEN,  "%s", arg1);
            }

            MALLOC(MODL->brch[ibrch].arg1, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg1,                    temparg    );
        } else if (arg1 != NULL && STRLEN(arg1) == 0  && (impstr & 0x100) != 0) {
            SPRINT0(1, "WARNING:: creating default arg1 ($0)");
            snprintf(temparg, MAX_LINE_LEN, "$0");

            MALLOC(MODL->brch[ibrch].arg1, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg1,                    temparg    );
        } else if (arg1 != NULL && STRLEN(arg1) == 0) {
            MALLOC(MODL->brch[ibrch].arg1, char, 2);
            strcpy(MODL->brch[ibrch].arg1, "0");
        } else {
            status = OCSM_ILLEGAL_NARG;
        }
    }

    if (narg >= 2) {
        if (arg2 != NULL && STRLEN(arg2) > 0) {
            if ((impstr & 0x080) != 0 && arg2[0] != '$' && arg2[0] != '!') {
                SPRINT0(1, "WARNING:: converting arg2 to an implicit string");
                snprintf(temparg, MAX_LINE_LEN, "$%s", arg2);
            } else {
                snprintf(temparg, MAX_LINE_LEN,  "%s", arg2);
            }

            MALLOC(MODL->brch[ibrch].arg2, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg2,                    temparg    );
        } else if (arg2 != NULL && STRLEN(arg2) == 0  && (impstr & 0x080) != 0) {
            SPRINT0(1, "WARNING:: creating default arg2 ($0)");
            snprintf(temparg, MAX_LINE_LEN, "$0");

            MALLOC(MODL->brch[ibrch].arg2, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg2,                    temparg    );
        } else if (arg2 != NULL && STRLEN(arg2) == 0) {
            MALLOC(MODL->brch[ibrch].arg2, char, 2);
            strcpy(MODL->brch[ibrch].arg2, "0");
        } else {
            status = OCSM_ILLEGAL_NARG;
        }
    }

    if (narg >= 3) {
        if (arg3 != NULL && STRLEN(arg3) > 0) {
            if ((impstr & 0x040) != 0 && arg3[0] != '$' && arg3[0] != '!') {
                SPRINT0(1, "WARNING:: converting arg3 to an implicit string");
                snprintf(temparg, MAX_LINE_LEN, "$%s", arg3);
            } else {
                snprintf(temparg, MAX_LINE_LEN,  "%s", arg3);
            }

            MALLOC(MODL->brch[ibrch].arg3, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg3,                    temparg    );
        } else if (arg3 != NULL && STRLEN(arg3) == 0  && (impstr & 0x040) != 0) {
            SPRINT0(1, "WARNING:: creating default arg3 ($0)");
            snprintf(temparg, MAX_LINE_LEN, "$0");

            MALLOC(MODL->brch[ibrch].arg3, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg3,                    temparg    );
        } else if (arg3 != NULL && STRLEN(arg3) == 0) {
            MALLOC(MODL->brch[ibrch].arg3, char, 2);
            strcpy(MODL->brch[ibrch].arg3, "0");
        } else {
            status = OCSM_ILLEGAL_NARG;
        }
    }

    if (narg >= 4) {
        if (arg4 != NULL && STRLEN(arg4) > 0) {
            if ((impstr & 0x020) != 0 && arg4[0] != '$' && arg4[0] != '!') {
                SPRINT0(1, "WARNING:: converting arg4 to an implicit string");
                snprintf(temparg, MAX_LINE_LEN, "$%s", arg4);
            } else {
                snprintf(temparg, MAX_LINE_LEN,  "%s", arg4);
            }

            MALLOC(MODL->brch[ibrch].arg4, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg4,                    temparg    );
        } else if (arg4 != NULL && STRLEN(arg4) == 0  && (impstr & 0x020) != 0) {
            SPRINT0(1, "WARNING:: creating default arg4 ($0)");
            snprintf(temparg, MAX_LINE_LEN, "$0");

            MALLOC(MODL->brch[ibrch].arg4, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg4,                    temparg    );
        } else if (arg4 != NULL && STRLEN(arg4) == 0) {
            MALLOC(MODL->brch[ibrch].arg4, char, 2);
            strcpy(MODL->brch[ibrch].arg4, "0");
        } else {
            status = OCSM_ILLEGAL_NARG;
        }
    }

    if (narg >= 5) {
        if (arg5 != NULL && STRLEN(arg5) > 0) {
            if ((impstr & 0x010) != 0 && arg5[0] != '$' && arg5[0] != '!') {
                SPRINT0(1, "WARNING:: converting arg5 to an implicit string");
                snprintf(temparg, MAX_LINE_LEN, "$%s", arg5);
            } else {
                snprintf(temparg, MAX_LINE_LEN,  "%s", arg5);
            }

            MALLOC(MODL->brch[ibrch].arg5, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg5,                    temparg    );
        } else if (arg5 != NULL && STRLEN(arg5) == 0  && (impstr & 0x010) != 0) {
            SPRINT0(1, "WARNING:: creating default arg5 ($0)");
            snprintf(temparg, MAX_LINE_LEN, "$0");

            MALLOC(MODL->brch[ibrch].arg5, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg5,                    temparg    );
        } else if (arg5 != NULL && STRLEN(arg5) == 0) {
            MALLOC(MODL->brch[ibrch].arg5, char, 2);
            strcpy(MODL->brch[ibrch].arg5, "0");
        } else {
            status = OCSM_ILLEGAL_NARG;
        }
    }

    if (narg >= 6) {
        if (arg6 != NULL && STRLEN(arg6) > 0) {
            if ((impstr & 0x008) != 0 && arg6[0] != '$' && arg6[0] != '!') {
                SPRINT0(1, "WARNING:: converting arg6 to an implicit string");
                snprintf(temparg, MAX_LINE_LEN, "$%s", arg6);
            } else {
                snprintf(temparg, MAX_LINE_LEN,  "%s", arg6);
            }

            MALLOC(MODL->brch[ibrch].arg6, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg6,                    temparg    );
        } else if (arg6 != NULL && STRLEN(arg6) == 0  && (impstr & 0x008) != 0) {
            SPRINT0(1, "WARNING:: creating default arg6 ($0)");
            snprintf(temparg, MAX_LINE_LEN, "$0");

            MALLOC(MODL->brch[ibrch].arg6, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg6,                    temparg    );
        } else if (arg6 != NULL && STRLEN(arg6) == 0) {
            MALLOC(MODL->brch[ibrch].arg6, char, 2);
            strcpy(MODL->brch[ibrch].arg6, "0");
        } else {
            status = OCSM_ILLEGAL_NARG;
        }
    }

    if (narg >= 7) {
        if (arg7 != NULL && STRLEN(arg7) > 0) {
            if ((impstr & 0x004) != 0 && arg7[0] != '$' && arg7[0] != '!') {
                SPRINT0(1, "WARNING:: converting arg7 to an implicit string");
                snprintf(temparg, MAX_LINE_LEN, "$%s", arg7);
            } else {
                snprintf(temparg, MAX_LINE_LEN,  "%s", arg7);
            }

            MALLOC(MODL->brch[ibrch].arg7, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg7,                    temparg    );
        } else if (arg7 != NULL && STRLEN(arg7) == 0  && (impstr & 0x004) != 0) {
            SPRINT0(1, "WARNING:: creating default arg7 ($0)");
            snprintf(temparg, MAX_LINE_LEN, "$0");

            MALLOC(MODL->brch[ibrch].arg7, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg7,                    temparg    );
        } else if (arg7 != NULL && STRLEN(arg7) == 0) {
            MALLOC(MODL->brch[ibrch].arg7, char, 2);
            strcpy(MODL->brch[ibrch].arg7, "0");
        } else {
            status = OCSM_ILLEGAL_NARG;
        }
    }

    if (narg >= 8) {
        if (arg8 != NULL && STRLEN(arg8) > 0) {
            if ((impstr & 0x002) != 0 && arg8[0] != '$' && arg8[0] != '!') {
                SPRINT0(1, "WARNING:: converting arg8 to an implicit string");
                snprintf(temparg, MAX_LINE_LEN, "$%s", arg8);
            } else {
                snprintf(temparg, MAX_LINE_LEN,  "%s", arg8);
            }

            MALLOC(MODL->brch[ibrch].arg8, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg8,                    temparg    );
        } else if (arg8 != NULL && STRLEN(arg8) == 0  && (impstr & 0x002) != 0) {
            SPRINT0(1, "WARNING:: creating default arg8 ($0)");
            snprintf(temparg, MAX_LINE_LEN, "$0");

            MALLOC(MODL->brch[ibrch].arg8, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg8,                    temparg    );
        } else if (arg8 != NULL && STRLEN(arg8) == 0) {
            MALLOC(MODL->brch[ibrch].arg8, char, 2);
            strcpy(MODL->brch[ibrch].arg8, "0");
        } else {
            status = OCSM_ILLEGAL_NARG;
        }
    }

    if (narg >= 9) {
        if (arg9 != NULL && STRLEN(arg9) > 0) {
            if ((impstr & 0x001) != 0 && arg9[0] != '$' && arg9[0] != '!') {
                SPRINT0(1, "WARNING:: converting arg9 to an implicit string");
                snprintf(temparg, MAX_LINE_LEN, "$%s", arg9);
            } else {
                snprintf(temparg, MAX_LINE_LEN,  "%s", arg9);
            }

            MALLOC(MODL->brch[ibrch].arg9, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg9,                    temparg    );
        } else if (arg9 != NULL && STRLEN(arg9) == 0  && (impstr & 0x001) != 0) {
            SPRINT0(1, "WARNING:: creating default arg9 ($0)");
            snprintf(temparg, MAX_LINE_LEN, "$0");

            MALLOC(MODL->brch[ibrch].arg9, char, (int)(STRLEN(temparg)+1));
            strcpy(MODL->brch[ibrch].arg9,                    temparg    );
        } else if (arg9 != NULL && STRLEN(arg9) == 0) {
            MALLOC(MODL->brch[ibrch].arg9, char, 2);
            strcpy(MODL->brch[ibrch].arg9, "0");
        } else {
            status = OCSM_ILLEGAL_NARG;
        }
    }

    /* recursively add Branches in an UDC, but not in a .cpc */
    if (strstr(filename, ".cpc") != NULL) {
        /* do not divert since .cpc file contains all Branches */
    } else if (MODL->brch[ibrch].arg1 == NULL) {
        /* nothing.  needed to avoid clang warning */
    } else if (type == OCSM_UDPRIM                                  &&
               (strncmp(MODL->brch[ibrch].arg1, "$/",    2) == 0 ||
                strncmp(MODL->brch[ibrch].arg1, "$$/",   3) == 0 ||
                strncmp(MODL->brch[ibrch].arg1, "$$$/",  4) == 0 ||
                strncmp(MODL->brch[ibrch].arg1, "$$$$/", 5) == 0   )  ) {

        if (MODL->level < 10) {
            (MODL->level)++;
            MODL->scope[MODL->level] = MODL->scope[MODL->level-1] + 1;

            /* if primtype starts with "/~/", start from $HOME */
            if        (strncmp(MODL->brch[ibrch].arg1, "$/~/", 4) == 0) {
#ifndef WIN32
                homepath = getenv("HOME");
                if (homepath != NULL) {
                    snprintf(udcfilename, MAX_EXPR_LEN, "%s%c%s.udc", homepath, SLASH, &(MODL->brch[ibrch].arg1[4]));
                } else {
                    (void) signalError2(MODL, OCSM_FILE_NOT_FOUND, filename, linenum,
                                        "environment variable $HOME not specified");
                    goto cleanup;
                }
#else
                homedrive = getenv("HOMEDRIVE");
                homepath  = getenv("HOMEPATH");
                if (homedrive != NULL && homepath != NULL) {
                    snprintf(udcfilename, MAX_EXPR_LEN, "%s%s%c%s.udc", homedrive, homepath, SLASH, &(MODL->brch[ibrch].arg1[4]));
                } else {
                    (void) signalError2(MODL, OCSM_FILE_NOT_FOUND, filename, linenum,
                                        "environment variable $HOMEDRIVE of $HOMEPATH not specified");
                    goto cleanup;
                }
#endif

            /* if primtype starts with "/", look in current directory */
            } else if (strncmp(MODL->brch[ibrch].arg1, "$/",  2) == 0) {
                snprintf(udcfilename, MAX_EXPR_LEN, "%s.udc", &(MODL->brch[ibrch].arg1[2]));

            /* if primtype starts with "$/", look in same directory as .csm file */
            } else if (strncmp(MODL->brch[ibrch].arg1, "$$/", 3) == 0) {
                STRNCPY(pathname, filename, MAX_EXPR_LEN);
                i = STRLEN(pathname) - 1;
                while (pathname[i] != '/' && pathname[i] != '\\') {
                    pathname[i] = '\0';
                    i--;
                    if (i < 0) break;
                }
                snprintf(udcfilename, MAX_EXPR_LEN, "%s%s.udc", pathname, &(MODL->brch[ibrch].arg1[3]));

            /* if primtype starts with "$$/", look in ESP_ROOT directory */
            } else if (strncmp(MODL->brch[ibrch].arg1, "$$$/", 4) == 0) {
                esp_root = getenv("ESP_ROOT");
                if (esp_root != NULL) {
                    snprintf(udcfilename, MAX_EXPR_LEN, "%s%cudc%c%s.udc", esp_root, SLASH, SLASH, &(MODL->brch[ibrch].arg1[4]));
                } else {
                    status = signalError2(MODL, OCSM_FILE_NOT_FOUND, filename, linenum,
                                          "ESP_ROOT is not set");
                    goto cleanup;
                }

            /* if primtype starts with "$$$/", look in ESP_UDC_PATH directory */
            } else if (strncmp(MODL->brch[ibrch].arg1, "$$$$/", 5) == 0) {
                esp_root = getenv("ESP_UDC_PATH");
                if (esp_root != NULL) {
                    snprintf(udcfilename, MAX_EXPR_LEN, "%s%c%s.udc", esp_root, SLASH, &(MODL->brch[ibrch].arg1[5]));
                } else {
                    status = signalError2(MODL, OCSM_FILE_NOT_FOUND, filename, linenum,
                                          "ESP_UDC_PATH is not set");
                    goto cleanup;
                }
            } else {
                status =signalError2(MODL, OCSM_FILE_NOT_FOUND, filename, linenum,
                                     "\"%s\" is not in the correct form", &(MODL->brch[ibrch].arg1[1]));
                goto cleanup;
            }

            /* make sure we have the correct type of slashes */
            for (i = 0; i < STRLEN(udcfilename); i++) {
                if (udcfilename[i] == '/' || udcfilename[i] == '\\') {
                    udcfilename[i] = SLASH;
                }
            }

            /* compress udcfilname by removing /../ patterns */
            status = compressFilename(udcfilename);
            if (status < SUCCESS) {
                (void) signalError2(MODL, status, filename, linenum,
                                    "error when trying to compress \"%s\"", udcfilename);
                goto cleanup;
            }

            SPRINT1(1, "\n>>> Diverting to file \"%s\"\n", udcfilename);

            /* recursively call ocsmLoad */
            dum = (void*)MODL;
            status = ocsmLoad(udcfilename, &dum);
            if (status < SUCCESS) {
                (void) signalError2(MODL, status, filename, linenum,
                             "error when trying to process \"%s\"", udcfilename);
                goto cleanup;
            }
        } else {
            status = signalError2(MODL, OCSM_NESTED_TOO_DEEPLY, filename, linenum,
                                  "maximum file depth reached");
            goto cleanup;
        }
    }

    /* increment number of groups if Branch will create a Body */
    if (bclass == OCSM_PRIMITIVE ||
        bclass == OCSM_GROWN     ||
        bclass == OCSM_APPLIED   ||
        bclass == OCSM_BOOLEAN   ||
        bclass == OCSM_TRANSFORM ||
        type   == OCSM_SKEND       ) {
        (MODL->ngroup)++;
    }

cleanup:
    if (status < SUCCESS && MODL != NULL) {
        MODL->nbrch--;
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetBrch - get info about a Branch                              *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetBrch(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            int    *type,               /* (out) Branch type */
            int    *bclass,             /* (out) Branch class */
            int    *actv,               /* (out) Branch Activity */
            int    *ichld,              /* (out) ibrch of child (or 0 if root) */
            int    *ileft,              /* (out) ibrch of left parent (or 0) */
            int    *irite,              /* (out) ibrch of rite parent (or 0) */
            int    *narg,               /* (out) number of Arguments */
            int    *nattr)              /* (out) number of Attributes */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmGetBrch);

    /* --------------------------------------------------------------- */

    /* default return values */
    *type   = 0;
    *bclass = 0;
    *actv   = 0;
    *ichld  = 0;
    *ileft  = 0;
    *irite  = 0;
    *narg   = 0;
    *nattr  = 0;

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* return pertinent information */
    *type   = MODL->brch[ibrch].type;
    *bclass = MODL->brch[ibrch].bclass;
    *actv   = MODL->brch[ibrch].actv;
    *ichld  = MODL->brch[ibrch].ichld;
    *ileft  = MODL->brch[ibrch].ileft;
    *irite  = MODL->brch[ibrch].irite;
    *narg   = MODL->brch[ibrch].narg;
    *nattr  = MODL->brch[ibrch].nattr;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSetBrch - set activity for a Branch                            *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSetBrch(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            int    actv)                /* (in)  Branch activity */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmSetBrch);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* check that a valid activity was given */
    if (actv != OCSM_ACTIVE    &&
        actv != OCSM_SUPPRESSED  ) {
        status = OCSM_ILLEGAL_ACTIVITY;
        goto cleanup;
    }

    /* check that branch type can be suppressed */
    if (MODL->brch[ibrch].bclass != OCSM_PRIMITIVE &&
        MODL->brch[ibrch].bclass != OCSM_GROWN     &&
        MODL->brch[ibrch].bclass != OCSM_APPLIED   &&
        MODL->brch[ibrch].bclass != OCSM_TRANSFORM   ) {
        status = OCSM_CANNOT_BE_SUPPRESSED;
        goto cleanup;
    }

    /* if this has a perturbed Body, free it first */
    status = removePerturbation(MODL);
    CHECK_STATUS(removePerturbation);

    /* remove any velocity information */
    status = removeVels(MODL, 0);
    CHECK_STATUS(removeVels);

    /* save the activity */
    MODL->brch[ibrch].actv = actv;

    /* mark the Branch as dirty */
    MODL->brch[ibrch].dirty = 1;

    /* mark MODL as not being checked */
    MODL->checked = 0;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmDelBrch - delete a Branch (or whole Sketch if SKBEG)           *
 *                                                                      *
 ************************************************************************
 */

int
ocsmDelBrch(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch)               /* (in)  Branch index (1:nbrch) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       itype, jbrch, iattr;

    ROUTINE(ocsmDelBrch);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check for valid Branch index */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* if this has a perturbed Body, free it first */
    status = removePerturbation(MODL);
    CHECK_STATUS(removePerturbation);

    /* remove any velocity information */
    status = removeVels(MODL, 0);
    CHECK_STATUS(removeVels);

    /* mark MODL as not being checked */
    MODL->checked = 0;

    /* remember the type if ibrch */
    itype = MODL->brch[ibrch].type;

    if (itype == OCSM_SKBEG) {
        SPRINT0(1, "WARNING:: deleting entire Sketch");
        (MODL->nwarn)++;
    }

    /* delete the Branch(es) */
    while (ibrch <= MODL->nbrch) {

        /* free up the storage associated with ibrch */
        for (iattr = 0; iattr < MODL->brch[ibrch].nattr; iattr++) {
            FREE(MODL->brch[ibrch].attr[iattr].name);
            FREE(MODL->brch[ibrch].attr[iattr].defn);
        }

        FREE(MODL->brch[ibrch].name    );
        FREE(MODL->brch[ibrch].filename);
        FREE(MODL->brch[ibrch].attr    );
        FREE(MODL->brch[ibrch].mprp    );
        FREE(MODL->brch[ibrch].arg1    );
        FREE(MODL->brch[ibrch].arg2    );
        FREE(MODL->brch[ibrch].arg3    );
        FREE(MODL->brch[ibrch].arg4    );
        FREE(MODL->brch[ibrch].arg5    );
        FREE(MODL->brch[ibrch].arg6    );
        FREE(MODL->brch[ibrch].arg7    );
        FREE(MODL->brch[ibrch].arg8    );
        FREE(MODL->brch[ibrch].arg9    );

        /* move all Branches up to write over deleted Branch */
        for (jbrch = ibrch; jbrch < MODL->nbrch; jbrch++) {
            MODL->brch[jbrch].name     = MODL->brch[jbrch+1].name;
            MODL->brch[jbrch].type     = MODL->brch[jbrch+1].type;
            MODL->brch[jbrch].bclass   = MODL->brch[jbrch+1].bclass;
            MODL->brch[jbrch].level    = MODL->brch[jbrch-1].level;
            MODL->brch[jbrch].indent   = MODL->brch[jbrch+1].indent;
            MODL->brch[jbrch].filename = MODL->brch[jbrch+1].filename;
            MODL->brch[jbrch].linenum  = MODL->brch[jbrch+1].linenum;
            MODL->brch[jbrch].actv     = MODL->brch[jbrch+1].actv;
            MODL->brch[jbrch].dirty    = 1;
            MODL->brch[jbrch].nattr    = MODL->brch[jbrch+1].nattr;
            MODL->brch[jbrch].attr     = MODL->brch[jbrch+1].attr;
            MODL->brch[jbrch].ileft    = MODL->brch[jbrch+1].ileft;
            MODL->brch[jbrch].irite    = MODL->brch[jbrch+1].irite;
            MODL->brch[jbrch].ichld    = MODL->brch[jbrch+1].ichld;
            MODL->brch[jbrch].nmprp    = MODL->brch[jbrch+1].nmprp;
            MODL->brch[jbrch].mprp     = MODL->brch[jbrch+1].mprp;
            MODL->brch[jbrch].narg     = MODL->brch[jbrch+1].narg;
            MODL->brch[jbrch].arg1     = MODL->brch[jbrch+1].arg1;
            MODL->brch[jbrch].arg2     = MODL->brch[jbrch+1].arg2;
            MODL->brch[jbrch].arg3     = MODL->brch[jbrch+1].arg3;
            MODL->brch[jbrch].arg4     = MODL->brch[jbrch+1].arg4;
            MODL->brch[jbrch].arg5     = MODL->brch[jbrch+1].arg5;
            MODL->brch[jbrch].arg6     = MODL->brch[jbrch+1].arg6;
            MODL->brch[jbrch].arg7     = MODL->brch[jbrch+1].arg7;
            MODL->brch[jbrch].arg8     = MODL->brch[jbrch+1].arg8;
            MODL->brch[jbrch].arg9     = MODL->brch[jbrch+1].arg9;
        }

        /* null pointers associated with the last Branch (since they were
           copied down) */
        MODL->brch[MODL->nbrch].name     = NULL;
        MODL->brch[MODL->nbrch].filename = NULL;
        MODL->brch[MODL->nbrch].attr     = NULL;
        MODL->brch[MODL->nbrch].mprp     = NULL;
        MODL->brch[MODL->nbrch].arg1     = NULL;
        MODL->brch[MODL->nbrch].arg2     = NULL;
        MODL->brch[MODL->nbrch].arg3     = NULL;
        MODL->brch[MODL->nbrch].arg4     = NULL;
        MODL->brch[MODL->nbrch].arg5     = NULL;
        MODL->brch[MODL->nbrch].arg6     = NULL;
        MODL->brch[MODL->nbrch].arg7     = NULL;
        MODL->brch[MODL->nbrch].arg8     = NULL;
        MODL->brch[MODL->nbrch].arg9     = NULL;

        /* decrement the number of Branches */
        (MODL->nbrch)--;

        /* if the remembered type is not SKBEG, we are done */
        if (itype != OCSM_SKBEG) break;

        /* otherwise keep deleting Branches that were in the Sketch */
        if (MODL->brch[ibrch].type   != OCSM_SKEND &&
            MODL->brch[ibrch].bclass != OCSM_SKETCH  ) break;
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmPrintBrchs - print Branches to file                            *
 *                                                                      *
 ************************************************************************
 */

int
ocsmPrintBrchs(void   *modl,            /* (in)  pointer to MODL */
               char   filename[])       /* (in)  file to which output is appended (or "" for stdout) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       ibrch, i, nindent, iattr, imprp;
    FILE      *fp=NULL;

    ROUTINE(ocsmPrintBrchs);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    if (strlen(filename) == 0) {
        fp = stdout;
    } else {
        fp = fopen(filename, "a");
        if (fp == NULL) {
            status = OCSM_FILE_NOT_FOUND;
            goto cleanup;
        }
    }

    /* run ocsmCheck to set up necessary linkages amongst Branches */
    status = ocsmCheck(MODL);
    CHECK_STATUS(ocsmCheck);

    /* print header */
    fprintf(fp, "    ibrch                    type             ileft irite ichld args...\n");

    /* print global Attributes */
    for (iattr = 0; iattr < MODL->nattr; iattr++) {
        if (MODL->attr[iattr].type != ATTRCSYS) {
            fprintf(fp, "                                              attr: %-20s %-20s\n",
                    MODL->attr[iattr].name,
                    MODL->attr[iattr].defn);
        } else {
            fprintf(fp, "                                              csys: %-20s %-20s\n",
                    MODL->attr[iattr].name,
                    MODL->attr[iattr].defn);
            }
    }

    /* print each Branch */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {

        fprintf(fp, "    %5d", ibrch);

        if (MODL->brch[ibrch].ileft == -2 &&
            MODL->brch[ibrch].irite == -2 &&
            MODL->brch[ibrch].ichld == -2   ) {
            fprintf(fp, " [-] ");
        } else if (MODL->brch[ibrch].dirty > 0) {
            if (MODL->brch[ibrch].actv == OCSM_ACTIVE    ) fprintf(fp, " [a] ");
            if (MODL->brch[ibrch].actv == OCSM_SUPPRESSED) fprintf(fp, " [s] ");
            if (MODL->brch[ibrch].actv == OCSM_INACTIVE  ) fprintf(fp, " [i] ");
            if (MODL->brch[ibrch].actv == OCSM_DEFERRED  ) fprintf(fp, " [d] ");
        } else {
            if (MODL->brch[ibrch].actv == OCSM_ACTIVE    ) fprintf(fp, " (a) ");
            if (MODL->brch[ibrch].actv == OCSM_SUPPRESSED) fprintf(fp, " (s) ");
            if (MODL->brch[ibrch].actv == OCSM_INACTIVE  ) fprintf(fp, " (i) ");
            if (MODL->brch[ibrch].actv == OCSM_DEFERRED  ) fprintf(fp, " (d) ");
        }

        fprintf(fp, " %-14s", MODL->brch[ibrch].name);

        nindent = MODL->brch[ibrch].indent;
        for (i = 0; i < nindent; i++) {
            fprintf(fp, ".");
        }

        fprintf(fp, "%-9s", ocsmGetText(MODL->brch[ibrch].type));

        for (i = nindent; i < 8; i++) {
            fprintf(fp, " ");
        }

        fprintf(fp, "%5d %5d %5d", MODL->brch[ibrch].ileft,
                                   MODL->brch[ibrch].irite,
                                   MODL->brch[ibrch].ichld);

        if (MODL->brch[ibrch].narg >= 1) {
            fprintf(fp, " {%s}",   MODL->brch[ibrch].arg1    );
        }
        if (MODL->brch[ibrch].narg >= 2) {
            fprintf(fp, " {%s}",   MODL->brch[ibrch].arg2    );
        }
        if (MODL->brch[ibrch].narg >= 3) {
            fprintf(fp, " {%s}",   MODL->brch[ibrch].arg3    );
        }
        if (MODL->brch[ibrch].narg >= 4) {
            fprintf(fp, " {%s}",   MODL->brch[ibrch].arg4    );
        }
        if (MODL->brch[ibrch].narg >= 5) {
            fprintf(fp, " {%s}",   MODL->brch[ibrch].arg5    );
        }
        if (MODL->brch[ibrch].narg >= 6) {
            fprintf(fp, " {%s}",   MODL->brch[ibrch].arg6    );
        }
        if (MODL->brch[ibrch].narg >= 7) {
            fprintf(fp, " {%s}",   MODL->brch[ibrch].arg7    );
        }
        if (MODL->brch[ibrch].narg >= 8) {
            fprintf(fp, " {%s}",   MODL->brch[ibrch].arg8    );
        }
        if (MODL->brch[ibrch].narg >= 9) {
            fprintf(fp, " {%s}",   MODL->brch[ibrch].arg9    );
        }

        fprintf(fp, "\n");

        for (iattr = 0; iattr < MODL->brch[ibrch].nattr; iattr++) {
            if (MODL->brch[ibrch].attr[iattr].type != ATTRCSYS) {
                fprintf(fp, "                                              attr: %-20s %-20s\n",
                        MODL->brch[ibrch].attr[iattr].name,
                        MODL->brch[ibrch].attr[iattr].defn);
            } else {
                fprintf(fp, "                                              csys: %-20s %-20s\n",
                        MODL->brch[ibrch].attr[iattr].name,
                        MODL->brch[ibrch].attr[iattr].defn);
            }
        }

        for (imprp = 0; imprp < MODL->brch[ibrch].nmprp; imprp++) {
            fprintf(fp, "                                              mprp: %-20s %10.5f\n",
                    MODL->brch[ibrch].mprp[imprp].name,
                    MODL->brch[ibrch].mprp[imprp].val);
        }
    }

cleanup:
    if (fp != NULL && strlen(filename) > 0) fclose(fp);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetArg - get an Argument for a Branch                          *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetArg(void   *modl,                /* (in)  pointer to MODL */
           int    ibrch,                /* (in)  Branch index (1:nbrch) */
           int    iarg,                 /* (in)  Argument index (1:narg) */
           char   defn[],               /* (out) Argument definition (at least MAX_STRVAL_LEN long) */
           double *value,               /* (out) Argument value */
           double *dot)                 /* (out) Argument velocity */
{
    int       status = SUCCESS;         /* (out) return status */

    char      str[MAX_STRVAL_LEN];
    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmGetArg);

    /* --------------------------------------------------------------- */

    /* default return values */
    defn[0] = '\0';
    *value  = 0;
    str[0]  = '\0';

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* check that valid iarg is given */
    if (iarg < 1 || iarg > MODL->brch[ibrch].narg) {
        status = OCSM_ILLEGAL_ARG_INDEX;
        goto cleanup;
    }

    /* return the definition and current value */
    if (iarg == 1) {
        STRNCPY(defn, MODL->brch[ibrch].arg1, MAX_STRVAL_LEN);
    }
    if (iarg == 2) {
        STRNCPY(defn, MODL->brch[ibrch].arg2, MAX_STRVAL_LEN);
    }
    if (iarg == 3) {
        STRNCPY(defn, MODL->brch[ibrch].arg3, MAX_STRVAL_LEN);
    }
    if (iarg == 4) {
        STRNCPY(defn, MODL->brch[ibrch].arg4, MAX_STRVAL_LEN);
    }
    if (iarg == 5) {
        STRNCPY(defn, MODL->brch[ibrch].arg5, MAX_STRVAL_LEN);
    }
    if (iarg == 6) {
        STRNCPY(defn, MODL->brch[ibrch].arg6, MAX_STRVAL_LEN);
    }
    if (iarg == 7) {
        STRNCPY(defn, MODL->brch[ibrch].arg7, MAX_STRVAL_LEN);
    }
    if (iarg == 8) {
        STRNCPY(defn, MODL->brch[ibrch].arg8, MAX_STRVAL_LEN);
    }
    if (iarg == 9) {
        STRNCPY(defn, MODL->brch[ibrch].arg9, MAX_STRVAL_LEN);
    }

    status = str2val(defn, MODL, value, dot, str);

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSetArg - set an Argument for a Branch                          *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSetArg(void   *modl,                /* (in)  pointer to MODL */
           int    ibrch,                /* (in)  Branch index (1:nbrch) */
           int    iarg,                 /* (in)  Argument index (1:narg) */
           char   defn[])               /* (in)  Argument definition */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmSetArg);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* check that valid iarg is given */
    if (iarg < 1 || iarg > MODL->brch[ibrch].narg) {
        status = OCSM_ILLEGAL_ARG_INDEX;
        goto cleanup;
    }

    /* if this has a perturbed Body, free it first */
    status = removePerturbation(MODL);
    CHECK_STATUS(removePerturbation);

    /* remove any velocity information */
    status = removeVels(MODL, 0);
    CHECK_STATUS(removeVels);

    /* save the definition */
    if        (iarg == 1) {
        if (strcmp(defn, MODL->brch[ibrch].arg1) != 0) {
            FREE(  MODL->brch[ibrch].arg1);
            MALLOC(MODL->brch[ibrch].arg1, char, (int)(STRLEN(defn)+1));
            strcpy(MODL->brch[ibrch].arg1,                    defn    );
        }
    } else if (iarg == 2) {
        if (strcmp(defn, MODL->brch[ibrch].arg2) != 0) {
            FREE(  MODL->brch[ibrch].arg2);
            MALLOC(MODL->brch[ibrch].arg2, char, (int)(STRLEN(defn)+1));
            strcpy(MODL->brch[ibrch].arg2,                    defn    );
        }
    } else if (iarg == 3) {
        if (strcmp(defn, MODL->brch[ibrch].arg3) != 0) {
            FREE(  MODL->brch[ibrch].arg3);
            MALLOC(MODL->brch[ibrch].arg3, char, (int)(STRLEN(defn)+1));
            strcpy(MODL->brch[ibrch].arg3,                    defn    );
        }
    } else if (iarg == 4) {
        if (strcmp(defn, MODL->brch[ibrch].arg4) != 0) {
            FREE(  MODL->brch[ibrch].arg4);
            MALLOC(MODL->brch[ibrch].arg4, char, (int)(STRLEN(defn)+1));
            strcpy(MODL->brch[ibrch].arg4,                    defn    );
        }
    } else if (iarg == 5) {
        if (strcmp(defn, MODL->brch[ibrch].arg5) != 0) {
            FREE(  MODL->brch[ibrch].arg5);
            MALLOC(MODL->brch[ibrch].arg5, char, (int)(STRLEN(defn)+1));
            strcpy(MODL->brch[ibrch].arg5,                    defn    );
        }
    } else if (iarg == 6) {
        if (strcmp(defn, MODL->brch[ibrch].arg6) != 0) {
            FREE(  MODL->brch[ibrch].arg6);
            MALLOC(MODL->brch[ibrch].arg6, char, (int)(STRLEN(defn)+1));
            strcpy(MODL->brch[ibrch].arg6,                    defn    );
        }
    } else if (iarg == 7) {
        if (strcmp(defn, MODL->brch[ibrch].arg7) != 0) {
            FREE(  MODL->brch[ibrch].arg7);
            MALLOC(MODL->brch[ibrch].arg7, char, (int)(STRLEN(defn)+1));
            strcpy(MODL->brch[ibrch].arg7,                    defn    );
        }
    } else if (iarg == 8) {
        if (strcmp(defn, MODL->brch[ibrch].arg8) != 0) {
            FREE(  MODL->brch[ibrch].arg8);
            MALLOC(MODL->brch[ibrch].arg8, char, (int)(STRLEN(defn)+1));
            strcpy(MODL->brch[ibrch].arg8,                    defn    );
        }
    } else if (iarg == 9) {
        if (strcmp(defn, MODL->brch[ibrch].arg9) != 0) {
            FREE(  MODL->brch[ibrch].arg9);
            MALLOC(MODL->brch[ibrch].arg9, char, (int)(STRLEN(defn)+1));
            strcpy(MODL->brch[ibrch].arg9,                    defn    );
        }
    }

    /* mark the Branch as dirty */
    MODL->brch[ibrch].dirty = 1;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmRetAttr - return an Attribute for a Branch by index            *
 *                                                                      *
 ************************************************************************
 */

int
ocsmRetAttr(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            int    iattr,               /* (in)  Attribute index (1:nattr) */
            char   aname[],             /* (out) Attribute name  (at least MAX_STRVAL_LEN long) */
            char   avalue[])            /* (out) Attribute value (at least MAX_STRVAL_LEN long) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmRetAttr);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* check that value iattr is given */
    if (iattr < 1 || iattr > MODL->brch[ibrch].nattr) {
        status = OCSM_ILLEGAL_ARG_INDEX;
        goto cleanup;
    }

    /* return the name and value */
    if (MODL->brch[ibrch].attr[iattr-1].type != ATTRCSYS) {
        STRNCPY(aname,  MODL->brch[ibrch].attr[iattr-1].name, MAX_STRVAL_LEN);
        STRNCPY(avalue, MODL->brch[ibrch].attr[iattr-1].defn, MAX_STRVAL_LEN);
    } else {
        status = OCSM_ILLEGAL_ARG_INDEX;
        goto cleanup;
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetAttr - get an Attribute for a Branch by name                *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetAttr(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            char   aname[],             /* (in)  Attribute name */
            char   avalue[])            /* (out) Attribute value (at least MAX_STRVAL_LEN long) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       iattr, jattr;

    ROUTINE(ocsmGetAttr);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 0 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* global Attribute */
    if (ibrch == 0) {

        /* determine the Attribute index (if any) */
        iattr = -1;
        for (jattr = 0; jattr < MODL->nattr; jattr++) {
            if (strcmp(MODL->attr[jattr].name, aname) == 0) {
                iattr = jattr;
                break;
            }
        }

        /* error if Attribute name is not found */
        if (iattr < 0) {
            status = OCSM_NAME_NOT_FOUND;
            goto cleanup;
        }

        /* return the value */
        STRNCPY(avalue, MODL->attr[iattr].defn, MAX_STRVAL_LEN);

    /* Attribute for Branch */
    } else {

        /* determine the Attribute index (if any) */
        iattr = -1;
        for (jattr = 0; jattr < MODL->brch[ibrch].nattr; jattr++) {
            if (strcmp(MODL->brch[ibrch].attr[jattr].name, aname) == 0 &&
                       MODL->brch[ibrch].attr[jattr].type != ATTRCSYS    ) {
                iattr = jattr;
                break;
            }
        }

        /* error if Attribute name is not found */
        if (iattr < 0) {
            status = OCSM_NAME_NOT_FOUND;
            goto cleanup;
        }

        /* return the value */
        STRNCPY(avalue, MODL->brch[ibrch].attr[iattr].defn, MAX_STRVAL_LEN);
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSetAttr - set an Attribute for a Branch                        *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSetAttr(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            char   aname[],             /* (in)  Attribute name */
            char   avalue[])            /* (in)  Attribute value (or blank to delete) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       iattr, jattr, jbrch;
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(ocsmSetAttr);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 0 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* if aname is a NULL, return now */
    if (STRLEN(aname) == 0) {
        SPRINT0(1, "WARNING:: NULL name given for Attribute");
        (MODL->nwarn)++;
        goto cleanup;

    /* if avalue is a NULL string, return now */
    } else if (strcmp(avalue, "$") == 0) {
        SPRINT1(1, "WARNING:: NULL value given for Attribute \"%s\"", aname);
        (MODL->nwarn)++;
        goto cleanup;
    }

    /* global Attribute */
    if (ibrch == 0) {

        /* determine the Attribute index (if any) */
        iattr = -1;
        for (jattr = 0; jattr < MODL->nattr; jattr++) {
            if (strcmp(MODL->attr[jattr].name, aname) == 0) {
                iattr = jattr;
                break;
            }
        }

        /* adding or modifying an Attribute */
        if (STRLEN(avalue) > 0) {
            /* if Attribute is not found already, create a new one */
            if (iattr < 0) {
                MODL->nattr++;

                RALLOC(MODL->attr, attr_T, MODL->nattr);

                iattr = MODL->nattr - 1;
                MODL->attr[iattr].name = NULL;
                MODL->attr[iattr].defn = NULL;
                MODL->attr[iattr].type = ATTRREAL;
            }

            /* set the Attribute's name and value */
            FREE(  MODL->attr[iattr].name);
            MALLOC(MODL->attr[iattr].name,  char, (int)(STRLEN(aname)+1));
            strcpy(MODL->attr[iattr].name,                     aname    );

            FREE(  MODL->attr[iattr].defn);
            MALLOC(MODL->attr[iattr].defn, char, (int)(STRLEN(avalue)+1));
            strcpy(MODL->attr[iattr].defn,                    avalue    );

        /* deleting an Attribute */
        } else {
            if (iattr < 0) {
                status = OCSM_NAME_NOT_FOUND;
                goto cleanup;
            }

            MODL->nattr--;

            for (jattr = iattr; jattr < MODL->nattr; jattr++) {
                MODL->attr[jattr].name = MODL->attr[jattr+1].name;
                MODL->attr[jattr].defn = MODL->attr[jattr+1].defn;
                MODL->attr[jattr].type = MODL->attr[jattr+1].type;
            }

            FREE(MODL->attr[MODL->nattr].name);
            FREE(MODL->attr[MODL->nattr].defn);
        }

        /* mark all the Branches as dirty */
        for (jbrch = 1; jbrch <= MODL->nbrch; jbrch++) {
            MODL->brch[jbrch].dirty = 1;
        }

    /* Attribute for Branch */
    } else {

        /* determine the Attribute index (if any) */
        iattr = -1;
        for (jattr = 0; jattr < MODL->brch[ibrch].nattr; jattr++) {
            if (strcmp(MODL->brch[ibrch].attr[jattr].name, aname) == 0) {
                iattr = jattr;
                break;
            }
        }

        /* adding or modifying an Attribute */
        if (STRLEN(avalue) > 0) {
            /* if Attribute is not found already, create a new one */
            if (iattr < 0) {
                MODL->brch[ibrch].nattr++;

                RALLOC(MODL->brch[ibrch].attr, attr_T, MODL->brch[ibrch].nattr);

                iattr = MODL->brch[ibrch].nattr - 1;
                MODL->brch[ibrch].attr[iattr].name = NULL;
                MODL->brch[ibrch].attr[iattr].defn = NULL;
                MODL->brch[ibrch].attr[iattr].type = ATTRREAL;
            } else if (MODL->brch[ibrch].attr[iattr].type == ATTRCSYS) {
                status = OCSM_NAME_NOT_UNIQUE;
                goto cleanup;
            }

            /* set the Attribute's name and value */
            FREE(  MODL->brch[ibrch].attr[iattr].name);
            MALLOC(MODL->brch[ibrch].attr[iattr].name,  char, (int)(STRLEN(aname)+1));
            strcpy(MODL->brch[ibrch].attr[iattr].name,                     aname    );

            FREE(  MODL->brch[ibrch].attr[iattr].defn);
            MALLOC(MODL->brch[ibrch].attr[iattr].defn, char, (int)(STRLEN(avalue)+1));
            strcpy(MODL->brch[ibrch].attr[iattr].defn,                    avalue    );

        /* deleting an Attribute */
        } else {
            if (iattr < 0) {
                status = OCSM_NAME_NOT_FOUND;
                goto cleanup;
            } else if (MODL->brch[ibrch].attr[iattr].type == ATTRCSYS) {
                status = OCSM_NAME_NOT_FOUND;
                goto cleanup;
            }

            MODL->brch[ibrch].nattr--;

            for (jattr = iattr; jattr < MODL->brch[ibrch].nattr; jattr++) {
                MODL->brch[ibrch].attr[jattr].name = MODL->brch[ibrch].attr[jattr+1].name;
                MODL->brch[ibrch].attr[jattr].defn = MODL->brch[ibrch].attr[jattr+1].defn;
                MODL->brch[ibrch].attr[jattr].type = MODL->brch[ibrch].attr[jattr+1].type;
            }

            FREE(MODL->brch[ibrch].attr[MODL->brch[ibrch].nattr].name);
            FREE(MODL->brch[ibrch].attr[MODL->brch[ibrch].nattr].defn);
        }

        /* mark the Branch as dirty */
        MODL->brch[ibrch].dirty = 1;
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmTraceAttrs - trace definition of all Attributes                *
 *                                                                      *
 ************************************************************************
 */

int
ocsmTraceAttrs(void   *modl,            /* (in)  pointer to MODL */
               char   pattern[],        /* (in)  pattern of Attributes to match */
               char   *info[])          /* (out) info about the Attributes (freeable) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       mchar, ibrch, iattr;
    char      line[MAX_LINE_LEN+1];
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(ocsmTraceAttrs);

    /* --------------------------------------------------------------- */

    /* initialize output buffer */
    mchar = 1000;
    MALLOC(*info, char, mchar);
    *info[0] = '\0';

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    snprintf(line, MAX_LINE_LEN, "Trace of Attributes matching \"%s\":\n", pattern);
    if (strlen(*info)+strlen(line) >= mchar) {
        mchar += strlen(line) + 1000;
        RALLOC(*info, char, mchar);
    }
    strcat(*info, line);

    /* global Attributes */
    for (iattr = 0; iattr < MODL->nattr; iattr++) {
        if (matches(pattern, MODL->attr[iattr].name) == 0) continue;

        snprintf(line, MAX_LINE_LEN, "    \"%s\" is a global Attribute\n",
                 MODL->attr[iattr].name);
        if (strlen(*info)+strlen(line) >= mchar) {
            mchar += strlen(line) + 1000;
            RALLOC(*info, char, mchar);
        }
        strcat(*info, line);
    }

    /* loop through all Branches */
    for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {
        if (MODL->brch[ibrch].ileft == -2 &&
            MODL->brch[ibrch].irite == -2 &&
            MODL->brch[ibrch].ichld == -2   ) continue;

        for (iattr = 0; iattr < MODL->brch[ibrch].nattr; iattr++) {
            if (matches(pattern, MODL->brch[ibrch].attr[iattr].name) == 0) continue;

            snprintf(line, MAX_LINE_LEN, "    \"%s\" applied to [[%s:%d]]\n",
                     MODL->brch[ibrch].attr[iattr].name, MODL->brch[ibrch].filename, MODL->brch[ibrch].linenum);
            if (strlen(*info)+strlen(line) >= mchar) {
                mchar += strlen(line) + 1000;
                RALLOC(*info, char, mchar);
            }
            strcat(*info, line);
        }
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmRetCsys - return a Csystem for a Branch by index               *
 *                                                                      *
 ************************************************************************
 */

int
ocsmRetCsys(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            int    icsys,               /* (in)  Csystem index (1:nattr) */
            char   cname[],             /* (out) Csystem name  (at least MAX_STRVAL_LEN long) */
            char   cvalue[])            /* (out) Csystem value (at least MAX_STRVAL_LEN long) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmRetCsys);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* check that value icsys is given */
    if (icsys < 1 || icsys > MODL->brch[ibrch].nattr) {
        status = OCSM_ILLEGAL_ARG_INDEX;
        goto cleanup;
    }

    /* return the name and value */
    if (MODL->brch[ibrch].attr[icsys-1].type == ATTRCSYS) {
        STRNCPY(cname,  MODL->brch[ibrch].attr[icsys-1].name, MAX_STRVAL_LEN);
        STRNCPY(cvalue, MODL->brch[ibrch].attr[icsys-1].defn, MAX_STRVAL_LEN);
    } else {
        status = OCSM_ILLEGAL_ARG_INDEX;
        goto cleanup;
    }

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetCsys - get a Csystem for a Branch by name                   *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetCsys(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            char   cname[],             /* (in)  Csystem name */
            char   cvalue[])            /* (out) Csystem value (at least MAX_STRVAL_LEN long) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       icsys, jcsys;

    ROUTINE(ocsmGetCsys);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* determine the Csystem index (if any) */
    icsys = -1;
    for (jcsys = 0; jcsys < MODL->brch[ibrch].nattr; jcsys++) {
        if (strcmp(MODL->brch[ibrch].attr[jcsys].name, cname) == 0 &&
                   MODL->brch[ibrch].attr[jcsys].type == ATTRCSYS    ) {
            icsys = jcsys;
            break;
        }
    }

    /* error if Csystem name is not found */
    if (icsys < 0) {
        status = OCSM_NAME_NOT_FOUND;
        goto cleanup;
    }

    /* return the value */
    STRNCPY(cvalue, MODL->brch[ibrch].attr[icsys].defn, MAX_STRVAL_LEN);

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSetCsys - set a Csystem for a Branch                           *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSetCsys(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            char   cname[],             /* (in)  Csystem name */
            char   cvalue[])            /* (in)  Csystem value (or blank to delete) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       icsys, jcsys;
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(ocsmSetCsys);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* if cname is NULL, return now */
    if (STRLEN(cname) == 0) {
        SPRINT0(1, "WARNING:: NULL name given for Csystem");
        (MODL->nwarn)++;
        goto cleanup;

    /* if cvalue is a NULL string, return now */
    } else if (strcmp(cvalue, "$") == 0) {
        SPRINT1(1, "WARNING:: NULL value given for Csystem \"%s\"", cname);
        (MODL->nwarn)++;
        goto cleanup;
    }

    /* determine the Csystem index (if any) */
    icsys = -1;
    for (jcsys = 0; jcsys < MODL->brch[ibrch].nattr; jcsys++) {
        if (strcmp(MODL->brch[ibrch].attr[jcsys].name, cname) == 0) {
            icsys = jcsys;
            break;
        }
    }

    /* adding or modifying a Csystem */
    if (STRLEN(cvalue) > 0) {
        /* if Csystem is not found already, create a new one */
        if (icsys < 0) {
            MODL->brch[ibrch].nattr++;

            RALLOC(MODL->brch[ibrch].attr, attr_T, MODL->brch[ibrch].nattr);

            icsys = MODL->brch[ibrch].nattr - 1;
            MODL->brch[ibrch].attr[icsys].name = NULL;
            MODL->brch[ibrch].attr[icsys].defn = NULL;
            MODL->brch[ibrch].attr[icsys].type = ATTRCSYS;
        } else if (MODL->brch[ibrch].attr[icsys].type != ATTRCSYS) {
            status = OCSM_NAME_NOT_UNIQUE;
            goto cleanup;
        }

        /* set the Csystem's name and value */
        FREE(  MODL->brch[ibrch].attr[icsys].name);
        MALLOC(MODL->brch[ibrch].attr[icsys].name,  char, (int)(STRLEN(cname )+1));
        strcpy(MODL->brch[ibrch].attr[icsys].name,                     cname     );

        FREE(  MODL->brch[ibrch].attr[icsys].defn);
        MALLOC(MODL->brch[ibrch].attr[icsys].defn, char, (int)(STRLEN(cvalue)+1));
        strcpy(MODL->brch[ibrch].attr[icsys].defn,                    cvalue    );

    /* deleting a Csystem */
    } else {
        if (icsys < 0) {
            status = OCSM_NAME_NOT_FOUND;
            goto cleanup;
        } else if (MODL->brch[ibrch].attr[icsys].type != ATTRCSYS) {
            status = OCSM_NAME_NOT_FOUND;
            goto cleanup;
        }

        MODL->brch[ibrch].nattr--;

        for (jcsys = icsys; jcsys < MODL->brch[ibrch].nattr; jcsys++) {
            MODL->brch[ibrch].attr[jcsys].name = MODL->brch[ibrch].attr[jcsys+1].name;
            MODL->brch[ibrch].attr[jcsys].defn = MODL->brch[ibrch].attr[jcsys+1].defn;
            MODL->brch[ibrch].attr[jcsys].type = MODL->brch[ibrch].attr[jcsys+1].type;
        }

        FREE(MODL->brch[ibrch].attr[MODL->brch[ibrch].nattr].name);
        FREE(MODL->brch[ibrch].attr[MODL->brch[ibrch].nattr].defn);
    }

    /* mark the Branch as dirty */
    MODL->brch[ibrch].dirty = 1;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmPrintAttrs - print global Attributes to file                   *
 *                                                                      *
 ************************************************************************
 */

int
ocsmPrintAttrs(void   *modl,            /* (in)  pointer to MODL */
               char   filename[])       /* (in)  file to which output is appended (or "" for stdout) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       iattr;
    FILE      *fp=NULL;

    ROUTINE(ocsmPrintAttrs);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    if (strlen(filename) == 0) {
        fp = stdout;
    } else {
        fp = fopen(filename, "a");
        if (fp == NULL) {
            status = OCSM_FILE_NOT_FOUND;
            goto cleanup;
        }
    }

    for (iattr = 0; iattr < MODL->nattr; iattr++) {
        fprintf(fp, "    %-24s -> %24s\n",
                MODL->attr[iattr].name,
                MODL->attr[iattr].defn);
    }

cleanup:
    if (fp != NULL && strlen(filename) > 0) fclose(fp);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetName - get the name of a Branch                             *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetName(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            char   name[])              /* (out) Branch name (at least MAX_STRVAL_LEN long) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmGetName);

    /* --------------------------------------------------------------- */

    /* default return values */
    name[0] = '\0';

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* return the name */
    STRNCPY(name, MODL->brch[ibrch].name, MAX_STRVAL_LEN);

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSetName - set the name for a Branch                            *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSetName(void   *modl,               /* (in)  pointer to MODL */
            int    ibrch,               /* (in)  Branch index (1:nbrch) */
            char   name[])              /* (in)  Branch name */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       jbrch;

    ROUTINE(ocsmSetName);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (name == NULL) {
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    }

    /* check that valid ibrch is given */
    if (ibrch < 1 || ibrch > MODL->nbrch) {
        status = OCSM_ILLEGAL_BRCH_INDEX;
        goto cleanup;
    }

    /* make sure that the new name is unique */
    for (jbrch = 1; jbrch <= MODL->nbrch; jbrch++) {
        if (strcmp(name, MODL->brch[jbrch].name) == 0) {
            status = OCSM_NAME_NOT_UNIQUE;
            goto cleanup;
        } else if (strncmp(name, "Brch_", 5) == 0) {
            status = OCSM_NAME_NOT_UNIQUE;
            goto cleanup;
        }
    }

    /* name cannot contain ">" */
    if (strstr(name, ">") != NULL) {
        status = OCSM_ILLEGAL_VALUE;
        goto cleanup;
    }

    /* set the name */
    FREE(MODL->brch[ibrch].name);

    MALLOC(MODL->brch[ibrch].name, char, (int)(STRLEN(name)+1));
    strcpy(MODL->brch[ibrch].name,                    name    );

    /* mark the Branch as dirty */
    MODL->brch[ibrch].dirty = 1;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetSketch - get string data associated with a Sketch           *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetSketch(void   *modl,             /* (in)  pointer to MODL */
              int    ibrch,             /* (in)  Branch index (1:nbrch) within Sketch */
              int    maxlen,            /* (in)  length of begs, vars, cons, and segs */
              char   begs[],            /* (out) string with SKBEG variables */
              char   vars[],            /* (out) string with Sketch variables */
              char   cons[],            /* (out) string with Sketch constraints */
              char   segs[])            /* (out) string with Sketch segments */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       jbrch, iskbeg, iskend, index1, index2, ibeg, iend;
    double    value, xval, yval, zval, rel, dot;
    char      *token=NULL, str[MAX_STRVAL_LEN];

    ROUTINE(ocsmGetSketch);

    /* --------------------------------------------------------------- */

    MODL->sigCode    = 0;
    MODL->sigMesg[0] = '\0';

    /* default returns */
    begs[0] = '\0';
    vars[0] = '\0';
    cons[0] = '\0';
    segs[0] = '\0';

    if        (MODL->brch[ibrch].type == OCSM_SKBEG) {
        iskbeg = ibrch;
        iskend = ibrch + 1;
    } else if (MODL->brch[ibrch].type == OCSM_SKEND) {
        iskbeg = ibrch - 1;
        iskend = ibrch;
    } else {
        iskbeg = ibrch;
        iskend = ibrch;
    }

    /* find SKBEG at or before ibrch */
    while (iskbeg > 0 && MODL->brch[iskbeg].type != OCSM_SKBEG) {
        if (MODL->brch[iskbeg].type == OCSM_SKEND) {
            status = signalError(MODL, OCSM_ILLEGAL_BRCH_INDEX,
                                 "ibrch=%d is not within a Sketch (SKEND found first)", ibrch);
            goto cleanup;
        }
        iskbeg--;
        if (iskbeg <= 0) {
            status = signalError(MODL, OCSM_ILLEGAL_BRCH_INDEX,
                                 "ibrch=%d is not within a Sketch (at beginning)", ibrch);
            goto cleanup;
        }
    }

    /* find SKEND at or after ibrch */
    while (iskend <= MODL->nbrch && MODL->brch[iskend].type != OCSM_SKEND) {
        if (MODL->brch[iskend].type == OCSM_SKBEG) {
            status = signalError(MODL, OCSM_ILLEGAL_BRCH_INDEX,
                                 "ibrch=%d is not within a Sketch (SKBEG found first)", ibrch);
            goto cleanup;
        }
        iskend++;
        if (iskend <= 0) {
            status = signalError(MODL, OCSM_ILLEGAL_BRCH_INDEX,
                                 "ibrch=%d is not within a Sketch (at end)", ibrch);
            goto cleanup;
        }
    }

    MALLOC(token, char, maxlen+1);

    /* build the begs string */
    status = str2val(MODL->brch[iskbeg].arg1, MODL, &xval, &dot, str);
    CHECK_STATUS(str2val);
    if (STRLEN(str) > 0) {
        status = signalError(MODL, OCSM_WRONG_PMTR_TYPE,
                             "xval cannot have string value (%s)", ibrch, str);
        goto cleanup;
    }

    status = str2val(MODL->brch[iskbeg].arg2, MODL, &yval, &dot, str);
    CHECK_STATUS(str2val);
    if (STRLEN(str) > 0) {
        status = signalError(MODL, OCSM_WRONG_PMTR_TYPE,
                             "yval cannot have string value (%s)", ibrch, str);
        goto cleanup;
    }

    status = str2val(MODL->brch[iskbeg].arg3, MODL, &zval, &dot, str);
    CHECK_STATUS(str2val);
    if (STRLEN(str) > 0) {
        status = signalError(MODL, OCSM_WRONG_PMTR_TYPE,
                             "zval cannot have string value (%s)", ibrch, str);
        goto cleanup;
    }

    status = str2val(MODL->brch[iskbeg].arg3, MODL, &rel,  &dot, str);
    CHECK_STATUS(str2val);
    if (STRLEN(str) > 0) {
        status = signalError(MODL, OCSM_WRONG_PMTR_TYPE,
                             "rel cannot have string value (%s)", ibrch, str);
        goto cleanup;
    }

    snprintf(begs, maxlen, "%s;%f;%s;%f;%s;%f;%s;%f;",
             MODL->brch[iskbeg].arg1, xval,
             MODL->brch[iskbeg].arg2, yval,
             MODL->brch[iskbeg].arg3, zval,
             MODL->brch[iskbeg].arg4, rel );

    /* if there are no statements within the Sketch, return an empty Sketch */
    if (iskend == iskbeg+1) {
        goto cleanup;
    }

    /* make sure that there there are only valid statements in the Sketch */
    for (jbrch = iskbeg+1; jbrch < iskend; jbrch++) {
        if (MODL->brch[jbrch].type != OCSM_SKVAR  &&
            MODL->brch[jbrch].type != OCSM_SKCON  &&
            MODL->brch[jbrch].type != OCSM_LINSEG &&
            MODL->brch[jbrch].type != OCSM_ARC    &&
            MODL->brch[jbrch].type != OCSM_SPLINE &&
            MODL->brch[jbrch].type != OCSM_SSLOPE &&
            MODL->brch[jbrch].type != OCSM_BEZIER   ) {
            status = signalError(MODL, OCSM_ILLEGAL_STATEMENT,
                                 "wrong type (%s) in Sketch", ocsmGetText(MODL->brch[jbrch].type));
            goto cleanup;
        }
    }

    /* make sure that a SKVAR follows the SKBEG */
    if (MODL->brch[iskbeg+1].type != OCSM_SKVAR) {
        status = signalError(MODL, OCSM_ILLEGAL_STATEMENT,
                             "SKVAR does not follow SKBEG");
        goto cleanup;
    }

    /* create the Sketch variables from the SKVAR statement */
    if (STRLEN(MODL->brch[iskbeg+1].arg2) < maxlen-1) {
        STRNCPY(vars, MODL->brch[iskbeg+1].arg2, maxlen);
    } else {
        status = signalError(MODL, OCSM_INTERNAL_ERROR,
                             "buffer overflow writing Sketch variables");
        goto cleanup;
    }

    /* create the Sketch constraints from the SKCON statements */
    jbrch = iskbeg + 2;
    while (jbrch < iskend && MODL->brch[jbrch].type == OCSM_SKCON) {
        status = str2val(MODL->brch[jbrch].arg2, MODL, &value, &dot, str);
        CHECK_STATUS(str2val);
        if (STRLEN(str) > 0) {
            status = signalError(MODL, OCSM_WRONG_PMTR_TYPE,
                                 "stretch constraint value cannot be a string (%s)", ibrch, str);
            goto cleanup;
        }
        index1 = NINT(value);

        status = str2val(MODL->brch[jbrch].arg3, MODL, &value, &dot, str);
        CHECK_STATUS(str2val);
        if (STRLEN(str) > 0) {
            status = signalError(MODL, OCSM_WRONG_PMTR_TYPE,
                                 "stretch constraint value cannot be a string (%s)", ibrch, str);
            goto cleanup;
        }
        index2 = NINT(value);

        snprintf(token, maxlen, "%s;%d;%d;%s;",
                 &(MODL->brch[jbrch].arg1[1]), index1, index2, &(MODL->brch[jbrch].arg4[1]));

        if (STRLEN(cons)+STRLEN(token) < maxlen-1) {
            STRNCAT(cons, token, maxlen);
        } else {
            status = signalError(MODL, OCSM_INTERNAL_ERROR,
                                 "buffer overflow writing Sketch constraints");
            goto cleanup;
        }

        jbrch++;
    }

    /* create the Sketch segments from the LINSEG, ARC, SPLINE, and BEZIER statements */
    ibeg = 1;
    while (jbrch < iskend) {
        if (strncmp(MODL->brch[jbrch].arg1, "::x[", 4) != 0) {
            status = signalError(MODL, OCSM_ILLEGAL_STATEMENT,
                                 "statement does not use ::x[] for x-argument");
            goto cleanup;
        }
        if (strncmp(MODL->brch[jbrch].arg2, "::y[", 4) != 0) {
            status = signalError(MODL, OCSM_ILLEGAL_STATEMENT,
                                 "statement does not use ::y[] for y-argument");
            goto cleanup;
        }
        if (strncmp(MODL->brch[jbrch].arg3, "::z[", 4) != 0 &&
            strcmp( MODL->brch[jbrch].arg3, "0"      ) != 0   ) {
            status = signalError(MODL, OCSM_ILLEGAL_STATEMENT,
                                 "statement does not use ::z[] or 0 for z-argument");
            goto cleanup;
        }

        if (strcmp(MODL->brch[jbrch].arg1, "::x[1]") == 0) {
            iend = 1;
        } else {
            iend = ibeg + 1;
        }

        if        (MODL->brch[jbrch].type == OCSM_LINSEG) {
            snprintf(token, maxlen, "L;%d;%d;", ibeg, iend);
        } else if (MODL->brch[jbrch].type == OCSM_ARC) {
            snprintf(token, maxlen, "C;%d;%d;", ibeg, iend);
        } else if (MODL->brch[jbrch].type == OCSM_SPLINE) {
            snprintf(token, maxlen, "S;%d;%d;", ibeg, iend);
        } else if (MODL->brch[jbrch].type == OCSM_BEZIER) {
            snprintf(token, maxlen, "B;%d;%d;", ibeg, iend);
        }

        STRNCAT(segs, token, maxlen);

        ibeg = iend;
        jbrch++;
    }

cleanup:
    FREE(token);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSolveSketch - solve for new Sketch variables                   *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSolveSketch(void   *modl,           /* (in)  pointer to MODL */
                char   vars_in[],       /* (in)  string with Sketch variables */
                char   cons[],          /* (in)  string with Sketch constraints */
                                        /* (out) string with Sketch constraint changes */
                char   vars_out[])      /* (out) string (1024 long) with new Sketch variables
                                                 or suggested modified constraints
                                                 (if starts with "*") */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;
    sket_T    *sket=NULL;

    int       i, im1, ip1, icount, lentok, index1, index2, ivar, nvar, jpmtr, jndex;
    int       ii, jj, ibeg, iend;
    double    val;
    char      token[MAX_EXPR_LEN], type[MAX_EXPR_LEN], value[MAX_EXPR_LEN];
    char      prefix[MAX_EXPR_LEN], suffix[MAX_EXPR_LEN];
    char      cons_mod[MAX_EXPR_LEN];

    ROUTINE(ocsmSolveSketch);

    /* --------------------------------------------------------------- */

    MALLOC(sket, sket_T, 1);

    /* default (empty) return */
    vars_out[0]      = '\0';
    MODL->sigMesg[0] = '\0';

    /* initialize the Sketch */
    sket->type   = 0;
    sket->size   = 0;
    sket->solved = 0;
    sket->signal = 0;
    sket->nseg   = 0;
    sket->nvar   = 0;
    sket->ncon   = 0;

    /* count the semicolons in vars_in to detemine number of variables */
    nvar = 0;
    for (i = 0; i < STRLEN(vars_in); i++) {
        if (vars_in[i] == ';') nvar++;
    }

    if (nvar%3 == 0) {
        nvar /= 3;
        sket->size = nvar;
    } else {
        status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                             "vars_in contains %d semicolons (which is not evenly divisible by 3)", nvar);
        goto cleanup;
    }

    /* remove internal Parameters that may be left over from a failure
          in a Sketch that was being solved */
    status = delPmtrByName(MODL, "::d");
    if (status != OCSM_NAME_NOT_FOUND) {
        CHECK_STATUS(delPmtrByName);
    }

    status = delPmtrByName(MODL, "::z");
    if (status != OCSM_NAME_NOT_FOUND) {
        CHECK_STATUS(delPmtrByName);
    }

    status = delPmtrByName(MODL, "::y");
    if (status != OCSM_NAME_NOT_FOUND) {
        CHECK_STATUS(delPmtrByName);
    }

    status = delPmtrByName(MODL, "::x");
    if (status != OCSM_NAME_NOT_FOUND) {
        CHECK_STATUS(delPmtrByName);
    }

    /* make the new Sketch variables */
    status = ocsmNewPmtr(MODL, "::x", OCSM_LOCALVAR, 1, sket->size);
    CHECK_STATUS(ocsmNewPmtr);
    sket->ix = MODL->npmtr;

    status = ocsmNewPmtr(MODL, "::y", OCSM_LOCALVAR, 1, sket->size);
    CHECK_STATUS(ocsmNewPmtr);
    sket->iy = MODL->npmtr;

    status = ocsmNewPmtr(MODL, "::z", OCSM_LOCALVAR, 1, sket->size);
    CHECK_STATUS(ocsmNewPmtr);
    sket->iz = MODL->npmtr;

    status = ocsmNewPmtr(MODL, "::d", OCSM_LOCALVAR, 1, sket->size);
    CHECK_STATUS(ocsmNewPmtr);
    sket->id = MODL->npmtr;

    /* pull the variables out of vars_in */
    icount = 0;
    lentok = 0;
    for (i = 0; i < STRLEN(vars_in); i++) {
        if (vars_in[i] != ';') {
            token[lentok++] = vars_in[i];
            token[lentok  ] = '\0';
        } else {
            val = strtod(token, NULL);
            if        (icount%3 == 0) {
                MODL->pmtr[sket->ix].value[icount/3] = val;
                MODL->pmtr[sket->ix].dot[  icount/3] = 0;
            } else if (icount%3 == 1) {
                MODL->pmtr[sket->iy].value[icount/3] = val;
                MODL->pmtr[sket->iy].dot[  icount/3] = 0;
            } else {
                MODL->pmtr[sket->id].value[icount/3] = val;
                MODL->pmtr[sket->id].dot[  icount/3] = 0;
            }

            icount++;
            lentok = 0;
        }
    }

    /* make a list of the Sketch variables */
    sket->nvar = 0;
    for (i = 1; i <= sket->size; i++) {
        sket->ipmtr[sket->nvar] = sket->ix;
        sket->index[sket->nvar] = i-1;
        sket->nvar++;

        sket->ipmtr[sket->nvar] = sket->iy;
        sket->index[sket->nvar] = i-1;
        sket->nvar++;

        if (fabs(MODL->pmtr[sket->id].value[i-1]) > EPS06) {
            sket->ipmtr[sket->nvar] = sket->id;
            sket->index[sket->nvar] = i-1;
            sket->nvar++;
        }
    }

    for (ivar = 0; ivar < sket->nvar; ivar++) {
        jpmtr = sket->ipmtr[ivar];
        jndex = sket->index[ivar];

        SPRINT3(1, "    -> creating  %s[%d] = %10.5f", MODL->pmtr[jpmtr].name, jndex+1,
                MODL->pmtr[jpmtr].value[jndex]);
    }

    /* make the constraints */
    icount = 0;
    lentok = 0;
    index1 = 0;
    index2 = 0;
    for (i = 0; i < STRLEN(cons); i++) {
        if (cons[i] != ';') {
            token[lentok++] = cons[i];
            token[lentok  ] = '\0';
        } else {
            if        (icount%4 == 0) {
                strcpy(type, token);
            } else if (icount%4 == 1) {
                index1 = strtol(token, NULL, 10);
            } else if (icount%4 == 2) {
                index2 = strtol(token, NULL, 10);
            } else {
                strcpy(value, token);

                /* (recursivly) modify value if it contains "::L[", "::I[", "::R[", or "::S[" */
                while (strstr(value, "::L[") != NULL || strstr(value, "::I[") != NULL ||
                       strstr(value, "::R[") != NULL || strstr(value, "::S[") != NULL   ){

                    /* pull out the prefix */
                    prefix[0] = '\0';
                    ibeg      =   0 ;
                    for (ii = 0; ii < STRLEN(value); ii++) {
                        if (value[ii] == ':' && value[ii+1] == ':' && value[ii+3] == '[') {
                            if (value[ii+2] == 'L' || value[ii+2] == 'I' ||
                                value[ii+2] == 'R' || value[ii+2] == 'S'   ) {
                                if (ii > 0) {
                                    STRNCPY(prefix, value, ii);
                                    prefix[ii] = '\0';
                                }
                                ibeg = strtol(&(value[ii+4]), NULL, 10);
                                break;
                            }
                        }
                    }

                    /* check that argument was okay */
                    if (ibeg < 1 || ibeg > sket->size) {
                        status = signalError(MODL, OCSM_ILLEGAL_VALUE,
                                             "::%c[%d] is not defined", value[ii+2], ibeg);
                        goto cleanup;
                    } else if (ibeg < sket->size) {
                        iend = ibeg + 1;
                    } else {
                        iend = 1;
                    }

                    /* pull out the suffix */
                    suffix[0] = '\0';
                    for (jj = ii+3; jj < STRLEN(value); jj++) {
                        if (value[jj] == ']') {
                            strcpy(suffix, &(value[jj+1]));
                            break;
                        }
                    }

                    /* create the replacement text */
                    if        (value[ii+2] == 'L') {
                        snprintf(value, MAX_EXPR_LEN, "%shypot(::y[%d]-::y[%d],::x[%d]-::x[%d])%s",
                                 prefix, iend, ibeg, iend, ibeg, suffix);
                    } else if (value[ii+2] == 'I') {
                        snprintf(value, MAX_EXPR_LEN, "%satan2d(::y[%d]-::y[%d],::x[%d]-::x[%d])%s",
                                 prefix, iend, ibeg, iend, ibeg, suffix);
                    } else if (value[ii+2] == 'R') {
                        snprintf(value, MAX_EXPR_LEN, "%sradius(::x[%d],::y[%d],::d[%d],::x[%d],::y[%d])%s",
                                 prefix, ibeg, ibeg, iend, iend, iend, suffix);
                    } else if (value[ii+2] == 'S') {
                        snprintf(value, MAX_EXPR_LEN, "%ssweep(::x[%d],::y[%d],::d[%d],::x[%d],::y[%d])%s",
                                 prefix, ibeg, ibeg, iend, iend, iend, suffix);
                    }
                }

                if        (strcmp(type, "X") == 0 && index2 <= 0) {
                    sket->ctype[sket->ncon] = 'X';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "::x[%d]-(%s)",
                             index1, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "X") == 0) {
                    sket->ctype[sket->ncon] = 'X';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "Xcent(::x[%d],::y[%d],::d[%d],::x[%d],::y[%d])-(%s)",
                             index1, index1, index2, index2, index2, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "Y") == 0 && index2 <= 0) {
                    sket->ctype[sket->ncon] = 'Y';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "::y[%d]-(%s)",
                             index1, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "Y") == 0) {
                    sket->ctype[sket->ncon] = 'Y';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "Ycent(::x[%d],::y[%d],::d[%d],::x[%d],::y[%d])-(%s)",
                             index1, index1, index2, index2, index2, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "P") == 0) {
                    sket->ctype[sket->ncon] = 'P';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    im1 = index1 - 1; if (im1 == 0           ) im1 = sket->size;
                    ip1 = index1 + 1; if (ip1 == sket->size+1) ip1 = 1;
                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "(abs(turnang(::x[%d],::y[%d],::d[%d],::x[%d],::y[%d],::d[%d],::x[%d],::y[%d]))-90)/57.3",
                             im1, im1, index1, index1, index1, ip1, ip1, ip1);
                } else if (strcmp(type, "T") == 0) {
                    sket->ctype[sket->ncon] = 'T';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    im1 = index1 - 1; if (im1 == 0           ) im1 = sket->size;
                    ip1 = index1 + 1; if (ip1 == sket->size+1) ip1 = 1;
                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "turnang(::x[%d],::y[%d],::d[%d],::x[%d],::y[%d],::d[%d],::x[%d],::y[%d])/57.3",
                             im1, im1, index1, index1, index1, ip1, ip1, ip1);
                } else if (strcmp(type, "A") == 0) {
                    sket->ctype[sket->ncon] = 'A';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    im1 = index1 - 1; if (im1 == 0           ) im1 = sket->size;
                    ip1 = index1 + 1; if (ip1 == sket->size+1) ip1 = 1;
                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "(turnang(::x[%d],::y[%d],::d[%d],::x[%d],::y[%d],::d[%d],::x[%d],::y[%d])-smallang(%s))/57.3",
                             im1, im1, index1, index1, index1, ip1, ip1, ip1, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "Z") == 0) {
                    sket->ctype[sket->ncon] = 'Z';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    if        (index2 == -2) {
                        snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                                 "::x[%d]-::x[%d]",
                                 index1, index1+1);
                    } else if (index2 == -3) {
                        snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                                 "::y[%d]-::y[%d]",
                                 index1-1, index1);
                    }
                } else if (strcmp(type, "W") == 0) {
                    sket->ctype[sket->ncon] = 'W';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "::x[%d]-::x[%d]-(%s)",
                             index2, index1, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "D") == 0) {
                    sket->ctype[sket->ncon] = 'D';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "::y[%d]-::y[%d]-(%s)",
                             index2, index1, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "H") == 0) {
                    sket->ctype[sket->ncon] = 'H';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "::y[%d]-::y[%d]",
                             index1, index2);
                } else if (strcmp(type, "V") == 0) {
                    sket->ctype[sket->ncon] = 'V';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "::x[%d]-::x[%d]",
                             index1, index2);
                } else if (strcmp(type, "I") == 0) {
                    sket->ctype[sket->ncon] = 'I';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "smallang(atan2d(::y[%d]-::y[%d],::x[%d]-::x[%d])-(%s))/57.3",
                             index2, index1, index2, index1, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "L") == 0) {
                    sket->ctype[sket->ncon] = 'L';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "hypot(::y[%d]-::y[%d],::x[%d]-::x[%d])-(%s)",
                             index1, index2, index1, index2, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "R") == 0) {
                    sket->ctype[sket->ncon] = 'R';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "radius(::x[%d],::y[%d],::d[%d],::x[%d],::y[%d])-(%s)",
                             index1, index1, index2, index2, index2, /*@ignore@*/value/*@end@*/);
                } else if (strcmp(type, "S") == 0) {
                    sket->ctype[sket->ncon] = 'S';
                    sket->ipnt[ sket->ncon] = index1;
                    sket->ip1[  sket->ncon] = index2;

                    snprintf(sket->con[sket->ncon++], MAX_EXPR_LEN,
                             "(sweep(::x[%d],::y[%d],::d[%d],::x[%d],::y[%d])-(%s))/57.3",
                             index1, index1, index2, index2, index2, /*@ignore@*/value/*@end@*/);
                }
            }

            icount++;
            lentok = 0;
        }
    }

    /* check that number of constraints matches number of variables */
    if        (sket->nvar > sket->ncon) {
        status = fixSketch(sket, vars_in, cons_mod);
        CHECK_STATUS(fixSketch);

        if (STRLEN(cons_mod) > 0) {
            strcpy(vars_out, cons_mod);
        } else {
            (void) signalError(MODL, OCSM_SINGULAR_MATRIX,
                               "under-constrained since nvar=%d but ncon=%d", sket->nvar, sket->ncon);
        }
        goto removeTemps;
    } else if (sket->nvar < sket->ncon) {
        status = fixSketch(sket, vars_in, cons_mod);
        CHECK_STATUS(fixSketch);

        if (STRLEN(cons_mod) > 0) {
            strcpy(vars_out, cons_mod);
        } else {
            (void) signalError(MODL, OCSM_SINGULAR_MATRIX,
                               "over-constrained since nvar=%d but ncon=%d", sket->nvar, sket->ncon);
        }
        goto removeTemps;
    }

    /* try to solve the sketch */
    status = solveSketch(MODL, sket);

    /* if solver fails, simply return nvar=0 */
    if (status == OCSM_SINGULAR_MATRIX) {
        status = fixSketch(sket, vars_in, cons_mod);
        CHECK_STATUS(fixSketch);

        if (STRLEN(cons_mod) > 0) {
            strcpy(vars_out, cons_mod);
        } else {
            (void) signalError(MODL, OCSM_SINGULAR_MATRIX,
                               "constraints are not independent");
        }
        goto removeTemps;
    } else if (status < SUCCESS) {
        goto removeTemps;
    }

    /* solver succeeded, so write the new variables */
    for (ivar = 0; ivar < nvar; ivar++) {
        snprintf(token, MAX_EXPR_LEN, "%.6lf;%.6lf;%.6lf;",
                 MODL->pmtr[sket->ix].value[ivar],
                 MODL->pmtr[sket->iy].value[ivar],
                 MODL->pmtr[sket->id].value[ivar]);
        strcat(vars_out, token);
    }

removeTemps:
    /* remove ::x[], ::y[], ::z[], and ::d[] if they were generated by a skvar statement */
    if (sket->nvar > 0) {
        status = delPmtrByName(MODL, "::d");
        if (status != OCSM_NAME_NOT_FOUND) {
            CHECK_STATUS(delPmtrByName);
        }

        status = delPmtrByName(MODL, "::z");
        if (status != OCSM_NAME_NOT_FOUND) {
            CHECK_STATUS(delPmtrByName);
        }

        status = delPmtrByName(MODL, "::y");
        if (status != OCSM_NAME_NOT_FOUND) {
            CHECK_STATUS(delPmtrByName);
        }

        status = delPmtrByName(MODL, "::x");
        if (status != OCSM_NAME_NOT_FOUND) {
            CHECK_STATUS(delPmtrByName);
        }
    }

cleanup:
    FREE(sket);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmSaveSketch - overwrite Branches associated with a Sketch       *
 *                                                                      *
 ************************************************************************
 */

int
ocsmSaveSketch(void   *modl,            /* (in)  pointer to MODL */
               int    ibrch,            /* (in)  Branch index (1:nbrch) within Sketch */
               char   vars[],           /* (in)  string with Sketch variables */
               char   cons[],           /* (in)  string with Sketch constraints */
               char   segs[])           /* (in)  string with Sketch segments */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       iskbeg, iskend, jbrch, iafter, icon, ncon, iseg, nseg, i, irel;
    double    value, dot;
    char      *arg1=NULL, *arg2=NULL, *arg3=NULL, *arg4=NULL, *arg5=NULL, *arg6=NULL;
    char      str[MAX_STRVAL_LEN];

    ROUTINE(ocsmSaveSketch);

    /* --------------------------------------------------------------- */

    if (outLevel >= 2) {
        SPRINT0(2, "ocsmSaveSketch: at beginning");
        status = ocsmPrintBrchs(MODL, "");
        CHECK_STATUS(ocsmPrintBrchs);
    }

    MALLOC(arg1, char, MAX_EXPR_LEN);
    MALLOC(arg2, char, MAX_EXPR_LEN);
    MALLOC(arg3, char, MAX_EXPR_LEN);
    MALLOC(arg4, char, MAX_EXPR_LEN);
    MALLOC(arg5, char, MAX_EXPR_LEN);
    MALLOC(arg6, char, MAX_EXPR_LEN);

    if        (MODL->brch[ibrch].type == OCSM_SKBEG) {
        iskbeg = ibrch;
        iskend = ibrch + 1;
    } else if (MODL->brch[ibrch].type == OCSM_SKEND) {
        iskbeg = ibrch - 1;
        iskend = ibrch;
    } else {
        iskbeg = ibrch;
        iskend = ibrch;
    }

    /* find SKBEG at or before ibrch */
    while (iskbeg > 0 && MODL->brch[iskbeg].type != OCSM_SKBEG) {
        if (MODL->brch[iskbeg].type == OCSM_SKEND) {
            status = signalError(MODL, OCSM_ILLEGAL_BRCH_INDEX,
                                 "ibrch=%d is not within a Sketch (SKEND found first)", ibrch);
            goto cleanup;
        }
        iskbeg--;
        if (iskbeg <= 0) {
            status = signalError(MODL, OCSM_ILLEGAL_BRCH_INDEX,
                                 "ibrch=%d is not within a Sketch (at beginning)", ibrch);
            goto cleanup;
        }
    }

    /* find SKEND at or after ibrch */
    while (iskend <= MODL->nbrch && MODL->brch[iskend].type != OCSM_SKEND) {
        if (MODL->brch[iskend].type == OCSM_SKBEG) {
            status = signalError(MODL, OCSM_ILLEGAL_BRCH_INDEX,
                                 "ibrch=%d is not within a Sketch (SKBEG found first)", ibrch);
            goto cleanup;
        }
        iskend++;
        if (iskend <= 0) {
            status = signalError(MODL, OCSM_ILLEGAL_BRCH_INDEX,
                                 "ibrch=%d is not within a Sketch (at end)", ibrch);
            goto cleanup;
        }
    }

    iafter = iskbeg;

    /* determine the number of constraints */
    ncon = 0;
    for (i = 0; i < STRLEN(cons); i++) {
        if (cons[i] == ';') ncon++;
    }
    ncon /= 4;

    /* determine the number of segments */
    nseg = 0;
    for (i = 0; i < STRLEN(segs); i++) {
        if (segs[i] == ';') nseg++;
    }
    nseg /= 3;

    /* delete the Branches between the SKBEG and the SKEND */
    for (jbrch = iskend-1; jbrch > iskbeg; jbrch--) {
        status = ocsmDelBrch(MODL, jbrch);
        CHECK_STATUS(ocsmDelBrch);
    }

    /* determine if Sketch is absolute or relative */
    value = 0;
    dot   = 0;
    status = str2val(MODL->brch[iskbeg].arg4, MODL, &value, &dot, str);
    CHECK_STATUS(str2val);
    if (STRLEN(str) > 0) {
        status = signalError(MODL, OCSM_WRONG_PMTR_TYPE,
                             "sketch type cannot have string value (%s)", ibrch, str);
        goto cleanup;
    }

    irel = NINT(value);

    /* add the SKVAR Branch */
    status = ocsmNewBrch(MODL, iafter++, OCSM_SKVAR, "", -1,
                         "$xy", vars, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    CHECK_STATUS(ocsmNewBrch);

    /* add the SKCON Branches */
    for (icon = 0; icon < ncon; icon++) {
        arg1[0] = '$';
        arg4[0] = '$';
        getToken(cons, 4*icon  , ';', MAX_EXPR_LEN, &(arg1[1]));
        getToken(cons, 4*icon+1, ';', MAX_EXPR_LEN,  arg2    );
        getToken(cons, 4*icon+2, ';', MAX_EXPR_LEN,  arg3    );
        getToken(cons, 4*icon+3, ';', MAX_EXPR_LEN, &(arg4[1]));

        status = ocsmNewBrch(MODL, iafter++, OCSM_SKCON, "", -1,
                             arg1, arg2, arg3, arg4, NULL, NULL, NULL, NULL, NULL);
        CHECK_STATUS(ocsmNewBrch);
    }

    /* add the LINSEG, ARC, SPLINE, and BEZIER Branches */
    for (iseg = 0; iseg < nseg; iseg++) {
        getToken(segs, 3*iseg,   ';', MAX_EXPR_LEN, arg1);
        getToken(segs, 3*iseg+2, ';', MAX_EXPR_LEN, arg6);

        if (irel == 0) {
            snprintf(arg2, MAX_EXPR_LEN, "::x[%s]", arg6);
            snprintf(arg3, MAX_EXPR_LEN, "::y[%s]", arg6);
            snprintf(arg4, MAX_EXPR_LEN, "::z[%s]", arg6);
            snprintf(arg5, MAX_EXPR_LEN, "::d[%s]", arg6);
        } else {
            snprintf(arg2, MAX_EXPR_LEN, "::x[%s]", arg6);
            snprintf(arg3, MAX_EXPR_LEN, "::y[%s]", arg6);
            snprintf(arg4, MAX_EXPR_LEN, "0"            );
            snprintf(arg5, MAX_EXPR_LEN, "::d[%s]", arg6);
        }

        if        (arg1[0] == 'L') {
            status = ocsmNewBrch(MODL, iafter++, OCSM_LINSEG, "", -1,
                                 arg2, arg3, arg4, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS(ocsmNewBrch);
        } else if (arg1[0] == 'C') {
            strcpy(arg6, "$xy");
            status = ocsmNewBrch(MODL, iafter++, OCSM_ARC, "", -1,
                                 arg2, arg3, arg4, arg5, arg6, NULL, NULL, NULL, NULL);
            CHECK_STATUS(ocsmNewBrch);
        } else if (arg1[0] == 'S') {
            status = ocsmNewBrch(MODL, iafter++, OCSM_SPLINE, "", -1,
                                 arg2, arg3, arg4, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS(ocsmNewBrch);
        } else if (arg1[0] == 'B') {
            status = ocsmNewBrch(MODL, iafter++, OCSM_BEZIER, "", -1,
                                 arg2, arg3, arg4, NULL, NULL, NULL, NULL, NULL, NULL);
            CHECK_STATUS(ocsmNewBrch);
        }
    }

    if (outLevel >= 2) {
        SPRINT0(2, "ocsmSaveSketch: after additions");
        status = ocsmPrintBrchs(MODL, "");
        CHECK_STATUS(ocsmPrintBrchs);
    }

cleanup:
    FREE(arg6);
    FREE(arg5);
    FREE(arg4);
    FREE(arg3);
    FREE(arg2);
    FREE(arg1);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmMakeEBody - make an EBody from a given Body                    *
 *                                                                      *
 ************************************************************************
 */

int
ocsmMakeEBody(void   *modl,             /* (in)  pointer to MODL */
              int    ibody,             /* (in)  Body index (1:nbody) */
              double dihedral,          /* (in)  dihedral angle (deg) for implicit .Keeps */
    /*@null@*/char   entList[])         /* (in)  entities to use in EBody (or NULL to delete) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       nnode, inode, nedge, iedge, nface, iface, nkeep, ikeep, ngroup, igroup;
    double    bbox[6], size, params[3];
    char      *pEnd;
    ego       *enodes=NULL, *eedges=NULL, *efaces=NULL;
    ego       newBody, newTess, newEBody, newEFace, *group=NULL;

    ROUTINE(ocsmMakeEBody);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ibody is given */
    if (ibody < 1 || ibody > MODL->nbody) {
        status = OCSM_ILLEGAL_BODY_INDEX;
        goto cleanup;
    }

    /* if attrName is NULL,  then we should delete the EBody (if it exists) */
    if (entList == NULL) {
        if (MODL->body[ibody].eetess != NULL) {
            status = EG_deleteObject(MODL->body[ibody].eetess);
            CHECK_STATUS(EG_deleteObject);

            MODL->body[ibody].eetess = NULL;
        }

        if (MODL->body[ibody].eebody != NULL) {
            status = EG_deleteObject(MODL->body[ibody].eebody);
            CHECK_STATUS(EG_deleteObject);

            MODL->body[ibody].eebody = NULL;
        }

    } else {

        /* make a copy of the Body */
        status = EG_copyObject(MODL->body[ibody].ebody, NULL, &newBody);
        CHECK_STATUS(EG_copyObject);

        status = EG_getBodyTopos(newBody, NULL, NODE, &nnode, &enodes);
        CHECK_STATUS(EG_getBodyTopos);

        status = EG_getBodyTopos(newBody, NULL, EDGE, &nedge, &eedges);
        CHECK_STATUS(EG_getBodyTopos);

        status = EG_getBodyTopos(newBody, NULL, FACE, &nface, &efaces);
        CHECK_STATUS(EG_getBodyTopos);

        /* initialize our place in the entList */
        pEnd = entList - 1;

        /* mark the Nodes to be kept */
        nkeep  = strtol(pEnd+1, &pEnd, 10);
        for (ikeep = 0; ikeep < nkeep; ikeep++) {
            SPLINT_CHECK_FOR_NULL(enodes);

            inode  = strtol(pEnd+1, &pEnd, 10);
            status = EG_attributeAdd(enodes[inode-1], ".Keep", ATTRINT,
                                     1, &inode, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }

        /* mark the Edges to be kept */
        nkeep  = strtol(pEnd+1, &pEnd, 10);
        for (ikeep = 0; ikeep < nkeep; ikeep++) {
            SPLINT_CHECK_FOR_NULL(eedges);

            iedge  = strtol(pEnd+1, &pEnd, 10);
            status = EG_attributeAdd(eedges[iedge-1], ".Keep", ATTRINT,
                                     1, &iedge, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }

        /* copy the Tessellation onto the newBody */
        status = EG_copyObject(MODL->body[ibody].etess, newBody, &newTess);
        CHECK_STATUS(EG_copyObject);

        /* initialize the EBody */
        status = EG_initEBody(newTess, dihedral, &newEBody);
        CHECK_STATUS(EG_initEBody);

        /* make the EFaces */
        ngroup = strtol(pEnd+1, &pEnd, 10);
        while (ngroup > 0) {
            SPLINT_CHECK_FOR_NULL(efaces);

            MALLOC(group, ego, ngroup);
            for (igroup = 0; igroup < ngroup; igroup++) {
                iface  = strtol(pEnd+1, &pEnd, 10);
                group[igroup] = efaces[iface-1];
            }

            status = EG_makeEFace(newEBody, ngroup, group, &newEFace);
            CHECK_STATUS(EG_makeEFace);

            FREE(group);

            /* get ready for next group */
            ngroup = strtol(pEnd+1, &pEnd, 10);
        }

        /* close the EBody */
        status = EG_finishEBody(newEBody);
        CHECK_STATUS(EG_finishEBody);

        /* tessellate the EBody */
        status = EG_getBoundingBox(newEBody, bbox);
        CHECK_STATUS(EG_getBoundingBox);

        size = sqrt(SQR(bbox[3]-bbox[0]) + SQR(bbox[4]-bbox[1]) + SQR(bbox[5]-bbox[2]));

        params[0] = TESS_PARAM_0 * size;
        params[1] = TESS_PARAM_1 * size;
        params[2] = TESS_PARAM_2;

        SPRINT4(1, "--> Tessellating EBody %6d     (%12.5e %12.5e %7.3f)",
                ibody, params[0], params[1], params[2]);

        status = EG_makeTessBody(newEBody, params, &(MODL->body[ibody].eetess));
        CHECK_STATUS(EG_makeTessBody);

        status = EG_attributeAdd(MODL->body[ibody].eetess, ".tessType", ATTRSTRING,
                                 4, NULL, NULL, "Tris");
        CHECK_STATUS(EG_attributeAdd);

        /* remember this EBody */
        MODL->body[ibody].eebody = newEBody;
    }

cleanup:
    if (enodes != NULL) EG_free(enodes);
    if (eedges != NULL) EG_free(eedges);
    if (efaces != NULL) EG_free(efaces);

    FREE(group);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmNewPmtr - create a new Parameter                               *
 *                                                                      *
 ************************************************************************
 */

int
ocsmNewPmtr(void   *modl,               /* (in)  pointer to MODL */
            char   name[],              /* (in)  Parameter name */
            int    type,                /* (in)  Parameter type */
            int    nrow,                /* (in)  number of rows */
            int    ncol)                /* (in)  number of columns */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       ipmtr, i, old_nrow, old_ncol;
    int       resize=0;
    char      newName[MAX_NAME_LEN];
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(ocsmNewPmtr);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* make the newName by prepending the prefix if name starts with
       a colon */
    if (name[0] == ':' && STRLEN(MODL->prefix) > 0) {
        if (STRLEN(name)+STRLEN(MODL->prefix) >= MAX_NAME_LEN) {
            status = OCSM_ILLEGAL_PMTR_NAME;
            goto cleanup;
        }

        snprintf(newName, MAX_NAME_LEN, "%s%s", MODL->prefix, name);
    } else {
        STRNCPY(newName, name, MAX_NAME_LEN);
    }

    /* if an UNKNOWN and Parameter is already a DESPMTR, CFGPMTR, or UNKNOWN, just return */
    if (type == OCSM_UNKNOWN) {
        for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
            if (strcmp(MODL->pmtr[ipmtr].name, newName) == 0) {
                if (MODL->pmtr[ipmtr].type == OCSM_DESPMTR ||
                    MODL->pmtr[ipmtr].type == OCSM_CFGPMTR ||
                    MODL->pmtr[ipmtr].type == OCSM_UNKNOWN   ) {
                    goto cleanup;
                }
            }
        }
    }

    /* if an UNKNOWN and Parameter is already a OUTPMTR or LOCALVAR, update
       the input type */
    if (type == OCSM_UNKNOWN) {
        for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
            if (strcmp(MODL->pmtr[ipmtr].name, newName) == 0) {
                if (MODL->pmtr[ipmtr].type == OCSM_OUTPMTR ||
                    MODL->pmtr[ipmtr].type == OCSM_LOCALVAR   ) {
                    type = MODL->pmtr[ipmtr].type;
                    break;
                }
            }
        }
    }

    /* if a CONPMTR, DESPMTR, or CFGPMTR remove any INTERNAL Parameters
       that might already exist (from a previous ocsmBuild) */
    if (type == OCSM_CONPMTR || type == OCSM_DESPMTR || type == OCSM_CFGPMTR) {
        for (ipmtr = MODL->npmtr; ipmtr > 0; ipmtr--) {
            if (MODL->pmtr[ipmtr].type == OCSM_LOCALVAR) {
                status = ocsmDelPmtr(MODL, ipmtr);
                CHECK_STATUS(ocsmDelPmtr);
            }
        }
    }

    /* if an OUTPMTR Parameter already exists, mark it as needing to be resized */
    resize = 0;
    if (type == OCSM_OUTPMTR) {
        for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
            if (strcmp(MODL->pmtr[ipmtr].name, newName) == 0) {
                if (MODL->pmtr[ipmtr].type == OCSM_OUTPMTR) {
                    resize = ipmtr;
                    break;
                }
            }
        }

    /* check if pmtrName is already defined at this scope */
    } else {
        for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
            if (strcmp(MODL->pmtr[ipmtr].name, newName) == 0                 &&
                MODL->pmtr[ipmtr].scope == MODL->scope[MODL->level]  ) {
                if (type == OCSM_LOCALVAR && MODL->pmtr[ipmtr].nrow == nrow
                                          && MODL->pmtr[ipmtr].ncol == ncol) {
                    status = SUCCESS;
                } else if (type == OCSM_LOCALVAR) {
                    resize = ipmtr;
                    break;
                } else {
                    status = OCSM_NAME_ALREADY_DEFINED;
                }
                goto cleanup;
            }
        }
    }

    /* check that pmtrName is a valid name */
    if (STRLEN(newName) == 0 || STRLEN(newName) >= MAX_NAME_LEN) {
        status = OCSM_ILLEGAL_PMTR_NAME;
        goto cleanup;
    } else if (newName[0] == '@') {
    } else if (isalpha(newName[0]) == 0 && newName[0] != ':') {
        status = OCSM_ILLEGAL_PMTR_NAME;
        goto cleanup;
    } else {
        for (i = 1; i < STRLEN(newName); i++) {
            if        (isalpha(newName[i]) != 0) {
            } else if (isdigit(newName[i]) != 0) {
            } else if (newName[i] == '_') {
            } else if (newName[i] == '@') {
            } else if (newName[i] == ':') {
            } else {
                status = OCSM_ILLEGAL_PMTR_NAME;
                goto cleanup;
            }
        }
    }

    /* check for a valid Parameter type */
    if (type != OCSM_DESPMTR  && type != OCSM_CFGPMTR && type != OCSM_OUTPMTR &&
        type != OCSM_LOCALVAR && type != OCSM_CONPMTR && type != OCSM_UNKNOWN   ) {
        status = OCSM_ILLEGAL_TYPE;
        goto cleanup;
    }

    /* extend the Parameter list (if needed) */
#ifndef __clang_analyzer__
    if (MODL->npmtr >= MODL->mpmtr) {
        MODL->mpmtr += 25;
        RALLOC(MODL->pmtr, pmtr_T, MODL->mpmtr+1);

        for (ipmtr = MODL->npmtr+1; ipmtr <= MODL->mpmtr; ipmtr++) {
            MODL->pmtr[ipmtr].name  = NULL;
            MODL->pmtr[ipmtr].mprop = 0;
            MODL->pmtr[ipmtr].value = NULL;
            MODL->pmtr[ipmtr].dot   = NULL;
            MODL->pmtr[ipmtr].lbnd  = NULL;
            MODL->pmtr[ipmtr].ubnd  = NULL;
            MODL->pmtr[ipmtr].str   = NULL;
        }
    }
#endif

    /* create the new Parameter */
    if (resize == 0) {
        MODL->npmtr += 1;
        ipmtr = MODL->npmtr;

    /* or resize existing Parameter */
    } else {
        ipmtr = resize;
    }

    /* store its name and defn */
    FREE(   MODL->pmtr[ipmtr].name);
    MALLOC( MODL->pmtr[ipmtr].name, char,    STRLEN(newName)+1);
    STRNCPY(MODL->pmtr[ipmtr].name, newName, STRLEN(newName)+1);

    if (nrow <= 0 || ncol <= 0) {
        FREE(MODL->pmtr[ipmtr].value);
        FREE(MODL->pmtr[ipmtr].dot  );
        FREE(MODL->pmtr[ipmtr].lbnd );
        FREE(MODL->pmtr[ipmtr].ubnd );
        FREE(MODL->pmtr[ipmtr].str  );

        MALLOC( MODL->pmtr[ipmtr].str,   char,   MAX_STRVAL_LEN);

        nrow = 0;
        ncol = 0;

    } else if (resize == 0) {
        FREE(MODL->pmtr[ipmtr].value);
        FREE(MODL->pmtr[ipmtr].dot  );
        FREE(MODL->pmtr[ipmtr].lbnd );
        FREE(MODL->pmtr[ipmtr].ubnd );
        FREE(MODL->pmtr[ipmtr].str  );

        MALLOC(MODL->pmtr[ipmtr].value, double, nrow*ncol);
        MALLOC(MODL->pmtr[ipmtr].dot,   double, nrow*ncol);

        if (type == OCSM_DESPMTR || type == OCSM_CFGPMTR || type == OCSM_UNKNOWN) {
            MALLOC(MODL->pmtr[ipmtr].lbnd,  double, nrow*ncol);
            MALLOC(MODL->pmtr[ipmtr].ubnd,  double, nrow*ncol);
        }

        /* initialize to large values */
        for (i = 0; i < nrow*ncol; i++) {
            MODL->pmtr[ipmtr].value[i] = -HUGEQ;
            MODL->pmtr[ipmtr].dot[  i] = 0;

            if (type == OCSM_DESPMTR || type == OCSM_CFGPMTR || type == OCSM_UNKNOWN) {
                MODL->pmtr[ipmtr].lbnd[i] = -HUGEQ;
                MODL->pmtr[ipmtr].ubnd[i] = +HUGEQ;
            }
        }

    } else {
        FREE(MODL->pmtr[ipmtr].str);

        old_nrow = MODL->pmtr[ipmtr].nrow;
        old_ncol = MODL->pmtr[ipmtr].ncol;

        RALLOC(MODL->pmtr[ipmtr].value, double, nrow*ncol);
        RALLOC(MODL->pmtr[ipmtr].dot,   double, nrow*ncol);
        if (type == OCSM_DESPMTR || type == OCSM_CFGPMTR || type == OCSM_UNKNOWN) {
            RALLOC(MODL->pmtr[ipmtr].lbnd,  double, nrow*ncol);
            RALLOC(MODL->pmtr[ipmtr].ubnd,  double, nrow*ncol);
        }

        /* if not previously an array, initialize to large values */
        if (old_nrow == 0 || old_ncol == 0) {
            for (i = 0; i < nrow*ncol; i++) {
                MODL->pmtr[ipmtr].value[i] = -HUGEQ;
                MODL->pmtr[ipmtr].dot[  i] = 0;
                if (type == OCSM_DESPMTR || type == OCSM_CFGPMTR || type == OCSM_UNKNOWN) {
                    MODL->pmtr[ipmtr].lbnd[i] = -HUGEQ;
                    MODL->pmtr[ipmtr].ubnd[i] = +HUGEQ;
                }
            }

        /* propagate last value if new array is larger */
        } else {
            for (i = old_nrow*old_ncol; i < nrow*ncol; i++) {
                MODL->pmtr[ipmtr].value[i] = MODL->pmtr[ipmtr].value[i-1];
                MODL->pmtr[ipmtr].dot[  i] = MODL->pmtr[ipmtr].dot[  i-1];
                if (type == OCSM_DESPMTR || type == OCSM_CFGPMTR || type == OCSM_UNKNOWN) {
                    MODL->pmtr[ipmtr].lbnd[i] = MODL->pmtr[ipmtr].lbnd[i-1];
                    MODL->pmtr[ipmtr].ubnd[i] = MODL->pmtr[ipmtr].ubnd[i-1];
                }
            }
        }
    }

    if (type == OCSM_DESPMTR || type == OCSM_CFGPMTR || type == OCSM_CONPMTR) {
        MODL->pmtr[ipmtr].scope = 0;
    } else {
        MODL->pmtr[ipmtr].scope = MODL->scope[MODL->level];
    }

    MODL->pmtr[ipmtr].type  = type;
    MODL->pmtr[ipmtr].nrow  = nrow;
    MODL->pmtr[ipmtr].ncol  = ncol;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmDelPmtr - delete a Parameter                                   *
 *                                                                      *
 ************************************************************************
 */

int
ocsmDelPmtr(void   *modl,               /* (in)  pointer to MODL */
            int    ipmtr)               /* (in)  Parameter index (1:npmtr) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       jpmtr;

    ROUTINE(ocsmDelPmtr);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ipmtr is given */
    if (ipmtr < 1 || ipmtr > MODL->npmtr) {
        status = OCSM_ILLEGAL_PMTR_INDEX;
        goto cleanup;
    } else if (MODL->pmtr[ipmtr].scope != MODL->scope[MODL->level]) {
        status = OCSM_ILLEGAL_PMTR_INDEX;
        goto cleanup;
    }

    /* remove data associated with this Parameter */
    FREE(MODL->pmtr[ipmtr].name );
    FREE(MODL->pmtr[ipmtr].value);
    FREE(MODL->pmtr[ipmtr].dot  );
    FREE(MODL->pmtr[ipmtr].lbnd );
    FREE(MODL->pmtr[ipmtr].ubnd );
    FREE(MODL->pmtr[ipmtr].str  );

    /* move all Parameters up to write over deleted Parameter */
    for (jpmtr = ipmtr; jpmtr < MODL->npmtr; jpmtr++) {
        MODL->pmtr[jpmtr].name  = MODL->pmtr[jpmtr+1].name;
        MODL->pmtr[jpmtr].type  = MODL->pmtr[jpmtr+1].type;
        MODL->pmtr[jpmtr].scope = MODL->pmtr[jpmtr+1].scope;
        MODL->pmtr[jpmtr].mprop = MODL->pmtr[jpmtr+1].mprop;
        MODL->pmtr[jpmtr].nrow  = MODL->pmtr[jpmtr+1].nrow;
        MODL->pmtr[jpmtr].ncol  = MODL->pmtr[jpmtr+1].ncol;
        MODL->pmtr[jpmtr].value = MODL->pmtr[jpmtr+1].value;
        MODL->pmtr[jpmtr].dot   = MODL->pmtr[jpmtr+1].dot;
        MODL->pmtr[jpmtr].lbnd  = MODL->pmtr[jpmtr+1].lbnd;
        MODL->pmtr[jpmtr].ubnd  = MODL->pmtr[jpmtr+1].ubnd;
        MODL->pmtr[jpmtr].str   = MODL->pmtr[jpmtr+1].str;
    }

    /* null pointers associated with last Parameter (since they were
       copied down) */
    MODL->pmtr[MODL->npmtr].name  = NULL;
    MODL->pmtr[MODL->npmtr].value = NULL;
    MODL->pmtr[MODL->npmtr].dot   = NULL;
    MODL->pmtr[MODL->npmtr].lbnd  = NULL;
    MODL->pmtr[MODL->npmtr].ubnd  = NULL;
    MODL->pmtr[MODL->npmtr].str   = NULL;

    /* decrement the number of Parameters */
    (MODL->npmtr)--;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmFindPmtr - find (or create) a Parameter                        *
 *                                                                      *
 ************************************************************************
 */

int
ocsmFindPmtr(void   *modl,              /* (in)  pointer to MODL */
             char   name[],             /* (in)  Parameter name */
             int    type,               /* (in)  Parameter type    (or 0 to not create) */
             int    nrow,               /* (in)  number of rows */
             int    ncol,               /* (in)  number of columns */
             int    *ipmtr)             /* (out) Parameter index (1:npmtr) (or 0 if not found) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       jpmtr;
    char      newName[MAX_NAME_LEN];
    void      *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(ocsmFindPmtr);

    /* --------------------------------------------------------------- */

    *ipmtr = 0;

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* make the newName by prepending the prefix if name starts with
       a colon */
    if (name[0] == ':' && STRLEN(MODL->prefix) > 0) {
        if (STRLEN(name)+STRLEN(MODL->prefix) >= MAX_NAME_LEN) {
            status = OCSM_ILLEGAL_PMTR_NAME;
            goto cleanup;
        }

        snprintf(newName, MAX_NAME_LEN, "%s%s", MODL->prefix, name);
    } else {
        STRNCPY(newName, name, MAX_NAME_LEN);
    }

    /* if type is zero, just look for the name */
    if (type == 0) {
        for (jpmtr = 1; jpmtr <= MODL->npmtr; jpmtr++) {
            if (strcmp(MODL->pmtr[jpmtr].name, newName) == 0          &&
                MODL->pmtr[jpmtr].scope == MODL->scope[MODL->level]  ) {
                *ipmtr = jpmtr;
                break;
            }
        }
        goto cleanup;
    }

    /* check if name is already defined at this level */
    for (jpmtr = 1; jpmtr <= MODL->npmtr; jpmtr++) {
        if (strcmp(MODL->pmtr[jpmtr].name, newName) == 0                 &&
                   MODL->pmtr[jpmtr].scope == MODL->scope[MODL->level]  ) {

            if (type == MODL->pmtr[jpmtr].type) {
                if (nrow != MODL->pmtr[jpmtr].nrow ||
                    ncol != MODL->pmtr[jpmtr].ncol   ) {
                    if (nrow*ncol > (MODL->pmtr[jpmtr].nrow)*(MODL->pmtr[jpmtr].ncol)) {
                        RALLOC(MODL->pmtr[jpmtr].value, double, nrow*ncol);
                        RALLOC(MODL->pmtr[jpmtr].dot,   double, nrow*ncol);
                    }

                    MODL->pmtr[jpmtr].nrow = nrow;
                    MODL->pmtr[jpmtr].ncol = ncol;
                }

                *ipmtr = jpmtr;
                goto cleanup;
            } else {
                *ipmtr = jpmtr;
                status = OCSM_NAME_ALREADY_DEFINED;
                goto cleanup;
            }
        }
    }

    /* if we got here, create a new Parameter */
    status = ocsmNewPmtr(MODL, newName, type, nrow, ncol);
    CHECK_STATUS(ocsmNewPmtr);

    *ipmtr = MODL->npmtr;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmGetPmtr - get info about a Parameter                           *
 *                                                                      *
 ************************************************************************
 */

int
ocsmGetPmtr(void   *modl,               /* (in)  pointer to MODL */
            int    ipmtr,               /* (in)  Parameter index (1:npmtr) */
            int    *type,               /* (out) Parameter type */
            int    *nrow,               /* (out) number of rows */
            int    *ncol,               /* (out) number of columns */
            char   name[])              /* (out) Parameter name (at least MAX_NAME_LEN long) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    ROUTINE(ocsmGetPmtr);

    /* --------------------------------------------------------------- */

    /* default return values */
    name[0] = '\0';
    *type   = 0;
    *nrow   = 0;
    *ncol   = 0;

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* check that valid ipmtr is given */
    if (ipmtr < 1 || ipmtr > MODL->npmtr) {
        status = OCSM_ILLEGAL_PMTR_INDEX;
        goto cleanup;
    } else if (MODL->pmtr[ipmtr].scope != MODL->scope[MODL->level]) {
        status = OCSM_ILLEGAL_PMTR_INDEX;
        goto cleanup;
    }

    /* return the name, type, and size */
    STRNCPY(name, MODL->pmtr[ipmtr].name, MAX_NAME_LEN);
    *type  =      MODL->pmtr[ipmtr].type;
    *nrow  =      MODL->pmtr[ipmtr].nrow;
    *ncol  =      MODL->pmtr[ipmtr].ncol;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmPrintPmtrs - print despmtrs and outpmtrs to file               *
 *                                                                      *
 ************************************************************************
 */

int
ocsmPrintPmtrs(void   *modl,            /* (in)  pointer to MODL */
               char   filename[])       /* (in)  file to which output is appended (or "" for stdout) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       ipmtr, maxlen, irow, icol, index, count;
    char      typeChar;
    FILE      *fp=NULL;

    ROUTINE(ocsmPrintPmtrs);

    /* --------------------------------------------------------------- */

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    if (strlen(filename) == 0) {
        fp = stdout;
    } else {
        fp = fopen(filename, "a");
        if (fp == NULL) {
            status = OCSM_FILE_NOT_FOUND;
            goto cleanup;
        }
    }

    /* find maximum name length */
    maxlen = 0;
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (STRLEN(MODL->pmtr[ipmtr].name) > maxlen) {
            maxlen = STRLEN(MODL->pmtr[ipmtr].name);
        }
    }

    fprintf(fp, "    ipmtr\n");

    /* loop through all constant Parameters and print its name and values */
    count = 0;
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (MODL->pmtr[ipmtr].type  == OCSM_CONPMTR) {
            fprintf(fp, "    %5d [c]  %-*s            %11.5f",
                    ipmtr, maxlen, MODL->pmtr[ipmtr].name, MODL->pmtr[ipmtr].value[0]);
            for (index = 1; index < MODL->pmtr[ipmtr].ncol; index++) {
                fprintf(fp, "   %11.5f", MODL->pmtr[ipmtr].value[index]);
            }
            fprintf(fp, "\n");
            count++;
        }
    }

    /* loop through all DESPMTR and CFGPMTR Parameters and print its name, value, and defn */
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (MODL->pmtr[ipmtr].type == OCSM_DESPMTR ||
            MODL->pmtr[ipmtr].type == OCSM_CFGPMTR   ) {

            if (MODL->pmtr[ipmtr].type == OCSM_DESPMTR) {
                typeChar = 'd';
            } else {
                typeChar = 'f';
            }

            index = 0;
            for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                    fprintf(fp, "    %5d [%c]  %-*s  [%3d,%3d] %11.5f %11.5f    >>",
                            ipmtr, typeChar, maxlen, MODL->pmtr[ipmtr].name,
                            irow, icol, MODL->pmtr[ipmtr].value[index],
                                        MODL->pmtr[ipmtr].dot[  index]);

                    if (MODL->pmtr[ipmtr].lbnd[index] <= -0.99*HUGEQ) {
                        fprintf(fp, " unbounded  << ");
                    } else {
                        fprintf(fp, "%11.5f << ", MODL->pmtr[ipmtr].lbnd[ index]);
                    }

                    if (MODL->pmtr[ipmtr].ubnd[index] >= +0.99*HUGEQ) {
                        fprintf(fp, " unbounded \n");
                    } else {
                        fprintf(fp, "%11.5f\n",   MODL->pmtr[ipmtr].ubnd[ index]);
                    }

                    index++;
                }
            }
            count++;
        }
    }

    /* loop through all OUTPMTR Parameters and print its name, value, and defn */
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (MODL->pmtr[ipmtr].type == OCSM_OUTPMTR) {

            index = 0;
            for (irow = 1; irow <= MODL->pmtr[ipmtr].nrow; irow++) {
                for (icol = 1; icol <= MODL->pmtr[ipmtr].ncol; icol++) {
                    if        (fabs(MODL->pmtr[ipmtr].value[index]) > 0.99*HUGEQ &&
                               fabs(MODL->pmtr[ipmtr].dot[  index]) > 0.99*HUGEQ   ) {
                        fprintf(fp, "    %5d [o]  %-*s  [%3d,%3d]  undefined   undefined\n",
                                ipmtr, maxlen, MODL->pmtr[ipmtr].name,
                                irow, icol);
                    } else if (fabs(MODL->pmtr[ipmtr].dot[  index]) > 0.99*HUGEQ) {
                        fprintf(fp, "    %5d [o]  %-*s  [%3d,%3d] %11.5f  undefined\n",
                                ipmtr, maxlen, MODL->pmtr[ipmtr].name,
                                irow, icol, MODL->pmtr[ipmtr].value[index]);
                    } else if (fabs(MODL->pmtr[ipmtr].value[index]) > 0.99*HUGEQ) {
                        fprintf(fp, "    %5d [o]  %-*s  [%3d,%3d]  undefined  %11.5f\n",
                                ipmtr, maxlen, MODL->pmtr[ipmtr].name,
                                irow, icol, MODL->pmtr[ipmtr].dot[  index]);
                    } else {
                        fprintf(fp, "    %5d [o]  %-*s  [%3d,%3d] %11.5f %11.5f\n",
                                ipmtr, maxlen, MODL->pmtr[ipmtr].name,
                                irow, icol, MODL->pmtr[ipmtr].value[index],
                                            MODL->pmtr[ipmtr].dot[  index]);
                    }
                    index++;
                }
            }
            count++;
        }
    }

    /* return message if no Paramaters */
    if (count <= 0) {
        fprintf(fp, "    --none--\n");
        goto cleanup;
    }

cleanup:
    if (fp != NULL && strlen(filename) > 0) fclose(fp);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   ocsmTracePmtrs - trace definition and use of all Parameters        *
 *                                                                      *
 ************************************************************************
 */

int
ocsmTracePmtrs(void   *modl,            /* (in)  pointer to MODL */
               char   pattern[],        /* (in)  pattern of Parameters to match */
               char   *info[])          /* (out) info about the Parameters (freeable) */
{
    int       status = SUCCESS;         /* (out) return status */

    modl_T    *MODL = (modl_T*)modl;

    int       mchar, ipmtr, ibrch, linenum, narg, iarg, ibeg, iend, irpn, used;
    char      line[MAX_LINE_LEN+1], *thisArg, *filelist=NULL, *templine=NULL, *arg1=NULL, *arg2=NULL;
    void      *temp;
    void      *realloc_temp=NULL;              /* used by RALLOC macro */
    FILE      *fp_csm=NULL;
    rpn_T     *rpn=NULL;                /* Rpn-code */

    ROUTINE(ocsmTracePmtrs);

    /* --------------------------------------------------------------- */

    /* initialize output buffer */
    mchar = 1000;
    MALLOC(*info, char, mchar);
    *info[0] = '\0';

    /* check magic number */
    if (MODL == NULL) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    } else if (MODL->magic != OCSM_MAGIC) {
        status = OCSM_NOT_MODL_STRUCTURE;
        goto cleanup;
    }

    /* get info needed for CONPMTRs, CFGPMTRs, and DESPMTRs */
    MALLOC(templine, char, MAX_LINE_LEN);
    MALLOC(arg1,     char, MAX_LINE_LEN);
    MALLOC(arg2,     char, MAX_LINE_LEN);

    status = ocsmGetFilelist(MODL, &filelist);
    CHECK_STATUS(ocsmGetFilelist);

    SPLINT_CHECK_FOR_NULL(filelist);

    snprintf(line, MAX_LINE_LEN, "Trace of top-level Parameters matching \"%s\":\n", pattern);
    if (strlen(*info)+strlen(line) >= mchar) {
        mchar += strlen(line) + 1000;
        RALLOC(*info, char, mchar);
    }
    strcat(*info, line);

    /* loop through all the Parameters */
    for (ipmtr = 1; ipmtr <= MODL->npmtr; ipmtr++) {
        if (matches(pattern, MODL->pmtr[ipmtr].name) == 0) continue;

        /* skip if an at-Parameter */
        if (MODL->pmtr[ipmtr].name[0] == '@') continue;

        snprintf(line, MAX_LINE_LEN, "    %-32s %s\n",
                 MODL->pmtr[ipmtr].name, ocsmGetText(MODL->pmtr[ipmtr].type));
        if (strlen(*info)+strlen(line) >= mchar) {
            mchar += strlen(line) + 1000;
            RALLOC(*info, char, mchar);
        }
        strcat(*info, line);

        /* if Parameter is a CONPMTR, CFGPMTR, or DESPMTR, search through
           all files to find out where is was defined */
        if (MODL->pmtr[ipmtr].type == OCSM_CONPMTR ||
            MODL->pmtr[ipmtr].type == OCSM_CFGPMTR ||
            MODL->pmtr[ipmtr].type == OCSM_DESPMTR   ) {

            ibeg = 0;
            for (iend = 1; iend < STRLEN(filelist); iend++) {
                if (filelist[iend] == '|') {
                    filelist[iend] = '\0';
                } else {
                    continue;
                }

                if (STRLEN(&filelist[ibeg]) > 0) {
                    fp_csm = fopen(&filelist[ibeg], "r");
                    if (fp_csm == NULL) {
                        status = OCSM_FILE_NOT_FOUND;
                        goto cleanup;
                    }

                    linenum = 0;
                    while (1) {
                        temp = fgets(templine, MAX_LINE_LEN, fp_csm);
                        if (temp == NULL) break;

                        linenum++;

                        narg = sscanf(templine, "%2047s\n", arg1);
                        if (narg > 0) {
                            if (strcmp(arg1, "CONPMTR") == 0 || strcmp(arg1, "conpmtr") == 0 ||
                                strcmp(arg1, "CFGPMTR") == 0 || strcmp(arg1, "cfgpmtr") == 0 ||
                                strcmp(arg1, "DESPMTR") == 0 || strcmp(arg1, "despmtr") == 0   ) {

                                narg = sscanf(templine, "%2047s %2047s\n", arg1, arg2);
                                if (narg > 1 && strcmp(arg2, MODL->pmtr[ipmtr].name) == 0) {
                                    snprintf(line, MAX_LINE_LEN, "        set  in [[%s:%d]]\n",
                                             &filelist[ibeg], linenum);
                                    if (strlen(*info)+strlen(line) >= mchar) {
                                        mchar += strlen(line) + 1000;
                                        RALLOC(*info, char, mchar);
                                    }
                                    strcat(*info, line);
                                }
                            } else if (strcmp(arg1, "END") == 0 || strcmp(arg1, "end") == 0) {
                                break;
                            }
                        }
                    }

                    fclose(fp_csm);
                    fp_csm = NULL;
                }

                ibeg = iend + 1;
                filelist[iend] = '|';
            }

        /* otherwise it is a LOCALVAR or OUTPMTR, so find the
           SET, PATBEG, or GETATTR at which the Parameter was mentioned */
        } else {
            for (ibrch = 1; ibrch <= MODL->nbrch; ibrch++) {