# Import pyCAPS module
import pyCAPS

# Import os module
import os
import sys
import shutil
import argparse

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'Astros Single Load Optimization Control Case PyTest Example',
                                 prog = 'astros_OptControl_PyTest',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = "." + os.sep, nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument('-noAnalysis', action='store_true', default = False, help = "Don't run analysis code")
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

#------------------------------------------------------------------
# ASTROS_ROOT should be the path containing ASTRO.D01 and ASTRO.IDX
#------------------------------------------------------------------
if args.noAnalysis == False:
    try:
       ASTROS_ROOT = os.environ["ASTROS_ROOT"]
       os.putenv("PATH", ASTROS_ROOT + os.pathsep + os.getenv("PATH"))
    except KeyError:
       print("Please set the environment variable ASTROS_ROOT")
       sys.exit(1)
#------------------------------------------------------------------

workDir= os.path.join(str(args.workDir[0]), "AstrosOptControlTest")

# Initialize capsProblem object
geometryScript = os.path.join("..","csmData","feaSimplePlate.csm")
capsProblem = pyCAPS.Problem(problemName=workDir,
                             capsFile=geometryScript, 
                             outLevel=args.outLevel)

# Create project name
projectName = "AstrosSingleLoadPlateOptControl"

# Load Astros aim
capsProblem.analysis.create( aim = "astrosAIM",
                             name = "astros",
                             autoExec = not args.noAnalysis )

# Set project name so a mesh file is generated
capsProblem.analysis["astros"].input.Proj_Name = projectName

# Set meshing parameters
capsProblem.analysis["astros"].input.Edge_Point_Max = 4

capsProblem.analysis["astros"].input.Quad_Mesh = True

capsProblem.analysis["astros"].input.Tess_Params = [.25,.01,15]

# Set analysis type
capsProblem.analysis["astros"].input.Analysis_Type = "Static"

# Set materials
madeupium    = {"materialType" : "isotropic",
                "youngModulus" : 72.0E9 ,
                "poissonRatio": 0.33,
                "density" : 2.8E3}

capsProblem.analysis["astros"].input.Material = {"Madeupium": madeupium}

# Set properties
shell  = {"propertyType" : "Shell",
          "membraneThickness" : 0.006,
          "material"        : "madeupium",
          "bendingInertiaRatio" : 1.0, # Default
          "shearMembraneRatio"  : 5.0/6.0} # Default

capsProblem.analysis["astros"].input.Property = {"plate": shell}

# Set constraints
constraint = {"groupName" : "plateEdge",
              "dofConstraint" : 123456}

capsProblem.analysis["astros"].input.Constraint = {"edgeConstraint": constraint}

# Set load
load = {"groupName" : "plate",
        "loadType" : "Pressure",
        "pressureForce" : 2.e6}

# Set loads
capsProblem.analysis["astros"].input.Load = {"appliedPressure": load}

# Set Optimization Control
optControl = {"fullyStressedDesign" : 5}

capsProblem.analysis["astros"].input.Optimization_Control = optControl

# Set analysis
# No analysis case information needs to be set for a single static load case
if args.noAnalysis:
    capsProblem.analysis["astros"].preAnalysis()
    exit()

# Run AIM analysis
capsProblem.analysis["astros"].runAnalysis()
