#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "egads.h"


//#define PRELINKWRITE
//#define POSTLINKWRITE
#define MAXRATIO  1.14           /* acceptable ratio of connection lengths */
#define MINRATIO  0.3333         /* smallest acceptable distance arcLen */
#define DOTCREASE 0.9            /* Face dots to consider a crease */

#define CROSS(a,b,c)      a[0] = (b[1]*c[2]) - (b[2]*c[1]);\
                          a[1] = (b[2]*c[0]) - (b[0]*c[2]);\
                          a[2] = (b[0]*c[1]) - (b[1]*c[0])
#define DOT(a,b)         (a[0]*b[0] + a[1]*b[1] + a[2]*b[2])


typedef struct {
  int iCntr;                     /* other side fan index (bias 0) or -1 */
  int iNode;                     /* the Node index */
  int nFan;                      /* number of fan links */
  int linkFans[4];               /* the link indices (0 bias) */
} Fan;

typedef struct Segment {
  ego            edge;           /* the Edge */
  int            nodes[2];       /* the 2 bounding Node Indices */
  int            sense;          /* the sense for an Edge */
  double         trange[2];      /* the range for an Edge */
  struct Segment *prev;          /* previous segment */
  struct Segment *next;          /* next segment */
} Segment;

typedef struct {
  Segment *segment;              /* current segment */
  double  xyz[3];                /* 3D position */
  double  dist;                  /* distance from Node */
  double  t;                     /* t on Edge */
  int     iEdge;                 /* Edge index or Node index for N2N */
} Intersect;

typedef struct {
  ego           edge;            /* the Edge split */
  double        t;               /* the value along the Edge for the Node */
  double        xyz[3];          /* 3D position */
  int           side;            /* 1 or 2 for contour side */
} newNode;

typedef struct {
  int     nLink;                 /* number of links */
  int     *links;                /* index node1 & node2 (2*nLink in len) */
  Segment **segments;            /* segment node1 & node2 (2*nLink in len) */
  int     nNewNode;              /* number of newNodes */
  newNode *newNodes;             /* the new Nodes (- indices) */
} Links;


  extern /*@kept@*/ /*@null@*/ ego EG_context( const ego object );
  extern int  EG_outLevel( const ego object );
  extern int  EG_sameThread( const ego object );
  extern int  EG_invEvaLimits( const ego geom, const double *limits,
                               const double *xyz, double *param,
                               double *result );


static int
EG_sameCurveSample(const ego edge1, const ego edge2)
{
  int    i, iper, stat;
  double d, tol, t, trange1[2], trange2[2], xyz[3], result[9];
  
  stat = EG_getRange(edge1, trange1, &iper);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS ERROR: EG_getRange 1 = %d (EG_sameCurveSample)!\n", stat);
    return stat;
  }
  stat = EG_getRange(edge2, trange2, &iper);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS ERROR: EG_getRange 2 = %d (EG_sameCurveSample)!\n", stat);
    return stat;
  }
  stat = EG_tolerance(edge1, &tol);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS ERROR: EG_tolerance = %d (EG_sameCurveSample)!\n", stat);
    return stat;
  }
  
  for (i = 1; i < 10; i++) {
    t    = trange1[0] + i*(trange1[1] - trange1[0])/10.0;
    stat = EG_evaluate(edge1, &t, result);
    if (stat != EGADS_SUCCESS) return EGADS_OUTSIDE;
    stat = EG_invEvaluate(edge2, result, &t, xyz);
    if (stat != EGADS_SUCCESS) return EGADS_OUTSIDE;
    if ((t < trange2[0]) || (t > trange2[1])) return EGADS_OUTSIDE;
    d = sqrt((result[0]-xyz[0])*(result[0]-xyz[0]) +
             (result[1]-xyz[1])*(result[1]-xyz[1]) +
             (result[2]-xyz[2])*(result[2]-xyz[2]));
    if (d > tol) return EGADS_OUTSIDE;
  }
  
  return EGADS_SUCCESS;
}


static void
EG_patchSplitterAttr(const char *attr, int nedge, const ego *edges,
                     const ego node, ego body)
{
  int          i, j, k, n, stat, aType, aLen, npts = 1;
  double       d, tol, xyz[3], xyzn[3];
  ego          *objs;
  const int    *ints;
  const double *reals, *xyzs;
  const char   *str;
  
  /* reassign node(s) */
  stat = EG_attributeRet(node, ".Nodes", &aType, &aLen, &ints, &xyzs, &str);
  if ((stat == EGADS_SUCCESS) && (aType == ATTRREAL)) {
    npts = aLen/3;
    if (npts < 1) npts = 1;
  }
  if (npts == 1) {
    stat = EG_evaluate(node, NULL, xyzn);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Error: EG_evaluate = %d (EG_patchSplitterAttr)!\n", stat);
      return;
    }
    xyzs = &xyzn[0];
  }
  stat = EG_getBodyTopos(body, NULL, NODE, &n, &objs);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_getBodyTopos Node = %d (EG_patchSplitterAttr)!\n",
           stat);
    return;
  }
  
  for (k = 0; k < npts; k++) {
    for (i = 0; i < n; i++) {
      stat = EG_attributeRet(objs[i], attr, &aType, &aLen, &ints, &reals, &str);
      if (stat == EGADS_SUCCESS) continue;
      stat = EG_tolerance(objs[i], &tol);
      if (stat != EGADS_SUCCESS) {
        printf(" EGADS Warning: tolerance %d N = %d (EG_patchSplitterAttr)!\n",
               i+1, stat);
        continue;
      }
      stat = EG_evaluate(objs[i], NULL, xyz);
      if (stat != EGADS_SUCCESS) {
        printf(" EGADS Warning: evaluate %d N = %d (EG_patchSplitterAttr)!\n",
               i+1, stat);
        continue;
      }
      d = sqrt((xyzs[3*k  ]-xyz[0])*(xyzs[3*k  ]-xyz[0]) +
               (xyzs[3*k+1]-xyz[1])*(xyzs[3*k+1]-xyz[1]) +
               (xyzs[3*k+2]-xyz[2])*(xyzs[3*k+2]-xyz[2]));
      if (d < tol) {
        j    = -(k+1);
        stat = EG_attributeAdd(objs[i], attr, ATTRINT, 1, &j, NULL, NULL);
        if (stat != EGADS_SUCCESS)
          printf(" EGADS Warning: getBodyTopos %d N = %d (patchSplitterAttr)!\n",
                 i+1, stat);
        break;
      }
    }
    if (i != n) continue;
    printf(" EGADS Warning: No matching Node found for %d (patchSplitterAttr)!\n",
           k+1);
    EG_free(objs);
    return;
  }
  EG_free(objs);
  
  /* reassign edges */
  stat = EG_getBodyTopos(body, NULL, EDGE, &n, &objs);
  if (stat != EGADS_SUCCESS) {
    printf("EGADS Error: EG_getBodyTopos Node = %d (EG_patchSplitterAttr)!\n",
           stat);
    return;
  }
  
  for (i = 0; i < n; i++) {
    stat = EG_attributeRet(objs[i], attr, &aType, &aLen, &ints, &reals, &str);
    if (stat == EGADS_SUCCESS) continue;
    for (j = 0; j < nedge; j++) {
      stat = EG_sameCurveSample(objs[i], edges[j]);
      if (stat == EGADS_SUCCESS) {
        stat = EG_attributeDup(edges[j], objs[i]);
        if (stat != EGADS_SUCCESS)
          printf(" EGADS Warning: EG_attributeDup %d = %d (patchSplitterAttr)!\n",
                 i+1, stat);
        break;
      }
    }
    if (j != nedge) continue;
    printf(" EGADS Warning: No matching Edge found for %d (patchSplitterAttr)!\n",
           i+1);
    EG_free(objs);
    return;
  }
  
  EG_free(objs);
}


static int
EG_senseEdge(ego body, const ego face, const ego edge)
{
  int    i, j, stat, cnt, sense, oclass, mtype, nLoop, nEdge, outLevel, lor;
  int    *sensel, *senses;
  double uvbox[4];
  ego    geom, *loops, *edges;
  
  outLevel = EG_outLevel(body);
  
  stat = EG_getTopology(face, &geom, &oclass, &mtype, uvbox, &nLoop, &loops,
                        &sensel);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_getTopology Face %d = %d (EG_senseEdge)!\n",
             EG_indexBodyTopo(body, face), stat);
    return stat;
  }
  for (i = 0; i < nLoop; i++) {
    lor = 1;
    if (abs(sensel[i]) == 2) lor = -1;
    stat = EG_getTopology(loops[i], &geom, &oclass, &mtype, NULL,
                          &nEdge, &edges, &senses);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_getTopology Loop %d/%d = %d (EG_senseEdge)!\n",
               EG_indexBodyTopo(body, face), i+1, stat);
      return stat;
    }
    sense = 0;
    for (cnt = j = 0; j < nEdge; j++) {
      if (edges[j] != edge) continue;
      sense = senses[j];
      cnt++;
    }
    if (cnt == 2) return 0;
    if (cnt == 1) return lor*sense;
  }
  
  return EGADS_EXISTS;
}


static void
EG_freeSegments(int ncontour, Segment **segs)
{
  int     i;
  Segment *start, *seg, *next;
  
  for (i = 0; i < ncontour; i++) {
    seg = start = segs[i];
    if (seg == NULL) continue;
    do {
      next = seg->next;
      EG_free(seg);
      seg = next;
    } while (seg != start);
  }
}


static int
EG_getSegmentTs(Segment *seg, Links *links, int *nts, double **ts)
{
  int    i, j, k, n;
  double t, *tx;
  
  *nts = 0;
  *ts  = NULL;
  for (n = i = 0; i < links->nNewNode; i++)
    if (seg->edge == links->newNodes[i].edge) n++;
  
  tx = (double *) EG_alloc((n+2)*sizeof(double));
  if (tx == NULL) return EGADS_MALLOC;
  
  tx[0] = seg->trange[0];
  tx[1] = seg->trange[1];
  n     = 2;
  for (i = 0; i < links->nNewNode; i++) {
    if (seg->edge != links->newNodes[i].edge) continue;
    t = links->newNodes[i].t;
    for (j = 0; j < n-1; j++)
      if ((tx[j] <= t) && (tx[j+1] >= t)) {
        for (k = n-1; k > j; k--) tx[k+1] = tx[k];
        tx[j+1] = t;
        break;
      }
    n++;
  }
  
  *nts = n;
  *ts  = tx;
  return EGADS_SUCCESS;
}


#ifdef DEBUG
static void
EG_printSegList(ego body, Segment *start)
{
  ego     edge;
  Segment *seg;
  
  seg = start;
  if (seg == NULL) return;

  do {
    edge = seg->edge;
    printf("   Edge %3d: Sense = %2d  Nodes = %3d %3d  tRange = %10.6lf %10.6lf\n",
           EG_indexBodyTopo(body, edge), seg->sense,
           seg->nodes[0], seg->nodes[1], seg->trange[0], seg->trange[1]);
    seg = seg->next;
  } while (seg != start);
  
}
#endif


static int
EG_fillSegList(int n, Segment *temp, Segment **start)
{
  int     i, j, stat, node, first, obj, *order, *mark;
  Segment **list = NULL;
  
  if (n <= 0) return EGADS_TOPOCNT;
  
  order = (int *) EG_alloc(2*n*sizeof(int));
  if (order == NULL) return EGADS_MALLOC;
  for (i = 0; i < 2*n; i++) order[i] = -1;
  mark  = &order[n];
  list  = (Segment **) EG_alloc(n*sizeof(Segment));
  if (list == NULL) {
    stat = EGADS_MALLOC;
    goto done;
  }
  for (i = 0; i < n; i++) list[i] = NULL;
  
  /* order the segments */
  order[0] = 0;
  mark[0]  = 0;
  if (temp[0].sense == 1) {
    node  = temp[0].nodes[1];
    first = temp[0].nodes[0];
  } else {
    node  = temp[0].nodes[0];
    first = temp[0].nodes[1];
  }
  for (i = 1; i < n; i++) {
    for (j = 1; j < n; j++) {
      if (mark[j] == 0) continue;
      if (temp[j].sense == 1) {
        obj = temp[j].nodes[0];
      } else {
        obj = temp[j].nodes[1];
      }
      if (obj == node) {
        order[i] = j;
        mark[j] = 0;
        if (temp[j].sense == 1) {
          node = temp[j].nodes[1];
        } else {
          node = temp[j].nodes[0];
        }
        break;
      }
    }
  }
  if (node != first) {
    stat = EGADS_NOTFOUND;
    goto done;
  }

  /* check for multiple contours where there should be one */
  for (j = i = 0; i < n; i++)
    if (order[i] == -1) j++;
  if (j != 0) {
#ifdef DEBUG
    printf(" order =");
    for (i = 0; i < n; i++) printf(" %d", order[i]);
    printf("\n");
#endif
    stat = EGADS_OCSEGFLT;
    goto done;
  }
  
  /* make the threaded list */
  stat = EGADS_MALLOC;
  for (i = 0; i < n; i++) {
    list[i] = EG_alloc(sizeof(Segment));
    if (list[i] == NULL) goto done;
  }
  for (i = 0; i < n; i++) {
    j = order[i];
    list[i]->edge      = temp[j].edge;
    list[i]->nodes[0]  = temp[j].nodes[0];
    list[i]->nodes[1]  = temp[j].nodes[1];
    list[i]->sense     = temp[j].sense;
    list[i]->trange[0] = temp[j].trange[0];
    list[i]->trange[1] = temp[j].trange[1];
    if (i == 0) {
      list[i]->prev    = list[n-1];
    } else {
      list[i]->prev    = list[i-1];
    }
    if (i == n-1) {
      list[i]->next    = list[0];
    } else {
      list[i]->next    = list[i+1];
    }
  }
  
  *start = list[0];
  stat   = EGADS_SUCCESS;
  
done:
  if (list != NULL) {
    if (stat != EGADS_SUCCESS)
      for (i = 0; i < n; i++)
        if (list[i] != NULL) EG_free(list[i]);
    EG_free(list);
  }
  EG_free(order);
  
  return stat;
}


static int
EG_addNewNode(ego body, int iEdge, double t, int side, int *nnnode,
              newNode **nnodes, int *index)
{
  int     i, stat, outLevel, nobj;
  double  d, etol, tol, xyz[3], result[9];
  ego     edge, *objs;
  newNode *newnodes;
  
  *index   = 0;
  outLevel = EG_outLevel(body);

  stat = EG_objectBodyTopo(body, EDGE, iEdge, &edge);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_objectBodyTopo = %d (EG_addNewNode)!\n", stat);
    return stat;
  }
  stat = EG_evaluate(edge, &t, result);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_evaluate = %d (EG_addNewNode)!\n", stat);
    return stat;
  }
  stat = EG_tolerance(edge, &etol);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_tolerance Edge = %d (EG_addNewNode)!\n", stat);
    return stat;
  }
  stat = EG_getBodyTopos(body, NULL, NODE, &nobj, &objs);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_getBodyTopos = %d (EG_addNewNode)!\n", stat);
    return stat;
  }
  
  /* do we hit an existing Node? */
  for (i = 0; i < nobj; i++) {
    stat = EG_tolerance(objs[i], &tol);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_tolerance Node %d = %d (EG_addNewNode)!\n",
               i+1, stat);
      EG_free(objs);
      return stat;
    }
    stat = EG_evaluate(objs[i], NULL, xyz);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_evaluate = %d (EG_addNewNode)!\n", stat);
      EG_free(objs);
      return stat;
    }
    d = sqrt((xyz[0]-result[0])*(xyz[0]-result[0]) +
             (xyz[1]-result[1])*(xyz[1]-result[1]) +
             (xyz[2]-result[2])*(xyz[2]-result[2]));
    if (d <= tol) {
      if (outLevel > 1)
        printf(" EGADS Info: new Node hit Node %d (EG_addNewNode)!\n", i+1);
      *index = i+1;
      EG_free(objs);
      return EGADS_SUCCESS;
    }
  }
  EG_free(objs);
  
  /* do we hit a stored new Node? */
  newnodes = *nnodes;
  for (i = 0; i < *nnnode; i++) {
    d = sqrt((newnodes[i].xyz[0]-result[0])*(newnodes[i].xyz[0]-result[0]) +
             (newnodes[i].xyz[1]-result[1])*(newnodes[i].xyz[1]-result[1]) +
             (newnodes[i].xyz[2]-result[2])*(newnodes[i].xyz[2]-result[2]));
    if (d <= etol) {
      if (outLevel > 0)
        printf(" EGADS Info: new Node hit new Node %d (EG_addNewNode)!\n",
               i+1);
      *index = -(i+1);
      return EGADS_OUTSIDE;
    }
  }
  
  /* no -- store away info for the new Node */
  if (*nnnode == 0) {
    newnodes = (newNode *) EG_alloc(sizeof(newNode));
    if (newnodes == NULL) {
      printf(" EGADS Error: Malloc of newNode (EG_addNewNode)!\n");
      return EGADS_MALLOC;
    }
  } else {
    newnodes = (newNode *) EG_reall(newnodes, (*nnnode+1)*sizeof(newNode));
    if (newnodes == NULL) {
      printf(" EGADS Error: Malloc of newNode (EG_addNewNode)!\n");
      return EGADS_MALLOC;
    }
  }
  newnodes[*nnnode].edge   = edge;
  newnodes[*nnnode].t      = t;
  newnodes[*nnnode].xyz[0] = result[0];
  newnodes[*nnnode].xyz[1] = result[1];
  newnodes[*nnnode].xyz[2] = result[2];
  newnodes[*nnnode].side   = side;
  
  *index  = -(*nnnode+1);
  *nnodes = newnodes;
  *nnnode = *nnnode+1;
  return EGADS_SUCCESS;
}


static int
EG_addLink(Links *links, int side, int iNode1, Segment *seg1,
                                   int iNode2, Segment *seg2)
{
  int i, in1, in2, *tmp;
  Segment *sg1, *sg2, **tmpsg;
  
  /* first slot always has the segments associated with Attribute 1 */
  if (side == 1) {
    in1 = iNode1;
    in2 = iNode2;
    sg1 = seg1;
    sg2 = seg2;
  } else {
    in2 = iNode1;
    in1 = iNode2;
    sg2 = seg1;
    sg1 = seg2;
  }
  
  for (i = 0; i < links->nLink; i++) {
    if ((links->links[2*i  ] == in1) && (links->links[2*i+1] == in2)) {
#ifdef DEBUG
      printf(" EGADS Info: Link found again %d %d\n", in1, in2);
#endif
      return EGADS_OUTSIDE;
    }
  }
  if (links->nLink == 0) {
    links->links = (int *) EG_alloc(2*sizeof(int));
    if (links->links == NULL) return EGADS_MALLOC;
    links->segments = (Segment **) EG_alloc(2*sizeof(Segment *));
    if (links->segments == NULL) return EGADS_MALLOC;
  } else {
    tmp = (int *) EG_reall(links->links, 2*(links->nLink+1)*sizeof(int));
    if (tmp == NULL) return EGADS_MALLOC;
    links->links = tmp;
    tmpsg = (Segment **) EG_reall(links->segments,
                                  2*(links->nLink+1)*sizeof(Segment *));
    if (tmpsg == NULL) return EGADS_MALLOC;
    links->segments = tmpsg;
  }
  links->links[2*links->nLink  ]    = in1;
  links->links[2*links->nLink+1]    = in2;
  links->segments[2*links->nLink  ] = sg1;
  links->segments[2*links->nLink+1] = sg2;
  links->nLink++;
  
  return EGADS_SUCCESS;
}


static int
EG_dotFaceSrc(ego body, const char *aNsrc, ego srcNode,
                        const char *aNdst, double *xyz)
{
  int          i, j, k, stat, nface, cnt, concave, aType, aLen, ne1, ne2;
  double       d, t, angle, dot, uv[2], dir[3], norm[3], nrm[3], result[18];
  double       coord[3], srcXYZ[3], *deru, *derv;
  ego          *faces, face, edge, *edge1 = NULL, *edge2 = NULL;
  const int    *ints;
  const double *reals;
  const char   *str;
  
  deru = &result[3];
  derv = &result[6];
  
  /* get the coordinates */
  stat = EG_evaluate(srcNode, NULL, srcXYZ);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_evaluate = %d (EG_dotFaceSrc)!\n", stat);
    return stat;
  }

  /* get the direction */
  dir[0] = xyz[0] - srcXYZ[0];
  dir[1] = xyz[1] - srcXYZ[1];
  dir[2] = xyz[2] - srcXYZ[2];
  d = sqrt(dir[0]*dir[0] + dir[1]*dir[1] + dir[2]*dir[2]);
  if (d == 0.0) {
    printf(" EGADS Error: Zero vector (EG_dotFaceSrc)!\n");
    return EGADS_DEGEN;
  }
  dir[0] /= d;
  dir[1] /= d;
  dir[2] /= d;
  
  /* get the Faces touching the src Node */
  stat = EG_getBodyTopos(body, srcNode, FACE, &nface, &faces);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: getBodyTopos Node for Faces = %d (EG_dotFaceSrc)!\n",
           stat);
    return stat;
  }
  
  /* check for sharp crease in retained Faces */
  nrm[0] = nrm[1] = nrm[2] = 0.0;
  face   = NULL;
  for (concave = cnt = i = 0; i < nface; i++) {
    stat = EG_attributeRet(faces[i], aNsrc, &aType, &aLen, &ints, &reals, &str);
    if (stat == EGADS_SUCCESS) continue;
    stat = EG_attributeRet(faces[i], aNdst, &aType, &aLen, &ints, &reals, &str);
    if (stat == EGADS_SUCCESS) continue;
    stat = EG_invEvaluate(faces[i], srcXYZ, uv, result);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Warning: Face invEvaluation0 = %d (EG_dotFaceSrc)!\n",
             stat);
      continue;
    }
    stat = EG_evaluate(faces[i], uv, result);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Warning: Face evaluation0 = %d (EG_dotFaceSrc)!\n", stat);
      continue;
    }
    d = DOT(deru, deru);
    if (d != 0.0) {
      d        = 1.0/sqrt(d);
      deru[0] *= d;
      deru[1] *= d;
      deru[2] *= d;
    }
    d = DOT(derv, derv);
    if (d != 0.0) {
      d        = 1.0/sqrt(d);
      derv[0] *= d;
      derv[1] *= d;
      derv[2] *= d;
    }
    CROSS(norm, deru, derv);
    if (faces[i]->mtype == SREVERSE) {
      norm[0] = -norm[0];
      norm[1] = -norm[1];
      norm[2] = -norm[2];
    }
    cnt++;
    if (cnt == 1) {
      face   = faces[i];
      nrm[0] = norm[0];
      nrm[1] = norm[1];
      nrm[2] = norm[2];
    } else {
      if (cnt == 2) {
        stat = EG_getBodyTopos(body, face, EDGE, &ne1, &edge1);
        if ((stat != EGADS_SUCCESS) || (edge1 == NULL)) {
          printf(" EGADS Warning: Face getBodyTopos1 = %d (EG_dotFaceSrc)!\n",
                 stat);
        } else {
          stat = EG_getBodyTopos(body, faces[i], EDGE, &ne2, &edge2);
          if ((stat != EGADS_SUCCESS) || (edge2 == NULL)) {
            printf(" EGADS Warning: Face getBodyTopos2 = %d (EG_dotFaceSrc)!\n",
                   stat);
            EG_free(edge1);
          } else {
            /* find shared Edge */
            edge = NULL;
            for (k = 0; k < ne2; k++) {
              for (j = 0; j < ne1; j++) {
                if (edge1[j] != edge2[k]) continue;
                edge = edge1[j];
                break;
              }
              if (edge != NULL) break;
            }
            EG_free(edge2);
            EG_free(edge1);
            if (edge != NULL) {
              stat = EG_invEvaluate(edge, srcXYZ, &t, coord);
              if (stat != EGADS_SUCCESS) {
                printf(" EGADS Warning: Edge invEval = %d (EG_dotFaceSrc)!\n",
                       stat);
              } else {
                stat = EG_getWindingAngle(edge, t, &angle);
                if (stat != EGADS_SUCCESS) {
                  printf(" EGADS Warning: WindingAngle = %d (EG_dotFaceSrc)!\n",
                         stat);
                } else {
                  if (angle < 120.0) {
                    concave = 1;
/*                  printf("  Winding Angle for Edge %d @ %lf = %lf\n",
                           EG_indexBodyTopo(body, edge), t, angle);   */
                  }
                }
              }
            } else {
              printf(" EGADS Warning: Edge Not Found (EG_dotFaceSrc)!\n");
            }
          }
        }
      } else {
        concave = 0;
      }
      if ((concave == 0) && (DOT(norm, nrm) < DOTCREASE)) {
        EG_free(faces);
        return EGADS_SUCCESS;
      }
    }
  }
  
  /* look at the dot product for each retained Face */
  for (i = 0; i < nface; i++) {
    stat = EG_attributeRet(faces[i], aNsrc, &aType, &aLen, &ints, &reals, &str);
    if (stat == EGADS_SUCCESS) continue;
    stat = EG_attributeRet(faces[i], aNdst, &aType, &aLen, &ints, &reals, &str);
    if (stat == EGADS_SUCCESS) continue;
    stat = EG_invEvaluate(faces[i], srcXYZ, uv, result);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Warning: Face invEvaluation = %d (EG_dotFaceSrc)!\n",
             stat);
      continue;
    }
    stat = EG_evaluate(faces[i], uv, result);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Warning: Face evaluation = %d (EG_dotFaceSrc)!\n", stat);
      continue;
    }
    d = DOT(deru, deru);
    if (d != 0.0) {
      d        = 1.0/sqrt(d);
      deru[0] *= d;
      deru[1] *= d;
      deru[2] *= d;
    }
    d = DOT(derv, derv);
    if (d != 0.0) {
      d        = 1.0/sqrt(d);
      derv[0] *= d;
      derv[1] *= d;
      derv[2] *= d;
    }
    CROSS(norm, deru, derv);
    if (faces[i]->mtype == SREVERSE) {
      norm[0] = -norm[0];
      norm[1] = -norm[1];
      norm[2] = -norm[2];
    }
    dot = DOT(norm, dir);
    if (dot > -0.05) continue;
    EG_free(faces);
    return EGADS_OUTSIDE;
  }
  EG_free(faces);
  
  return EGADS_SUCCESS;
}


static int
EG_dotFace(ego body, const char *aNsrc, ego srcNode,
                     const char *aNdst, Intersect *nodeIntersect)
{
  int          i, stat, nface, cnt, aType, aLen;
  double       d, dot, uv[2], dir[3], norm[3], nrm[3], srcXYZ[3], result[18];
  double       *deru, *derv;
  ego          node, *faces;
  const int    *ints;
  const double *reals;
  const char   *str;
#ifdef DEBUG
  int          npositive = 0;
#endif
  
  deru = &result[3];
  derv = &result[6];
  
  /* get the coordinates */
  stat = EG_evaluate(srcNode, NULL, srcXYZ);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_evaluate = %d (EG_dotFace)!\n", stat);
    return stat;
  }

  /* get the direction */
  dir[0] = nodeIntersect->xyz[0] - srcXYZ[0];
  dir[1] = nodeIntersect->xyz[1] - srcXYZ[1];
  dir[2] = nodeIntersect->xyz[2] - srcXYZ[2];
  d = sqrt(dir[0]*dir[0] + dir[1]*dir[1] + dir[2]*dir[2]);
  if (d == 0.0) {
    printf(" EGADS Error: Zero vector (EG_dotFace)!\n");
    return EGADS_DEGEN;
  }
  dir[0] /= d;
  dir[1] /= d;
  dir[2] /= d;
  
  /* get the Faces touching our target Node */
  stat = EG_objectBodyTopo(body, NODE, nodeIntersect->iEdge, &node);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: objectBodyTopo for Node %d = %d (EG_dotFace)!\n",
           nodeIntersect->iEdge, stat);
    return stat;
  }
  stat = EG_getBodyTopos(body, node, FACE, &nface, &faces);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: getBodyTopos Node %d for Faces = %d (EG_dotFace)!\n",
           nodeIntersect->iEdge, stat);
    return stat;
  }
  
  /* check for sharp crease in retained Faces */
  nrm[0] = nrm[1] = nrm[2] = 0.0;
  for (cnt = i = 0; i < nface; i++) {
    stat = EG_attributeRet(faces[i], aNsrc, &aType, &aLen, &ints, &reals, &str);
    if (stat == EGADS_SUCCESS) continue;
    stat = EG_attributeRet(faces[i], aNdst, &aType, &aLen, &ints, &reals, &str);
    if (stat == EGADS_SUCCESS) continue;
    stat = EG_invEvaluate(faces[i], nodeIntersect->xyz, uv, result);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Warning: Face invEvaluation0 = %d (EG_dotFace)!\n", stat);
      continue;
    }
    stat = EG_evaluate(faces[i], uv, result);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Warning: Face evaluation0 = %d (EG_dotFace)!\n", stat);
      continue;
    }
    d = DOT(deru, deru);
    if (d != 0.0) {
      d        = 1.0/sqrt(d);
      deru[0] *= d;
      deru[1] *= d;
      deru[2] *= d;
    }
    d = DOT(derv, derv);
    if (d != 0.0) {
      d        = 1.0/sqrt(d);
      derv[0] *= d;
      derv[1] *= d;
      derv[2] *= d;
    }
    CROSS(norm, deru, derv);
    if (faces[i]->mtype == SREVERSE) {
      norm[0] = -norm[0];
      norm[1] = -norm[1];
      norm[2] = -norm[2];
    }
    cnt++;
    if (cnt == 1) {
      nrm[0] = norm[0];
      nrm[1] = norm[1];
      nrm[2] = norm[2];
    } else {
      if (DOT(norm, nrm) < DOTCREASE) {
        EG_free(faces);
        return EGADS_SUCCESS;
      }
    }
  }
  
  /* look at the dot product for each retained Face */
  for (i = 0; i < nface; i++) {
    stat = EG_attributeRet(faces[i], aNsrc, &aType, &aLen, &ints, &reals, &str);
    if (stat == EGADS_SUCCESS) continue;
    stat = EG_attributeRet(faces[i], aNdst, &aType, &aLen, &ints, &reals, &str);
    if (stat == EGADS_SUCCESS) continue;
    stat = EG_invEvaluate(faces[i], nodeIntersect->xyz, uv, result);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Warning: Face invEvaluation = %d (EG_dotFace)!\n", stat);
      continue;
    }
    stat = EG_evaluate(faces[i], uv, result);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Warning: Face evaluation = %d (EG_dotFace)!\n", stat);
      continue;
    }
    d = DOT(deru, deru);
    if (d != 0.0) {
      d        = 1.0/sqrt(d);
      deru[0] *= d;
      deru[1] *= d;
      deru[2] *= d;
    }
    d = DOT(derv, derv);
    if (d != 0.0) {
      d        = 1.0/sqrt(d);
      derv[0] *= d;
      derv[1] *= d;
      derv[2] *= d;
    }
    CROSS(norm, deru, derv);
    if (faces[i]->mtype == SREVERSE) {
      norm[0] = -norm[0];
      norm[1] = -norm[1];
      norm[2] = -norm[2];
    }
    dot = DOT(norm, dir);
#ifdef DEBUG
    printf(" Face %2d: dot = %lf\n", EG_indexBodyTopo(body, faces[i]), dot);
    if (dot >= 0.0) npositive++;
#else
    if (dot <  0.0) continue;
    EG_free(faces);
    return EGADS_OUTSIDE;
#endif
  }
  EG_free(faces);
#ifdef DEBUG
  printf("\n");
  if (npositive != 0) return EGADS_OUTSIDE;
#endif
  
  return EGADS_SUCCESS;
}


static void
EG_initializeIntersect(Intersect *intersect)
{
  intersect->dist    = 1.e201;
  intersect->t       = 0.0;
  intersect->iEdge   = 0;
  intersect->segment = NULL;
  intersect->xyz[0]  = intersect->xyz[1] = intersect->xyz[2] = 0.0;
}


static void
EG_fillIntersect(Intersect *intersect, Segment *segment, double *xyz,
                 double dist, double t, int iEdge)
{
  if (dist  >= intersect->dist) return;
  if (iEdge <= EGADS_SUCCESS) {
    printf(" EGADS Internal: iEdge = %d (EG_fillIntersect)!\n", iEdge);
    return;
  }
  
  intersect->dist    = dist;
  intersect->t       = t;
  intersect->iEdge   = iEdge;
  intersect->segment = segment;
  intersect->xyz[0]  = xyz[0];
  intersect->xyz[1]  = xyz[1];
  intersect->xyz[2]  = xyz[2];
}


static void
EG_findIntersections(ego body, const char *aNsrc, Segment *src,
                               const char *aNdst, Segment *dst,
                     int side, int both, Links *links)
{
#ifdef DEBUG
  int       iEdge;
#endif
  int       i, stat, status, iNode, oNode, outLevel, index, N2N;
  double    d, t, ratio, trange[2], xyz[3], result[3], nodxyz[3];
  ego       srcNode, node;
  Segment   *segment, *seg;
  Intersect intersect, node2node;

  if ((src == NULL) || (dst == NULL)) return;
  outLevel = EG_outLevel(body);

  segment = src;
  do {
    /* get range and end-points for Edge */
                             iNode = segment->nodes[0];
    if (segment->sense == 1) iNode = segment->nodes[1];
    stat = EG_objectBodyTopo(body, NODE, iNode, &srcNode);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: objectBodyTopo iN = %d (EG_findIntersections)!\n",
               stat);
      return;
    }
    stat = EG_evaluate(srcNode, NULL, result);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_evaluate iN = %d (EG_findIntersections)!\n",
               stat);
      return;
    }
#ifdef DEBUG
    iEdge = EG_indexBodyTopo(body, segment->edge);
#endif

    /* initialize intersections */
    EG_initializeIntersect(&intersect);
    EG_initializeIntersect(&node2node);
    
    /* look at nearest intersections */
    seg = dst;
    do {
      for (i = 0; i < 4; i++) {
        trange[0] = seg->trange[0] +  i   *(seg->trange[1]-seg->trange[0])/4.0;
        trange[1] = seg->trange[0] + (i+1)*(seg->trange[1]-seg->trange[0])/4.0;
        stat = EG_invEvaLimits(seg->edge, trange, result, &t, xyz);
        if (stat != EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Warning: invEvaLimits = %d (EG_findIntersections)!\n",
                   stat);
        } else if (both == 0) {
          if (EG_dotFaceSrc(body, aNsrc, srcNode, aNdst, xyz) == EGADS_SUCCESS) {
            d = sqrt((xyz[0]-result[0])*(xyz[0]-result[0]) +
                     (xyz[1]-result[1])*(xyz[1]-result[1]) +
                     (xyz[2]-result[2])*(xyz[2]-result[2]));
            EG_fillIntersect(&intersect, seg, xyz, d, t,
                             EG_indexBodyTopo(body, seg->edge));
          }
        } else {
          /* both is non-zero for flend13 cases!  */
          d = sqrt((xyz[0]-result[0])*(xyz[0]-result[0]) +
                   (xyz[1]-result[1])*(xyz[1]-result[1]) +
                   (xyz[2]-result[2])*(xyz[2]-result[2]));
          EG_fillIntersect(&intersect, seg, xyz, d, t,
                           EG_indexBodyTopo(body, seg->edge));
        }
      }
      
      t     = seg->trange[0];
      oNode = seg->nodes[0];
      if (seg->sense == 1) {
        t     = seg->trange[1];
        oNode = seg->nodes[1];
      }
      stat = EG_objectBodyTopo(body, NODE, oNode, &node);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: objectBodyTopo oN = %d (EG_findIntersections)!\n",
                 stat);
        return;
      }
      stat = EG_evaluate(node, NULL, nodxyz);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: EG_evaluate oN = %d (EG_findIntersections)!\n",
                 stat);
        return;
      }
      if (EG_dotFaceSrc(body, aNsrc, srcNode, aNdst, nodxyz) == EGADS_SUCCESS) {
        d = sqrt((nodxyz[0]-result[0])*(nodxyz[0]-result[0]) +
                 (nodxyz[1]-result[1])*(nodxyz[1]-result[1]) +
                 (nodxyz[2]-result[2])*(nodxyz[2]-result[2]));
        EG_fillIntersect(&node2node, seg, nodxyz, d, t, oNode);
      }

      seg = seg->next;
    } while (seg != dst);

    /* insert a Node-to-Node link */
    N2N = 0;
    d   = 0.0;
    if ((node2node.dist < 1.e200) && (intersect.dist < 1.e200)) {
      d     = sqrt((node2node.xyz[0]-intersect.xyz[0])*
                   (node2node.xyz[0]-intersect.xyz[0]) +
                   (node2node.xyz[1]-intersect.xyz[1])*
                   (node2node.xyz[1]-intersect.xyz[1]) +
                   (node2node.xyz[2]-intersect.xyz[2])*
                   (node2node.xyz[2]-intersect.xyz[2]));
      ratio = node2node.dist/intersect.dist;
      if (ratio < MAXRATIO) {
        if (EG_dotFace(body, aNsrc, srcNode, aNdst,
                       &node2node) == EGADS_SUCCESS) {
          stat = EG_addLink(links, side, iNode, segment, node2node.iEdge, seg);
          if ((stat != EGADS_SUCCESS) && (stat != EGADS_OUTSIDE))
            if (outLevel > 0)
              printf(" EGADS Warning: addLinkN2N = %d (EG_findIntersections)!\n",
                     stat);
#ifdef DEBUG
          if (stat == EGADS_SUCCESS)
            printf(" Node %2d w/ Node %2d: Rat = %9.6lf -- dist = %lf  %2d\n",
                   iNode, node2node.iEdge, ratio, node2node.dist, links->nLink);
#endif
          N2N = 1;
        }
      }
    }

    /* if no N2N then insert a Link to a new Node */
    if ((intersect.dist < 1.e200) && (N2N == 0) &&
        (d > MINRATIO*intersect.dist)) {
      stat = EG_addNewNode(body, intersect.iEdge, intersect.t, 3-side,
                           &links->nNewNode, &links->newNodes, &index);
      if ((stat != EGADS_SUCCESS) && (stat != EGADS_OUTSIDE))
        if (outLevel > 0)
          printf(" EGADS Warning: addNewNode0 = %d (EG_findIntersections)!\n",
                 stat);
      /* don't add link if we hit an existing Node */
      if (index <= 0) {
        status = EG_addLink(links, side, iNode, segment, index, seg);
        if ((status != EGADS_SUCCESS) && (status != EGADS_OUTSIDE))
          if (outLevel > 0)
            printf(" EGADS Warning: addLink1 = %d (EG_findIntersections)!\n",
                   stat);
#ifdef DEBUG
        if (status == EGADS_SUCCESS)
          if (index > 0) {
            printf(" Node %2d w/ Node %2d: Rat = %9.6lf -- dist = %lf  %2d\n",
                   iNode, index, 0.0, 0.0, links->nLink);
          } else {
            printf(" Node %2d w/ Edge %2d: t   = %9.6lf -- dist = %lf  %2d",
                   iNode, intersect.iEdge, intersect.t, intersect.dist,
                   links->nLink);
            if (stat == EGADS_OUTSIDE) printf(" *");
            printf("\n");
          }
#endif
      }
    }
    
    segment = segment->next;
  } while (segment != src);

}


static void
EG_missedIntersections(ego body, const char *aNsrc, Segment *src,
                                 const char *aNdst, Segment *dst,
                       int side, Links *links)
{
  int       i, hit, stat, status, iNode, oNode, outLevel, index, N2N, nts;
  double    d, t, ratio, xyz[3], result[3], nodxyz[3], *ts;
  ego       node, srcNode;
  Segment   *segment, *seg;
  Intersect intersect, node2node;
  
  if ((src == NULL) || (dst == NULL)) return;
  outLevel = EG_outLevel(body);

  segment = src;
  do {
    /* get range and end-points for Edge */
                             iNode = segment->nodes[0];
    if (segment->sense == 1) iNode = segment->nodes[1];
    for (hit = i = 0; i < links->nLink; i++)
      if ((links->links[2*i  ] == iNode) || (links->links[2*i+1] == iNode)) {
        hit++;
        break;
      }
    if (hit != 0) {
      segment = segment->next;
      continue;
    }
    stat = EG_objectBodyTopo(body, NODE, iNode, &srcNode);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: objectBodyTopo = %d (EG_missedIntersections)!\n",
               stat);
      return;
    }
    stat = EG_evaluate(srcNode, NULL, result);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_evaluate = %d (EG_missedIntersections)!\n",
               stat);
      return;
    }
    
    /* initialize intersections */
    EG_initializeIntersect(&intersect);
    EG_initializeIntersect(&node2node);
    
    /* look for the nearest intersection */
    seg = dst;
    do {
      ts = NULL;
      stat = EG_getSegmentTs(seg, links, &nts, &ts);
      if ((stat == EGADS_SUCCESS) && (nts > 2) && (ts != NULL)) {
        for (i = 0; i < nts-1; i++) {
          stat = EG_invEvaLimits(seg->edge, &ts[i], result, &t, xyz);
          if (stat != EGADS_SUCCESS) {
            if (outLevel > 0)
              printf(" EGADS Warning: %d invEvaLims = %d (missedIntersections)!\n",
                     i+1, stat);
          } else {
            if (EG_dotFaceSrc(body, aNsrc, srcNode, aNdst, xyz) ==
                EGADS_SUCCESS) {
              d = sqrt((xyz[0]-result[0])*(xyz[0]-result[0]) +
                       (xyz[1]-result[1])*(xyz[1]-result[1]) +
                       (xyz[2]-result[2])*(xyz[2]-result[2]));
              EG_fillIntersect(&intersect, seg, xyz, d, t,
                               EG_indexBodyTopo(body, seg->edge));
            }
          }
        }
        EG_free(ts);
      } else {
        if (ts != NULL) EG_free(ts);
        stat = EG_invEvaLimits(seg->edge, seg->trange, result, &t, xyz);
        if (stat != EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Warning: invEvaLimits = %d (EG_missedIntersections)!\n",
                   stat);
        } else {
          d = sqrt((xyz[0]-result[0])*(xyz[0]-result[0]) +
                   (xyz[1]-result[1])*(xyz[1]-result[1]) +
                   (xyz[2]-result[2])*(xyz[2]-result[2]));
          EG_fillIntersect(&intersect, seg, xyz, d, t,
                           EG_indexBodyTopo(body, seg->edge));
        }
      }
      
      t     = seg->trange[0];
      oNode = seg->nodes[0];
      if (seg->sense == 1) {
        t     = seg->trange[1];
        oNode = seg->nodes[1];
      }
      stat = EG_objectBodyTopo(body, NODE, oNode, &node);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: objBodyTopo oN = %d (missedIntersections)!\n",
                 stat);
        return;
      }
      stat = EG_evaluate(node, NULL, nodxyz);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: EG_evaluate oN = %d (missedIntersections)!\n",
                 stat);
        return;
      }
      if (EG_dotFaceSrc(body, aNsrc, srcNode, aNdst, nodxyz) == EGADS_SUCCESS) {
        d = sqrt((nodxyz[0]-result[0])*(nodxyz[0]-result[0]) +
                 (nodxyz[1]-result[1])*(nodxyz[1]-result[1]) +
                 (nodxyz[2]-result[2])*(nodxyz[2]-result[2]));
        EG_fillIntersect(&node2node, seg, nodxyz, d, t, oNode);
      }

      seg = seg->next;
    } while (seg != dst);

    /* insert a Node-to-Node link (if exists) */
    N2N = 0;
    if (node2node.dist < 1.e200) {
      ratio = node2node.dist/intersect.dist;
      if (ratio < MAXRATIO) {
        stat = EG_addLink(links, side, iNode, segment, node2node.iEdge, seg);
        if ((stat != EGADS_SUCCESS) && (stat != EGADS_OUTSIDE))
          if (outLevel > 0)
            printf(" EGADS Warning: addLinkN2N = %d (missedIntersections)!\n",
                   stat);
#ifdef DEBUG
        if (stat == EGADS_SUCCESS)
          printf(" Node %2d w/ Node %2d: Rat = %9.6lf -- dist = %lf  %2d\n",
                 iNode, node2node.iEdge, ratio, node2node.dist, links->nLink);
#endif
        N2N = 1;
      }
    }

    /* if no N2N then insert a Link to a new Node */
    if ((N2N == 0) && (intersect.iEdge != 0)) {
      stat = EG_addNewNode(body, intersect.iEdge, intersect.t, 3-side,
                           &links->nNewNode, &links->newNodes, &index);
      if ((stat != EGADS_SUCCESS) && (stat != EGADS_OUTSIDE))
        if (outLevel > 0)
          printf(" EGADS Warning: addNewNode = %d (EG_missedIntersections)!\n",
                 stat);
      status = EG_addLink(links, side, iNode, segment, index, seg);
      if ((status != EGADS_SUCCESS) && (status != EGADS_OUTSIDE))
        if (outLevel > 0)
          printf(" EGADS Warning: addLink = %d (EG_missedIntersections)!\n",
                 stat);
#ifdef DEBUG
      if (status == EGADS_SUCCESS)
        if (index > 0) {
          printf(" Node %2d w/ Node %2d: Rat = %9.6lf -- dist = %lf  %2d\n",
                 iNode, index, 0.0, 0.0, links->nLink);
        } else {
          printf(" Node %2d w/ Edge %2d: t   = %9.6lf -- dist = %lf  %2d",
                 iNode, intersect.iEdge, intersect.t, intersect.dist,
                 links->nLink);
          if (stat == EGADS_OUTSIDE) printf(" *");
          printf("\n");
        }
#endif
    }
    
    segment = segment->next;
  } while (segment != src);

}


static void
EG_phase3Intersections(ego body, Segment *src, Segment *dst, int side,
                       Links *links)
{
  int       i, hit, stat, status, iNode, oNode, outLevel, index, N2N;
  double    d, t, ratio, xyz[3], result[3], nodxyz[3];
  ego       node, srcNode;
  Segment   *segment, *seg;
  Intersect intersect, node2node;
  
  if ((src == NULL) || (dst == NULL)) return;
  outLevel = EG_outLevel(body);

  segment = src;
  do {
    /* get range and end-points for Edge */
                             iNode = segment->nodes[0];
    if (segment->sense == 1) iNode = segment->nodes[1];
    for (hit = i = 0; i < links->nLink; i++)
      if ((links->links[2*i  ] == iNode) || (links->links[2*i+1] == iNode)) {
        hit++;
        break;
      }
    if (hit != 0) {
      segment = segment->next;
      continue;
    }
    stat = EG_objectBodyTopo(body, NODE, iNode, &srcNode);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: objectBodyTopo = %d (EG_phase3Intersections)!\n",
               stat);
      return;
    }
    stat = EG_evaluate(srcNode, NULL, result);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_evaluate = %d (EG_phase3Intersections)!\n",
               stat);
      return;
    }
    
    /* initialize intersections */
    EG_initializeIntersect(&intersect);
    EG_initializeIntersect(&node2node);
    
    /* look for the nearest intersection */
    seg = dst;
    do {
      stat = EG_invEvaLimits(seg->edge, seg->trange, result, &t, xyz);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Warning: invEvaLimits = %d (EG_phase3Intersections)!\n",
                 stat);
      } else {
        d = sqrt((xyz[0]-result[0])*(xyz[0]-result[0]) +
                 (xyz[1]-result[1])*(xyz[1]-result[1]) +
                 (xyz[2]-result[2])*(xyz[2]-result[2]));
        EG_fillIntersect(&intersect, seg, xyz, d, t,
                         EG_indexBodyTopo(body, seg->edge));
      }
      
      t     = seg->trange[0];
      oNode = seg->nodes[0];
      if (seg->sense == 1) {
        t     = seg->trange[1];
        oNode = seg->nodes[1];
      }
      stat = EG_objectBodyTopo(body, NODE, oNode, &node);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: objBodyTopo oN = %d (phase3Intersections)!\n",
                 stat);
        return;
      }
      stat = EG_evaluate(node, NULL, nodxyz);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: EG_evaluate oN = %d (phase3Intersections)!\n",
                 stat);
        return;
      }
      d = sqrt((nodxyz[0]-result[0])*(nodxyz[0]-result[0]) +
               (nodxyz[1]-result[1])*(nodxyz[1]-result[1]) +
               (nodxyz[2]-result[2])*(nodxyz[2]-result[2]));
      EG_fillIntersect(&node2node, seg, nodxyz, d, t, oNode);

      seg = seg->next;
    } while (seg != dst);

    /* insert a Node-to-Node link (if exists) */
    N2N = 0;
    if (node2node.dist < 1.e200) {
      ratio = node2node.dist/intersect.dist;
      if (ratio < MAXRATIO) {
        stat = EG_addLink(links, side, iNode, segment, node2node.iEdge, seg);
        if ((stat != EGADS_SUCCESS) && (stat != EGADS_OUTSIDE))
          if (outLevel > 0)
            printf(" EGADS Warning: addLinkN2N = %d (phase3Intersections)!\n",
                   stat);
#ifdef DEBUG
        if (stat == EGADS_SUCCESS)
          printf(" Node %2d w/ Node %2d: Rat = %9.6lf -- dist = %lf  %2d\n",
                 iNode, node2node.iEdge, ratio, node2node.dist, links->nLink);
#endif
        N2N = 1;
      }
    }

    /* if no N2N then insert a Link to a new Node */
    if ((N2N == 0) && (intersect.iEdge != 0)) {
      stat = EG_addNewNode(body, intersect.iEdge, intersect.t, 3-side,
                           &links->nNewNode, &links->newNodes, &index);
      if ((stat != EGADS_SUCCESS) && (stat != EGADS_OUTSIDE))
        if (outLevel > 0)
          printf(" EGADS Warning: addNewNode = %d (EG_phase3Intersections)!\n",
                 stat);
      status = EG_addLink(links, side, iNode, segment, index, seg);
      if ((status != EGADS_SUCCESS) && (status != EGADS_OUTSIDE))
        if (outLevel > 0)
          printf(" EGADS Warning: addLink = %d (EG_phase3Intersections)!\n",
                 stat);
#ifdef DEBUG
      if (status == EGADS_SUCCESS)
        if (index > 0) {
          printf(" Node %2d w/ Node %2d: Rat = %9.6lf -- dist = %lf  %2d\n",
                 iNode, index, 0.0, 0.0, links->nLink);
        } else {
          printf(" Node %2d w/ Edge %2d: t   = %9.6lf -- dist = %lf  %2d",
                 iNode, intersect.iEdge, intersect.t, intersect.dist,
                 links->nLink);
          if (stat == EGADS_OUTSIDE) printf(" *");
          printf("\n");
        }
#endif
    }
    if ((N2N == 0) && (intersect.iEdge == 0))
      printf(" EGADS Error: No Link to Node %2d (EG_phase3Intersections)!\n",
             iNode);
    
    segment = segment->next;
  } while (segment != src);

}


static int
EG_floodFaces(ego body, const ego face, int n, int *rmfMap, ego *e2fMap)
{
  int i, eIndex, fIndex, stat, nobj, outLevel;
  ego *objs;
  
  outLevel = EG_outLevel(body);

  stat = EG_getBodyTopos(body, face, EDGE, &nobj, &objs);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_getBodyTopos %d = %d (EG_floodFaces)!\n",
             EG_indexBodyTopo(body, face), stat);
    return stat;
  }
  
  for (i = 0; i < nobj; i++) {
    eIndex = EG_indexBodyTopo(body, objs[i]);
    if (eIndex <= EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_indexBodyTopo %d = %d (EG_floodFaces)!\n",
               i+1, eIndex);
      EG_free(objs);
      return eIndex;
    }
    if ((e2fMap[2*eIndex-2] != face) && (e2fMap[2*eIndex-2] != NULL)) {
      fIndex = EG_indexBodyTopo(body, e2fMap[2*eIndex-2]);
      if (fIndex <= EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: EG_indexBodyTopo Face0 = %d (EG_floodFaces)!\n",
                 stat);
        EG_free(objs);
        return stat;
      }
      if (rmfMap[fIndex-1] == -1) {
        rmfMap[fIndex-1] = n;
        stat = EG_floodFaces(body, e2fMap[2*eIndex-2], n, rmfMap, e2fMap);
        if (stat != EGADS_SUCCESS) {
          EG_free(objs);
          return stat;
        }
      }
    }
    if ((e2fMap[2*eIndex-1] != face) && (e2fMap[2*eIndex-1] != NULL)) {
      fIndex = EG_indexBodyTopo(body, e2fMap[2*eIndex-1]);
      if (fIndex <= EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: EG_indexBodyTopo Face1 = %d (EG_floodFaces)!\n",
                 fIndex);
        EG_free(objs);
        return fIndex;
      }
      if (rmfMap[fIndex-1] == -1) {
        rmfMap[fIndex-1] = n;
        stat = EG_floodFaces(body, e2fMap[2*eIndex-1], n, rmfMap, e2fMap);
        if (stat != EGADS_SUCCESS) {
          EG_free(objs);
          return stat;
        }
      }
    }
  }
  EG_free(objs);
  
  return EGADS_SUCCESS;
}


static int
EG_makeSegList(ego body, const char *aName1, const char *aName2,
               int *ncontour, Segment ***psegs1, Segment ***psegs2, int *both)
{
  int          i, j, k, m, n, stat, outLevel, nface, nedge, aType, aLen;
  int          cnt1, cnt2, cnt3, oclass, mtype, *senses, *rmfMap = NULL;
  ego          geom, *nds, *objs;
  ego          *faces = NULL, *edges = NULL, *e2fMap = NULL;
  double       trange[2];
  Segment      *temp, **segs1 = NULL, **segs2 = NULL;
  const int    *ints;
  const double *reals;
  const char   *string;
  
  *psegs1  = NULL;
  *psegs2  = NULL;
  *both    = cnt3 = 0;
  outLevel = EG_outLevel(body);
  
  /* get all of the Faces & Edges */
  stat = EG_getBodyTopos(body, NULL, FACE, &nface, &faces);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_getBodyTopos Face = %d (EG_makeSegList)!\n",
             stat);
    return stat;
  }
  if (faces == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: getBodyTopos faces == NULL (EG_makeSegList)!\n");
    return EGADS_MALLOC;
  }
  stat = EG_getBodyTopos(body, NULL, EDGE, &nedge, &edges);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_getBodyTopos Edge = %d (EG_makeSegList)!\n",
             stat);
    goto cleanup;
  }
  if (edges == NULL) {
    stat = EGADS_MALLOC;
    if (outLevel > 0)
      printf(" EGADS Error: getBodyTopos edges == NULL (EG_makeSegList)!\n");
    goto cleanup;
  }
#ifdef DEBUG
  printf(" nFaces = %d   nEdges = %d\n", nface, nedge);

  /* make sure we can get the Edge sense correct! */
  if (body->mtype == SOLIDBODY) {
    int eIndex, *esens;
    
    esens = (int *) EG_alloc(2*nedge*sizeof(int));
    if (esens == NULL) {
      if (outLevel > 0)
        printf(" EGADS Error: Edge Sense Malloc %d (EG_makeSegList)!\n",
               2*nedge);
      stat = EGADS_MALLOC;
      goto cleanup;
    }
    for (i = 0; i < 2*nedge; i++) esens[i] = 0;
    for (i = 0; i < nface; i++) {
      stat = EG_getBodyTopos(body, faces[i], EDGE, &n, &objs);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: EG_getBodyTopos %d sense = %d (makeSegList)!\n",
                 i+1, stat);
        EG_free(objs);
        goto cleanup;
      }
      for (j = 0; j < n; j++) {
        eIndex = EG_indexBodyTopo(body, objs[j]);
        k = EG_senseEdge(body, faces[i], objs[j]);
        if (k == EGADS_EXISTS) {
          printf(" Edge %d NOT found in Face %d!\n", eIndex, i+1);
        } else if (k == -1) {
          esens[2*eIndex-2]++;
        } else if (k ==  0) {
          printf(" Edge %d found Twice in Face %d!\n", eIndex, i+1);
        } else if (k ==  1) {
          esens[2*eIndex-1]++;
        } else {
          printf(" Edge %d Error in Face %d = %d!\n", eIndex, i+1, k);
        }
      }
      EG_free(objs);
    }

    for (j = i = 0; i < nedge; i++) {
      if (edges[i]->mtype == DEGENERATE) continue;
      if ((esens[2*i  ] != 1) || (esens[2*i+1] != 1)) j++;
    }
    if (j != 0)
      for (i = 0; i < nedge; i++)
        printf(" Edge %2d: %d %d\n", i+1, esens[2*i  ], esens[2*i+1]);
      
    EG_free(esens);
  }
#endif
  
  /* count Edges with our attributes */
  for (cnt1 = cnt2 = i = 0; i < nedge; i++) {
    if (edges[i]->mtype == DEGENERATE) continue;
    stat = EG_attributeRet(edges[i], aName1, &aType, &aLen,
                           &ints, &reals, &string);
    if (stat == EGADS_SUCCESS) cnt1++;
    stat = EG_attributeRet(edges[i], aName2, &aType, &aLen,
                           &ints, &reals, &string);
    if (stat == EGADS_SUCCESS) cnt2++;
  }
  if ((cnt1 == 0) || (cnt2 == 0)) {
    if (outLevel > 0)
      printf(" EGADS Error: nAttr Edges = %d %d (EG_makeSegList)!\n",
             cnt1, cnt2);
    stat = EGADS_ATTRERR;
    goto cleanup;
  }
  
  /* count Faces with our attributes */
  for (cnt1 = cnt2 = cnt3 = i = 0; i < nface; i++) {
    j    = 0;
    stat = EG_attributeRet(faces[i], aName1, &aType, &aLen,
                           &ints, &reals, &string);
    if (stat == EGADS_SUCCESS) {
      cnt1++;
      j++;
    }
    stat = EG_attributeRet(faces[i], aName2, &aType, &aLen,
                           &ints, &reals, &string);
    if (stat == EGADS_SUCCESS) {
      cnt2++;
      j++;
    }
    if (j == 2) cnt3++;
  }
  if (cnt1 + cnt2 == 0) {
    if (outLevel > 0)
      printf(" EGADS Error: nAttr Faces = %d %d (EG_makeSegList)!\n",
             cnt1, cnt2);
    stat = EGADS_ATTRERR;
    goto cleanup;
  }
  if (cnt3 != 0) {
    if ((cnt3 != cnt1) || (cnt3 != cnt2)) {
      if (outLevel > 0)
        printf(" EGADS Error: nAttr Faces = %d %d cnt = %d (EG_makeSegList)!\n",
               cnt1, cnt2, cnt3);
      stat = EGADS_ATTRERR;
      goto cleanup;
    }
  }
  
  /* build the Edge to Faces map */
  e2fMap = (ego *) EG_alloc(2*nedge*sizeof(ego));
  if (e2fMap == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Allocating %d egos (EG_makeSegList)!\n",
             2*nedge);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  for (i = 0; i < 2*nedge; i++) e2fMap[i] = NULL;
  for (i = 0; i < nface; i++) {
    stat = EG_attributeRet(faces[i], aName1, &aType, &aLen,
                           &ints, &reals, &string);
    if (stat != EGADS_SUCCESS)
      stat = EG_attributeRet(faces[i], aName2, &aType, &aLen,
                             &ints, &reals, &string);
    if (stat != EGADS_SUCCESS) continue;
    stat = EG_getBodyTopos(body, faces[i], EDGE, &n, &objs);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_getBodyTopos %d e2fMap = %d (makeSegList)!\n",
               i+1, stat);
      EG_free(objs);
      goto cleanup;
    }
    for (j = 0; j < n; j++) {
      if (objs[j]->mtype == DEGENERATE) continue;
      k = EG_indexBodyTopo(body, objs[j]);
      if (k <= EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Warning: indexBodyTopo %d/%d = %d (EG_makeSegList)!\n",
                 i+1, j+1, stat);
        continue;
      }
      if (e2fMap[2*k-2] == NULL) {
        e2fMap[2*k-2] = faces[i];
      } else if (e2fMap[2*k-1] == NULL) {
        e2fMap[2*k-1] = faces[i];
      } else {
        if (outLevel > 0)
          printf(" EGADS Error: Edge %d has 3 or more Faces (makeSegList)!\n",
                 k);
        stat = EGADS_TOPOCNT;
        EG_free(objs);
        goto cleanup;
      }
    }
    EG_free(objs);
  }
  /* check our marked Edges */
  for (i = 0; i < nedge; i++) {
    if (edges[i]->mtype == DEGENERATE) continue;
    stat = EG_attributeRet(edges[i], aName1, &aType, &aLen,
                           &ints, &reals, &string);
    if (stat != EGADS_SUCCESS)
      stat = EG_attributeRet(edges[i], aName2, &aType, &aLen,
                             &ints, &reals, &string);
    if (stat != EGADS_SUCCESS) continue;
    if (e2fMap[2*i  ] == NULL) {
      if (outLevel > 0)
        printf(" EGADS Error: Edge %d should have a Face (EG_makeSegList)!\n",
               i+1);
      stat = EGADS_TOPOCNT;
      goto cleanup;
    } else if (e2fMap[2*i+1] != NULL) {
      if (outLevel > 0)
        printf(" EGADS Error: Edge %d has nFace > 1 (EG_makeSegList)!\n", i+1);
      stat = EGADS_TOPOCNT;
      goto cleanup;
    }
  }
  
  /* build the removed Faces map & get nContour */
  rmfMap = (int *) EG_alloc(nface*sizeof(int));
  if (rmfMap == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Allocating %d ints (EG_makeSegList)!\n", nface);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  for (i = 0; i < nface; i++) rmfMap[i] = -1;
  for (n = i = 0; i < nface; i++) {
    if (rmfMap[i] != -1) continue;
    stat = EG_attributeRet(faces[i], aName1, &aType, &aLen,
                           &ints, &reals, &string);
    if (stat != EGADS_SUCCESS)
      stat = EG_attributeRet(faces[i], aName2, &aType, &aLen,
                             &ints, &reals, &string);
    if (stat != EGADS_SUCCESS) continue;
    rmfMap[i] = n;
    stat = EG_floodFaces(body, faces[i], n, rmfMap, e2fMap);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_floodFaces %d = %d (EG_makeSegList)!\n",
               i+1, stat);
      goto cleanup;
    }
    n++;
  }
  *ncontour = n;

#ifdef DEBUG
  printf(" nContour = %d   nRmFaces =", n);
  for (i = 0; i < n; i++) {
    for (j = k = 0; k < nface; k++)
      if (rmfMap[k] == i) j++;
    printf(" %d", j);
  }
  printf("\n");
#endif
  segs1 = (Segment **) EG_alloc(n*sizeof(Segment *));
  if (segs1 == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Allocating %d Segment1 (EG_makeSegList)!\n", n);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  for (i = 0; i < n; i++) segs1[i] = NULL;
  segs2 = (Segment **) EG_alloc(n*sizeof(Segment *));
  if (segs2 == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Allocating %d Segment1 (EG_makeSegList)!\n", n);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  for (i = 0; i < n; i++) segs2[i] = NULL;
  
  /* build the threaded contour lists */
  for (i = 0; i < n; i++) {
    for (cnt2 = cnt1 = j = 0; j < nedge; j++) {
      if (edges[j]->mtype == DEGENERATE) continue;
      stat = EG_attributeRet(edges[j], aName1, &aType, &aLen,
                             &ints, &reals, &string);
      if (stat == EGADS_SUCCESS) {
        k = EG_indexBodyTopo(body, e2fMap[2*j]);
        if (k <= EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Error: indexBody on Edge1 %d = %d (makeSegList)!\n",
                   j+1, k);
          stat = k;
          goto cleanup;
        }
        if (rmfMap[k-1] == i) cnt1++;
      }
      stat = EG_attributeRet(edges[j], aName2, &aType, &aLen,
                             &ints, &reals, &string);
      if (stat == EGADS_SUCCESS) {
        k = EG_indexBodyTopo(body, e2fMap[2*j]);
        if (k <= EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Error: indexBody on Edge2 %d = %d (makeSegList)!\n",
                   j+1, k);
          stat = k;
          goto cleanup;
        }
        if (rmfMap[k-1] == i) cnt2++;
      }
    }
#ifdef DEBUG
    printf(" Contour %d: Edge cnts = %d %d\n", i+1, cnt1, cnt2);
#endif
    k = cnt1;
    if (cnt2 > k) k = cnt2;
    temp = (Segment *) EG_alloc(k*sizeof(Segment));
    if (temp == NULL) {
      if (outLevel > 0)
        printf(" EGADS Error: Allocating %d Segments (EG_makeSegList)!\n", k);
      stat = EGADS_MALLOC;
      goto cleanup;
    }
    
    /* fill in contour1 data */
    for (cnt1 = j = 0; j < nedge; j++) {
      if (edges[j]->mtype == DEGENERATE) continue;
      stat = EG_attributeRet(edges[j], aName1, &aType, &aLen,
                             &ints, &reals, &string);
      if (stat == EGADS_SUCCESS) {
        k = EG_indexBodyTopo(body, e2fMap[2*j]);
        if (k <= EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Error: indexBody on Edge1 %d = %d (makeSegList)!\n",
                   j+1, k);
          stat = k;
          EG_free(temp);
          goto cleanup;
        }
        if (rmfMap[k-1] == i) {
          stat = EG_getTopology(edges[j], &geom, &oclass, &mtype, trange,
                                &m, &nds, &senses);
          if (stat != EGADS_SUCCESS) {
            if (outLevel > 0)
              printf(" EGADS Error: getTopo on Edge1 %d = %d (makeSegList)!\n",
                     j+1, temp[cnt1].sense);
            stat = temp[cnt1].sense;
            if (stat == EGADS_SUCCESS) stat = EGADS_TOPOERR;
            EG_free(temp);
            goto cleanup;
          }
          temp[cnt1].nodes[0]  = EG_indexBodyTopo(body, nds[0]);
          temp[cnt1].nodes[1]  = temp[cnt1].nodes[0];
          if (m != 1) temp[cnt1].nodes[1] = EG_indexBodyTopo(body, nds[1]);
          temp[cnt1].edge      = edges[j];
          temp[cnt1].trange[0] = trange[0];
          temp[cnt1].trange[1] = trange[1];
          temp[cnt1].sense     = EG_senseEdge(body, e2fMap[2*j], edges[j]);
          if (abs(temp[cnt1].sense) != 1) {
            if (outLevel > 0)
              printf(" EGADS Error: senseEdge on Edge1 %d = %d (makeSegList)!\n",
                     j+1, temp[cnt1].sense);
            stat = temp[cnt1].sense;
            if (stat == EGADS_SUCCESS) stat = EGADS_TOPOERR;
            EG_free(temp);
            goto cleanup;
          }
          cnt1++;
        }
      }
    }
    stat = EG_fillSegList(cnt1, temp, &segs1[i]);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        if (stat == EGADS_NOTFOUND) {
          printf(" EGADS Error: %d/%d contour1 not closed (EG_makeSegList)!\n",
                 i+1, n);
        } else if (stat == EGADS_OCSEGFLT) {
          printf(" EGADS Error: %d/%d contour1 too complex (EG_makeSegList)!\n",
                 i+1, n);
        } else {
          printf(" EGADS Error: fillSegList on %d contr1 = %d (makeSegList)!\n",
                 i+1, stat);
        }
      EG_free(temp);
      goto cleanup;
    }
    
    /* fill in contour2 data */
    for (cnt2 = j = 0; j < nedge; j++) {
      if (edges[j]->mtype == DEGENERATE) continue;
      stat = EG_attributeRet(edges[j], aName2, &aType, &aLen,
                             &ints, &reals, &string);
      if (stat == EGADS_SUCCESS) {
        k = EG_indexBodyTopo(body, e2fMap[2*j]);
        if (k <= EGADS_SUCCESS) {
          if (outLevel > 0)
            printf(" EGADS Error: indexBody on Edge2 %d = %d (makeSegList)!\n",
                   j+1, k);
          stat = k;
          EG_free(temp);
          goto cleanup;
        }
        if (rmfMap[k-1] == i) {
          stat = EG_getTopology(edges[j], &geom, &oclass, &mtype, trange,
                                &m, &nds, &senses);
          if (stat != EGADS_SUCCESS) {
            if (outLevel > 0)
              printf(" EGADS Error: getTopo on Edge2 %d = %d (makeSegList)!\n",
                     j+1, temp[cnt1].sense);
            stat = temp[cnt1].sense;
            if (stat == EGADS_SUCCESS) stat = EGADS_TOPOERR;
            EG_free(temp);
            goto cleanup;
          }
          temp[cnt2].nodes[0]  = EG_indexBodyTopo(body, nds[0]);
          temp[cnt2].nodes[1]  = temp[cnt2].nodes[0];
          if (m != 1) temp[cnt2].nodes[1] = EG_indexBodyTopo(body, nds[1]);
          temp[cnt2].edge      = edges[j];
          temp[cnt2].trange[0] = trange[0];
          temp[cnt2].trange[1] = trange[1];
          temp[cnt2].sense     = EG_senseEdge(body, e2fMap[2*j], edges[j]);
          if (abs(temp[cnt2].sense) != 1) {
            if (outLevel > 0)
              printf(" EGADS Error: senseEdge on Edge1 %d = %d (makeSegList)!\n",
                     j+1, temp[cnt2].sense);
            stat = temp[cnt2].sense;
            if (stat == EGADS_SUCCESS) stat = EGADS_TOPOERR;
            EG_free(temp);
            goto cleanup;
          }
          cnt2++;
        }
      }
    }
    stat = EG_fillSegList(cnt2, temp, &segs2[i]);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        if (stat == EGADS_NOTFOUND) {
          printf(" EGADS Error: %d/%d contour2 not closed (EG_makeSegList)!\n",
                 i+1, n);
        } else if (stat == EGADS_OCSEGFLT) {
          printf(" EGADS Error: %d/%d contour2 too complex (EG_makeSegList)!\n",
                 i+1, n);
        } else {
          printf(" EGADS Error: fillSegList on %d contr2 = %d (makeSegList)!\n",
                 i+1, stat);
        }
      EG_free(temp);
      goto cleanup;
    }

    EG_free(temp);
  }
  
cleanup:
  if (stat != EGADS_SUCCESS) {
    if (segs2 != NULL) {
      EG_freeSegments(*ncontour, segs2);
      EG_free(segs2);
      segs2 = NULL;
    }
    if (segs1 != NULL) {
      EG_freeSegments(*ncontour, segs1);
      EG_free(segs1);
      segs1 = NULL;
    }
    *ncontour = 0;
  }
  if (rmfMap != NULL) EG_free(rmfMap);
  if (e2fMap != NULL) EG_free(e2fMap);
  if (faces  != NULL) EG_free(faces);
  if (edges  != NULL) EG_free(edges);
  
  *psegs1 = segs1;
  *psegs2 = segs2;
  *both   = cnt3;
  return stat;
}
  

static int
EG_newNode2seg(int newNode, Segment *segment, Links *links)
{
  Segment *seg, *newSeg;
  
  /* segments may not match!
     spin around to find the appropriate Edge that includes our t */
  seg = segment;
  do {
    if (seg->edge == links->newNodes[newNode-1].edge) {
      if ((links->newNodes[newNode-1].t >= seg->trange[0]) &&
          (links->newNodes[newNode-1].t <= seg->trange[1])) {
        /* we have the segment to split */
        newSeg = (Segment *) EG_alloc(sizeof(Segment));
        if (newSeg == NULL) {
          printf(" EGADS Error: Malloc on a Segment (EG_newNode2seg)!\n");
          return EGADS_MALLOC;
        }
        /* copy our segment */
        newSeg->edge      = seg->edge;
        newSeg->nodes[0]  = seg->nodes[0];
        newSeg->nodes[1]  = seg->nodes[1];
        newSeg->sense     = seg->sense;
        newSeg->trange[0] = seg->trange[0];
        newSeg->trange[1] = seg->trange[1];
        newSeg->prev      = seg->prev;
        newSeg->next      = seg->next;
        /* adjust the end-point info */
        if (seg->sense == 1) {
             seg->nodes[1]  = -newNode;
          newSeg->nodes[0]  = -newNode;
             seg->trange[1] = links->newNodes[newNode-1].t;
          newSeg->trange[0] = links->newNodes[newNode-1].t;
        } else {
             seg->nodes[0]  = -newNode;
          newSeg->nodes[1]  = -newNode;
             seg->trange[0] = links->newNodes[newNode-1].t;
          newSeg->trange[1] = links->newNodes[newNode-1].t;
        }
        /* insert the new segment into the list */
           seg->next = newSeg;
        newSeg->prev = seg;
        
        return EGADS_SUCCESS;
      }
    }
    
    seg = seg->next;
  } while (seg != segment);
  
  return EGADS_NOTFOUND;
}


static int
EG_insertNodes(ego body, Links *links, ego *newbod)
{
  int    i, outLevel, nbody, oclass, mtype, aType, aLen, *map, *senses;
  int    stat, index, nnode,  nedge;
  ego    node, *nodes, *edges, context, nbod, geom, *bodies, model;
  double *xyzs;
  const int    *ints;
  const double *reals;
  const char   *string;
  
  *newbod = NULL;
  context = EG_context(body);
  if  (context == NULL) return EGADS_NOTCNTX;
  
  if (links->nNewNode == 0) {
    /* we are OK -- make a copy */
    stat = EG_copyObject(body, NULL, newbod);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Error: EG_copyObject = %d (EG_insertNodes)!\n", stat);
      return stat;
    }
    return EGADS_SUCCESS;
  }
  
  /* mark the Nodes and Edges */
  stat = EG_getBodyTopos(body, NULL, NODE, &nnode, &nodes);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_getBodyTopos Node = %d (EG_insertNodes)!\n", stat);
    return stat;
  }
  for (i = 0; i < nnode; i++)
    EG_attributeAdd(nodes[i], ".IDmate", ATTRINT, 1, &i, NULL, NULL);
  
  stat = EG_getBodyTopos(body, NULL, EDGE, &nedge, &edges);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_getBodyTopos Edge = %d (EG_insertNodes)!\n", stat);
    EG_free(nodes);
    return stat;
  }
  for (i = 0; i < nedge; i++)
    EG_attributeAdd(edges[i], ".IDmate", ATTRINT, 1, &i, NULL, NULL);
  
  /* make a dummy Node */
  stat = EG_makeTopology(context, NULL, NODE, 0, links->newNodes[0].xyz, 0,
                         NULL, NULL, &node);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_makeTopology = %d (EG_insertNodes)!\n", stat);
    EG_free(edges);
    EG_free(nodes);
    return stat;
  }
  if (links->nNewNode > 1) {
    xyzs = (double *) EG_alloc(3*links->nNewNode*sizeof(double));
    if (xyzs == NULL) {
      printf(" EGADS Error: malloc of %d xyzs (EG_insertNodes)!\n",
             links->nNewNode);
      EG_deleteObject(node);
      EG_free(edges);
      EG_free(nodes);
      return stat;
    }
    for (i = 0; i < links->nNewNode; i++) {
      xyzs[3*i  ] = links->newNodes[i].xyz[0];
      xyzs[3*i+1] = links->newNodes[i].xyz[1];
      xyzs[3*i+2] = links->newNodes[i].xyz[2];
    }
    EG_attributeAdd(node, ".Nodes", ATTRREAL, 3*links->nNewNode,
                    NULL, xyzs, NULL);
    EG_free(xyzs);
  }
  
  /* insert the new Nodes */
  outLevel = EG_setOutLevel(context, 0);
  stat = EG_generalBoolean(body, node, SPLITTER, 0.0, &model);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_generalBoolean = %d (EG_insertNodes)!\n", stat);
    EG_deleteObject(node);
    EG_free(edges);
    EG_free(nodes);
    return stat;
  }
  (void) EG_setOutLevel(context, outLevel);
  
  stat = EG_getTopology(model, &geom, &oclass, &mtype, NULL, &nbody, &bodies,
                        &senses);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_getTopology M = %d (EG_insertNodes)!\n", stat);
    EG_deleteObject(node);
    EG_deleteObject(model);
    EG_free(edges);
    EG_free(nodes);
    return stat;
  }
  if (nbody != 1) {
    printf(" EGADS Error: nBody = %d (EG_insertNodes)!\n", nbody);
    EG_deleteObject(node);
    EG_deleteObject(model);
    EG_free(edges);
    EG_free(nodes);
    return EGADS_INDEXERR;
  }
  EG_patchSplitterAttr(".IDmate", nedge, edges, node, bodies[0]);
  EG_deleteObject(node);
  
  /* remove the attributes added to Body */
  for (i = 0; i < nedge; i++) EG_attributeDel(edges[i], ".IDmate");
  for (i = 0; i < nnode; i++) EG_attributeDel(nodes[i], ".IDmate");
  EG_free(edges);
  EG_free(nodes);
  
  stat = EG_copyObject(bodies[0], NULL, &nbod);
  EG_deleteObject(model);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_copyObject = %d (EG_insertNodes)!\n", stat);
    return stat;
  }
  
  /* remove our Edge attributes in the new Body */
  stat = EG_getBodyTopos(nbod, NULL, EDGE, &nedge, &edges);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_getBodyTopos New Edge = %d (EG_insertNodes)!\n",
           stat);
    EG_deleteObject(nbod);
    return stat;
  }
  for (i = 0; i < nedge; i++) EG_attributeDel(edges[i], ".IDmate");
  EG_free(edges);
  
  /* map the links to the new Body */
  stat = EG_getBodyTopos(nbod, NULL, NODE, &nnode, &nodes);
  if (stat != EGADS_SUCCESS) {
    printf(" EGADS Error: EG_getBodyTopos new Node = %d (EG_insertNodes)!\n",
           stat);
    EG_deleteObject(nbod);
    return stat;
  }
  map = (int *) EG_alloc(nnode*sizeof(int));
  if (map == NULL) {
    printf(" EGADS Error: map malloc of size %d (EG_insertNodes)!\n", nnode);
    EG_free(nodes);
    EG_deleteObject(nbod);
    return EGADS_MALLOC;
  }
  for (i = 0; i < nnode; i++) {
    stat = EG_attributeRet(nodes[i], ".IDmate", &aType, &aLen,
                           &ints, &reals, &string);
    if (stat != EGADS_SUCCESS) {
      printf(" EGADS Error: EG_attributeRet Node %d = %d (EG_insertNodes)!\n",
             i+1, stat);
      EG_free(map);
      EG_free(nodes);
      EG_deleteObject(nbod);
      return stat;
    }
    index = ints[0] + links->nNewNode;
    map[index] = i + 1;
    EG_attributeDel(nodes[i], ".IDmate");
  }
  EG_free(nodes);
  
  /* reassign the link end-points */
  for (i = 0; i < 2*links->nLink; i++) {
    index = links->links[i] + links->nNewNode;
    if (links->links[i] > 0) index--;
    links->links[i] = map[index];
  }
  EG_free(map);
  
  *newbod = nbod;
  return EGADS_SUCCESS;
}


static int
EG_reorderFan(Links *links, int side, Segment *segment, Fan *fan)
{
  int     i, j, m, n, iNode, in[4], jn[4], lf[4];
  Segment *seg, *start;
  
                              n = 1;
  if (fan->linkFans[1] != -1) n = 2;
  if (fan->linkFans[2] != -1) n = 3;
  if (fan->linkFans[3] != -1) n = 4;
  if (n == 1) return EGADS_SUCCESS;
  
  for (i = 0; i < n; i++) {
    j     = fan->linkFans[i];
    in[i] = links->links[2*j+side-1];
    lf[i] = j;
    jn[i] = 0;
  }
  
  /* find the first non-match */
  start = segment;
  do {
                            iNode = start->nodes[0];
    if  (start->sense == 1) iNode = start->nodes[1];
    for (i = 0; i < n; i++)
      if (iNode == in[i]) break;
    if (i == n) break;
    start = start->next;
  } while (start != segment);
  
  /* locate the contiguous links */
  m   = 0;
  seg = start;
  do {
                          iNode = seg->nodes[0];
    if  (seg->sense == 1) iNode = seg->nodes[1];
    for (i = 0; i < n; i++) {
      if (iNode == in[i]) {
        jn[m] = i;
        m++;
        break;
      }
    }
    if  (m == n) break;
    if ((m != 0) && (i == n)) return EGADS_SEQUERR;
    seg = seg->next;
  } while (seg != start);
  
  for (i = 0; i < n; i++) fan->linkFans[i] = lf[jn[i]];
  
  return EGADS_SUCCESS;
}


static int
EG_buildContours(ego body, int *nodeTable, Links *links, int nFan, Fan *fans,
                 Segment *segs1, Segment *segs2, int iCntr,
                 int *ncnt, int *cntr1, int *cntr2)
{
  int     i, m, n, ccnt, iNode, in1, in2, index, link, *nind;
  Segment *seg, *start;
  
  /* get the count */
  for (ccnt = i = 0; i < nFan; i++)
    if (fans[i].iCntr == iCntr) ccnt += fans[i].nFan;
  seg = segs1;
  do {
    ccnt++;
    seg = seg->next;
  } while (seg != segs1);
  seg = segs2;
  do {
    ccnt++;
    seg = seg->next;
  } while (seg != segs2);
  ccnt /= 2;
  
  /* allocate our Node index vector */
  nind = (int *) EG_alloc((ccnt+5)*sizeof(int));
  if (nind == NULL) return EGADS_MALLOC;

  /* fill side1 and Node index vector */
                         iNode = segs1->nodes[1];
  if (segs1->sense == 1) iNode = segs1->nodes[0];
  n     = 0;
  m     = *ncnt;
  index = nodeTable[iNode-1];
  for (i = fans[index].nFan; i >= 0; i--) {
    link    = fans[index].linkFans[i];
    nind[n] = links->links[2*link+1];
    n++;
  }
  for (i = fans[index].nFan; i > 0; i--, m++) {
    cntr1[2*m  ] = iNode;
    cntr1[2*m+1] = 0;
  }
  seg = segs1;
  do {
                          iNode = seg->nodes[0];
    if  (seg->sense == 1) iNode = seg->nodes[1];
    index = nodeTable[iNode-1];
    for (i = fans[index].nFan; i >= 0; i--) {
      link    = fans[index].linkFans[i];
      nind[n] = links->links[2*link+1];
      n++;
    }
    cntr1[2*m  ] = EG_indexBodyTopo(body, seg->edge);
    cntr1[2*m+1] = seg->sense;
    m++;
    if (seg->next != segs1)
      for (i = fans[index].nFan; i > 0; i--, m++) {
        cntr1[2*m  ] = iNode;
        cntr1[2*m+1] = 0;
      }
    seg = seg->next;
  } while (seg != segs1);
  i  = m;
  i -= *ncnt;
  if (i != ccnt) {
    printf(" EGADS Error: Mismatch for Segs1 %d %d (buildContours)!\n",
           i, ccnt);
    EG_free(nind);
    return EGADS_CONSTERR;
  }

#ifdef DEBUG
  printf("\n seg2 ->\n");
  for (i = 0; i <= ccnt; i++) printf(" %2d: %2d\n", i, nind[i]);
  printf("\n");
#endif
  
  /* match Nodes with segs2 (backwards) */
  n = *ncnt;
  for (i = 0; i < ccnt; i++, n++) {
    if (nind[i] != nind[i+1]) break;
    cntr2[2*n  ] = nind[i];
    cntr2[2*n+1] = 0;
  }
  start = segs2;
  do {
    if (start->sense == 1) {
      in1 = start->nodes[1];
      in2 = start->nodes[0];
    } else {
      in1 = start->nodes[0];
      in2 = start->nodes[1];
    }
    if ((in1 == nind[i]) && (in2 == nind[i+1])) break;
    start = start->prev;
  } while (start != segs2);
  if ((in1 != nind[i]) || (in2 != nind[i+1])) {
    printf(" EGADS Error: Cannot find start for Segs2 %d %d (buildContours)!\n",
           nind[i], nind[i+1]);
    EG_free(nind);
    return EGADS_CONSTERR;
  }
  
  /* match Nodes */
  seg = start;
  do {
    while (nind[i] == nind[i+1]) {
      cntr2[2*n  ] = nind[i];
      cntr2[2*n+1] = 0;
      n++;
      i++;
    }
    if (seg->sense == 1) {
      in1 = seg->nodes[1];
      in2 = seg->nodes[0];
    } else {
      in1 = seg->nodes[0];
      in2 = seg->nodes[1];
    }
    if ((in1 != nind[i]) || (in2 != nind[i+1])) {
      printf(" EGADS Error: Cannot find Edge in Segs2 %d %d (buildContours)!\n",
             nind[i], nind[i+1]);
      EG_free(nind);
      return EGADS_CONSTERR;
    }
    cntr2[2*n  ] = EG_indexBodyTopo(body, seg->edge);
    cntr2[2*n+1] = -seg->sense;
    n++;
    i++;
    seg = seg->prev;
  } while (seg != start);
  while (i < ccnt) {
    cntr2[2*n  ] = nind[i];
    cntr2[2*n+1] = 0;
    n++;
    i++;
  }
  if (i != ccnt) {
    printf(" EGADS Error: Mismatch for Segs2 %d %d (buildContours)!\n",
           i, ccnt);
    EG_free(nind);
    return EGADS_CONSTERR;
  }

#ifdef DEBUG
  printf(" contours %d ->\n", iCntr);
  for (i = *ncnt; i < m; i++)
    printf(" %2d: %2d %2d    %2d %2d\n", i, cntr1[2*i  ], cntr1[2*i+1],
                                            cntr2[2*i  ], cntr2[2*i+1]);
  printf("\n");
#endif
  
  EG_free(nind);
  *ncnt = *ncnt + ccnt;
  return EGADS_SUCCESS;
}


int
EG_mateContours(ego body, const char *aName1, const char *aName2, ego *newBody,
                int *ncontour, int **nent, int **contour1, int **contour2)
{
  int     i, j, n, stat, outLevel, in1, in2, index, nnode, ccnt, nFan, both;
  int     *nentity = NULL, *cntr1 = NULL, *cntr2 = NULL, *nodeTable = NULL;
  ego     context, newbod = NULL;
  Segment *seg, **segs1 = NULL, **segs2 = NULL;
  Fan     fan, *fans = NULL;
  Links   links;
#if defined(PRELINKWRITE) || defined(POSTLINKWRITE)
  double  xyz1[3], xyz2[3];
  ego     obj;
  FILE    *fp = NULL;
#endif

  *newBody  = NULL;
  *ncontour = 0;
  *nent     = NULL;
  *contour1 = NULL;
  *contour2 = NULL;
  if  (body == NULL)               return EGADS_NULLOBJ;
  if  (body->magicnumber != MAGIC) return EGADS_NOTOBJ;
  if  (body->oclass != BODY)       return EGADS_NOTBODY;
  if ((body->mtype != SOLIDBODY) &&
      (body->mtype != SHEETBODY))  return EGADS_TOPOERR;
  if  (body->blind == NULL)        return EGADS_NODATA;
  if  (EG_sameThread(body))        return EGADS_CNTXTHRD;
  outLevel = EG_outLevel(body);
  context  = EG_context(body);
  if  (context == NULL)            return EGADS_NOTCNTX;
  
  links.nLink    = 0;
  links.links    = NULL;
  links.segments = NULL;
  links.nNewNode = 0;
  links.newNodes = NULL;
  
  /* make the segment lists for the input Body */
  stat = EG_makeSegList(body, aName1, aName2, ncontour, &segs1, &segs2, &both);
  if ((stat != EGADS_SUCCESS) || (segs1 == NULL) || (segs2 == NULL)) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_makeSegList Body = %d (EG_mateContours)!\n",
             stat);
    return stat;
  }

  /* find the intersections: Nodes with Nodes and/or Nodes with Edges */
  for (i = 0; i < *ncontour; i++) {
    EG_findIntersections(body, aName1, segs1[i], aName2, segs2[i], 1, both,
                         &links);
    EG_findIntersections(body, aName2, segs2[i], aName1, segs1[i], 2, both,
                         &links);
  }
  /* Phase 2 */
  for (i = 0; i < *ncontour; i++) {
    EG_missedIntersections(body, aName1, segs1[i], aName2, segs2[i], 1, &links);
    EG_missedIntersections(body, aName2, segs2[i], aName1, segs1[i], 2, &links);
  }
  /* Phase 3 */
  for (i = 0; i < *ncontour; i++) {
    EG_phase3Intersections(body, segs1[i], segs2[i], 1, &links);
    EG_phase3Intersections(body, segs2[i], segs1[i], 2, &links);
  }
#ifdef DEBUG
  for (i = 0; i < *ncontour; i++) {
    printf(" Contour %d\n", i+1);
    EG_printSegList(body, segs1[i]);
    printf("\n");
    EG_printSegList(body, segs2[i]);
    printf("\n");
  }
#endif
  if ((links.links == NULL) || (links.segments == NULL)) {
    if (outLevel > 0)
      printf(" EGADS Error: No links found (mateContours)!\n");
    stat = EGADS_NOTFOUND;
    goto cleanup;
  }
    
  /* insert the new Nodes into the segment lists */
  for (i = 0; i < links.nLink; i++) {
    if (links.links[2*i  ] < 0) {
      stat = EG_newNode2seg(-links.links[2*i  ], links.segments[2*i  ], &links);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: EG_newNode2seg 1 = %d (mateContours)!\n", stat);
        goto cleanup;
      }
    }
    if (links.links[2*i+1] < 0) {
      stat = EG_newNode2seg(-links.links[2*i+1], links.segments[2*i+1], &links);
      if (stat != EGADS_SUCCESS) {
        if (outLevel > 0)
          printf(" EGADS Error: EG_newNode2seg 2 = %d (mateContours)!\n", stat);
        goto cleanup;
      }
    }
  }
  
#ifdef DEBUG
  for (i = 0; i < *ncontour; i++) {
    printf(" Contour %d\n", i+1);
    EG_printSegList(body, segs1[i]);
    printf("\n");
    EG_printSegList(body, segs2[i]);
    printf("\n");
  }
#endif
  
  /* build fans where appropriate */
  
#ifdef PRELINKWRITE
  fp = fopen("mateContour.txt", "w");
  if (fp == NULL) {
    printf(" Error: Cannot open 'mateContour.txt'!\n");
  } else {
    for (i = 0; i < links.nLink; i++) {
      xyz1[0] = xyz1[1] = xyz1[2] = 0.0;
      xyz2[0] = xyz2[1] = xyz2[2] = 0.0;
      in1 = links.links[2*i  ];
      if (in1 > 0) {
        (void) EG_objectBodyTopo(body, NODE, in1, &obj);
        (void) EG_evaluate(obj, NULL, xyz1);
      } else if (links.newNodes != NULL) {
        j       = -in1 - 1;
        xyz1[0] = links.newNodes[j].xyz[0];
        xyz1[1] = links.newNodes[j].xyz[1];
        xyz1[2] = links.newNodes[j].xyz[2];
      }
      in2 = links.links[2*i+1];
      if (in2 > 0) {
        (void) EG_objectBodyTopo(body, NODE, in2, &obj);
        (void) EG_evaluate(obj, NULL, xyz2);
      } else if (links.newNodes != NULL) {
        j       = -in2 - 1;
        xyz2[0] = links.newNodes[j].xyz[0];
        xyz2[1] = links.newNodes[j].xyz[1];
        xyz2[2] = links.newNodes[j].xyz[2];
      }
      if ((links.links[2*i  ] > 0) && (links.links[2*i+1] > 0)) {
        fprintf(fp, "%d %d 2 %lf %lf %lf  %lf %lf %lf\n", i+1, 0,
                xyz1[0], xyz1[1], xyz1[2], xyz2[0], xyz2[1], xyz2[2]);
      } else {
        fprintf(fp, "%d %d 0 %lf %lf %lf  %lf %lf %lf\n", i+1, 0,
                xyz1[0], xyz1[1], xyz1[2], xyz2[0], xyz2[1], xyz2[2]);
      }
    }
    fprintf(fp, "0 0 0 0 0 0 0 0 0\n");
    fclose(fp);
  }
#endif

  /* insert the Nodes */
  stat = EG_insertNodes(body, &links, &newbod);
  if ((stat != EGADS_SUCCESS) || (newbod == NULL)) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_insertNodes = %d (mateContours)!\n", stat);
    goto cleanup;
  }

#ifdef POSTLINKWRITE
  fp = fopen("mateContour.txt", "w");
  if (fp == NULL) {
    printf(" Error: Cannot open 'mateContour.txt'!\n");
  } else {
    for (i = 0; i < links.nLink; i++) {
      xyz1[0] = xyz1[1] = xyz1[2] = 0.0;
      xyz2[0] = xyz2[1] = xyz2[2] = 0.0;
      in1  = links.links[2*i  ];
      stat = EG_objectBodyTopo(newbod, NODE, in1, &obj);
      if (stat != EGADS_SUCCESS) {
        printf(" Link1 %d EG_objectBodyTopo = %d\n", i+1, stat);
      } else {
        stat = EG_evaluate(obj, NULL, xyz1);
        if (stat != EGADS_SUCCESS)
          printf(" Link1 %d EG_evaluate = %d\n", i+1, stat);
      }
      in2  = links.links[2*i+1];
      stat = EG_objectBodyTopo(newbod, NODE, in2, &obj);
      if (stat != EGADS_SUCCESS) {
        printf(" Link2 %d EG_objectBodyTopo = %d\n", i+1, stat);
      } else {
        stat = EG_evaluate(obj, NULL, xyz2);
        if (stat != EGADS_SUCCESS)
          printf(" Link2 %d EG_evaluate = %d\n", i+1, stat);
      }
      fprintf(fp, "%d %d 2 %lf %lf %lf  %lf %lf %lf\n", i+1, 0,
              xyz1[0], xyz1[1], xyz1[2], xyz2[0], xyz2[1], xyz2[2]);
    }
    fprintf(fp, "0 0 0 0 0 0 0 0 0\n");
    fclose(fp);
  }
#endif

  /* create the return results */
  EG_freeSegments(*ncontour, segs2);
  EG_free(segs2);
  EG_freeSegments(*ncontour, segs1);
  EG_free(segs1);
  segs1 = segs2 = NULL;
  
  /* make the segment lists for the new Body */
  stat = EG_makeSegList(newbod, aName1, aName2, &n, &segs1, &segs2, &both);
  if ((stat != EGADS_SUCCESS) || (segs1 == NULL) || (segs2 == NULL)) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_makeSegList Body = %d (EG_mateContours)!\n",
             stat);
    return stat;
  }
  if (n != *ncontour) {
    if (outLevel > 0)
      printf(" EGADS Error: nContour mismatch %d %d (EG_mateContours)!\n",
             n, *ncontour);
    stat = EGADS_TOPOCNT;
    goto cleanup;
  }

#ifdef DEBUG
  for (i = 0; i < *ncontour; i++) {
    printf(" Contour %d\n", i+1);
    EG_printSegList(newbod, segs1[i]);
    printf("\n");
    EG_printSegList(newbod, segs2[i]);
    printf("\n");
  }
  printf(" links ->\n");
  for (i = 0; i < links.nLink; i++)
    printf(" %2d: %2d %2d\n", i, links.links[2*i  ], links.links[2*i+1]);
#endif
  
  /* create the Node fan information */
  stat = EG_getBodyTopos(newbod, NULL, NODE, &nnode, NULL);
  if (stat != EGADS_SUCCESS) {
    if (outLevel > 0)
      printf(" EGADS Error: EG_getBodyTopos N = %d (EG_mateContours)!\n", stat);
    goto cleanup;
  }
  nodeTable = (int *) EG_alloc(nnode*sizeof(int));
  if (nodeTable == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Malloc of %d ints (EG_mateContours)!\n", nnode);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  for (i = 0; i < nnode; i++) nodeTable[i] = -1;
  for (nFan = i = 0; i < links.nLink; i++) {
    in1 = links.links[2*i  ];
    in2 = links.links[2*i+1];
    if (nodeTable[in1-1] == -1) {
      nodeTable[in1-1] = 0;
      nFan++;
    }
    if (nodeTable[in2-1] == -1) {
      nodeTable[in2-1] = 0;
      nFan++;
    }
  }
  for (i = 0; i < nnode; i++) nodeTable[i] = -1;
  fans = (Fan *) EG_alloc(nFan*sizeof(Fan));
  if (fans == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Malloc of %d Fans (EG_mateContours)!\n", n);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  for (i = 0; i < nFan; i++) {
    fans[i].iCntr       = 0;
    fans[i].iNode       = 0;
    fans[i].nFan        = 0;
    fans[i].linkFans[0] = fans[i].linkFans[1] = -1;
    fans[i].linkFans[2] = fans[i].linkFans[3] = -1;
  }
  for (n = i = 0; i < links.nLink; i++) {
    in1 = links.links[2*i  ];
    in2 = links.links[2*i+1];
    if (nodeTable[in1-1] == -1) {
      nodeTable[in1-1]    = n;
      fans[n].iNode       = in1;
      fans[n].linkFans[0] = i;
      n++;
    } else {
      index = nodeTable[in1-1];
      if (fans[index].linkFans[1] == -1) {
        fans[index].linkFans[1] = i;
      } else if (fans[index].linkFans[2] == -1) {
        fans[index].linkFans[2] = i;
      } else if (fans[index].linkFans[3] == -1) {
        fans[index].linkFans[3] = i;
      } else {
        if (outLevel > 0)
          printf(" EGADS Error: Fan limit for Node %d (EG_mateContours)!\n",
                 in1);
        stat = EGADS_INDEXERR;
        goto cleanup;
      }
    }
    if (nodeTable[in2-1] == -1) {
      nodeTable[in2-1]    = n;
      fans[n].iNode       = in2;
      fans[n].linkFans[0] = i;
      n++;
    } else {
      index = nodeTable[in2-1];
      if (fans[index].linkFans[1] == -1) {
        fans[index].linkFans[1] = i;
      } else if (fans[index].linkFans[2] == -1) {
        fans[index].linkFans[2] = i;
      } else if (fans[index].linkFans[3] == -1) {
        fans[index].linkFans[3] = i;
      } else {
        if (outLevel > 0)
          printf(" EGADS Error: Fan limit for Node %d (EG_mateContours)!\n",
                 in2);
        stat = EGADS_INDEXERR;
        goto cleanup;
      }
    }
  }
  
  /* mark which contour */
  if (*ncontour > 1) {
    for (i = 0; i < *ncontour; i++) {
      seg = segs1[i];
      do {
        index = nodeTable[seg->nodes[0]-1];
        if (index == -1) {
          if (outLevel > 0)
            printf(" EGADS Error: Node %d not covered (EG_mateContours)!\n",
                   seg->nodes[0]);
          stat = EGADS_INDEXERR;
          goto cleanup;
        }
        fans[index].iCntr = i;
        index = nodeTable[seg->nodes[1]-1];
        fans[index].iCntr = i;
        if (index == -1) {
          if (outLevel > 0)
            printf(" EGADS Error: Node %d not covered (EG_mateContours)!\n",
                   seg->nodes[1]);
          stat = EGADS_INDEXERR;
          goto cleanup;
        }
        seg = seg->next;
      } while (seg != segs1[i]);
      
      seg = segs2[i];
      do {
        index = nodeTable[seg->nodes[0]-1];
        if (index == -1) {
          if (outLevel > 0)
            printf(" EGADS Error: Node %d not covered (EG_mateContours)!\n",
                   seg->nodes[0]);
          stat = EGADS_INDEXERR;
          goto cleanup;
        }
        fans[index].iCntr = i;
        index = nodeTable[seg->nodes[1]-1];
        if (index == -1) {
          if (outLevel > 0)
            printf(" EGADS Error: Node %d not covered (EG_mateContours)!\n",
                   seg->nodes[1]);
          stat = EGADS_INDEXERR;
          goto cleanup;
        }
        fans[index].iCntr = i;
        seg = seg->next;
      } while (seg != segs2[i]);
    }
  }
  
  /* order the fans */
  for (ccnt = i = 0; i < n; i++) {
    fan = fans[i];
    if (fan.linkFans[1] == -1) continue;
    fan.nFan++;
    if (fan.linkFans[2] != -1) fan.nFan++;
    if (fan.linkFans[3] != -1) fan.nFan++;
    ccnt += fan.nFan;
    index = fan.linkFans[0];
    if (links.links[2*index  ] == fan.iNode) {
      /* we are in segs1 */
      stat = EG_reorderFan(&links, 2, segs2[fan.iCntr], &fan);
      if (stat != EGADS_SUCCESS) {
        if (stat == EGADS_SEQUERR) {
          if (outLevel > 0)
            printf(" EGADS Error: reorderFan 1 = CrossOver (mateContours)!\n");
        } else {
          if (outLevel > 0)
            printf(" EGADS Error: reorderFan 1 = %d (EG_mateContours)!\n",
                   stat);
        }
        goto cleanup;
      }
      fans[i] = fan;
    } else if (links.links[2*index+1] == fan.iNode) {
      /* we are in segs2 */
      stat = EG_reorderFan(&links, 1, segs1[fan.iCntr], &fan);
      if (stat != EGADS_SUCCESS) {
        if (stat == EGADS_SEQUERR) {
          if (outLevel > 0)
            printf(" EGADS Error: reorderFan 2 = CrossOver (mateContours)!\n");
        } else {
          if (outLevel > 0)
            printf(" EGADS Error: reorderFan 2 = %d (EG_mateContours)!\n",
                   stat);
        }
        goto cleanup;
      }
      fans[i] = fan;
    } else {
      printf(" EGADS Info: Cannot find Node %d in Link %d (EG_mateContours)!\n",
             fans[i].iNode, index);
    }
  }
  /* get the total contour count */
  for (i = 0; i < *ncontour; i++) {
    seg = segs1[i];
    do {
      ccnt++;
      seg = seg->next;
    } while (seg != segs1[i]);
    
    seg = segs2[i];
    do {
      ccnt++;
      seg = seg->next;
    } while (seg != segs2[i]);
  }
  ccnt /= 2;
#ifdef DEBUG
  printf(" Fans %d ->\n", ccnt);
  for (i = 0; i < nnode; i++) {
    index = nodeTable[i];
    if (index == -1) continue;
    printf(" %2d: %2d %2d -- %2d", fans[index].iNode, fans[index].iCntr,
           fans[index].nFan, fans[index].linkFans[0]);
    if (fans[index].linkFans[1] != -1) printf(" %2d", fans[index].linkFans[1]);
    if (fans[index].linkFans[2] != -1) printf(" %2d", fans[index].linkFans[2]);
    if (fans[index].linkFans[3] != -1) printf(" %2d", fans[index].linkFans[3]);
    printf("\n");
  }
#endif
  
  /* connect the links, fans and the segment lists */
  nentity = (int *) EG_alloc(*ncontour*sizeof(int));
  if (nentity == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Malloc of %d int entities (EG_mateContours)!\n",
             *ncontour);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  cntr1 = (int *) EG_alloc(2*ccnt*sizeof(int));
  if (cntr1 == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Malloc of %d ints for cntr1 (EG_mateContours)!\n",
             ccnt);
    stat = EGADS_MALLOC;
    goto cleanup;
  }
  cntr2 = (int *) EG_alloc(2*ccnt*sizeof(int));
  if (cntr2 == NULL) {
    if (outLevel > 0)
      printf(" EGADS Error: Malloc of %d ints for cntr2 (EG_mateContours)!\n",
             ccnt);
    stat = EGADS_MALLOC;
    goto cleanup;
  }

  for (n = j = i = 0; i < *ncontour; i++) {
    stat = EG_buildContours(newbod, nodeTable, &links, nFan, fans,
                            segs1[i], segs2[i], i, &n, cntr1, cntr2);
    if (stat != EGADS_SUCCESS) {
      if (outLevel > 0)
        printf(" EGADS Error: EG_buildContours %d = %d (EG_mateContours)!\n",
               i, stat);
      goto cleanup;
    }
    nentity[i] = n - j;
    j = n;
  }
  /* do we have the right number of entries? */
  if (n != ccnt) {
    if (outLevel > 0)
      printf(" EGADS Error: Contour count mismatch %d %d (EG_mateContours)!\n",
             n, ccnt);
    stat = EGADS_INDEXERR;
    goto cleanup;
  }
  /* check to make sure we don't have Nodes connecting to nodes */
  for (j = i = 0; i < n; i++)
    if ((cntr1[2*i+1] == 0) && (cntr2[2*i+1] == 0)) {
      printf(" EGADS Error: slot %d has 2 Node markers %d %d (mateContours)!\n",
             i+1, cntr1[2*i], cntr2[2*i]);
      j++;
    }
  if (j != 0) {
    stat = EGADS_INDEXERR;
    goto cleanup;
  }
  
  stat = EGADS_SUCCESS;
  
cleanup:
  if (fans           != NULL) EG_free(fans);
  if (nodeTable      != NULL) EG_free(nodeTable);
  if (links.links    != NULL) EG_free(links.links);
  if (links.segments != NULL) EG_free(links.segments);
  if (links.newNodes != NULL) EG_free(links.newNodes);
  if (segs2 != NULL) {
    EG_freeSegments(*ncontour, segs2);
    EG_free(segs2);
  }
  if (segs1 != NULL) {
    EG_freeSegments(*ncontour, segs1);
    EG_free(segs1);
  }
  
  if (stat != EGADS_SUCCESS) {
    if (newbod != NULL)  EG_deleteObject(newbod);
    if (nentity != NULL) EG_free(nentity);
    if (cntr1   != NULL) EG_free(cntr1);
    if (cntr2   != NULL) EG_free(cntr2);
    *ncontour = 0;
  } else {
    *newBody  = newbod;
    *nent     = nentity;
    *contour1 = cntr1;
    *contour2 = cntr2;
  }

  return stat;
}
