## [import]
# Import pyCAPS module
import pyCAPS

# Import os module
import os
import argparse
## [import]

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'FRICTION Pytest Example',
                                 prog = 'friction_PyTest.py',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = ["."+os.sep], nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument('-noAnalysis', action='store_true', default = False, help = "Don't run analysis code")
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

## [localVariable]
# Create working directory variable
workDir = os.path.join(str(args.workDir[0]), "FrictionAnalysisTest")
## [localVariable]

# -----------------------------------------------------------------
# Load CSM file
# -----------------------------------------------------------------
## [geometry]
geometryScript = os.path.join("..","csmData","frictionWingTailFuselage.csm")
capsProblem = pyCAPS.Problem(problemName=workDir,
                             capsFile=geometryScript,
                             outLevel=args.outLevel)

capsProblem.geometry.despmtr.area = 10.0
## [geometry]

# -----------------------------------------------------------------
# Load desired aim
# -----------------------------------------------------------------
print ("Loading AIM")
## [loadAIM]
friction = capsProblem.analysis.create( aim = "frictionAIM", unitSystem = "US" )
## [loadAIM]

# -----------------------------------------------------------------
# Set new Mach/Alt parameters
# -----------------------------------------------------------------
print ("Setting Mach & Altitude Values")
## [setInputs]
friction.input.Mach = [0.5, 1.5]

# Note: friction wants ft (defined in the AIM) - Automatic unit conversion to ft
friction.input.Altitude = [9000, 18200.0]*pyCAPS.Unit("m")

# Specify transition location applied to all surfaces
friction.input.BL_Transition = 0.1

# Specify lifting surfaces
friction.input.LiftSurface = {"Wing":{"BL_Transition":0.2},
                              "HTail":{},
                              "VTail":{}}

# Specify body of revolution surfaces
friction.input.RevolveSurface = {"Fuse":{"BL_Transition":0.05}}
## [setInputs]

# -----------------------------------------------------------------
# Get Output Data from Friction (execution is automatic just-in-time)
# -----------------------------------------------------------------
## [output]
Cdtotal = friction.output.CDtotal
CdForm  = friction.output.CDform
CdFric  = friction.output.CDfric
## [output]

print("Total drag =", Cdtotal )
print("Form drag =", CdForm)
print("Friction drag =", CdFric)

# Get all dynamic ouputs
for i in friction.dynout.keys():
    print(str(i) + " = " + str(friction.dynout[i].value))

assert(len(Cdtotal) == 2)
assert(len(CdForm) == 2)
assert(len(CdFric) == 2)

assert(abs(Cdtotal[0] - 0.012756893521617161) < 1e-4)
assert(abs(Cdtotal[1] - 0.011861060846919087) < 1e-4)

assert(abs(CdForm[0] - 0.0025723837005405996) < 1e-4)
assert(abs(CdForm[1] - 0.0024034739741570534) < 1e-4)

assert(abs(CdFric[0] - 0.010184509821076562) < 1e-4)
assert(abs(CdFric[1] - 0.009457586872762033) < 1e-4)
