import pyCAPS
import unittest
import os
import glob
import shutil

class TestSkeleton(unittest.TestCase):

#==============================================================================
    @classmethod
    def setUpClass(cls):
        # This will only be executed once at startup
        cls.problemName = "skeletonExample"
        cls.iProb = 1
        cls.cleanUp()

    @classmethod
    def tearDownClass(cls):
        # This is only executed once to tear down
        cls.cleanUp()

    @classmethod
    def cleanUp(cls):
        # Remove problem directories
        dirs = glob.glob( cls.problemName + '*')
        for dir in dirs:
            if os.path.isdir(dir):
                shutil.rmtree(dir)

#==============================================================================
    def test_defaultInput(self):
        
        # Instantiate our CAPS problem "capsProblem"
        capsProblem = pyCAPS.Problem(problemName=self.problemName+str(self.iProb),
                                     capsFile="case.csm",
                                     outLevel = 0); self.__class__.iProb += 1

        # Load our skeletal aim
        skel = capsProblem.analysis.create(aim = "skeletonAIM")

        # Check the default input values
        self.assertEqual(skel.input.InputVariable, False)
        self.assertEqual(skel.input.num, 8)
        self.assertIsNone(skel.input.Mach)
        self.assertIsNone(skel.input.Mesh_Format)
        self.assertEqual(skel.input.Table, {"Entry1":"Value1","Entry2":"Value2","Entry3":"Value3"})

#==============================================================================
    def test_setInput(self):
        
        # Instantiate our CAPS problem "capsProblem"
        capsProblem = pyCAPS.Problem(problemName=self.problemName+str(self.iProb),
                                     capsFile="case.csm",
                                     outLevel = 0); self.__class__.iProb += 1

        # Load our skeletal aim
        skel = capsProblem.analysis.create(aim = "skeletonAIM")
        
        # Try setting inputs
        skel.input.InputVariable = True
        skel.input.num = 16
        skel.input.Mach = 0.5
        skel.input.Mesh_Format = ["vtk","stl"]
        skel.input.Table = {"fee":42, "fi":2.0, "fum":"yay!"}
        
        # Check the inputs are properly set
        self.assertEqual(skel.input.InputVariable, True)
        self.assertEqual(skel.input.num, 16)
        self.assertEqual(skel.input.Mach, 0.5)
        self.assertEqual(skel.input.Mesh_Format, ["vtk","stl"])
        self.assertEqual(skel.input.Table, {"fee":42, "fi":2.0, "fum":"yay!"})

#==============================================================================
    def test_getOutput(self):

        # Instantiate our CAPS problem "capsProblem"
        capsProblem = pyCAPS.Problem(problemName=self.problemName+str(self.iProb),
                                     capsFile="case.csm",
                                     outLevel = 0); self.__class__.iProb += 1

        # Load our skeletal aim
        skel = capsProblem.analysis.create(aim = "skeletonAIM")

        # Set the input
        skel.input.num = 16.0
        
        # AIM autoExecutes
        
        # Check the output
        self.assertAlmostEqual(skel.output.sqrtNum, 16**0.5, 8)

if __name__ == '__main__':
    unittest.main()