import unittest

import os
import glob
import shutil

import sys

import pyCAPS

class TestFriction(unittest.TestCase):

    @classmethod
    def setUpClass(cls):

        cls.file = os.path.join("..","csmData","frictionWingTailFuselage.csm")
        cls.problemName = "workDir_friction"
        cls.iProb = 1
        cls.cleanUp()

        cls.capsProblem = pyCAPS.Problem(cls.problemName, capsFile=cls.file, outLevel=0)


    @classmethod
    def tearDownClass(cls):
        del cls.capsProblem
        cls.cleanUp()

    @classmethod
    def cleanUp(cls):

        # Remove analysis directories
        dirs = glob.glob( cls.problemName + '*')
        for dir in dirs:
            if os.path.isdir(dir):
                shutil.rmtree(dir)


#==============================================================================
    def test_setInput(self):

        friction = self.capsProblem.analysis.create(aim = "frictionAIM")

        friction.input.Mach = 0.5
        friction.input.Mach = [0.5, 1.5]

        friction.input.Altitude = 9000*pyCAPS.Unit("m")
        friction.input.Altitude = [9000, 18200.0]*pyCAPS.Unit("m")

        friction.input.BL_Transition = 0.2
        friction.input.TwTaw = 1.1

        friction.input.LiftSurface = {"Wing":{"BL_Transition":0.2},
                                      "HTail":{},
                                      "VTail":{}}

        friction.input.RevolveSurface = {"Fuse":{"BL_Transition":0.05}}

#==============================================================================
    # Test re-enter
    def test_reenter(self):
        
        ft = pyCAPS.Unit("ft")

        friction = self.capsProblem.analysis.create(aim = "frictionAIM")

        friction.input.Mach = 0.5
        friction.input.Altitude = 9000*pyCAPS.Unit("m")

        friction.input.LiftSurface = {"Wing":{"BL_Transition":0.2},
                                      "HTail":{},
                                      "VTail":{}}

        friction.input.RevolveSurface = {"Fuse":{"BL_Transition":0.05}}

        Cdtotal = friction.output.CDtotal
        Cdform  = friction.output.CDform
        Cdfric  = friction.output.CDfric
        self.assertAlmostEqual(Cdtotal, 0.0127568935216171, 4)
        self.assertAlmostEqual(Cdform , 0.0025723837005406, 4)
        self.assertAlmostEqual(Cdfric , 0.0101845098210765, 4)

        Wing = friction.dynout.Wing
        Fuse = friction.dynout.Fuse
        WingTrue = {'ReL': 6137401.31294, 'CDtotal': 0.006977890354803, 'CDform': 0.001791263145675, 'CDfric': 0.005186627209128, 'Swet': 2.206944450475e+02*ft**2, 'RefL': 4.235546091653e+00*ft, 'ToC': 0.1201844631294, 'FormFactor': 1.345361845656}
        FuseTrue = {'ReL': 39408980.35608, 'CDtotal': 0.003804664172345, 'CDform': 0.0002748517604039, 'CDfric': 0.003529812411941, 'Swet': 1.741518683686e+02*ft**2, 'RefL': 2.719694284473e+01*ft, 'ToC': 0.1231202056488, 'FormFactor': 1.077865826375}

        for key in WingTrue.keys():
            self.assertAlmostEqual(Wing[key], WingTrue[key], 4)
            self.assertAlmostEqual(Fuse[key], FuseTrue[key], 4)

        friction.input.Mach = [0.5, 1.5]
        friction.input.Altitude = [9000, 18200.0]*pyCAPS.Unit("m")

        Cdtotal = friction.output.CDtotal
        self.assertEqual(len(Cdtotal), 2)
        self.assertAlmostEqual(Cdtotal[0], 0.012756893521617161, 4)
        self.assertAlmostEqual(Cdtotal[1], 0.011861060846919087, 4)

        area = self.capsProblem.geometry.despmtr.area
        self.capsProblem.geometry.despmtr.area = 20

        Cdtotal = friction.output.CDtotal
        self.assertEqual(len(Cdtotal), 2)
        self.assertAlmostEqual(Cdtotal[0], 0.02278401000407973, 4)
        self.assertAlmostEqual(Cdtotal[1], 0.02113273301572688, 4)

        self.capsProblem.geometry.despmtr.area = area

#==============================================================================
    # Test Lift only
    def test_LiftSurface(self):

        friction = self.capsProblem.analysis.create(aim = "frictionAIM")

        friction.input.Mach = 0.5
        friction.input.Altitude = 9000*pyCAPS.Unit("m")

        friction.input.LiftSurface = {"Wing":{"BL_Transition":0.2},
                                      "HTail":{},
                                      "VTail":{}}

        Cdtotal = friction.output.CDtotal
        self.assertAlmostEqual(Cdtotal, 0.008952229349271857, 4)

#==============================================================================
    # Test Lift only
    def test_RevolveSurface(self):

        friction = self.capsProblem.analysis.create(aim = "frictionAIM")

        friction.input.Mach = 0.5
        friction.input.Altitude = 9000*pyCAPS.Unit("m")

        friction.input.RevolveSurface = {"Fuse":{"BL_Transition":0.05}}

        Cdtotal = friction.output.CDtotal
        self.assertAlmostEqual(Cdtotal, 0.0038046641723453066, 4)


#==============================================================================
    # Test SI units
    def test_SI(self):
        
        ft = pyCAPS.Unit("ft")

        friction = self.capsProblem.analysis.create(aim = "frictionAIM", unitSystem = "SI")

        friction.input.Mach = 0.5
        friction.input.Altitude = 9000*pyCAPS.Unit("m")

        friction.input.LiftSurface = {"Wing":{"BL_Transition":0.2},
                                      "HTail":{},
                                      "VTail":{}}

        friction.input.RevolveSurface = {"Fuse":{"BL_Transition":0.05}}

        Cdtotal = friction.output.CDtotal
        Cdform  = friction.output.CDform
        Cdfric  = friction.output.CDfric
        self.assertAlmostEqual(Cdtotal, 0.0127568935216171, 4)
        self.assertAlmostEqual(Cdform , 0.0025723837005406, 4)
        self.assertAlmostEqual(Cdfric , 0.0101845098210765, 4)

        Wing = friction.dynout.Wing
        Fuse = friction.dynout.Fuse
        WingTrue = {'ReL': 6137382.405053, 'CDtotal': 0.006977890354803, 'CDform': 0.001791263145675, 'CDfric': 0.005186627209128, 'Swet': 2.206944450475e+02*ft**2, 'RefL': 4.235546091653e+00*ft, 'ToC': 0.1201844631294, 'FormFactor': 1.345361845656}
        FuseTrue = {'ReL': 39408858.94629, 'CDtotal': 0.003804664172345, 'CDform': 0.0002748517604039, 'CDfric': 0.003529812411941, 'Swet': 1.741518683686e+02*ft**2, 'RefL': 2.719694284473e+01*ft, 'ToC': 0.1231202056488, 'FormFactor': 1.077865826375}

        for key in WingTrue.keys():
            self.assertAlmostEqual(Wing[key], WingTrue[key], 4)
            self.assertAlmostEqual(Fuse[key], FuseTrue[key], 4)
            
if __name__ == '__main__':
    unittest.main()
