/*
 ************************************************************************
 *                                                                      *
 * udfFlend3 -- create a flend on an attributed Body                    *
 *                                                                      *
 *            Written by John Dannenhoffer @ Syracuse University        *
 *            Patterned after code written by Bob Haimes  @ MIT         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2011/2026  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

//$$$#define DEBUG 1
#define NOG1

#define NUMUDPARGS       5
#define NUMUDPINPUTBODYS 1
#include "udpUtilities.h"

/* shorthands for accessing argument values and velocities */
#define ATTR1IN(IUDP)  ((char   *) (udps[IUDP].arg[0].val))
#define ATTR2IN(IUDP)  ((char   *) (udps[IUDP].arg[1].val))
#define FRAC1(  IUDP)  ((double *) (udps[IUDP].arg[2].val))[0]
#define FRAC2(  IUDP)  ((double *) (udps[IUDP].arg[3].val))[0]
#define ATTROUT(IUDP)  ((char   *) (udps[IUDP].arg[4].val))

/* data about possible arguments */
static char  *argNames[NUMUDPARGS] = {"attr1in",  "attr2in",  "frac1",  "frac2",  "attrout",  };
static int    argTypes[NUMUDPARGS] = {ATTRSTRING, ATTRSTRING, ATTRREAL, ATTRREAL, ATTRSTRING, };
static int    argIdefs[NUMUDPARGS] = {0,          0,          0,        0,        0,          };
static double argDdefs[NUMUDPARGS] = {0.,         0.,         0.,       0.,       0.,         };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

#include "OpenCSM.h"
#include "mateContours.h"
#include <assert.h>

/* neural net structure */
typedef struct {
    int     ninput;           /* number of nodes in the input layer */
    int     nhidden;          /* number of nodes in each hidden layer */
    int     noutput;          /* number of nodes in te output layer */

    double  *weight0;         /* weights for input scaling */
    double  *weight1;         /* weights between input   and hidden1 */
    double  *weight2;         /* weights between hidden1 and hidden2 */
    double  *weight3;         /* weights between hidden2 and output */
    double  *weight4;         /* weights for output scaling */

    double  *bias0;           /* biases for input scaling */
    double  *bias1;           /* biases between input   and hidden1 */
    double  *bias2;           /* biases between hidden1 and hidden2 */
    double  *bias3;           /* biases between hidden2 and output */
    double  *bias4;           /* biases for output scaling */
} nnet_T;

/* prototype of unpublished EGADS function */
extern int EG_relPosTs(ego ecurve, int npnt, /*@null@*/ const double newS[],
                       double newT[], double newXYZ[]);

/* prototypes of routine defined below */
static int setupNodeInfo(ego ebody, char attr1[], char attr2[], int nodeCount[], double nodeInfo[]);
#ifndef NOG1
static int getXYZforEdge(ego ebody, ego eedge, char attr[], double t, double xyz[]);
#endif
static int getXYZforEntity(ego ebody, int iedge, int isens, char attr[], double nodeInfo[], double xyz[]);
static int makeFlendCurve(ego context, nnet_T *nn, double xyz1[], double xyz2[], double frac1, double frac2, ego *newCurve);
static void xform(double m[], double x[]);

static int  nnet_make(nnet_T **nnet);
static int  nnet_eval(nnet_T *nnet, double inputs[], double outputs[]);
static int  nnet_free(nnet_T *nnet);
static double sigmoid(double x);
#ifndef NOG1
static int makeTangent(const ego ebody, nnet_T *nn, /*@null@*/ ego bottom,
                       char *attrb, double fb, /*@null@*/ ego top, char *attrt,
                       double ft, /*@null@*/ double *nodexyz, ego *esurf);
#endif

/* message (to be shared by all functions) */
static char message[1024];


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, *senses, periodic, attrType, attrLen;
    int     ncontour, *nent=NULL, *contr1=NULL, *contr2=NULL;
    int     ipair, icontour, ient, iedge1, isens1, iedge2, isens2;
    int     count1, count2, nfoo, ifoo;
    int     nnode, iedge, nedge, nface, brchattr[2], bodyattr[2];
    int     ibody, ntemp = 0, sense[4], fillStyle;
    int     nlist=0, iface, attrs[2], inode;
    int     *nodeCount=NULL;
    CINT    *tempIlist;
    double  xyz0[18], xyz1[18], xyz2[18], xyz3[18], data[18];
    double  value, dot, trange[4], uvrange[4];
    double  *nodeInfo=NULL;
    CDOUBLE *tempRlist;
    char    tempstr[MAX_STR_LEN];
    CCHAR   *tempClist;
    ego     context, eref, *ebodys, *echilds;
    ego     *enodes=NULL, *eedges=NULL, *efaces=NULL, *efoo=NULL, *faceList=NULL;
    ego     etemp[4], enode, eedge, ecurve, eloop, esurface, eedge01, eedge23;
    udp_T   *udps = *Udps;
    nnet_T  *nn=NULL;
    void    *modl;

#ifdef DEBUG
    int     i, j, k;
#endif

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("attr1in(0) = %s\n", ATTR1IN(0));
    printf("attr2in(0) = %s\n", ATTR2IN(0));
    printf("frac1(  0) = %f\n", FRAC1(  0));
    printf("frac2(  0) = %f\n", FRAC2(  0));
    printf("attrout(0) = %s\n", ATTROUT(0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    message[0] = '\0';

    /* get context and pointer to the OpenCSM MODL */
    status = EG_getContext(emodel, &context);
    CHECK_STATUS(EG_getContext);

    status = EG_getUserPointer(context, (void**)(&(modl)));
    CHECK_STATUS(EG_getUserPointer);

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 1024, "expecting a Model\n");
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 1024, "expecting Model to contain one Body (not %d)\n", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* verify that the given attributes are different */
    if (strcmp(ATTR1IN(0), ATTR2IN(0)) == 0) {
        snprintf(message, 1024, "attr1 and attr2 are the same\n");
        status = EGADS_RANGERR;
        goto cleanup;
    }

    /* make sure input is a SheetBody or SolidBody */
    status = EG_getTopology(ebodys[0], &eref, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    if (mtype != FACEBODY && mtype != SHEETBODY && mtype != SOLIDBODY) {
        snprintf(message, 1024, "Body is not a SheetBody or SolidBody\n");
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    printf("attr1in(%d) = %s\n", numUdp, ATTR1IN(numUdp));
    printf("attr2in(%d) = %s\n", numUdp, ATTR2IN(numUdp));
    printf("frac1(  %d) = %f\n", numUdp, FRAC1(  numUdp));
    printf("frac2(  %d) = %f\n", numUdp, FRAC2(  numUdp));
    printf("attrout(%d) = %s\n", numUdp, ATTROUT(numUdp));
#endif

    /* summarize Edges and Faces with ATTR1IN and ATTR2IN */
#ifdef DEBUG
    status = EG_getBodyTopos(ebodys[0], NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(ebodys[0], NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(eedges);
    SPLINT_CHECK_FOR_NULL(efaces);

    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_attributeRet(eedges[iedge], ATTR1IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            printf("Edge %5d has attribute %s\n", iedge+1, ATTR1IN(0));
        }
    }

    for (iface = 0; iface < nface; iface++) {
        status = EG_attributeRet(efaces[iface], ATTR1IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            printf("Face %5d has attribute %s\n", iface+1, ATTR1IN(0));
        }
    }

    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_attributeRet(eedges[iedge], ATTR2IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            printf("Edge %5d has attribute %s\n", iedge+1, ATTR2IN(0));
        }
    }

    for (iface = 0; iface < nface; iface++) {
        status = EG_attributeRet(efaces[iface], ATTR2IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            printf("Face %5d has attribute %s\n", iface+1, ATTR2IN(0));
        }
    }

    EG_free(eedges);   eedges = NULL;
    EG_free(efaces);   efaces = NULL;
#endif

    /* make the neural net based upon data in a nnet.dat file */
    status = nnet_make(&nn);
    CHECK_STATUS(nnet_make);

    SPLINT_CHECK_FOR_NULL(nn);

    /* add Nodes to the input Body so that the Loops with the given
       attributes have matching Nodes (which may involve splitting
       Edges)*/
    status = EG_mateContours(ebodys[0], ATTR1IN(0), ATTR2IN(0),
                             ebody, &ncontour, &nent, &contr1, &contr2);
    CHECK_STATUS(EG_mateContours);

#ifdef DEBUG
    printf("*body after EG_mateContours:\n");
    ocsmPrintBodyTopo(*ebody);
#endif

    SPLINT_CHECK_FOR_NULL(*ebody);
    SPLINT_CHECK_FOR_NULL(nent  );
    SPLINT_CHECK_FOR_NULL(contr1);
    SPLINT_CHECK_FOR_NULL(contr2);

    /* get tangency information at all Nodes associated with
       Edges that have either ATR1IN or ATTR2IN */
    status = EG_getBodyTopos(*ebody, NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    EG_free(enodes);   enodes = NULL;

    MALLOC(nodeCount, int,      nnode);
    MALLOC(nodeInfo,  double, 9*nnode);

    for (inode = 0; inode < nnode; inode++) {
        nodeCount[  inode  ] = 0;
        nodeInfo[ 9*inode  ] = 0;
        nodeInfo[ 9*inode+1] = 0;
        nodeInfo[ 9*inode+2] = 0;
        nodeInfo[ 9*inode+3] = 0;
        nodeInfo[ 9*inode+4] = 0;
        nodeInfo[ 9*inode+5] = 0;
        nodeInfo[ 9*inode+6] = 0;
        nodeInfo[ 9*inode+7] = 0;
        nodeInfo[ 9*inode+8] = 0;
    }

    status = setupNodeInfo(*ebody, ATTR1IN(0), ATTR2IN(0), nodeCount, nodeInfo);
    CHECK_STATUS(setupNodeInfo);


    /* print the matching Edges/Nodes */
#ifdef DEBUG
    printf("ncontour=%d\n", ncontour);
    k = 0;
    for (i = 0; i < ncontour; i++) {
        for (j = 0; j < nent[i]; j++) {
            printf("i=%3d, j=%3d, contr1=%5d,%5d contr2=%5d,%5d\n", i, j,
                   contr1[2*k], contr1[2*k+1], contr2[2*k], contr2[2*k+1]);
            k++;
        }
    }
#endif

    /* go to cleanup here to see the results of EG_mateContours (before
       the flend Faces are added */
//$$$    goto cleanup;

    /* get the Nodes, Edges, and Faces associated with the new Body */
    SPLINT_CHECK_FOR_NULL(*ebody);

    status = EG_getBodyTopos(*ebody, NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(*ebody, NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(*ebody, NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(enodes);
    SPLINT_CHECK_FOR_NULL(eedges);
    SPLINT_CHECK_FOR_NULL(efaces);

    /* initialize the attributes that will be put on the new Faces */
    status = ocsmEvalExpr(modl, "@nbody", &value, &dot, tempstr);
    CHECK_STATUS(ocsmEvalExpr);

    ibody = value + 1.01;

    brchattr[0] = -1;                   // fixed in buildPrimitive because _markFaces_ is not set
    bodyattr[0] = ibody;

    /* any Edge that has both Faces associated with ATTR1IN should also get ATTR1IN.
       same for ATTR2IN */
    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_getBodyTopos(*ebody, eedges[iedge], FACE, &nfoo, &efoo);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(efoo);

        count1 = 0;
        count2 = 0;

        for (ifoo = 0; ifoo < nfoo; ifoo++) {
            status = EG_attributeRet(efoo[ifoo], ATTR1IN(0), &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                count1++;
            }

            status = EG_attributeRet(efoo[ifoo], ATTR2IN(0), &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) {
                count2++;
            }
        }

        if (count1 >= nfoo) {
            status = EG_attributeAdd(eedges[iedge], ATTR1IN(0), ATTRINT, 1,
                                     &count1, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }

        if (count2 >= nfoo) {
            status = EG_attributeAdd(eedges[iedge], ATTR2IN(0), ATTRINT, 1,
                                     &count2, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }

        EG_free(efoo);   efoo = NULL;
    }

    /* allocate a list of the Faces to sew together */
    nlist = nface;
    for (icontour = 0; icontour < ncontour; icontour++) {
        nlist += nent[icontour];
    }

    MALLOC(faceList, ego, nlist);

    /* build a list of the Faces that are not associated with ATTR1IN or ATTR2IN */
    nlist = 0;
    for (iface = 0; iface < nface; iface++) {
        status = EG_attributeRet(efaces[iface], ATTR1IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        status = EG_attributeRet(efaces[iface], ATTR2IN(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) continue;

        faceList[nlist++] = efaces[iface];
    }

    /* loop through each set of contours (eg, separate flends on the
       two sides of the aircraft) */
    ipair  = 0;
    for (icontour = 0; icontour < ncontour; icontour++) {

        /* make a Face for each entity pair */
        for (ient = 0; ient < nent[icontour]; ient++) {
            iedge1 = contr1[2*ipair  ];
            isens1 = contr1[2*ipair+1];
            iedge2 = contr2[2*ipair  ];
            isens2 = contr2[2*ipair+1];

            /* get the Nodes associated with this entity in loop1 */
            if (isens1 == 0) {
                status = EG_objectBodyTopo(*ebody, NODE, iedge1, &enode);
                CHECK_STATUS(EG_objectBodyTopo);

                etemp[3] = enode;
                etemp[0] = enode;
            } else {
                status = EG_objectBodyTopo(*ebody, EDGE, iedge1, &eedge);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                if (isens1 > 0) {
                    etemp[3] = echilds[0];
                    etemp[0] = echilds[1];
                } else {
                    etemp[0] = echilds[0];
                    etemp[3] = echilds[1];
                }
            }

            /* get the Nodes associated with this entity in loop2 */
            if (isens2 == 0) {
                status = EG_objectBodyTopo(*ebody, NODE, iedge2, &enode);
                CHECK_STATUS(EG_objectBodyTopo);

                etemp[1] = enode;
                etemp[2] = enode;
            } else {
                status = EG_objectBodyTopo(*ebody, EDGE, iedge2, &eedge);
                CHECK_STATUS(EG_objectBodyTopo);

                status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                if (isens2 < 0) {
                    etemp[1] = echilds[0];
                    etemp[2] = echilds[1];
                } else {
                    etemp[2] = echilds[0];
                    etemp[1] = echilds[1];
                }
            }

            /* set up the control points for the curve between Nodes 0 and 1 */
            status = getXYZforEntity(*ebody, iedge1, +isens1, ATTR1IN(0), nodeInfo, xyz0);
            CHECK_STATUS(getXYZforEntity);

            status = getXYZforEntity(*ebody, iedge2, +isens2, ATTR2IN(0), nodeInfo, xyz1);
            CHECK_STATUS(getXYZforEntity);

#ifdef DEBUG
            printf("\nxyz0:  %10.5f %10.5f %10.5f\n", xyz0[0], xyz0[1], xyz0[2]);
            printf(  "       %10.5f %10.5f %10.5f\n", xyz0[3], xyz0[4], xyz0[5]);
            printf(  "xyz1:  %10.5f %10.5f %10.5f\n", xyz1[0], xyz1[1], xyz1[2]);
            printf(  "       %10.5f %10.5f %10.5f\n", xyz1[3], xyz1[4], xyz1[5]);
            printf("iedge1=%3d, isens1=%3d, iedge2=%3d, isens2=%3d\n", iedge1, isens1, iedge2, isens2);
#endif

            status = makeFlendCurve(context, nn, xyz0, xyz1, FRAC1(0), FRAC2(0), &ecurve);
            CHECK_STATUS(makeFlendCurve);

            status = EG_getRange(ecurve, trange, &periodic);
            CHECK_STATUS(EG_getRange);

            status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                                     2, &etemp[0], NULL, &eedge01);
            CHECK_STATUS(EG_makeTopology);

            /* set up the control points for the curve between Nodes 2 and 3 */
            status = getXYZforEntity(*ebody, iedge2, -isens2, ATTR2IN(0), nodeInfo, xyz2);
            CHECK_STATUS(getXYZforEntity);

            status = getXYZforEntity(*ebody, iedge1, -isens1, ATTR1IN(0), nodeInfo, xyz3);
            CHECK_STATUS(getXYZforEntity);

#ifdef DEBUG
            printf("\nxyz2:  %10.5f %10.5f %10.5f\n", xyz2[0], xyz2[1], xyz2[2]);
            printf(  "       %10.5f %10.5f %10.5f\n", xyz2[3], xyz2[4], xyz2[5]);
            printf(  "xyz3:  %10.5f %10.5f %10.5f\n", xyz3[0], xyz3[1], xyz3[2]);
            printf(  "       %10.5f %10.5f %10.5f\n", xyz3[3], xyz3[4], xyz3[5]);
#endif

            status = makeFlendCurve(context, nn, xyz2, xyz3, FRAC2(0), FRAC1(0), &ecurve);
            CHECK_STATUS(makeFlendCurve);

            status = EG_getRange(ecurve, trange, &periodic);
            CHECK_STATUS(EG_getRange);

            status = EG_makeTopology(context, ecurve, EDGE, TWONODE, trange,
                                     2, &etemp[2], NULL, &eedge23);
            CHECK_STATUS(EG_makeTopology);

            /* build the Loop */
            if (isens1 == 0) {
                ntemp = 3;

                etemp[0] = eedge01;
                sense[0] = SFORWARD;

                etemp[1] = eedges[iedge2-1];
                sense[1] = -isens2;

                etemp[2] = eedge23;
                sense[2] = SFORWARD;
            } else if (isens2 == 0) {
                ntemp = 3;

                etemp[0] = eedge23;
                sense[0] = SFORWARD;

                etemp[1] = eedges[iedge1-1];
                sense[1] = isens1;

                etemp[2] = eedge01;
                sense[2] = SFORWARD;
            } else {
                ntemp = 4;

                etemp[3] = eedges[iedge1-1];
                sense[3] = isens1;

                etemp[0] = eedge01;
                sense[0] = SFORWARD;

                etemp[1] = eedges[iedge2-1];
                sense[1] = -isens2;

                etemp[2] = eedge23;
                sense[2] = SFORWARD;
            }

            status = EG_makeTopology(context, NULL, LOOP, CLOSED, NULL,
                                     ntemp, etemp, sense, &eloop);
            CHECK_STATUS(EG_makeTopology);

            /* build the surface and flip its U if there is a Node
               in contour2 (to make the U's consistent with the
               other Faces in the flend */

            fillStyle = 0;

            /* note: fillStyle=1 looks better but cannot be applied
               when the top and/or bottom are lines.  if we want to
               use fillStyle=1, then a change will be need to be made
               in EG_isoCline to convert linear Bsplines to cubic
               Bsplines */
            status = EG_isoCline(eloop, fillStyle, 0.0, &esurface);
            CHECK_STATUS(EG_isoCline);

            if (isens2 == 0) {
                etemp[0] = esurface;

                status = EG_flipObject(etemp[0], &esurface);
                CHECK_STATUS(EG_flipObject);
            }

#ifndef NOG1
            /* adjust the control points to be G1 */
            if (isens1 == 0) {
                /* xyz0 == xyz3 */
                status = makeTangent(*ebody, nn, NULL, ATTR1IN(0), FRAC1(0),
                                     eedges[iedge2-1], ATTR2IN(0), FRAC2(0),
                                     xyz0, &esurface);
            } else if (isens2 == 0) {
                /* xyz1 == xyz2 */
                status = makeTangent(*ebody, nn, eedges[iedge1-1], ATTR1IN(0),
                                     FRAC1(0), NULL, ATTR2IN(0), FRAC2(0),
                                     xyz1, &esurface);
            } else {
                status = makeTangent(*ebody, nn, eedges[iedge1-1], ATTR1IN(0),
                                     FRAC1(0), eedges[iedge2-1], ATTR2IN(0),
                                     FRAC2(0), NULL, &esurface);
            }
            CHECK_STATUS(makeTangent);
#endif
          
            /* build the Face and attribute it for OpenCSM */
            status = EG_getRange(esurface, uvrange, &periodic);
            CHECK_STATUS(EG_getRange);

            status = EG_makeFace(esurface, SFORWARD, uvrange, &faceList[nlist++]);
            CHECK_STATUS(EG_makeFace);

            brchattr[1] = icontour * 1000 + ient;
            bodyattr[1] = icontour * 1000 + ient;

            status = EG_attributeAdd(faceList[nlist-1], "_brch", ATTRINT, 2,
                                     brchattr, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);

            status = EG_attributeAdd(faceList[nlist-1], "_body", ATTRINT, 2,
                                     bodyattr, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);

            if (strlen(ATTROUT(0)) > 0) {
                attrs[0] = icontour;
                attrs[1] = ient;

                status = EG_attributeAdd(faceList[nlist-1], ATTROUT(0), ATTRINT, 2,
                                         attrs, NULL, NULL);
                CHECK_STATUS(EG_attributeAdd);
            }

            /* get ready for next entity */
            ipair++;
        }
    }

    /* sew the Faces together and extract the only Body */
    status = EG_sewFaces(nlist, faceList, 1e-4, 0, &emodel);
    CHECK_STATUS(EG_sewFaces);

    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    if (nchild > 1) {
        printf("error: nchild=%d\n", nchild);
        exit(0);
    }

    /* update nnode, nedge, and nface */
    EG_free(enodes);
    EG_free(eedges);
    EG_free(efaces);

    status = EG_getBodyTopos(echilds[0], NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(echilds[0], NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(echilds[0], NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(enodes);
    SPLINT_CHECK_FOR_NULL(eedges);
    SPLINT_CHECK_FOR_NULL(efaces);

    /* loop through all Edges that are between newly-created flend Faces */
    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_getInfo(eedges[iedge], &oclass, &mtype, NULL, NULL, NULL);
        CHECK_STATUS(EG_getInfo);
        if (mtype == DEGENERATE) continue;

        status = EG_getBodyTopos(echilds[0], eedges[iedge], FACE, &nfoo, &efoo);
        CHECK_STATUS(EG_getBodyTopos);

        if (nfoo < 2) {
            EG_free(efoo);
            continue;
        }

        SPLINT_CHECK_FOR_NULL(efoo);

        status = EG_attributeRet(efoo[0], "_body", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);
        if (attrType != ATTRINT || tempIlist[0] != ibody) {
            EG_free(efoo);
            continue;
        }

        status = EG_attributeRet(efoo[1], "_body", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);
        if (attrType != ATTRINT || tempIlist[0] != ibody) {
            EG_free(efoo);
            continue;
        }

        EG_free(efoo);

        if (strlen(ATTROUT(0)) > 0) {
            status = EG_attributeAdd(eedges[iedge], ATTROUT(0), ATTRINT, 1,
                                     &iedge, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }
    }

    /* remove arrays created by EG_mateContours */
    EG_free(nent  );
    EG_free(contr1);
    EG_free(contr2);

    *ebody = echilds[0];
    /* set the output value */

    /* return the new Body */
    udps[numUdp].ebody = *ebody;

cleanup:
    FREE(nodeCount);
    FREE(nodeInfo );

    if (enodes != NULL) EG_free(enodes);
    if (eedges != NULL) EG_free(eedges);
    if (efaces != NULL) EG_free(efaces);

    FREE(faceList);

    if (nn != NULL) (void) nnet_free(nn);

    if (strlen(message) > 0) {
        MALLOC(*string, char, 1024);
        strncpy(*string, message, 1023);
    } else if (status != EGADS_SUCCESS) {
        *string = udpErrorStr(status);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
   /*@unused@*/int    npnt,             /* (in)  number of points */
   /*@unused@*/int    entType,          /* (in)  OCSM entity type */
   /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
   /*@unused@*/double vels[])           /* (out) velocities */
{
    int iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        return EGADS_NOTMODEL;
    }

    /* this routine is not written yet */
    return EGADS_NOLOAD;
}


/*
 ************************************************************************
 *                                                                      *
 *   setupNodeInfo - set up XYZ, norm and tang at each Node             *
 *                                                                      *
 ************************************************************************
 */

static int
setupNodeInfo(ego    ebody,             /* (in)  Body */
              char   attr1[],           /* (in)  first  attribute */
              char   attr2[],           /* (in)  second attribute */
              int    nodeCount[],       /* (both) number of Edges incident to Node */
              double nodeInfo[])        /* (both) xyz, norm, and tang at each Node */
{
    int status = EGADS_SUCCESS;

    int     nloop, iloop, nface, iface, nedge, iedge, nnode, inode;
    int     oclass, mtype, attrType, attrLen, *senses;
    int     ntempEdge, ntempNode, itemp, mtypeFace, i;
    int     *sensesLoop, *sensesEdges;
    CINT    *tempIlist;
    double  dataFace[18], dataEdge[18], uv_best[2], xyz_best[3];
    double  tangx, tangy, tangz, tang, normx, normy, normz, norm;
    double  tempx, tempy, tempz;
    double  trange[2], uvrange[4];
    CDOUBLE *tempRlist;
    CCHAR   *tempClist;
    ego     *etempEdges, *etempNodes, *eloops, *enodes, *eedges, *efaces, eref;

    ROUTINE (setupNodeInfo);

    /* --------------------------------------------------------------- */

    status = EG_getBodyTopos(ebody, NULL, NODE, &nnode, &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    EG_free(enodes);

    status = EG_getBodyTopos(ebody, NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(eedges);

    /* loop through the Edges with attr1 that are adjacent to a Face
       without attr1 and deposit at its Nodes the normal and tangential
       directions */
    for (iedge = 0; iedge < nedge; iedge++) {
#ifdef DEBUG
        printf("looking at iedge=%d\n", EG_indexBodyTopo(ebody, eedges[iedge]));
#endif

        status = EG_getInfo(eedges[iedge], &oclass, &mtype, NULL, NULL, NULL);
        CHECK_STATUS(EG_getInfo);
        if (mtype == DEGENERATE) continue;

        status = EG_attributeRet(eedges[iedge], attr1, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) continue;

        status = EG_getBodyTopos(ebody, eedges[iedge], FACE, &nface, &efaces);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(efaces);

        for (iface = 0; iface < nface; iface++) {
            status = EG_attributeRet(efaces[iface], attr1, &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) continue;

#ifdef DEBUG
            printf("     found iface=%d\n", EG_indexBodyTopo(ebody, efaces[iface]));
#endif
            status = EG_getTopology(efaces[iface], &eref, &oclass, &mtypeFace,
                                    uvrange, &nloop, &eloops, &sensesLoop);
            CHECK_STATUS(EG_getTopology);

            for (iloop = 0; iloop < nloop; iloop++) {
                status = EG_getTopology(eloops[iloop], &eref, &oclass, &mtype,
                                        uvrange, &ntempEdge, &etempEdges, &sensesEdges);
                CHECK_STATUS(EG_getTopology);

                for (itemp = 0; itemp < ntempEdge; itemp++) {
                    if (etempEdges[itemp] != eedges[iedge]) continue;

                    status = EG_getTopology(etempEdges[itemp], &eref, &oclass, &mtype,
                                            trange, &ntempNode, &etempNodes, &senses);
                    CHECK_STATUS(EG_getTopology);

                    if (mtype == DEGENERATE) continue;

                    for (i = 0; i < 2; i++) {
                        inode = EG_indexBodyTopo(ebody, etempNodes[i]) - 1;

                        status = EG_evaluate(etempEdges[itemp], &trange[i], dataEdge);
                        CHECK_STATUS(EG_evaluate);

                        status = EG_invEvaluate(efaces[iface], dataEdge, uv_best, xyz_best);
                        CHECK_STATUS(EG_invEvaluate);

                        status = EG_evaluate(efaces[iface], uv_best, dataFace);
                        CHECK_STATUS(EG_evaluate);

                        tangx = sensesEdges[itemp] * dataEdge[3];
                        tangy = sensesEdges[itemp] * dataEdge[4];
                        tangz = sensesEdges[itemp] * dataEdge[5];
                        tang  = sqrt(tangx * tangx + tangy * tangy + tangz * tangz);

                        normx = dataFace[4] * dataFace[8] - dataFace[5] * dataFace[7];
                        normy = dataFace[5] * dataFace[6] - dataFace[3] * dataFace[8];
                        normz = dataFace[3] * dataFace[7] - dataFace[4] * dataFace[6];
                        norm  = mtypeFace * sqrt(normx * normx + normy * normy + normz * normz);
#ifdef DEBUG
                        printf("        at inode=%3d, tang=%10.5f %10.5f %10.5f, norm=%10.5f %10.5f %10.5f\n",
                               inode+1, tangx/tang, tangy/tang, tangz/tang, normx/norm, normy/norm, normz/norm);
#endif
                        nodeCount[  inode  ] ++;
                        nodeInfo[ 9*inode  ] = dataEdge[0];
                        nodeInfo[ 9*inode+1] = dataEdge[1];
                        nodeInfo[ 9*inode+2] = dataEdge[2];
                        nodeInfo[ 9*inode+3] += normx / norm;
                        nodeInfo[ 9*inode+4] += normy / norm;
                        nodeInfo[ 9*inode+5] += normz / norm;
                        nodeInfo[ 9*inode+6] += tangx / tang;
                        nodeInfo[ 9*inode+7] += tangy / tang;
                        nodeInfo[ 9*inode+8] += tangz / tang;
#ifdef DBUG
                        printf("                      nodeInfo=%10.5f %10.5f %10.5f   %10.5f %10.5f %10.5f   %10.5f %10.5f %10.5f\n",
                               nodeInfo[9*inode  ], nodeInfo[9*inode+1], nodeInfo[9*inode+2],
                               nodeInfo[9*inode+3], nodeInfo[9*inode+4], nodeInfo[9*inode+5],
                               nodeInfo[9*inode+6], nodeInfo[9*inode+7], nodeInfo[9*inode+8]);
#endif
                    }
                }
            }
        }

        EG_free(efaces);   efaces = NULL;
    }

    /* loop through the Edges with attr2 that are adjacent to a Face
       without attr2 and deposit at its Nodes the normal and tangential
       directions */
    for (iedge = 0; iedge < nedge; iedge++) {
#ifdef DEBUG
        printf("looking at iedge=%d\n", EG_indexBodyTopo(ebody, eedges[iedge]));
#endif

        status = EG_getInfo(eedges[iedge], &oclass, &mtype, NULL, NULL, NULL);
        CHECK_STATUS(EG_getInfo);
        if (mtype == DEGENERATE) continue;

        status = EG_attributeRet(eedges[iedge], attr2, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) continue;

        status = EG_getBodyTopos(ebody, eedges[iedge], FACE, &nface, &efaces);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(efaces);

        for (iface = 0; iface < nface; iface++) {
            status = EG_attributeRet(efaces[iface], attr2, &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == EGADS_SUCCESS) continue;

#ifdef DEBUG
            printf("     found iface=%d\n", EG_indexBodyTopo(ebody, efaces[iface]));
#endif
            status = EG_getTopology(efaces[iface], &eref, &oclass, &mtypeFace,
                                    uvrange, &nloop, &eloops, &sensesLoop);
            CHECK_STATUS(EG_getTopology);

            for (iloop = 0; iloop < nloop; iloop++) {
                status = EG_getTopology(eloops[iloop], &eref, &oclass, &mtype,
                                        uvrange, &ntempEdge, &etempEdges, &sensesEdges);
                CHECK_STATUS(EG_getTopology);

                for (itemp = 0; itemp < ntempEdge; itemp++) {
                    if (etempEdges[itemp] != eedges[iedge]) continue;

                    status = EG_getTopology(etempEdges[itemp], &eref, &oclass, &mtype,
                                            trange, &ntempNode, &etempNodes, &senses);
                    CHECK_STATUS(EG_getTopology);

                    if (mtype == DEGENERATE) continue;

                    for (i = 0; i < 2; i++) {
                        inode = EG_indexBodyTopo(ebody, etempNodes[i]) - 1;

                        status = EG_evaluate(etempEdges[itemp], &trange[i], dataEdge);
                        CHECK_STATUS(EG_evaluate);

                        status = EG_invEvaluate(efaces[iface], dataEdge, uv_best, xyz_best);
                        CHECK_STATUS(EG_invEvaluate);

                        status = EG_evaluate(efaces[iface], uv_best, dataFace);
                        CHECK_STATUS(EG_evaluate);

                        tangx = sensesEdges[itemp] * dataEdge[3];
                        tangy = sensesEdges[itemp] * dataEdge[4];
                        tangz = sensesEdges[itemp] * dataEdge[5];
                        tang  = sqrt(tangx * tangx + tangy * tangy + tangz * tangz);

                        normx = dataFace[4] * dataFace[8] - dataFace[5] * dataFace[7];
                        normy = dataFace[5] * dataFace[6] - dataFace[3] * dataFace[8];
                        normz = dataFace[3] * dataFace[7] - dataFace[4] * dataFace[6];
                        norm  = mtypeFace * sqrt(normx * normx + normy * normy + normz * normz);
#ifdef DEBUG
                        printf("        at inode=%3d, tang=%10.5f %10.5f %10.5f, norm=%10.5f %10.5f %10.5f\n",
                               inode+1, tangx/tang, tangy/tang, tangz/tang, normx/norm, normy/norm, normz/norm);
#endif
                        nodeCount[  inode  ] ++;
                        nodeInfo[ 9*inode  ] = dataEdge[0];
                        nodeInfo[ 9*inode+1] = dataEdge[1];
                        nodeInfo[ 9*inode+2] = dataEdge[2];
                        nodeInfo[ 9*inode+3] += normx / norm;
                        nodeInfo[ 9*inode+4] += normy / norm;
                        nodeInfo[ 9*inode+5] += normz / norm;
                        nodeInfo[ 9*inode+6] += tangx / tang;
                        nodeInfo[ 9*inode+7] += tangy / tang;
                        nodeInfo[ 9*inode+8] += tangz / tang;
                    }
                }
            }
        }

        EG_free(efaces);   efaces = NULL;
    }

    EG_free(eedges);   eedges = NULL;

    /* compute direction of flend curve by taking tang.cross.norm */
    for (inode = 0; inode < nnode; inode++) {
        tempx = nodeInfo[9*inode+5] * nodeInfo[9*inode+7] - nodeInfo[9*inode+4] * nodeInfo[9*inode+8];
        tempy = nodeInfo[9*inode+3] * nodeInfo[9*inode+8] - nodeInfo[9*inode+5] * nodeInfo[9*inode+6];
        tempz = nodeInfo[9*inode+4] * nodeInfo[9*inode+6] - nodeInfo[9*inode+3] * nodeInfo[9*inode+7];

#ifdef DEBUG
        printf("%3d count=%3d, xyz=%10.5f %10.5f %10.5f, norm=%10.5f %10.5f %10.5f, tang=%10.5f %10.5f %10.5f, t.x.n=%10.5f %10.5f %10.5f\n",
               inode+1, nodeCount[inode],
               nodeInfo[9*inode  ], nodeInfo[9*inode+1], nodeInfo[9*inode+2],
               nodeInfo[9*inode+3], nodeInfo[9*inode+4], nodeInfo[9*inode+5],
               nodeInfo[9*inode+6], nodeInfo[9*inode+7], nodeInfo[9*inode+8], tempx, tempy, tempz);
#endif

        nodeInfo[9*inode+3] = tempx;
        nodeInfo[9*inode+4] = tempy;
        nodeInfo[9*inode+5] = tempz;
    }

    for (inode = 0; inode < nnode; inode++) {
        if (nodeCount[inode] == 0) {

        } else if (nodeCount[inode] == 2) {
            nodeInfo[9*inode+3] /= 4;
            nodeInfo[9*inode+4] /= 4;
            nodeInfo[9*inode+5] /= 4;
        } else {
            printf("trouble here, inode=%d\n", inode);
            exit(0);
        }
    }

    status = EGADS_SUCCESS;

cleanup:
    return status;
}


#ifndef NOG1
/*
 ************************************************************************
 *                                                                      *
 *   getXYZforEdge - find xyz for Edge at given t                       *
 *                                                                      *
 ************************************************************************
 */

static int
getXYZforEdge(ego    ebody,             /* (in)  Body */
              ego    eedge,             /* (in)  Edge ego */
              char   attr[],            /* (in)  attribute associated with Edge */
              double t,                 /* (in)  point at which to evaluate */
              double xyz[])             /* (out) xyz, d(xyz)/d(t) */
{
    int status = EGADS_SUCCESS;

    int      nloop, iloop, nface, iface, nedge, iedge, isens;
    int      oclass, mtype, *senses, attrType, attrLen;
    CINT     *tempIlist;
    double   uv[2], data[18];
    double   normx, normy, normz, tangx, tangy, tangz, dirnx, dirny, dirnz, temp;
    CDOUBLE  *tempRlist;
    CCHAR    *tempClist;
    ego      eface, eref, *eloops, *eedges, *efaces=NULL;

    ROUTINE (getXYZforEdge);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("enter getXYZforEdge(iedge=%d, attr=%s)\n", EG_indexBodyTopo(ebody, eedge), attr);
#endif

    /* make sure the Edge has the specified attribute */
    status = EG_attributeRet(eedge, attr, &attrType, &attrLen,
                             &tempIlist, &tempRlist, &tempClist);
    CHECK_STATUS(EG_attributeRet);

    /* find a Face associated with eedge that does not have attr */
    status = EG_getBodyTopos(ebody, eedge, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(efaces);

    eface = NULL;
    for (iface = 0; iface < nface; iface++) {
        status = EG_attributeRet(efaces[iface], attr, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_NOTFOUND) {
            eface = efaces[iface];
            break;
        }
    }

    if (eface == NULL) {
        status = EGADS_NOTFOUND;
        goto cleanup;
    }

    /* get the Face normal */
    status = EG_getTopology(eface, &eref, &oclass, &mtype,
                            data, &nloop, &eloops, &senses);
    CHECK_STATUS(EG_getTopology);
    
    status = EG_getEdgeUV(eface, eedge, 0, t, uv);
    CHECK_STATUS(EG_getEdgeUV);

    status = EG_evaluate(eface, uv, data);
    CHECK_STATUS(EG_evaluate);

    normx = mtype * (data[4] * data[8] - data[5] * data[7]);
    normy = mtype * (data[5] * data[6] - data[3] * data[8]);
    normz = mtype * (data[3] * data[7] - data[4] * data[6]);

    /* determine if eface is to the left or right of eedge */
    isens = 0;
    for (iloop = 0; iloop < nloop; iloop++) {
        if (isens != 0) continue;
        
        status = EG_getTopology(eloops[iloop], &eref, &oclass, &mtype,
                                data, &nedge, &eedges, &senses);
        CHECK_STATUS(EG_getTopology);

        for (iedge = 0; iedge < nedge; iedge++) {
            if (eedge == eedges[iedge]) {
                isens = senses[iedge];
                break;
            }
        }
    }

    /* get the tangential direction associated with the Edge */
    status = EG_evaluate(eedge, &t, data);
    CHECK_STATUS(EG_evaluate);

    tangx = isens * data[3];
    tangy = isens * data[4];
    tangz = isens * data[5];

    /* compute direction of flend curve by taking tang.cross.norm */
    dirnx = normz * tangy - normy * tangz;
    dirny = normx * tangz - normz * tangx;
    dirnz = normy * tangx - normx * tangy;

    temp = sqrt(dirnx * dirnx + dirny * dirny + dirnz * dirnz);

    /* fill the output vector */
    xyz[0] = data[0];
    xyz[1] = data[1];
    xyz[2] = data[2];

    xyz[3] = dirnx / temp;
    xyz[4] = dirny / temp;
    xyz[5] = dirnz / temp;

cleanup:
    if (efaces != NULL) EG_free(efaces);

    return status;
}
#endif


/*
 ************************************************************************
 *                                                                      *
 *   getXYZforEntity - find xyz for Edge or Face at end of entity       *
 *                                                                      *
 ************************************************************************
 */

static int
getXYZforEntity(ego    ebody,           /* (in)  Body */
                int    iedge,           /* (in)  Edge or Node index (bias-1) */
                int    isens,           /* (in)  =0 for Node, otherwise Edge */
                char   attr[],          /* (in)  attribute associated with Edge */
                double nodeInfo[],      /* (in)  info about X, dXdn, and dXdT at Nodes */
                double xyz[])           /* (out) xyz, d(xyz)/d(t) */
{
    int status = EGADS_SUCCESS;

    int      inode, ntemp, nedge, jedge, nface, jface, count;
    int      oclass, mtype, *senses, attrType, attrLen;
    CINT     *tempIlist;
    double   trange[4], uv[2], data[18];
    double   xnorm1, ynorm1, znorm1, xnorm2, ynorm2, znorm2, temp, dot;
    CDOUBLE  *tempRlist;
    CCHAR    *tempClist;
    ego      enode, eedge, eref, *etemp, *eedges=NULL, *efaces=NULL;

    ROUTINE (getXYZforEntity);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("enter getXYZforEntity(iedge=%d, isens=%d, attr=%s)\n", iedge, isens, attr);
#endif

    /* make sure the Edge has the specified attribute */
    if (isens != 0) {
        status = EG_objectBodyTopo(ebody, EDGE, iedge, &eedge);
        CHECK_STATUS(EG_objectBodyTopo);

        status = EG_attributeRet(eedge, attr, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        CHECK_STATUS(EG_attributeRet);
    }

    /* find enode at beg of entity */
    if (isens < 0) {
        status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                trange, &ntemp, &etemp, &senses);
        CHECK_STATUS(EG_getTopology);

        enode = etemp[0];
    } else if (isens > 0) {
        status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                trange, &ntemp, &etemp, &senses);
        CHECK_STATUS(EG_getTopology);

        enode = etemp[1];
    } else {
        inode = iedge;

        status = EG_objectBodyTopo(ebody, NODE, inode, &enode);
        CHECK_STATUS(EG_objectBodyTopo);
    }
#ifdef DEBUG
    printf("inode=%d\n", EG_indexBodyTopo(ebody, enode));
#endif

    /* find count of Edge(s) that are not associated with attr and
       which are incident to enode */
    count = 0;

    status = EG_getBodyTopos(ebody, enode, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(eedges);

    for (jedge = 0; jedge < nedge; jedge++) {
        status = EG_attributeRet(eedges[jedge], attr, &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) {
            eedge = eedges[jedge];
            count ++;
        }
    }

    EG_free(eedges);   eedges = NULL;
#ifdef DEBUG
    printf("count=%d\n", count);
#endif

//$$$    int count_save = count;

    /* if there is one Edge, check to see if the normals on the two
       incident Faces are nearly parallel.  and if they are, change
       the count to 0 so that the flend curve is not forced to be
       parallel to the Edge */
    if (count == 1) {
        status = EG_getBodyTopos(ebody, enode, FACE, &nface, &efaces);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(efaces);

        xnorm1 = 0;
        ynorm1 = 0;
        znorm1 = 0;
        xnorm2 = 0;
        ynorm2 = 0;
        znorm2 = 0;

        for (jface = 0; jface < nface; jface++) {
            status = EG_attributeRet(efaces[jface], attr, &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status != EGADS_SUCCESS) {
                xnorm2 = xnorm1;
                ynorm2 = ynorm1;
                znorm2 = znorm1;

                status = EG_getEdgeUV(efaces[jface], enode, 0, 0, uv);
                CHECK_STATUS(EG_getEdgeUV);

                status = EG_evaluate(efaces[jface], uv, data);
                CHECK_STATUS(EG_evaluate);

                xnorm1 = data[4] * data[8] - data[5] * data[7];
                ynorm1 = data[5] * data[6] - data[3] * data[8];
                znorm1 = data[3] * data[7] - data[4] * data[6];
                temp   = sqrt(xnorm1 * xnorm1 + ynorm1 * ynorm1 + znorm1 * znorm1);
                xnorm1 /= temp;
                ynorm1 /= temp;
                znorm1 /= temp;
            }
        }

        dot = xnorm1 * xnorm2 + ynorm1 * ynorm2 + znorm1 * znorm2;
        if (dot > 0.985) {              // approximatey 10 deg
            count = 0;
#ifdef DEBUG
            printf("faces almost normal, so changing count=0\n");
#endif
        }

        EG_free(efaces);   efaces = NULL;
    }

    /* if there are no Edges to set the direction, find Faces that are not
       associated with attr and which are indicent to enode */
    if (count == 0) {
        inode = EG_indexBodyTopo(ebody, enode) - 1;

        xyz[0] = nodeInfo[9*inode  ];
        xyz[1] = nodeInfo[9*inode+1];
        xyz[2] = nodeInfo[9*inode+2];

        /* vector out of Face is tang.cross.norm (assuming Face is on
           left of Edge, which was taken care of in setupNodeInfo) */
        xyz[3] = nodeInfo[9*inode+3];
        xyz[4] = nodeInfo[9*inode+4];
        xyz[5] = nodeInfo[9*inode+5];

    /* if there was one Edge incident to enode and not associated
       with attr, evaluate the Edge at enode */
    } else if (count == 1) {
        SPLINT_CHECK_FOR_NULL(eedge);

        status = EG_getTopology(eedge, &eref, &oclass, &mtype,
                                trange, &ntemp, &etemp, &senses);
        CHECK_STATUS(EG_getTopology);

#ifdef DEBUG
        printf("enode   =%3d (%llx)\n", EG_indexBodyTopo(ebody, enode   ), (long long)enode   );
        printf("eedge   =%3d (%llx)\n", EG_indexBodyTopo(ebody, eedge   ), (long long)eedge   );
        printf("etemp[0]=%3d (%llx)\n", EG_indexBodyTopo(ebody, etemp[0]), (long long)etemp[0]);
        printf("etemp[1]=%3d (%llx)\n", EG_indexBodyTopo(ebody, etemp[1]), (long long)etemp[1]);
#endif

        if (etemp[0] == enode) {
            status = EG_evaluate(eedge, &trange[0], data);
            CHECK_STATUS(EG_evaluate);

            xyz[0] =  data[0];
            xyz[1] =  data[1];
            xyz[2] =  data[2];
            xyz[3] = -data[3];
            xyz[4] = -data[4];
            xyz[5] = -data[5];
        } else if (etemp[1] == enode) {
            status = EG_evaluate(eedge, &trange[1], data);
            CHECK_STATUS(EG_evaluate);

            xyz[0] =  data[0];
            xyz[1] =  data[1];
            xyz[2] =  data[2];
            xyz[3] = +data[3];
            xyz[4] = +data[4];
            xyz[5] = +data[5];
        }

        /* more than one Edge found */
    } else {
        printf("error2, count=%d\n", count); exit(0);
    }

cleanup:
    if (eedges != NULL) EG_free(eedges);
    if (efaces != NULL) EG_free(efaces);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   makeFlendCurve - make a flend curve between two points             *
 *                                                                      *
 ************************************************************************
 */

static int
makeFlendCurve(ego     context,         /* (in)  context */
               nnet_T  *nn,             /* (in)  neural network data */
               double  xyz1[],          /* (in)  data at beg of flend curve */
               double  xyz2[],          /* (in)  data at end of flend curve */
               double  frac1,           /* (in)  flend strength at beg */
               double  frac2,           /* (in)  flend strength at end */
               ego     *ecurve)         /* (out) resulting Curve */
{
    int status = EGADS_SUCCESS;

    int    header[4], i;
    double A[3], B[3], D[3], E[3];
    double mat1[12], mat2[12], mat3[12], mat4[12], mat5[12];
    double dist, phi, sinphi, cosphi, psi, sinpsi, cospsi, ang1, ang2, sinang, cosang;
    double temp, fleft, frite, fmid;
    double inputdata[3], outputdata[3];
    double knots[9], cps[15], rdata[24];

    ROUTINE (makeFlendCurve);

    /* --------------------------------------------------------------- */

    /* set up bspline header and knot vector */
    header[0] = 0;
    header[1] = 3;            /* degree */
    header[2] = 5;            /* ncp    */
    header[3] = 9;            /* nknot  */

    knots[0] = 0.0;
    knots[1] = 0.0;
    knots[2] = 0.0;
    knots[3] = 0.0;
    knots[4] = 0.5;
    knots[5] = 1.0;
    knots[6] = 1.0;
    knots[7] = 1.0;
    knots[8] = 1.0;

#ifdef DEBUG
    printf("xyz1: %10.5f %10.5f %10.5f   %10.5f %10.5f %10.5f\n", xyz1[0], xyz1[1], xyz1[2], xyz1[3], xyz1[4], xyz1[5]);
    printf("xyz2: %10.5f %10.5f %10.5f   %10.5f %10.5f %10.5f\n", xyz2[0], xyz2[1], xyz2[2], xyz2[3], xyz2[4], xyz2[5]);
#endif

    /* normalize the tangent vectors */
    temp = sqrt(xyz1[3] * xyz1[3] + xyz1[4] * xyz1[4] + xyz1[5] * xyz1[5]);
    xyz1[3] /= temp;
    xyz1[4] /= temp;
    xyz1[5] /= temp;

    temp = sqrt(xyz2[3] * xyz2[3] + xyz2[4] * xyz2[4] + xyz2[5] * xyz2[5]);
    xyz2[3] /= temp;
    xyz2[4] /= temp;
    xyz2[5] /= temp;

    /* make copies of the inputs (to be used in the transformations) */
    A[0] = xyz1[0];             A[1] = xyz1[1];             A[2] = xyz1[2];
    B[0] = xyz1[0] + xyz1[3];   B[1] = xyz1[1] + xyz1[4];   B[2] = xyz1[2] + xyz1[5];
    D[0] = xyz2[0] + xyz2[3],   D[1] = xyz2[1] + xyz2[4];   D[2] = xyz2[2] + xyz2[5];
    E[0] = xyz2[0];             E[1] = xyz2[1];             E[2] = xyz2[2];

    /* get the distance between A and E */
    dist = sqrt( (E[0] - A[0]) * (E[0] - A[0])
               + (E[1] - A[1]) * (E[1] - A[1])
               + (E[2] - A[2]) * (E[2] - A[2]));

    /* translate A to origin (ie, A[0]=A[1]=A[2]=0) */
    mat1[ 0] = 1;   mat1[ 1] = 0;   mat1[ 2] = 0;  mat1[ 3] = -A[0];
    mat1[ 4] = 0;   mat1[ 5] = 1;   mat1[ 6] = 0;  mat1[ 7] = -A[1];
    mat1[ 8] = 0;   mat1[ 9] = 0;   mat1[10] = 1;  mat1[11] = -A[2];

    xform(mat1, A);
    xform(mat1, B);
    xform(mat1, D);
    xform(mat1, E);

    /* rotate Y to put E in x-y plane (ie, E[2]=0) */
    phi  = atan2(E[2]-A[2], E[0]-A[0]);
    sinphi = sin(phi);
    cosphi = cos(phi);

    mat2[ 0] = +cosphi;  mat2[ 1] = 0;  mat2[ 2] = +sinphi;   mat2[ 3] = 0;
    mat2[ 4] = 0;        mat2[ 5] = 1;  mat2[ 6] = 0;         mat2[ 7] = 0;
    mat2[ 8] = -sinphi;  mat2[ 9] = 0;  mat2[10] = +cosphi;   mat2[11] = 0;

    xform(mat2, A);
    xform(mat2, B);
    xform(mat2, D);
    xform(mat2, E);

    /* rotate Z to put E on x axis (ie, E[1]=0) */
    psi  = atan2(E[1]-A[1], E[0]-A[0]);
    sinpsi = sin(psi);
    cospsi = cos(psi);

    mat3[ 0] = +cospsi;  mat3[ 1] = +sinpsi;  mat3[ 2] = 0;   mat3[ 3] = 0;
    mat3[ 4] = -sinpsi;  mat3[ 5] = +cospsi;  mat3[ 6] = 0;   mat3[ 7] = 0;
    mat3[ 8] = 0;        mat3[ 9] = 0;        mat3[10] = 1;   mat3[11] = 0;

    xform(mat3, A);
    xform(mat3, B);
    xform(mat3, D);
    xform(mat3, E);

    /* scale to put E at (1,0,0) (ie, E[0]=1) */
    mat4[ 0] = 1/dist;  mat4[ 1] = 0;       mat4[ 2] = 0;       mat4[ 3] = 0;
    mat4[ 4] = 0;       mat4[ 5] = 1/dist;  mat4[ 6] = 0;       mat4[ 7] = 0;
    mat4[ 8] = 0;       mat4[ 9] = 0;       mat4[10] = 1/dist;  mat4[11] = 0;

    xform(mat4, A);
    xform(mat4, B);
    xform(mat4, D);
    xform(mat4, E);

    /* rotate X to put xy plane between (AB) and (ED) */
    ang1   = atan2(B[2]-A[2], B[1]-A[1]);
    ang2   = atan2(D[2]-E[2], D[1]-E[1]);
    sinang = sin((ang1+ang2)/2);
    cosang = cos((ang1+ang2)/2);

    mat5[ 0] = 1;  mat5[ 1] = 0;        mat5[ 2] = 0;        mat5[ 3] = 0;
    mat5[ 4] = 0;  mat5[ 5] = +cosang;  mat5[ 6] = +sinang;  mat5[ 7] = 0;
    mat5[ 8] = 0;  mat5[ 9] = -sinang;  mat5[10] = +cosang;  mat5[11] = 0;

    xform(mat5, A);
    xform(mat5, B);
    xform(mat5, D);
    xform(mat5, E);

    /* look up fleft, frite, and fmid from the neural net */
    inputdata[0] = atan2(B[1]-A[1], B[0]-A[0]) / PIo180;
    inputdata[1] = atan2(D[1]-E[1], E[0]-D[0]) / PIo180;
    inputdata[2] = 0.5;

    if (inputdata[0] < 0) {
        inputdata[0] *= -1;
        inputdata[1] *= -1;
    }

    if (fabs(inputdata[0]-inputdata[1]) < EPS06) {
        status = nnet_eval(nn, inputdata, outputdata);
        CHECK_STATUS(nnet_eval);

        fleft = (outputdata[0] + outputdata[1]) / 2;
        frite = (outputdata[0] + outputdata[1]) / 2;
        fmid  = 0.50;
    } else if (fabs(inputdata[0]) <= fabs(inputdata[1])) {
        status = nnet_eval(nn, inputdata, outputdata);
        CHECK_STATUS(nnet_eval);

        fleft = outputdata[0];
        frite = outputdata[1];
        fmid  = outputdata[2];
    } else {
        temp         = inputdata[0];
        inputdata[0] = inputdata[1];
        inputdata[1] = temp;

        status = nnet_eval(nn, inputdata, outputdata);
        CHECK_STATUS(nnet_eval);

        fleft =     outputdata[1];
        frite =     outputdata[0];
        fmid  = 1 - outputdata[2];
    }

    /* override if frac1 and/or frac2 are given */
    if (frac1 > 0) {
        fleft = frac1;
        fmid  = 0.5;
    }
    if (frac2 > 0) {
        frite = frac2;
        fmid  = 0.5;
    }

    /* set up the control points */
    cps[3*0  ] = xyz1[0];
    cps[3*0+1] = xyz1[1];
    cps[3*0+2] = xyz1[2];

    cps[3*4  ] = xyz2[0];
    cps[3*4+1] = xyz2[1];
    cps[3*4+2] = xyz2[2];

    cps[3*1  ] = xyz1[0] + fleft * dist * xyz1[3];
    cps[3*1+1] = xyz1[1] + fleft * dist * xyz1[4];
    cps[3*1+2] = xyz1[2] + fleft * dist * xyz1[5];

    cps[3*3  ] = xyz2[0] + frite * dist * xyz2[3];
    cps[3*3+1] = xyz2[1] + frite * dist * xyz2[4];
    cps[3*3+2] = xyz2[2] + frite * dist * xyz2[5];

    cps[3*2  ] = (1-fmid) * cps[3*1  ] + fmid * cps[3*3  ];
    cps[3*2+1] = (1-fmid) * cps[3*1+1] + fmid * cps[3*3+1];
    cps[3*2+2] = (1-fmid) * cps[3*1+2] + fmid * cps[3*3+2];

    /* create the cubic bspline curve */
    for (i = 0; i < 9; i++) {
        rdata[i  ] = knots[i];
    }
    for (i = 0; i < 15; i++) {
        rdata[i+9] = cps[i];
    }
    status = EG_makeGeometry(context, CURVE, BSPLINE, NULL,
                             header, rdata, ecurve);
    CHECK_STATUS(EG_makeGeometry);

cleanup:
    return status;
}

static void
xform(double m[],
      double x[])
{
    double temp[3];

    temp[0] = m[ 0] * x[0] + m[ 1] * x[1] + m[ 2] * x[2] + m[ 3];
    temp[1] = m[ 4] * x[0] + m[ 5] * x[1] + m[ 6] * x[2] + m[ 7];
    temp[2] = m[ 8] * x[0] + m[ 9] * x[1] + m[10] * x[2] + m[11];

    x[0] = temp[0];
    x[1] = temp[1];
    x[2] = temp[2];

    return;
}


/*
 ***********************************************************************
 *                                                                     *
 *   nnet_make - make a neural network                                 *
 *                                                                     *
 ***********************************************************************
 */

static int
nnet_make(nnet_T  **nnet)               /* (out) created neural net */
{
    int    status = SUCCESS;            /* (out) return status */

    /* --------------------------------------------------------------- */

    nnet_T   *nn=NULL;

    ROUTINE(nnet_init);

    /* --------------------------------------------------------------- */

/* input layer scaling */
#define WEIGHT0( input        )  nn->weight0[           input        ]
#define BIAS0(   input        )  nn->bias0[             input        ]

/* hidden layer 1 variables */
#define WEIGHT1( input,hidden1)  nn->weight1[nn->ninput*hidden1+input]
#define BIAS1(         hidden1)  nn->bias1[             hidden1      ]

/* hidden layer 2 variables */
#define WEIGHT2( hidden1,hidden2)  nn->weight2[nn->nhidden*hidden2+hidden1]
#define BIAS2(           hidden2)  nn->bias2[              hidden2        ]

/* output layer variables */
#define WEIGHT3( hidden2,output)  nn->weight3[nn->nhidden*output+hidden2]
#define BIAS3(           output)  nn->bias3[              output        ]

/* output scaling */
#define WEIGHT4( output        )  nn->weight4[output]
#define BIAS4(   output        )  nn->bias4[  output]

    /* default return */
    *nnet = NULL;

    /* make a new nnet */
    MALLOC(nn, nnet_T, 1);

    nn->weight0 = NULL;
    nn->weight1 = NULL;
    nn->weight2 = NULL;
    nn->weight3 = NULL;
    nn->weight4 = NULL;

    nn->bias0 = NULL;
    nn->bias1 = NULL;
    nn->bias2 = NULL;
    nn->bias3 = NULL;
    nn->bias4 = NULL;

    /* get the header info */
    nn->ninput  = 3;
    nn->nhidden = 6;
    nn->noutput = 3;

    /* allocate all memory */
    MALLOC(nn->weight0, double, nn->ninput);
    MALLOC(nn->bias0,   double, nn->ninput);

    MALLOC(nn->weight1, double, nn->nhidden*nn->ninput);
    MALLOC(nn->bias1,   double, nn->nhidden           );

    MALLOC(nn->weight2, double, nn->nhidden*nn->nhidden);
    MALLOC(nn->bias2,   double, nn->nhidden            );

    MALLOC(nn->weight3, double, nn->noutput*nn->nhidden);
    MALLOC(nn->bias3,   double, nn->noutput            );

    MALLOC(nn->weight4, double, nn->noutput);
    MALLOC(nn->bias4,   double, nn->noutput);

    /* input scaling */
    BIAS0(  0) = -4.245097676301e-05;
    WEIGHT0(0) =  5.884791633842e-03;
    BIAS0(  1) =  5.001934769873e-01;
    WEIGHT0(1) =  2.858330704576e-03;
    BIAS0(  2) = -1.712750417306e-04;
    WEIGHT0(2) =  1.000204906402e+00;

    /* first hidden layer */
    BIAS1(    0) = -5.967890043793e-01;
    WEIGHT1(0,0) =  3.257264740507e+00;
    WEIGHT1(1,0) = -3.105117808389e+00;
    WEIGHT1(2,0) =  8.281137707605e-01;
    BIAS1(    1) = -4.359697537547e-01;
    WEIGHT1(0,1) = -7.320837271725e-01;
    WEIGHT1(1,1) =  1.260230782030e+01;
    WEIGHT1(2,1) =  3.947844968092e-01;
    BIAS1(    2) =  7.799773623491e+00;
    WEIGHT1(0,2) = -6.306931617842e-01;
    WEIGHT1(1,2) = -1.727292048620e+01;
    WEIGHT1(2,2) =  4.251968442707e-01;
    BIAS1(    3) =  3.372269915967e+00;
    WEIGHT1(0,3) = -5.241938091357e+00;
    WEIGHT1(1,3) =  4.149303255206e+00;
    WEIGHT1(2,3) =  4.500140028654e-01;
    BIAS1(    4) =  4.767240325985e+00;
    WEIGHT1(0,4) =  1.231307455748e+00;
    WEIGHT1(1,4) = -9.498043822496e+00;
    WEIGHT1(2,4) = -5.372563578076e-02;
    BIAS1(    5) = -1.253827993056e+01;
    WEIGHT1(0,5) =  1.049787450325e+01;
    WEIGHT1(1,5) =  1.220624883453e+01;
    WEIGHT1(2,5) = -2.743934813313e-04;

    /* second hidden layer */
    BIAS2(    0) =  1.288617557584e+00;
    WEIGHT2(0,0) = -6.229919338643e-01;
    WEIGHT2(1,0) =  8.152140843373e+00;
    WEIGHT2(2,0) =  4.377117396523e+00;
    WEIGHT2(3,0) = -4.921264064495e+00;
    WEIGHT2(4,0) = -1.319125507811e+01;
    WEIGHT2(5,0) =  3.728942315957e+00;
    BIAS2(    1) = -5.159598139120e-01;
    WEIGHT2(0,1) =  4.932391642196e+00;
    WEIGHT2(1,1) = -2.205587754510e+01;
    WEIGHT2(2,1) =  7.842258398384e-02;
    WEIGHT2(3,1) =  1.273017773856e+01;
    WEIGHT2(4,1) = -1.820564620113e+00;
    WEIGHT2(5,1) = -7.676681056305e+00;
    BIAS2(    2) =  7.771479572856e-01;
    WEIGHT2(0,2) =  6.242162765319e+00;
    WEIGHT2(1,2) =  2.574853140818e+00;
    WEIGHT2(2,2) =  7.470764587742e+00;
    WEIGHT2(3,2) = -4.764315924756e+00;
    WEIGHT2(4,2) = -1.569187585999e+01;
    WEIGHT2(5,2) =  1.530212352956e+00;
    BIAS2(    3) = -1.681015568706e-01;
    WEIGHT2(0,3) = -1.947099536458e+00;
    WEIGHT2(1,3) =  4.828449324959e+00;
    WEIGHT2(2,3) = -5.411482389997e+00;
    WEIGHT2(3,3) = -5.577385831053e+00;
    WEIGHT2(4,3) =  6.877180770372e+00;
    WEIGHT2(5,3) =  4.392494275238e+00;
    BIAS2(    4) = -3.265681938746e+00;
    WEIGHT2(0,4) =  5.305662528118e+00;
    WEIGHT2(1,4) =  1.279993107856e+00;
    WEIGHT2(2,4) =  7.066280157683e+00;
    WEIGHT2(3,4) =  4.830045143417e+00;
    WEIGHT2(4,4) = -1.379907460549e+01;
    WEIGHT2(5,4) = -2.080182568084e+00;
    BIAS2(    5) = -7.878198261176e-01;
    WEIGHT2(0,5) =  4.340617159645e-02;
    WEIGHT2(1,5) = -6.403597257879e+00;
    WEIGHT2(2,5) =  1.494689929446e+00;
    WEIGHT2(3,5) =  8.505888606562e+00;
    WEIGHT2(4,5) = -6.829529695252e+00;
    WEIGHT2(5,5) =  4.720808937779e-03;

    /* output layer */
    BIAS3(    0) = -6.374236213468e+00;
    WEIGHT3(0,0) = -2.051515188266e+00;
    WEIGHT3(1,0) =  1.411311218255e+01;
    WEIGHT3(2,0) =  7.210642076360e+00;
    WEIGHT3(3,0) =  7.122052911231e+00;
    WEIGHT3(4,0) =  8.963885627894e+00;
    WEIGHT3(5,0) = -8.140729125490e+00;
    BIAS3(    1) =  3.591707577827e+00;
    WEIGHT3(0,1) =  1.022323754854e+01;
    WEIGHT3(1,1) =  6.324006386081e-01;
    WEIGHT3(2,1) =  1.420097216902e+01;
    WEIGHT3(3,1) = -9.721932176827e+00;
    WEIGHT3(4,1) = -1.161161945794e+00;
    WEIGHT3(5,1) = -1.446625505883e+01;
    BIAS3(    2) = -9.084974011096e+00;
    WEIGHT3(0,2) = -3.308681891698e+00;
    WEIGHT3(1,2) =  9.158089686317e-01;
    WEIGHT3(2,2) = -3.642650889270e+00;
    WEIGHT3(3,2) =  1.515997170087e+01;
    WEIGHT3(4,2) =  1.012772861450e+01;
    WEIGHT3(5,2) = -1.289174782999e+01;

    /* output scaling */
    BIAS4(  0) =  5.000000000000e-02;
    WEIGHT4(0) =  9.500000000000e-01;
    BIAS4(  1) =  5.000000000000e-02;
    WEIGHT4(1) =  9.500000000000e-01;
    BIAS4(  2) =  5.000000000000e-02;
    WEIGHT4(2) =  9.000000000000e-01;

    *nnet = nn;

cleanup:
#undef WEIGHT0
#undef WEIGHT1
#undef WEIGHT2
#undef WEIGHT3
#undef WEIGHT4

#undef BIAS0
#undef BIAS1
#undef BIAS2
#undef BIAS3
#undef BIAS4

    return status;
}


/*
 ***********************************************************************
 *                                                                     *
 *   nnet_eval - evaluate a nneural net                                *
 *                                                                     *
 ***********************************************************************
 */

static int
nnet_eval(nnet_T  *nn,                  /* (in)  pointer to neural net */
          double  inputs[],             /* (in)  inputs  to neural net */
          double  outputs[])            /* (out) outputs to neural net */
{
    int    status = SUCCESS;            /* (out) return status */

    /* --------------------------------------------------------------- */

    /* neural network working variables */
    double    *active0=NULL;
    double    *active1=NULL;
    double    *active2=NULL;
    double    *active3=NULL;
    double    sum;
    int       input, hidden1, hidden2, output;

    ROUTINE(nnet_eval);

    /* --------------------------------------------------------------- */

/* input layer scaling */
#define WEIGHT0( input        )  nn->weight0[           input        ]
#define BIAS0(   input        )  nn->bias0[             input        ]
#define ACTIVE0( input        )  active0[               input        ]

/* hidden layer 1 variables */
#define WEIGHT1( input,hidden1)  nn->weight1[nn->ninput*hidden1+input]
#define BIAS1(         hidden1)  nn->bias1[             hidden1      ]
#define ACTIVE1(       hidden1)  active1[               hidden1      ]

/* hidden layer 2 variables */
#define WEIGHT2( hidden1,hidden2)  nn->weight2[nn->nhidden*hidden2+hidden1]
#define BIAS2(           hidden2)  nn->bias2[              hidden2        ]
#define ACTIVE2(         hidden2)  active2[                hidden2        ]

/* output layer variables */
#define WEIGHT3( hidden2,output)  nn->weight3[nn->nhidden*output+hidden2]
#define BIAS3(           output)  nn->bias3[              output        ]
#define ACTIVE3(         output)  active3[                output        ]

/* output scaling */
#define WEIGHT4( output        )  nn->weight4[output]
#define BIAS4(   output        )  nn->bias4[  output]

    /* assertions (for stanalizer */
    assert(nn->ninput  == 3);
    assert(nn->nhidden == 6);
    assert(nn->noutput == 3);

    /* allocate all memory */
    MALLOC(active0, double, nn->ninput );
    MALLOC(active1, double, nn->nhidden);
    MALLOC(active2, double, nn->nhidden);
    MALLOC(active3, double, nn->noutput);

    /* scale the inputs */
    for (input = 0; input < nn->ninput; input++) {
        ACTIVE0(input) = BIAS0(input) + WEIGHT0(input) * inputs[input];
    }

    /* forward propagate to first hidden layer */
    for (hidden1 = 0; hidden1 < nn->nhidden; hidden1++) {
        sum = BIAS1(hidden1);
        for (input = 0; input < nn->ninput; input++) {
            sum += WEIGHT1(input,hidden1) * ACTIVE0(input);
        }
        ACTIVE1(hidden1) = sigmoid(sum);
    }

    /* forward propagate to second hidden layer */
    for (hidden2 = 0; hidden2 < nn->nhidden; hidden2++) {
        sum = BIAS2(hidden2);
        for (hidden1 = 0; hidden1 < nn->nhidden; hidden1++) {
            sum += WEIGHT2(hidden1,hidden2) * ACTIVE1(hidden1);
        }
        ACTIVE2(hidden2) = sigmoid(sum);
    }

    /* forward propagate to output layer */
    for (output = 0; output < nn->noutput; output++) {
        sum = BIAS3(output);
        for (hidden2 = 0; hidden2 < nn->nhidden; hidden2++) {
            sum += WEIGHT3(hidden2,output) * ACTIVE2(hidden2);
        }
        ACTIVE3(output) = sigmoid(sum);
    }

    /* scale and store the outputs */
    for (output = 0; output < nn->noutput; output++) {
        outputs[output] = BIAS4(output) + WEIGHT4(output) * ACTIVE3(output);
    }

cleanup:
    FREE(active0);
    FREE(active1);
    FREE(active2);
    FREE(active3);

#undef WEIGHT0
#undef BIAS0
#undef ACTIVE0

#undef WEIGHT1
#undef BIAS1
#undef ACTIVE1

#undef WEIGHT2
#undef BIAS2
#undef ACTIVE2

#undef WEIGHT3
#undef BIAS3
#undef ACTIVE3

#undef WEIGHT4
#undef BIAS4

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   nnet_free - free data associated with a neural net                 *
 *                                                                      *
 ************************************************************************
 */
static int
nnet_free(nnet_T   *nn)                 /* (in)  neural network to free */
{
    int    status = SUCCESS;            /* (out) return status */

    ROUTINE(nnet_free);

    /* --------------------------------------------------------------- */

    if (nn == NULL) goto cleanup;

    FREE(nn->weight0);
    FREE(nn->weight1);
    FREE(nn->weight2);
    FREE(nn->weight3);
    FREE(nn->weight4);

    FREE(nn->bias0);
    FREE(nn->bias1);
    FREE(nn->bias2);
    FREE(nn->bias3);
    FREE(nn->bias4);

    FREE(nn);
cleanup:
    return status;
}


/*
 ***********************************************************************
 *                                                                     *
 *   sigmoid - sigmoid function                                        *
 *                                                                     *
 ***********************************************************************
 */

static double
sigmoid(double  x) {
    return 1.0 / (1.0 + exp(-x));
}


#ifndef NOG1
/*
 ***********************************************************************
 *                                                                     *
 *   makeTangent - adjust CPs to make surface G1                       *
 *                                                                     *
 ***********************************************************************
 */

static int
makeTangent(const ego ebody,         /* (in)   the Body */
            nnet_T  *nn,             /* (in)   neural network data */
            /*@null@*/ ego bottom,   /* (in)   Edge or NULL (Node) for bottom */
            char attrb[],            /* (in)   Attribute for bottom */
            double fracb,            /* (in)   flend strength at bottom */
            /*@null@*/ ego top,      /* (in)   Edge or NULL (Node) for top */
            char attrt[],            /* (in)   Attribute for top */
            double fract,            /* (in)   flend strength at top */
            /*@null@*/ double *nxyz, /* (in)   xyz, d(xyz)/d(t) for Node */
            ego *esurf)              /* (both) input surf, return with update */
{
  int    i, j, k, status, oclass, mtype, *cints, *ints = NULL;
  double t, xyz[3], xyzb[6], xyzt[6], *cp, *cpc, *creals, *reals = NULL;
  ego    context, esurface, rGeom, ecurve;
  
  ROUTINE(makeTangent);
  
  if ((top == NULL) && (bottom == NULL)) {
    printf(" Error: bottom and top are both NULL!\n");
    status = EGADS_REFERCE;
    goto cleanup;
  }

  esurface = *esurf;
  status   = EG_getContext(esurface, &context);
  CHECK_STATUS(EG_getContext);
  
  /* get the BSpline surface to modify */
  status = EG_getGeometry(esurface, &oclass, &mtype, &rGeom, &ints, &reals);
  CHECK_STATUS(EG_getGeometry);
  if ((oclass != SURFACE) || (mtype != BSPLINE) ||
      (ints == NULL)      || (reals == NULL)) {
    printf(" Error: Input entity not a BSPLINE surface!\n");
    status = EGADS_GEOMERR;
    goto cleanup;
  }

#ifdef DEBUG
  printf(" *** Flend Surface (%d) Type = %d ***\n", oclass, mtype);
  printf("     %d  %d %d %d  %d %d %d\n", ints[0], ints[1], ints[2],
         ints[3], ints[4], ints[5], ints[6]);
  printf("     Uknots = %6.4lf", reals[0]);
  for (j = i = 1; i < ints[3]; i++, j++) {
    printf(" %6.4lf", reals[i]);
    if (j == 8) {
      if (i != ints[3]-1) printf("\n             ");
      j = -1;
    }
  }
  printf("\n");
  printf("     Vknots = %6.4lf", reals[ints[3]]);
  j = 1;
  for (i = ints[3]+1; i < ints[3]+ints[6]; i++, j++) {
    printf(" %6.4lf", reals[i]);
    if (j == 8) {
      if (i != ints[3]+ints[6]-1) printf("\n             ");
      j = -1;
    }
  }
  printf("\n");
#endif
  
  /* do we need to do anything? */
  if (ints[5] == 2) goto cleanup;
  if (ints[5] - ints[4] < 3) goto cleanup;

  /* adjust the interior control points */
  cp = &reals[ints[3]+ints[6]];
  
  /* for now -> leave a buffer of 1 cp */
  for (j = 2; j < ints[5]-2; j++) {
    i = 0;
    if (bottom != NULL) {
      status = EG_invEvaluate(bottom, &cp[3*(j*ints[2]+i)], &t, xyz);
      CHECK_STATUS(EG_invEvaluate);
      status = getXYZforEdge(ebody, bottom, attrb, t, xyzb);
      CHECK_STATUS(getXYZforEdge);
    } else if (nxyz != NULL) {
      for (k = 0; k < 6; k++) xyzb[k] = nxyz[k];
    } else {
      printf(" Error: bottom and nxyz are both NULL!\n");
      status = EGADS_REFERCE;
      goto cleanup;
    }
    i = ints[2]-1;
    if (top != NULL) {
      status = EG_invEvaluate(top, &cp[3*(j*ints[2]+i)], &t, xyz);
      CHECK_STATUS(EG_invEvaluate);
      status = getXYZforEdge(ebody, top, attrt, t, xyzt);
      CHECK_STATUS(getXYZforEdge);
    } else if (nxyz != NULL) {
      for (k = 0; k < 6; k++) xyzt[k] = nxyz[k];
    } else {
      printf(" Error: top and nxyz are both NULL!\n");
      status = EGADS_REFERCE;
      goto cleanup;
    }
    
    /* note: can improve performance by not making an ego */
    status = makeFlendCurve(context, nn, xyzb, xyzt, fracb, fract, &ecurve);
    CHECK_STATUS(makeFlendCurve);
    status = EG_getGeometry(ecurve, &oclass, &mtype, &rGeom, &cints, &creals);
    EG_deleteObject(ecurve);
    CHECK_STATUS(EG_getGeometry);
    
    /* adjust the inner control points -- we know this is a BSPLINE curve! */
    cpc = &creals[ints[3]];
    for (i = 1; i < ints[2]-1; i++) {
      cp[3*(j*ints[2]+i)  ] = cpc[3*i  ];
      cp[3*(j*ints[2]+i)+1] = cpc[3*i+1];
      cp[3*(j*ints[2]+i)+2] = cpc[3*i+2];
    }
    EG_free(cints);
    EG_free(creals);
  }
  
  /* make the updated surface then remove the input surface */
  status = EG_makeGeometry(context, SURFACE, BSPLINE, NULL, ints, reals, esurf);
  CHECK_STATUS(EG_makeGeometry);
  EG_deleteObject(esurface);

cleanup:
  if (ints  != NULL) EG_free(ints);
  if (reals != NULL) EG_free(reals);
  
  return status;
}
#endif
