# Import pyCAPS module
import pyCAPS

# Import os module
import os
import shutil
import argparse

# Setup and read command line options. Please note that this isn't required for pyCAPS
parser = argparse.ArgumentParser(description = 'missile DATCOM Pytest Example',
                                 prog = 'missileDatcom_PyTest',
                                 formatter_class = argparse.ArgumentDefaultsHelpFormatter)

#Setup the available commandline options
parser.add_argument('-workDir', default = "./", nargs=1, type=str, help = 'Set working/run directory')
parser.add_argument('-noAnalysis', action='store_true', default = False, help = "Don't run analysis code")
parser.add_argument("-outLevel", default = 1, type=int, choices=[0, 1, 2], help="Set output verbosity")
args = parser.parse_args()

# Create project name
projectName = "MissileDatcomTest"

# Working directory
workDir = os.path.join(str(args.workDir[0]), projectName)

# Load CSM file
geometryScript = os.path.join("..","csmData","elissim","elissim.csm")
capsProblem = pyCAPS.Problem(problemName=workDir,
                             capsFile=geometryScript,
                             outLevel=args.outLevel)

# Set configuration parameters (CFGPMTR)

# Set configuration parameters (CFGPMTR)
capsProblem.geometry.cfgpmtr.VIEW.Concept = 0
capsProblem.geometry.cfgpmtr.VIEW.Vlm = 1
capsProblem.geometry.cfgpmtr.VIEW.Cfd = 0

# Set design parameters (DESPMTR)
capsProblem.geometry.despmtr.Fuse.centrWidth = 10.0
capsProblem.geometry.despmtr.Fuse.centrHeight = 10.0

# Create the missileDatcom AIM
missileDatcom = capsProblem.analysis.create(aim = "missileDatcomAIM",
                                            autoExec = not args.noAnalysis)

# Set the name of the missileDatcom executable
missileDatcom.input.missileDatcom = "misdat_2014e.exe"

# Set body geometry inputs
missileDatcom.input.BODoption = 1
missileDatcom.input.BODtrunc = False
missileDatcom.input.BODtnose = "CONICAL"
missileDatcom.input.BODtaft = "CONICAL"

#missileDatcom.input.FinSet = {"FinSet1": {"SecNACA" : "NACA-4-0012"}}


# Set analysis conditions
missileDatcom.input.Mach = [0.2, 1.5, 3.]
missileDatcom.input.Altitude = [0., 5000, 200]
missileDatcom.input.Alpha = [-2, 2, 6]
missileDatcom.input.BL_Type = "TURB"
missileDatcom.input.Surface_Roughness = 0

if args.noAnalysis == False:
    missileDatcom.runAnalysis()
else:
    missileDatcom.preAnalysis()
    # Copy old results if no analysis available
    shutil.copy2(os.path.join("..","analysisData","missileDatcom",projectName+"_for042.csv"),
                 os.path.join(missileDatcom.analysisDir,"for042.csv"))
    missileDatcom.postAnalysis()

for i in missileDatcom.output.keys():
    print(i, " = ", missileDatcom.output[i].value)
