/*
 *      CAPS: Computational Aircraft Prototype Syntheses
 *
 *             FlightStream AIM
 *
 *     Written by John Dannenhoffer (Syracuse) and
 *                Marshall Galbraith (MIT)
 *
 *      Copyright 2014-2025, Massachusetts Institute of Technology
 *      Licensed under The GNU Lesser General Public License, version 2.1
 *      See http://www.opensource.org/licenses/lgpl-2.1.php
 *
 */

/*!\mainpage Introduction
 * \tableofcontents
 * \section overviewFLIGHTSTREAM FlightStream AIM Overview
 *
 * A module in the Computational Aircraft Prototype Syntheses (CAPS)
 * has been developed to interact (primarily through input files) with
 * Research in Flight's FlightStream \cite FlightStream. FlightStream predicts
 * the aerodynamic performance of a vehicle via a panel method, which
 * makes it very fast.
 *
 * An outline of the AIM's inputs and outputs are provided in \ref
 * aimInputsFLIGHTSTREAM and \ref aimOutputsFLIGHTSTREAM, respectively.
 *
 * Geometric attributes recognized by the AIM are provided in \ref
 * attributeFLIGHTSTREAM.
 *
 * \section assumptionsFLIGHTSTREAM Geometry Requirements and Assumptions
 * The FlightStream coordinate system assumption (X -- downstream) needs to be followed.
 *
 * The FligthStream AIM supports two types of geometry representations:
 *  -# Solid body (SOLIDBODY) or manifold sheet body (SHEETBODY)\n
 *     This geometry format requires the use of a surface meshing AIM that is linked to the Surface_Mesh input (see \ref aimInputsFLIGHTSTREAM).
 *  -# Sectional geometry\n
 *     Sectional geometry information used to generate FligthStream CCS geometry file (see CCS inputs in \ref aimInputsFLIGHTSTREAM).
 *     Currently supported CCS bodies are:
 *     - General Components (Fuselage, Fairings, Pods, etc.)
 *     - Lifting Surface Components (Wing, Fin, etc.) with Standard Deflection Control Surfaces
 *     - Annular Components
 *     - Revolution Components
 *
 * \subsection secGeomFLIGHTSTREAM Sectional Geometry Requirements
 *
 * Within <b>OpenCSM</b> there are a number of airfoil generation UDPs (User Defined Primitives). These include NACA 4
 * series, a more general NACA 4/5/6 series generator, Sobieczky's PARSEC parameterization and Kulfan's CST
 * parameterization. All of these UDPs generate <b>EGADS</b> <em>FaceBodies</em> where the <em>Face</em>'s underlying
 * <em>Surface</em>  is planar and the bounds of the <em>Face</em> is a closed set of <em>Edges</em> whose
 * underlying <em>Curves</em> contain the airfoil shape.
 *
 * <b>Important Airfoil Geometry Assumptions</b>
 * - There must be a <em>Node</em> that represents the <em>Leading Edge</em> point
 * - For a sharp trailing edge, there must be a <em>Nodes</em> at the <em>Trailing Edge</em>
 * - For a blunt trailing edge, the airfoil curve may be open, or closed by a single <em>Edge</em> connecting the upper/lower <em>Nodes</em>
 * - For a <em>FaceBody</em>, the airfoil coordinates traverse counter-clockwise around the <em>Face</em> normal. The <b>OpenCSM</b> <em>REORDER</em> operation may be used to flip the <em>Face</em> normal.
 * - For a <em>WireBody</em>, the airfoil coordinates traverse in the order of the loop
 *
 * <b>Note:</b> Additional spurious <em>Nodes</em> on the upper and lower <em>Edges</em> of the airfoil are acceptable.
 *
 * The FlightStream CCS surfaces are generated from a set of <em> FaceBodys</em> with the same capsGroup attributes,
 * and the geometric details extracted from the geometry. Attempts are made to orient and sort <em>FaceBody</em>, but users are encouraged to check
 * the airfoil orientation in the CCS input file when setting up a new problem.
 *
 * It should be noted that general construction in either <b> OpenCSM</b> or even <b> EGADS</b> will be supported
 * as long as the topology described above is used. But care should be taken when constructing the airfoil shape
 * so that a discontinuity (i.e.,  simply <em>C<sup>0</sup></em>) is not generated at the <em>Node</em> representing
 * the <em>Leading Edge</em>. This can be done by fitting a spline through the entire shape as one and then intersecting the single
 *  <em>Edge</em> to place the LE <em>Node</em>.
 *
 * <b>CCS General Components</b><br>
 * The <em>FaceBody</em> sections for general components and can be oriented in any direction with any shape. However, sections may not have perpendicular normal vectors.
 *
 * <b>CCS Lifting Surface Components</b><br>
 * The <em> FaceBody</em> must contain at least two edges and two nodes, but may contain any number of <em> Edges</em> otherwise.
 * If the <em> FaceBody</em> contains more nodes, the node with the smallest <b> x</b> value is used to
 * define the leading edge, the node with the largest <b> x</b> defines the trailing edge. The airfoil may have a
 * single <em> Edge</em> that defines a straight blunt trailing edge. The airfoil shapes in the CCS file are generated by
 * sampling the <em> Curves</em> of each section.
 *
 * Trailing edge control surfaces can be added to the above example by making use of the vlmControlName attribute (see \ref attributeFLIGHTSTREAM regarding the attribution
 * specifics).  To add a <b>RightFlap</b> and <b>LeftFlap</b> to the
 * the naca UDP entries are augmented with the following attributes.
 *
 * \code {.py}
 * # left tip
 * udprim    naca      Thickness thick     Camber    camber
 * attribute vlmControl_LeftFlap 80 # Hinge line is at 80% of the chord (control surface between hinge and trailing edge)
 * ...
 *
 * # root
 * udprim    naca      Thickness thick     Camber    camber
 * attribute vlmControl_LeftFlap 80 # Hinge line is at 80% of the chord (control surface between hinge and trailing edge)
 * attribute vlmControl_RightFlap 80 # Hinge line is at 80% of the chord (control surface between hinge and trailing edge)
 * ...
 *
 * # right tip
 * udprim    naca      Thickness thick     Camber    camber
 * attribute vlmControl_RightFlap 80 # Hinge line is at 80% of the chord (control surface between hinge and trailing edge)
 * ...
 * \endcode
 *
 * Note how the root airfoil contains two attributes for both the left and right flaps.
 * Only trailing edge control surfaces are currently supported by FlightStream.
 *
 * In the pyCAPS script the \ref aimInputsFLIGHTSTREAM, <b>CCS_Control</b>, must be defined.
 * \code {.py}
 * flap = {"deflectionAngle" : 10.0}
 *
 *  flightstream.input.CCS_Control = {"LeftFlap": flap, "RightFlap": flap}
 \endcode
 *
 * Notice how the information defined in the <b>flap</b> variable is assigned to the vlmControl<b>Name</b> portion of the
 * attributes added to the *.csm file.
 *
 * <b>CCS Annular Components</b><br>
 * Annular components are made up of multiple annular airfoil sections.
 * The same section rules for Lifting Surface components apply. However, blunt trailing edges are currently not supported by FlightStream.
 *
 * <b>CCS Revolution Components</b><br>
 * Revolution components are made up of a single airfoil sections along with an axis of rotation.
 * The same section rules for Lifting Surface components apply. However, blunt trailing edges are currently not supported by FlightStream.
 * The axis of rotation body must be a LINE and have the same capsGroup attribute as the other sections, as well as a capsType=Axis string attribute.
 *
 *
 */

 /*! \page attributeFLIGHTSTREAM Attribution
 *
 * The following list of attributes drives the FlightStream geometric definition.
 *
 * - <b> capsGroup</b> For sectional CCS geometry, this string attribute labels the <em> FaceBody</em> as to which AVL Surface the section
 *  is assigned. This should be something like: <em> Main\_Wing</em>, <em> Horizontal\_Tail</em>, etc. This informs
 *  the AVL AIM to collect all <em> FaceBodies</em> that match this attribute into a single AVL Surface.<br>
 *  Note: If a capsGroup contains only one section then the section is treated as a slender body, and only the numChord and spaceChord in the "AVL_Surface" (\ref vlmSurface) input will be used.
 *
 * - <b> vlmControl"Name"</b> This string attribute attaches a control surface to the Lifting Surface CCS geometry <em>FaceBody</em>. The hinge
 * location is defined as the double value between 0 or 1.0.  The range as percentage from 0 to 100
 * will also work. The name of the
 * control surface is the string information after vlmControl (or vlmControl\_).  For Example, to define a control
 * surface named Aileron the following are identical (<em>attribute vlmControlAileron 0.8 </em> or
 * <em>attribute vlmControl\_Aileron 80</em>) . Multiple <em> vlmControl</em> attributes, with different
 * names, can be defined on a single <em> FaceBody</em>.
 *
 * - <b> vlmNumSpan </b> This attribute may be set on any given airfoil cross-section to overwrite the number of
 * spanwise elements placed on the surface (globally set - see keyword "numSpanPerSection" and "numSpanTotal" in \ref vlmSurface )
 * between two sections. Note, that the AIM internally sorts the sections in ascending y (or z) order, so care
 * should be taken to select the correct section for the desired intent.
 *
 * - <b> vlmNumChord </b> This attribute may be set on any given airfoil cross-section to overwrite the number of
 * chordwise elements placed on the surface (globally set - see keyword "numChord" in \ref vlmSurface ).<br>
 * vlmNumChord must be the same if set on multiple sections of a surface.
 *
 * - <b> capsLength</b> This attribute defines the length units that
 *  the *.csm file is generated in and is not optional for FligtStream.
 *  The FlightStream input grid will be scaled to either the default length of METER
 *  or the user specified length unit (see \ref aimUnitsFLIGHTSTREAM).
 *
 * - <b> capsReferenceChord</b> and <b> capsReferenceSpan</b>
 * [Optional] These attributes may exist on any <em> Body</em>. Their
 * value will be used as the reference moment lengths in FlightStream's
 * input file with their units assumed to be consistent with the
 * attribute "capsLength". These values may be alternatively set through an input
 * value, "ReferenceChord" (see \ref aimInputsFLIGHTSTREAM)
 *
 * - <b> capsReferenceArea</b> [Optional] This attribute may exist on
 * any <em> Body</em>.  Its value will be used as the reference area
 * in FlightStream's input file with its units assumed to be consistent with
 * the attribute "capsLength". This value may be alternatively set
 * through an input value, "ReferenceArea" (see \ref aimInputsFLIGHTSTREAM)
 *
 * - <b> capsReferenceX</b>, <b> capsReferenceY</b>,  and <b> capsReferenceZ</b>
 * [Optional] These attribute may exist on any <em> Body</em>. Their
 * value will be used as the reference moment lengths in FlightStream's
 * input file with their units assumed to be consistent with the
 * attribute "capsLength". These values may be alternatively set through an input
 * value, "ReferenceX" (see \ref aimInputsFLIGHTSTREAM)
 *
 * - <b> capsReferenceX</b>, <b> capsReferenceY</b>,  and <b> capsReferenceZ</b>
 * [Optional] These attribute may exist on any <em> Body</em>. Their
 * value will be used as the reference moment lengths in FlightStream's
 * input file with their units assumed to be consistent with the
 * attribute "capsLength". These values may be alternatively set through an input
 * value, "ReferenceX" (see \ref aimInputsFLIGHTSTREAM)
 *
 */

//#define DEBUG

#include <string.h>
#include <math.h>
#include "capsTypes.h"
#include "aimUtil.h"

#include "miscUtils.h"
#include "jsonUtils.h"
#include "meshUtils.h"
#include "cfdUtils.h"
#include "vlmUtils.h"

#include "wavefrontWriter.h"

#ifdef WIN32
   #define snprintf    _snprintf
   #define strcasecmp  stricmp
   #define strncasecmp _strnicmp
   #define strtok_r    strtok_s
#else
   #include <unistd.h>
   #include <limits.h>
#endif

/* define the indicies of the inputs and outputs */
enum aimInputs
{
    inProj_Name = 1,             /* index is 1-biased */
    inFlightStream,
    inMach,
    inAlpha,
    inBeta,
    inSweepRangeMach,
    inSweepRangeVelocity,
    inSweepRangeAlpha,
    inSweepRangeBeta,
    inSweepClear_Solution_After_Each_Run,
    inSweepReference_Velocity_Equals_Freestream,
    inSweepExport_Surface_Data,
    inFluid_Properties,
    inAltitude,
    inSolver_Model,
    inWake_Termination,
    inStabilization,
    inWall_Collision_Avoidance,
    inCCS_GeneralSurface,
    inCCS_LiftingSurface,
    inCCS_Control,
    inCCS_Revolution,
    inCCS_Annular,
    inReferenceChord,
    inReferenceSpan,
    inReferenceArea,
    inReferenceX,
    inReferenceY,
    inReferenceZ,
   // inReferenceLength,
    inReferenceVelocity,
    inPressure_Scale_Factor,
    inConvergenceTol,
    inSolverIterations,
    inSolverConvergence,
    inExport_Solver_Analysis,
    inFlightScriptPre,
    inFlightScriptPost,
//    inDesign_Variable,
//    inDesign_Functional,
    inHidden,
    inMax_Parallel_Threads,
    inSaveFSM,
    inMesh_Morph,
    inSurface_Mesh,
    NUMINPUT = inSurface_Mesh    /* Total number of inputs */
};

enum aimOutputs
{
  outAlpha = 1,                /* index is 1-based */
  outBeta,
  outMach,
  outVelocity,
  outDensity,
  outPressure,
  outSonicSpeed,
  outTemperature,
  outViscosity,
  outCx,
  outCy,
  outCz,
  outCL,
  outCDi,
  outCDo,
  outCMx,
  outCMy,
  outCMz,
  NUMOUTPUT = outCMz           /* Total number of outputs */
};

#define MXCHAR  255

#define ANALYSIS_CSV

#ifdef ANALYSIS_PLOAD_BDF
static const char* loadbdf = "surface_load.bdf";
#elif defined(ANALYSIS_VTK)
static const char* cpvtk = "surface_cp.vtk";
#elif defined(ANALYSIS_CSV)
static const char* cpcsv = "surface_cp.csv";
static const char* pcsv  = "surface_p.csv";
#else
#error "must define ANALYSIS_PLOAD_BDF, ANALYSIS_VTK, or ANALYSIS_CSV"
#endif

static const char *resultsFile = "results.txt";
static const char *resultsSweepFile = "sweep_results.txt";
static const char *settingsFile = "settings.txt";

typedef struct {

  // Fluid properties in specific units
  double rhoff;        // kg/m^3
  double pff;          // Pa
  double sonicspeedff; // m/s
  double tff;          // K
  double muff;         // Pa-sec

  // Reference quanteties
  double speedRef;     // m/s
  double machRef;

  // variables that could be obtained via attributes on the Body
  double Cref;       // capsReferenceChord
  double Bref;       // capsReferenceSpan
  double Sref;       // capsReferenceArea
  double Xref;       // capsReferenceX   (cg)
  double Yref;       // capsReferenceY   (cg)
  double Zref;       // capsReferenceZ   (cg)

  // outputs from FlightStream
  capsValue Alpha;       // Angle of Attack
  capsValue Beta;        // Side slip
  capsValue Mach;        // Freestream mach
  capsValue Velocity;    // Freestream velocity
  capsValue Density;     // Freestream density
  capsValue Pressure;    // Freestream pressure
  capsValue SonicSpeed;  // Freestream speed of sound
  capsValue Temperature; // Freestream temperature
  capsValue Viscosity;   // Freestream viscosity
  capsValue Cx;          // X-force coefficient
  capsValue Cy;          // Y-force coefficient
  capsValue Cz;          // Z-force coefficient
  capsValue CL;          // lift coefficient
  capsValue CDi;         // drag coefficient (inviscid)
  capsValue CDo;         // drag coefficient (total)
  capsValue CMx;         // X-moment coefficient
  capsValue CMy;         // Y-moment coefficient
  capsValue CMz;         // Z-moment coefficient

  // CCS general surface information
  int numCCSGenSurface;
  vlmSurfaceStruct *ccsGenSurface;

  // CCS lifting surface information
  int numCCSLiftSurface;
  vlmSurfaceStruct *ccsLiftSurface;

  // CCS control surface information
  int numCCSControl;
  vlmControlStruct *ccsControl;

  // CCS body of revolution information
  int numCCSRevolution;
  vlmSurfaceStruct *ccsRevolution;

  // CCS Nacelle & Annular Duct information
  int numCCSAnnular;
  vlmSurfaceStruct *ccsAnnular;

  // Attribute to index map
  mapAttrToIndexStruct groupMap;

  // Design information
  cfdDesignStruct design;

  // Units structure
  cfdUnitsStruct units;

  double lengthScaleFactor;

  // Mesh reference obtained from meshing AIM
  const aimMeshRef *meshRefIn;
  aimMeshRef meshRefMorph, meshRefDisp;

} aimStorage;


static int initialize_aimStorage(aimStorage *flightstreamInstance)
{

  // Set initial values for flightstreamInstance
  flightstreamInstance->rhoff = 0;
  flightstreamInstance->pff = 0;
  flightstreamInstance->sonicspeedff = 0;
  flightstreamInstance->tff = 0;
  flightstreamInstance->muff = 0;

  flightstreamInstance->speedRef = 0;
  flightstreamInstance->machRef = 0;

  flightstreamInstance->Cref = 0;
  flightstreamInstance->Bref = 0;
  flightstreamInstance->Sref = 0;
  flightstreamInstance->Xref = 0;
  flightstreamInstance->Yref = 0;
  flightstreamInstance->Zref = 0;

  aim_initValue(&flightstreamInstance->Alpha);
  aim_initValue(&flightstreamInstance->Beta);
  aim_initValue(&flightstreamInstance->Mach);
  aim_initValue(&flightstreamInstance->Velocity);
  aim_initValue(&flightstreamInstance->Density);
  aim_initValue(&flightstreamInstance->Pressure);
  aim_initValue(&flightstreamInstance->SonicSpeed);
  aim_initValue(&flightstreamInstance->Temperature);
  aim_initValue(&flightstreamInstance->Viscosity);
  aim_initValue(&flightstreamInstance->Cx);
  aim_initValue(&flightstreamInstance->Cy);
  aim_initValue(&flightstreamInstance->Cz);
  aim_initValue(&flightstreamInstance->CL);
  aim_initValue(&flightstreamInstance->CDi);
  aim_initValue(&flightstreamInstance->CDo);
  aim_initValue(&flightstreamInstance->CMx);
  aim_initValue(&flightstreamInstance->CMy);
  aim_initValue(&flightstreamInstance->CMz);

  flightstreamInstance->numCCSGenSurface = 0;
  flightstreamInstance->ccsGenSurface = NULL;

  flightstreamInstance->numCCSLiftSurface = 0;
  flightstreamInstance->ccsLiftSurface = NULL;

  flightstreamInstance->numCCSControl = 0;
  flightstreamInstance->ccsControl = NULL;

  flightstreamInstance->numCCSRevolution = 0;
  flightstreamInstance->ccsRevolution = NULL;

  flightstreamInstance->numCCSAnnular = 0;
  flightstreamInstance->ccsAnnular = NULL;

  // Container for attribute to index map
  (void)initiate_mapAttrToIndexStruct(&flightstreamInstance->groupMap);

  // Design information
  initiate_cfdDesignStruct(&flightstreamInstance->design);

  // Units
  initiate_cfdUnitsStruct(&flightstreamInstance->units);
  flightstreamInstance->lengthScaleFactor = 1.0;

  flightstreamInstance->meshRefIn = NULL;
  aim_initMeshRef(&flightstreamInstance->meshRefMorph, aimUnknownMeshType);
  aim_initMeshRef(&flightstreamInstance->meshRefDisp, aimUnknownMeshType);

  return CAPS_SUCCESS;
}


static int destroy_aimStorage(aimStorage *flightstreamInstance)
{
  int i;

  aim_freeValue(&flightstreamInstance->Alpha);
  aim_freeValue(&flightstreamInstance->Beta);
  aim_freeValue(&flightstreamInstance->Mach);
  aim_freeValue(&flightstreamInstance->Velocity);
  aim_freeValue(&flightstreamInstance->Density);
  aim_freeValue(&flightstreamInstance->Pressure);
  aim_freeValue(&flightstreamInstance->SonicSpeed);
  aim_freeValue(&flightstreamInstance->Temperature);
  aim_freeValue(&flightstreamInstance->Viscosity);
  aim_freeValue(&flightstreamInstance->Cx);
  aim_freeValue(&flightstreamInstance->Cy);
  aim_freeValue(&flightstreamInstance->Cz);
  aim_freeValue(&flightstreamInstance->CL);
  aim_freeValue(&flightstreamInstance->CDi);
  aim_freeValue(&flightstreamInstance->CDo);
  aim_freeValue(&flightstreamInstance->CMx);
  aim_freeValue(&flightstreamInstance->CMy);
  aim_freeValue(&flightstreamInstance->CMz);

  // Free general surfaces
  if (flightstreamInstance->ccsGenSurface != NULL) {
    for (i = 0; i < flightstreamInstance->numCCSGenSurface; i++) {
      destroy_vlmSurfaceStruct(&flightstreamInstance->ccsGenSurface[i]);
    }
    AIM_FREE(flightstreamInstance->ccsGenSurface);
    flightstreamInstance->numCCSGenSurface = 0;
  }

  // Free lifting surfaces
  if (flightstreamInstance->ccsLiftSurface != NULL) {
    for (i = 0; i < flightstreamInstance->numCCSLiftSurface; i++) {
      destroy_vlmSurfaceStruct(&flightstreamInstance->ccsLiftSurface[i]);
    }
    AIM_FREE(flightstreamInstance->ccsLiftSurface);
    flightstreamInstance->numCCSLiftSurface = 0;
  }

  // Free controls
  if (flightstreamInstance->ccsControl != NULL) {
    for (i = 0; i < flightstreamInstance->numCCSControl; i++) {
      destroy_vlmControlStruct(&flightstreamInstance->ccsControl[i]);
    }
    AIM_FREE(flightstreamInstance->ccsControl);
    flightstreamInstance->numCCSControl = 0;
  }

  // Free Revolution
  if (flightstreamInstance->ccsRevolution != NULL) {
    for (i = 0; i < flightstreamInstance->numCCSRevolution; i++) {
      destroy_vlmSurfaceStruct(&flightstreamInstance->ccsRevolution[i]);
    }
    AIM_FREE(flightstreamInstance->ccsRevolution);
    flightstreamInstance->numCCSRevolution = 0;
  }

  // Free Annular
  if (flightstreamInstance->ccsAnnular != NULL) {
    for (i = 0; i < flightstreamInstance->numCCSAnnular; i++) {
      destroy_vlmSurfaceStruct(&flightstreamInstance->ccsAnnular[i]);
    }
    AIM_FREE(flightstreamInstance->ccsAnnular);
    flightstreamInstance->numCCSAnnular = 0;
  }

  // Attribute to index map
  (void) destroy_mapAttrToIndexStruct(&flightstreamInstance->groupMap);

  // Design information
  (void) destroy_cfdDesignStruct(&flightstreamInstance->design);

  // Units
  destroy_cfdUnitsStruct(&flightstreamInstance->units);
  flightstreamInstance->lengthScaleFactor = 1.0;

  // Surface Mesh
  aim_freeMeshRef(&flightstreamInstance->meshRefMorph);
  aim_freeMeshRef(&flightstreamInstance->meshRefDisp);
  flightstreamInstance->meshRefIn = NULL;

  initialize_aimStorage(flightstreamInstance);

  return CAPS_SUCCESS;
}


// Write out the airfoil cross-section given an ego body
static int
ccs_writeSection(void *aimInfo,
                 FILE *fp,
                 double lengthScaleFactor,
                 const vlmSectionStruct *vlmSection)
{
  int status; // Function return status

  int i; // Indexing

  int numPoint = 101;

  ego tess=NULL;
  double *xCoord=NULL, *yCoord=NULL, *zCoord=NULL;

  if (vlmSection->type == vlmSecNode) {

    fprintf(fp, "CrossSection;%+16.12e;%+16.12e;%+16.12e\n",
            lengthScaleFactor*vlmSection->xyzLE[0],
            lengthScaleFactor*vlmSection->xyzLE[1],
            lengthScaleFactor*vlmSection->xyzLE[2]);

  } else {
    status = vlm_getSectionCoord(aimInfo,
                                 vlmSection,
                                 0, //normalize,
                                 numPoint,
                                 &xCoord,
                                 &yCoord,
                                 &zCoord,
                                 &tess);
    AIM_STATUS(aimInfo, status);
    AIM_NOTNULL(xCoord, aimInfo, status);
    AIM_NOTNULL(yCoord, aimInfo, status);
    AIM_NOTNULL(zCoord, aimInfo, status);

    fprintf(fp, "CrossSection");
    for( i = 0; i < numPoint; i++) {
      fprintf(fp, ";%+16.12e;%+16.12e;%+16.12e",
              lengthScaleFactor*xCoord[i],
              lengthScaleFactor*yCoord[i],
              lengthScaleFactor*zCoord[i]);
    }
    fprintf(fp, "\n");
  }

  status = CAPS_SUCCESS;

cleanup:
  AIM_FREE(xCoord);
  AIM_FREE(yCoord);
  AIM_FREE(zCoord);

/*@-nullpass@*/
  EG_deleteObject(tess);
/*@+nullpass@*/

  return status;
}

typedef struct {

  char *name; // control surface name

  vlmSectionStruct *rootSection;
  vlmSectionStruct *tipSection;

  vlmControlStruct *rootControlSurface;
  vlmControlStruct *tipControlSurface;

} fsControlStruct;

static void
ccs_Meshing(FILE *fp, const vlmSurfaceStruct *surface)
{
  char nptsS[20];
  char growth_typeS[20];
  char growth_rateS[20];
  char periodicityS[20];

  if (surface->Nchord > 0)
    snprintf(nptsS, 20, "%d", surface->Nchord);
  else
    strcpy(nptsS, "auto");

  if (surface->Cgrowth_type > 0)
    snprintf(growth_typeS, 20, "%d", surface->Cgrowth_type);
  else
    strcpy(growth_typeS, "auto");

  if (surface->Cgrowth_rate > 0)
    snprintf(growth_rateS, 20, "%16.12e", surface->Cgrowth_rate);
  else
    strcpy(growth_rateS, "auto");

  if (surface->Cperiodicity > 0)
    snprintf(periodicityS, 20, "%d", surface->Cperiodicity);
  else
    strcpy(periodicityS, "auto");

  //Mesh_U;u_pts;growth_type;growth_rate;periodicity
  fprintf(fp, "Mesh_U;%s;%s;%s;%s\n",
          nptsS, growth_typeS, growth_rateS, periodicityS);


  if (surface->NspanTotal > 0)
    snprintf(nptsS, 20, "%d", surface->NspanTotal);
  else
    strcpy(nptsS, "auto");

  if (surface->NspanSection > 0)
    snprintf(nptsS, 20, "%d", surface->NspanSection*(surface->numSection-1));
  else
    strcpy(nptsS, "auto");

  if (surface->Sgrowth_type > 0)
    snprintf(growth_typeS, 20, "%d", surface->Sgrowth_type);
  else
    strcpy(growth_typeS, "auto");

  if (surface->Sgrowth_rate > 0)
    snprintf(growth_rateS, 20, "%16.12e", surface->Sgrowth_rate);
  else
    strcpy(growth_rateS, "auto");

  if (surface->Speriodicity > 0)
    snprintf(periodicityS, 20, "%d", surface->Speriodicity);
  else
    strcpy(periodicityS, "auto");

  //Mesh_V;v_pts;growth_type;growth_rate;periodicity
  fprintf(fp, "Mesh_V;%s;%s;%s;%s\n",
          nptsS, growth_typeS, growth_rateS, periodicityS);
}


static int
ccs_GeneralSurface(void *aimInfo, FILE *fp, double lengthScaleFactor, const vlmSurfaceStruct *surface,
                   int *iBoundary,
                   int *numWTerm, int **wtermSurf)
{
  int status = CAPS_SUCCESS;
  int i, sectionIndex;

  fprintf(fp, "Component;%s\n", surface->name);
  fprintf(fp, "LiftingSurface;false\n");

  ccs_Meshing(fp, surface);

  // write out sections
  for (i = 0; i < surface->numSection; i++) {
    sectionIndex = surface->vlmSection[i].sectionIndex;
    status = ccs_writeSection(aimInfo, fp, lengthScaleFactor, &surface->vlmSection[sectionIndex]);
    AIM_STATUS(aimInfo, status);
  }
  fprintf(fp, "\n");

  AIM_REALL((*wtermSurf), (*numWTerm) + 1, int, aimInfo, status);
  (*wtermSurf)[(*numWTerm)++] = (*iBoundary)++;

  status = CAPS_SUCCESS;
cleanup:

  return status;
}


static int
ccs_addControl(void *aimInfo,
               vlmSectionStruct *rootSection,
               vlmSectionStruct *tipSection,
               vlmControlStruct *rootControlSurface,
               vlmControlStruct *tipControlSurface,
               int *numControl,
               fsControlStruct **controlList)
{
  int status;
  int i;
  fsControlStruct *control = NULL;

  for (i = 0; i < (*numControl); i++) {
    if (strcmp(rootControlSurface->name, (*controlList)[i].name) == 0) {
      control = &(*controlList)[i];
      break;
    }
  }
  if (control == NULL) {
    AIM_REALL((*controlList), (*numControl)+1, fsControlStruct, aimInfo, status);
    control = &(*controlList)[(*numControl)];
    (*numControl)++;

    control->name = NULL;
    control->rootSection = rootSection;
    control->tipSection = tipSection;
    control->rootControlSurface = rootControlSurface;
    control->tipControlSurface = tipControlSurface;

    AIM_STRDUP(control->name, rootControlSurface->name, aimInfo, status);
  }

  AIM_NOTNULL(control, aimInfo, status);
/*@-kepttrans@*/
  control->tipSection        = tipSection;
  control->tipControlSurface = tipControlSurface;
/*@+kepttrans@*/

  status = CAPS_SUCCESS;
cleanup:
  return status;
}


static int
ccs_LiftingSurface(void *aimInfo, FILE *fp,
                   double lengthScaleFactor,
                   const vlmSurfaceStruct *surface,
                   int *iBoundary,
                   int *numSharp, int **sharpSurf,
                   int *numProx, int **proxSurf,
                   int *numBase, int **baseSurf)
{
  int status = CAPS_SUCCESS;
  int i, j, k, sectionIndex;
  int found;
  double *xyzRoot, *xyzTip, *xyzLE;
  double spanSurf, spanCntrl, v[2], u[2][2];
  double hinge_height, slot_gap_percentage;

  int nSharp = 0;
  int nProx = 0;
  int nBase = 0;
  int liftSurfBase = (int)true;
  int bluntTE = (int)false;

  int numControl = 0;
  fsControlStruct *controlList = NULL;

  int *fullSpanControl = NULL;

  vlmSectionStruct *rootSection;
  vlmSectionStruct *tipSection;
  vlmControlStruct *rootControlSurface;
  vlmControlStruct *tipControlSurface;

  // Compute the span
  sectionIndex = surface->vlmSection[0].sectionIndex;
  xyzRoot = surface->vlmSection[sectionIndex].xyzLE;

  sectionIndex = surface->vlmSection[surface->numSection-1].sectionIndex;
  xyzTip = surface->vlmSection[sectionIndex].xyzLE;

  spanSurf = sqrt(pow(xyzTip[1] - xyzRoot[1],2) + pow(xyzTip[2] - xyzRoot[2],2));

  fprintf(fp, "Component;%s\n", surface->name);
  fprintf(fp, "LiftingSurface;true\n");
  fprintf(fp, "Parameter;V_Loft_C0\n"); // Assume spanwise C0 for now

  // gather control surfaces
  for (i = 0; i < surface->numSection-1; i++) {

    sectionIndex = surface->vlmSection[i].sectionIndex;
    rootSection = &surface->vlmSection[sectionIndex];

    sectionIndex = surface->vlmSection[i+1].sectionIndex;
    tipSection = &surface->vlmSection[sectionIndex];

    for (j = 0; j < rootSection->numControl; j++) {

      rootControlSurface = &rootSection->vlmControl[j];

      // find matching control surface info on tip section
      found = (int) false;
      tipControlSurface = NULL;
      for (k = 0; k < tipSection->numControl; k++) {
        if (strcmp(rootControlSurface->name, tipSection->vlmControl[k].name) == 0) {
          tipControlSurface = &tipSection->vlmControl[k];
          found = (int) true;
          break;
        }
      }
      if (!found) continue;

      AIM_NOTNULL(rootControlSurface, aimInfo, status);
      AIM_NOTNULL(tipControlSurface, aimInfo, status);
      status = ccs_addControl(aimInfo,
                              rootSection,
                              tipSection,
                              rootControlSurface,
                              tipControlSurface,
                              &numControl,
                              &controlList);
      AIM_STATUS(aimInfo, status);
    }
  }

  AIM_ALLOC(fullSpanControl, numControl, int, aimInfo, status);
  for (i = 0; i < numControl; i++) fullSpanControl[i] = (int)false;

  // write out control surfaces
  for (i = 0; i < numControl; i++) {
    AIM_NOTNULL(controlList, aimInfo, status);

    rootSection = controlList[i].rootSection;
    tipSection  = controlList[i].tipSection;

    rootControlSurface = controlList[i].rootControlSurface;
    tipControlSurface  = controlList[i].tipControlSurface;

    if (rootControlSurface->leOrTe == 0) {
      // Leading edge control surface
      AIM_ERROR(aimInfo, "FligthStream currently does not support leading edge devices.");
      status = CAPS_BADVALUE;
      goto cleanup;
    }

    // Trailing edge control surface
    xyzLE = rootSection->xyzLE;
    spanCntrl = sqrt(pow(xyzLE[1] - xyzRoot[1],2) + pow(xyzLE[2] - xyzRoot[2],2));
    v[0] = spanCntrl/spanSurf;

    xyzLE = tipSection->xyzLE;
    spanCntrl = sqrt(pow(xyzLE[1] - xyzRoot[1],2) + pow(xyzLE[2] - xyzRoot[2],2));
    v[1] = spanCntrl/spanSurf;

    hinge_height = rootControlSurface->hingeHeight;
    slot_gap_percentage = rootControlSurface->slotGap*100;

    // Don't allow a slot gap for full span control surfaces
    if (v[0] < 0.001 && v[1] > 0.999) {
      slot_gap_percentage = 0;
      liftSurfBase = (int)false;
      fullSpanControl[i] = (int)true;
    } else {
      nProx++; // the control surface boundary
      nProx++; // control surface side boundary
    }

    // FlightStream defines u from TE->LE
    u[0][0] = 1-rootControlSurface->percentChord;
    u[0][1] = 1-tipControlSurface->percentChord;

//    u[1][0] = 1-rootControlSurface->percentChord[1];
//    u[1][1] = 1-tipControlSurface->percentChord[1];

//    if (rootControlSurface->percentChord[1] < 0) {
    // Standard control surface
    // Subsurface;Name;v0;v1;u0;u1;hinge_height;rotation_angle;slot_gap_percentage
      fprintf(fp, "Subsurface;%s;%16.12f;%16.12f;%16.12f;%16.12f;%16.12f;%16.12f;%16.12f\n",
              controlList[i].name, v[0], v[1], u[0][0], u[0][1], hinge_height, rootControlSurface->deflectionAngle, slot_gap_percentage);

      if (fabs(rootControlSurface->deflectionAngle) <= 10) {
        // Add wake termination nodes for small deflection angles
        fprintf(fp, "Parameter;Subsurface_Wake_Nodes;%d\n", i+1);
      }
//    }

    // Morphing control surface
    // MorphCS;Name;v0;v1;u0;u1

//    else {
//
//      int cove_type = 1;
//      const char* want_flap = "true";
//      double flap_gap = 0.1;
//
//      // Flap/Cove
//      // Flap_cove;Name;v0;v1;u00;u01;u10;u11;cove_type;want_flap;flap_gap;slot_gap_percentage
//      fprintf(fp, "Flap_cove;%s;%16.12f;%16.12f;%16.12f;%16.12f;%16.12f;%16.12f;%d;%s;%16.12f;%16.12f\n",
//              controlList[i].name, v[0], v[1], u[0][0], u[1][0], u[0][1], u[1][1], cove_type, want_flap, flap_gap, slot_gap_percentage);
//    }
  }

  ccs_Meshing(fp, surface);

  // Check for blunt trailing edges
  for (i = 0; i < surface->numSection; i++) {
    if (surface->vlmSection[i].teClass == EDGE) {
      if (surface->blendTE == (int)true) {
        fprintf(fp, "Parameter;Blend_trailing_edges\n");
      } else {
        fprintf(fp, "Parameter;Blunt_trailing_edges\n");
        bluntTE = (int)true;
      }
      fprintf(fp, "Parameter;Open_Cross_Sections\n");
      break;
    }
  }
  if (bluntTE == (int)false) {
    nSharp += numControl; // control surface TE
    if (numControl > 0 && fullSpanControl[0] == (int)false)
      nSharp++;           // lifting surface TE
    fprintf(fp, "Parameter;Mark_trailing_edges\n");
  }

  if (bluntTE == (int)true) {
    nBase += numControl; // control surface base
    if (numControl > 0 && fullSpanControl[0] == (int)false)
      nBase++;           // lifting surface base
  }

  // add sharpSurf bounds
  if (nSharp > 0) {
    AIM_REALL((*sharpSurf), (*numSharp) + nSharp, int, aimInfo, status);
  }
  // add proxSurf bounds
  if (nProx > 0) {
    AIM_REALL((*proxSurf), (*numProx) + nProx, int, aimInfo, status);
  }
  // add baseSurf bounds
  if (nBase > 0) {
    AIM_REALL((*baseSurf), (*numBase) + nBase, int, aimInfo, status);
  }

  // lifting surface
  if (nSharp > 0 && numControl > 0 && fullSpanControl[0] == (int)false)
    (*sharpSurf)[(*numSharp)++] = (*iBoundary);
  (*iBoundary)++;

  // add control surfaces
  for (i = 0; i < numControl; i++) {
    if (nSharp > 0)
      (*sharpSurf)[(*numSharp)++] = (*iBoundary);
    if (nProx > 0)
      (*proxSurf)[(*numProx)+i] = (*iBoundary);
    (*iBoundary)++;
  }

  if (bluntTE == (int)true) {
    j = 0;
    if (nBase > 0 && liftSurfBase == (int)true) {
      (*baseSurf)[(*numBase)] = (*iBoundary);
      (*iBoundary)++; // lifting surface base
      j = 1;
    }
    for (i = 0; i < numControl; i++) {
      (*baseSurf)[(*numBase)+j+i] = (*iBoundary);
      (*iBoundary)++;                    // control surface base
    }
  }

  // control surface side boundaries
  for (i = 0; i < numControl; i++) {
    if (fullSpanControl[i] == (int)false) {
      (*proxSurf)[(*numProx)+numControl+i] = (*iBoundary);
      (*iBoundary)++;
    }
  }
  (*numProx) += nProx;
  (*numBase) += nBase;

  // write out sections
  for (i = 0; i < surface->numSection; i++) {
    sectionIndex = surface->vlmSection[i].sectionIndex;
    status = ccs_writeSection(aimInfo, fp, lengthScaleFactor, &surface->vlmSection[sectionIndex]);
    AIM_STATUS(aimInfo, status);
  }
  fprintf(fp, "\n");

  status = CAPS_SUCCESS;
cleanup:

  if (controlList != NULL) {
    for (i = 0; i < numControl; i++) {
      AIM_FREE(controlList[i].name);
    }
    AIM_FREE(controlList);
  }
  AIM_FREE(fullSpanControl);

  return status;
}


static int
ccs_Revolution(void *aimInfo, FILE *fp,
               double lengthScaleFactor,
               const vlmSurfaceStruct *surface,
               int *iBoundary,
               int *numSharp, int **sharpSurf,
               int *numWTerm, int **wtermSurf)
{
  int i, status = CAPS_SUCCESS;
  int bluntTE = (int)false;

  if (surface->numSection != 1) {
    AIM_ERROR(aimInfo, "Body of revolution '%s' should only have one section!", surface->name);
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  fprintf(fp, "Component;%s\n", surface->name);
  fprintf(fp, "LiftingSurface;false\n");

  fprintf(fp, "Parameter;RevolveBody;%16.12e;%16.12e;%16.12e;%16.12e;%16.12e;%16.12e;%16.12e;%16.12e\n",
          surface->x0[0], surface->x0[1], surface->x0[2],
         -surface->axis0[0], -surface->axis0[1], -surface->axis0[2], // Negate axis to avoid inverted geometry
          surface->startAngle, surface->endAngle);

  // Check for blunt trailing edges
  for (i = 0; i < surface->numSection; i++) {
    if (surface->vlmSection[i].teClass == EDGE) {
      fprintf(fp, "Parameter;Blunt_trailing_edges\n");
      fprintf(fp, "Parameter;Open_Cross_Sections\n");

      AIM_ERROR(aimInfo, "Revolution surface '%s' must have Sharp trailing edge!", surface->name);
      status = CAPS_BADVALUE;
      goto cleanup;

      //bluntTE = (int)true;
      //break;
    }
  }
  if (bluntTE == (int)false) {
    fprintf(fp, "Parameter;Mark_trailing_edges\n");
    AIM_REALL((*sharpSurf), (*numSharp) + 1, int, aimInfo, status);
    (*sharpSurf)[(*numSharp)++] = (*iBoundary);
  }

  ccs_Meshing(fp, surface);

  // write out section
  status = ccs_writeSection(aimInfo, fp, lengthScaleFactor, &surface->vlmSection[0]);
  AIM_STATUS(aimInfo, status);
  fprintf(fp, "\n");


  AIM_REALL((*wtermSurf), (*numWTerm) + 1, int, aimInfo, status);
  (*wtermSurf)[(*numWTerm)++] = (*iBoundary)++;

  status = CAPS_SUCCESS;
cleanup:

  return status;
}


static int
ccs_Annular(void *aimInfo, FILE *fp,
            double lengthScaleFactor,
            const vlmSurfaceStruct *surface,
            int *iBoundary,
            int *numSharp, int **sharpSurf,
            int *numWTerm, int **wtermSurf)
{
  int status = CAPS_SUCCESS;
  int i, sectionIndex;
  int bluntTE = (int)false;

  fprintf(fp, "Component;%s\n", surface->name);
  fprintf(fp, "LiftingSurface;false\n");

//  fprintf(fp, "Parameter;Nacelle;%16.12e;%16.12e;%16.12e;%16.12e;%16.12e;%16.12e\n",
//          surface->x0[0], surface->x0[1], surface->x0[2],
//          surface->axis[0], surface->axis[1], surface->axis[2]);

  fprintf(fp, "Parameter;Nacelle;0;0;0;0;0;0;\n");
  fprintf(fp, "Parameter;V_Loft_C2\n"); // Assume annular C2 for now

  // Check for blunt trailing edges
  for (i = 0; i < surface->numSection; i++) {
    if (surface->vlmSection[i].teClass == EDGE) {
      fprintf(fp, "Parameter;Blunt_trailing_edges\n");
      fprintf(fp, "Parameter;Open_Cross_Sections\n");

      AIM_ERROR(aimInfo, "Annular surface '%s' must have Sharp trailing edge!", surface->name);
      status = CAPS_BADVALUE;
      goto cleanup;

      //bluntTE = (int)true;
      //break;
    }
  }
  if (bluntTE == (int)false) {
    fprintf(fp, "Parameter;Mark_trailing_edges\n");
    AIM_REALL((*sharpSurf), (*numSharp) + 1, int, aimInfo, status);
    (*sharpSurf)[(*numSharp)++] = (*iBoundary);
  }

  ccs_Meshing(fp, surface);

  // write out sections
  for (i = 0; i < surface->numSection; i++) {
    sectionIndex = surface->vlmSection[i].sectionIndex;
    status = ccs_writeSection(aimInfo, fp, lengthScaleFactor, &surface->vlmSection[sectionIndex]);
    AIM_STATUS(aimInfo, status);
  }
  fprintf(fp, "\n");

  AIM_REALL((*wtermSurf), (*numWTerm) + 1, int, aimInfo, status);
  (*wtermSurf)[(*numWTerm)++] = (*iBoundary)++;

  status = CAPS_SUCCESS;
cleanup:

  return status;
}


static int
writeCCS(void *aimInfo, FILE *fp, const aimStorage *flightstreamInstance,
         int *numSharp, int **sharpSurf,
         int *numProx, int **proxSurf,
         int *numBase, int **baseSurf,
         int *numWTerm, int **wtermSurf)
{
  int status = CAPS_SUCCESS;
  int i;
  int iBoundary = 1;

  fprintf(fp, "Aircraft;CAPS_Airplane\n");
  fprintf(fp, "Parameter;Units;%s\n", flightstreamInstance->units.length);
  fprintf(fp, "\n");

  for (i = 0; i < flightstreamInstance->numCCSLiftSurface; i++) {
    status = ccs_LiftingSurface(aimInfo, fp,
                                flightstreamInstance->lengthScaleFactor,
                                &flightstreamInstance->ccsLiftSurface[i],
                                &iBoundary,
                                numSharp, sharpSurf,
                                numProx, proxSurf,
                                numBase, baseSurf);
    AIM_STATUS(aimInfo, status);
  }

  for (i = 0; i < flightstreamInstance->numCCSGenSurface; i++) {
    status = ccs_GeneralSurface(aimInfo, fp,
                                flightstreamInstance->lengthScaleFactor,
                                &flightstreamInstance->ccsGenSurface[i],
                                &iBoundary,
                                numWTerm, wtermSurf);
    AIM_STATUS(aimInfo, status);
  }

  for (i = 0; i < flightstreamInstance->numCCSRevolution; i++) {
    status = ccs_Revolution(aimInfo, fp,
                            flightstreamInstance->lengthScaleFactor,
                            &flightstreamInstance->ccsRevolution[i],
                            &iBoundary,
                            numSharp, sharpSurf,
                            numWTerm, wtermSurf);
    AIM_STATUS(aimInfo, status);
  }

  for (i = 0; i < flightstreamInstance->numCCSAnnular; i++) {
    status = ccs_Annular(aimInfo, fp,
                         flightstreamInstance->lengthScaleFactor,
                         &flightstreamInstance->ccsAnnular[i],
                         &iBoundary,
                         numSharp, sharpSurf,
                         numWTerm, wtermSurf);
    AIM_STATUS(aimInfo, status);
  }

  status = CAPS_SUCCESS;
cleanup:
  return status;
}


// Get the loads from a capsTuple
static int
_getFluid_Properties(void *aimInfo,
                     int numPropTuple,
                     capsTuple propTuple[],
                     aimStorage *flightstreamInstance) {

  /*! \page flightstreamFLUID_PROPERTIES FlightStream Fluid Properties
   * Structure for the load tuple  = ("Property", "Value").
   */

  int status; //Function return

  int i; // Indexing

  char *keyValue = NULL; // Key values from tuple searches
  const char *keyWord = NULL; // Key words to find in the tuples

  printf("\nGetting Fluid Properties.......\n");

  if (numPropTuple != 5) {
    AIM_ERROR(aimInfo, "All 5 fluid properties must be specified. Only %d provided:");
    for (i = 0; i < numPropTuple; i++)
      AIM_ADDLINE(aimInfo, "%s", propTuple[i].name);
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  // Loop through tuples and get each property
  for (i = 0; i < numPropTuple; i++) {

    /*! \page flightstreamFLUID_PROPERTIES
     *
     * <ul>
     *  <li> <B>density = "(no default)"</B> </li> <br>
     *  Reference density
     * </ul>
     *
     */
    keyWord = "density";
    if (strcasecmp(propTuple[i].name, keyWord) == 0) {
      status = string_toDoubleUnits(aimInfo, propTuple[i].value, "kg/m^3", &flightstreamInstance->rhoff);
      AIM_STATUS(aimInfo, status, "while parsing '%s'", propTuple[i].value);
    }

    /*! \page flightstreamFLUID_PROPERTIES
     *
     * <ul>
     *  <li> <B>pressure = "(no default)"</B> </li> <br>
     *  Reference pressure
     * </ul>
     *
     */
    keyWord = "pressure";
    if (strcasecmp(propTuple[i].name, keyWord) == 0) {
      status = string_toDoubleUnits(aimInfo, propTuple[i].value, "Pa", &flightstreamInstance->pff);
      AIM_STATUS(aimInfo, status, "while parsing '%s'", propTuple[i].value);
    }

    /*! \page flightstreamFLUID_PROPERTIES
     *
     * <ul>
     *  <li> <B>sonic_velocity = "(no default)"</B> </li> <br>
     *  Reference speed of sound
     * </ul>
     *
     */
    keyWord = "sonic_velocity";
    if (strcasecmp(propTuple[i].name, keyWord) == 0) {
      status = string_toDoubleUnits(aimInfo, propTuple[i].value, "m/s", &flightstreamInstance->sonicspeedff);
      AIM_STATUS(aimInfo, status, "while parsing '%s'", propTuple[i].value);
    }

    /*! \page flightstreamFLUID_PROPERTIES
     *
     * <ul>
     *  <li> <B>temperature = "(no default)"</B> </li> <br>
     *  Reference temperature
     * </ul>
     *
     */
    keyWord = "temperature";
    if (strcasecmp(propTuple[i].name, keyWord) == 0) {
      status = string_toDoubleUnits(aimInfo, propTuple[i].value, "K", &flightstreamInstance->tff);
      AIM_STATUS(aimInfo, status, "while parsing '%s'", propTuple[i].value);
    }

    /*! \page flightstreamFLUID_PROPERTIES
     *
     * <ul>
     *  <li> <B>viscosity = "(no default)"</B> </li> <br>
     *  Reference viscosity
     * </ul>
     *
     */
    keyWord = "viscosity";
    if (strcasecmp(propTuple[i].name, keyWord) == 0) {
      status = string_toDoubleUnits(aimInfo, propTuple[i].value, "Pa*s", &flightstreamInstance->muff);
      AIM_STATUS(aimInfo, status, "while parsing '%s'", propTuple[i].value);
    }
  }

  if (flightstreamInstance->rhoff <= 0) {
    AIM_ERROR(aimInfo, "Positive 'density' must be specified. Provided Fluid_Properties:");
    for (i = 0; i < numPropTuple; i++)
      AIM_ADDLINE(aimInfo, "%s -> %s", propTuple[i].name, propTuple[i].value);
    status = CAPS_BADVALUE;
    goto cleanup;
  }
  if (flightstreamInstance->pff <= 0) {
    AIM_ERROR(aimInfo, "Positive 'pressure' must be specified. Provided Fluid_Properties:");
    for (i = 0; i < numPropTuple; i++)
      AIM_ADDLINE(aimInfo, "%s -> %s", propTuple[i].name, propTuple[i].value);
    status = CAPS_BADVALUE;
    goto cleanup;
  }
  if (flightstreamInstance->sonicspeedff <= 0) {
    AIM_ERROR(aimInfo, "Positive 'sonic_velocity' must be specified. Provided Fluid_Properties:");
    for (i = 0; i < numPropTuple; i++)
      AIM_ADDLINE(aimInfo, "%s -> %s", propTuple[i].name, propTuple[i].value);
    status = CAPS_BADVALUE;
    goto cleanup;
  }
  if (flightstreamInstance->tff <= 0) {
    AIM_ERROR(aimInfo, "Positive 'temperature' must be specified. Provided Fluid_Properties:");
    for (i = 0; i < numPropTuple; i++)
      AIM_ADDLINE(aimInfo, "%s -> %s", propTuple[i].name, propTuple[i].value);
    status = CAPS_BADVALUE;
    goto cleanup;
  }
  if (flightstreamInstance->muff <= 0) {
    AIM_ERROR(aimInfo, "Positive 'viscosity' must be specified. Provided Fluid_Properties:");
    for (i = 0; i < numPropTuple; i++)
      AIM_ADDLINE(aimInfo, "%s -> %s", propTuple[i].name, propTuple[i].value);
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  printf("\tDone getting Fluid Properties\n");
  status = CAPS_SUCCESS;
cleanup:
  AIM_FREE(keyValue);

  return status;
}


// Get FlightStream data transfer files
static int
fs_dataTransfer(void *aimInfo,
                capsValue  aimInputs[],
                const aimStorage *fsInstance)
{

    /*! \page dataTransferFLIGHTSTREAM FlightStream Data Transfer
     *
     * \section dataToCart3D Data transfer to FlightStream (FieldIn)
     *
     * <ul>
     *  <li> <B>"Displacement"</B> </li> <br>
     *   Retrieves nodal displacements (as from a structural solver)
     *   and updates FlightStream surface mesh.
     * </ul>
     */

    int status; // Function return status
    int i, ibound, iglobal; // Indexing

    // Discrete data transfer variables
    capsDiscr *discr;
    char **boundNames = NULL;
    int numBoundName = 0;
    enum capsdMethod dataTransferMethod;
    int numDataTransferPoint;
    int dataTransferRank;
    double *dataTransferData;
    char *units;

    const aimMeshRef *meshRefIn;

    aimMesh    mesh;

    // Data transfer variable
    double *dxyz = NULL;

    int foundDisplacement = (int) false;

    mesh.meshRef = NULL;
    mesh.meshData = NULL;

    status = aim_getBounds(aimInfo, &numBoundName, &boundNames);
    if (status == CAPS_NOTFOUND) return CAPS_SUCCESS;
    AIM_STATUS(aimInfo, status);

    // Get mesh
    meshRefIn = (aimMeshRef *) aimInputs[inSurface_Mesh-1].vals.AIMptr;

    if ( aimInputs[inMesh_Morph-1].vals.integer == (int) true &&
        meshRefIn == NULL) { // If we are mighty morphing
      meshRefIn = &fsInstance->meshRefMorph;
    }

    foundDisplacement = (int) false;
    for (ibound = 0; ibound < numBoundName; ibound++) {
      AIM_NOTNULL(boundNames, aimInfo, status);

      status = aim_getDiscr(aimInfo, boundNames[ibound], &discr);
      if (status != CAPS_SUCCESS) continue;

      status = aim_getDataSet(discr,
                              "Displacement",
                              &dataTransferMethod,
                              &numDataTransferPoint,
                              &dataTransferRank,
                              &dataTransferData,
                              &units);
      if (status != CAPS_SUCCESS) continue;

      foundDisplacement = (int) true;

      // Is the rank correct?
      if (dataTransferRank != 3) {
        AIM_ERROR(aimInfo, "Displacement transfer data found however rank is %d not 3!!!!", dataTransferRank);
        status = CAPS_BADRANK;
        goto cleanup;
      }

    } // Loop through bound names

    if (foundDisplacement != (int) true ) {
        status = CAPS_SUCCESS;
        goto cleanup;
    }

    status = aim_freeMeshRef(&((aimStorage *)fsInstance)->meshRefDisp);
    AIM_STATUS(aimInfo, status);

    /* Create a local mesh file name */
    status = aim_localMeshRef(aimInfo, meshRefIn, &((aimStorage *)fsInstance)->meshRefDisp);
    AIM_STATUS(aimInfo, status);

    /*@-immediatetrans@*/
    mesh.meshData = NULL;
    mesh.meshRef = &((aimStorage *)fsInstance)->meshRefDisp;
    /*@+immediatetrans@*/

    status = mesh_surfaceMeshData(aimInfo, &fsInstance->groupMap, &mesh);
    AIM_STATUS(aimInfo, status);
    AIM_NOTNULL(mesh.meshData, aimInfo, status);

    /*@-immediatetrans@*/
    ((aimStorage *)fsInstance)->meshRefIn = &((aimStorage *)fsInstance)->meshRefDisp;
    /*@+immediatetrans@*/

    AIM_ALLOC(dxyz, 3*mesh.meshData->nVertex, double, aimInfo, status);
    memset(dxyz, 0, 3*mesh.meshData->nVertex*sizeof(double));

    // gather displacements first to avoid double counting edges/nodes
    for (ibound = 0; ibound < numBoundName; ibound++) {
      AIM_NOTNULL(boundNames, aimInfo, status);

      status = aim_getDiscr(aimInfo, boundNames[ibound], &discr);
      if (status != CAPS_SUCCESS) continue;

      status = aim_getDataSet(discr,
                              "Displacement",
                              &dataTransferMethod,
                              &numDataTransferPoint,
                              &dataTransferRank,
                              &dataTransferData,
                              &units);
      if (status != CAPS_SUCCESS) continue;

      if (numDataTransferPoint != discr->nPoints &&
          numDataTransferPoint > 1) {
        AIM_ERROR(aimInfo, "Developer error!! %d != %d", numDataTransferPoint, discr->nPoints);
        status = CAPS_MISMATCH;
        goto cleanup;
      }

      for (i = 0; i < discr->nPoints; i++) {
        iglobal = discr->tessGlobal[2*i+1];

        if (numDataTransferPoint == 1) {
          // A single point means this is an initialization phase
          status = aim_convert(aimInfo, 3, units, dataTransferData, fsInstance->units.length, &dxyz[3*(iglobal-1)+0]);
          AIM_STATUS(aimInfo, status);
        } else {
          // Apply delta displacements
          status = aim_convert(aimInfo, 3, units, &dataTransferData[3*i+0], fsInstance->units.length, &dxyz[3*(iglobal-1)+0]);
          AIM_STATUS(aimInfo, status);
        }
      }
    } // Loop through bound names

    // Apply the displacements
    for (i = 0; i < mesh.meshData->nVertex; i++) {
      mesh.meshData->verts[i][0] += dxyz[3*i+0];
      mesh.meshData->verts[i][1] += dxyz[3*i+1];
      mesh.meshData->verts[i][2] += dxyz[3*i+2];
    }

    /* write the mesh */
    status = aim_writeMesh(aimInfo, MESHWRITER, fsInstance->units.length, &mesh);
    AIM_STATUS(aimInfo, status);

    status = CAPS_SUCCESS;

// Clean-up
cleanup:
    AIM_FREE(dxyz);
    AIM_FREE(boundNames);

    aim_freeMeshData(mesh.meshData);
    AIM_FREE(mesh.meshData);

    return status;
}

/**********************************************************************/
/* aimInitialize - initialize the AIM                                 */
/**********************************************************************/

int
aimInitialize(int        qFlag,         /* (in)  -1 indiactes query only */
 /*@unused@*/ const char unitSys[],     /* (in)  unit system */
              void       *aimInfo,      /* (in)  AIM context */
              void       **instStore,   /* (out) AIM instance storage */
              int        *major,        /* (out) major version number */
              int        *minor,        /* (out) minor version number */
              int        *nIn,          /* (out) number of inputs to AIM */
              int        *nOut,         /* (out) number of outputs from AIM */
              int        *nFields,      /* (out) number of DataSet fields */
              char       ***fnames,     /* (out) array  of DataSet names */
              int        **franks,      /* (out) array  of DataSet ranks */
              int        **fInOut)      /* (out) array  of field flags */
{

    int status = CAPS_SUCCESS;

    int  *ints=NULL, i;
    char **strs=NULL;
    const char *keyWord;
    char *keyValue = NULL;
    double real = 1;
    cfdUnitsStruct *units=NULL;

    aimStorage *flightstreamInstance = NULL;

    /* -------------------------------------------------------------- */

#ifdef DEBUG
    printf("flightstreamAIM/aimInitialize(qFlag=%d, unitSys=%s)\n", qFlag, unitSys);
#endif

    /* specify the version number */
    *major = 1;
    *minor = 0;

    /* specify the number of analysis input and out "parameters" */
    *nIn     = NUMINPUT;
    *nOut    = NUMOUTPUT;

    /* if this is simply a query, return now */
    if (qFlag == -1) {
        return CAPS_SUCCESS;
    }

    /* specify the field variables this analysis can generate and consume */
    *nFields = 3;

    /* specify the name of each field variable */
    AIM_ALLOC(strs, *nFields, char *, aimInfo, status);
    strs[0]  = EG_strdup("RelPressure");
    strs[1]  = EG_strdup("AbsPressure");
    strs[2]  = EG_strdup("Displacement");
    for (i = 0; i < *nFields; i++)
      if (strs[i] == NULL) {
        status = EGADS_MALLOC;
        goto cleanup;
      }
    *fnames  = strs;

    /* specify the dimension of each field variable */
    AIM_ALLOC(ints, *nFields, int, aimInfo, status);

    ints[0]  = 1;
    ints[1]  = 1;
    ints[2]  = 3;
    *franks   = ints;
    ints = NULL;

    /* specify if a field is an input field or output field */
    AIM_ALLOC(ints, *nFields, int, aimInfo, status);

    ints[0]  = FieldOut;
    ints[1]  = FieldOut;
    ints[2]  = FieldIn;
    *fInOut  = ints;
    ints = NULL;

    // Allocate flightstreamInstance
    AIM_ALLOC(flightstreamInstance, 1, aimStorage, aimInfo, status);

    initialize_aimStorage(flightstreamInstance);

    units = &flightstreamInstance->units;

    *instStore = flightstreamInstance;


    /*! \page aimUnitsFLIGHTSTREAM AIM Units
     *  FlightStream expects units for all inputs, and by default the AIM uses SI units, i.e.
     *   - mass : kg
     *   - length : meter
     *   - time : seconds
     *   - temperature : K
     *  A unit system may be optionally specified during AIM instance initiation to use a different set of base units.
     *  A unit system may be specified via a JSON string dictionary for example:
     *  unitSys = "{"mass": "lb", "length": "feet", "time":"seconds", "temperature": "R"}"
     */
    if (unitSys != NULL) {

      // Do we have a json string?
      if (strncmp( unitSys, "{", 1) != 0) {
        AIM_ERROR(aimInfo, "unitSys ('%s') is expected to be a JSON string dictionary", unitSys);
        return CAPS_BADVALUE;
      }

      /*! \page aimUnitsFLIGHTSTREAM
       *  \section jsonStringFLIGHTSTREAM JSON String Dictionary
       *  The key arguments of the dictionary are described in the following:
       *
       *  <ul>
       *  <li> <B>mass = "None"</B> </li> <br>
       *  Mass units - e.g. "kilogram", "k", "slug", ...
       *  </ul>
       */
      keyWord = "mass";
      status  = search_jsonDictionary(unitSys, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {
        units->mass = string_removeQuotation(keyValue);
        AIM_FREE(keyValue);
        real = 1;
        status = aim_convert(aimInfo, 1, units->mass, &real, "kg", &real);
        AIM_STATUS(aimInfo, status, "unitSys ('%s'): %s is not a %s unit", unitSys, units->mass, keyWord);
      } else {
        AIM_ERROR(aimInfo, "unitSys ('%s') does not contain '%s'", unitSys, keyWord);
        status = CAPS_BADVALUE;
        goto cleanup;
      }

      /*! \page aimUnitsFLIGHTSTREAM
       *  <ul>
       *  <li> <B>length = "None"</B> </li> <br>
       *  Length units - FlightStream support: "inch", "millimeter", "feet", "mile", "meter", "kilometer", "mils", "micron", "centimeter", "microinch"
       *  </ul>
       */
      keyWord = "length";
      status  = search_jsonDictionary(unitSys, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {
        units->length = string_removeQuotation(keyValue);
        AIM_FREE(keyValue);
        real = 1;
        status = aim_convert(aimInfo, 1, units->length, &real, "m", &real);
        AIM_STATUS(aimInfo, status, "unitSys ('%s'): %s is not a %s unit", unitSys, units->length, keyWord);
      } else {
        AIM_ERROR(aimInfo, "unitSys ('%s') does not contain '%s'", unitSys, keyWord);
        status = CAPS_BADVALUE;
        goto cleanup;
      }

      /*! \page aimUnitsFLIGHTSTREAM
       *  <ul>
       *  <li> <B>time = "None"</B> </li> <br>
       *  Time units - e.g. "second", "s", "minute", ...
       *  </ul>
       */
      keyWord = "time";
      status  = search_jsonDictionary(unitSys, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {
        units->time = string_removeQuotation(keyValue);
        AIM_FREE(keyValue);
        real = 1;
        status = aim_convert(aimInfo, 1, units->time, &real, "s", &real);
        AIM_STATUS(aimInfo, status, "unitSys ('%s'): %s is not a %s unit", unitSys, units->time, keyWord);
      } else {
        AIM_ERROR(aimInfo, "unitSys ('%s') does not contain '%s'", unitSys, keyWord);
        status = CAPS_BADVALUE;
        goto cleanup;
      }

      /*! \page aimUnitsFLIGHTSTREAM
       *  <ul>
       *  <li> <B>temperature = "None"</B> </li> <br>
       *  Temperature units - e.g. "Kelvin", "K", "degC", ...
       *  </ul>
       */
      keyWord = "temperature";
      status  = search_jsonDictionary(unitSys, keyWord, &keyValue);
      if (status == CAPS_SUCCESS) {
        units->temperature = string_removeQuotation(keyValue);
        AIM_FREE(keyValue);
        real = 1;
        status = aim_convert(aimInfo, 1, units->temperature, &real, "K", &real);
        AIM_STATUS(aimInfo, status, "unitSys ('%s'): %s is not a %s unit", unitSys, units->temperature, keyWord);
      } else {
        AIM_ERROR(aimInfo, "unitSys ('%s') does not contain '%s'", unitSys, keyWord);
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    } else {

      // default SI units
      AIM_STRDUP(units->mass, "kg", aimInfo, status);
      AIM_STRDUP(units->length, "meter", aimInfo, status);
      AIM_STRDUP(units->time, "seconds", aimInfo, status);
      AIM_STRDUP(units->temperature, "K", aimInfo, status);

    }

    status = cfd_cfdDerivedUnits(aimInfo, units);
    AIM_STATUS(aimInfo, status);

cleanup:
  return status;
}


/**********************************************************************/
/* aimInputs - return information about index'th analysis input       */
/**********************************************************************/

int
aimInputs(
/*@unused@*/void    *instStore,         /* (in)  AIM instance storage */
          void      *aimInfo,           /* (in)  AIM context */
          int       index,              /* (in)  input index (1-nIn] */
          char      **ainame,           /* (out) name of analysis input */
          capsValue *inval)             /* (out) description of analysis input */
{
    int status = CAPS_SUCCESS;
    aimStorage *flightstreamInstance;

    cfdUnitsStruct *units=NULL;

    /* -------------------------------------------------------------- */

#ifdef DEBUG
    printf("flightstreamAIM/aimInputs(index=%d)\n", index);
#endif

    *ainame = NULL;

    /*! \page aimInputsFLIGHTSTREAM AIM Inputs
     * The following list outlines the FlightStream inputs along with their
     * default values available through the AIM interface.
     */

    // entities to be defined for each aimInput
    // default values are in [brackets]

    // *ainame             = EG_strdup("name")
    // inval->type         = Boolean, Integer, Double, String, Tuple, Pointer, DoubleDeriv, or PointerMesh
    // inval->nullVal      = [NotNull], IsNull, NotAllowed, IsPartial
    // inval->units        = [NULL] or EG_strdup("bar")
    // inval->lfixed       = [Fixed] or Change (length is fixed)
    // inval->sfixed       = [Fixed] or Change (shape  is fixed)
    // inval->dim          = [0], 1, or 2 (maximum dimensions)
    // inval->nrow         = [1] or n (only if dim>0)
    // inval->ncol         = [1] or n (only if dim>1)
    // inval->length       = inval->nrow * inval->ncol
    // inval->units        = NULL or EG_strdup("meter") or EG_strdup("degree") or ...
    // inval->vals.real    = number or AIM_ALLOC(inval->vals.reals, inval->length, double, aimInfo, status)
    // inval->vals.integer = number or AIM_ALLOC(inval->vals.integer, inval->length, int, aimInfo, status)
    // inval->vals.string  = NULL or EG_strdup("flightstream_CAPS")


    flightstreamInstance = (aimStorage *) instStore;

    if (flightstreamInstance != NULL) units = &flightstreamInstance->units;

    // FlightStream Inputs
    if (index == inProj_Name) {
        *ainame             = EG_strdup("Proj_Name");
        inval->type         = String;
        inval->nullVal      = NotNull;
        AIM_STRDUP(inval->vals.string, "flightstream_CAPS", aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> ProjName = "flightstream_CAPS"</B> <br>
         * Name for files generated by fightstream AIM.
         */

    } else if (index == inFlightStream) {
        *ainame             = EG_strdup("FlightStream");
        inval->type         = String;
        inval->nullVal      = NotNull;
        AIM_STRDUP(inval->vals.string, "FlightStream", aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> FlightStream = "FlightStream"</B> <br>
         * The name of the FlightStream executable. May include full path.
         */

    } else if (index == inMach) {
        *ainame             = EG_strdup("Mach"); // Mach number
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = NULL;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Mach = NULL </B> <br>
         *  Mach number
         */

    } else if (index == inAlpha) {
        *ainame             = EG_strdup("Alpha");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = EG_strdup("degree");
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Alpha = NULL </B> <br>
         *  Angle of attack
         */

    } else if (index == inBeta) {
        *ainame             = EG_strdup("Beta");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = EG_strdup("degree");
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Beta = NULL </B> <br>
         *  Sideslip angle
         */

    } else if (index == inSweepRangeMach) {
        *ainame             = EG_strdup("SweepRangeMach");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = NULL;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->nrow         = 3;
        AIM_ALLOC(inval->vals.reals, inval->nrow, double, aimInfo, status);
        inval->vals.reals[0] = 0;
        inval->vals.reals[1] = 0;
        inval->vals.reals[2] = 0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> SweepRangeMach = NULL </B> <br>
         *  Use FligthStream Sweeper Toolbox to solver for a range of Mach numbers.<br>
         *  SweepRangeMach is a vector of length 3 with [start, end, delta] Mach numbers.<br>
         *  Only one of 'Mach', 'SweepRangeMach', or 'SweepRangeVelocity' must be specified.
         */

    } else if (index == inSweepRangeVelocity) {
        *ainame             = EG_strdup("SweepRangeVelocity");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->nrow         = 3;
        AIM_ALLOC(inval->vals.reals, inval->nrow, double, aimInfo, status);
        inval->vals.reals[0] = 0;
        inval->vals.reals[1] = 0;
        inval->vals.reals[2] = 0;
        if (units != NULL)
            AIM_STRDUP(inval->units, units->speed, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> SweepRangeVelocity = NULL </B> <br>
         *  Use FligthStream Sweeper Toolbox to solver for a range of velocities.<br>
         *  SweepRangeVelocity is a vector of length 3 with [start, end, delta] velocities.<br>
         *  Only one of 'Mach', 'SweepRangeMach', or 'SweepRangeVelocity' must be specified.
         */

    } else if (index == inSweepRangeAlpha) {
        *ainame             = EG_strdup("SweepRangeAlpha");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = EG_strdup("degree");
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->nrow         = 3;
        AIM_ALLOC(inval->vals.reals, inval->nrow, double, aimInfo, status);
        inval->vals.reals[0] = 0;
        inval->vals.reals[1] = 0;
        inval->vals.reals[2] = 0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> SweepRangeAlpha = NULL </B> <br>
         *  Use FligthStream Sweeper Toolbox to solver for a range of angles of attack.<br>
         *  SweepRangeAlpha is a vector of length 3 with [start, end, delta] angles.<br>
         *  Only one of 'Alpha' or 'SweepRangeAlpha' must be specified.
         */

    } else if (index == inSweepRangeBeta) {
        *ainame             = EG_strdup("SweepRangeBeta");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = EG_strdup("degree");
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->nrow         = 3;
        AIM_ALLOC(inval->vals.reals, inval->nrow, double, aimInfo, status);
        inval->vals.reals[0] = 0;
        inval->vals.reals[1] = 0;
        inval->vals.reals[2] = 0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> SweepRangeBeta = NULL </B> <br>
         *  Use FligthStream Sweeper Toolbox to solver for a range of sideslip angles.<br>
         *  SweepRangeBeta is a vector of length 3 with [start, end, delta] angles.<br>
         *  Only one of 'Beta' or 'SweepRangeBeta' must be specified.
         */

    } else if (index == inSweepClear_Solution_After_Each_Run) {
        *ainame             = EG_strdup("SweepClear_Solution_After_Each_Run");
        inval->type         = Boolean;
        inval->nullVal      = IsNull;
        inval->units        = NULL;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.integer = (int)false;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> SweepClear_Solution_After_Each_Run = False </B> <br>
         *  Re-initialize the solution between each analysis when using FligthStream Sweeper Toolbox.
         */

    } else if (index == inSweepReference_Velocity_Equals_Freestream) {
        *ainame             = EG_strdup("SweepReference_Velocity_Equals_Freestream");
        inval->type         = Boolean;
        inval->nullVal      = IsNull;
        inval->units        = NULL;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.integer = (int)false;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> SweepReference_Velocity_Equals_Freestream = False </B> <br>
         *  Use the updated reference velocity when using FligthStream Sweeper Toolbox to sweep over Mach or Velocity.
         */

    } else if (index == inSweepExport_Surface_Data) {
        *ainame             = EG_strdup("SweepExport_Surface_Data");
        inval->type         = String;
        inval->nullVal      = IsNull;
        inval->units        = NULL;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> SweepExport_Surface_Data = NULL </B> <br>
         * Export surface data while using FligthStream Sweeper Toolbox.
         * Only one file format may be specified. Available options:<br>
         *   - DAT
         *   - VTK
         *   - CSV
         *   - TXT
         */

    } else if (index == inFluid_Properties) {
        *ainame             = EG_strdup("Fluid_Properties");
        inval->type         = Tuple;
        inval->nullVal      = IsNull;
        inval->lfixed       = Change;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->vals.tuple   = NULL;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Fluid_Properties = NULL </B> <br>
         *  Reference fluid properties. Altitude must be NULL. See \ref flightstreamFLUID_PROPERTIES.
         */

    } else if (index == inAltitude) {
        *ainame             = EG_strdup("Altitude");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = EG_strdup("feet");
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Altitude = NULL (default) </B> <br>
         *  Altitude used to compute Fluid Properties. The Fluid_Properties input must be NULL.
         */

    } else if (index == inSolver_Model) {
        *ainame             = EG_strdup("Solver_Model");
        inval->type         = String;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.string  = NULL;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Solver_Model = NULL (default) </B> <br>
         *  One of: "INCOMPRESSIBLE", "SUBSONIC_PRANDTL_GLAUERT", "TRANSONIC_FIELD_PANEL", "SUPERSONIC_LINEAR_DOUBLET", "TANGENT_CONE", "MODIFIED_NEWTONIAN"
         */

    } else if (index == inWake_Termination) {
        *ainame             = EG_strdup("Wake_Termination");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;
        if (units != NULL)
            AIM_STRDUP(inval->units, units->length, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Wake_Termination = NULL (default) </B> <br>
         *  Wake termination x-location
         */

    } else if (index == inStabilization) {
        *ainame             = EG_strdup("Stabilization");
        inval->type         = Double;
        inval->nullVal      = NotNull;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 1.0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Stabilization = 1.0 (default) </B> <br>
         *  Stabilization value for Solver_Model "INCOMPRESSIBLE", "SUBSONIC_PRANDTL_GLAUERT", or "TRANSONIC_FIELD_PANEL"
         */

    } else if (index == inWall_Collision_Avoidance) {
        *ainame             = EG_strdup("Wall_Collision_Avoidance");
        inval->type         = Boolean;
        inval->nullVal      = NotNull;
        inval->lfixed       = Fixed;
        inval->sfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.integer = (int)false;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Wall_Collision_Avoidance = false (default) </B> <br>
         *  Enable wake Wall Collision Avoidance
         */

    } else if (index == inCCS_GeneralSurface) {
        *ainame             = EG_strdup("CCS_GeneralSurface");
        inval->type         = Tuple;
        inval->nullVal      = IsNull;
        inval->lfixed       = Change;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->vals.tuple   = NULL;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> CCS_GeneralSurface = NULL (default) </B> <br>
         * General surface information for CCS geometry. See \ref vlmSurface for additional details.
         */

    } else if (index == inCCS_LiftingSurface) {
        *ainame             = EG_strdup("CCS_LiftingSurface");
        inval->type         = Tuple;
        inval->nullVal      = IsNull;
        inval->lfixed       = Change;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->vals.tuple   = NULL;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> CCS_LiftingSurface = NULL (default) </B> <br>
         * Lifting surface information for CCS geometry. See \ref vlmSurface for additional details.
         */

    } else if (index == inCCS_Control) {
        *ainame             = EG_strdup("CCS_Control");
        inval->type         = Tuple;
        inval->nullVal      = IsNull;
        inval->lfixed       = Change;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->vals.tuple   = NULL;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> CCS_Control = NULL (default) </B> <br>
         * Control surface information for CCS geometry. See \ref vlmControl for additional details.
         */

    } else if (index == inCCS_Revolution) {
        *ainame             = EG_strdup("CCS_Revolution");
        inval->type         = Tuple;
        inval->nullVal      = IsNull;
        inval->lfixed       = Change;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->vals.tuple   = NULL;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> CCS_Revolution = NULL (default) </B> <br>
         * Body of Revolution information for CCS geometry. Only one airfoil section and one body for the axis of revolution must be provided.
         * See \ref vlmSurface for additional details.
         */

    } else if (index == inCCS_Annular) {
        *ainame             = EG_strdup("CCS_Annular");
        inval->type         = Tuple;
        inval->nullVal      = IsNull;
        inval->lfixed       = Change;
        inval->sfixed       = Fixed;
        inval->dim          = Vector;
        inval->vals.tuple   = NULL;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> CCS_Annular = NULL (default) </B> <br>
         * Nacelle and Annular body information for CCS geometry. See \ref vlmSurface for additional details.
         */

    } else if (index == inReferenceChord) {
        *ainame             = EG_strdup("ReferenceChord");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;
        if (units != NULL)
            AIM_STRDUP(inval->units, units->length, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>ReferenceChord = NULL </B> <br>
         * This sets the reference chord for used in force and moment
         *  calculations.  Alternatively, the geometry (body) attribute
         *  (see \ref attributeFLIGHTSTREAM) "capsReferenceChord" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceSpan) {
        *ainame             = EG_strdup("ReferenceSpan");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;
        if (units != NULL)
            AIM_STRDUP(inval->units, units->length, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>ReferenceSpan = NULL </B> <br>
         * This sets the reference span for used in force and moment
         *  calculations.  Alternatively, the geometry (body) attribute
         *  (see \ref attributeFLIGHTSTREAM) "capsReferenceSpan" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceArea) {
        *ainame             = EG_strdup("ReferenceArea");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;
        if (units != NULL)
            AIM_STRDUP(inval->units, units->area, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>ReferenceArea = NULL </B> <br>
         * This sets the reference area for used in force and moment
         *  calculations.  Alternatively, the geometry (body) attribute
         *  (see \ref attributeFLIGHTSTREAM) "capsReferenceArea" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceX) {
        *ainame             = EG_strdup("ReferenceX");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;
        if (units != NULL)
            AIM_STRDUP(inval->units, units->length, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>ReferenceX = NULL </B> <br>
         * This sets the reference X for moment calculations.
         * Alternatively, the geometry (body) attribute
         *  (see \ref attributeFLIGHTSTREAM) "capsReferenceX" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceY) {
        *ainame             = EG_strdup("ReferenceY");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;
        if (units != NULL)
            AIM_STRDUP(inval->units, units->length, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>ReferenceX = NULL </B> <br>
         * This sets the reference Y for moment calculations.
         * Alternatively, the geometry (body) attribute
         *  (see \ref attributeFLIGHTSTREAM) "capsReferenceY" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceZ) {
        *ainame             = EG_strdup("ReferenceZ");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;
        if (units != NULL)
            AIM_STRDUP(inval->units, units->length, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>ReferenceX = NULL </B> <br>
         * This sets the reference Z for moment calculations.
         * Alternatively, the geometry (body) attribute
         *  (see \ref attributeFLIGHTSTREAM) "capsReferenceZ" maybe used to
         *  specify this variable (note: values set through the AIM input
         *  will supersede the attribution value).
         */

    } else if (index == inReferenceVelocity) {
        *ainame             = EG_strdup("ReferenceVelocity");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0.0;
        if (units != NULL)
            AIM_STRDUP(inval->units, units->speed, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>ReferenceVelocity = NULL </B> <br>
         * This sets the reference velocity
         */

    } else if (index == inPressure_Scale_Factor) {
        *ainame             = EG_strdup("Pressure_Scale_Factor");
        inval->type         = Double;
        inval->vals.real    = 1.0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>Pressure_Scale_Factor = 1.0</B> <br>
         * Value to scale Pressure data when transferring data. Data is scaled based on Pressure = Pressure_Scale_Factor*Pressure.
         */

    } else if (index == inConvergenceTol) {
        *ainame             = EG_strdup("ConvergenceTol");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = NULL;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 1.0e-5;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>ConvergenceTol = 1.0e-5 (default) </B> <br>
         * Solver convergence tolerance
         */

    } else if (index == inSolverIterations) {
        *ainame             = EG_strdup("SolverIterations");
        inval->type         = Integer;
        inval->nullVal      = IsNull;
        inval->units        = NULL;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.integer = 0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>SolverIterations = NULL </B> <br>
         * Maximum number of solver iterations
         */

    } else if (index == inSolverConvergence) {
        *ainame             = EG_strdup("SolverConvergence");
        inval->type         = Double;
        inval->nullVal      = IsNull;
        inval->units        = NULL;
        inval->lfixed       = Fixed;
        inval->dim          = Scalar;
        inval->vals.real    = 0;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>SolverConvergence = NULL </B> <br>
         * Solver convergence tolerance
         */

    } else if (index == inExport_Solver_Analysis) {
        *ainame             = EG_strdup("Export_Solver_Analysis");
        inval->type         = String;
        inval->dim          = Vector;
        inval->lfixed       = Change;
        inval->sfixed       = Fixed;
        inval->vals.string  = NULL;
        inval->nullVal      = IsNull;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>Export_Solver_Analysis = NULL</B> <br>
         * List of file types to export. Available options:<br>
         *   - Tecplot
         *   - VTK
         *   - CSV
         *   - BDF
         *   - Force_Distributions
         */

    } else if (index == inFlightScriptPre) {
        *ainame             = EG_strdup("FlightScriptPre");
        inval->type         = String;
        inval->dim          = Vector;
        inval->lfixed       = Change;
        inval->sfixed       = Fixed;
        inval->vals.string  = NULL;
        inval->nullVal      = IsNull;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>FlightScriptPre = NULL</B> <br>
         * List of flight script commands injected in script.txt prior to INITIALIZE_SOLVER
         */

    } else if (index == inFlightScriptPost) {
        *ainame             = EG_strdup("FlightScriptPost");
        inval->type         = String;
        inval->dim          = Vector;
        inval->lfixed       = Change;
        inval->sfixed       = Fixed;
        inval->vals.string  = NULL;
        inval->nullVal      = IsNull;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>FlightScriptPost = NULL</B> <br>
         * List of flight script commands to append at the end of script.txt
         */
#if 0
    } else if (index == inDesign_Variable) {
        *ainame             = EG_strdup("Design_Variable");
        inval->type         = Tuple;
        inval->nullVal      = IsNull;
        inval->lfixed       = Change;
        inval->vals.tuple   = NULL;
        inval->dim          = Vector;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Design_Variable = NULL</B> <br>
         * List of AnalysisIn and/or GeometryIn variable names used to compute sensitivities of Design_Functional for optimization, see \ref cfdDesignVariable for additional details.
         */

    } else if (index == inDesign_Functional) {
        *ainame             = EG_strdup("Design_Functional");
        inval->type         = Tuple;
        inval->nullVal      = IsNull;
        inval->lfixed       = Change;
        inval->vals.tuple   = NULL;
        inval->dim          = Vector;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Design_Functional = NULL</B> <br>
         * The design functional tuple is used to input functional information for optimization, see \ref cfdDesignFunctional for additional details.
         * Using this requires Design_SensFile = False.
         */
#endif
    } else if (index == inHidden) {
        *ainame              = EG_strdup("Hidden");
        inval->type          = Boolean;
        inval->lfixed        = Fixed;
        inval->vals.integer  = (int) true;
        inval->dim           = Scalar;
        inval->nullVal       = NotNull;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Hidden = True</B> <br>
         * Hide FlightStream GUI during execution on Windows
         */

    } else if (index == inMax_Parallel_Threads) {
        *ainame              = EG_strdup("Max_Parallel_Threads");
        inval->type          = Integer;
        inval->lfixed        = Fixed;
        inval->vals.integer  = 0;
        inval->dim           = Scalar;
        inval->nullVal       = IsNull;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Max_Parallel_Threads = NULL</B> <br>
         * Maximum number of threads for FligthStream analysis
         */

    } else if (index == inSaveFSM) {
        *ainame              = EG_strdup("SaveFSM");
        inval->type          = Boolean;
        inval->lfixed        = Fixed;
        inval->vals.integer  = (int) false;
        inval->dim           = Scalar;
        inval->nullVal       = NotNull;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> SaveFSM = False</B> <br>
         * Save a FlightStream .fsm file after analysis
         */

    } else if (index == inMesh_Morph) {
        *ainame              = EG_strdup("Mesh_Morph");
        inval->type          = Boolean;
        inval->lfixed        = Fixed;
        inval->vals.integer  = (int) false;
        inval->dim           = Scalar;
        inval->nullVal       = NotNull;

        /*! \page aimInputsFLIGHTSTREAM
         * - <B> Mesh_Morph = False</B> <br>
         * Project previous surface mesh onto new geometry.
         */

    } else if (index == inSurface_Mesh) {
        *ainame             = EG_strdup("Surface_Mesh");
        inval->type         = PointerMesh;
        inval->dim          = Vector;
        inval->lfixed       = Change;
        inval->sfixed       = Change;
        inval->vals.AIMptr  = NULL;
        inval->nullVal      = IsNull;
        AIM_STRDUP(inval->meshWriter, MESHWRITER, aimInfo, status);
        if (units != NULL)
            AIM_STRDUP(inval->units, units->length, aimInfo, status);

        /*! \page aimInputsFLIGHTSTREAM
         * - <B>Surface_Mesh = NULL</B> <br>
         * A Surface_Mesh link.
         */

    } else {
        AIM_ERROR(aimInfo, "Unknown input index $%d", index);
        status = CAPS_RANGEERR;
        goto cleanup;
    }

    AIM_NOTNULL(*ainame, aimInfo, status);

cleanup:
    if (status != CAPS_SUCCESS) {
        AIM_FREE(*ainame);
    }

    return status;
}


/**********************************************************************/
/* aimOutputs - return information about index'th analysis outputs    */
/**********************************************************************/

int
aimOutputs(/*@unused@*/ void *instStore,    /* (in)  AIM instance storage */
           /*@unused@*/ void *aimInfo,      /* (in)  AIM context */
           int index,                       /* (in)  output index (1-nOut) */
           char **aoname,                   /* (out) name of analysis output */
           capsValue *outval)               /* (out) description of analysis output */
{
  int status = CAPS_SUCCESS;

  /*! \page aimOutputsFLIGHTSTREAM AIM Outputs
   * The following list outlines the FlightStream outputs available through
   * the AIM interface.
   */

  /* -------------------------------------------------------------- */

#ifdef DEBUG
  printf("flightstreamAIM/aimOutputs(index=%d)\n", index);
#endif

  outval->type       = Double;
  outval->lfixed     = Fixed;
  outval->sfixed     = Fixed;
  outval->units      = NULL;
  outval->dim        = 0;
  outval->length     = 1;
  outval->nrow       = 1;
  outval->ncol       = 1;
  outval->vals.real  = 0.0;
  outval->vals.reals = NULL;

  if        (index == outAlpha) {
    *aoname = EG_strdup("Alpha");
  } else if (index == outBeta) {
    *aoname = EG_strdup("Beta");
  } else if (index == outMach) {
    *aoname = EG_strdup("Mach");
  } else if (index == outVelocity) {
    *aoname = EG_strdup("Velocity");
  } else if (index == outDensity) {
    *aoname = EG_strdup("Density");
  } else if (index == outPressure) {
    *aoname = EG_strdup("Pressure");
  } else if (index == outSonicSpeed) {
    *aoname = EG_strdup("SonicSpeed");
  } else if (index == outTemperature) {
    *aoname = EG_strdup("Temperature");
  } else if (index == outViscosity) {
    *aoname = EG_strdup("Viscosity");
  } else if (index == outCx) {
    *aoname = EG_strdup("Cx");
  } else if (index == outCy) {
    *aoname = EG_strdup("Cy");
  } else if (index == outCz) {
    *aoname = EG_strdup("Cz");
  } else if (index == outCL) {
    *aoname = EG_strdup("CL");
  } else if (index == outCDi) {
    *aoname = EG_strdup("CDi");
  } else if (index == outCDo) {
    *aoname = EG_strdup("CDo");
  } else if (index == outCMx) {
    *aoname = EG_strdup("CMx");
  } else if (index == outCMy) {
    *aoname = EG_strdup("CMy");
  } else if (index == outCMz) {
    *aoname = EG_strdup("CMz");
  } else {
    printf("flightstreamAIM/aimOutputs index = %d NOT Found\n", index);
    status = CAPS_NOTFOUND;
    goto cleanup;
  }

  /*! \page aimOutputsFLIGHTSTREAM
   * Analysis settings:
   * - <B>Alpha</B> = Angle of attack
   * - <B>Beta</B>  = Sideslip angle
   * - <B>Mach</B> = Fresstream mach
   * - <B>Velocity</B> = Fresstream velocity
   * - <B>Density</B> = Fresstream density
   * - <B>Pressure</B> = Fresstream pressure
   * - <B>SonicSpeed</B> = Fresstream speed of sound
   * - <B>Temperature</B> = Fresstream temperature
   * - <B>Viscosity</B> = Fresstream viscosity
   *
   * Aerodynamic coefficients:
   * - <B>Cx</B> = X-force coefficient
   * - <B>Cx</B> = X-force coefficient
   * - <B>Cx</B> = X-force coefficient
   * - <B>CL</B> = Lift coefficient
   * - <B>CDi</B> = Induced drag coefficient
   * - <B>CDo</B> = Skin friction drag coefficient
   * - <B>CMx</B> = X-moment coefficient
   * - <B>CMy</B> = Y-moment coefficient
   * - <B>CMz</B> = Z-moment coefficient
   *
   * NOTE: Coefficients are normalized by the input reference velocity, not the freestream velocity.
   *
   * Component aerodynamic coefficients are also available as dynamic outputs for a single point analysis.
   */

  AIM_NOTNULL(*aoname, aimInfo, status);

cleanup:
  return status;
}


/**********************************************************************/
/* aimUpdateState - update the AIM's internal state                   */
/**********************************************************************/

int
aimUpdateState(void      *instStore,    /* (in)  AIM instance storage */
               void      *aimInfo,      /* (in)  AIM context */
               capsValue aimInputs[])   /* (in)  array of analysis inputs */
{
  // Function return flag
  int status = CAPS_SUCCESS;

  int i, isurf;
  int foundSref=(int)false;
  int foundCref=(int)false;
  int foundBref=(int)false;
  int foundXref=(int)false;
  int foundYref=(int)false;
  int foundZref=(int)false;

  int nbound = 0;
  char **boundNames = NULL;

  // AIM input bodies
  const char *intent;
  int  numBody;
  ego *bodies = NULL;

  // EGADS return values
  int          atype, alen;
  const int    *ints;
  const char   *string;
  const double *reals;

  cfdUnitsStruct *units = NULL;

  const char *lengthUnits=NULL;

  aimStorage *flightstreamInstance = (aimStorage *)instStore;
  units = &flightstreamInstance->units;

  /* -------------------------------------------------------------- */

#ifdef DEBUG
  printf("flightstreamAIM/aimUpdateState()\n");
#endif

  AIM_NOTNULL(aimInputs, aimInfo, status);

//    if (aimInputs[inSurface_Mesh-1].nullVal == IsNull &&
//        aimInputs[inMesh_Morph-1].vals.integer == (int) false) {
//      AIM_ANALYSISIN_ERROR(aimInfo, inSurface_Mesh, "'Surface_Mesh' input must be linked to an output 'Surface_Mesh'");
//      status = CAPS_BADVALUE;
//      goto cleanup;
//    }

  // Get AIM bodies
  status = aim_getBodies(aimInfo, &intent, &numBody, &bodies);
  AIM_STATUS(aimInfo, status);

#ifdef DEBUG
  printf("flightstreamAIM/aimUpdateState -> numBody=%d\n", numBody);
#endif

  if ((numBody <= 0) || (bodies == NULL)) {
    AIM_ERROR(aimInfo, "No body\n");
    status = CAPS_SOURCEERR;
    goto cleanup;
  }

  if ( aimInputs[inSurface_Mesh-1].nullVal == NotNull &&
      (aimInputs[inCCS_GeneralSurface-1].nullVal == NotNull ||
       aimInputs[inCCS_LiftingSurface-1].nullVal == NotNull ||
       aimInputs[inCCS_Revolution-1].nullVal == NotNull ||
       aimInputs[inCCS_Annular-1].nullVal == NotNull)) {
    AIM_ERROR(aimInfo, "Either 'Surface_Mesh' or at least one of CCS geometry must be specified");
    AIM_ADDLINE(aimInfo, "Specified inputs:");
    if (aimInputs[inSurface_Mesh-1].nullVal == NotNull)
      AIM_ADDLINE(aimInfo, "\tSurface_Mesh");
    if (aimInputs[inCCS_GeneralSurface-1].nullVal == NotNull)
      AIM_ADDLINE(aimInfo, "\tCCS_GeneralSurface");
    if (aimInputs[inCCS_LiftingSurface-1].nullVal == NotNull)
      AIM_ADDLINE(aimInfo, "\tCCS_LiftingSurface");
    if (aimInputs[inCCS_Revolution-1].nullVal == NotNull)
      AIM_ADDLINE(aimInfo, "\tCCS_Revolution");
    if (aimInputs[inCCS_Annular-1].nullVal == NotNull)
      AIM_ADDLINE(aimInfo, "\tCCS_Annular");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if ((aimInputs[inMach-1         ].nullVal != IsNull && (aimInputs[inSweepRangeMach-1].nullVal != IsNull || aimInputs[inSweepRangeVelocity-1].nullVal != IsNull)) ||
      (aimInputs[inSweepRangeMach-1    ].nullVal != IsNull && (aimInputs[inMach-1     ].nullVal != IsNull || aimInputs[inSweepRangeVelocity-1].nullVal != IsNull)) ||
      (aimInputs[inSweepRangeVelocity-1].nullVal != IsNull && (aimInputs[inMach-1     ].nullVal != IsNull || aimInputs[inSweepRangeMach-1    ].nullVal != IsNull)) ||
      (aimInputs[inSweepRangeVelocity-1].nullVal == IsNull &&  aimInputs[inMach-1     ].nullVal == IsNull && aimInputs[inSweepRangeMach-1    ].nullVal == IsNull)) {
    AIM_ERROR(aimInfo, "Only one of 'Mach', 'SweepRangeMach', or 'SweepRangeVelocity' must be specified");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inAlpha-1].nullVal == aimInputs[inSweepRangeAlpha-1].nullVal) {
    AIM_ERROR(aimInfo, "Only one of 'Alpha' or 'SweepRangeAlpha' must be specified");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inBeta-1].nullVal == aimInputs[inSweepRangeBeta-1].nullVal) {
    AIM_ERROR(aimInfo, "Only one of 'Beta' or 'SweepRangeBeta' must be specified");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  status = aim_getBounds(aimInfo, &nbound, &boundNames);
  AIM_STATUS(aimInfo, status);

  if ((aimInputs[inSweepRangeMach-1    ].nullVal != IsNull ||
       aimInputs[inSweepRangeVelocity-1].nullVal != IsNull ||
       aimInputs[inSweepRangeAlpha-1   ].nullVal != IsNull ||
       aimInputs[inSweepRangeBeta-1    ].nullVal != IsNull) &&
      nbound > 0) {
    AIM_ERROR(aimInfo, "The Sweeper Toolbox cannot be used with bounds transfer");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inFluid_Properties-1].nullVal == aimInputs[inAltitude-1].nullVal) {
    AIM_ERROR(aimInfo, "Only one of 'Fluid_Properties' or 'Altitude' must be specified");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inSolver_Model-1].nullVal == IsNull) {
    AIM_ERROR(aimInfo, "'Solver_Model' must be specified");
    status = CAPS_BADVALUE;
    goto cleanup;
  }
  string_toUpperCase(aimInputs[inSolver_Model-1].vals.string);

  if (aimInputs[inFluid_Properties-1].nullVal == NotNull) {
    status = _getFluid_Properties(aimInfo,
                                  aimInputs[inFluid_Properties-1].length,
                                  aimInputs[inFluid_Properties-1].vals.tuple,
                                  flightstreamInstance);
    AIM_STATUS(aimInfo, status);
  }
#if 0
  // Get design variables
  if (aimInputs[inDesign_Variable-1].nullVal == NotNull &&
      (flightstreamInstance->design.numDesignVariable == 0 ||
       aim_newAnalysisIn(aimInfo, inDesign_Variable) == CAPS_SUCCESS)) {

      if (aimInputs[inDesign_Functional-1].nullVal == IsNull) {
          AIM_ERROR(aimInfo, "\"Design_Variable\" has been set, but no values have been provided for \"Design_Functional\" and \"Design_SensFile\" is False!");
          status = CAPS_BADVALUE;
          goto cleanup;
      }
/*@-nullpass@*/
      status = cfd_getDesignVariable(aimInfo,
                                     aimInputs[inDesign_Variable-1].length,
                                     aimInputs[inDesign_Variable-1].vals.tuple,
                                     &flightstreamInstance->design.numDesignVariable,
                                     &flightstreamInstance->design.designVariable);
/*@+nullpass@*/
      AIM_STATUS(aimInfo, status);
  }
#endif
  status = aim_capsLength(aimInfo, &lengthUnits);
  AIM_NOTFOUND(aimInfo, status);

  if (status == CAPS_NOTFOUND) {
      AIM_ERROR(aimInfo, "capsLength attribute must be specified for FlightStream");
      goto cleanup;
  }
  AIM_NOTNULL(lengthUnits, aimInfo, status);

  flightstreamInstance->lengthScaleFactor = 1.0;
  status = aim_convert(aimInfo, 1, lengthUnits, &flightstreamInstance->lengthScaleFactor, units->length, &flightstreamInstance->lengthScaleFactor);
  AIM_STATUS(aimInfo, status);

#ifdef DEBUG
  printf("flightstreamAIM/aimUpdateState -> scaleFactor=%f\n", lengthScaleFactor);
#endif

  // Loop over bodies and look for reference quantity attributes
  for (i=0; i < numBody; i++) {
    status = EG_attributeRet(bodies[i], "capsReferenceChord",
                             &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {
      if (atype == ATTRREAL && alen == 1) {
        flightstreamInstance->Cref = reals[0] * flightstreamInstance->lengthScaleFactor;
        foundCref = (int)true;
      } else {
        AIM_ERROR(aimInfo, "capsReferenceChord should be a scalar\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }

    status = EG_attributeRet(bodies[i], "capsReferenceSpan",
                             &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {
      if (atype == ATTRREAL && alen == 1) {
        flightstreamInstance->Bref = reals[0] * flightstreamInstance->lengthScaleFactor;
        foundBref = (int)true;
      } else {
        AIM_ERROR(aimInfo, "capsReferenceSpan should be a scalar\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }

    status = EG_attributeRet(bodies[i], "capsReferenceArea",
                             &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {
      if (atype == ATTRREAL && alen == 1) {
        flightstreamInstance->Sref = reals[0] * flightstreamInstance->lengthScaleFactor * flightstreamInstance->lengthScaleFactor;
        foundSref = (int)true;
      } else {
        AIM_ERROR(aimInfo, "capsReferenceArea should be a scalar\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }

    status = EG_attributeRet(bodies[i], "capsReferenceX",
                             &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {

      if (atype == ATTRREAL && alen == 1) {
        flightstreamInstance->Xref = reals[0] * flightstreamInstance->lengthScaleFactor;
        foundXref = (int)true;
      } else {
        AIM_ERROR(aimInfo, "capsReferenceX should be followed by a single real value!\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }

    status = EG_attributeRet(bodies[i], "capsReferenceY",
                             &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS) {

      if (atype == ATTRREAL && alen == 1) {
        flightstreamInstance->Yref = reals[0] * flightstreamInstance->lengthScaleFactor;
        foundYref = (int)true;
      } else {
        AIM_ERROR(aimInfo, "capsReferenceY should be followed by a single real value!\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }

    status = EG_attributeRet(bodies[i], "capsReferenceZ",
                             &atype, &alen, &ints, &reals, &string);
    if (status == EGADS_SUCCESS){

      if (atype == ATTRREAL && alen == 1) {
        flightstreamInstance->Zref = reals[0] * flightstreamInstance->lengthScaleFactor;
        foundZref = (int)true;
      } else {
        AIM_ERROR(aimInfo, "capsReferenceZ should be followed by a single real value!\n");
        status = CAPS_BADVALUE;
        goto cleanup;
      }
    }
  }

  if (aimInputs[inReferenceArea-1].nullVal == NotNull) {
    flightstreamInstance->Sref = aimInputs[inReferenceArea-1].vals.real;
    foundSref = (int)true;
  }
  if (foundSref == (int)false) {
    AIM_ERROR(aimInfo, "capsReferenceArea is not set on any body and 'ReferenceArea' input not set!");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inReferenceChord-1].nullVal == NotNull) {
    flightstreamInstance->Cref = aimInputs[inReferenceChord-1].vals.real;
    foundCref = (int)true;
  }
  if (foundCref == (int)false) {
    AIM_ERROR(aimInfo, "capsReferenceChord is not set on any body and 'ReferenceChord' input not set!");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inReferenceSpan-1].nullVal == NotNull) {
    flightstreamInstance->Bref = aimInputs[inReferenceSpan-1].vals.real;
    foundBref = (int)true;
  }
  if (foundBref == (int)false) {
    AIM_ERROR(aimInfo, "capsReferenceSpan is not set on any body and 'ReferenceSpan' input not set!");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inReferenceX-1].nullVal == NotNull) {
    flightstreamInstance->Xref = aimInputs[inReferenceX-1].vals.real;
    foundXref = (int)true;
  }
  if (foundXref == (int)false) {
    AIM_ERROR(aimInfo, "capsReferenceX is not set on any body and 'ReferenceX' input not set!");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inReferenceY-1].nullVal == NotNull) {
    flightstreamInstance->Yref = aimInputs[inReferenceY-1].vals.real;
    foundYref = (int)true;
  }
  if (foundYref == (int)false) {
    AIM_ERROR(aimInfo, "capsReferenceY is not set on any body and 'ReferenceY' input not set!");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inReferenceZ-1].nullVal == NotNull) {
    flightstreamInstance->Zref = aimInputs[inReferenceZ-1].vals.real;
    foundZref = (int)true;
  }
  if (foundZref == (int)false) {
    AIM_ERROR(aimInfo, "capsReferenceZ is not set on any body and 'ReferenceZ' input not set!");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inReferenceVelocity-1].nullVal == IsNull) {
    AIM_ERROR(aimInfo, "Input ReferenceVelocity not set!");
    status = CAPS_BADVALUE;
    goto cleanup;
  }

  if (aimInputs[inCCS_GeneralSurface-1].nullVal == NotNull ||
      aimInputs[inCCS_LiftingSurface-1].nullVal == NotNull ||
      aimInputs[inCCS_Revolution-1    ].nullVal == NotNull ||
      aimInputs[inCCS_Annular-1       ].nullVal == NotNull) {

    // CCS file format is specified

    if (aim_newGeometry(aimInfo) == CAPS_SUCCESS) {

      // Get capsGroup name and index mapping to make sure all bodies have a capsGroup value
      status = create_CAPSGroupAttrToIndexMap(numBody,
                                              bodies,
                                              0, // Only search down to the body level of the EGADS body
                                              &flightstreamInstance->groupMap);
      AIM_STATUS(aimInfo, status);
    }

    if (aim_newGeometry(aimInfo) == CAPS_SUCCESS ||
        aim_newAnalysisIn(aimInfo, inCCS_GeneralSurface) == CAPS_SUCCESS) {

      // Get general surface information
      status = get_vlmSurface(aimInfo,
                              aimInputs[inCCS_GeneralSurface-1].length,
                              aimInputs[inCCS_GeneralSurface-1].vals.tuple,
                              &flightstreamInstance->groupMap,
                              0,   // default Nchord
                              1.0, // default Cspace
                              "degree",
                              &flightstreamInstance->numCCSGenSurface,
                              &flightstreamInstance->ccsGenSurface);
      AIM_NOTFOUND(aimInfo, status);

      // Accumulate section data
      status = vlm_getSections(aimInfo, numBody, bodies, NULL, flightstreamInstance->groupMap, vlmBODY,
                               flightstreamInstance->numCCSGenSurface, &flightstreamInstance->ccsGenSurface);
      AIM_STATUS(aimInfo, status);
    }

    if (aim_newGeometry(aimInfo) == CAPS_SUCCESS ||
        aim_newAnalysisIn(aimInfo, inCCS_LiftingSurface) == CAPS_SUCCESS ||
        aim_newAnalysisIn(aimInfo, inCCS_Control) == CAPS_SUCCESS
        ) {

      if (aim_newGeometry(aimInfo) == CAPS_SUCCESS ||
          aim_newAnalysisIn(aimInfo, inCCS_Control) == CAPS_SUCCESS) {

          // Get control surface information
          if (aimInputs[inCCS_Control-1].nullVal == NotNull) {

              status = get_vlmControl(aimInfo,
                                      aimInputs[inCCS_Control-1].length,
                                      aimInputs[inCCS_Control-1].vals.tuple,
                                      "degree",
                                      &flightstreamInstance->numCCSControl,
                                      &flightstreamInstance->ccsControl);
              AIM_STATUS(aimInfo, status);
          }
      }

      if (aim_newGeometry(aimInfo) == CAPS_SUCCESS ||
          aim_newAnalysisIn(aimInfo, inCCS_LiftingSurface) == CAPS_SUCCESS) {

          // Get lifting surface information
          status = get_vlmSurface(aimInfo,
                                  aimInputs[inCCS_LiftingSurface-1].length,
                                  aimInputs[inCCS_LiftingSurface-1].vals.tuple,
                                  &flightstreamInstance->groupMap,
                                  0,   // default Nchord
                                  1.0, // default Cspace
                                  "degree",
                                  &flightstreamInstance->numCCSLiftSurface,
                                  &flightstreamInstance->ccsLiftSurface);
          AIM_STATUS(aimInfo, status);

          // Accumulate section data
          status = vlm_getSections(aimInfo, numBody, bodies, NULL, flightstreamInstance->groupMap, vlmGENERIC,
                                   flightstreamInstance->numCCSLiftSurface, &flightstreamInstance->ccsLiftSurface);
          AIM_STATUS(aimInfo, status);
          AIM_NOTNULL(flightstreamInstance->ccsLiftSurface, aimInfo, status);
      }

      // Loop through surfaces and transfer control surface data to sections
      for (isurf = 0; isurf < flightstreamInstance->numCCSLiftSurface; isurf++) {
          status = get_ControlSurface(aimInfo,
                                      flightstreamInstance->numCCSControl,
                                      flightstreamInstance->ccsControl,
                                      &flightstreamInstance->ccsLiftSurface[isurf]);
          AIM_STATUS (aimInfo, status);
      }

      if (aim_newGeometry(aimInfo) == CAPS_SUCCESS ||
          aim_newAnalysisIn(aimInfo, inCCS_Revolution) == CAPS_SUCCESS) {

        // Get general surface information
        status = get_vlmSurface(aimInfo,
                                aimInputs[inCCS_Revolution-1].length,
                                aimInputs[inCCS_Revolution-1].vals.tuple,
                                &flightstreamInstance->groupMap,
                                0,   // default Nchord
                                1.0, // default Cspace
                                "degree",
                                &flightstreamInstance->numCCSRevolution,
                                &flightstreamInstance->ccsRevolution);
        AIM_NOTFOUND(aimInfo, status);

        // Accumulate section data
        status = vlm_getSections(aimInfo, numBody, bodies, NULL, flightstreamInstance->groupMap, vlmREVOLVE,
                                 flightstreamInstance->numCCSRevolution, &flightstreamInstance->ccsRevolution);
        AIM_STATUS(aimInfo, status);
      }

      if (aim_newGeometry(aimInfo) == CAPS_SUCCESS ||
          aim_newAnalysisIn(aimInfo, inCCS_Annular) == CAPS_SUCCESS) {

        // Get general surface information
        status = get_vlmSurface(aimInfo,
                                aimInputs[inCCS_Annular-1].length,
                                aimInputs[inCCS_Annular-1].vals.tuple,
                                &flightstreamInstance->groupMap,
                                0,   // default Nchord
                                1.0, // default Cspace
                                "degree",
                                &flightstreamInstance->numCCSAnnular,
                                &flightstreamInstance->ccsAnnular);
        AIM_NOTFOUND(aimInfo, status);

        // Accumulate section data
        status = vlm_getSections(aimInfo, numBody, bodies, NULL, flightstreamInstance->groupMap, vlmANNULAR,
                                 flightstreamInstance->numCCSAnnular, &flightstreamInstance->ccsAnnular);
        AIM_STATUS(aimInfo, status);
      }
    }


  } else {
    // Get mesh
    flightstreamInstance->meshRefIn = (aimMeshRef *) aimInputs[inSurface_Mesh-1].vals.AIMptr;

    if ( aimInputs[inMesh_Morph-1].vals.integer == (int) true &&
        flightstreamInstance->meshRefIn == NULL) { // If we are mighty morphing

      // Lets "load" the meshRef now since it's not linked
      status = aim_loadMeshRef(aimInfo, &flightstreamInstance->meshRefMorph);
      AIM_STATUS(aimInfo, status);

      // Mighty Morph the mesh
      status = aim_morphMeshUpdate(aimInfo, &flightstreamInstance->meshRefMorph, numBody, bodies);
      AIM_STATUS(aimInfo, status);

      AIM_ALLOC(flightstreamInstance->meshRefMorph.fileName, PATH_MAX, char, aimInfo, status);
      status = aim_file(aimInfo, aimInputs[inProj_Name-1].vals.string, flightstreamInstance->meshRefMorph.fileName);
      AIM_STATUS(aimInfo, status);

      /*@-immediatetrans@*/
      flightstreamInstance->meshRefIn = &flightstreamInstance->meshRefMorph;
      /*@+immediatetrans@*/
    }
    AIM_NOTNULL(flightstreamInstance->meshRefIn, aimInfo, status);

    // Get attribute to index mapping
    status = create_MeshRefToIndexMap(aimInfo, flightstreamInstance->meshRefIn, &flightstreamInstance->groupMap);
    AIM_STATUS(aimInfo, status);
  }

#if 0
  status = cfd_cfdCoefficientUnits(aimInfo,
                                   flightstreamInstance->Cref, units->length,
                                   flightstreamInstance->Sref, units->area,
                                   aimInputs[Freestream_Density-1 ].vals.real, aimInputs[Freestream_Density-1 ].units,
                                   aimInputs[inReferenceVelocity-1].vals.real, aimInputs[inReferenceVelocity-1].units,
                                   aimInputs[Freestream_Pressure-1].vals.real, aimInputs[Freestream_Pressure-1].units,
                                   units);
  AIM_STATUS(aimInfo, status);
#endif

#ifdef DEBUG
  printf("flightstreamInstance->projectName = %s\n", flightstreamInstance->projectName);
  printf("flightstreamInstance->Cref        = %f\n", flightstreamInstance->Cref       );
  printf("flightstreamInstance->Bref        = %f\n", flightstreamInstance->Bref       );
  printf("flightstreamInstance->Sref        = %f\n", flightstreamInstance->Sref       );
  printf("flightstreamInstance->Xref        = %f\n", flightstreamInstance->Xref       );
  printf("flightstreamInstance->Yref        = %f\n", flightstreamInstance->Yref       );
  printf("flightstreamInstance->Zref        = %f\n", flightstreamInstance->Zref       );
#endif

  status = CAPS_SUCCESS;
cleanup:
  AIM_FREE(boundNames);
  return status;
}


/**********************************************************************/
/* aimPreAnalysis - generate FlightStream input file                  */
/**********************************************************************/

int
aimPreAnalysis(const void *instStore,   /* (in)  AIM instance storage */
               void       *aimInfo,     /* (in)  AIM context */
               capsValue  aimInputs[])  /* (in)  array of analysis inputs */
{
  // Function return flag
  int status = CAPS_SUCCESS;

  //double alt, temp, vel;
  FILE   *fp = NULL;

  int i;
  int nbound = 0;
  char **boundNames = NULL;

  int numSharp = 0;
  int *sharpSurf = NULL;

  int numProx = 0;
  int *proxSurf = NULL;

  int numBase = 0;
  int *baseSurf = NULL;

  int numWTerm = 0;
  int *wtermSurf = NULL; // Wake Termination Node Surfaces

  char meshFile[PATH_MAX];
  char aimFile[PATH_MAX];
  const char *file_type = NULL;
  const char *export = NULL;
  const char *flightscript = NULL;
  const char *vsp3_root = NULL;

  aimMesh    mesh;

  const aimStorage *flightstreamInstance = (const aimStorage *)instStore;
  const cfdUnitsStruct *units = &flightstreamInstance->units;

  /* -------------------------------------------------------------- */

#ifdef DEBUG
  printf("flightstreamAIM/aimPreAnalysis()\n");
#endif

  mesh.meshRef = NULL;
  mesh.meshData = NULL;

  AIM_NOTNULL(aimInputs, aimInfo, status);

  if (aimInputs[inCCS_GeneralSurface-1].nullVal == NotNull ||
      aimInputs[inCCS_LiftingSurface-1].nullVal == NotNull ||
      aimInputs[inCCS_Revolution-1    ].nullVal == NotNull ||
      aimInputs[inCCS_Annular-1       ].nullVal == NotNull) {

    file_type = "CSV";
    snprintf(meshFile, PATH_MAX, "%s%s", aimInputs[inProj_Name-1].vals.string, ".csv");

    fp = aim_fopen(aimInfo, meshFile, "w");
    if (fp == NULL) {
        AIM_ERROR(aimInfo, "Cannot open \"%s\"", meshFile);
        status = CAPS_IOERR;
        goto cleanup;
    }

    status = writeCCS(aimInfo, fp, flightstreamInstance,
                      &numSharp, &sharpSurf,
                      &numProx, &proxSurf,
                      &numBase, &baseSurf,
                      &numWTerm, &wtermSurf);
    AIM_STATUS(aimInfo, status);

    fclose(fp);
    fp = NULL;

  } else {

    if ( aimInputs[inMesh_Morph-1].vals.integer == (int) true) {
      if (aimInputs[inSurface_Mesh-1].nullVal == NotNull) { // If we are mighty morphing
        // store the current mesh for future iterations
        status = aim_storeMeshRef(aimInfo, (aimMeshRef *) aimInputs[inSurface_Mesh-1].vals.AIMptr, NULL);
        AIM_STATUS(aimInfo, status);
      } else {
        /*@-immediatetrans@*/
        mesh.meshData = NULL;
        mesh.meshRef = (aimMeshRef *)flightstreamInstance->meshRefIn;
        /*@+immediatetrans@*/

        status = mesh_surfaceMeshData(aimInfo, &flightstreamInstance->groupMap, &mesh);
        AIM_STATUS(aimInfo, status);
        AIM_NOTNULL(mesh.meshData, aimInfo, status);

        /* write the mesh */
        status = aim_writeMesh(aimInfo, MESHWRITER, NULL, &mesh);
        AIM_STATUS(aimInfo, status);
      }
    }

    status = fs_dataTransfer(aimInfo, aimInputs, flightstreamInstance);
    AIM_STATUS(aimInfo, status);

    file_type = "OBJ";
    snprintf(meshFile, PATH_MAX, "%s%s", flightstreamInstance->meshRefIn->fileName, MESHEXTENSION);
  }

  /*
  // compute velocity from standard atmosphere
  alt = 0.3048 * aimInputs[inAltitude-1].vals.real;
  if (alt <= 11000) {
      temp = 288.16 - 0.0065 * alt;
  } else if (alt <= 25000) {
      temp = 216.66;
  } else if (alt < 47000) {
      temp = 216.66 + 0.0030 * (alt - 25000);
  } else if (alt < 53000) {
      temp = 282.66;
  } else if (alt < 79000) {
      temp = 282.66 - 0.0045 * (alt - 53000);
  } else if (alt < 90000) {
      temp = 165.66;
  } else {
      temp = 165.66 + 0.0040 * (alt - 90000);
  }
  vel = aimInputs[inMach-1].vals.real * sqrt(1.40 * 287 * temp);
  */

  // write flightstream script file
  fp = aim_fopen(aimInfo, "script.txt", "w");
  if (fp == NULL) {
      AIM_ERROR(aimInfo, "Cannot open \"script.txt\"");
      status = CAPS_IOERR;
      goto cleanup;
  }

  fprintf(fp, "CLEAR_SOLUTION\n\n");

  if (strcmp(file_type, "CSV") == 0) {

    /* path for vspscript */
    vsp3_root = getenv("VSP3_ROOT");
    if (vsp3_root == NULL) {
      AIM_ERROR(aimInfo, "VSP3_ROOT must be set to directory containing vspscript!");
      status = CAPS_BADVALUE;
      goto cleanup;
    }

    fprintf(fp, "CCS_IMPORT\n");
    fprintf(fp, "CLOSE_COMPONENT_ENDS ENABLE\n");
    fprintf(fp, "UPDATE_PROPERTIES    DISABLE\n");
    fprintf(fp, "CLEAR_EXISTING       ENABLE\n");
    fprintf(fp, "FILE                 %s\n", meshFile);
    fprintf(fp, "\n");
    fprintf(fp, "BOOLEAN_UNITE_PATH\n");
    fprintf(fp, "OPENVSP_PATH %s\n",vsp3_root);
    fprintf(fp, "\n");
    fprintf(fp, "BOOLEAN_UNITE_MESH -1\n\n");
    fprintf(fp, "\n");
  } else {

    fprintf(fp, "IMPORT\n");
    fprintf(fp, "UNITS        %s\n", units->length);
    fprintf(fp, "FILE_TYPE    %s\n", file_type);
    fprintf(fp, "FILE         %s\n", meshFile);
    fprintf(fp, "CLEAR\n");
    fprintf(fp, "\n");
    fprintf(fp, "AUTO_DETECT_BASE_REGIONS\n");
    fprintf(fp, "\n");
  }

  // must be before initialization
  if (numSharp > 0 && sharpSurf != NULL) {
    fprintf(fp, "DETECT_TRAILING_EDGES_BY_SURFACE\n");
    fprintf(fp, "SURFACES %d\n", numSharp);
    for (i = 0; i < numSharp-1; i++)
      fprintf(fp, "%d,", sharpSurf[i]);
    fprintf(fp, "%d\n", sharpSurf[numSharp-1]);
    fprintf(fp, "\n");
  }

  if (numProx > 0 && proxSurf != NULL) {
    fprintf(fp, "SOLVER_PROXIMAL_BOUNDARIES %d\n", numProx);
    for (i = 0; i < numProx; i++)
      fprintf(fp, "%d\n", proxSurf[i]);
    fprintf(fp, "\n");
  }

  // add known base regions
  if (numBase > 0 && baseSurf != NULL) {
    for (i = 0; i < numBase; i++) {
      fprintf(fp, "CREATE_NEW_BASE_REGION %d EMPIRICAL 0.0\n", baseSurf[i]);
      fprintf(fp, "\n");
    }

    for (i = 0; i < numBase; i++) {
      fprintf(fp, "SET_BASE_REGION_TRAILING_EDGES %d\n", i+1);
      fprintf(fp, "\n");
    }
  }

  if (numWTerm > 0 && wtermSurf != NULL) {
    for (i = 0; i < numWTerm; i++) {
      fprintf(fp, "DETECT_WAKE_TERMINATION_NODES_BY_SURFACE %d\n", wtermSurf[i]);
      fprintf(fp, "\n");
    }
  }


  fprintf(fp, "SET_SIMULATION_LENGTH_UNITS %s\n", units->length);
  fprintf(fp, "\n");

  // create a new csys for the moment reference (X-streamwise, Y-left, Z-up)
  fprintf(fp, "CREATE_NEW_COORDINATE_SYSTEM\n");
  fprintf(fp, "EDIT_COORDINATE_SYSTEM\n");
  fprintf(fp, "FRAME 2\n");
  fprintf(fp, "NAME Moment_Reference\n");
  fprintf(fp, "ORIGIN_X %15.5f %s\n", flightstreamInstance->Xref, units->length);
  fprintf(fp, "ORIGIN_Y %15.5f %s\n", flightstreamInstance->Yref, units->length);
  fprintf(fp, "ORIGIN_Z %15.5f %s\n", flightstreamInstance->Zref, units->length);
  fprintf(fp, "VECTOR_X_X 1\n");
  fprintf(fp, "VECTOR_X_Y 0\n");
  fprintf(fp, "VECTOR_X_Z 0\n");
  fprintf(fp, "VECTOR_Y_X 0\n");
  fprintf(fp, "VECTOR_Y_Y 1\n");
  fprintf(fp, "VECTOR_Y_Z 0\n");
  fprintf(fp, "VECTOR_Z_X 0\n");
  fprintf(fp, "VECTOR_Z_Y 0\n");
  fprintf(fp, "VECTOR_Z_Z 1\n");
  fprintf(fp, "SET_SOLVER_ANALYSIS_LOADS_FRAME 2\n");
  fprintf(fp, "\n");

  if (aimInputs[inFluid_Properties-1].nullVal == NotNull) {
    fprintf(fp, "FLUID_PROPERTIES\n");
    fprintf(fp, "DENSITY                     %15.12e\n",   flightstreamInstance->rhoff);
    fprintf(fp, "PRESSURE                    %15.12e\n",   flightstreamInstance->pff);
    fprintf(fp, "SONIC_VELOCITY              %15.12e\n",   flightstreamInstance->sonicspeedff);
    fprintf(fp, "TEMPERATURE                 %15.12e\n",   flightstreamInstance->tff);
    fprintf(fp, "VISCOSITY                   %15.12e\n",   flightstreamInstance->muff);
  } else {
    fprintf(fp, "AIR_ALTITUDE                %15.12e %s\n",   aimInputs[inAltitude-1].vals.real, aimInputs[inAltitude-1].units);
  }
  // SOLVER_SET_REF_MACH_NUMBER does not currently work in FlightStream
  fprintf(fp, "#SOLVER_SET_REF_MACH_NUMBER  %15.12e\n",   aimInputs[inMach    -1].vals.real);
  fprintf(fp, "\n");
  fprintf(fp, "SOLVER_SET_REF_VELOCITY     %15.12e\n",   aimInputs[inReferenceVelocity-1].vals.real);
  fprintf(fp, "\n");
  fprintf(fp, "SOLVER_SET_REF_AREA         %15.12e\n",   flightstreamInstance->Sref);
  fprintf(fp, "\n");
  fprintf(fp, "SOLVER_SET_REF_LENGTH       %15.12e\n",   flightstreamInstance->Cref);
  fprintf(fp, "\n");
  fprintf(fp, "\n");
  if (aimInputs[inMach-1].nullVal != IsNull) {
    fprintf(fp, "SOLVER_SET_MACH_NUMBER      %15.12e\n",   aimInputs[inMach    -1].vals.real);
    fprintf(fp, "\n");
  }
  if (aimInputs[inAlpha-1].nullVal != IsNull) {
    fprintf(fp, "SOLVER_SET_AOA              %15.12e\n",   aimInputs[inAlpha   -1].vals.real);
    fprintf(fp, "\n");
  }
  if (aimInputs[inBeta-1].nullVal != IsNull) {
    fprintf(fp, "SOLVER_SET_SIDESLIP         %15.12e\n",   aimInputs[inBeta    -1].vals.real);
    fprintf(fp, "\n");
  }

  if (aimInputs[inSolverIterations-1].nullVal == NotNull) {
    fprintf(fp, "SOLVER_SET_ITERATIONS       %d\n",   aimInputs[inSolverIterations-1].vals.integer);
    fprintf(fp, "\n");
  }
  if (aimInputs[inSolverConvergence-1].nullVal == NotNull) {
    fprintf(fp, "SOLVER_SET_CONVERGENCE      %15.12e\n",   aimInputs[inSolverConvergence-1].vals.real);
    fprintf(fp, "\n");
  }

  if (aimInputs[inFlightScriptPre-1].nullVal == NotNull) {
    flightscript = aimInputs[inFlightScriptPre-1].vals.string;
    for (i = 0; i < aimInputs[inFlightScriptPre-1].length; i++) {
      fprintf(fp, "%s\n", flightscript);
      flightscript += strlen(flightscript)+1;
    }
  }
  fprintf(fp, "\n");

  fprintf(fp, "INITIALIZE_SOLVER\n");
  fprintf(fp, "SOLVER_MODEL                %s\n", aimInputs[inSolver_Model-1].vals.string);
  if (strcmp(file_type, "CSV") == 0) {
    fprintf(fp, "SURFACES                    -1\n");
  } else {
    fprintf(fp, "SURFACES                    %d\n", flightstreamInstance->groupMap.numAttribute);
    for (i = 0; i < flightstreamInstance->groupMap.numAttribute; i++)
      fprintf(fp, "%d,DISABLE\n", i+1);
  }
  if (aimInputs[inWake_Termination-1].nullVal == IsNull) {
    fprintf(fp, "WAKE_TERMINATION_X          DEFAULT\n");
  } else {
    fprintf(fp, "WAKE_TERMINATION_X          %16.12e\n", aimInputs[inWake_Termination-1].vals.real);
  }
  fprintf(fp, "WALL_COLLISION_AVOIDANCE    %s\n", aimInputs[inWall_Collision_Avoidance-1].vals.integer == (int)true ? "ENABLE" : "DISABLE");
  if (strcasecmp(aimInputs[inSolver_Model-1].vals.string, "INCOMPRESSIBLE") == 0 ||
      strcasecmp(aimInputs[inSolver_Model-1].vals.string, "SUBSONIC_PRANDTL_GLAUERT") == 0 ||
      strcasecmp(aimInputs[inSolver_Model-1].vals.string, "TRANSONIC_FIELD_PANEL") == 0)
    fprintf(fp, "STABILIZATION               ENABLE %16.12e\n", aimInputs[inStabilization-1].vals.real);
  fprintf(fp, "\n");
  if (aimInputs[inMax_Parallel_Threads-1].nullVal == NotNull) {
    fprintf(fp, "SET_MAX_PARALLEL_THREADS %d\n", aimInputs[inMax_Parallel_Threads-1].vals.integer);
    fprintf(fp, "\n");
  }


  if (aimInputs[inSweepRangeMach-1    ].nullVal != IsNull ||
      aimInputs[inSweepRangeVelocity-1].nullVal != IsNull ||
      aimInputs[inSweepRangeAlpha-1   ].nullVal != IsNull ||
      aimInputs[inSweepRangeBeta-1    ].nullVal != IsNull) {

    fprintf(fp, "EXECUTE_SOLVER_SWEEPER\n");
    fprintf(fp, "ANGLE_OF_ATTACK %s\n", aimInputs[inSweepRangeAlpha-1   ].nullVal != IsNull ? "ENABLE" : "DISABLE");
    fprintf(fp, "SIDE_SLIP_ANGLE %s\n", aimInputs[inSweepRangeBeta-1    ].nullVal != IsNull ? "ENABLE" : "DISABLE");
    fprintf(fp, "VELOCITY        %s\n", aimInputs[inSweepRangeMach-1    ].nullVal != IsNull ||
                                        aimInputs[inSweepRangeVelocity-1].nullVal != IsNull ? "ENABLE" : "DISABLE");
    fprintf(fp, "ANGLE_OF_ATTACK_START %15.12e\n", aimInputs[inSweepRangeAlpha-1].vals.reals[0]);
    fprintf(fp, "ANGLE_OF_ATTACK_STOP  %15.12e\n", aimInputs[inSweepRangeAlpha-1].vals.reals[1]);
    fprintf(fp, "ANGLE_OF_ATTACK_DELTA %15.12e\n", aimInputs[inSweepRangeAlpha-1].vals.reals[2]);
    fprintf(fp, "SIDE_SLIP_ANGLE_START %15.12e\n", aimInputs[inSweepRangeBeta-1].vals.reals[0]);
    fprintf(fp, "SIDE_SLIP_ANGLE_STOP  %15.12e\n", aimInputs[inSweepRangeBeta-1].vals.reals[1]);
    fprintf(fp, "SIDE_SLIP_ANGLE_DELTA %15.12e\n", aimInputs[inSweepRangeBeta-1].vals.reals[2]);
    if (aimInputs[inSweepRangeVelocity-1 ].nullVal != IsNull) {
      fprintf(fp, "VELOCITY_START %15.12e\n", aimInputs[inSweepRangeVelocity-1].vals.reals[0]);
      fprintf(fp, "VELOCITY_STOP  %15.12e\n", aimInputs[inSweepRangeVelocity-1].vals.reals[1]);
      fprintf(fp, "VELOCITY_DELTA %15.12e\n", aimInputs[inSweepRangeVelocity-1].vals.reals[2]);
    } else {
      fprintf(fp, "MACH_START %15.12e\n", aimInputs[inSweepRangeMach-1].vals.reals[0]);
      fprintf(fp, "MACH_STOP  %15.12e\n", aimInputs[inSweepRangeMach-1].vals.reals[1]);
      fprintf(fp, "MACH_DELTA %15.12e\n", aimInputs[inSweepRangeMach-1].vals.reals[2]);
    }
    if (aimInputs[inSweepExport_Surface_Data-1 ].nullVal == IsNull) {
      fprintf(fp, "EXPORT_SURFACE_DATA_PER_STEP DISABLE\n");
    } else {
      fprintf(fp, "EXPORT_SURFACE_DATA_PER_STEP %s\n", aimInputs[inSweepExport_Surface_Data-1 ].vals.string);
      status = aim_file(aimInfo, "", aimFile);
      AIM_STATUS(aimInfo, status);
      fprintf(fp, "%s\n", aimFile);
    }
    fprintf(fp, "CLEAR_SOLUTION_AFTER_EACH_RUN %s\n", aimInputs[inSweepClear_Solution_After_Each_Run-1].vals.integer == (int)true ? "ENABLE" : "DISABLE");
    fprintf(fp, "REFERENCE_VELOCITY_EQUALS_FREESTREAM %s\n", aimInputs[inSweepReference_Velocity_Equals_Freestream-1].vals.integer == (int)true ? "ENABLE" : "DISABLE");
    fprintf(fp, "APPEND_TO_EXISTING_SWEEP DISABLE\n");
    status = aim_file(aimInfo, resultsSweepFile, aimFile);
    AIM_STATUS(aimInfo, status);
    fprintf(fp, "%s\n", aimFile);
    fprintf(fp, "\n");

  } else {

    fprintf(fp, "START_SOLVER\n");
    fprintf(fp, "\n");

    fprintf(fp, "EXPORT_SOLVER_ANALYSIS_SPREADSHEET\n");
    status = aim_file(aimInfo, resultsFile, aimFile);
    AIM_STATUS(aimInfo, status);
    fprintf(fp, "%s\n", aimFile);
    fprintf(fp, "\n");

    // add custom exports
    if (aimInputs[inExport_Solver_Analysis-1].nullVal == NotNull) {
      status = aim_file(aimInfo, aimInputs[inProj_Name-1].vals.string, aimFile);
      AIM_STATUS(aimInfo, status);

      export = aimInputs[inExport_Solver_Analysis-1].vals.string;
      for (i = 0; i < aimInputs[inExport_Solver_Analysis-1].length; i++) {
        if (strcasecmp(export, "TECPLOT") == 0) {
          fprintf(fp, "EXPORT_SOLVER_ANALYSIS_TECPLOT\n");
          fprintf(fp, "%s.tec\n", aimFile);
          fprintf(fp, "\n");
        } else if (strcasecmp(export, "VTK") == 0) {
          fprintf(fp, "EXPORT_SOLVER_ANALYSIS_VTK\n");
          fprintf(fp, "%s.vtk\n", aimFile);
          fprintf(fp, "SURFACES -1\n");
          fprintf(fp, "\n");
        } else if (strcasecmp(export, "BDF") == 0) {
          fprintf(fp, "EXPORT_SOLVER_ANALYSIS_PLOAD_BDF\n");
          fprintf(fp, "%s.bdf\n", aimFile);
          fprintf(fp, "\n");
        } else {
          AIM_ERROR(aimInfo, "Unknown Export_Solver_Analysis = %s", export);
          status = CAPS_BADVALUE;
          goto cleanup;
        }
        export += strlen(export)+1;
      }
    }

    status = aim_getBounds(aimInfo, &nbound, &boundNames);
    AIM_STATUS(aimInfo, status);

    if (nbound > 0) {
#ifdef ANALYSIS_PLOAD_BDF
      status = aim_rmFile(aimInfo, loadbdf);
      AIM_STATUS(aimInfo, status);

      fprintf(fp, "EXPORT_SOLVER_ANALYSIS_PLOAD_BDF\n");
      fprintf(fp, "%s\n", loadbdf);
      fprintf(fp, "SURFACES -1\n");
      fprintf(fp, "\n");
#elif defined(ANALYSIS_VTK)
      status = aim_rmFile(aimInfo, cpvtk);
      AIM_STATUS(aimInfo, status);

      fprintf(fp, "EXPORT_SOLVER_ANALYSIS_VTK\n");
      fprintf(fp, "%s\n", cpvtk);
      fprintf(fp, "SURFACES -1\n\n");
      fprintf(fp, "SET_VTK_EXPORT_VARIABLES 2 DISABLE\n");
      fprintf(fp, "CP\n");
      fprintf(fp, "PSTATIC\n");
      fprintf(fp, "\n");
#elif defined(ANALYSIS_CSV)
      status = aim_rmFile(aimInfo, pcsv);
      AIM_STATUS(aimInfo, status);
      status = aim_file(aimInfo, pcsv, aimFile);
      AIM_STATUS(aimInfo, status);

      fprintf(fp, "EXPORT_SOLVER_ANALYSIS_CSV\n");
      fprintf(fp, "%s\n", aimFile);
      fprintf(fp, "FORMAT PRESSURE\n");
      fprintf(fp, "UNITS PASCALS\n");
      fprintf(fp, "FRAME 1\n");
      fprintf(fp, "SURFACES -1\n");
      fprintf(fp, "\n");

      status = aim_rmFile(aimInfo, cpcsv);
      AIM_STATUS(aimInfo, status);
      status = aim_file(aimInfo, cpcsv, aimFile);
      AIM_STATUS(aimInfo, status);

      fprintf(fp, "EXPORT_SOLVER_ANALYSIS_CSV\n");
      fprintf(fp, "%s\n", aimFile);
      fprintf(fp, "FORMAT CP-REFERENCE\n");
      fprintf(fp, "UNITS PASCALS\n");
      fprintf(fp, "FRAME 1\n");
      fprintf(fp, "SURFACES -1\n");
      fprintf(fp, "\n");
#else
#error "must define ANALYSIS_PLOAD_BDF or ANALYSIS_CSV"
#endif
    }
  }

  fprintf(fp, "\n");
  fprintf(fp, "EXPORT_LOG\n");
  status = aim_file(aimInfo, "log.txt", aimFile);
  AIM_STATUS(aimInfo, status);
  fprintf(fp, "%s\n", aimFile);
  fprintf(fp, "\n");

  fprintf(fp, "\n");
  fprintf(fp, "OUTPUT_SETTINGS_AND_STATUS\n");
  status = aim_file(aimInfo, settingsFile, aimFile);
  AIM_STATUS(aimInfo, status);
  fprintf(fp, "%s\n", aimFile);
  fprintf(fp, "\n");

  if (aimInputs[inSaveFSM-1].vals.integer == (int)true) {
    fprintf(fp, "SAVEAS\n");
    status = aim_file(aimInfo, aimInputs[inProj_Name-1].vals.string, aimFile);
    AIM_STATUS(aimInfo, status);
    fprintf(fp, "%s.fsm\n", aimFile);
    fprintf(fp, "\n");
  }

  if (aimInputs[inFlightScriptPost-1].nullVal == NotNull) {
    flightscript = aimInputs[inFlightScriptPost-1].vals.string;
    for (i = 0; i < aimInputs[inFlightScriptPost-1].length; i++) {
      fprintf(fp, "%s\n", flightscript);
      flightscript += strlen(flightscript)+1;
    }
  }
  fprintf(fp, "\n");


  fprintf(fp, "CLOSE_FLIGHTSTREAM\n");

  fclose(fp); fp = NULL;

cleanup:
  AIM_FREE(sharpSurf);
  AIM_FREE(proxSurf);
  AIM_FREE(baseSurf);
  AIM_FREE(wtermSurf);
  AIM_FREE(boundNames);
  if (fp != NULL) fclose(fp);

  aim_freeMeshData(mesh.meshData);
  AIM_FREE(mesh.meshData);

  return status;
}


/**********************************************************************/
/* aimExecute - execute FlightStream                                  */
/**********************************************************************/
int aimExecute(/*@unused@*/ const void *instStore, void *aimInfo,
               int *state)
{
  /*! \page aimExecuteFLIGHTSTREAM AIM Execution
   *
   * If auto execution is enabled when creating an FlightStream AIM,
   * the AIM will execute FlightStream just-in-time on Linux with the command line:
   *
   * \code{.sh}
   * FlightStream script.txt > flightstreamOut.txt
   * \endcode
   *
   * and on Windows with the command:
   *
   * \code{.sh}
   * FlightStream -hidden -script script.txt > flightstreamOut.txt
   * \endcode
   *
   * In both cases the FlightStream executable is assumed to in the PATH environment variable.
   *
   * The analysis can be also be explicitly executed with caps_execute in the C-API
   * or via Analysis.runAnalysis in the pyCAPS API.
   *
   * Calling preAnalysis and postAnalysis is NOT allowed when auto execution is enabled.
   *
   * Auto execution can also be disabled when creating an FlightStream AIM object.
   * In this mode, caps_execute and Analysis.runAnalysis can be used to run the analysis,
   * or FlightStream can be executed by calling preAnalysis, system, and posAnalysis as demonstrated
   * below with a pyCAPS example:
   *
   * \code{.py}
   * print ("\n\preAnalysis......")
   * flightstream.preAnalysis()
   *
   * print ("\n\nRunning......")
   * flightstream.system("FlightStream.exe -hidden -script script.txt"); # Run via system call in inputs analysis directory
   *
   * print ("\n\postAnalysis......")
   * flightstream.postAnalysis()
   * \endcode
   */
  int status = CAPS_SUCCESS;
  capsValue *flightstream = NULL;
#ifdef WIN32
  capsValue *hidden = NULL;
#endif

#define fslen PATH_MAX+42
  char fscmd[fslen];

  *state = 0;

  aim_getValue(aimInfo, inFlightStream, ANALYSISIN, &flightstream);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(flightstream, aimInfo, status);

#ifdef WIN32
  aim_getValue(aimInfo, inHidden, ANALYSISIN, &hidden);
  AIM_STATUS(aimInfo, status);
  AIM_NOTNULL(flightstream, aimInfo, status);

  strcpy(fscmd, flightstream->vals.string);
  if (hidden->vals.integer == (int)true)
    strncat(fscmd, " -hidden", fslen-strlen(fscmd));
  strncat(fscmd, " -script script.txt > flightstreamOut.txt", fslen-strlen(fscmd));
#else
  snprintf(fscmd, fslen, "%s script.txt > flightstreamOut.txt", flightstream->vals.string);
#endif
  //printf(" Executing: %s\n", fscmd);
  status = aim_system(aimInfo, "", fscmd);
  AIM_STATUS(aimInfo, status, "Failed to execute: %s", fscmd);

#undef fslen
cleanup:
  return status;
}


/**********************************************************************/
/* aimPostAnalysis - read FlightStream output file                    */
/**********************************************************************/

int
aimPostAnalysis(/*@unused@*/ void *instStore,       /* (in)  AIM instance storage */
                /*@unused@*/ void *aimInfo,         /* (in)  AIM context */
                /*@unused@*/ int restart,           /* (in)  0=normal, 1=restart */
                /*@unused@*/ capsValue aimInputs[]) /* (in)  array of analysis inputs */
{
  int    status = CAPS_SUCCESS;
  int    found = (int)false;

  int i;
  char tmp[128];
  double dtmp;
  long pos = 0;
  int nlen;

#if 0
  int j, k, idv, irow, icol, ibody; // Indexing
  int index;

  int numFunctional=0, nGeomIn = 0, numDesignVariable = 0;
  int **functional_map=NULL;
  double **functional_xyz=NULL;
  double functional_dvar;

  int numNode = 0, *numPoint=NULL;
  const char *name;
  char **names=NULL;
  double **dxyz = NULL;

  const char *projectName =NULL;

  capsValue *values=NULL, *geomInVal;

  // Mesh reference obtained from meshing AIM
  const aimMeshRef *meshRef = NULL;
#endif
  // outputs from FlightStream
  double coeffs[9];
  const char* coeffNames[9] = {"Cx", "Cy", "Cz", "CL", "CDi", "CDo", "CMx", "CMy", "CMz"};
  capsValue surfVal;

  capsValue *values[NUMOUTPUT] = {NULL};

  //cfdDesignVariableStruct *dvar=NULL;

  size_t linecap=0, linelen=0;
  char   *line=NULL, *rest=NULL, *token=NULL;

  FILE   *fp=NULL;
  aimStorage *flightstreamInstance = (aimStorage *)instStore;
  const cfdUnitsStruct *units = &flightstreamInstance->units;
  double speedScale = 1.0;

  /* -------------------------------------------------------------- */

  aim_initValue(&surfVal);

  AIM_NOTNULL(aimInputs, aimInfo, status);

  values[ 0] = &flightstreamInstance->Alpha;
  values[ 1] = &flightstreamInstance->Beta;
  values[ 2] = &flightstreamInstance->Mach;
  values[ 3] = &flightstreamInstance->Velocity;
  values[ 4] = &flightstreamInstance->Density;
  values[ 5] = &flightstreamInstance->Pressure;
  values[ 6] = &flightstreamInstance->SonicSpeed;
  values[ 7] = &flightstreamInstance->Temperature;
  values[ 8] = &flightstreamInstance->Viscosity;
  values[ 9] = &flightstreamInstance->Cx;
  values[10] = &flightstreamInstance->Cy;
  values[11] = &flightstreamInstance->Cz;
  values[12] = &flightstreamInstance->CL;
  values[13] = &flightstreamInstance->CDi;
  values[14] = &flightstreamInstance->CDo;
  values[15] = &flightstreamInstance->CMx;
  values[16] = &flightstreamInstance->CMy;
  values[17] = &flightstreamInstance->CMz;

  for (i = 0; i < NUMOUTPUT; i++)
    aim_freeValue(values[i]);

#ifdef DEBUG
  printf("flightstreamAIM/aimPostAnalysis()\n");
#endif


  fp = aim_fopen(aimInfo, settingsFile, "r");
  if (fp == NULL) {
    AIM_ERROR(aimInfo, "Unable to open \"%s\"\n", settingsFile);
    status = CAPS_IOERR;
    goto cleanup;
  }

  /* read the solver settings */
  while (getline(&line, &linecap, fp) > 0) {
    AIM_NOTNULL(line, aimInfo, status);

    rest = line;
    token = strtok_r(rest, ",", &rest);
    if (strcmp(token,"Density") == 0) {
      token = strtok_r(rest, ",", &rest);
      status = sscanf(token,"%lf",&dtmp);
      if (status != 1) AIM_STATUS(aimInfo, (status = CAPS_IOERR));
      status = aim_convert(aimInfo, 1, rest, &dtmp, "kg/m^3", &flightstreamInstance->rhoff);
      AIM_STATUS(aimInfo, status);
    } else if (strcmp(token,"Pressure") == 0) {
      token = strtok_r(rest, ",", &rest);
      status = sscanf(token,"%lf",&dtmp);
      if (status != 1) AIM_STATUS(aimInfo, (status = CAPS_IOERR));
      status = aim_convert(aimInfo, 1, rest, &dtmp, "Pa", &flightstreamInstance->pff);
      AIM_STATUS(aimInfo, status);
    } else if (strcmp(token,"Sonic Velocity") == 0) {
      token = strtok_r(rest, ",", &rest);
      status = sscanf(token,"%lf",&dtmp);
      if (status != 1) AIM_STATUS(aimInfo, (status = CAPS_IOERR));
      status = aim_convert(aimInfo, 1, rest, &dtmp, "m/s", &flightstreamInstance->sonicspeedff);
      AIM_STATUS(aimInfo, status);
    } else if (strcmp(token,"Temperature") == 0) {
      token = strtok_r(rest, ",", &rest);
      status = sscanf(token,"%lf",&dtmp);
      if (status != 1) AIM_STATUS(aimInfo, (status = CAPS_IOERR));
      status = aim_convert(aimInfo, 1, rest, &dtmp, "K", &flightstreamInstance->tff);
      AIM_STATUS(aimInfo, status);
    } else if (strcmp(token,"Viscosity") == 0) {
      token = strtok_r(rest, ",", &rest);
      status = sscanf(token,"%lf",&dtmp);
      if (status != 1) AIM_STATUS(aimInfo, (status = CAPS_IOERR));
      status = aim_convert(aimInfo, 1, rest, &dtmp, "Pa-sec", &flightstreamInstance->muff);
      AIM_STATUS(aimInfo, status);
    }
  }

  AIM_STRDUP(flightstreamInstance->Alpha.units      , "degree"    , aimInfo, status);
  AIM_STRDUP(flightstreamInstance->Beta.units       , "degree"    , aimInfo, status);
  AIM_STRDUP(flightstreamInstance->Velocity.units   , units->speed, aimInfo, status);
  AIM_STRDUP(flightstreamInstance->Density.units    , "kg/m^3"    , aimInfo, status);
  AIM_STRDUP(flightstreamInstance->Pressure.units   , "Pa"        , aimInfo, status);
  AIM_STRDUP(flightstreamInstance->SonicSpeed.units , "m/s"       , aimInfo, status);
  AIM_STRDUP(flightstreamInstance->Temperature.units, "K"         , aimInfo, status);
  AIM_STRDUP(flightstreamInstance->Viscosity.units  , "Pa-sec"    , aimInfo, status);

  speedScale = 1;
  status = aim_convert(aimInfo, 1, units->speed, &speedScale, "m/s", &speedScale);
  AIM_STATUS(aimInfo, status);

  fclose(fp);
  fp = NULL;

  if (aimInputs[inSweepRangeMach-1    ].nullVal != IsNull ||
      aimInputs[inSweepRangeVelocity-1].nullVal != IsNull ||
      aimInputs[inSweepRangeAlpha-1   ].nullVal != IsNull ||
      aimInputs[inSweepRangeBeta-1    ].nullVal != IsNull) {

    /* sweeper toolbox analysis */
    fp = aim_fopen(aimInfo, resultsSweepFile, "r");
    if (fp == NULL) {
      AIM_ERROR(aimInfo, "Unable to open \"%s\"\n", resultsSweepFile);
      status = CAPS_IOERR;
      goto cleanup;
    }

    /* read the file until we find a line that starts with "     Total" */
    while (!feof(fp)) {
      linelen = getline(&line, &linecap, fp);
      if (linelen == 0) AIM_STATUS(aimInfo, (status = CAPS_IOERR));

      AIM_NOTNULL(line, aimInfo, status);

      rest = line;
      while(*rest == ' ') rest++;
      if (strncmp(rest, "AOA (deg)", 9) == 0) {
        // skip the '---' below the header
        linelen = getline(&line, &linecap, fp);
        if (linelen == 0) AIM_STATUS(aimInfo, (status = CAPS_IOERR));
        pos = ftell(fp);
        break;
      }
    }

    // count the number of rows of data
    nlen = 0;
    while (!feof(fp)) {
      linelen = getline(&line, &linecap, fp);
      if (linelen == 0) AIM_STATUS(aimInfo, (status = CAPS_IOERR));

      AIM_NOTNULL(line, aimInfo, status);

      rest = line;
      while(*rest == ' ') rest++;
      if (strncmp(rest, "---------", 9) != 0)
        nlen++;
      else
        break;
    }

    for (i = 0; i < NUMOUTPUT; i++) {
      values[i]->type = Double;
      values[i]->dim  = Vector;
      values[i]->nrow = values[i]->length = nlen;
      if (nlen > 1) {
        AIM_ALLOC(values[i]->vals.reals, values[i]->length, double, aimInfo, status);
      }
    }

    // go back to the beginning of the data
    status = fseek(fp, pos, SEEK_SET);
    if (status != 0) { status = CAPS_IOERR; AIM_STATUS(aimInfo, status); }

    i = 0;
    while (!feof(fp)) {
      linelen = getline(&line, &linecap, fp);
      if (linelen == 0) AIM_STATUS(aimInfo, (status = CAPS_IOERR));

      AIM_NOTNULL(line, aimInfo, status);

      rest = line;
      while(*rest == ' ') rest++;
      if (strncmp(rest, "---------", 9) == 0) break;

      if (nlen > 1) {
        status = sscanf(rest, "%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf",
                        &(flightstreamInstance->Alpha.vals.reals[i]),
                        &(flightstreamInstance->Beta.vals.reals[i]),
                        &(flightstreamInstance->Velocity.vals.reals[i]),
                        &(flightstreamInstance->Cx.vals.reals[i]),
                        &(flightstreamInstance->Cy.vals.reals[i]),
                        &(flightstreamInstance->Cz.vals.reals[i]),
                        &(flightstreamInstance->CL.vals.reals[i]),
                        &(flightstreamInstance->CDi.vals.reals[i]),
                        &(flightstreamInstance->CDo.vals.reals[i]),
                        &(flightstreamInstance->CMx.vals.reals[i]),
                        &(flightstreamInstance->CMy.vals.reals[i]),
                        &(flightstreamInstance->CMz.vals.reals[i]));

        flightstreamInstance->Mach.vals.reals[i]        = (flightstreamInstance->Velocity.vals.reals[i]*speedScale)/flightstreamInstance->sonicspeedff;
        flightstreamInstance->Density.vals.reals[i]     = flightstreamInstance->rhoff;
        flightstreamInstance->Pressure.vals.reals[i]    = flightstreamInstance->pff;
        flightstreamInstance->SonicSpeed.vals.reals[i]  = flightstreamInstance->sonicspeedff;
        flightstreamInstance->Temperature.vals.reals[i] = flightstreamInstance->tff;
        flightstreamInstance->Viscosity.vals.reals[i]   = flightstreamInstance->muff;

      } else {
        status = sscanf(rest, "%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf",
                        &(flightstreamInstance->Alpha.vals.real),
                        &(flightstreamInstance->Beta.vals.real),
                        &(flightstreamInstance->Velocity.vals.real),
                        &(flightstreamInstance->Cx.vals.real),
                        &(flightstreamInstance->Cy.vals.real),
                        &(flightstreamInstance->Cz.vals.real),
                        &(flightstreamInstance->CL.vals.real),
                        &(flightstreamInstance->CDi.vals.real),
                        &(flightstreamInstance->CDo.vals.real),
                        &(flightstreamInstance->CMx.vals.real),
                        &(flightstreamInstance->CMy.vals.real),
                        &(flightstreamInstance->CMz.vals.real));

        flightstreamInstance->Mach.vals.real        = (flightstreamInstance->Velocity.vals.real*speedScale)/flightstreamInstance->sonicspeedff;
        flightstreamInstance->Density.vals.real     = flightstreamInstance->rhoff;
        flightstreamInstance->Pressure.vals.real    = flightstreamInstance->pff;
        flightstreamInstance->SonicSpeed.vals.real  = flightstreamInstance->sonicspeedff;
        flightstreamInstance->Temperature.vals.real = flightstreamInstance->tff;
        flightstreamInstance->Viscosity.vals.real   = flightstreamInstance->muff;

      }
      if (status != 12) {
        AIM_ERROR(aimInfo, "Failed to read sweep forces from \"%s\"\n", resultsSweepFile);
        status = CAPS_IOERR;
        goto cleanup;
      }

      i++;
    }

    fclose(fp);
    fp = NULL;

  } else {

    for (i = 0; i < NUMOUTPUT; i++) {
      values[i]->type = Double;
      values[i]->dim  = Scalar;
      values[i]->nrow = values[i]->ncol = values[i]->length = 1;
    }

    flightstreamInstance->Alpha.vals.real       = aimInputs[inAlpha-1].vals.real;
    flightstreamInstance->Beta.vals.real        = aimInputs[inBeta-1].vals.real;
    flightstreamInstance->Mach.vals.real        = aimInputs[inMach-1].vals.real;
    flightstreamInstance->Velocity.vals.real    = aimInputs[inMach-1].vals.real*flightstreamInstance->sonicspeedff/speedScale;
    flightstreamInstance->Density.vals.real     = flightstreamInstance->rhoff;
    flightstreamInstance->Pressure.vals.real    = flightstreamInstance->pff;
    flightstreamInstance->SonicSpeed.vals.real  = flightstreamInstance->sonicspeedff;
    flightstreamInstance->Temperature.vals.real = flightstreamInstance->tff;
    flightstreamInstance->Viscosity.vals.real   = flightstreamInstance->muff;

    /* single point analysis */
    fp = aim_fopen(aimInfo, resultsFile, "r");
    if (fp == NULL) {
      AIM_ERROR(aimInfo, "Unable to open \"%s\"\n", resultsFile);
      status = CAPS_IOERR;
      goto cleanup;
    }

    /* read the file until we find a line that starts with "     Total" */
    while (!feof(fp)) {
      linelen = getline(&line, &linecap, fp);
      if (linelen == 0) AIM_STATUS(aimInfo, (status = CAPS_IOERR));

      AIM_NOTNULL(line, aimInfo, status);

      rest = line;
      while(*rest == ' ') rest++;
      token = strtok_r(rest, ",", &rest);
      if (token != NULL) {
        // deal with possible null characters in the file...
        if (rest == NULL)
          rest = token+strlen(token);
        while ((rest - line)/sizeof(char) < linelen && (rest[0] == '\0' || rest[0] == ',')) {
          rest++;
        }

        if (strcmp(token, "Total") == 0) {

          /* now that we have found the line, get the various coefficients */
          status = sscanf(rest, "%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf",
                          &(flightstreamInstance->Cx.vals.real),
                          &(flightstreamInstance->Cy.vals.real),
                          &(flightstreamInstance->Cz.vals.real),
                          &(flightstreamInstance->CL.vals.real),
                          &(flightstreamInstance->CDi.vals.real),
                          &(flightstreamInstance->CDo.vals.real),
                          &(flightstreamInstance->CMx.vals.real),
                          &(flightstreamInstance->CMy.vals.real),
                          &(flightstreamInstance->CMz.vals.real));
          if (status != 9) {
            AIM_ERROR(aimInfo, "Failed to read total forces from \"%s\"\n", resultsFile);
            status = CAPS_IOERR;
            goto cleanup;
          }

          found = (int)true;
          break;

        } else if (rest != NULL && restart == 0) {

          /* try to read a component output */
          status = sscanf(rest, "%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%lf",
                          &coeffs[0],
                          &coeffs[1],
                          &coeffs[2],
                          &coeffs[3],
                          &coeffs[4],
                          &coeffs[5],
                          &coeffs[6],
                          &coeffs[7],
                          &coeffs[8]);

          if (status == 9) {

            surfVal.type = Tuple;
            AIM_ALLOC(surfVal.vals.tuple, 9, capsTuple, aimInfo, status);
            for (i = 0; i < 9; i++) {
              surfVal.vals.tuple[i].name  = NULL;
              surfVal.vals.tuple[i].value = NULL;
            }
            surfVal.length = surfVal.nrow = 9;
            surfVal.dim = Vector;

            for (i = 0; i < 9; i++) {
              AIM_STRDUP(surfVal.vals.tuple[i].name , coeffNames[i], aimInfo, status);
              snprintf(tmp, 128, "%16.12e", coeffs[i]);
              AIM_STRDUP(surfVal.vals.tuple[i].value, tmp, aimInfo, status);
            }

            /* create the dynamic output */
            status = aim_makeDynamicOutput(aimInfo, token, &surfVal);
            AIM_STATUS(aimInfo, status);
          }
        }
      }
    }

    fclose(fp);
    fp = NULL;

    if (found == (int)false) {
      /* getting here means we did not find results */
      AIM_ERROR(aimInfo, "Did not find \"Total\" results\n");
      status = CAPS_IOERR;
      goto cleanup;
    }
  }



#if 0
  if (inputs[inDesign_Functional-1].nullVal == NotNull) {

    // Get mesh
    meshRef = flightstreamInstance->meshRefIn;
    AIM_NOTNULL(meshRef, aimInfo, status);

    /* check for GeometryIn variables*/
    nGeomIn = 0;
    for (i = 0; i < flightstreamInstance->design.numDesignVariable; i++) {

      name = flightstreamInstance->design.designVariable[i].name;

      // Loop over the geometry in values and compute sensitivities for all bodies
      index = aim_getIndex(aimInfo, name, GEOMETRYIN);
      if (index == CAPS_NOTFOUND) continue;
      if (index < CAPS_SUCCESS ) {
        status = index;
        AIM_STATUS(aimInfo, status);
      }

      if(aim_getGeomInType(aimInfo, index) != 0) {
        AIM_ERROR(aimInfo, "GeometryIn value %s is a configuration parameter and not a valid design parameter - can't get sensitivity\n",
                  name);
        status = CAPS_BADVALUE;
        goto cleanup;
      }

      nGeomIn++;
    }

    projectName = inputs[inProj_Name-1].vals.string;

    // Read <Proj_Name>.sens
    snprintf(tmp, 128, "%s%s", projectName, ".sens");
    fp = aim_fopen(aimInfo, tmp, "r");
    if (fp == NULL) {
      AIM_ERROR(aimInfo, "Unable to open: %s", tmp);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // Number of nodes and functionals and AnalysIn design variables in the file
    status = fscanf(fp, "%d %d", &numFunctional, &numDesignVariable);
    if (status == EOF || status != 2) {
      AIM_ERROR(aimInfo, "Failed to read sens file number of functionals and analysis design variables");
      status = CAPS_IOERR; goto cleanup;
    }
    if (flightstreamInstance->design.numDesignVariable != numDesignVariable+nGeomIn) {
      AIM_ERROR(aimInfo, "Incorrect number of AnalysisIn derivatives in sens file. Expected %d and found %d",
                flightstreamInstance->design.numDesignVariable-nGeomIn, numDesignVariable);
      status = CAPS_IOERR; goto cleanup;
    }

    AIM_ALLOC(numPoint, numFunctional, int, aimInfo, status);
    for (i = 0; i < numFunctional; i++) numPoint[i] = 0;

    AIM_ALLOC(functional_map, numFunctional, int*, aimInfo, status);
    for (i = 0; i < numFunctional; i++) functional_map[i] = NULL;

    AIM_ALLOC(functional_xyz, numFunctional, double*, aimInfo, status);
    for (i = 0; i < numFunctional; i++) functional_xyz[i] = NULL;

    AIM_ALLOC(names, numFunctional, char*, aimInfo, status);
    for (i = 0; i < numFunctional; i++) names[i] = NULL;

    AIM_ALLOC(values, numFunctional, capsValue, aimInfo, status);
    for (i = 0; i < numFunctional; i++) aim_initValue(&values[i]);

    for (i = 0; i < numFunctional; i++) {
      values[i].type = DoubleDeriv;

      /* allocate derivatives */
      AIM_ALLOC(values[i].derivs, flightstreamInstance->design.numDesignVariable, capsDeriv, aimInfo, status);
      for (idv = 0; idv < flightstreamInstance->design.numDesignVariable; idv++) {
        values[i].derivs[idv].name  = NULL;
        values[i].derivs[idv].deriv = NULL;
        values[i].derivs[idv].len_wrt = 0;
      }
      values[i].nderiv = flightstreamInstance->design.numDesignVariable;
    }

    // Read in Functional name, value and dFunctinoal/dxyz
    for (i = 0; i < numFunctional; i++) {

      status = fscanf(fp, "%s", tmp);
      if (status == EOF) {
        AIM_ERROR(aimInfo, "Failed to read sens file functional name");
        status = CAPS_IOERR; goto cleanup;
      }

      AIM_STRDUP(names[i], tmp, aimInfo, status);

      status = fscanf(fp, "%lf", &values[i].vals.real);
      if (status == EOF || status != 1) {
        AIM_ERROR(aimInfo, "Failed to read sens file functional value");
        status = CAPS_IOERR; goto cleanup;
      }

      status = fscanf(fp, "%d", &numPoint[i]);
      if (status == EOF || status != 1) {
        AIM_ERROR(aimInfo, "Failed to read sens file number of points");
        status = CAPS_IOERR; goto cleanup;
      }

      AIM_ALLOC(functional_map[i],   numPoint[i], int   , aimInfo, status);
      AIM_ALLOC(functional_xyz[i], 3*numPoint[i], double, aimInfo, status);

      for (j = 0; j < numPoint[i]; j++) {
        status = fscanf(fp, "%d %lf %lf %lf", &functional_map[i][j],
                        &functional_xyz[i][3*j+0],
                        &functional_xyz[i][3*j+1],
                        &functional_xyz[i][3*j+2]);
        if (status == EOF || status != 4) {
          AIM_ERROR(aimInfo, "Failed to read sens file data");
          status = CAPS_IOERR; goto cleanup;
        }
      }


      /* read additional derivatives from .sens file */
      for (k = nGeomIn; k < flightstreamInstance->design.numDesignVariable; k++) {

        /* get derivative name */
        status = fscanf(fp, "%s", tmp);
        if (status == EOF) {
          AIM_ERROR(aimInfo, "Failed to read sens file design variable name");
          status = CAPS_IOERR; goto cleanup;
        }

        found = (int)false;
        for (idv = 0; idv < flightstreamInstance->design.numDesignVariable; idv++)
          if ( strcasecmp(flightstreamInstance->design.designVariable[idv].name, tmp) == 0) {
            found = (int)true;
            break;
          }
        if (found == (int)false) {
          AIM_ERROR(aimInfo, "Design variable '%s' in sens file not in Design_Varible input", tmp);
          status = CAPS_IOERR; goto cleanup;
        }

        AIM_STRDUP(values[i].derivs[idv].name, tmp, aimInfo, status);

        status = fscanf(fp, "%d", &values[i].derivs[idv].len_wrt);
        if (status == EOF || status != 1) {
          AIM_ERROR(aimInfo, "Failed to read sens file number of design variable derivatives");
          status = CAPS_IOERR; goto cleanup;
        }

        AIM_ALLOC(values[i].derivs[idv].deriv, values[i].derivs[idv].len_wrt, double, aimInfo, status);
        for (j = 0; j < values[i].derivs[idv].len_wrt; j++) {

          status = fscanf(fp, "%lf", &values[i].derivs[idv].deriv[j]);
          if (status == EOF || status != 1) {
            AIM_ERROR(aimInfo, "Failed to read sens file design variable derivative");
            status = CAPS_IOERR; goto cleanup;
          }
        }
      }
    }

    AIM_ALLOC(dxyz, meshRef->nmap, double*, aimInfo, status);
    for (ibody = 0; ibody < meshRef->nmap; ibody++) dxyz[ibody] = NULL;

    /* set derivatives */
    for (idv = 0; idv < flightstreamInstance->design.numDesignVariable; idv++) {

      name = flightstreamInstance->design.designVariable[idv].name;

      // Loop over the geometry in values and compute sensitivities for all bodies
      index = aim_getIndex(aimInfo, name, GEOMETRYIN);
      status = aim_getValue(aimInfo, index, GEOMETRYIN, &geomInVal);
      if (status == CAPS_BADINDEX) continue;
      AIM_STATUS(aimInfo, status);

      for (i = 0; i < numFunctional; i++) {
        AIM_STRDUP(values[i].derivs[idv].name, name, aimInfo, status);

        AIM_ALLOC(values[i].derivs[idv].deriv, geomInVal->length, double, aimInfo, status);
        values[i].derivs[idv].len_wrt  = geomInVal->length;
        for (j = 0; j < geomInVal->length; j++)
          values[i].derivs[idv].deriv[j] = 0;
      }

      for (irow = 0; irow < geomInVal->nrow; irow++) {
        for (icol = 0; icol < geomInVal->ncol; icol++) {

          // get the sensitivity for each body
          for (ibody = 0; ibody < meshRef->nmap; ibody++) {
            if (meshRef->maps[ibody].tess == NULL) continue;
            status = aim_tessSensitivity(aimInfo,
                                         name,
                                         irow+1, icol+1, // row, col
                                         meshRef->maps[ibody].tess,
                                         &numNode, &dxyz[ibody]);
            AIM_STATUS(aimInfo, status, "Sensitivity for: %s\n", name);
            AIM_NOTNULL(dxyz[ibody], aimInfo, status);
          }

          for (i = 0; i < numFunctional; i++) {
            functional_dvar = values[i].derivs[idv].deriv[geomInVal->ncol*irow + icol];

            for (j = 0; j < numPoint[i]; j++) {
              k = functional_map[i][j]-1; // 1-based indexing into surface mesh

              functional_dvar += functional_xyz[i][3*j+0]*dxyz[ibody][3*k + 0]  // dx/dGeomIn
                                                                      + functional_xyz[i][3*j+1]*dxyz[ibody][3*k + 1]  // dy/dGeomIn
                                                                                                             + functional_xyz[i][3*j+2]*dxyz[ibody][3*k + 2]; // dz/dGeomIn
            }
            values[i].derivs[idv].deriv[geomInVal->ncol*irow + icol] = functional_dvar;
          }

          for (ibody = 0; ibody < meshRef->nmap; ibody++)
            AIM_FREE(dxyz[ibody]);
        }
      }
    }

    /* create the dynamic output */
    for (i = 0; i < numFunctional; i++) {
      status = aim_makeDynamicOutput(aimInfo, names[i], &values[i]);
      AIM_STATUS(aimInfo, status);
    }
  }
#endif

  status = CAPS_SUCCESS;

cleanup:
  if (fp   != NULL) fclose(fp);
  if (line != NULL) free(line);
#if 0
  if (functional_map != NULL) {
    for (i = 0; i < numFunctional; i++)
      AIM_FREE(functional_map[i]);
    AIM_FREE(functional_map);
  }
  if (functional_xyz != NULL) {
    for (i = 0; i < numFunctional; i++)
      AIM_FREE(functional_xyz[i]);
    AIM_FREE(functional_xyz);
  }
  AIM_FREE(numPoint);
  if (names != NULL) {
    for (i = 0; i < numFunctional; i++)
      AIM_FREE(names[i]);
    AIM_FREE(names);
  }
  if (meshRef != NULL && dxyz != NULL) {
    for (ibody = 0; ibody < meshRef->nmap; ibody++)
      AIM_FREE(dxyz[ibody]);
    AIM_FREE(dxyz);
  }
  if (values != NULL) {
    for (i = 0; i < numFunctional; i++)
      aim_freeValue(&values[i]);
    AIM_FREE(values);
  }
#endif
  return status;
}


/**********************************************************************/
/* aimCalcOutput - retreive FlightStream output information           */
/**********************************************************************/

int
aimCalcOutput(/*@unused@*/ void *instStore,    /* (in)  AIM instance storage */
              /*@unused@*/ void *aimInfo,      /* (in)  AIM context */
              /*@unused@*/ int index,          /* (in)  analysis output */
              /*@unused@*/ capsValue *outval)  /* (in)  pointer to capsValue to fill */
{
  int status = CAPS_SUCCESS;

  aimStorage *flightstreamInstance = (aimStorage *)instStore;

  /* -------------------------------------------------------------- */

#ifdef DEBUG
  printf("flightstreamAIM/aimCalcOutput(index=%d)\n", index);
#endif

  if        (index == outAlpha) {
    *outval = flightstreamInstance->Alpha;
    aim_initValue(&flightstreamInstance->Alpha);
  } else if (index == outBeta) {
    *outval = flightstreamInstance->Beta;
    aim_initValue(&flightstreamInstance->Beta);
  } else if (index == outMach) {
    *outval = flightstreamInstance->Mach;
    aim_initValue(&flightstreamInstance->Mach);
  } else if (index == outVelocity) {
    *outval = flightstreamInstance->Velocity;
    aim_initValue(&flightstreamInstance->Velocity);
  } else if (index == outDensity) {
    *outval = flightstreamInstance->Density;
    aim_initValue(&flightstreamInstance->Density);
  } else if (index == outPressure) {
    *outval = flightstreamInstance->Pressure;
    aim_initValue(&flightstreamInstance->Pressure);
  } else if (index == outSonicSpeed) {
    *outval = flightstreamInstance->SonicSpeed;
    aim_initValue(&flightstreamInstance->SonicSpeed);
  } else if (index == outTemperature) {
    *outval = flightstreamInstance->Temperature;
    aim_initValue(&flightstreamInstance->Temperature);
  } else if (index == outViscosity) {
    *outval = flightstreamInstance->Viscosity;
    aim_initValue(&flightstreamInstance->Viscosity);
  } else if (index == outCx) {
    *outval = flightstreamInstance->Cx;
    aim_initValue(&flightstreamInstance->Cx);
  } else if (index == outCy) {
    *outval = flightstreamInstance->Cy;
    aim_initValue(&flightstreamInstance->Cy);
  } else if (index == outCz) {
    *outval = flightstreamInstance->Cz;
    aim_initValue(&flightstreamInstance->Cz);
  } else if (index == outCL) {
    *outval = flightstreamInstance->CL;
    aim_initValue(&flightstreamInstance->CL);
  } else if (index == outCDi) {
    *outval = flightstreamInstance->CDi;
    aim_initValue(&flightstreamInstance->CDi);
  } else if (index == outCDo) {
    *outval = flightstreamInstance->CDo;
    aim_initValue(&flightstreamInstance->CDo);
  } else if (index == outCMx) {
    *outval = flightstreamInstance->CMx;
    aim_initValue(&flightstreamInstance->CMx);
  } else if (index == outCMy) {
    *outval = flightstreamInstance->CMy;
    aim_initValue(&flightstreamInstance->CMy);
  } else if (index == outCMz) {
    *outval = flightstreamInstance->CMz;
    aim_initValue(&flightstreamInstance->CMz);
  } else {
    AIM_ERROR(aimInfo, "Developer error: Unknown index %d", index);
    status = CAPS_NOTIMPLEMENT;
    goto cleanup;
  }

cleanup:
  return status;
}


/**********************************************************************/
/* aimCleanup - free up memory allocated in aimInitialize             */
/**********************************************************************/

void
aimCleanup(void *instStore)             /* (in)  AIM instance storage */
{

    /* -------------------------------------------------------------- */

#ifdef DEBUG
    printf("flightstreamAIM/aimCleanup()\n");
#endif

    aimStorage *flightstreamInstance = (aimStorage *)instStore;

    destroy_aimStorage(flightstreamInstance);

    AIM_FREE(flightstreamInstance);
}


// ********************** AIM Function Break *****************************
int
aimDiscr(char *tname, capsDiscr *discr)
{

  int i; // Indexing

  int status; // Function return status

  int numBody;

  // EGADS objects
  ego *bodies = NULL, *tess = NULL;

  const char   *intents;

  // Volume Mesh obtained from meshing AIM
  const aimMeshRef *meshRef;

  aimStorage *flightstreamInstance;

  flightstreamInstance = (aimStorage *) discr->instStore;

#ifdef DEBUG
  printf(" flightstreamAIM/aimDiscr: tname = %s!\n", tname);
#endif

  if (tname == NULL) return CAPS_NOTFOUND;

  // Currently this ONLY works if the capsTranfer lives on single body!
  status = aim_getBodies(discr->aInfo, &intents, &numBody, &bodies);
  AIM_STATUS(discr->aInfo, status);
  if (bodies == NULL) {
    AIM_ERROR(discr->aInfo, "NULL Bodies!\n");
    return CAPS_NULLOBJ;
  }

  // Get mesh
  meshRef = flightstreamInstance->meshRefIn;
  AIM_NOTNULL(meshRef, discr->aInfo, status);

  if (aim_newGeometry(discr->aInfo) == CAPS_SUCCESS) {
    // Get capsGroup name and index mapping to make sure all faces have a capsGroup value
    status = create_CAPSGroupAttrToIndexMap(numBody,
                                            bodies,
                                            1, // Only search down to the face level of the EGADS body
                                            &flightstreamInstance->groupMap);
    AIM_STATUS(discr->aInfo, status);
  }

  // Lets check the volume mesh

  // Do we have an individual surface mesh for each body
  if (meshRef->nmap != numBody) {
    AIM_ERROR(  discr->aInfo, "Number of surface mesh in the linked volume mesh (%d) does not match the number");
    AIM_ADDLINE(discr->aInfo,"of bodies (%d) - data transfer is NOT possible.", meshRef->nmap,numBody);
    status = CAPS_MISMATCH;
    goto cleanup;
  }

  // Lets store away our tessellation now
  AIM_ALLOC(tess, meshRef->nmap, ego, discr->aInfo, status);
  for (i = 0; i < meshRef->nmap; i++) {
    tess[i] = meshRef->maps[i].tess;
  }

  status = mesh_fillDiscr(tname, &flightstreamInstance->groupMap, meshRef->nmap, tess, discr);
  AIM_STATUS(discr->aInfo, status);

#ifdef DEBUG
  printf(" flightstreamAIM/aimDiscr: Finished!!\n");
#endif

  status = CAPS_SUCCESS;

cleanup:
  AIM_FREE(tess);
  return status;
}

// ********************** AIM Function Break *****************************
void aimFreeDiscrPtr(void *ptr)
{
  // Extra information to store into the discr void pointer - just a int array
  EG_free(ptr);
}

// ********************** AIM Function Break *****************************
int
aimLocateElement(capsDiscr *discr, double *params, double *param,
                 int *bIndex, int *eIndex, double *bary)
{
    return aim_locateElement(discr, params, param, bIndex, eIndex, bary);
}


// ********************** AIM Function Break *****************************
int
aimTransfer(capsDiscr *discr, const char *name, int npts, int rank, double *data,
            char **units)
{
  /*! \page dataTransferFLIGHTSTREAM AIM Data Transfer
   *
   * The FlightStream AIM has the ability to transfer surface data (e.g. pressure distributions) to and from the AIM
   * using the conservative and interpolative data transfer schemes in CAPS.
   *
   * \section dataFromFlightsteram Data transfer from FlightStream (FieldOut)
   *
   * <ul>
   * <li> <B>"RelPressure" </B> </li> <br>
   *  Loads the relative to freestream pressure distribution from FlightStream vtk file.
   *  This distribution may be scaled based on
   *  Pressure = Pressure_Scale_Factor*RelPressure, where "Pressure_Scale_Factor"
   *  is an AIM input (\ref aimInputsFLIGHTSTREAM)
   * </ul>
   *
   * <ul>
   * <li> <B>"AbsPressure" </B> </li> <br>
   *  Loads the absolute pressure distribution from FlightStream vtk file.
   *  This distribution may be scaled based on
   *  Pressure = Pressure_Scale_Factor*AbsPressure, where "Pressure_Scale_Factor"
   *  is an AIM input (\ref aimInputsFLIGHTSTREAM)
   * </ul>
   *
   */
  int    i, j, global, status, bIndex;
  double **rvec=NULL, scale = 1.0;
  capsValue *Pressure_Scale_Factor_Value=NULL;
  int state, nglobal;
  ego        body;
  FILE *fp = NULL;
#ifdef ANALYSIS_PLOAD_BDF
  int ID, inode, jnode;
  char str[10];
  size_t linecap = 0;
  char *line = NULL; // Temporary line holder
  const char *PLOAD4 = "$$  PLOAD4 Data";
#elif defined(ANALYSIS_VTK)
  size_t linecap = 0;
  char *line = NULL; // Temporary line holder
  const char *CP = "SCALARS Cp FLOAT";
  const char *PSTATIC = "SCALARS Static_pressure_ratio FLOAT";
  double vref = 0, qref=0;
  capsValue *ReferenceVelocity=NULL;
#elif defined(ANALYSIS_CSV)
  double x, y, z;
  double vref = 0, qref=0;
  capsValue *ReferenceVelocity=NULL;
#else
#error "must define ANALYSIS_PLOAD_BDF, ANALYSIS_VTK, or, ANALYSIS_CSV"
#endif

  const aimStorage *flightstreamInstance = (const aimStorage *)discr->instStore;
  const aimMeshRef *meshRef = flightstreamInstance->meshRefIn;

#ifdef DEBUG
  printf(" flightstreamAIM/aimTransfer name = %s  npts = %d/%d!\n",
         name, npts, len_wrt);
#endif

  if (strcmp(name, "RelPressure") == 0) {

    if (rank != 1) {
      AIM_ERROR(discr->aInfo, "Rank (%d) must be 1 for 'RelPressure'!", rank);
      status = CAPS_NOTIMPLEMENT;
      goto cleanup;
    }

#ifdef ANALYSIS_PLOAD_BDF
    fp = aim_fopen(discr->aInfo, loadbdf, "r");
    if (fp == NULL) {
      AIM_ERROR(discr->aInfo, "Cannot open \"%s\"", loadbdf);
      status = CAPS_IOERR;
      goto cleanup;
    }

    /* try and read the bdf file */
    while (getline(&line, &linecap, fp) > 0) {
      AIM_NOTNULL(line, discr->aInfo, status);
      if (strncmp(line, PLOAD4, strlen(PLOAD4)) == 0) {
        break;
      }
    }

    AIM_NOTNULL(line, discr->aInfo, status);
    if (strncmp(line, PLOAD4, strlen(PLOAD4)) != 0) {
      AIM_ERROR(discr->aInfo, "Could not find 'PLOAD4' data in \"%s\"", loadbdf);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // Skip '$$'
    status = getline(&line, &linecap, fp);
    if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));

    AIM_ALLOC(rvec, meshRef->nmap, double*, discr->aInfo, status);
    for (i = 0; i < meshRef->nmap; i++) {
      rvec[i] = NULL;
    }

    jnode = 1;
    for (i = 0; i < meshRef->nmap; i++) {
      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(discr->aInfo, status);
      AIM_ALLOC(rvec[i], nglobal, double, discr->aInfo, status);

      for (j = 0; j < nglobal; j++, jnode++) {
        status = getline(&line, &linecap, fp);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
        status = sscanf(line, "%s %d %d %lf", str, &ID, &inode, &rvec[i][j]);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
        if (inode != jnode) {
          AIM_ERROR(discr->aInfo, "While reading %s, %d != %d!", loadbdf, inode, jnode);
          status = CAPS_NOTIMPLEMENT;
          goto cleanup;
        }
        // subtract off the reference pressure
        // both BDF pressure and pref are in Pa
        rvec[i][j] -= flightstreamInstance->pff;
      }
    }

    // Convert from Pa to working pressure
    scale = 1;
    status = aim_convert(discr->aInfo, 1, "Pa", &scale, flightstreamInstance->units.pressure, &scale);
    AIM_STATUS(discr->aInfo, status);

    // set the units
    AIM_STRDUP(*units, "Pa", discr->aInfo, status);

#elif defined(ANALYSIS_VTK)
    fp = aim_fopen(discr->aInfo, cpvtk, "r");
    if (fp == NULL) {
      AIM_ERROR(discr->aInfo, "Cannot open \"%s\"", cpvtk);
      status = CAPS_IOERR;
      goto cleanup;
    }

    /* try and read the vtk file */
    while (getline(&line, &linecap, fp) > 0) {
      AIM_NOTNULL(line, discr->aInfo, status);
      if (strncmp(line, CP, strlen(CP)) == 0) {
        break;
      }
    }

    AIM_NOTNULL(line, discr->aInfo, status);
    if (strncmp(line, CP, strlen(CP)) != 0) {
      AIM_ERROR(discr->aInfo, "Could not find 'PLOAD4' data in \"%s\"", cpvtk);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // Skip 'LOOKUP_TABLE default'
    status = getline(&line, &linecap, fp);
    if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));

    AIM_ALLOC(rvec, meshRef->nmap, double*, discr->aInfo, status);
    for (i = 0; i < meshRef->nmap; i++) {
      rvec[i] = NULL;
    }

    for (i = 0; i < meshRef->nmap; i++) {
      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(discr->aInfo, status);
      AIM_ALLOC(rvec[i], nglobal, double, discr->aInfo, status);

      for (j = 0; j < nglobal; j++) {
        status = getline(&line, &linecap, fp);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
        status = sscanf(line, "%lf", &rvec[i][j]);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
      }
    }

    status = aim_getValue(discr->aInfo, inReferenceVelocity, ANALYSISIN, &ReferenceVelocity);
    AIM_STATUS(discr->aInfo, status);
    AIM_NOTNULL(ReferenceVelocity, discr->aInfo, status);

    // Compute dynamic pressure
    status = aim_convert(discr->aInfo, 1,
                         ReferenceVelocity->units,  &ReferenceVelocity->vals.real,
                         "m/s", &vref);
    AIM_STATUS(discr->aInfo, status);

    qref = 0.5 * flightstreamInstance->rhoff * vref*vref;

    status = aim_convert(discr->aInfo, 1, "Pa", &qref, flightstreamInstance->units.pressure, &qref);
    AIM_STATUS(discr->aInfo, status);
    scale *= qref;

    // set the units
    AIM_STRDUP(*units, flightstreamInstance->units.pressure, discr->aInfo, status);

#elif defined(ANALYSIS_CSV)

    fp = aim_fopen(discr->aInfo, cpcsv, "r");
    if (fp == NULL) {
      AIM_ERROR(discr->aInfo, "Cannot open \"%s\"", cpcsv);
      status = CAPS_IOERR;
      goto cleanup;
    }

    AIM_ALLOC(rvec, meshRef->nmap, double*, discr->aInfo, status);
    for (i = 0; i < meshRef->nmap; i++) {
      rvec[i] = NULL;
    }

    for (i = 0; i < meshRef->nmap; i++) {
      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(discr->aInfo, status);
      AIM_ALLOC(rvec[i], nglobal, double, discr->aInfo, status);

      for (j = 0; j < nglobal; j++) {
        status = fscanf(fp, "%lf, %lf, %lf, %lf\n", &x, &y, &z, &rvec[i][j]);
        if (status != 4) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
      }
    }

    status = aim_getValue(discr->aInfo, inReferenceVelocity, ANALYSISIN, &ReferenceVelocity);
    AIM_STATUS(discr->aInfo, status);
    AIM_NOTNULL(ReferenceVelocity, discr->aInfo, status);

    // Compute dynamic pressure
    status = aim_convert(discr->aInfo, 1,
                         ReferenceVelocity->units,  &ReferenceVelocity->vals.real,
                         "m/s", &vref);
    AIM_STATUS(discr->aInfo, status);

    qref = 0.5 * flightstreamInstance->rhoff * vref*vref;

    status = aim_convert(discr->aInfo, 1, "Pa", &qref, flightstreamInstance->units.pressure, &qref);
    AIM_STATUS(discr->aInfo, status);
    scale *= qref;

    // set the units
    AIM_STRDUP(*units, "Pa", discr->aInfo, status);

#else
#error "must define ANALYSIS_PLOAD_BDF, ANALYSIS_VTK, or ANALYSIS_CSV"
#endif


    // Custom additional scale factor
    status = aim_getValue(discr->aInfo, inPressure_Scale_Factor, ANALYSISIN, &Pressure_Scale_Factor_Value);
    AIM_STATUS(discr->aInfo, status);
    AIM_NOTNULL(Pressure_Scale_Factor_Value, discr->aInfo, status);
    scale *= Pressure_Scale_Factor_Value->vals.real;

  } else if (strcmp(name, "AbsPressure") == 0) {

    if (rank != 1) {
      AIM_ERROR(discr->aInfo, "Rank (%d) must be 1 for 'AbsPressure'!", rank);
      status = CAPS_NOTIMPLEMENT;
      goto cleanup;
    }

#ifdef ANALYSIS_PLOAD_BDF
    fp = aim_fopen(discr->aInfo, loadbdf, "r");
    if (fp == NULL) {
      AIM_ERROR(discr->aInfo, "Cannot open \"%s\"", loadbdf);
      status = CAPS_IOERR;
      goto cleanup;
    }

    /* try and read the bdf file */
    while (getline(&line, &linecap, fp) > 0) {
      AIM_NOTNULL(line, discr->aInfo, status);
      if (strncmp(line, PLOAD4, strlen(PLOAD4)) == 0) {
        break;
      }
    }

    AIM_NOTNULL(line, discr->aInfo, status);
    if (strncmp(line, PLOAD4, strlen(PLOAD4)) != 0) {
      AIM_ERROR(discr->aInfo, "Could not find 'PLOAD4' data in \"%s\"", loadbdf);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // Skip '$$'
    status = getline(&line, &linecap, fp);
    if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));

    AIM_ALLOC(rvec, meshRef->nmap, double*, discr->aInfo, status);
    for (i = 0; i < meshRef->nmap; i++) {
      rvec[i] = NULL;
    }

    jnode = 1;
    for (i = 0; i < meshRef->nmap; i++) {
      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(discr->aInfo, status);
      AIM_ALLOC(rvec[i], nglobal, double, discr->aInfo, status);

      for (j = 0; j < nglobal; j++, jnode++) {
        status = getline(&line, &linecap, fp);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
        status = sscanf(line, "%s %d %d %lf", str, &ID, &inode, &rvec[i][j]);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
        if (inode != jnode) {
          AIM_ERROR(discr->aInfo, "While reading %s, %d != %d!", loadbdf, inode, jnode);
          status = CAPS_NOTIMPLEMENT;
          goto cleanup;
        }
        // subtract off the reference pressure
        // both BDF pressure and pref are in Pa
        rvec[i][j] -= flightstreamInstance->pff;
      }
    }

    // Convert from Pa to working pressure
    scale = 1;
    status = aim_convert(discr->aInfo, 1, "Pa", &scale, flightstreamInstance->units.pressure, &scale);
    AIM_STATUS(discr->aInfo, status);

    // set the units
    AIM_STRDUP(*units, "Pa", discr->aInfo, status);

#elif defined(ANALYSIS_VTK)
    fp = aim_fopen(discr->aInfo, cpvtk, "r");
    if (fp == NULL) {
      AIM_ERROR(discr->aInfo, "Cannot open \"%s\"", cpvtk);
      status = CAPS_IOERR;
      goto cleanup;
    }

    /* try and read the vtk file */
    while (getline(&line, &linecap, fp) > 0) {
      AIM_NOTNULL(line, discr->aInfo, status);
      if (strncmp(line, CP, strlen(CP)) == 0) {
        break;
      }
    }

    AIM_NOTNULL(line, discr->aInfo, status);
    if (strncmp(line, CP, strlen(CP)) != 0) {
      AIM_ERROR(discr->aInfo, "Could not find 'PLOAD4' data in \"%s\"", cpvtk);
      status = CAPS_IOERR;
      goto cleanup;
    }

    // Skip 'LOOKUP_TABLE default'
    status = getline(&line, &linecap, fp);
    if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));

    AIM_ALLOC(rvec, meshRef->nmap, double*, discr->aInfo, status);
    for (i = 0; i < meshRef->nmap; i++) {
      rvec[i] = NULL;
    }

    for (i = 0; i < meshRef->nmap; i++) {
      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(discr->aInfo, status);
      AIM_ALLOC(rvec[i], nglobal, double, discr->aInfo, status);

      for (j = 0; j < nglobal; j++) {
        status = getline(&line, &linecap, fp);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
        status = sscanf(line, "%lf", &rvec[i][j]);
        if (status <= 0) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
      }
    }

    status = aim_getValue(discr->aInfo, inReferenceVelocity, ANALYSISIN, &ReferenceVelocity);
    AIM_STATUS(discr->aInfo, status);
    AIM_NOTNULL(ReferenceVelocity, discr->aInfo, status);

    // Compute dynamic pressure
    status = aim_convert(discr->aInfo, 1,
                         ReferenceVelocity->units,  &ReferenceVelocity->vals.real,
                         "m/s", &vref);
    AIM_STATUS(discr->aInfo, status);

    qref = 0.5 * flightstreamInstance->rhoff * vref*vref;

    status = aim_convert(discr->aInfo, 1, "Pa", &qref, flightstreamInstance->units.pressure, &qref);
    AIM_STATUS(discr->aInfo, status);
    scale *= qref;

    // set the units
    AIM_STRDUP(*units, flightstreamInstance->units.pressure, discr->aInfo, status);

#elif defined(ANALYSIS_CSV)

    fp = aim_fopen(discr->aInfo, pcsv, "r");
    if (fp == NULL) {
      AIM_ERROR(discr->aInfo, "Cannot open \"%s\"", pcsv);
      status = CAPS_IOERR;
      goto cleanup;
    }

    AIM_ALLOC(rvec, meshRef->nmap, double*, discr->aInfo, status);
    for (i = 0; i < meshRef->nmap; i++) {
      rvec[i] = NULL;
    }

    for (i = 0; i < meshRef->nmap; i++) {
      status = EG_statusTessBody(meshRef->maps[i].tess, &body, &state, &nglobal);
      AIM_STATUS(discr->aInfo, status);
      AIM_ALLOC(rvec[i], nglobal, double, discr->aInfo, status);

      for (j = 0; j < nglobal; j++) {
        status = fscanf(fp, "%lf, %lf, %lf, %lf\n", &x, &y, &z, &rvec[i][j]);
        if (status != 4) AIM_STATUS(discr->aInfo, (status = CAPS_IOERR));
      }
    }

    // set the units
    AIM_STRDUP(*units, "Pa", discr->aInfo, status);

#else
#error "must define ANALYSIS_PLOAD_BDF, ANALYSIS_VTK, or ANALYSIS_CSV"
#endif


    // Custom additional scale factor
    status = aim_getValue(discr->aInfo, inPressure_Scale_Factor, ANALYSISIN, &Pressure_Scale_Factor_Value);
    AIM_STATUS(discr->aInfo, status);
    AIM_NOTNULL(Pressure_Scale_Factor_Value, discr->aInfo, status);
    scale *= Pressure_Scale_Factor_Value->vals.real;
  }


  /* move the appropriate parts of the tessellation to data */
  AIM_NOTNULL(rvec, discr->aInfo, status);
  for (i = 0; i < npts; i++) {
    /* points might span multiple bodies */
    bIndex = discr->tessGlobal[2*i  ];
    global = discr->tessGlobal[2*i+1];
    for (j = 0; j < rank; j++)
      data[rank*i+j] = rvec[bIndex-1][rank*(global-1)+j] * scale;
  }

  status = CAPS_SUCCESS;
cleanup:
  if (rvec != NULL) {
    for (i = 0; i < meshRef->nmap; ++i) {
      AIM_FREE(rvec[i]);
    }
  }
  AIM_FREE(rvec);
#if defined(ANALYSIS_PLOAD_BDF) || defined(ANALYSIS_VTK)
  if (line != NULL) free(line);
#endif

  if (fp != NULL) fclose(fp);

  return status;
}


// ********************** AIM Function Break *****************************
int
aimInterpolation(capsDiscr *discr, /*@unused@*/ const char *name, int bIndex,
                 int eIndex, double *bary, int rank, double *data,
                 double *result)
{
#ifdef DEBUG
    printf(" flightstreamAIM/aimInterpolation  %s!\n", name);
#endif

    return  aim_interpolation(discr, name, bIndex, eIndex,
                              bary, rank, data, result);
}


// ********************** AIM Function Break *****************************
int
aimInterpolateBar(capsDiscr *discr, /*@unused@*/ const char *name, int bIndex,
                  int eIndex, double *bary, int rank, double *r_bar,
                  double *d_bar)
{
#ifdef DEBUG
    printf(" flightstreamAIM/aimInterpolateBar  %s!\n", name);
#endif

    return  aim_interpolateBar(discr, name, bIndex, eIndex,
                               bary, rank, r_bar, d_bar);
}


// ********************** AIM Function Break *****************************
int
aimIntegration(capsDiscr *discr, /*@unused@*/ const char *name, int bIndex,
               int eIndex, int rank, double *data, double *result)
{
#ifdef DEBUG
    printf(" flightstreamAIM/aimIntegration  %s!\n", name);
#endif

    return aim_integration(discr, name, bIndex, eIndex, rank,
                           data, result);
}


// ********************** AIM Function Break *****************************
int
aimIntegrateBar(capsDiscr *discr, /*@unused@*/ const char *name, int bIndex,
                int eIndex, int rank, double *r_bar, double *d_bar)
{
#ifdef DEBUG
    printf(" flightstreamAIM/aimIntegrateBar  %s!\n", name);
#endif

    return aim_integrateBar(discr, name, bIndex, eIndex, rank,
                            r_bar, d_bar);
}
