// This software has been cleared for public release on 05 Nov 2020, case number 88ABW-2020-3462.

#include <stdio.h>
#include <string.h>

#include "aimUtil.h"
#include "jsonUtils.h"
#include "miscUtils.h"


// JSON

// Return whether the string value represents a JSON string
// Currently only checks whether the first character is '{'
int json_isDict(const char *string)
{
    return strncmp(string, "{", 1) == 0;
}

// Convert a string in tuple form to an array of strings - tuple is assumed to be bounded by '(' and ')' and comma separated
//  for example ("3.0", 5, "foo", ("f", 1, 4), [1,2,3]) - note the strings of the outer tuple should NOT contain commas, Tuple elements
//  consisting of internal tuples and arrays are okay (only 1 level deep).  If the string coming in is not a tuple the string
// is simply copied. Also quotations are removed from values elements of the (outer) tuple.
int json_parseTuple(/*@null@*/ const char *stringToParse, int *arraySize,
                    char **stringArray[])
{
    int i, j;         // Array indexing
    size_t stringToParseLen;

    int matchLength;  // String length of matching pattern
    int startIndex; // Keep track of where we are in the string array

    int arrayIndex;
    int haveArray = (int) false;
    char *quoteString = NULL; // Temporary string to hold the found string
    char *noQuoteString = NULL;  // Temporary string to hold the found string with quotation marks removed

    int foundInternal = (int) false;

    // Debug function
    //printf("Search string - %s length %d\n", stringToParse, strlen(stringToParse));

    if (stringToParse == NULL) return CAPS_BADVALUE;

    stringToParseLen = strlen(stringToParse);

    // Check to make sure first and last of the incoming string denote an array
    if (stringToParse[0] == '[' ) {

        // Debug function
        //printf("[ found\n");

        if(stringToParse[stringToParseLen-1] == ']') { // Lets count how many commas we have

            // Debug function
            //printf("] found\n");

            haveArray = (int) true;
            *arraySize = 1;
            foundInternal = 0;
            for (i = 1; i < stringToParseLen-1; i++) {

                if (stringToParse[i] == '{') foundInternal++;
                if (stringToParse[i] == '}' ) foundInternal--;

                if (stringToParse[i] == '(') foundInternal++;
                if (stringToParse[i] == ')' ) foundInternal--;

                if (stringToParse[i] == '[') foundInternal++;
                if (stringToParse[i] == ']') foundInternal--;

                if (foundInternal > 0) continue;

                if (stringToParse[i] == ',') *arraySize += 1;
            }

        } else {

            *arraySize = 1;
        }

    } else {

        *arraySize = 1;
    }

    // Debug function
    //printf("Number of strings (arraySize) = %d\n", *arraySize);

    // Allocate stringArray
    *stringArray = (char **) EG_alloc(*arraySize*sizeof(char **));
    if (*stringArray == NULL) {
        *arraySize = 0;
        return EGADS_MALLOC;
    }

    if (*arraySize == 1 && haveArray == (int) false) {

        // Debug function
        //printf("We don't have an array\n");

        noQuoteString = string_removeQuotation(stringToParse);

        (*stringArray)[0] = (char *) EG_alloc((strlen(noQuoteString)+1) * sizeof(char));

        // Check for malloc error
        if ((*stringArray)[0] == NULL) {

            // Free string array
            AIM_FREE(*stringArray);

            *arraySize = 0;

            // Free no quote array
            AIM_FREE(noQuoteString);

            return EGADS_MALLOC;
        }

        strncpy((*stringArray)[0], noQuoteString, strlen(noQuoteString));
        (*stringArray)[0][strlen(noQuoteString)] = '\0';

        AIM_FREE (noQuoteString);

    } else {

        startIndex = 1;
        arrayIndex = 0;
        foundInternal = 0;
        // Parse string based on defined pattern
        for (i = 1; i < stringToParseLen; i++) {

            if (stringToParse[i] == '{') foundInternal++;
            if (stringToParse[i] == '}') foundInternal--;

            if (stringToParse[i] == '(') foundInternal++;
            if (stringToParse[i] == ')') foundInternal--;

            if (stringToParse[i] == '[') foundInternal++;
            if (stringToParse[i] == ']') foundInternal--;

            if (foundInternal > 0) continue;

            if (stringToParse[i] == ',' ||
                i == stringToParseLen-1) {

                matchLength = i-startIndex;

                // Debug function
                //printf("MatchLength = %d\n", matchLength);

                // Make sure we aren't exceeding our arrays bounds
                if (arrayIndex >= *arraySize) {
                    return CAPS_MISMATCH;
                }

                // If not just a blank space
                if (matchLength > 0) {

                    // Allocate the quoted string array
                    quoteString = (char *) EG_alloc( (matchLength+1) * sizeof(char));
                    if (quoteString == NULL) return EGADS_MALLOC;

                    strncpy(quoteString, stringToParse + startIndex, matchLength);
                    quoteString[matchLength] = '\0';

                    // Remove quotations
                    noQuoteString = string_removeQuotation(quoteString);

                    // Free quote string array
                    AIM_FREE(quoteString);

                    // Allocate string array element based on no quote string
                    (*stringArray)[arrayIndex] = (char *) EG_alloc( (strlen(noQuoteString)+1) * sizeof(char));

                    // Check for malloc error
                    if ((*stringArray)[arrayIndex] == NULL) {

                        // Free string array
                        AIM_FREE(quoteString);
                        for (j = 0; j <  arrayIndex; j++) AIM_FREE((*stringArray)[j]);
                        AIM_FREE(*stringArray);

                        *arraySize = 0;

                        // Free no quote string
                        AIM_FREE(noQuoteString);

                        return EGADS_MALLOC;
                    }

                    // Copy no quote string into array
                    strncpy((*stringArray)[arrayIndex], noQuoteString, strlen(noQuoteString));
                    (*stringArray)[arrayIndex][strlen(noQuoteString)] = '\0';

                    // Free no quote array
                    AIM_FREE(noQuoteString);

                    // Increment start indexes
                    arrayIndex += 1;
                    startIndex = i+1; // +1 to skip the commas
                    while (startIndex < stringToParseLen &&
                           stringToParse[startIndex] == ' ') startIndex++;
                }
            }
        }
    }

    // Debug function - print out number array
    //for (i = 0; i < *arraySize; i++) printf("Value = %s\n", (*stringArray)[i]);

    // Free quote array
    AIM_FREE(quoteString);

    // Free no quote array
    AIM_FREE(noQuoteString);


    return CAPS_SUCCESS;
}

// Search for a json entry in stringToSearch
static
int json_entry(const char **stringStart, char **value) {

    int  i; // Indexing for loop

    int  keyIndexStart; // Starting index of keyValue

    int  keyLength = 0; // Length of keyValue
    int  nesting = 0;

    const char *stringToSearch = *stringStart;

    // Get the starting index of the keyValue (using pointer arithmetic)
    keyIndexStart = 0;

    while (stringToSearch[keyIndexStart] == ' ') keyIndexStart += 1; // Skip whitespace after :, some JSON writers have spaces others don't
    while (stringToSearch[keyIndexStart] == '{') keyIndexStart += 1;

    // See how long our keyValue is - at most length of incoming string
    for (i = 0; i < (int) strlen(stringToSearch); i++) {

        if (stringToSearch[keyIndexStart+keyLength] == '[' ||
            stringToSearch[keyIndexStart+keyLength] == '{') nesting++;

        // Array possibly nested
        if (stringToSearch[keyIndexStart] == '[' ||
            stringToSearch[keyIndexStart] == '{') {

            if (stringToSearch[keyIndexStart+keyLength] == ']' ||
                stringToSearch[keyIndexStart+keyLength] == '}') {
                nesting--;
              if (nesting == 0) {
                  keyLength += 1; // include closing bracket
                  break;
              }
            }

            keyLength += 1;

        } else {

            keyLength += 1;
            if (stringToSearch[keyIndexStart+keyLength] == ':' ||
                stringToSearch[keyIndexStart+keyLength] == ',' ||
                stringToSearch[keyIndexStart+keyLength] == '}') break;
        }
    }

    // Free keyValue (if not already null) and allocate
    AIM_FREE(*value);

    if (keyLength > 0) {
        *value = (char *) EG_alloc((keyLength+1) * sizeof(char));
        if (*value == NULL) return EGADS_MALLOC;

    } else {
        //printf("No match found for %s\n",keyWord);
        *value = NULL; //"\0";
        return CAPS_NOTFOUND;
    }

    // Copy matched expression to keyValue
    strncpy((*value), stringToSearch + keyIndexStart, keyLength);
    (*value)[keyLength] = '\0';
    *stringStart += keyIndexStart + keyLength;

    return CAPS_SUCCESS;
}

// Search for a Key-Value pair in stringStart, and advance stringStart
static
int json_getKeyValue(const char **stringStart, char **keyWord, char **keyValue) {

  int status = CAPS_SUCCESS;
  char *keyWordQuote = NULL;
  size_t len, i=0;

  status = json_entry(stringStart, &keyWordQuote);
  if (status != CAPS_SUCCESS) return status;

  *keyWord = string_removeQuotation(keyWordQuote);
  AIM_FREE(keyWordQuote);

  len = strlen(*stringStart);
  i = 0;
  while (i < len && (*stringStart)[i] != ':') i++;
  i++;
  if (i > len) {
    return CAPS_NOTFOUND;
  }
  (*stringStart) += i;

  status = json_entry(stringStart, keyValue);
  if (status != CAPS_SUCCESS) return status;

  len = strlen(*stringStart);
  i = 0;
  while (i < len && (*stringStart)[i] != ',' && (*stringStart)[i] != '}') i++;
  i++;
  if (i > len) {
    return CAPS_NOTFOUND;
  }
  (*stringStart) += i;

  return CAPS_SUCCESS;
}

// Search for a Key-Value pair in stringToSearch
int json_keyValue(const char *stringToSearch, char **keyWord, char **keyValue) {
  return json_getKeyValue(&stringToSearch, keyWord, keyValue);
}

// Search for all json Key-Value pairs in stringToSearch
int json_dictTuple(void *aimInfo, const char *jsonDict, int *numTuple, capsTuple **tuple) {

  int status = CAPS_SUCCESS;
  size_t i=0;
  char *keyWord = NULL;
  char *keyValue = NULL;

  AIM_NOTNULL(jsonDict, aimInfo, status);

  // first cleanup
  for (i = 0; i < *numTuple; i++) {
    AIM_FREE((*tuple)[i].name);
    AIM_FREE((*tuple)[i].value);
  }
  AIM_FREE((*tuple));
  *numTuple = 0;

  do {
    status = json_getKeyValue(&jsonDict, &keyWord, &keyValue);
    if (status == CAPS_NOTFOUND) break;
    AIM_STATUS(aimInfo, status);

    AIM_REALL((*tuple), (*numTuple)+1, capsTuple, aimInfo, status);
    (*tuple)[(*numTuple)].name = keyWord;
    (*tuple)[(*numTuple)].value = keyValue;
    (*numTuple)++;
    keyWord = NULL;
    keyValue = NULL;

  } while (1);

  status = CAPS_SUCCESS;
cleanup:
  AIM_FREE(keyWord);
  AIM_FREE(keyValue);

  return status;
}


// Simple json dictionary parser - currently doesn't support nested arrays for keyValue
int search_jsonDictionary(const char *stringToSearch, const char *keyWord, char **keyValue) {

    int  i; // Indexing for loop

    int  keyIndexStart; // Starting index of keyValue

    int  keyLength = 0; // Length of keyValue
    int  nesting = 0;

    char pattern[255]; // Substring - keyWord

    char *patternMatch = NULL; // Mathced substring - keyWord

    // Build pattern
    snprintf(pattern, 255, "\"%s\":", keyWord);

    // Debug function
    //printf("Pattern - [%s]\n", pattern);
    //printf("Search string - [%s]\n", stringToSearch);

    // Search string for pattern - return pointer to match
    patternMatch = strstr(stringToSearch, pattern);

    // If pattern is found - get keyValue
    if (patternMatch != NULL) {

        // Get the starting index of the keyValue (using pointer arithmetic)
        keyIndexStart = (patternMatch - stringToSearch) + strlen(pattern);

        if (stringToSearch[keyIndexStart] == ' ') keyIndexStart += 1; // Skip whitespace after :, some JSON writers have spaces others don't

        // See how long our keyValue is - at most length of incoming string
        for (i = 0; i < (int) strlen(stringToSearch); i++) {

            if (stringToSearch[keyIndexStart+keyLength] == '[' ||
                stringToSearch[keyIndexStart+keyLength] == '{') nesting++;

            // Array possibly nested
            if (stringToSearch[keyIndexStart] == '[' ||
                stringToSearch[keyIndexStart] == '{') {

                if (stringToSearch[keyIndexStart+keyLength] == ']' ||
                    stringToSearch[keyIndexStart+keyLength] == '}') {
                    nesting--;
                  if (nesting == 0) {
                      keyLength += 1; // include closing bracket
                      break;
                  }
                }

                keyLength += 1;

            } else {

                keyLength += 1;
                if (stringToSearch[keyIndexStart+keyLength] == ',' ||
                    stringToSearch[keyIndexStart+keyLength] == '}') break;
            }
        }

        // Debug function
        //printf("Key start %d\n"    , keyIndexStart);
        //printf("Size of keyLength - %d\n", keyLength);

        // Free keyValue (if not already null) and allocate
        AIM_FREE(*keyValue);

        if (keyLength > 0) {
            *keyValue = (char *) EG_alloc((keyLength+1) * sizeof(char));
            if (*keyValue == NULL) return EGADS_MALLOC;

        } else {
            //printf("No match found for %s\n",keyWord);
            *keyValue = NULL; //"\0";
            return CAPS_NOTFOUND;
        }

        // Copy matched expression to keyValue
        strncpy((*keyValue), stringToSearch + keyIndexStart, keyLength);
        (*keyValue)[keyLength] = '\0';

    } else {

        //printf("No match found for %s\n",keyWord);
        *keyValue = NULL; //"\0";
        return CAPS_NOTFOUND;
    }

    // Debug function
    // printf("keyValue = [%s]\n", *keyValue);

    return CAPS_SUCCESS;
}

// Get raw string `value` with given `key` in `jsonDict` string
// Simple wrapper over search_jsonDictionary function
int json_get(const char *jsonDict, const char *key, char **value)
{

    if (!json_isDict(jsonDict)) {
        return CAPS_BADVALUE;
    }

    return search_jsonDictionary(jsonDict, key, value);
}

// Get string `value` with given `key` in `jsonDict` string
// NOTE: overwrites value pointer with newly allocated string
int json_getString(const char *jsonDict, const char *key, char **value)
{
    int status;
    char *valueStr = NULL;

    if (*value != NULL) {
        AIM_FREE(*value);
    }

    status = json_get(jsonDict, key, &valueStr);
    if (status == CAPS_SUCCESS) {
        *value = string_removeQuotation(valueStr);
    }

    if (valueStr != NULL) AIM_FREE(valueStr);

    return status;
}

// Get array of strings `value` with given `key` in `jsonDict` string
int json_getStringArray(const char *jsonDict, const char *key, int size, char *value[])
{
    int status;
    char *valueStr = NULL;

    status = json_get(jsonDict, key, &valueStr);
    if (status == CAPS_SUCCESS) {
        status = string_toStringArray(valueStr, size, value);
    }

    if (valueStr != NULL) EG_free(valueStr);

    return status;
}

// Get dynamic array of strings `value` with given `key` in `jsonDict` string
int json_getStringDynamicArray(const char *jsonDict, const char *key,
                               int *size, char **value[])
{
    int status;
    char *valueStr = NULL;

    status = json_get(jsonDict, key, &valueStr);
    if (status == CAPS_SUCCESS) {
        status = string_toStringDynamicArray(valueStr, size, value);
    }

    if (valueStr != NULL) EG_free(valueStr);

    return status;
}

// Get integer `value` with given `key` in `jsonDict` string
int json_getInteger(const char *jsonDict, const char *key, int *value)
{
    int status;
    char *valueStr = NULL;

    status = json_get(jsonDict, key, &valueStr);
    if (status == CAPS_SUCCESS) {
        status = string_toInteger(valueStr, value);
    }

    if (valueStr != NULL) EG_free(valueStr);

    return status;
}

// Get array of integers `value` with given `key` in `jsonDict` string
int json_getIntegerArray(const char *jsonDict, const char *key, int size, int value[])
{
    int status;
    char *valueStr = NULL;

    status = json_get(jsonDict, key, &valueStr);
    if (status == CAPS_SUCCESS) {
        status = string_toIntegerArray(valueStr, size, value);
    }

    if (valueStr != NULL) EG_free(valueStr);

    return status;
}

// Get dynamic array of integers `value` with given `key` in `jsonDict` string
int json_getIntegerDynamicArray(const char *jsonDict, const char *key,
                                int *size, int *value[])
{
    int status;
    char *valueStr = NULL;

    status = json_get(jsonDict, key, &valueStr);
    if (status == CAPS_SUCCESS) {
        status = string_toIntegerDynamicArray(valueStr, size, value);
    }

    if (valueStr != NULL) EG_free(valueStr);

    return status;
}

// Get double `value` with given `key` in `jsonDict` string
int json_getDouble(const char *jsonDict, const char *key, double *value)
{
    int status;
    char *valueStr = NULL;

    status = json_get(jsonDict, key, &valueStr);
    if (status == CAPS_SUCCESS) {
        status = string_toDouble(valueStr, value);
    }

    if (valueStr != NULL) EG_free(valueStr);

    return status;
}

// Get array of doubles `value` with given `key` in `jsonDict` string
int json_getDoubleArray(const char *jsonDict, const char *key, int size, double value[])
{
    int status;
    char *valueStr = NULL;

    status = json_get(jsonDict, key, &valueStr);
    if (status == CAPS_SUCCESS) {
        status = string_toDoubleArray(valueStr, size, value);
    }

    if (valueStr != NULL) EG_free(valueStr);

    return status;
}

// Get dynamic array of doubles `value` with given `key` in `jsonDict` string
int json_getDoubleDynamicArray(const char *jsonDict, const char *key,
                               int *size, double *value[])
{
    int status;
    char *valueStr = NULL;

    status = json_get(jsonDict, key, &valueStr);
    if (status == CAPS_SUCCESS) {
        status = string_toDoubleDynamicArray(valueStr, size, value);
    }

    if (valueStr != NULL) EG_free(valueStr);

    return status;
}
