/*
 ************************************************************************
 *                                                                      *
 * udfOffset -- make an offset of a planar WireBody or SheetBody        *
 *                                                                      *
 *            Written by John Dannenhoffer @ Syracuse University        *
 *            Patterned after code written by Bob Haimes  @ MIT         *
 *                                                                      *
 ************************************************************************
 */

/*
 * Copyright (C) 2011/2025  John F. Dannenhoffer, III (Syracuse University)
 *
 * This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *     MA  02110-1301  USA
 */

//#define DEBUG 1

#define NUMUDPARGS 7
#define NUMUDPINPUTBODYS 1
#include "udpUtilities.h"

/* shorthands for accessing argument values and velocities */
#define NODELIST(    IUDP,I)  ((int    *) (udps[IUDP].arg[0].val))[I]
#define NODELIST_SIZ(IUDP)    ( int     )  udps[IUDP].arg[0].size
#define NODEDIST(    IUDP,I)  ((double *) (udps[IUDP].arg[1].val))[I]
#define NODEDIST_SIZ(IUDP)    ( int     )  udps[IUDP].arg[1].size
#define EDGELIST(    IUDP,I)  ((int    *) (udps[IUDP].arg[2].val))[I]
#define EDGELIST_SIZ(IUDP)    ( int     )  udps[IUDP].arg[2].size
#define FACELIST(    IUDP,I)  ((int    *) (udps[IUDP].arg[3].val))[I]
#define FACELIST_SIZ(IUDP)    ( int     )  udps[IUDP].arg[3].size
#define DIST(        IUDP)    ((double *) (udps[IUDP].arg[4].val))[0]
#define DIST_SIZ(    IUDP)    ( int     )  udps[IUDP].arg[4].size
#define ATTRNAME(    IUDP)    ((char   *) (udps[IUDP].arg[5].val))
#define METHOD(      IUDP)    ((int    *) (udps[IUDP].arg[6].val))[0]
#define METHOD_SIZ(  IUDP)    ( int     )  udps[IUDP].arg[6].size

/* data about possible arguments */
static char  *argNames[NUMUDPARGS] = {"nodelist", "nodedist", "edgelist", "facelist", "dist",   "attrname", "method", };
static int    argTypes[NUMUDPARGS] = {ATTRINT,    ATTRREAL,   ATTRINT,    ATTRINT,    ATTRREAL, ATTRSTRING, ATTRINT,  };
static int    argIdefs[NUMUDPARGS] = {0,          0,          0,          0,          0,        0,          1,        };
static double argDdefs[NUMUDPARGS] = {0.,         1.,         0.,         0.,         0.,       0.,         0.,       };

/* get utility routines: udpErrorStr, udpInitialize, udpReset, udpSet,
                         udpGet, udpVel, udpClean, udpMesh */
#include "udpUtilities.c"

#include "OpenCSM.h"
#include <assert.h>

#define EPS12     1.0e-12
#define EPS30     1.0e-30

#define MIN(A,B)  (((A) < (B)) ? (A) : (B))
#define MAX(A,B)  (((A) < (B)) ? (B) : (A))

#ifdef GRAFIC
   #include "grafic.h"
#endif

/* structures */
typedef struct {
    double  xyz[3];           /* X,Y,Z coordinates */
    double  dist;             /* distance from Wnode */
    int     nedge;            /* number of incident Wedges */
    ego     enode;            /* ego in brep */
} wnode_T;

typedef struct {
    int     tag;              /* 1=can be offset from */
    int     ibeg;             /* Wnode at beginning */
    int     iend;             /* Wnode at end */
    int     ileft;            /* Wface on left */
    int     irite;            /* Wface on rite */
    int     ibleft;           /* Wedge at beginning associated with ileft */
    int     ibrite;           /* Wedge at beginning associated with irite */
    int     ieleft;           /* Wedge at end       associated with ileft */
    int     ierite;           /* Wedge at end       associated with irite */
    double  tbeg;             /* t at ibeg */
    double  tend;             /* t at end */
    ego     ecurve;           /* curve */
    ego     epleft;           /* pcurve associated with ileft */
    ego     eprite;           /* pcurve associated with irite */
    ego     eedge;            /* ego in brep */
} wedge_T;

typedef struct {
    int     tag;              /* 1=can be split */
    int     new;              /* =1 if new scribed Face */
    int     mtype;            /* SFORWARD or SREVERSE */
    ego     esurface;         /* surface */
    ego     eface;            /* ego in brep */
} wface_T;

typedef struct {
    int     nnode;            /* number of Wnodes */
    int     nedge;            /* number of Wedges */
    int     nface;            /* number of Wfaces */
    int     nface_orig;       /* number of Wfaces after wrepInit */
    wnode_T *node;            /* array  of Wnodes */
    wedge_T *edge;            /* array  of Wedges */
    wface_T *face;            /* array  of Wfaces */
    ego     ebody;            /* ego in brep */
} wrep_T;

typedef struct {
    int     iface;            /* Wface associated with this offset */
    int     ibeg;             /* Wnode at the beginning of this offset */
    int     iend;             /* Wnode at the end       of this offset */
    double  tbeg;             /* t     at the beginning of this offset */
    double  tend;             /* t     at the end       of this offset */
    ego     ecurve;           /* EGADS curve associated with this offset */
} offset_T;

/* prototypes */
static int makePlanarOffset(ego ebodyIn, ego *ebodyOut, int *NumUdp, udp_T *udps);
static int intersectLines(int npnt1, double xyz1[], int npnt2, double xyz2[], int iplane, double *t1, double *t2);

static int makeSolidOffset1(ego ebodyIn, ego *ebodyOut, int *NumUdp, udp_T *udps);
static int makeOffsetCurve(wrep_T *wrep, int iface, int iedge, offset_T *offset);
static int wrepInit(ego ebody, wrep_T **wrep);
static int wrepBreakEdge(wrep_T *wrep, int iedge, double t, int *jnode);
static int wrepMakeEdge(wrep_T *wrep, int iface, int ibeg, int iend, /*@null@*/ego einput);
static int wrepMakeNode(wrep_T *wrep, int iface, double xyz[]);
static int wrep2ego(wrep_T wrep, ego context, ego *ebody);
static int wrepPrint(wrep_T *wrep);
static int wrepFree(wrep_T *wrep);
static int makeSolidOffset2(ego ebodyIn, ego *ebodyOut, int *NumUdp, udp_T *udps);

static void printLoops(int nloop, int ibeg[], int iend[], double seg[]);
#ifdef DEBUG
static void printPoints(int nscribe, int facnum[], int begend[], int nper[], double points[]);
#endif

static int
my_scribeFaces(ego    ebody,      /* (in)  the original Body */
     /*@null@*/char   aName[],    /* (in)  scribe attribute name */
               int    nscribe,    /* (in)  the number of scribes */
               int    iface[],    /* (in)  array of Faces to scribe (bias-1) */
               int    startEnd[], /* (in)  start and ending Edge (bias-1), 2*nscribe in len */
               int    nper[],     /* (in)  array of number of points per scribe */
               double xyz[],      /* (in)  array of coordinates */
               ego    *newBody,   /* (out) newly scribed Body */
               int    *NumUdp);

#define ISWAP(A,B)        iswap=A; A=B; B=iswap;
#define RSWAP(A,B)        rswap=A; A=B; B=rswap;
#define TESS_PARAM_0      0.0250
#define TESS_PARAM_1      0.0075
#define TESS_PARAM_2      20.0

//#define MAKE_PLOTFILES 1

/* message (to be shared by all functions) */
static char message[1024];


/*
 ************************************************************************
 *                                                                      *
 *   udpExecute - execute the primitive                                 *
 *                                                                      *
 ************************************************************************
 */

int
udpExecute(ego  emodel,                 /* (in)  Model containing Body */
           ego  *ebody,                 /* (out) Body pointer */
           int  *nMesh,                 /* (out) number of associated meshes */
           char *string[])              /* (out) error message */
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, *senses;
    double  data[18];
    ego     eref, *ebodys, *echilds;
    udp_T   *udps = *Udps;

#ifdef DEBUG
    int     i;
#endif

    ROUTINE(udpExecute);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("udpExecute(emodel=%llx)\n", (long long)emodel);
    printf("nodelist(0) =");
    for (i = 0; i < NODELIST_SIZ(0); i++) {
        printf(" %d", NODELIST(0,i));
    }
    printf("\n");
    printf("nodedist(0) =");
    for (i = 0; i < NODEDIST_SIZ(0); i++) {
        printf(" %f", NODEDIST(0,i));
    }
    printf("\n");
    printf("edgelist(0) =");
    for (i = 0; i < EDGELIST_SIZ(0); i++) {
        printf(" %d", EDGELIST(0,i));
    }
    printf("\n");
    printf("facelist(0) =");
    for (i = 0; i < FACELIST_SIZ(0); i++) {
        printf(" %d", FACELIST(0,i));
    }
    printf("\n");
    printf("dist(    0) = %f\n", DIST(    0));
    printf("attrname(0) = %s\n", ATTRNAME(0));
    printf("method(  0) = %d\n", METHOD(  0));
#endif

    /* default return values */
    *ebody  = NULL;
    *nMesh  = 0;
    *string = NULL;

    message[0] = '\0';

    /* check that Model was input that contains one Body */
    status = EG_getTopology(emodel, &eref, &oclass, &mtype,
                            data, &nchild, &ebodys, &senses);
    CHECK_STATUS(EG_getTopology);

    if (oclass != MODEL) {
        snprintf(message, 1024, "expecting a Model\n");
        status = EGADS_NOTMODEL;
        goto cleanup;
    } else if (nchild != 1) {
        snprintf(message, 1024, "expecting Model to contain one Body (not %d)\n", nchild);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    /* make sure input is WireBody or SheetBody */
    status = EG_getTopology(ebodys[0], &eref, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    /* if planar, make planar offset */
    if (mtype == WIREBODY || mtype == FACEBODY || mtype == SHEETBODY) {
        status = makePlanarOffset(ebodys[0], ebody, NumUdp, udps);
        CHECK_STATUS(makePlanarOffset);

    /* of a solid, make offsets from specified Edges on specified Faces */
    } else if (mtype == SOLIDBODY) {

#ifdef GRAFIC
        {
            int     nface, iface, nloop, iloop, nedge, iedge, i, *senses, periodic;
            double  data[18], trange[4], tt;
            ego     *efaces, *eloops, *eedges, esurf, eref, topRef, prev, next;

            void    *realloc_temp=NULL;

            int     io_kbd=5, io_scr=6, indgr=1+2+4+16+64;
            int     *ilin=NULL, *isym=NULL, *nper=NULL, mline, nline, mdata, ndata;
            float   *udata=NULL, *vdata=NULL;
            char    pltitl[255];

            status = EG_getBodyTopos(ebodys[0], NULL, FACE, &nface, &efaces);
            CHECK_STATUS(EG_getBodyTopos);

            mline = 100;
            MALLOC(ilin, int, mline);
            MALLOC(isym, int, mline);
            MALLOC(nper, int, mline);

            mdata = 1000;
            MALLOC(udata, float, mdata);
            MALLOC(vdata, float, mdata);

            for (iface = 0; iface < nface; iface++) {
                ndata = 0;
                nline = 0;

                status = EG_getTopology(efaces[iface], &esurf, &oclass, &mtype,
                                        data, &nloop, &eloops, &senses);
                CHECK_STATUS(EG_getTopology);

                status = EG_getInfo(esurf, &oclass, &mtype, &topRef, &prev, &next);
                CHECK_STATUS(EG_getInfo);

                if (mtype == PLANE) continue;

                for (iloop = 0; iloop < nloop; iloop++) {
                    status = EG_getTopology(eloops[iloop], &eref, &oclass, &mtype,
                                            data, &nedge, &eedges, &senses);
                    CHECK_STATUS(EG_getTopology);

                    for (iedge = 0; iedge < nedge; iedge++) {
                        status = EG_getInfo(eedges[iedge], &oclass, &mtype, &topRef, &prev, &next);
                        CHECK_STATUS(EG_getInfo);

                        if (mtype == DEGENERATE) continue;

                        status = EG_getRange(eedges[iedge], trange, &periodic);
                        CHECK_STATUS(EG_getRange);

                        for (i = 0; i < 33; i++) {
                            tt = trange[0] + (trange[1] - trange[0]) * (double)(i) / 32.0;

                            status = EG_evaluate(eedges[iedge+nedge], &tt, data);
                            CHECK_STATUS(EG_evaluate);

                            udata[ndata] = data[0];
                            vdata[ndata] = data[1];
                            ndata++;

                            if (ndata >= mdata) {
                                mdata += 100;
                                RALLOC(udata, float, mdata);
                                RALLOC(vdata, float, mdata);
                            }
                        }

                        ilin[nline] = 1 + nline % 7;
                        isym[nline] = 1 + nline % 4;
                        nper[nline] = 33;
                        nline++;

                        if (nline >= mline) {
                            mline += 1000;
                            RALLOC(ilin, int, mline);
                            RALLOC(isym, int, mline);
                            RALLOC(nper, int, mline);
                        }
                    }
                }

                snprintf(pltitl, 254, "~u~v~Face %d", iface+1);
                grinit_(&io_kbd, &io_scr, "udfOffset", strlen("udfOffset"));
                grline_(ilin, isym, &nline, pltitl, &indgr, udata, vdata, nper, strlen(pltitl));
            }

            EG_free(efaces);

            FREE(ilin);
            FREE(isym);
            FREE(nper);
            FREE(udata);
            FREE(vdata);
        }
#endif

        if (METHOD(0) == 2) {
            status = makeSolidOffset2(ebodys[0], ebody, NumUdp, udps);
            CHECK_STATUS(makeSolidOffset2);
        } else {
            status = makeSolidOffset1(ebodys[0], ebody, NumUdp, udps);
            CHECK_STATUS(makeSolidOffset1);
        }

    } else {
        snprintf(message, 1024, "not a WireBody, SheetBody, or SolidBody");
        status = EGADS_NODATA;
        goto cleanup;
    }

    /* cache copy of arguments for future use */
    status = cacheUdp(NULL);
    CHECK_STATUS(cacheUdp);

#ifdef DEBUG
    printf("nodelist(%d) =", numUdp);
    for (i = 0; i < udps[numUdp].arg[1].size; i++) {
        printf(" %d", NODELIST(numUdp,i));
    }
    printf("\n");
    printf("nodedist(%d) =", numUdp);
    for (i = 0; i < udps[numUdp].arg[1].size; i++) {
        printf(" %f", NODEDIST(numUdp,i));
    }
    printf("\n");
    printf("edgelist(%d) =", numUdp);
    for (i = 0; i < udps[numUdp].arg[2].size; i++) {
        printf(" %d", EDGELIST(numUdp,i));
    }
    printf("\n");
    printf("facelist(%d) =", numUdp);
    for (i = 0; i < udps[numUdp].arg[3].size; i++) {
        printf(" %d", FACELIST(numUdp,i));
    }
    printf("\n");
    printf("dist(    %d) = %f\n", numUdp, DIST(numUdp));
#endif

    /* set the output value */

    /* return the new Body */
    udps[numUdp].ebody = *ebody;

cleanup:
    if (strlen(message) > 0) {
        MALLOC(*string, char, 1024);
        strncpy(*string, message, 1023);
    } else if (status != EGADS_SUCCESS) {
        *string = udpErrorStr(status);
    }

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   udpSensitivity - return sensitivity derivatives for the "real" argument *
 *                                                                      *
 ************************************************************************
 */

int
udpSensitivity(ego    ebody,            /* (in)  Body pointer */
   /*@unused@*/int    npnt,             /* (in)  number of points */
   /*@unused@*/int    entType,          /* (in)  OCSM entity type */
   /*@unused@*/int    entIndex,         /* (in)  OCSM entity index (bias-1) */
   /*@unused@*/double uvs[],            /* (in)  parametric coordinates for evaluation */
   /*@unused@*/double vels[])           /* (out) velocities */
{
    int iudp, judp;

    ROUTINE(udpSensitivity);

    /* --------------------------------------------------------------- */

    /* check that ebody matches one of the ebodys */
    iudp = 0;
    for (judp = 1; judp <= numUdp; judp++) {
        if (ebody == udps[judp].ebody) {
            iudp = judp;
            break;
        }
    }
    if (iudp <= 0) {
        return EGADS_NOTMODEL;
    }

    /* this routine is not written yet */
    return EGADS_NOLOAD;
}


/*
 ************************************************************************
 *                                                                      *
 *   makePlanarOffset - build offset Body for a planar WireBody or SheetBody *
 *                                                                      *
 ************************************************************************
 */

static int
makePlanarOffset(ego    ebodyIn,        /* (in)  input Body */
                 ego    *ebodyOut,      /* (out) output Body */
     /*@unused@*/int    *NumUdp,
                 udp_T  *udps)
{
    int     status = EGADS_SUCCESS;

    int     oclass, mtype, nchild, *senses, iplane, nloop, nedge, medge, iedge, jedge, ipnt, nchange, isave;
    int     periodic, sizes[2], ntemp1, ntemp2, *ibeg=NULL, *iend=NULL, *active=NULL, *newSenses=NULL;
    double  data[18], bbox[6], xplane=0, yplane=0, zplane=0, tt, trange[4], alen, fact, frac;
    double  **xyz=NULL, *tbeg=NULL, *tend=NULL;
    ego     context, eref, *echilds, *eedges=NULL, *eloops=NULL;
    ego     *newCurves=NULL, *newNodes=NULL, *newEdges=NULL;
    ego     newLoop, newFace, newShell, topRef, eprev, enext;

    int     npnt_min=5, npnt_max=201;

#ifdef GRAFIC
    float   *xplot=NULL, *yplot=NULL;
    int     io_kbd=5, io_scr=6, indgr=1+2+4+16+64;
    int     nline, nplot, *ilin=NULL, *isym=NULL, *nper=NULL;
    char    pltitl[80];
#endif

    ROUTINE(makePlanarOffset);

    /* --------------------------------------------------------------- */

    /* make sure DIST is only one non-zero scalar */
    if (udps[0].arg[4].size != 1) {
        snprintf(message, 1024, "\"dist\" must be a scalar");
        status = EGADS_RANGERR;
        goto cleanup;
    } else if (fabs(DIST(0)) < EPS06) {
        snprintf(message, 1024, "\"dist\" must be non-zero");
        status= EGADS_RANGERR;
        goto cleanup;
    }

    /* make sure body only contains one loop */
    status = EG_getBodyTopos(ebodyIn, NULL, LOOP, &nloop, &eloops);
    CHECK_STATUS(EG_getBodyTopos);

    if (nloop != 1) {
        snprintf(message, 1024, "input Body has %d Loops, but was expecting one\n", nloop);
        status = EGADS_NOTBODY;
        goto cleanup;
    }

    status = EG_getTopology(ebodyIn, &eref, &oclass, &mtype,
                            data, &nchild, &echilds, &senses);
    CHECK_STATUS(EG_getTopology);

    /* if a WireBody, make sure it is not non-manifold */
    if (mtype == WIREBODY) {
        status = EG_getBodyTopos(ebodyIn, NULL, EDGE, &ntemp1, NULL);
        CHECK_STATUS(EG_getBodyTopos);

        SPLINT_CHECK_FOR_NULL(eloops);

        status = EG_getTopology(eloops[0], &eref, &oclass, &mtype,
                                data, &ntemp2, &echilds, &senses);
        CHECK_STATUS(EG_getTopology);

        if (ntemp1 != ntemp2) {
            snprintf(message, 1024, "Input WireBody must be manifold\n");
            status = EGADS_NOTBODY;
            goto cleanup;
        }
    }

    /* make sure input is aligned with one of the coordinate planes */
    status = EG_getBoundingBox(ebodyIn, bbox);
    CHECK_STATUS(EG_getBoundingBox);

    if        (fabs(bbox[5]-bbox[2]) < EPS06) {
        iplane = 3;
        zplane = (bbox[5] + bbox[2]) / 2;
    } else if (fabs(bbox[4]-bbox[1]) < EPS06) {
        iplane = 2;
        yplane = (bbox[4] + bbox[1]) / 2;
    } else if (fabs(bbox[3]-bbox[0]) < EPS06) {
        iplane = 1;
        xplane = (bbox[3] + bbox[0]) / 2;
    } else {
        snprintf(message, 1024, "input Body must be aligned with a coordinate plane: dx=%f dy=%f dz=%f\n",
               bbox[3]-bbox[0], bbox[4]-bbox[1], bbox[5]-bbox[2]);
        status = EGADS_RANGERR;
        goto cleanup;
    }

    status = EG_getContext(ebodyIn, &context);
    CHECK_STATUS(EG_getContext);

    /* get the Edges associated with the input Body */
    SPLINT_CHECK_FOR_NULL(eloops);

    status = EG_getTopology(eloops[0], &eref, &oclass, &mtype,
                            data, &nedge, &eedges, &senses);
    CHECK_STATUS(EG_getTopology);

#ifdef DEBUG
    printf("Body has %d Edges\n", nedge);
#endif

    assert(nedge > 0);
    SPLINT_CHECK_FOR_NULL(eedges);

    /* allocate space for new Curves, Nodes, and Edges */
    MALLOC(ibeg,      int,     nedge  );
    MALLOC(iend,      int,     nedge  );
    MALLOC(tbeg,      double,  nedge  );
    MALLOC(tend,      double,  nedge  );
    MALLOC(active,    int,     nedge  );
    MALLOC(xyz,       double*, nedge  );
    MALLOC(newCurves, ego,     nedge  );
    MALLOC(newNodes,  ego,     nedge+1);
    MALLOC(newEdges,  ego,     nedge  );
    MALLOC(newSenses, int,     nedge  );

    for (iedge = 0; iedge < nedge; iedge++) {
        active[iedge] = 1;
        xyz[   iedge] = NULL;
    }

#ifdef GRAFIC
    MALLOC(ilin,  int,   4*nedge         );
    MALLOC(isym,  int,   4*nedge         );
    MALLOC(nper,  int,   4*nedge         );
    MALLOC(xplot, float, 4*nedge*npnt_max);
    MALLOC(yplot, float, 4*nedge*npnt_max);

    nplot = 0;
    nline = 0;
#endif

    /* make offset points for each of the Edges */
    for (iedge = 0; iedge < nedge; iedge++) {

        /* determine the number of offset points */
        status = EG_getRange(eedges[iedge], trange, &periodic);
        CHECK_STATUS(EG_getRange);

        status = EG_arcLength(eedges[iedge], trange[0], trange[1], &alen);
        CHECK_STATUS(EG_arclength);

        ibeg[iedge] = 0;
        iend[iedge] = 10 * alen / fabs(DIST(0));
        if (iend[iedge] < npnt_min) iend[iedge] = npnt_min;
        if (iend[iedge] > npnt_max) iend[iedge] = npnt_max;

#ifdef DEBUG
        printf("ibeg[%d]=%d,  iend[%d]=%d\n", iedge, ibeg[iedge], iedge, iend[iedge]);
#endif

        /* generate the points */
        MALLOC(xyz[iedge], double, 3*iend[iedge]+3);

        for (ipnt = ibeg[iedge]; ipnt <= iend[iedge]; ipnt++) {
            if (senses[iedge] == SFORWARD) {
                tt = trange[0] + (double)(ipnt-ibeg[iedge])/(double)(iend[iedge]-ibeg[iedge]) * (trange[1]-trange[0]);
            } else {
                tt = trange[1] + (double)(ipnt-ibeg[iedge])/(double)(iend[iedge]-ibeg[iedge]) * (trange[0]-trange[1]);
            }

            status = EG_evaluate(eedges[iedge], &tt, data);
            CHECK_STATUS(EG_evaluate);

#ifdef GRAFIC
            /* original curve */
            if (iplane == 3) {
                xplot[nplot] = data[0];
                yplot[nplot] = data[1];
                nplot++;
            } else if (iplane == 2) {
                xplot[nplot] = data[2];
                yplot[nplot] = data[0];
                nplot++;
            } else {
                xplot[nplot] = data[1];
                yplot[nplot] = data[2];
                nplot++;
            }
#endif

            /* apply the offset */
            if (senses[iedge] == SFORWARD) {
                fact = +DIST(0) / sqrt(data[3]*data[3] + data[4]*data[4] + data[5]*data[5]);
            } else {
                fact = -DIST(0) / sqrt(data[3]*data[3] + data[4]*data[4] + data[5]*data[5]);
            }

            if (iplane == 3) {
                xyz[iedge][3*ipnt  ] = data[0] + fact * data[4];
                xyz[iedge][3*ipnt+1] = data[1] - fact * data[3];
                xyz[iedge][3*ipnt+2] = zplane;
            } else if (iplane == 2) {
                xyz[iedge][3*ipnt  ] = data[0] - fact * data[5];
                xyz[iedge][3*ipnt+1] = yplane;
                xyz[iedge][3*ipnt+2] = data[2] + fact * data[3];
            } else {
                xyz[iedge][3*ipnt  ] = xplane;
                xyz[iedge][3*ipnt+1] = data[1] + fact * data[5];
                xyz[iedge][3*ipnt+2] = data[2] - fact * data[4];
            }
        }

#ifdef GRAFIC
        ilin[nline] = +GR_SOLID;
        isym[nline] = -GR_X;
        nper[nline] = iend[iedge] - ibeg[iedge] + 1;
        nline++;
#endif
    }

#ifdef GRAFIC
    for (iedge = 0; iedge < nedge; iedge++) {
        if (active[iedge] == 0) continue;

        /* offset curve */
        for (ipnt = ibeg[iedge]; ipnt <= iend[iedge]; ipnt++) {
            if (iplane == 3) {
                xplot[nplot] = xyz[iedge][3*ipnt  ];
                yplot[nplot] = xyz[iedge][3*ipnt+1];
                nplot++;
            } else if (iplane == 2) {
                xplot[nplot] = xyz[iedge][3*ipnt+2];
                yplot[nplot] = xyz[iedge][3*ipnt  ];
                nplot++;
            } else {
                xplot[nplot] = xyz[iedge][3*ipnt+1];
                yplot[nplot] = xyz[iedge][3*ipnt+2];
                nplot++;
            }
        }

        ilin[nline] = +GR_DOTTED;
        isym[nline] = +GR_CIRCLE;
        nper[nline] = iend[iedge] - ibeg[iedge] + 1;
        nline++;
    }
#endif

#ifdef DEBUG
    printf("before end adjustment\n");
    for (iedge = 0; iedge < nedge; iedge++) {
        printf("iedge=%d\n", iedge);
        for (ipnt = ibeg[iedge]; ipnt <= iend[iedge]; ipnt++) {
            printf("%3d %12.6f %12.6f %12.6f\n", ipnt, xyz[iedge][3*ipnt], xyz[iedge][3*ipnt+1], xyz[iedge][3*ipnt+2]);
        }
    }
#endif
#ifdef GRAFIC
    if (iplane == 3) {
        strcpy(pltitl, "~x~y~Original curves and offsets");
    } else if (iplane == 2) {
        strcpy(pltitl, "~z~x~Original curves and offsets");
    } else {
        strcpy(pltitl, "~y~z~Original curves and offsets");
    }

    grinit_(&io_kbd, &io_scr, "udfOffset", strlen("udfOffset"));
    grline_(ilin, isym, &nline, pltitl, &indgr, xplot, yplot, nper, strlen(pltitl));
#endif

    /* intersect each offset curve with its neighbors in order to
       find tbeg and tend of the intersections */
    status = EG_getInfo(eloops[0], &oclass, &mtype, &topRef, &eprev, &enext);
    CHECK_STATUS(EG_getInfo);

    medge = nedge;
    nchange = nedge - 1;
    while (nchange > 0) {
        nchange = 0;

        /* use whole range of each active Edge */
        for (iedge = 0; iedge < nedge; iedge++) {
            if (active[iedge] == 1) {
                tbeg[iedge] = 0;
                tend[iedge] = iend[iedge];
            }
        }

        /* find intersection for each pair of adjacent (active) Edges */
        for (iedge = 0; iedge < nedge; iedge++) {
            if (mtype == OPEN && iedge == medge-1) break;

            if (active[iedge] == 0) continue;     // ignore iedge if already inactive

            for (jedge = iedge+1; jedge != iedge; jedge++) {
                if (jedge >= nedge) jedge -= nedge;

                if (active[jedge] == 0) continue; // ignore jedge if already inactive

                status = intersectLines(iend[iedge]+1, xyz[iedge], iend[jedge]+1, xyz[jedge], iplane, &(tend[iedge]), &(tbeg[jedge]));
                if (status == OCSM_UDP_ERROR1) {
                    snprintf(message, 1024, "Edges %d and %d are parallel but not colinear\n", iedge+1, jedge+1);
                    goto cleanup;
                }
                CHECK_STATUS(intersectLines);

                break;
            }
        }

        /* inactivate any Edge in which t is not increasing */
        for (iedge = 0; iedge < nedge; iedge++) {
            if (active[iedge] == 1 && tend[iedge] <= tbeg[iedge]) {
                active[iedge] = 0;
                if (iedge == medge-1) medge--;
                nchange++;
            }
#ifdef DEBUG
            printf("    tbeg[%2d]=%12.4e, tend[%2d]=%12.4e,  active=%d\n", iedge, tbeg[iedge], iedge, tend[iedge], active[iedge]);
#endif
        }
    }

    /* reset ibeg and iend for each active Edge and adjust the first and last points
       so that the ends line up */
    for (iedge = 0; iedge < nedge; iedge++) {
        if (active[iedge] == 0) continue;

        if (nedge == 1) break;

        ibeg[iedge] = MAX(0, (int)tbeg[iedge]);
        frac        = tbeg[iedge] - ibeg[iedge];

        xyz[iedge][3*ibeg[iedge]  ] = (1-frac) * xyz[iedge][3*ibeg[iedge]  ] + frac * xyz[iedge][3*ibeg[iedge]+3];
        xyz[iedge][3*ibeg[iedge]+1] = (1-frac) * xyz[iedge][3*ibeg[iedge]+1] + frac * xyz[iedge][3*ibeg[iedge]+4];
        xyz[iedge][3*ibeg[iedge]+2] = (1-frac) * xyz[iedge][3*ibeg[iedge]+2] + frac * xyz[iedge][3*ibeg[iedge]+5];

        if (fabs(xyz[iedge][3*ibeg[iedge]  ]-xyz[iedge][3*ibeg[iedge]+3]) < EPS06 &&
            fabs(xyz[iedge][3*ibeg[iedge]+1]-xyz[iedge][3*ibeg[iedge]+4]) < EPS06 &&
            fabs(xyz[iedge][3*ibeg[iedge]+2]-xyz[iedge][3*ibeg[iedge]+5]) < EPS06   ) {
            ibeg[iedge]++;
        }

        iend[iedge] = MIN(iend[iedge], 1+(int)(tend[iedge]-EPS12));
        frac        = iend[iedge] - tend[iedge];

        xyz[iedge][3*iend[iedge]  ] = (1-frac) * xyz[iedge][3*iend[iedge]  ] + frac * xyz[iedge][3*iend[iedge]-3];
        xyz[iedge][3*iend[iedge]+1] = (1-frac) * xyz[iedge][3*iend[iedge]+1] + frac * xyz[iedge][3*iend[iedge]-2];
        xyz[iedge][3*iend[iedge]+2] = (1-frac) * xyz[iedge][3*iend[iedge]+2] + frac * xyz[iedge][3*iend[iedge]-1];

        if (fabs(xyz[iedge][3*iend[iedge]  ]-xyz[iedge][3*iend[iedge]-3]) < EPS06 &&
            fabs(xyz[iedge][3*iend[iedge]+1]-xyz[iedge][3*iend[iedge]-2]) < EPS06 &&
            fabs(xyz[iedge][3*iend[iedge]+2]-xyz[iedge][3*iend[iedge]-1]) < EPS06   ) {
            iend[iedge]--;
        }
    }

#ifdef DEBUG
    printf("after end adjustment\n");
    for (iedge = 0; iedge < nedge; iedge++) {
        printf("iedge=%d\n", iedge);
        for (ipnt = ibeg[iedge]; ipnt <= iend[iedge]; ipnt++) {
            printf("%3d %12.6f %12.6f %12.6f\n", ipnt, xyz[iedge][3*ipnt], xyz[iedge][3*ipnt+1], xyz[iedge][3*ipnt+2]);
        }
    }
#endif
    /* make the needed Nodes */
    isave = 0;
    jedge = 0;
    for (iedge = 0; iedge < nedge; iedge++) {
        if (active[iedge] == 0) continue;

#ifdef DEBUG
        printf("Node[%d] at (%10.5f %10.5f %10.5f\n", jedge, xyz[iedge][3*ibeg[iedge]], xyz[iedge][3*ibeg[iedge]+1], xyz[iedge][3*ibeg[iedge]+2]);
#endif

        status = EG_makeTopology(context, NULL, NODE, 0,
                                 &(xyz[iedge][3*ibeg[iedge]]), 0, NULL, NULL, &(newNodes[jedge]));
        CHECK_STATUS(EG_makeTopology);

        isave = iedge;

        jedge++;
    }

    if (mtype == CLOSED) {
        newNodes[jedge] = newNodes[0];
    } else {
#ifdef DEBUG
        printf("Node[%d] at (%10.5f %10.5f %10.5f\n", jedge, xyz[isave][3*ibeg[isave]], xyz[isave][3*ibeg[isave]+1], xyz[isave][3*ibeg[isave]+2]);
#endif

        status = EG_makeTopology(context, NULL, NODE, 0,
                                 &(xyz[isave][3*iend[isave]]), 0, NULL, NULL, &(newNodes[jedge]));
        CHECK_STATUS(EG_makeTopology);
    }

    /* make the Curves and Edges */
    jedge = 0;
    for (iedge = 0; iedge < nedge; iedge++) {
        if (active[iedge] == 0) continue;

        sizes[0] = iend[iedge] - ibeg[iedge] + 1;
        sizes[1] = 0;
        status = EG_approximate(context, 0, EPS06, sizes, &(xyz[iedge][3*ibeg[iedge]]), &newCurves[jedge]);
        CHECK_STATUS(EG_approximate);

        status = EG_getRange(newCurves[jedge], trange, &periodic);
        CHECK_STATUS(EG_getRange);

        newSenses[jedge] = SFORWARD;
        status = EG_makeTopology(context, newCurves[jedge], EDGE, TWONODE,
                                 trange, 2, &(newNodes[jedge]), &(newSenses[jedge]), &newEdges[jedge]);
        CHECK_STATUS(EG_makeTopology);

        jedge++;
    }

    /* make a Loop from the Edges */
    status = EG_makeTopology(context, NULL, LOOP, mtype,
                             NULL, jedge, newEdges, newSenses, &newLoop);
    CHECK_STATUS(EG_makeToplogy);

    /* get the mtype of the input (original) Body */
    status = EG_getInfo(ebodyIn, &oclass, &mtype, &topRef, &eprev, &enext);
    CHECK_STATUS(EG_getInfo);

    /* if input was a WireBody, we can make the WireBody directly from the Loop */
    if (mtype == WIREBODY) {
        status = EG_makeTopology(context, NULL, BODY, WIREBODY,
                                 NULL, 1, &newLoop, NULL, ebodyOut);
        CHECK_STATUS(EG_makeTopology);

    /* otherwise, make the Face, Sheet, Shell, and SheetBody */
    } else {
        status = EG_makeFace(newLoop, newSenses[0], NULL, &newFace);
        CHECK_STATUS(EG_makeFace);

        status = EG_makeTopology(context, NULL, SHELL, OPEN,
                                 NULL, 1, &newFace, NULL, &newShell);
        CHECK_STATUS(EG_makeTopology);

        status = EG_makeTopology(context, NULL, BODY, SHEETBODY,
                                 NULL, 1, &newShell, NULL, ebodyOut);
        CHECK_STATUS(EG_makeTopology);

        SPLINT_CHECK_FOR_NULL(*ebodyOut);

        /* tell OpenCSM to put _body, _brch, and Branch Attributes on the Faces */
        status = EG_attributeAdd(*ebodyOut, "__markFaces__", ATTRSTRING, 1, NULL, NULL, "true");
        CHECK_STATUS(EG_attributeAdd);
    }

cleanup:
    if (eloops != NULL) EG_free(eloops);

    if (xyz != NULL) {
        for (iedge = 0; iedge < nedge; iedge++) {
            FREE(xyz[iedge]);
        }
    }

    FREE(ibeg     );
    FREE(iend     );
    FREE(tbeg     );
    FREE(tend     );
    FREE(active   );
    FREE(xyz      );
    FREE(newCurves);
    FREE(newNodes );
    FREE(newEdges );
    FREE(newSenses);

#ifdef GRAFIC
    FREE(ilin );
    FREE(isym );
    FREE(nper );
    FREE(xplot);
    FREE(yplot);
#endif

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   interesctLines - find intersection of two discrete lines           *
 *                                                                      *
 ************************************************************************
 */

static int
intersectLines(int    npnt1,            /* (in)  number of points in line 1 */
               double xyz1[],           /* (in)  xyz array       for line 1 */
               int    npnt2,            /* (in)  number of points in line 2 */
               double xyz2[],           /* (in)  xyz array       for line 2 */
               int    iplane,           /* (in)  1=yz, 2=zx, 3=xy */
               double *t1,              /* (out) parametric coordinate for line 1 */
               double *t2)              /* (out) parametric coordinate for line 2 */
{
    int status = EGADS_SUCCESS;         /* (out) return status */

    int    ipnt1, ipnt2, redo;
    double xA, yA, xB, yB, xC, yC, xD, yD, det, s1, s2;

    ROUTINE(intersect);

    /* --------------------------------------------------------------- */

    /* default returns */
    *t1 = -HUGEQ;
    *t2 = +HUGEQ;

    ipnt1 = npnt1 - 2;                  // start at end of line 1
    ipnt2 = 0;                          // start at beg of line 2

    redo = 1;
    while (redo != 0) {
        redo = 0;

        /* get coordinates */
        if (iplane == 1) {
            xA = xyz1[3*ipnt1+1];
            yA = xyz1[3*ipnt1+2];
            xB = xyz1[3*ipnt1+4];
            yB = xyz1[3*ipnt1+5];

            xC = xyz2[3*ipnt2+1];
            yC = xyz2[3*ipnt2+2];
            xD = xyz2[3*ipnt2+4];
            yD = xyz2[3*ipnt2+5];
        } else if (iplane == 2) {
            xA = xyz1[3*ipnt1+2];
            yA = xyz1[3*ipnt1  ];
            xB = xyz1[3*ipnt1+5];
            yB = xyz1[3*ipnt1+3];

            xC = xyz2[3*ipnt2+2];
            yC = xyz2[3*ipnt2  ];
            xD = xyz2[3*ipnt2+5];
            yD = xyz2[3*ipnt2+3];
        } else {
            xA = xyz1[3*ipnt1  ];
            yA = xyz1[3*ipnt1+1];
            xB = xyz1[3*ipnt1+3];
            yB = xyz1[3*ipnt1+4];

            xC = xyz2[3*ipnt2  ];
            yC = xyz2[3*ipnt2+1];
            xD = xyz2[3*ipnt2+3];
            yD = xyz2[3*ipnt2+4];
        }

        /* find the intersection */
        det = (xB-xA) * (yC-yD) - (xC-xD) * (yB-yA);
        if (fabs(det) < EPS30) {
            if (fabs(xB-xC) < EPS06 && fabs(yB-yC) < EPS06) {
                *t1 = npnt1 - 1;
                *t2 = 0;
            } else {
                snprintf(message, 1014, "no intersection");
                status = OCSM_UDP_ERROR1;
            }

            goto cleanup;
        } else {
            s1 = ((xC-xA) * (yC-yD) - (xC-xD) * (yC-yA)) / det;
            s2 = ((xB-xA) * (yC-yA) - (xC-xA) * (yB-yA)) / det;
        }

        /* if not in this segment in line 1, decrease ipnt1 if possible and redo */
        if (s1 < 0) {
            if (ipnt1 > 1) {
                ipnt1--;
                redo ++;
            } else {
                *t2 = ipnt2 + s2;
                goto cleanup;
            }
        }

        /* if not in this segment in line 2, increase ipnt2 if possible and redo */
        if (s2 > 1) {
            if (ipnt2 < npnt2-2) {
                ipnt2++;
                redo ++;
            } else {
                *t1 = ipnt1 + s1;
                goto cleanup;
            }
        }
    }

    *t1 = ipnt1 + s1;
    *t2 = ipnt2 + s2;

cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   makeSolidOffset1 - build offset Body for a SolidBody (method 1)    *
 *                                                                      *
 ************************************************************************
 */

static int
makeSolidOffset1(ego    ebodyIn,        /* (in)  input  Body */
                 ego    *ebodyOut,      /* (out) output Body */
     /*@unused@*/int    *NumUdp,
                 udp_T  *udps)
{
    int     status = EGADS_SUCCESS;

    int      i, iface, iedge, jedge, medge, inode, jnode;
    int      itry, ntry=10, iswap;
    int      *ibegList=NULL, *iendList=NULL;
    double   data1[18], data2[18], tbreak, arclength;
    double   xyz0[18], uv1[2], uv2[2], xyz_temp[18], fact1, fact2, dist, dtest, den, t1;
    ego      context, epcurvei, epcurvej, etemp;
    wrep_T   *wrep=NULL;
    offset_T *offset=NULL;

#ifdef GRAFIC
    int      io_kbd=5, io_scr=6, indgr=1+2+4+16+64;
    int      nline, nplot, *ilin=NULL, *isym=NULL, *nper=NULL;
    int      ipnt, npnt=51;
    float    *uplot=NULL, *vplot=NULL;
    double   tt, xyz_out[3], uv_out[2], data[18];
    char     pltitl[80];
#endif

    ROUTINE(makeSolidOffset1);

    /* --------------------------------------------------------------- */

    /* check inputs */
    if (udps[0].arg[0].size != udps[0].arg[1].size) {
        snprintf(message, 1024, "\"nodelist\" and \"nodedist must have the same number of entries");
        status = EGADS_RANGERR;
        goto cleanup;
    }

    if (udps[0].arg[0].size > 1 || NODELIST(0,0) != 0) {
        for (i = 0; i < udps[0].arg[0].size; i++) {
            if (NODELIST(0,i) <= 0) {
                snprintf(message, 1024, "\"nodelist\" must contain all positive values");
                status = EGADS_RANGERR;
                goto cleanup;
            }
        }

        for (i = 0; i < udps[0].arg[1].size; i++) {
            if (NODEDIST(0,i) <= 0) {
                snprintf(message, 1024, "\"nodedist\" must contain all positive values");
                status = EGADS_RANGERR;
                goto cleanup;
            }
        }
    }

    for (i = 0; i < udps[0].arg[2].size; i++) {
        if (EDGELIST(0,i) <= 0) {
            snprintf(message, 1024, "\"edgelist\" must contain all positive values");
            status = EGADS_RANGERR;
            goto cleanup;
        }
    }

    for (i = 0; i < udps[0].arg[3].size; i++) {
        if (FACELIST(0,i) <= 0) {
            snprintf(message, 1024, "\"facelist\" must contain all positive values");
            status = EGADS_RANGERR;
            goto cleanup;
        }
    }

    if (udps[0].arg[4].size != 1) {
        snprintf(message, 1024, "\"dist\" must be a scalar");
        status = EGADS_RANGERR;
        goto cleanup;
    } else if (DIST(0) <= 0) {
        snprintf(message, 1024, "\"dist\" must be positive");
        status = EGADS_RANGERR;
        goto cleanup;
    }

    status = EG_getContext(ebodyIn, &context);
    CHECK_STATUS(EG_getContext);

    /* make the winged-edge data structure */
    status = wrepInit(ebodyIn, &wrep);
    CHECK_STATUS(wrepInit);

    SPLINT_CHECK_FOR_NULL(wrep);

    /* mark the Nodes, Edges, and Faces that are in the nodeList, edgeList, and faceList */
    for (inode = 1; inode <= wrep->nnode; inode++) {
        wrep->node[inode].dist = DIST(0);
    }

    if (udps[0].arg[0].size > 1 || NODELIST(0,0) != 0) {
        for (i = 0; i < udps[0].arg[0].size; i++) {
            inode = NODELIST(0,i);
            if (inode >= 1 && inode <= wrep->nnode) {
                wrep->node[inode].dist = NODEDIST(0,i);
            } else {
                snprintf(message, 1024, "\"nodelist[%d]\" (%d) is out of bounds", i+1, NODELIST(0,i));
                status = EGADS_RANGERR;
                goto cleanup;
            }
        }
    }

    for (i = 0; i < udps[0].arg[2].size; i++) {
        iedge = EDGELIST(0,i);
        if (iedge >= 1 && iedge <= wrep->nedge) {
            wrep->edge[iedge].tag = 1;
        } else {
            snprintf(message, 1024, "\"edgelist[%d]\" (%d) is out of bounds", i+1, EDGELIST(0,i));
            status = EGADS_RANGERR;
            goto cleanup;
        }
    }

    for (i = 0; i < udps[0].arg[3].size; i++) {
        iface = FACELIST(0, i);
        if (iface >= 1 && iface <= wrep->nface) {
            wrep->face[iface].tag = 1;
        } else {
            snprintf(message, 1024, "\"facelist[%d]\" (%d) is out of bounds", i+1, FACELIST(0,i));
            status = EGADS_RANGERR;
            goto cleanup;
        }
    }

#ifdef DEBUG
    printf("initial wrep\n");
    status = wrepPrint(wrep);
    CHECK_STATUS(wrepPrint);
#endif

#ifdef GRAFIC
    MALLOC(ilin,  int,   5     *wrep->nedge);
    MALLOC(isym,  int,   5     *wrep->nedge);
    MALLOC(nper,  int,   5     *wrep->nedge);
    MALLOC(uplot, float, 5*npnt*wrep->nedge);
    MALLOC(vplot, float, 5*npnt*wrep->nedge);
#endif

    /* create array of possible offset Wedges for any Wface */
    medge = 3 * (wrep->nedge+1);
    MALLOC(offset, offset_T, medge);

    /* split any Wedge that is not tagged and which adjoins a
       Wedge that is */
    for (iedge = 1; iedge <= wrep->nedge; iedge++) {
        if (wrep->edge[iedge].tag != 0) continue;           // tagged     Wedge
        if (wrep->edge[iedge].ecurve == NULL) continue;     // degenerate Wedge

        /* (possibly) split at beginning */
        if ((wrep->face[wrep->edge[iedge].ileft ].tag == 1 &&
             wrep->edge[wrep->edge[iedge].ibleft].tag == 1   ) ||
            (wrep->face[wrep->edge[iedge].irite ].tag == 1 &&
             wrep->edge[wrep->edge[iedge].ibrite].tag == 1   )   ) {
            status = EG_arcLength(wrep->edge[iedge].ecurve,
                                  wrep->edge[iedge].tbeg, wrep->edge[iedge].tend, &arclength);
            CHECK_STATUS(EG_arclength);

            dist = wrep->node[wrep->edge[iedge].ibeg].dist;

            if (fabs(dist-arclength) < EPS06) {
                /* splitting is not required */
            } else if (dist < arclength) {
                tbreak = wrep->edge[iedge].tbeg
                      + (wrep->edge[iedge].tend - wrep->edge[iedge].tbeg) * dist / arclength;

                for (itry = 0; itry < ntry; itry++) {
                    status = EG_arcLength(wrep->edge[iedge].ecurve,
                                          wrep->edge[iedge].tbeg, tbreak, &arclength);
                    CHECK_STATUS(EG_arcLength);

                    if (fabs(dist-arclength) < EPS06) break;

                    tbreak = wrep->edge[iedge].tbeg
                        + (tbreak - wrep->edge[iedge].tbeg) * dist / arclength;
                }

                status = wrepBreakEdge(wrep, iedge, tbreak, &jnode);
                CHECK_STATUS(wrepBreakEdge);
            } else {
                snprintf(message, 1024, "Wedge %d is shorter than offset distance", iedge);
                status = OCSM_UDP_ERROR1;
                goto cleanup;
            }
        }

        /* split at end */
        if ((wrep->face[wrep->edge[iedge].ileft ].tag == 1 &&
             wrep->edge[wrep->edge[iedge].ieleft].tag == 1   ) ||
            (wrep->face[wrep->edge[iedge].irite ].tag == 1 &&
             wrep->edge[wrep->edge[iedge].ierite].tag == 1   )   ) {
            status = EG_arcLength(wrep->edge[iedge].ecurve,
                                  wrep->edge[iedge].tbeg, wrep->edge[iedge].tend, &arclength);
            CHECK_STATUS(EG_arclength);

            dist = wrep->node[wrep->edge[iedge].iend].dist;

            if (fabs(dist-arclength) < EPS06) {
                /* splitting is not required */
            } else if (dist < arclength) {
                tbreak = wrep->edge[iedge].tend
                      + (wrep->edge[iedge].tbeg - wrep->edge[iedge].tend) * dist / arclength;

                for (itry = 0; itry < ntry; itry++) {
                    status = EG_arcLength(wrep->edge[iedge].ecurve,
                                          tbreak, wrep->edge[iedge].tend, &arclength);
                    CHECK_STATUS(EG_arcLength);

                    if (fabs(dist-arclength) < EPS06) break;

                    tbreak = wrep->edge[iedge].tend
                        + (tbreak - wrep->edge[iedge].tend) * dist / arclength;
                }

                status = wrepBreakEdge(wrep, iedge, tbreak, &jnode);
                CHECK_STATUS(wrepBreakEdge);
            } else {
                snprintf(message, 1024, "Wedge %d is shorter than offset distance", iedge);
                status = OCSM_UDP_ERROR1;
                goto cleanup;
            }

        }
    }

#ifdef DEBUG
        printf("after splitting Wedges\n");
        status = wrepPrint(wrep);
        CHECK_STATUS(wrepPrint);
#endif

    /* process each tagged Wface */
    for (iface = 1; iface <= wrep->nface; iface++) {
        if (wrep->face[iface].tag == 0) continue;

#ifdef GRAFIC
        nplot = 0;
        nline = 0;

        /* draw outline */
        for (iedge = 1; iedge <= wrep->nedge; iedge++) {
            if (wrep->edge[iedge].ileft != iface &&
                wrep->edge[iedge].irite != iface   ) continue;

            jedge = iedge;
            for (ipnt = 0; ipnt < npnt; ipnt++) {
                double uvout[2], xyzout[3];
                tt = wrep->edge[iedge].tbeg + (wrep->edge[iedge].tend - wrep->edge[iedge].tbeg) * (double)(ipnt) / (double)(npnt-1);

                status = EG_evaluate(wrep->edge[iedge].eedge, &tt, data);
                CHECK_STATUS(EG_evaluate);

                status = EG_invEvaluate(wrep->face[iface].esurface, data, uvout, xyzout);
                CHECK_STATUS(EG_invEvaluate);

                uplot[nplot] = uvout[0];
                vplot[nplot] = uvout[1];
                nplot++;
            }

            ilin[nline] = +GR_SOLID;
            isym[nline] = -GR_PLUS;
            nper[nline] = npnt;
            nline++;
        }
#endif

        /* initialize the offsets for this Wface */
        for (iedge = 0; iedge < medge; iedge++) {
            offset[iedge].iface  = 0;
            offset[iedge].ibeg   = 0;
            offset[iedge].iend   = 0;
            offset[iedge].tbeg   = 0;
            offset[iedge].tend   = 0;
            offset[iedge].ecurve = NULL;
        }

        /* loop through each Wedge that is tagged and that adjoins this Wface */
        for (iedge = wrep->nedge; iedge > 0; iedge--) {
            if (wrep->edge[iedge].tag == 0) continue;

            /* jedge is the previous Wedge */
            if        (wrep->edge[iedge].ileft == iface) {
                jedge = wrep->edge[iedge].ibleft;
                dist  = wrep->node[wrep->edge[iedge].ibeg].dist;
                fact1 = + dist * wrep->face[iface].mtype;
            } else if (wrep->edge[iedge].irite == iface) {
                jedge = wrep->edge[iedge].ibrite;
                dist  = wrep->node[wrep->edge[iedge].ibeg].dist;
                fact1 = - dist * wrep->face[iface].mtype;
            } else {
                continue;
            }

            /* no need to process if ibeg is already set */
            if (offset[iedge].ibeg != 0) {

            /* if the previous Wedge (jedge) is not tagged, set offset.ibeg to
               the end of jedge */
            } else if (wrep->edge[jedge].tag == 0) {
                if        (wrep->edge[jedge].ibeg == wrep->edge[iedge].ibeg) {
                    offset[iedge].ibeg = wrep->edge[jedge].iend;
                } else if (wrep->edge[jedge].iend == wrep->edge[iedge].ibeg) {
                    offset[iedge].ibeg = wrep->edge[jedge].ibeg;
                } else {
                    status = OCSM_INTERNAL_ERROR;
                    snprintf(message, 1024, "expecting Wedge %d to be attached to Wnode %d",
                             jedge, wrep->edge[iedge].iend);
                    goto cleanup;
                }

#ifdef GRAFIC
                status = EG_invEvaluate(wrep->face[iface].esurface, wrep->node[offset[iedge].ibeg].xyz, uv_out, xyz_out);
                CHECK_STATUS(EG_invEvaluate);

                uplot[nplot] = uv_out[0];
                vplot[nplot] = uv_out[1];
                nplot++;

                ilin[nline] = 0;
                isym[nline] = GR_STAR;
                nper[nline] = 1;
                nline++;
#endif

            /* if the previous Wedge (jedge) is tagged, find the intersection
               of offsets, create a Wnode and a Wedge */
            } else {
                status = EG_otherCurve(wrep->face[iface].esurface, wrep->edge[iedge].eedge, 0, &epcurvei);
                CHECK_STATUS(EG_otherCurve);

                status = EG_otherCurve(wrep->face[iface].esurface, wrep->edge[jedge].eedge, 0, &epcurvej);
                CHECK_STATUS(EG_otherCurve);

                /* beg of iedge, end of jedge */
                if ((wrep->edge[iedge].ileft == iface && wrep->edge[jedge].ileft == iface) ||
                    (wrep->edge[iedge].irite == iface && wrep->edge[jedge].irite == iface)   ) {
                    status = EG_evaluate(epcurvei, &wrep->edge[iedge].tbeg, data1);
                    CHECK_STATUS(EG_evaluate);

                    status = EG_evaluate(epcurvej, &wrep->edge[jedge].tend, data2);
                    CHECK_STATUS(EG_evaluate);

                    fact2 = +fact1;

                /* beg of iedge, beg of jedge */
                } else if ((wrep->edge[iedge].ileft == iface && wrep->edge[jedge].irite == iface) ||
                           (wrep->edge[iedge].irite == iface && wrep->edge[jedge].ileft == iface)   ) {
                    status = EG_evaluate(epcurvei, &wrep->edge[iedge].tbeg, data1);
                    CHECK_STATUS(EG_evaluate);

                    status = EG_evaluate(epcurvej, &wrep->edge[jedge].tbeg, data2);
                    CHECK_STATUS(EG_evaluate);

                    fact2 = -fact1;

                } else {
                    snprintf(message, 1024, "iedge=%d and jedge=%d do not have a common Wnode", iedge, jedge);
                    status = OCSM_INTERNAL_ERROR;
                    goto cleanup;
                }

                status = EG_evaluate(wrep->face[iface].esurface, data1, xyz0);
                CHECK_STATUS(EG_evaluate);

                for (itry = 0; itry < ntry; itry++) {
                    uv1[0] = data1[0] - fact1 * data1[3];
                    uv1[1] = data1[1] + fact1 * data1[2];

                    status = EG_evaluate(wrep->face[iface].esurface, uv1, xyz_temp);
                    CHECK_STATUS(EG_evaluate);

                    dtest = sqrt((xyz_temp[0]-xyz0[0]) * (xyz_temp[0]-xyz0[0])
                                +(xyz_temp[1]-xyz0[1]) * (xyz_temp[1]-xyz0[1])
                                +(xyz_temp[2]-xyz0[2]) * (xyz_temp[2]-xyz0[2]));

                    if (fabs(dtest-dist) < EPS06) break;
                    fact1 = fact1 * dist / dtest;
                }

                for (itry = 0; itry < ntry; itry++) {
                    uv2[0] = data2[0] - fact2 * data2[3];
                    uv2[1] = data2[1] + fact2 * data2[2];

                    status = EG_evaluate(wrep->face[iface].esurface, uv2, xyz_temp);
                    CHECK_STATUS(EG_evaluate);

                    dtest = sqrt((xyz_temp[0]-xyz0[0]) * (xyz_temp[0]-xyz0[0])
                                +(xyz_temp[1]-xyz0[1]) * (xyz_temp[1]-xyz0[1])
                                +(xyz_temp[2]-xyz0[2]) * (xyz_temp[2]-xyz0[2]));

                    if (fabs(dtest-dist) < EPS06) break;
                    fact2 = fact2 * dist / dtest;
                }

                den = (data1[3] * data2[2] - data1[2] * data2[3]);
                if (fabs(den) < EPS12) {
                    uv1[0] = (uv1[0] + uv2[0]) / 2;
                    uv1[1] = (uv1[1] + uv2[1]) / 2;
                } else {
                    t1 = ((uv2[1] - uv1[1]) * data2[2] - (uv2[0] - uv1[0]) * data2[3]) / den;
                    uv1[0] = uv1[0] + data1[2] * t1;
                    uv1[1] = uv1[1] + data1[3] * t1;
                }

#ifdef GRAFIC
                uplot[nplot] = data1[0];
                vplot[nplot] = data1[1];
                nplot++;

                uplot[nplot] = uv1[0];
                vplot[nplot] = uv1[1];
                nplot++;

                ilin[nline] = GR_DOTTED;
                isym[nline] = GR_CIRCLE;
                nper[nline] = 2;
                nline++;
#endif

                status = EG_evaluate(wrep->face[iface].esurface, uv1, data1);
                CHECK_STATUS(EG_evaluate);

                jnode = status = wrepMakeNode(wrep, iface, data1);
                CHECK_STATUS(wrepMakeNode);

                status = wrepMakeEdge(wrep, iface, wrep->edge[iedge].ibeg, jnode, NULL);
                CHECK_STATUS(wrepMakeEdge);

                offset[iedge].ibeg = jnode;

                if        (wrep->edge[jedge].ibeg == wrep->edge[iedge].ibeg) {
                    offset[jedge].ibeg = jnode;
                } else if (wrep->edge[jedge].iend == wrep->edge[iedge].ibeg) {
                    offset[jedge].iend = jnode;
                }
            }

            /* jedge is the next Wedge */
            if  (wrep->edge[iedge].ileft == iface) {
                jedge = wrep->edge[iedge].ieleft;
            } else {
                jedge = wrep->edge[iedge].ierite;
            }

            /* no need to process if iend is already set */
            if (offset[iedge].iend != 0) {

            /* if the next Wedge (jedge) is not tagged, set offset.iend to
               the end of jedge */
            } else if (wrep->edge[jedge].tag == 0) {
                if        (wrep->edge[jedge].ibeg == wrep->edge[iedge].iend) {
                    offset[iedge].iend = wrep->edge[jedge].iend;
                } else if (wrep->edge[jedge].iend == wrep->edge[iedge].iend) {
                    offset[iedge].iend = wrep->edge[jedge].ibeg;
                } else {
                    status = OCSM_INTERNAL_ERROR;
                    snprintf(message, 1024, "expecting Wedge %d to be attached to Wnode %d",
                             jedge, wrep->edge[iedge].iend);
                    goto cleanup;
                }

#ifdef GRAFIC
                status = EG_invEvaluate(wrep->face[iface].esurface, wrep->node[offset[iedge].iend].xyz, uv_out, xyz_out);
                CHECK_STATUS(EG_invEvaluate);

                uplot[nplot] = uv_out[0];
                vplot[nplot] = uv_out[1];
                nplot++;

                ilin[nline] = 0;
                isym[nline] = GR_STAR;
                nper[nline] = 1;
                nline++;
#endif

            /* if the next Wedge (jedge) is tagged, find the intersection
               of offsets, create a Wnode and a Wedge */
            } else {
                status = EG_otherCurve(wrep->face[iface].esurface, wrep->edge[iedge].eedge, 0, &epcurvei);
                CHECK_STATUS(EG_otherCurve);

                status = EG_otherCurve(wrep->face[iface].esurface, wrep->edge[jedge].eedge, 0, &epcurvej);
                CHECK_STATUS(EG_otherCurve);

                /* end of iedge, beg of jedge */
                if ((wrep->edge[iedge].ileft == iface && wrep->edge[jedge].ileft == iface) ||
                    (wrep->edge[iedge].irite == iface && wrep->edge[jedge].irite == iface)   ) {
                    status = EG_evaluate(epcurvei, &wrep->edge[iedge].tend, data1);
                    CHECK_STATUS(EG_evaluate);

                    status = EG_evaluate(epcurvej, &wrep->edge[jedge].tbeg, data2);
                    CHECK_STATUS(EG_evaluate);

                    fact2 = +fact1;

                /* end of iedge, end of jedge */
                } else if ((wrep->edge[iedge].ileft == iface && wrep->edge[jedge].irite == iface) ||
                           (wrep->edge[iedge].irite == iface && wrep->edge[jedge].ileft == iface)   ) {
                    status = EG_evaluate(epcurvei, &wrep->edge[iedge].tend, data1);
                    CHECK_STATUS(EG_evaluate);

                    status = EG_evaluate(epcurvej, &wrep->edge[jedge].tend, data2);
                    CHECK_STATUS(EG_evaluate);

                    fact2 = -fact1;

                } else {
                    snprintf(message, 1024, "iedge=%d and jedge=%d do not have a common Wnode", iedge, jedge);
                    status = OCSM_INTERNAL_ERROR;
                    goto cleanup;
                }

                status = EG_evaluate(wrep->face[iface].esurface, data1, xyz0);
                CHECK_STATUS(EG_evaluate);

                for (itry = 0; itry < ntry; itry++) {
                    uv1[0] = data1[0] - fact1 * data1[3];
                    uv1[1] = data1[1] + fact1 * data1[2];

                    status = EG_evaluate(wrep->face[iface].esurface, uv1, xyz_temp);
                    CHECK_STATUS(EG_evaluate);

                    dtest = sqrt((xyz_temp[0]-xyz0[0]) * (xyz_temp[0]-xyz0[0])
                                +(xyz_temp[1]-xyz0[1]) * (xyz_temp[1]-xyz0[1])
                                +(xyz_temp[2]-xyz0[2]) * (xyz_temp[2]-xyz0[2]));

                    if (fabs(dtest-dist) < EPS06) break;
                    fact1 = fact1 * dist / dtest;
                }

                for (itry = 0; itry < ntry; itry++) {
                    uv2[0] = data2[0] - fact2 * data2[3];
                    uv2[1] = data2[1] + fact2 * data2[2];

                    status = EG_evaluate(wrep->face[iface].esurface, uv2, xyz_temp);
                    CHECK_STATUS(EG_evaluate);

                    dtest = sqrt((xyz_temp[0]-xyz0[0]) * (xyz_temp[0]-xyz0[0])
                                +(xyz_temp[1]-xyz0[1]) * (xyz_temp[1]-xyz0[1])
                                +(xyz_temp[2]-xyz0[2]) * (xyz_temp[2]-xyz0[2]));

                    if (fabs(dtest-dist) < EPS06) break;
                    fact2 = fact2 * dist / dtest;
                }

                den = (data1[3] * data2[2] - data1[2] * data2[3]);
                if (fabs(den) < EPS12) {
                    uv1[0] = (uv1[0] + uv2[0]) / 2;
                    uv1[1] = (uv1[1] + uv2[1]) / 2;
                } else {
                    t1 = ((uv2[1] - uv1[1]) * data2[2] - (uv2[0] - uv1[0]) * data2[3]) / den;
                    uv1[0] = uv1[0] + data1[2] * t1;
                    uv1[1] = uv1[1] + data1[3] * t1;
                }

#ifdef GRAFIC
                uplot[nplot] = data1[0];
                vplot[nplot] = data1[1];
                nplot++;

                uplot[nplot] = uv1[0];
                vplot[nplot] = uv1[1];
                nplot++;

                ilin[nline] = GR_DOTTED;
                isym[nline] = GR_CIRCLE;
                nper[nline] = 2;
                nline++;
#endif

                status = EG_evaluate(wrep->face[iface].esurface, uv1, data1);
                CHECK_STATUS(EG_evaluate);

                jnode = status = wrepMakeNode(wrep, iface, data1);
                CHECK_STATUS(wrepMakeNode);

                status = wrepMakeEdge(wrep, iface, wrep->edge[iedge].iend, jnode, NULL);
                CHECK_STATUS(wrepMakeEdge);

                offset[iedge].iend = jnode;

                if        (wrep->edge[jedge].ibeg == wrep->edge[iedge].iend) {
                    offset[jedge].ibeg = jnode;
                } else if (wrep->edge[jedge].iend == wrep->edge[iedge].iend) {
                    offset[jedge].iend = jnode;
                }
            }

            /* create an offset curve and associated Wedge (which will make a Wface) */
            status = makeOffsetCurve(wrep, iface, iedge, &offset[iedge]);
            CHECK_STATUS(makeOffsetCurve);
        }

#ifdef DEBUG
        for (iedge = 1; iedge <= wrep->nedge; iedge++) {
            printf("iedge=%3d, iface=%3d, ibeg=%3d, iend=%3d, tbeg=%10.5f, tend=%10.5f, ecurve=%llx\n",
                   iedge, offset[iedge].iface, offset[iedge].ibeg, offset[iedge].iend, offset[iedge].tbeg, offset[iedge].tend, (long long)(offset[iedge].ecurve));
        }
#endif

        for (iedge = 1; iedge <= wrep->nedge; iedge++) {
            if (offset[iedge].ecurve == NULL) continue;

            /* orient the new Wedge so that its right side faces it Wedge */
            if        (wrep->edge[iedge].ileft == iface) {
                /* do nothing */
            } else if (wrep->edge[iedge].irite == iface) {
#ifdef DEBUG
                printf("swapping ends for iedge=%d\n", iedge);
#endif
                iswap              = offset[iedge].ibeg;
                offset[iedge].ibeg = offset[iedge].iend;
                offset[iedge].iend = iswap;

                status = EG_flipObject(offset[iedge].ecurve, &etemp);
                CHECK_STATUS(EG_flipObject);

                status = EG_deleteObject(offset[iedge].ecurve);
                CHECK_STATUS(EG_deleteObject);

                offset[iedge].ecurve = etemp;
            } else {
                snprintf(message, 1024, "neither ileft nor irite of iedge=%d is set to iface=%d", iedge, iface);
                status = OCSM_INTERNAL_ERROR;
                goto cleanup;
            }

            status = wrepMakeEdge(wrep, iface, offset[iedge].ibeg, offset[iedge].iend, offset[iedge].ecurve);
            CHECK_STATUS(wrepMakeEdge);

#ifdef GRAFIC
            jedge = status;
            for (ipnt = 0; ipnt < npnt; ipnt++) {
                double uvout[2], xyzout[3];
                tt = wrep->edge[jedge].tbeg + (wrep->edge[jedge].tend - wrep->edge[jedge].tbeg) * (double)(ipnt) / (double)(npnt-1);

                status = EG_evaluate(wrep->edge[jedge].ecurve, &tt, data);
                CHECK_STATUS(EG_evaluate);

                status = EG_invEvaluate(wrep->face[iface].esurface, data, uvout, xyzout);
                CHECK_STATUS(EG_invEvaluate);

                uplot[nplot] = uvout[0];
                vplot[nplot] = uvout[1];
                nplot++;
            }

            ilin[nline] = +GR_DASHED;
            isym[nline] = -GR_PLUS;
            nper[nline] = npnt;
            nline++;
#endif
        }

#ifdef GRAFIC
        snprintf(pltitl, 79, "udfScribe --- offset ends");
        grinit_(&io_kbd, &io_scr, pltitl, STRLEN(pltitl));

        if (wrep->face[iface].mtype > 0) {
            snprintf(pltitl, 79, "~u~v~Face %d", iface);
        } else {
            for (i = 0; i < nplot; i++) {
                uplot[i] *= -1;
            }
            snprintf(pltitl, 79, "~-u~v~Face %d", iface);
        }
        grline_(ilin, isym, &nline, pltitl, &indgr, uplot, vplot, nper, STRLEN(pltitl));
#endif
    }

#ifdef DEBUG
    printf("final wrep\n");
    status = wrepPrint(wrep);
    CHECK_STATUS(wrepPrint);

    for (iface = 1; iface <= wrep->nface; iface++) {
        printf("Wface %3d: Wedges", iface);
        for (iedge = 1; iedge <= wrep->nedge; iedge++) {
            if (wrep->edge[iedge].ileft == iface) {
                printf(" %3d", -iedge);
            } else if (wrep->edge[iedge].irite == iface) {
                printf(" %3d", +iedge);
            }
        }
        printf("\n");
    }
#endif

    /* put the "scribeFace" attribute on Faces that are within
       the scribe */
    for (iface = 1; iface <= wrep->nface; iface++) {
        wrep->face[iface].new = 0;

        if (wrep->face[iface].tag != 0 || iface > wrep->nface_orig) {
            for (iedge = 1; iedge <= wrep->nedge; iedge++) {
                if (wrep->edge[iedge].tag == 0) continue;

                if (wrep->edge[iedge].ileft == iface ||
                    wrep->edge[iedge].irite == iface   ) {
                    wrep->face[iface].new = 1;
                }
            }
        }
    }

    FREE(offset);

#ifdef DEBUG
    printf("final wrep\n");
    status = wrepPrint(wrep);
    CHECK_STATUS(wrepPrint);
#endif

    /* try to make a new ebody */
    status = wrep2ego(*wrep, context, ebodyOut);
    CHECK_STATUS(wrep2ego);

    SPLINT_CHECK_FOR_NULL(*ebodyOut);

#ifdef DEBUG
    printf("ebody\n");
    ocsmPrintEgo(*ebodyOut);
#endif

    /* call to avoid compiler warning */
    if (wrep->nnode == -2 && wrep->nedge == -2 && wrep->nface == -2) {
        status = wrepPrint(wrep);
        CHECK_STATUS(wrepPrint);
    }

cleanup:
    /* free the WREP structure */
    wrepFree(wrep);
    FREE(wrep);

    FREE(offset);
    FREE(ibegList);
    FREE(iendList);

#ifdef GRAFIC
    FREE(ilin);
    FREE(isym);
    FREE(nper);
    FREE(uplot);
    FREE(vplot);
#endif

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   makeOffsetCurve - make an (untrimmed) offset curve                 *
 *                                                                      *
 ************************************************************************
 */

static int
makeOffsetCurve(wrep_T    *wrep,        /* (in)  pointer to WREP structure */
                int       iface,        /* (in)  Wface index (bias-1) */
                int       iedge,        /* (in)  Wedge index (bias-1) */
                offset_T  *offset)      /* (in)  offset information */
{
    int     status = EGADS_SUCCESS;     /* (out) return status */

    int     npnt, ipnt;
    int     oclass, mtype, sizes[2], i, *header;
    double  data1[18], tt, *gdata;
    double  tang[2], duv[2], duv_beg[2], duv_end[2];
    double  data[18], uv_out[2], xyz_out[3], frac;
    double  *uv1=NULL, *uv2=NULL, *xyz=NULL;
    double  toler=EPS06;
    ego     epcurve, topRef, prev, next, context, etemp, eref;

    ROUTINE(makeOffsetCurve);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("enter makeOffsetCurve(iface=%d, iedge=%d)\n", iface, iedge);
#endif

    npnt = 21;
    MALLOC(uv1, double, 2*npnt);        /* parametric coordinates of the Wedge */
    MALLOC(uv2, double, 3*npnt);        /* parametric coordinates of the offset */
    MALLOC(xyz, double, 3*npnt);        /* physical   coordinates of the offset */

    /* get the pcurve associated with this iedge,iface */
    status = EG_getInfo(wrep->face[iface].esurface, &oclass, &mtype, &topRef, &prev, &next);
    CHECK_STATUS(EG_getInfo);

    if (mtype == PLANE) {
        status = EG_otherCurve(wrep->face[iface].esurface, wrep->edge[iedge].ecurve, 0.0, &epcurve);
        CHECK_STATUS(EG_otherCurve);
    } else if (wrep->edge[iedge].ileft == iface) {
        epcurve = wrep->edge[iedge].epleft;
    } else if (wrep->edge[iedge].irite == iface) {
        epcurve = wrep->edge[iedge].eprite;
    } else {
        snprintf(message, 1024, "Wface %d is not planar and does not point to Wedge %d", iface, iedge);
        status = OCSM_INTERNAL_ERROR;
        goto cleanup;
    }

    /* find the uv along the Wedge (to be used below as the anchor
       for each tuft) */
    for (ipnt = 0; ipnt < npnt; ipnt++) {
        tt = wrep->edge[iedge].tbeg
           + (wrep->edge[iedge].tend - wrep->edge[iedge].tbeg) * (double)(ipnt) / (double)(npnt-1);

        status = EG_evaluate(epcurve, &tt, data1);
        CHECK_STATUS(EG_evaluate);

        uv1[2*ipnt  ] = data1[0];
        uv1[2*ipnt+1] = data1[1];
    }

    /* find the difference in UV between the ends of the Wedge and the ends of the offset */
    status = EG_evaluate(epcurve, &wrep->edge[iedge].tbeg, data);
    CHECK_STATUS(EG_evaluate);

    tang[0] = data[2] / sqrt(data[2] * data[2] + data[3] * data[3]);     // norm[1] = +tang[0]
    tang[1] = data[3] / sqrt(data[2] * data[2] + data[3] * data[3]);     // norm[0] = -tang[1]

    status = EG_invEvaluate(wrep->face[iface].esurface, wrep->node[offset->ibeg].xyz, uv_out, xyz_out);
    CHECK_STATUS(EG_invEvaluate);

    duv_beg[0] = +tang[0] * (uv_out[0] - uv1[0]) + tang[1] * (uv_out[1] - uv1[1]);
    duv_beg[1] = -tang[1] * (uv_out[0] - uv1[0]) + tang[0] * (uv_out[1] - uv1[1]);


    status = EG_evaluate(epcurve, &wrep->edge[iedge].tend, data);
    CHECK_STATUS(EG_evaluate);

    tang[0] = data[2] / sqrt(data[2] * data[2] + data[3] * data[3]);
    tang[1] = data[3] / sqrt(data[2] * data[2] + data[3] * data[3]);

    status = EG_invEvaluate(wrep->face[iface].esurface, wrep->node[offset->iend].xyz, uv_out, xyz_out);
    CHECK_STATUS(EG_invEvaluate);

    duv_end[0] = +tang[0] * (uv_out[0] - uv1[2*npnt-2]) + tang[1] * (uv_out[1] - uv1[2*npnt-1]);
    duv_end[1] = -tang[1] * (uv_out[0] - uv1[2*npnt-2]) + tang[0] * (uv_out[1] - uv1[2*npnt-1]);

    /* straight pcurve (for now) */
    for (ipnt = 0; ipnt < npnt; ipnt++) {
        frac   = (double)(ipnt) / (double)(npnt-1);
        tt     = (1-frac) * wrep->edge[iedge].tbeg + frac * wrep->edge[iedge].tend;
        duv[0] = (1-frac) * duv_beg[0]             + frac * duv_end[0];
        duv[1] = (1-frac) * duv_beg[1]             + frac * duv_end[1];

        status = EG_evaluate(epcurve, &tt, data);
        CHECK_STATUS(EG_evaluate);

        tang[0] = data[2] / sqrt(data[2] * data[2] + data[3] * data[3]);
        tang[1] = data[3] / sqrt(data[2] * data[2] + data[3] * data[3]);

        uv2[3*ipnt  ] = uv1[2*ipnt  ] + tang[0] * duv[0] - tang[1] * duv[1];
        uv2[3*ipnt+1] = uv1[2*ipnt+1] + tang[1] * duv[0] + tang[0] * duv[1];
        uv2[3*ipnt+2] = 0;
    }

#ifdef GRAFIC
    if (0) {
        int      io_kbd=5, io_scr=6, indgr=1+2+4+16+64;
        int      nline=0, nplot=0, ilin[2], isym[2], nper[2];
        float    uplot[200], vplot[200];
        char     pltitl[80];

        for (ipnt = 0; ipnt < npnt; ipnt++) {
            uplot[nplot] = uv1[2*ipnt  ];
            vplot[nplot] = uv1[2*ipnt+1];
            nplot++;
        }
        ilin[nline] = GR_SOLID;
        isym[nline] = GR_SQUARE;
        nper[nline] = npnt;
        nline++;

        for (ipnt = 0; ipnt < npnt; ipnt++) {
            uplot[nplot] = uv2[3*ipnt  ];
            vplot[nplot] = uv2[3*ipnt+1];
            nplot++;
        }
        ilin[nline] = GR_SOLID;
        isym[nline] = GR_PLUS;
        nper[nline] = npnt;
        nline++;

        snprintf(pltitl, 79, "udfScribe");
        grinit_(&io_kbd, &io_scr, pltitl, STRLEN(pltitl));

        snprintf(pltitl, 79, "~u~v~makeOffsetCurve: iface=%d, iedge=%d", iface, iedge);
        grline_(ilin, isym, &nline, pltitl, &indgr, uplot, vplot, nper, STRLEN(pltitl));


    }
#endif

    /* generate the coordinates that are a given offset distance from the Wedge */
    for (ipnt = 0; ipnt < npnt; ipnt++) {

        status = EG_evaluate(wrep->face[iface].esurface, &uv2[3*ipnt], data1);
        CHECK_STATUS(EG_evaluate);

        /* this is needed to get tbeg and tend correct below */
        xyz[3*ipnt  ] = data1[0];
        xyz[3*ipnt+1] = data1[1];
        xyz[3*ipnt+2] = data1[2];
    }

    /* generate the EGADS curve */
    status = EG_getContext(wrep->face[iface].esurface, &context);
    CHECK_STATUS(EG_getContext);

    sizes[0] = npnt;
    sizes[1] = 0;
    status = EG_approximate(context, 0, EPS06, sizes, uv2, &etemp);
    CHECK_STATUS(EG_approximate);

    /* convert etemp (which is a BSPLINE Curve into a BSPLINE Pcurve) */
    status = EG_getGeometry(etemp, &oclass, &mtype, &eref, &header, &gdata);
    CHECK_STATUS(EG_getGeometry);

    if (oclass != CURVE || mtype != BSPLINE) {
        snprintf(message, 1024, "etemp: oclass=%d, mtype=%d", oclass, mtype);
        status = OCSM_INTERNAL_ERROR;
        goto cleanup;
    }

    SPLINT_CHECK_FOR_NULL(header);
    SPLINT_CHECK_FOR_NULL(gdata );

    status = EG_deleteObject(etemp);
    CHECK_STATUS(EG_deleteObject);

    for (i = 0; i < header[2]; i++) {
        gdata[header[3]+2*i  ] = gdata[header[3]+3*i  ];
        gdata[header[3]+2*i+1] = gdata[header[3]+3*i+1];
    }

    status = EG_makeGeometry(context, PCURVE, BSPLINE, eref, header, gdata, &epcurve);
    CHECK_STATUS(EG_makeGeometry);

    EG_free(header);
    EG_free(gdata );

    /* convert to EGADS curve */
    status = EG_otherCurve(wrep->face[iface].esurface, epcurve, toler, &(offset->ecurve));
    CHECK_STATUS(EG_otherCurve);

cleanup:
    FREE(uv1);
    FREE(uv2);
    FREE(xyz);

#ifdef DEBUG
    printf("exit makeOffsetCurve -> status=%d\n", status);
#endif

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   wrepInit - initialize a WREP structure from an EGADS Body          *
 *                                                                      *
 ************************************************************************
 */

static int
wrepInit(ego    ebody,                  /* (in)  Body to transform */
         wrep_T **wrep)                 /* (out) pointer to new WREP structure (freeable) */
{
    int     status = EGADS_SUCCESS;     /* (out) return status */

    int     oclass, mtype, nchild, ichild, *senses;
    int     inode, iedge, jedge, kedge, iface, nloop, iloop;
    double  data[118];
    ego     *enodes=NULL, *eedges=NULL, *efaces=NULL, eref, esurface, *echilds;
    ego     *eloops;
    wrep_T  *newWrep=NULL;

    ROUTINE(wrepInit);

    /* --------------------------------------------------------------- */

    *wrep = NULL;

    /* set up initial data structure */
    MALLOC(newWrep, wrep_T, 1);

    newWrep->node = NULL;
    newWrep->edge = NULL;
    newWrep->face = NULL;
    newWrep->ebody = ebody;

    status = EG_getBodyTopos(ebody, NULL, NODE, &(newWrep->nnode), &enodes);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(ebody, NULL, EDGE, &(newWrep->nedge), &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(ebody, NULL, FACE, &(newWrep->nface), &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    MALLOC(newWrep->node, wnode_T, newWrep->nnode+1);
    MALLOC(newWrep->edge, wedge_T, newWrep->nedge+1);
    MALLOC(newWrep->face, wface_T, newWrep->nface+1);

    /* fill in the Wnode data structure */
    for (inode = 1; inode <= newWrep->nnode; inode++) {
        SPLINT_CHECK_FOR_NULL(enodes);

        status = EG_getTopology(enodes[inode-1], &eref, &oclass, &mtype,
                                data, &nchild, &echilds, &senses);
        CHECK_STATUS(EG_getTopology);

        newWrep->node[inode].xyz[0] = data[0];
        newWrep->node[inode].xyz[1] = data[1];
        newWrep->node[inode].xyz[2] = data[2];
        newWrep->node[inode].nedge  = 0;
        newWrep->node[inode].enode  = enodes[inode-1];
    }

    /* fill in the Wedge data structure */
    for (iedge = 1; iedge <= newWrep->nedge; iedge++) {
        SPLINT_CHECK_FOR_NULL(eedges);

        status = EG_getTopology(eedges[iedge-1], &eref, &oclass, &mtype,
                                data, &nchild, &echilds, &senses);
        CHECK_STATUS(EG_getTopology);

        newWrep->edge[iedge].tag    = 0;
        newWrep->edge[iedge].ibeg   = 0;
        newWrep->edge[iedge].iend   = 0;
        newWrep->edge[iedge].ileft  = 0;
        newWrep->edge[iedge].irite  = 0;
        newWrep->edge[iedge].ibleft = 0;
        newWrep->edge[iedge].ibrite = 0;
        newWrep->edge[iedge].ieleft = 0;
        newWrep->edge[iedge].ierite = 0;
        newWrep->edge[iedge].tbeg   = data[0];
        newWrep->edge[iedge].tend   = data[1];
        newWrep->edge[iedge].ecurve = eref;
        newWrep->edge[iedge].epleft = NULL;
        newWrep->edge[iedge].eprite = NULL;
        newWrep->edge[iedge].eedge  = eedges[iedge-1];

        for (inode = 1; inode <= newWrep->nnode; inode++) {
            if (echilds[0] == newWrep->node[inode].enode) {
                newWrep->edge[iedge].ibeg = inode;
                newWrep->node[inode].nedge++;
            }
            if (echilds[1] == newWrep->node[inode].enode) {
                newWrep->edge[iedge].iend = inode;
                newWrep->node[inode].nedge++;
            }
        }
    }

    /* fill in the Wface data structure */
    for (iface = 1; iface <= newWrep->nface; iface++) {
        SPLINT_CHECK_FOR_NULL(efaces);

        status = EG_getTopology(efaces[iface-1], &esurface, &oclass, &mtype,
                                data, &nloop, &eloops, &senses);
        CHECK_STATUS(EG_getTopology);

        newWrep->face[iface].tag      = 0;
        newWrep->face[iface].mtype    = mtype;
        newWrep->face[iface].esurface = esurface;
        newWrep->face[iface].eface    = efaces[iface-1];

        for (iloop = 0; iloop < nloop; iloop++) {
            status = EG_getTopology(eloops[iloop], &eref, &oclass, &mtype,
                                    data, &nchild, &echilds, &senses);
            CHECK_STATUS(EG_getTopology);

            for (ichild = 0; ichild < nchild; ichild++) {
                iedge = status = EG_indexBodyTopo(ebody, echilds[ichild]);
                CHECK_STATUS(EG_indexBodyTopo);

                jedge = status = EG_indexBodyTopo(ebody, echilds[(ichild-1+nchild)%nchild]);
                CHECK_STATUS(EG_indexBodyTopo);

                kedge = status = EG_indexBodyTopo(ebody, echilds[(ichild+1+nchild)%nchild]);
                CHECK_STATUS(EG_indexBodyTopo);

                if (senses[ichild] == SFORWARD) {
                    newWrep->edge[iedge].ileft  = iface;
                    newWrep->edge[iedge].ibleft = jedge;
                    newWrep->edge[iedge].ieleft = kedge;
                    if (eref != NULL) {
                        newWrep->edge[iedge].epleft = echilds[ichild+nchild];
                    }
                } else {
                    newWrep->edge[iedge].irite  = iface;
                    newWrep->edge[iedge].ibrite = kedge;
                    newWrep->edge[iedge].ierite = jedge;
                    if (eref != NULL) {
                        newWrep->edge[iedge].eprite = echilds[ichild+nchild];
                    }
                }

            }
        }
    }

    newWrep->nface_orig = newWrep->nface;

    *wrep = newWrep;

cleanup:
    if (enodes != NULL) EG_free(enodes);
    if (eedges != NULL) EG_free(eedges);
    if (efaces != NULL) EG_free(efaces);

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   wrepBreakEdge - break a given Wedge at the given t                 *
 *                                                                      *
 ************************************************************************
 */

static int
wrepBreakEdge(wrep_T  *wrep,            /* (in)  pointer to a WREP structure */
              int     iedge,            /* (in)  Wedge to break */
              double  t,                /* (in)  parametric coordinate at split */
              int     *jnode)           /* (out) Wnode at the break */
{
    int     status = EGADS_SUCCESS;     /* (out) <0 return status */
                                        /* (out) >0 is Wedge index */

    int     jedge, kedge, oclass, mtype;
    double  data[18];
    ego     context, enodes[2], topRef, prev, next;
    void    *realloc_temp = NULL;            /* used by RALLOC macro */

    ROUTINE(wrepBreakEdge);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("enter wrepBreakEdge(iedge=%d, t=%10.5f)\n", iedge, t);
#endif

    /* try to adjust t to put it between tbeg and tend if a CIRCLE */
    status = EG_getInfo(wrep->edge[iedge].ecurve, &oclass, &mtype, &topRef, &prev, &next);
    CHECK_STATUS(EG_getInfo);

    if (mtype == CIRCLE) {
        if        (t-TWOPI >= wrep->edge[iedge].tbeg-EPS06 && t-TWOPI <= wrep->edge[iedge].tend+EPS06) {
            t -= TWOPI;
        } else if (t+TWOPI >= wrep->edge[iedge].tbeg-EPS06 && t+TWOPI <= wrep->edge[iedge].tend+EPS06) {
            t += TWOPI;
        }
    }

    /* return immediately if break is at end of Wedge */
    if        (fabs(t-wrep->edge[iedge].tbeg) < EPS06) {
        *jnode = wrep->edge[iedge].ibeg;
        status = iedge;
        goto cleanup;
    } else if (fabs(t-wrep->edge[iedge].tend) < EPS06) {
        *jnode = wrep->edge[iedge].iend;
        status = iedge;
        goto cleanup;
    }

    /* if t < tbeg, see if there is an adjoining Wedge that shares a curve with iedge */
    if (t < wrep->edge[iedge].tbeg) {
        if        (wrep->edge[wrep->edge[iedge].ibleft].ecurve == wrep->edge[iedge].ecurve) {
            iedge = wrep->edge[iedge].ibleft;
        } else if (wrep->edge[wrep->edge[iedge].ibrite].ecurve == wrep->edge[iedge].ecurve) {
            iedge = wrep->edge[iedge].ibrite;
        } else {
            snprintf(message, 1024, "cannot find adjoining Wedge that shares a curve with iedge=%d", iedge);
            status = OCSM_INTERNAL_ERROR;
            goto cleanup;
        }
    }

    /* if t > tend, see if there is an adjoinig Wedge that shares a curve with iedge */
    if (t > wrep->edge[iedge].tend) {
        if        (wrep->edge[wrep->edge[iedge].ieleft].ecurve == wrep->edge[iedge].ecurve) {
            iedge = wrep->edge[iedge].ieleft;
        } else if (wrep->edge[wrep->edge[iedge].ierite].ecurve == wrep->edge[iedge].ecurve) {
            iedge = wrep->edge[iedge].ierite;
        } else {
            snprintf(message, 1024, "cannot find adjoining Wedge that shares a curve with iedge=%d", iedge);
            status = OCSM_INTERNAL_ERROR;
            goto cleanup;
        }
    }

    status = EG_getContext(wrep->edge[iedge].eedge, &context);
    CHECK_STATUS(EG_getContext);

    /* get the coordinates of the new Wnode */
    status = EG_evaluate(wrep->edge[iedge].eedge, &t, data);
    CHECK_STATUS(EG_evaluate);

    /* add a Wnode */
    RALLOC(wrep->node, wnode_T, wrep->nnode+2);

    *jnode = wrep->nnode + 1;

    wrep->node[*jnode].xyz[0] = data[0];
    wrep->node[*jnode].xyz[1] = data[1];
    wrep->node[*jnode].xyz[2] = data[2];
    wrep->node[*jnode].nedge  = 2;
    wrep->node[*jnode].enode  = NULL;

    status = EG_makeTopology(context, NULL, NODE, 0,
                             data, 0, NULL, NULL, &(wrep->node[*jnode].enode));
    CHECK_STATUS(EG_makeTopology);

    (wrep->nnode)++;

    /* add a Wedge (the second part of iedge) */
    RALLOC(wrep->edge, wedge_T, wrep->nedge+2);

    jedge = wrep->nedge + 1;

    wrep->edge[jedge].tag    = wrep->edge[iedge].tag;
    wrep->edge[jedge].ibeg   = *jnode;
    wrep->edge[jedge].iend   = wrep->edge[iedge].iend;
    wrep->edge[jedge].ileft  = wrep->edge[iedge].ileft;
    wrep->edge[jedge].irite  = wrep->edge[iedge].irite;
    wrep->edge[jedge].ibleft = iedge;
    wrep->edge[jedge].ibrite = iedge;
    wrep->edge[jedge].ieleft = wrep->edge[iedge].ieleft;
    wrep->edge[jedge].ierite = wrep->edge[iedge].ierite;
    wrep->edge[jedge].tbeg   = t;
    wrep->edge[jedge].tend   = wrep->edge[iedge].tend;
    wrep->edge[jedge].ecurve = wrep->edge[iedge].ecurve;
    wrep->edge[jedge].epleft = wrep->edge[iedge].epleft;
    wrep->edge[jedge].eprite = wrep->edge[iedge].eprite;
    wrep->edge[jedge].eedge  = wrep->edge[iedge].eedge;

    data[0] = wrep->edge[jedge].tbeg;
    data[1] = wrep->edge[jedge].tend;

    enodes[0] = wrep->node[wrep->edge[jedge].ibeg].enode;
    enodes[1] = wrep->node[wrep->edge[jedge].iend].enode;

    status = EG_makeTopology(context, wrep->edge[jedge].ecurve, EDGE, TWONODE,
                             data, 2, enodes, NULL, &(wrep->edge[jedge].eedge));
    CHECK_STATUS(EG_makeTopology);

    (wrep->nedge)++;

    /* modify iedge (to be the first part of the original iedge) */
    wrep->edge[iedge].iend   = *jnode;
    wrep->edge[iedge].ieleft = jedge;
    wrep->edge[iedge].ierite = jedge;
    wrep->edge[iedge].tend   = t;

    data[0] = wrep->edge[iedge].tbeg;
    data[1] = wrep->edge[iedge].tend;

    enodes[0] = wrep->node[wrep->edge[iedge].ibeg].enode;
    enodes[1] = wrep->node[wrep->edge[iedge].iend].enode;

    status = EG_makeTopology(context, wrep->edge[iedge].ecurve, EDGE, TWONODE,
                             data, 2, enodes, NULL, &(wrep->edge[iedge].eedge));
    CHECK_STATUS(EG_makeTopology);

    /* modify Wedges that were incident on the end of iedge */
    kedge = wrep->edge[jedge].ieleft;
    if (wrep->edge[kedge].ibleft == iedge) {
        wrep->edge[kedge].ibleft =  jedge;
    }
    if (wrep->edge[kedge].ierite == iedge) {
        wrep->edge[kedge].ierite =  jedge;
    }

    kedge = wrep->edge[jedge].ierite;
    if (wrep->edge[kedge].ieleft == iedge) {
        wrep->edge[kedge].ieleft =  jedge;
    }
    if (wrep->edge[kedge].ibrite == iedge) {
        wrep->edge[kedge].ibrite =  jedge;
    }

    status = jedge;

cleanup:
#ifdef DEBUG
    printf("exit wrepBreakEdge -> status=%d, jnode=%d\n", status, *jnode);
#endif

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   wrepMakeEdge - build a new Wedge between Wnodes                    *
 *                                                                      *
 ************************************************************************
 */

static int
wrepMakeEdge(wrep_T  *wrep,             /* (in)  pointer to a WREP structure */
             int     iface,             /* (in)  Wface on which Wedge will reside */
             int     ibeg,              /* (in)  Wnode at begining of Wedge */
             int     iend,              /* (in)  Wnode at end      of Wedge */
   /*@null@*/ego     ecurvei)           /* (in)  ecurve for new Wedge */
{
    int     status = EGADS_SUCCESS;     /* (out) <0 return status */
                                        /* (out) >0 is Wedge index */

    int     oclass, mtype, knode, jedge, kedge, jface, kface, used;
    double  data[18], tout[2], uvout[4], xyzout[3];
    ego     ecurve, context, topRef, prev, next, epline, epleft, eprite;
    void    *realloc_temp = NULL;            /* used by RALLOC macro */

    ROUTINE(wrepMakeEdge);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("enter wrepMakeEdge(iface=%d, ibeg=%d, iend=%d, ecurvei=%llx)\n", iface, ibeg, iend, (long long)ecurvei);
#endif

    /* if ecurve is not given, create a new curve (straight trimmed pcurve) between ibeg and iend */
    if (ecurvei == NULL) {
        status = EG_getContext(wrep->face[iface].esurface, &context);
        CHECK_STATUS(EG_getContext);

        status = EG_invEvaluate(wrep->face[iface].esurface, wrep->node[ibeg].xyz, &uvout[0], xyzout);
        CHECK_STATUS(EG_invEvaluate);

        status = EG_invEvaluate(wrep->face[iface].esurface, wrep->node[iend].xyz, &uvout[2], xyzout);
        CHECK_STATUS(EG_invEvaluate);

        uvout[2] -= uvout[0];
        uvout[3] -= uvout[1];

        status = EG_makeGeometry(context, PCURVE, LINE, NULL, NULL, uvout, &epline);
        CHECK_STATUS(EG_makeGeometry);

        data[0] = uvout[0];
        data[1] = uvout[1];
        status = EG_invEvaluate(epline, data, &tout[0], xyzout);
        CHECK_STATUS(EG_invEvaluate);

        data[0] = uvout[2] + uvout[0];
        data[1] = uvout[3] + uvout[1];
        status = EG_invEvaluate(epline, data, &tout[1], xyzout);
        CHECK_STATUS(EG_invEvaluate);

        status = EG_makeGeometry(context, PCURVE, TRIMMED, epline, NULL, tout, &epleft);
        CHECK_STATUS(EG_makeGeometry);

        status = EG_otherCurve(wrep->face[iface].esurface, epleft, 0.0, &ecurve);
        CHECK_STATUS(EG_otherCurve);

        status = EG_deleteObject(epleft);
        CHECK_STATUS(EG_deleteObject);
    } else {
        ecurve = ecurvei;
    }

    status = EG_getInfo(wrep->face[iface].esurface, &oclass, &mtype, &topRef, &prev, &next);
    CHECK_STATUS(EG_getInfo);

    SPLINT_CHECK_FOR_NULL(ecurve);

    if (mtype == PLANE) {
        epleft = NULL;
        eprite = NULL;
    } else {
        status = EG_otherCurve(wrep->face[iface].esurface, ecurve, 0.0, &epleft);
        if (status == EGADS_CONSTERR) {
            snprintf(message, 1024, "perhaps the offset is not contained to the Faces in facelist");
            status = OCSM_UDP_ERROR2;
            goto cleanup;
        }
        CHECK_STATUS(EG_otherCurve);

        eprite = epleft;
    }

    /* create a new Wedge between ibeg and iend */
    RALLOC(wrep->edge, wedge_T, wrep->nedge+2);

    jedge = wrep->nedge + 1;

    wrep->edge[jedge].tag    = 0;
    wrep->edge[jedge].ibeg   = ibeg;
    wrep->edge[jedge].iend   = iend;
    wrep->edge[jedge].ileft  = 0;                 // corrected below
    wrep->edge[jedge].irite  = 0;                 // corrected below
    wrep->edge[jedge].ibleft = 0;                 // corrected below
    wrep->edge[jedge].ibrite = 0;                 // corrected below
    wrep->edge[jedge].ieleft = 0;                 // corrected below
    wrep->edge[jedge].ierite = 0;                 // corrected below
    wrep->edge[jedge].tbeg   = 0;                 // corrected below
    wrep->edge[jedge].tend   = 0;                 // corrected below
    wrep->edge[jedge].ecurve = ecurve;
    /*@-kepttrans@*/
    wrep->edge[jedge].epleft = epleft;
    wrep->edge[jedge].eprite = eprite;
    /*@+kepttrans@*/
    wrep->edge[jedge].eedge  = NULL;

    status = EG_invEvaluate(ecurve, wrep->node[ibeg].xyz, tout, xyzout);
    CHECK_STATUS(EG_invEvaluate);
    wrep->edge[jedge].tbeg = tout[0];

    status = EG_invEvaluate(ecurve, wrep->node[iend].xyz, tout, xyzout);
    CHECK_STATUS(EG_invEvaluate);
    wrep->edge[jedge].tend = tout[0];

    (wrep->nedge)++;

    /* attaching to an isolated Wnode */
    if        (wrep->node[ibeg].nedge < 1) {
        wrep->edge[jedge].ileft = iface;
        wrep->edge[jedge].irite = iface;
    } else if (wrep->node[iend].nedge < 1) {
        wrep->edge[jedge].ileft = iface;
        wrep->edge[jedge].irite = iface;

    /* if ibeg and iend have a common Wface, set the ileft and irite of the new Wedge.
       note that iface is favored */
    } else {
        for (kface = 1; kface <= wrep->nface; kface++) {
            used = 0;
            for (kedge = 1; kedge < jedge; kedge++) {
                if (wrep->edge[kedge].ibeg == ibeg || wrep->edge[kedge].iend == ibeg) {
                    if (wrep->edge[kedge].ileft == kface || wrep->edge[kedge].irite == kface) {
                        used = 1;
                        break;
                    }
                }
            }
            if (used == 0) continue;

            used = 0;
            for (kedge = 1; kedge < jedge; kedge++) {
                if (wrep->edge[kedge].ibeg == iend || wrep->edge[kedge].iend == iend) {
                    if (wrep->edge[kedge].ileft == kface || wrep->edge[kedge].irite == kface) {
                        used = 1;
                        break;
                    }
                }
            }
            if (used == 0) continue;

            if (kface == iface) {
                wrep->edge[jedge].ileft = kface;
                wrep->edge[jedge].irite = kface;
                break;
            } else if (wrep->edge[jedge].ileft == 0 && wrep->edge[jedge].irite == 0) {
                wrep->edge[jedge].ileft = kface;
                wrep->edge[jedge].irite = kface;
            }
        }
    }

    if (wrep->edge[jedge].ileft == 0 || wrep->edge[jedge].irite == 0) {
        wrepPrint(wrep);
        snprintf(message, 1024, "eithe rileft or irite of jedge=%d is zero", jedge);
        status = OCSM_INTERNAL_ERROR;
        goto cleanup;
    }

    /* determine the ileft and irite for the new Wedge and change
       the links at ibeg and iend to point to the new Wedge */
    for (kedge = 1; kedge < jedge; kedge++) {

        /* attaching to kedge, which has a Wnode with .nedge==1

           A <--             --> C
                 ---------->
           B -->    jedge    <-- D

         */

        /* case A */
        if (wrep->edge[kedge].ibeg == ibeg && wrep->edge[kedge].ibleft == 0
                                           && wrep->edge[kedge].ibrite == 0   ) {
#ifdef DEBUG
            printf("case A: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].ileft = wrep->edge[kedge].ileft;
            wrep->edge[jedge].irite = wrep->edge[kedge].irite;

            wrep->edge[kedge].ibleft = jedge;
            wrep->edge[kedge].ibrite = jedge;
            wrep->edge[jedge].ibleft = kedge;
            wrep->edge[jedge].ibrite = kedge;
        }

        /* case B */
        if (wrep->edge[kedge].iend == ibeg && wrep->edge[kedge].ieleft == 0
                                           && wrep->edge[kedge].ierite == 0   ) {
#ifdef DEBUG
            printf("case B: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].ileft = wrep->edge[kedge].ileft;
            wrep->edge[jedge].irite = wrep->edge[kedge].irite;

            wrep->edge[kedge].ieleft = jedge;
            wrep->edge[kedge].ierite = jedge;
            wrep->edge[jedge].ibleft = kedge;
            wrep->edge[jedge].ibrite = kedge;
        }

        /* case C */
        if (wrep->edge[kedge].ibeg == iend && wrep->edge[kedge].ibleft == 0
                                           && wrep->edge[kedge].ibrite == 0   ) {
#ifdef DEBUG
            printf("case C: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].ileft = wrep->edge[kedge].ileft;
            wrep->edge[jedge].irite = wrep->edge[kedge].irite;

            wrep->edge[kedge].ibleft = jedge;
            wrep->edge[kedge].ibrite = jedge;
            wrep->edge[jedge].ieleft = kedge;
            wrep->edge[jedge].ierite = kedge;
        }

        /* case D */
        if (wrep->edge[kedge].iend == iend && wrep->edge[kedge].ieleft == 0
                                           && wrep->edge[kedge].ierite == 0   ) {
#ifdef DEBUG
            printf("case D: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].ileft = wrep->edge[kedge].ileft;
            wrep->edge[jedge].irite = wrep->edge[kedge].irite;

            wrep->edge[kedge].ieleft = jedge;
            wrep->edge[kedge].ierite = jedge;
            wrep->edge[jedge].ieleft = kedge;
            wrep->edge[jedge].ierite = kedge;
        }

        /* attaching to kedge, which has a Wnode with .nedge>1

                 G E                 I K
                  \ ^               ^ /
                   v \    jedge    / v
                      ------------>
                   / ^             ^ \
                  v /               \ v
                 F H                 L J
        */

        /* case E */
        if (wrep->edge[kedge].ibeg == ibeg && wrep->edge[kedge].irite == wrep->edge[jedge].ileft) {
#ifdef DEBUG
            printf("case E: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].ileft = wrep->edge[kedge].irite;

            wrep->edge[kedge].ibrite = jedge;
            wrep->edge[jedge].ibleft = kedge;
        }

        /* case F */
        if (wrep->edge[kedge].ibeg == ibeg && wrep->edge[kedge].ileft == wrep->edge[jedge].irite) {
#ifdef DEBUG
            printf("case F: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].irite = wrep->edge[kedge].ileft;

            wrep->edge[kedge].ibleft = jedge;
            wrep->edge[jedge].ibrite = kedge;
        }

        /* case G */
        if (wrep->edge[kedge].iend == ibeg && wrep->edge[kedge].ileft == wrep->edge[jedge].ileft) {
#ifdef DEBUG
            printf("case G: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].ileft = wrep->edge[kedge].ileft;

            wrep->edge[kedge].ieleft = jedge;
            wrep->edge[jedge].ibleft = kedge;
        }

        /* case H */
        if (wrep->edge[kedge].iend == ibeg && wrep->edge[kedge].irite == wrep->edge[jedge].irite) {
#ifdef DEBUG
            printf("case H: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].irite = wrep->edge[kedge].irite;

            wrep->edge[kedge].ierite = jedge;
            wrep->edge[jedge].ibrite = kedge;
        }

        /* case I */
        if (wrep->edge[kedge].ibeg == iend && wrep->edge[kedge].ileft == wrep->edge[jedge].ileft) {
#ifdef DEBUG
            printf("case I: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].ileft = wrep->edge[kedge].ileft;

            wrep->edge[kedge].ibleft = jedge;
            wrep->edge[jedge].ieleft = kedge;
        }

        /* case J */
        if (wrep->edge[kedge].ibeg == iend && wrep->edge[kedge].irite == wrep->edge[jedge].irite) {
#ifdef DEBUG
            printf("case J: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].irite = wrep->edge[kedge].irite;

            wrep->edge[kedge].ibrite = jedge;
            wrep->edge[jedge].ierite = kedge;
        }

        /* case K */
        if (wrep->edge[kedge].iend == iend && wrep->edge[kedge].irite == wrep->edge[jedge].ileft) {
#ifdef DEBUG
            printf("case K: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].ileft = wrep->edge[kedge].irite;

            wrep->edge[kedge].ierite = jedge;
            wrep->edge[jedge].ieleft = kedge;
        }

        /* case L */
        if (wrep->edge[kedge].iend == iend && wrep->edge[kedge].ileft == wrep->edge[jedge].irite) {
#ifdef DEBUG
            printf("case L: jedge=%d, kedge=%d\n", jedge, kedge);
#endif
            wrep->edge[jedge].irite = wrep->edge[kedge].ileft;

            wrep->edge[kedge].ieleft = jedge;
            wrep->edge[jedge].ierite = kedge;
        }
    }

    /* if the number of incident Wedges on both ibeg and iend are greater than
       zero, this means we have closed a loop, so create a new Wface on the right of the new Wedge */
    if (wrep->node[ibeg].nedge > 0 && wrep->node[iend].nedge > 0) {
        RALLOC(wrep->face, wface_T, wrep->nface+2);

        jface = wrep->nface + 1;
#ifdef DEBUG
        printf("creating Wface %d\n", jface);
#endif

        wrep->face[jface].tag      = 0;
        wrep->face[jface].mtype    = wrep->face[iface].mtype;
        wrep->face[jface].esurface = wrep->face[iface].esurface;
        wrep->face[jface].eface    = wrep->face[iface].eface;

        (wrep->nface)++;

        /* change the ileft and irite pointers on all Wedges associated with the new Wface */
        wrep->edge[jedge].irite = jface;

        knode = wrep->edge[jedge].iend;
        kedge = wrep->edge[jedge].ierite;
        while (kedge != jedge) {
            if (kedge == 0) {
                wrepPrint(wrep);
                snprintf(message, 1024, "trouble setting ileft/irite for kedge=%d", kedge);
                status = OCSM_INTERNAL_ERROR;
                goto cleanup;
            }

            if        (wrep->edge[kedge].ieleft == 0) {
                wrep->edge[kedge].ileft = jface;
                wrep->edge[kedge].irite = jface;
                kedge = wrep->edge[kedge].ibleft;
            } else if (wrep->edge[kedge].ibeg == knode) {
                wrep->edge[kedge].irite = jface;
                knode = wrep->edge[kedge].iend;
                kedge = wrep->edge[kedge].ierite;
            } else if (wrep->edge[kedge].iend == knode) {
                wrep->edge[kedge].ileft = jface;
                knode = wrep->edge[kedge].ibeg;
                kedge = wrep->edge[kedge].ibleft;
            } else {
                wrepPrint(wrep);
                snprintf(message, 1024, "trouble setting ileft/irite for kedge=%d", kedge);
                status = OCSM_INTERNAL_ERROR;
                goto cleanup;
            }
        }
    }

    /* increment the number of Wedges incident on the two end-points */
    wrep->node[ibeg].nedge++;
    wrep->node[iend].nedge++;

    /* make sure all info is updated */
    if (wrep->edge[jedge].ileft  == 0 ||
        wrep->edge[jedge].irite  == 0 ||
        wrep->edge[jedge].ibleft == 0 ||
        wrep->edge[jedge].ibrite == 0   ) {
        printf("jedge=%d, ileft=%d, irite=%d, ibleft=%d, ibrite=%d\n", jedge,
               wrep->edge[jedge].ileft,
               wrep->edge[jedge].irite,
               wrep->edge[jedge].ibleft,
               wrep->edge[jedge].ibrite);
        snprintf(message, 1024, "info not updated for jedge=%d", jedge);
        status = OCSM_INTERNAL_ERROR;
        goto cleanup;
    }

    status = jedge;

cleanup:
#ifdef DEBUG
    wrepPrint(wrep);
    printf("exit wrepMakeEdge -> status=%d\n", status);
#endif

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   wrepMakeNode - build a new Wnode                                   *
 *                                                                      *
 ************************************************************************
 */

static int
wrepMakeNode(wrep_T  *wrep,             /* (in)  pointer to a WREP structure */
             int     iface,             /* (in)  Wface on which Wnode should lie */
             double  xyz[])             /* (in)  coordinate of new Wnode */
{
    int     status = EGADS_SUCCESS;     /* (out) <0 return status */
                                        /* (out) >0 is Wnode index */

    int     jnode;
    double  uv_close[2], xyz_close[3];
    ego     context;
    void    *realloc_temp = NULL;            /* used by RALLOC macro */

    ROUTINE(wrepMakeNode);

    /* --------------------------------------------------------------- */

#ifdef DEBUG
    printf("enter wrepMakeNode(iface=%d, xyz=%10.5f, %10.5f %10.5f)\n",
           iface, xyz[0], xyz[1], xyz[2]);
#endif

    /* check if Wnode already exists */
    for (jnode = 1; jnode <= wrep->nnode; jnode++) {
        if (fabs(wrep->node[jnode].xyz[0] - xyz[0]) < EPS06 &&
            fabs(wrep->node[jnode].xyz[1] - xyz[1]) < EPS06 &&
            fabs(wrep->node[jnode].xyz[2] - xyz[2]) < EPS06   ) {
            status = jnode;
            goto cleanup;
        }
    }

    status = EG_getContext(wrep->face[iface].esurface, &context);
    CHECK_STATUS(EG_getContext);

    /* adjust xyz so that it actually lies on the esurface */
    status = EG_invEvaluate(wrep->face[iface].esurface, xyz, uv_close, xyz_close);
    CHECK_STATUS(EG_invEvaluate);

    /* add a Wnode */
    RALLOC(wrep->node, wnode_T, wrep->nnode+2);

    jnode = wrep->nnode + 1;

    wrep->node[jnode].xyz[0] = xyz_close[0];
    wrep->node[jnode].xyz[1] = xyz_close[1];
    wrep->node[jnode].xyz[2] = xyz_close[2];
    wrep->node[jnode].nedge  = 0;
    wrep->node[jnode].enode  = NULL;

    status = EG_makeTopology(context, NULL, NODE, 0,
                             xyz_close, 0, NULL, NULL, &(wrep->node[jnode].enode));
    CHECK_STATUS(EG_makeTopology);

    (wrep->nnode)++;

    status = jnode;

cleanup:
#ifdef DEBUG
    printf("exit wrepMakeNode -> status=%d\n", status);
#endif

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   wrep2ego - convert a wrep into an EGADS Body                       *
 *                                                                      *
 ************************************************************************
 */

static int
wrep2ego(wrep_T  wrep,                  /* (in)  pointer to a WREP structure */
         ego     context,               /* (in)  EGADS context for Body */
         ego     *ebody)                /* (out) EGADS Body that was created */
{
    int     status = EGADS_SUCCESS;     /* (out) return status */

    int     inode, iedge, nedge, iface, nloop, iloop, jloop, ibeg, oclass, mtype;
    int     *senses=NULL, *iedges=NULL, *iused=NULL;
    double  data[18];
    ego     enodes[2], *eedges=NULL, *eloops=NULL, *efaces=NULL, eshell;
    ego     oldEdge, topRef, prev, next;

    ROUTINE(wrep2ego);

    /* --------------------------------------------------------------- */

    *ebody = NULL;

    /* build the Nodes */
    for (inode = 1; inode <= wrep.nnode; inode++) {
        status = EG_makeTopology(context, NULL, NODE, 0,
                                 wrep.node[inode].xyz, 0, NULL, NULL, &(wrep.node[inode].enode));
        CHECK_STATUS(EG_makeTopology);
    }

    /* build the Edges */
    for (iedge = 1; iedge <= wrep.nedge; iedge++) {
#ifdef DEBUG
        printf("building Edge %3d\n", iedge);
#endif
        oldEdge = wrep.edge[iedge].eedge;

        /* now make the Edge */
        data[0] = wrep.edge[iedge].tbeg;
        data[1] = wrep.edge[iedge].tend;

        enodes[0] = wrep.node[wrep.edge[iedge].ibeg].enode;
        enodes[1] = wrep.node[wrep.edge[iedge].iend].enode;

        if (wrep.edge[iedge].ecurve != NULL) {
            status = EG_makeTopology(context, wrep.edge[iedge].ecurve, EDGE, TWONODE,
                                     data, 2, enodes, NULL, &(wrep.edge[iedge].eedge));
            CHECK_STATUS(EG_makeTopology);
        } else {
            status = EG_makeTopology(context, NULL, EDGE, DEGENERATE,
                                     data, 1, enodes, NULL, &(wrep.edge[iedge].eedge));
            CHECK_STATUS(EG_makeTopology);
        }

        if (oldEdge != NULL) {
            status = EG_attributeDup(oldEdge, wrep.edge[iedge].eedge);
            CHECK_STATUS(EG_attributeDup);
        }
    }

    /* allocate temporary memory (to hold egos) */
    MALLOC(eloops, ego,   wrep.nedge+1);     // perhaps too big
    MALLOC(eedges, ego, 2*wrep.nedge+1);     // perhaps too big
    MALLOC(senses, int, 2*wrep.nedge+1);     // perhaps too big
    MALLOC(iedges, int,   wrep.nedge+1);     // perhaps too big
    MALLOC(efaces, ego,   wrep.nface+1);
    MALLOC(iused,  int,   wrep.nedge+1);

    /* build the Faces */
    for (iface = 1; iface <= wrep.nface; iface++) {
#ifdef DEBUG
        printf("building Face %3d\n", iface);
#endif
        nloop = 0;

        for (iedge = 1; iedge <= wrep.nedge; iedge++) {
            iused[iedge] = 0;
        }

        /* build the Loops in this Face */
        for (iloop = 0; iloop < wrep.nedge; iloop++) {
#ifdef DEBUG
            printf("    adding Loop %3d\n", iloop+1);
#endif
            ibeg      = 0;
            nedge     = 0;
            senses[0] = 0;

            /* find a non-degenerate Edge associated with this Face */
            for (iedge = 1; iedge <= wrep.nedge; iedge++) {
                if (iused[iedge] != 0) continue;
                if (wrep.edge[iedge].ibeg == wrep.edge[iedge].iend) continue;

                if        (wrep.edge[iedge].ileft == iface) {
#ifdef DEBUG
                    printf("        adding Edge %3d\n", iedge);
#endif
                    iedges[nedge] = iedge;
                    eedges[nedge] = wrep.edge[iedge].eedge;
                    senses[nedge] = SFORWARD;
                    nedge++;

                    iused[iedge] = 1;

                    ibeg  = iedge;
                    iedge = wrep.edge[iedge].ieleft;
                    break;
                } else if (wrep.edge[iedge].irite == iface) {
#ifdef DEBUG
                    printf("        adding Edge %3d\n", iedge);
#endif
                    iedges[nedge] = iedge;
                    eedges[nedge] = wrep.edge[iedge].eedge;
                    senses[nedge] = SREVERSE;
                    nedge++;

                    iused[iedge] = 1;

                    ibeg  = iedge;
                    iedge = wrep.edge[iedge].ibrite;
                    break;
                }
            }

            if (senses[0] == 0) break;            /* no Edge found */
            if (ibeg      == 0) break;

            /* keep adding Edges to this Loop until we get back to the beginning */
            while (iedge != ibeg) {
                if (wrep.edge[iedge].ileft == iface) {
#ifdef DEBUG
                    printf("        adding Edge %3d\n", iedge);
#endif
                    iedges[nedge] = iedge;
                    eedges[nedge] = wrep.edge[iedge].eedge;
                    senses[nedge] = SFORWARD;
                    nedge++;

                    iused[iedge] = 1;

                    iedge = wrep.edge[iedge].ieleft;
                } else if (wrep.edge[iedge].irite == iface) {
#ifdef DEBUG
                    printf("        adding Edge %3d\n", iedge);
#endif
                    iedges[nedge] = iedge;
                    eedges[nedge] = wrep.edge[iedge].eedge;
                    senses[nedge] = SREVERSE;
                    nedge++;

                    iused[iedge] = 1;

                    iedge = wrep.edge[iedge].ibrite;
                } else {
                    snprintf(message, 1024, "having trouble traversing the Loop");
                    status = OCSM_INTERNAL_ERROR;
                    goto cleanup;
                }
            }

            /* we should add pcurves here */
            status = EG_getInfo(wrep.face[iface].esurface, &oclass, &mtype, &topRef, &prev, &next);
            CHECK_STATUS(EG_getInfo);

            if (mtype != PLANE) {
                for (iedge = 0; iedge < nedge; iedge++) {
                    if (senses[iedge] == SFORWARD) {
                        eedges[iedge+nedge] = wrep.edge[iedges[iedge]].epleft;
                    } else {
                        eedges[iedge+nedge] = wrep.edge[iedges[iedge]].eprite;
                    }

                    if (eedges[iedge+nedge] == NULL) {
                        status = EG_otherCurve(wrep.face[iface].esurface, eedges[iedge], 0.0, &eedges[iedge+nedge]);
                        CHECK_STATUS(EG_otherCurve);
                    }
                }
            }

            /* make the Loop */
            if (mtype != PLANE) {
                status = EG_makeTopology(context, wrep.face[iface].esurface, LOOP, CLOSED,
                                         NULL, nedge, eedges, senses, &eloops[nloop]);
                CHECK_STATUS(EG_makeTopology);
            } else {
                status = EG_makeTopology(context, NULL, LOOP, CLOSED,
                                         NULL, nedge, eedges, senses, &eloops[nloop]);
                CHECK_STATUS(EG_makeTopology);
            }
            nloop++;
        }

        /* make the Face */
        senses[0] = SFORWARD;
        for (iloop = 1; iloop < nloop; iloop++) {
            senses[iloop] = SREVERSE;
        }
        status = EG_makeTopology(context, wrep.face[iface].esurface, FACE, SFORWARD,
                                 NULL, nloop, eloops, senses, &efaces[iface-1]);

        /* if we got a construction error, it might be because we have the wrong
           Loop identified as the outer Loop */
        for (iloop = 1; iloop < nloop; iloop++) {
            if (status != EGADS_CONSTERR) break;

            prev = eloops[0];
            for (jloop = 0; jloop < nloop-1; jloop++) {
                eloops[jloop] = eloops[jloop+1];
            }
            eloops[nloop-1] = prev;

            status = EG_makeTopology(context, wrep.face[iface].esurface, FACE, SFORWARD,
                                     NULL, nloop, eloops, senses, &efaces[iface-1]);
        }
        CHECK_STATUS(EG_makeTopology);

        /* copy the Attributes from the original Faces onto the new Face */
        status = EG_attributeDup(wrep.face[iface].eface, efaces[iface-1]);
        CHECK_STATUS(EG_attributeDup);

        /* put the "scribeFace" attribute on Faces that are within
           the scribe */
        if (wrep.face[iface].new == 1) {
            status = EG_attributeAdd(efaces[iface-1], "__offsetFace__", ATTRINT, 1,
                                     &iface, NULL, NULL);
            CHECK_STATUS(EG_attributeAdd);
        }
    }

    /* build a Shell */
#ifdef DEBUG
    printf("building Shell\n");
#endif
    status = EG_makeTopology(context, NULL, SHELL, CLOSED,
                             NULL, wrep.nface, efaces, NULL, &eshell);
    CHECK_STATUS(EG_makeTopology);

    /* build the new Body */
#ifdef DEBUG
    printf("building SolidBody\n");
#endif
    status = EG_makeTopology(context, NULL, BODY, SOLIDBODY,
                             NULL, 1, &eshell, NULL, ebody);
    CHECK_STATUS(EG_makeTopology);

cleanup:
    FREE(eloops);
    FREE(eedges);
    FREE(senses);
    FREE(iedges);
    FREE(efaces);
    FREE(iused );

    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   wrepPrint - print a WREP structure                                 *
 *                                                                      *
 ************************************************************************
 */

static int
wrepPrint(wrep_T *wrep)                 /* (in)  pointer to a WREP structure */
{
    int     status = EGADS_SUCCESS;     /* (out) return status */

    int     inode, iedge, iface;

    ROUTINE(wrepPrint);

    /* --------------------------------------------------------------- */

    printf("wrep associated with ebody=%12llx\n", (long long)wrep->ebody);

    printf("inode        x               y               z         nedg    dist           enode\n");
    for (inode = 1; inode <= wrep->nnode; inode++) {
        printf("%5d %15.8f %15.8f %15.8f %5d %10.7f %12llx\n", inode,
               wrep->node[inode].xyz[0],
               wrep->node[inode].xyz[1],
               wrep->node[inode].xyz[2],
               wrep->node[inode].nedge,
               wrep->node[inode].dist,
    (long long)wrep->node[inode].enode);
    }

    printf("iedge   tag  ibeg  iend ileft irite iblft ibrit ielft ierit        tbeg            tend           ecurve       epleft       eprite        eedge\n");
    for (iedge = 1; iedge <= wrep->nedge; iedge++) {
        printf("%5d %5d %5d %5d %5d %5d %5d %5d %5d %5d %15.8f %15.8f %12llx %12llx %12llx %12llx\n", iedge,
               wrep->edge[iedge].tag,
               wrep->edge[iedge].ibeg,
               wrep->edge[iedge].iend,
               wrep->edge[iedge].ileft,
               wrep->edge[iedge].irite,
               wrep->edge[iedge].ibleft,
               wrep->edge[iedge].ibrite,
               wrep->edge[iedge].ieleft,
               wrep->edge[iedge].ierite,
               wrep->edge[iedge].tbeg,
               wrep->edge[iedge].tend,
    (long long)wrep->edge[iedge].ecurve,
    (long long)wrep->edge[iedge].epleft,
    (long long)wrep->edge[iedge].eprite,
    (long long)wrep->edge[iedge].eedge);
    }

    printf("iface   tag mtype    esurface        eface\n");
    for (iface = 1; iface <= wrep->nface; iface++) {
        printf("%5d %5d %5d %12llx %12llx\n", iface,
               wrep->face[iface].tag,
               wrep->face[iface].mtype,
    (long long)wrep->face[iface].esurface,
    (long long)wrep->face[iface].eface);
    }

//cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   wrepFree - free a WREP structure                                   *
 *                                                                      *
 ************************************************************************
 */

static int
wrepFree(wrep_T   *wrep)                /* (in)  pointer to a WREP structure */
{
    int     status = EGADS_SUCCESS;     /* (out) return status */

    ROUTINE(wrepFree);

    /* --------------------------------------------------------------- */

    if (wrep != NULL) {
        FREE(wrep->node);
        FREE(wrep->edge);
        FREE(wrep->face);

        wrep->nnode = 0;
        wrep->nedge = 0;
        wrep->nface = 0;
    }

//cleanup:
    return status;
}


/*
 ************************************************************************
 *                                                                      *
 *   makeSolidOffset2 - build offset Body for a SolidBody (method 2)    *
 *                                                                      *
 ************************************************************************
 */

static int
makeSolidOffset2(ego     ebodyIn,       /* (in)  input  Body */
                 ego     *ebodyOut,     /* (out) output Body */
     /*@unused@*/int     *NumUdp,
                 udp_T   *udps)
{
    int        status = EGADS_SUCCESS;

    int        outLevel=0, oclass, mtype, nchild, *senses, i, npnt_face, ntri_face, ntri_save;
    int        npnt, ipnt, ntri, itri, ip0, ip1, ip2, iswap, nfrozen, imin;
    int        pptype, ppindx, stat, nseg, iseg, jseg, okay, ibest;
    int        ismth, nsmth=100, jm1, count=0, mloop, nloop=0, iloop, jloop;
    int        nnode, inode, nedge, iedge, nface, iface;
    int        npoint, npoint_beg, nscribe, nremove, attrType, attrLen;
    int        nchange, ipass, nlist, ilist, status0, status1;
    int        *state=NULL, *tris=NULL, *tricn=NULL, *ibeg=NULL, *iend=NULL, *closed=NULL;
    int        *facnum=NULL, *begend=NULL, *nper=NULL, *numedge=NULL;
    CINT       *ptype, *pindx, *tris_face, *tric_face, *tempIlist;
    double     data[18], bbox[6], size, params[3], dtest, dmin, frac, alen;
    double     frac01, frac02, frac12, frac10, frac20, frac21, rswap;
    double     xa, ya, za, da, xb, yb, zb, db, dot;
    double     uvbest[2], xyzbest[3], test, dbest;
    double     *xyz=NULL, *dist=NULL, *seg=NULL, *points=NULL, *xyzsmth=NULL;
    CDOUBLE    *xyz_face, *uv_face, *tempRlist;
    CCHAR      *tempClist;
    ego        tempBody, *enodes=NULL, *eedges=NULL, *efaces=NULL, *elist=NULL;
    ego        eref, etess, esurf, *echilds;
#ifdef MAKE_PLOTFILES
    FILE       *fp;
#endif
    clock_t    old_time, new_time;
    void       *realloc_temp=NULL;              /* used by RALLOC macro */

    ROUTINE(makeSolidOffset2);

#define IFACE(I)  seg[9*(I)  ]
#define IBEG( I)  seg[9*(I)+1]
#define XBEG( I)  seg[9*(I)+2]
#define YBEG( I)  seg[9*(I)+3]
#define ZBEG( I)  seg[9*(I)+4]
#define IEND( I)  seg[9*(I)+5]
#define XEND( I)  seg[9*(I)+6]
#define YEND( I)  seg[9*(I)+7]
#define ZEND( I)  seg[9*(I)+8]

    /* --------------------------------------------------------------- */

    old_time = clock();

    outLevel = status = ocsmSetOutLevel(-1);
    CHECK_STATUS(ocsmSetOutLevel);

    /* make a copy of the Body (so that it does not get removed
     when OpenCSM deletes emodel) */
    status = EG_copyObject(ebodyIn, NULL, &tempBody);
    CHECK_STATUS(EG_copyObject);

    {
        ego  context;
        void *modl;
        modl_T *MODL;

        status = EG_getContext(tempBody, &context);
        CHECK_STATUS(EG_getContext);

        status = EG_getUserPointer(context, (void**)(&(modl)));
        CHECK_STATUS(EG_getUserPointer);

        MODL = (modl_T *)modl;

        status = EG_attributeAdd(tempBody, "filename", ATTRSTRING, 0,
                                 NULL, NULL, MODL->brch[1].filename);
        CHECK_STATUS(EG_attributeAdd);
    }

    SPLINT_CHECK_FOR_NULL(tempBody);

    /* get the Edges and Faces associated with this Body */
    status = EG_getBodyTopos(tempBody, NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(tempBody, NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(eedges);
    SPLINT_CHECK_FOR_NULL(efaces);

    /* check arguments */
    if (DIST_SIZ(0) > 1) {
        snprintf(message, 1024, "\"dist\" should be a scalar");
        status = EGADS_RANGERR;
        goto cleanup;

    } else if (DIST(0) <= 0) {
        snprintf(message, 1024, "\"dist\" should be positive");
        status = EGADS_RANGERR;
        goto cleanup;

    } else if (NODELIST_SIZ(0) > 1 || NODELIST(0,0) != 0) {
        snprintf(message, 1024, "\"nodelist\" cannot be specified for method=2");
        status = EGADS_RANGERR;
        goto cleanup;

    } else if (NODEDIST_SIZ(0) > 1 || NODEDIST(0,0) != 1) {
        snprintf(message, 1024, "\"nodedist\" cannot be specified for method=2");
        status = EGADS_RANGERR;
        goto cleanup;

    }

    for (i = 0; i < EDGELIST_SIZ(0); i++) {
        if (EDGELIST(0,i) <= 0) {
            snprintf(message, 1024, "\"edgelist[%d]\" should be positive", i);
            status = EGADS_RANGERR;
            goto cleanup;
        }
    }

    for (i = 0; i < FACELIST_SIZ(0); i++) {
        if (FACELIST(0,i) <= 0) {
            snprintf(message, 1024, "\"facelist[%d]\" should be positive", i);
            status = EGADS_RANGERR;
            goto cleanup;
        }
    }

    new_time = clock();
    SPRINT1(1, "%8.3f sec - after initialization",
            (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC));
    old_time = new_time;

    /* tessellate the Body, making sure that there are always about 4
       triangles within the specified distance */
    status = EG_getBoundingBox(tempBody, bbox);
    CHECK_STATUS(EG_getBoundingBox);

    size = sqrt(SQR(bbox[3]-bbox[0]) + SQR(bbox[4]-bbox[1]) + SQR(bbox[5]-bbox[2]));

    params[0] = TESS_PARAM_0 * size;
    params[1] = TESS_PARAM_1 * size;
    params[2] = TESS_PARAM_2;

    if (params[0] > DIST(0)/3) {
        params[0] = DIST(0) / 3;
    }

    status = EG_makeTessBody(tempBody, params, &etess);
    CHECK_STATUS(EG_makeTessBody);

    new_time = clock();
    SPRINT1(1, "%8.3f sec - after tessellation",
           (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC));
    old_time = new_time;

    /* find the number of points in the complete triangulation */
    status = EG_statusTessBody(etess, &eref, &stat, &npnt);
    CHECK_STATUS(EG_statusTessBody);

    /* find the total number of triangles in the Faces in facelist */
    ntri = 0;
    for (i = 0; i < FACELIST_SIZ(0); i++) {
        status = EG_getTessFace(etess, FACELIST(0,i),
                                &npnt_face, &xyz_face, &uv_face, &ptype, &pindx,
                                &ntri_face, &tris_face, &tric_face);
        CHECK_STATUS(EG_getTessFace);

        ntri += ntri_face;
    }

    /* allocate arrays that are needed by the level-set method below */
    MALLOC(tris,  int,    3*ntri);
    MALLOC(tricn, int,    4*ntri);
    MALLOC(xyz,   double, 3*npnt);
    MALLOC(state, int,      npnt);
    MALLOC(dist,  double,   npnt);

    ntri_save = ntri;

    /* make a tessellation which is the union of the tessellations
       of the Faces in facelist */
    ntri = 0;
    for (i = 0; i < FACELIST_SIZ(0); i++) {
        status = EG_getTessFace(etess, FACELIST(0,i),
                                &npnt_face, &xyz_face, &uv_face, &ptype, &pindx,
                                &ntri_face, &tris_face, &tric_face);
        CHECK_STATUS(EG_getTessFace);

        for (itri = 0; itri < ntri_face; itri++) {
            status = EG_localToGlobal(etess, FACELIST(0,i), tris_face[3*itri  ], &tris[3*ntri  ]);
            CHECK_STATUS(EG_localToGlobal);

            status = EG_localToGlobal(etess, FACELIST(0,i), tris_face[3*itri+1], &tris[3*ntri+1]);
            CHECK_STATUS(EG_localToGlobal);

            status = EG_localToGlobal(etess, FACELIST(0,i), tris_face[3*itri+2], &tris[3*ntri+2]);
            CHECK_STATUS(EG_localToGlobal);

            tricn[4*ntri  ] = tric_face[3*itri  ];
            tricn[4*ntri+1] = tric_face[3*itri+1];
            tricn[4*ntri+2] = tric_face[3*itri+2];
            tricn[4*ntri+3] = FACELIST(0,i);

            ntri++;
        }
    }

    new_time = clock();
    SPRINT3(1, "%8.3f sec - after assembly of global tessellation (npnt=%d, ntri=%d)",
            (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC), npnt, ntri);
    old_time = new_time;

#ifdef DEBUG
    printf("npnt=%5d\n", npnt);
    printf("ntri=%5d\n", ntri);
#endif

    /* note that this MALLOC assumes that every triangle will
       contribute a segment.  this is obviously way too big */
    MALLOC(seg, double, 9*ntri);

    /* set up the global point array and initialize all the
       points as free */
    for (ipnt = 0; ipnt < npnt; ipnt++) {
        status = EG_getGlobal(etess, ipnt+1, &pptype, &ppindx, &(xyz[3*ipnt]));
        CHECK_STATUS(EG_getGlobal);

        state[ipnt] = 0;     // free
        dist[ ipnt] = HUGEQ;
    }

    /* classify the points as frozen if adjoining one of the Edges
       in edgelist */
    for (itri = 0; itri < ntri; itri++) {
        for (i = 0; i < EDGELIST_SIZ(0); i++) {
            if (tricn[4*itri  ] == -EDGELIST(0,i)) {
                state[tris[3*itri+1]-1] = 1;
                dist[ tris[3*itri+1]-1] = 0;

                state[tris[3*itri+2]-1] = 1;
                dist[ tris[3*itri+2]-1] = 0;
            }
            if (tricn[4*itri+1] == -EDGELIST(0,i)) {
                state[tris[3*itri+2]-1] = 1;
                dist[ tris[3*itri+2]-1] = 0;

                state[tris[3*itri  ]-1] = 1;
                dist[ tris[3*itri  ]-1] = 0;
            }
            if (tricn[4*itri+2] == -EDGELIST(0,i)) {
                state[tris[3*itri  ]-1] = 1;
                dist[ tris[3*itri  ]-1] = 0;

                state[tris[3*itri+1]-1] = 1;
                dist[ tris[3*itri+1]-1] = 0;
            }
        }
    }

    /* while there still are triangles in the triangle list */
    while (ntri > 0) {
        dmin = HUGEQ;
        imin = -1;

        /* loop through the active triangles */
        for (itri = 0; itri < ntri; itri++) {
            ip0 = tris[3*itri  ] - 1;
            ip1 = tris[3*itri+1] - 1;
            ip2 = tris[3*itri+2] - 1;

            if (state[ip0] == 1 && state[ip1] == 1 && state[ip2] == 1) {
                nfrozen = 3;
            } else if (state[ip0] == 1 && state[ip1] == 1) {
                nfrozen = 2;
            } else if (state[ip1] == 1 && state[ip2] == 1) {
                nfrozen = 2;
                iswap = ip0;
                ip0   = ip1;
                ip1   = ip2;
                ip2   = iswap;
            } else if (state[ip2] == 1 && state[ip0] == 1) {
                nfrozen = 2;
                iswap = ip1;
                ip1   = ip0;
                ip0   = ip2;
                ip2   = iswap;
            } else if (state[ip0] == 1) {
                nfrozen = 1;
            } else if (state[ip1] == 1) {
                nfrozen = 1;
                iswap = ip1;
                ip1   = ip2;
                ip2   = ip0;
                ip0   = iswap;
            } else if (state[ip2] == 1) {
                nfrozen = 1;
                iswap = ip2;
                ip2   = ip1;
                ip1   = ip0;
                ip0   = iswap;
            } else {
                nfrozen = 0;
            }

            /* if triangle has 3 frozen points, remove it from the
               active triangle list */
            if (nfrozen == 3) {
                ISWAP(tris[ 3*itri  ], tris[ 3*ntri-3]);
                ISWAP(tris[ 3*itri+1], tris[ 3*ntri-2]);
                ISWAP(tris[ 3*itri+2], tris[ 3*ntri-1]);
                ISWAP(tricn[4*itri  ], tricn[4*ntri-4]);
                ISWAP(tricn[4*itri+1], tricn[4*ntri-3]);
                ISWAP(tricn[4*itri+2], tricn[4*ntri-2]);
                ISWAP(tricn[4*itri+3], tricn[4*ntri-1]);

                ntri--;
                itri--;

            /* otherwise if the triangle has 2 frozen points, find the
               distance between ip2 and the line between ip0 and ip1 */
            } else if (nfrozen == 2) {

                /*    free
                       ip2
                       / \
                      / : \
                     /  :  \
                    /   :   \
                  ip0--------ip1
                frozen     frozen
                */

                if (state[ip0] != 1 || state[ip1] != 1 || state[ip2] != 0) {
                    snprintf(message, 1024, "nfrozen=2, itri=%5d (%5d, %5d, %5d)",
                             itri, ip0, ip1, ip2);
                    status = OCSM_UDP_ERROR1;
                    goto cleanup;
                }

                frac = (  (xyz[3*ip2  ]-xyz[3*ip0  ])*(xyz[3*ip1  ]-xyz[3*ip0  ])
                        + (xyz[3*ip2+1]-xyz[3*ip0+1])*(xyz[3*ip1+1]-xyz[3*ip0+1])
                        + (xyz[3*ip2+2]-xyz[3*ip0+2])*(xyz[3*ip1+2]-xyz[3*ip0+2]))
                     / (  (xyz[3*ip1  ]-xyz[3*ip0  ])*(xyz[3*ip1  ]-xyz[3*ip0  ])
                        + (xyz[3*ip1+1]-xyz[3*ip0+1])*(xyz[3*ip1+1]-xyz[3*ip0+1])
                        + (xyz[3*ip1+2]-xyz[3*ip0+2])*(xyz[3*ip1+2]-xyz[3*ip0+2]));

                if (frac < 0) frac = 0;
                if (frac > 1) frac = 1;

                dtest = (1-frac)+dist[ip0] + frac*dist[ip1]
                      + sqrt(SQR((1-frac)*xyz[3*ip0  ] + frac*xyz[3*ip1  ] - xyz[3*ip2  ])
                            +SQR((1-frac)*xyz[3*ip0+1] + frac*xyz[3*ip1+1] - xyz[3*ip2+1])
                            +SQR((1-frac)*xyz[3*ip0+2] + frac*xyz[3*ip1+2] - xyz[3*ip2+2]));

                /* if the dtest < dist, update dist and remember if it is
                   the smallest new dist so far */
                if (dtest < dist[ip2]) {
                    dist[ip2] = dtest;
                }
                if (dtest < dmin) {
                    dmin = dtest;
                    imin = ip2;
                }

            /* otherwise if the triangle has 1 frozen point, find the
               distances between ip0 and ip1 and between ip0 and ip2 */
            } else if (nfrozen == 1) {

                /*   frozen
                       ip0
                       / \
                      /   \
                     / - - \
                    /       \
                  ip1--------ip1
                 free       free
                */

                if (state[ip0] != 1 || state[ip1] != 0 || state[ip2] != 0) {
                    snprintf(message, 1024, "nfrozen=1, itri=%5d (%5d, %5d, %5d)",
                             itri, ip0, ip1, ip2);
                    status = OCSM_UDP_ERROR1;
                    goto cleanup;
                }

                dtest = dist[ip0]
                      +sqrt(SQR(xyz[3*ip1  ]-xyz[3*ip0  ])
                           +SQR(xyz[3*ip1+1]-xyz[3*ip0+1])
                           +SQR(xyz[3*ip1+2]-xyz[3*ip0+2]));

                /* if the dtest < dist, update dist and remember if it is
                   the smallest new dist so far */
                if (dtest < dist[ip1]) {
                    dist[ip1] = dtest;
                }
                if (dtest < dmin) {
                    dmin = dtest;
                    imin = ip1;
                }

                dtest = dist[ip0]
                      + sqrt(SQR(xyz[3*ip2  ]-xyz[3*ip0  ])
                            +SQR(xyz[3*ip2+1]-xyz[3*ip0+1])
                            +SQR(xyz[3*ip2+2]-xyz[3*ip0+2]));

                /* if the dtest < d, update d and remember if it is
                   the smallest new d so far */
                if (dtest < dist[ip2]) {
                    dist[ip2] = dtest;
                }
                if (dtest < dmin) {
                    dmin = dtest;
                    imin = ip2;
                }
            }
        }

        /* freeze the distance at imin (if found) */
        if (imin >= 0) {
            state[imin] = 1;

            /* jump out if dmin is large compared with DIST */
            if (dmin > 2*DIST(0)) break;
        } else {
            break;
        }
    }

    new_time = clock();
    SPRINT1(1, "%8.3f sec - after computing distance field",
            (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC));
    old_time = new_time;

    /* find the segments that comprise the offset curve.
       seg contains the following 9 columns:
       xbeg, ybeg, zbeg, xend, yend, zend, iface, iedgebeg, iedgeend */

    /* generate a segment for every triangle with one point on the
       other side of DIST from the other two points */
    nseg = 0;
    for (itri = 0; itri < ntri_save; itri++) {
        ip0 = tris[3*itri  ] - 1;
        ip1 = tris[3*itri+1] - 1;
        ip2 = tris[3*itri+2] - 1;

        if      (dist[ip0] <= DIST(0) && dist[ip1] >  DIST(0) && dist[ip2] >  DIST(0)) {
            frac01 = (DIST(0) - dist[ip0]) / (dist[ip1] - dist[ip0]);
            frac02 = (DIST(0) - dist[ip0]) / (dist[ip2] - dist[ip0]);

            XBEG( nseg) = (1-frac01) * xyz[3*ip0  ] + frac01 * xyz[3*ip1  ];
            YBEG( nseg) = (1-frac01) * xyz[3*ip0+1] + frac01 * xyz[3*ip1+1];
            ZBEG( nseg) = (1-frac01) * xyz[3*ip0+2] + frac01 * xyz[3*ip1+2];
            XEND( nseg) = (1-frac02) * xyz[3*ip0  ] + frac02 * xyz[3*ip2  ];
            YEND( nseg) = (1-frac02) * xyz[3*ip0+1] + frac02 * xyz[3*ip2+1];
            ZEND( nseg) = (1-frac02) * xyz[3*ip0+2] + frac02 * xyz[3*ip2+2];
            IFACE(nseg) =         tricn[4*itri+3];
            IBEG( nseg) = -MIN(0, tricn[4*itri+2]);
            IEND( nseg) = -MIN(0, tricn[4*itri+1]);

            if (fabs(XEND(nseg)-XBEG(nseg)) > EPS09 ||
                fabs(YEND(nseg)-YBEG(nseg)) > EPS09 ||
                fabs(ZEND(nseg)-ZBEG(nseg)) > EPS09   ) {
                nseg++;
            } else {
                SPRINT7(0, "WARNING:: segment %3d is too short (%12.7f %12.7f %12.7f) to (%12.7f %12.7f %12.7f)\n", nseg,
                        XBEG(nseg), YBEG(nseg), ZBEG(nseg),
                        XEND(nseg), YEND(nseg), ZEND(nseg));
            }
        } else if (dist[ip0] >  DIST(0) && dist[ip1] <= DIST(0) && dist[ip2] <= DIST(0)) {
            frac01 = (DIST(0) - dist[ip0]) / (dist[ip1] - dist[ip0]);
            frac02 = (DIST(0) - dist[ip0]) / (dist[ip2] - dist[ip0]);

            XEND( nseg) = (1-frac01) * xyz[3*ip0  ] + frac01 * xyz[3*ip1  ];
            YEND( nseg) = (1-frac01) * xyz[3*ip0+1] + frac01 * xyz[3*ip1+1];
            ZEND( nseg) = (1-frac01) * xyz[3*ip0+2] + frac01 * xyz[3*ip1+2];
            XBEG( nseg) = (1-frac02) * xyz[3*ip0  ] + frac02 * xyz[3*ip2  ];
            YBEG( nseg) = (1-frac02) * xyz[3*ip0+1] + frac02 * xyz[3*ip2+1];
            ZBEG( nseg) = (1-frac02) * xyz[3*ip0+2] + frac02 * xyz[3*ip2+2];
            IFACE(nseg) =         tricn[4*itri+3];
            IEND( nseg) = -MIN(0, tricn[4*itri+2]);
            IBEG( nseg) = -MIN(0, tricn[4*itri+1]);

            if (fabs(XEND(nseg)-XBEG(nseg)) > EPS09 ||
                fabs(YEND(nseg)-YBEG(nseg)) > EPS09 ||
                fabs(ZEND(nseg)-ZBEG(nseg)) > EPS09   ) {
                nseg++;
            } else {
                SPRINT7(0, "WARNING:: segment %3d is too short (%12.7f %12.7f %12.7f) to (%12.7f %12.7f %12.7f)\n", nseg,
                        XBEG(nseg), YBEG(nseg), ZBEG(nseg),
                        XEND(nseg), YEND(nseg), ZEND(nseg));
            }
        } else if (dist[ip1] <= DIST(0) && dist[ip2] >  DIST(0) && dist[ip0] >  DIST(0)) {
            frac12 = (DIST(0) - dist[ip1]) / (dist[ip2] - dist[ip1]);
            frac10 = (DIST(0) - dist[ip1]) / (dist[ip0] - dist[ip1]);

            XBEG( nseg) = (1-frac12) * xyz[3*ip1  ] + frac12 * xyz[3*ip2  ];
            YBEG( nseg) = (1-frac12) * xyz[3*ip1+1] + frac12 * xyz[3*ip2+1];
            ZBEG( nseg) = (1-frac12) * xyz[3*ip1+2] + frac12 * xyz[3*ip2+2];
            XEND( nseg) = (1-frac10) * xyz[3*ip1  ] + frac10 * xyz[3*ip0  ];
            YEND( nseg) = (1-frac10) * xyz[3*ip1+1] + frac10 * xyz[3*ip0+1];
            ZEND( nseg) = (1-frac10) * xyz[3*ip1+2] + frac10 * xyz[3*ip0+2];
            IFACE(nseg) =         tricn[4*itri+3];
            IBEG( nseg) = -MIN(0, tricn[4*itri  ]);
            IEND( nseg) = -MIN(0, tricn[4*itri+2]);

            if (fabs(XEND(nseg)-XBEG(nseg)) > EPS09 ||
                fabs(YEND(nseg)-YBEG(nseg)) > EPS09 ||
                fabs(ZEND(nseg)-ZBEG(nseg)) > EPS09   ) {
                nseg++;
            } else {
                SPRINT7(0, "WARNING:: segment %3d is too short (%12.7f %12.7f %12.7f) to (%12.7f %12.7f %12.7f)\n", nseg,
                        XBEG(nseg), YBEG(nseg), ZBEG(nseg),
                        XEND(nseg), YEND(nseg), ZEND(nseg));
            }
        } else if (dist[ip1] >  DIST(0) && dist[ip2] <= DIST(0) && dist[ip0] <= DIST(0)) {
            frac12 = (DIST(0) - dist[ip1]) / (dist[ip2] - dist[ip1]);
            frac10 = (DIST(0) - dist[ip1]) / (dist[ip0] - dist[ip1]);

            XEND( nseg) = (1-frac12) * xyz[3*ip1  ] + frac12 * xyz[3*ip2  ];
            YEND( nseg) = (1-frac12) * xyz[3*ip1+1] + frac12 * xyz[3*ip2+1];
            ZEND( nseg) = (1-frac12) * xyz[3*ip1+2] + frac12 * xyz[3*ip2+2];
            XBEG( nseg) = (1-frac10) * xyz[3*ip1  ] + frac10 * xyz[3*ip0  ];
            YBEG( nseg) = (1-frac10) * xyz[3*ip1+1] + frac10 * xyz[3*ip0+1];
            ZBEG( nseg) = (1-frac10) * xyz[3*ip1+2] + frac10 * xyz[3*ip0+2];
            IFACE(nseg) =         tricn[4*itri+3];
            IEND( nseg) = -MIN(0, tricn[4*itri  ]);
            IBEG( nseg) = -MIN(0, tricn[4*itri+2]);

            if (fabs(XEND(nseg)-XBEG(nseg)) > EPS09 ||
                fabs(YEND(nseg)-YBEG(nseg)) > EPS09 ||
                fabs(ZEND(nseg)-ZBEG(nseg)) > EPS09   ) {
                nseg++;
            } else {
                SPRINT7(0, "WARNING:: segment %3d is too short (%12.7f %12.7f %12.7f) to (%12.7f %12.7f %12.7f)\n", nseg,
                        XBEG(nseg), YBEG(nseg), ZBEG(nseg),
                        XEND(nseg), YEND(nseg), ZEND(nseg));
            }
        } else if (dist[ip2] <= DIST(0) && dist[ip0] >  DIST(0) && dist[ip1] >  DIST(0)) {
            frac20 = (DIST(0) - dist[ip2]) / (dist[ip0] - dist[ip2]);
            frac21 = (DIST(0) - dist[ip2]) / (dist[ip1] - dist[ip2]);

            XBEG( nseg) = (1-frac20) * xyz[3*ip2  ] + frac20 * xyz[3*ip0  ];
            YBEG( nseg) = (1-frac20) * xyz[3*ip2+1] + frac20 * xyz[3*ip0+1];
            ZBEG( nseg) = (1-frac20) * xyz[3*ip2+2] + frac20 * xyz[3*ip0+2];
            XEND( nseg) = (1-frac21) * xyz[3*ip2  ] + frac21 * xyz[3*ip1  ];
            YEND( nseg) = (1-frac21) * xyz[3*ip2+1] + frac21 * xyz[3*ip1+1];
            ZEND( nseg) = (1-frac21) * xyz[3*ip2+2] + frac21 * xyz[3*ip1+2];
            IFACE(nseg) =         tricn[4*itri+3];
            IBEG( nseg) = -MIN(0, tricn[4*itri+1]);
            IEND( nseg) = -MIN(0, tricn[4*itri  ]);

            if (fabs(XEND(nseg)-XBEG(nseg)) > EPS09 ||
                fabs(YEND(nseg)-YBEG(nseg)) > EPS09 ||
                fabs(ZEND(nseg)-ZBEG(nseg)) > EPS09   ) {
                nseg++;
            } else {
                SPRINT7(0, "WARNING:: segment %3d is too short (%12.7f %12.7f %12.7f) to (%12.7f %12.7f %12.7f)\n", nseg,
                        XBEG(nseg), YBEG(nseg), ZBEG(nseg),
                        XEND(nseg), YEND(nseg), ZEND(nseg));
            }
        } else if (dist[ip2] >  DIST(0) && dist[ip0] <= DIST(0) && dist[ip1] <= DIST(0)) {
            frac20 = (DIST(0) - dist[ip2]) / (dist[ip0] - dist[ip2]);
            frac21 = (DIST(0) - dist[ip2]) / (dist[ip1] - dist[ip2]);

            XEND( nseg) = (1-frac20) * xyz[3*ip2  ] + frac20 * xyz[3*ip0  ];
            YEND( nseg) = (1-frac20) * xyz[3*ip2+1] + frac20 * xyz[3*ip0+1];
            ZEND( nseg) = (1-frac20) * xyz[3*ip2+2] + frac20 * xyz[3*ip0+2];
            XBEG( nseg) = (1-frac21) * xyz[3*ip2  ] + frac21 * xyz[3*ip1  ];
            YBEG( nseg) = (1-frac21) * xyz[3*ip2+1] + frac21 * xyz[3*ip1+1];
            ZBEG( nseg) = (1-frac21) * xyz[3*ip2+2] + frac21 * xyz[3*ip1+2];
            IFACE(nseg) =         tricn[4*itri+3];
            IEND( nseg) = -MIN(0, tricn[4*itri+1]);
            IBEG( nseg) = -MIN(0, tricn[4*itri  ]);

            if (fabs(XEND(nseg)-XBEG(nseg)) > EPS09 ||
                fabs(YEND(nseg)-YBEG(nseg)) > EPS09 ||
                fabs(ZEND(nseg)-ZBEG(nseg)) > EPS09   ) {
                nseg++;
            } else {
                SPRINT7(0, "WARNING:: segment %3d is too short (%12.7f %12.7f %12.7f) to (%12.7f %12.7f %12.7f)\n", nseg,
                        XBEG(nseg), YBEG(nseg), ZBEG(nseg),
                        XEND(nseg), YEND(nseg), ZEND(nseg));
            }
        }
    }

    new_time = clock();
    SPRINT1(1, "%8.3f sec - after generation of segments",
            (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC));
    old_time = new_time;

#ifdef DEBUG
    printLoops(nloop, ibeg, iend, seg);
#endif

    nloop =  0;
    mloop = 10;
    MALLOC(ibeg,   int, mloop);
    MALLOC(iend,   int, mloop);
    MALLOC(closed, int, mloop);

    /* sort the segments so that they form a continuous chain */
    ibeg[nloop] = 0;
    for (iseg = 0; iseg < nseg-1; iseg++) {

        /* first, swap iseg with the first segment starting at iseg that
           has an associated Edge (if any) */
        if (iseg == ibeg[nloop]) {
            for (jseg = iseg; jseg < nseg; jseg++) {
                if (IBEG(jseg) != 0) {
                    RSWAP(XBEG( jseg), XBEG( iseg));
                    RSWAP(YBEG( jseg), YBEG( iseg));
                    RSWAP(ZBEG( jseg), ZBEG( iseg));
                    RSWAP(XEND( jseg), XEND( iseg));
                    RSWAP(YEND( jseg), YEND( iseg));
                    RSWAP(ZEND( jseg), ZEND( iseg));
                    RSWAP(IFACE(jseg), IFACE(iseg));
                    RSWAP(IBEG( jseg), IBEG( iseg));
                    RSWAP(IEND( jseg), IEND( iseg));

                    break;
                }
            }
        }

        okay = 0;
        for (jseg = iseg+1; jseg < nseg; jseg++) {
            /* end(iseg) == beg(jseg) --- swap iseg+1 with jseg */
            if        (fabs(XEND(iseg)-XBEG(jseg)) < EPS06 &&
                       fabs(YEND(iseg)-YBEG(jseg)) < EPS06 &&
                       fabs(ZEND(iseg)-ZBEG(jseg)) < EPS06   ) {
                if (iseg+1 < jseg) {
                    RSWAP(XBEG( iseg+1), XBEG( jseg));
                    RSWAP(YBEG( iseg+1), YBEG( jseg));
                    RSWAP(ZBEG( iseg+1), ZBEG( jseg));
                    RSWAP(XEND( iseg+1), XEND( jseg));
                    RSWAP(YEND( iseg+1), YEND( jseg));
                    RSWAP(ZEND( iseg+1), ZEND( jseg));
                    RSWAP(IFACE(iseg+1), IFACE(jseg));
                    RSWAP(IBEG( iseg+1), IBEG( jseg));
                    RSWAP(IEND( iseg+1), IEND( jseg));
                }
                okay = 1;
                break;
            }
        }

        /* if we cannot find a match for end of iseg (either because is forms a
           complete loop or because we have found the end of an open loop.  so complete
           the loop and start another */
        if (okay == 0) {
            if (iseg > ibeg[nloop]) {
                iend[nloop] = iseg;
                nloop++;
            } else if (fabs(XBEG(iseg)-XEND(iseg)) > EPS06 ||
                       fabs(YBEG(iseg)-YEND(iseg)) > EPS06 ||
                       fabs(ZBEG(iseg)-ZEND(iseg)) > EPS06   ) {
                iend[nloop] = iseg;
                nloop++;
            }

            if (nloop >= mloop-2) {
                mloop += 10;
                RALLOC(ibeg,   int, mloop);
                RALLOC(iend,   int, mloop);
                RALLOC(closed, int, mloop);
            }

            if (iseg < nseg-1) {
                ibeg[nloop] = iseg + 1;
            }
        }
    }

    if (nseg-1 > ibeg[nloop]) {
        iend[nloop] = nseg-1;
        nloop++;
    } else if (fabs(XBEG(iseg-1)-XEND(nseg-1)) > EPS06 ||
               fabs(YBEG(iseg-1)-YEND(nseg-1)) > EPS06 ||
               fabs(ZBEG(iseg-1)-ZEND(nseg-1)) > EPS06   ) {
        iend[nloop] = iseg;
        nloop++;
    }

    /* if there is only one segment in the last loop and the
       length of that segment is small, remove it */
    while (nloop > 0) {
        if (ibeg[nloop-1] != iend[nloop-1]           ||
            fabs(XEND(nseg-1)-XBEG(nseg-1)) >= EPS06 ||
            fabs(YEND(nseg-1)-YBEG(nseg-1)) >= EPS06 ||
            fabs(ZEND(nseg-1)-ZBEG(nseg-1)) >= EPS06   ) {
            break;
        }

        nseg--;
        nloop--;
    }

    /* if we find consecutive segments associated with different Faces,
       fix the Edge markings. this can occur when a very short
       segment is rejected (above) */

    jseg = 0;      // beginning of current segment

    for (iseg = 0; iseg < nseg-1; iseg++) {

        /* xyz_beg(iseg+1) == xyz_end(iseg) */
        if (fabs(XEND(iseg)-XBEG(iseg+1)) < EPS06 &&
            fabs(YEND(iseg)-YBEG(iseg+1)) < EPS06 &&
            fabs(ZEND(iseg)-ZBEG(iseg+1)) < EPS06   ) {

            /* if Faces match there should not be any Edge pointers */
            if (IFACE(iseg) == IFACE(iseg+1)) {
                if (IEND(iseg) != 0 || IBEG(iseg+1) != 0) {
                    snprintf(message, 1024, "error 1 at iseg=%d", iseg);
                    status = OCSM_UDP_ERROR1;
                    goto cleanup;
                }

            /* if the Faces match... */
            } else {

                /* ...both Edge pointers are 0, so find a suitable Edge */
                if (IEND(iseg) == 0 && IBEG(iseg+1) == 0) {
                    ibest = 0;
                    dbest = HUGEQ;
                    for (iedge = 0; iedge < nedge; iedge++) {
                        status = EG_invEvaluate(eedges[iedge], &XBEG(iseg), uvbest, xyzbest);
                        CHECK_STATUS(EG_invEvaluate);

                        dtest = SQR(XBEG(iseg)-xyzbest[0])
                            +   SQR(YBEG(iseg)-xyzbest[1])
                            +   SQR(ZBEG(iseg)-xyzbest[2]);

                        if (dtest < dbest) {
                            ibest = iedge + 1;
                            dbest = dtest;
                        }
                    }

                    if (ibest > 0) {
                        IEND(iseg  ) = ibest;
                        IBEG(iseg+1) = ibest;
                    } else {
                        snprintf(message, 1024, "error 6 at iseg=%d", iseg);
                        status = OCSM_UDP_ERROR1;
                        goto cleanup;
                    }

                /* ...if iend(iseg) == 0 but ibeg(iseg+1) != 0, fix iend(iseg) */
                } else if (IEND(iseg) == 0) {
                    IEND(iseg) = IBEG(iseg+1);

                /* ...if (ibeg(iseg+1) == 0 but iend(iseg) != 0, fix ibeg(iseg+1) */
                } else if (IBEG(iseg+1) == 0) {
                    IBEG(iseg+1) = IEND(iseg);

                /* ...iend(iseg) and ibeg(iseg+1) do not match */
                } else if (IEND(iseg) != IBEG(iseg+1)) {
                    snprintf(message, 1024, "error 2 at iseg=%d", iseg);
                    status = OCSM_UDP_ERROR1;
                    goto cleanup;
                }
            }

        /* xyz_end(iseg) matches xyz_beg(jseg), so we have an isolated loop */
        } else if (fabs(XEND(iseg)-XBEG(jseg)) < EPS06 &&
                   fabs(YEND(iseg)-YBEG(jseg)) < EPS06 &&
                   fabs(ZEND(iseg)-ZBEG(jseg)) < EPS06   ) {

            /*  if the Faces match, then iend(iseg) and ibeg(jseg) should both be 0 */
            if (IFACE(iseg) == IFACE(jseg)) {
                if (IEND(iseg) != 0 || IBEG(jseg) != 0) {
                    snprintf(message, 1024, "error 4 at iseg=%d, jseg=%d", iseg, jseg);
                    status = OCSM_UDP_ERROR1;
                    goto cleanup;
                }
            }

            jseg = iseg + 1;

        /* there is a jump in coordinates, so start a new loop
           on the current Face */
        } else {

            /* both iend(iseg) and ibeg(iseg+1) should be 0 */
            if (IEND(iseg) == 0 || IBEG(iseg+1) == 0) {
                snprintf(message, 1024, "error 5 at iseg=%d", iseg);
                status = OCSM_UDP_ERROR1;
                goto cleanup;
            }

            jseg = iseg + 1;
        }
    }

    /* print the loops */
#ifdef DEBUG
    printLoops(nloop, ibeg, iend, seg);
#endif

#ifdef MAKE_PLOTFILES
    fp = fopen("offset.plot", "a");
    if (fp != NULL) {
        for (iloop = 0; iloop < nloop; iloop++) {
            fprintf(fp, "%5d %5d orig_%d|r\n", iend[iloop]-ibeg[iloop]+2, 1, *NumUdp);
            for (jseg = ibeg[iloop]; jseg <= iend[iloop]; jseg++) {
                fprintf(fp, "%10.5f %10.5f %10.5f\n", XBEG(jseg), YBEG(jseg), ZBEG(jseg));
            }
            fprintf(fp, "%10.5f %10.5f %10.5f\n", XEND(iend[iloop]), YEND(iend[iloop]), ZEND(iend[iloop]));
        }
        fclose(fp);
        fp = NULL;
    }
#endif

    /* smooth the segments */
    MALLOC(xyzsmth, double, 3*nseg);

    for (ismth = 0; ismth < nsmth; ismth++) {
        for (iloop = 0; iloop < nloop; iloop++) {
            if (fabs(XBEG(ibeg[iloop])-XEND(iend[iloop])) < EPS06 &&
                fabs(YBEG(ibeg[iloop])-YEND(iend[iloop])) < EPS06 &&
                fabs(ZBEG(ibeg[iloop])-ZEND(iend[iloop])) < EPS06   ) {
                closed[iloop] = 1;
            } else {
                closed[iloop] = 0;
            }

            for (jseg = ibeg[iloop]; jseg <= iend[iloop]; jseg++) {
                if (jseg == ibeg[iloop]) {
                    jm1 = iend[iloop];
                } else {
                    jm1 = jseg - 1;
                }

                /* do not smooth points on an Edge */
                if (IBEG(jseg) != 0) continue;

                /* do not smooth the ends of an open loop */
                if (closed[iloop] == 0 && (jseg == ibeg[iloop] || jseg == iend[iloop])) continue;

                /* compute the laplace smoothing changes */
                xyzsmth[3*jseg  ] = (XBEG(jm1) + 2 * XBEG(jseg) + XEND(jseg)) / 4;
                xyzsmth[3*jseg+1] = (YBEG(jm1) + 2 * YBEG(jseg) + YEND(jseg)) / 4;
                xyzsmth[3*jseg+2] = (ZBEG(jm1) + 2 * ZBEG(jseg) + ZEND(jseg)) / 4;
            }

            /* apply the changes */
            for (jseg = ibeg[iloop]; jseg <= iend[iloop]; jseg++) {
                if (jseg == ibeg[iloop]) {
                    jm1 = iend[iloop];
                } else {
                    jm1 = jseg - 1;
                }

                if (IBEG(jseg) != 0) continue;
                if (closed[iloop] == 0 && (jseg == ibeg[iloop] || jseg == iend[iloop])) continue;

                XEND(jm1) = xyzsmth[3*jseg  ];
                YEND(jm1) = xyzsmth[3*jseg+1];
                ZEND(jm1) = xyzsmth[3*jseg+2];

                XBEG(jseg) = xyzsmth[3*jseg  ];
                YBEG(jseg) = xyzsmth[3*jseg+1];
                ZBEG(jseg) = xyzsmth[3*jseg+2];
            }
        }
    }

    FREE(xyzsmth);

#ifdef MAKE_PLOTFILES
    fp = fopen("offset.plot", "a");
    if (fp != NULL) {
        for (iloop = 0; iloop < nloop; iloop++) {
            fprintf(fp, "%5d %5d smoothed_%d|b\n", iend[iloop]-ibeg[iloop]+2, 1, *NumUdp);
            for (jseg = ibeg[iloop]; jseg <= iend[iloop]; jseg++) {
                fprintf(fp, "%10.5f %10.5f %10.5f\n", XBEG(jseg), YBEG(jseg), ZBEG(jseg));
            }
            fprintf(fp, "%10.5f %10.5f %10.5f\n", XEND(iend[iloop]), YEND(iend[iloop]), ZEND(iend[iloop]));
        }
        fclose(fp);
        fp = NULL;
    }
#endif

    new_time = clock();
    SPRINT1(1, "%8.3f sec - after smoothing segments",
            (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC));
    old_time = new_time;

#ifdef DEBUG
    printLoops(nloop, ibeg, iend, seg);
#endif

    /* project the points to the Edge or Face */
    for (iloop = 0; iloop < nloop; iloop++) {
        esurf = NULL;

        for (jseg = ibeg[iloop]; jseg <= iend[iloop]; jseg++) {
            if (jseg == ibeg[iloop]) {
                jm1 = iend[iloop];
            } else {
                jm1 = jseg - 1;
            }

            if (IBEG(jseg) != 0) {
                iedge = NINT(IBEG(jseg));
                status = EG_invEvaluate(eedges[iedge-1], &XBEG(jseg), uvbest, xyzbest);
                CHECK_STATUS(EG_invEvaluate);

                esurf = NULL;
            } else if (esurf == NULL) {
                iface = NINT(IFACE(jseg));
                status = EG_invEvaluate(efaces[iface-1], &XBEG(jseg), uvbest, xyzbest);
                CHECK_STATUS(EG_invEvaluate);

                status = EG_getTopology(efaces[iface-1], &esurf, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);
            } else {
                status = EG_invEvaluateGuess(esurf, &XBEG(jseg), uvbest, xyzbest);
                CHECK_STATUS(EG_invEvaluateGuess);
            }

            test = sqrt(SQR(XBEG(jseg)-xyzbest[0])
                       +SQR(YBEG(jseg)-xyzbest[1])
                       +SQR(ZBEG(jseg)-xyzbest[2]));
            if (test > DIST(0)) {
                SPRINT2(0, "WARNING:: invEval fail, jseg=%d, test=%12.5e", jseg, test);
            }

            XBEG(jseg) = xyzbest[0];
            YBEG(jseg) = xyzbest[1];
            ZBEG(jseg) = xyzbest[2];

            if (closed[iloop] == 0 && IEND(jseg) != 0) {
                iedge = NINT(IEND(jseg));
                status = EG_invEvaluate(eedges[iedge-1], &XEND(jseg), uvbest, xyzbest);
                CHECK_STATUS(EG_invEvaluate);

                test = sqrt(SQR(XBEG(jseg)-xyzbest[0])
                           +SQR(YBEG(jseg)-xyzbest[1])
                           +SQR(ZBEG(jseg)-xyzbest[2]));
                if (test > DIST(0)) {
                SPRINT2(0, "WARNING:: invEval fail, jseg=%d, test=%12.5e", jseg, test);
                }

                XEND(jseg) = xyzbest[0];
                YEND(jseg) = xyzbest[1];
                ZEND(jseg) = xyzbest[2];

                esurf = NULL;
            } else if (closed[iloop] == 1 || jseg > ibeg[iloop]) {
                XEND(jm1) = xyzbest[0];
                YEND(jm1) = xyzbest[1];
                ZEND(jm1) = xyzbest[2];
            }
        }
    }

    new_time = clock();
    SPRINT1(1, "%8.3f sec - after projecting to BRep",
            (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC));
    old_time = new_time;

#ifdef DEBUG
    printLoops(nloop, ibeg, iend, seg);
#endif

#ifdef MAKE_PLOTFILES
    fp = fopen("offset.plot", "a");
    if (fp != NULL) {
        for (iloop = 0; iloop < nloop; iloop++) {
            fprintf(fp, "%5d %5d projected_%d|c\n", iend[iloop]-ibeg[iloop]+2, 1, *NumUdp);
            for (jseg = ibeg[iloop]; jseg <= iend[iloop]; jseg++) {
                fprintf(fp, "%10.5f %10.5f %10.5f\n", XBEG(jseg), YBEG(jseg), ZBEG(jseg));
            }
            fprintf(fp, "%10.5f %10.5f %10.5f\n", XEND(iend[iloop]), YEND(iend[iloop]), ZEND(iend[iloop]));
        }
        fclose(fp);
        fp = NULL;
    }
#endif

    /* remove any loops that are very short */
    for (iloop = 0; iloop < nloop; iloop++) {
        alen = 0;
        for (iseg = ibeg[iloop]; iseg <= iend[iloop]; iseg++) {
            alen += sqrt( (XEND(iseg)-XBEG(iseg)) * (XEND(iseg)-XBEG(iseg))
                        + (YEND(iseg)-YBEG(iseg)) * (YEND(iseg)-YBEG(iseg))
                        + (ZEND(iseg)-ZBEG(iseg)) * (ZEND(iseg)-ZBEG(iseg)));
        }

        if (alen < EPS03) {
            iseg = ibeg[iloop];
            for (jseg = ibeg[iloop+1]; jseg <= iend[nloop-1]; jseg++) {
                XBEG( iseg) = XBEG( jseg);
                YBEG( iseg) = YBEG( jseg);
                ZBEG( iseg) = ZBEG( jseg);
                XEND( iseg) = XEND( jseg);
                YEND( iseg) = YEND( jseg);
                ZEND( iseg) = ZEND( jseg);
                IFACE(iseg) = IFACE(jseg);
                IBEG( iseg) = IBEG( jseg);
                IEND( iseg) = IEND( jseg);

                iseg++;
            }

            for (jloop = iloop+1; jloop < nloop; jloop++) {
                ibeg[jloop-1] = ibeg[jloop];
                iend[jloop-1] = iend[jloop];
            }

            nloop--;
            iloop--;
        }
    }

    nseg = iend[nloop-1] + 1;

    new_time = clock();
    SPRINT1(1, "%8.3f sec - after removing short loops",
            (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC));
    old_time = new_time;

#ifdef DEBUG
    printLoops(nloop, ibeg, iend, seg);
#endif

    /* create the inputs for EG_scribeFaces */
    nscribe = 0;
    for (iseg = 0; iseg < nseg; iseg++) {
        if (IBEG(iseg) != 0) {
            nscribe++;
        } else if (iseg < nseg-1 &&
                   (fabs(XEND(iseg)-XBEG(iseg+1)) > EPS06 ||
                    fabs(YEND(iseg)-YBEG(iseg+1)) > EPS06 ||
                    fabs(ZEND(iseg)-ZBEG(iseg+1)) > EPS06   )) {
            nscribe++;
        }
    }

    /* if there are no Edges associated with the segments,
       we have only one scribe */
    if (nscribe == 0) {
        nscribe = 1;
    }

    /* allocate arrays for EG_scribeFaces */
    MALLOC(points, double, 3*(nseg+2*nscribe) );
    MALLOC(facnum, int,              nscribe+1);
    MALLOC(begend, int,            2*nscribe+2);
    MALLOC(nper,   int,              nscribe+1);

    npoint     = 0;
    npoint_beg = 0;
    nremove    = 0;

    /* extract the scribes from the segments */
    nscribe   = 0;
    facnum[0] = NINT(IFACE(nscribe));
    begend[0] = NINT(IBEG( nscribe));

    for (iseg = 0; iseg < nseg; iseg++) {

        /* add iseg to points */
        points[3*npoint  ] = XBEG(iseg);
        points[3*npoint+1] = YBEG(iseg);
        points[3*npoint+2] = ZBEG(iseg);
        npoint++;

        /* if this is the first point in this scribe, add the
           next point too */
        if (npoint == npoint_beg+1) {
            iseg++;

            points[3*npoint  ] = XBEG(iseg);
            points[3*npoint+1] = YBEG(iseg);
            points[3*npoint+2] = ZBEG(iseg);
            npoint++;

            count = 0;

        /* if it is not the first point, check to see if we can
           remove the point before the one just added */
        } else {

            /* note that "a" represents the segment between npoint-3 and
               npoint-2 and that "b" represents the segment between
               npoint-2 and npoint-1 */
            xa = points[3*npoint-6] - points[3*npoint-9];
            ya = points[3*npoint-5] - points[3*npoint-8];
            za = points[3*npoint-4] - points[3*npoint-7];
            da = sqrt(xa * xa + ya * ya + za * za);

            xb = points[3*npoint-3] - points[3*npoint-6];
            yb = points[3*npoint-2] - points[3*npoint-5];
            zb = points[3*npoint-1] - points[3*npoint-4];
            db = sqrt(xb * xb + yb * yb + zb * zb);

            /* if either of the intervals is very small, remove npoint-2
               by copying npoint-1 over it */
            if (da < EPS06 || db < EPS06) {
                points[3*npoint-6] = points[3*npoint-3];
                points[3*npoint-5] = points[3*npoint-2];
                points[3*npoint-4] = points[3*npoint-1];
                npoint--;
                count++;

            /* if we have already removed 5 consecutive segments, do not
               remove this one because it could result in a flat spot */
            } else if (count > 5) {
                count = 0;

            /* if "a" and "b" are nearly colinear (dot product is close
               to 1.0, remove npoint-1 by copying npoint-1 over it */
            } else {
                dot = (xa * xb + ya * yb + za * zb) / da / db;

                if (dot > 0.9975) {     // 4 degrees
                    points[3*npoint-6] = points[3*npoint-3];
                    points[3*npoint-5] = points[3*npoint-2];
                    points[3*npoint-4] = points[3*npoint-1];
                    npoint--;
                    count++;
                } else {
                    count = 0;
                }
            }
        }

        /* see if we have found a segment with an Edge at its end, or is
           the last segment, or if the next segment does not match
           this segment (which happens when we have two or more
           closed loops) */
        if (IEND(iseg) != 0 || iseg == nseg-1 ||
            fabs(XEND(iseg)-XBEG(iseg+1)) > EPS06 ||
            fabs(YEND(iseg)-YBEG(iseg+1)) > EPS06 ||
            fabs(ZEND(iseg)-ZBEG(iseg+1)) > EPS06   ) {

            /* add the point at the end */
            points[3*npoint  ] = XEND(iseg);
            points[3*npoint+1] = YEND(iseg);
            points[3*npoint+2] = ZEND(iseg);
            npoint++;

            /* if the distance between this point and the previous is
               too small, remove the previous one */
            xb = points[3*npoint-3] - points[3*npoint-6];
            yb = points[3*npoint-2] - points[3*npoint-5];
            zb = points[3*npoint-1] - points[3*npoint-4];
            db = sqrt(xb * xb + yb * yb + zb * zb);

            if (db < EPS06) {
                points[3*npoint-6] = points[3*npoint-3];
                points[3*npoint-5] = points[3*npoint-2];
                points[3*npoint-4] = points[3*npoint-1];
                npoint--;
            }

            /* finish up this scribe */
            begend[2*nscribe+1] = NINT(IEND(iseg));
            nper[    nscribe  ] = npoint - npoint_beg;

            /* if this scribe begins and ends on the same Edge and
               if the intermediate points are all very close to
               that Edge, then we should remove this scribe */
            if (begend[2*nscribe] == begend[2*nscribe+1] && begend[2*nscribe] != 0) {
                okay = 0;

                for (ipnt = npoint_beg; ipnt < npoint; ipnt++) {
                    status = EG_invEvaluate(eedges[begend[2*nscribe]-1], &points[3*ipnt], uvbest, xyzbest);
                    CHECK_STATUS(EG_invEvaluate);

                    dtest = sqrt(SQR(points[3*ipnt  ]-xyzbest[0])
                                +SQR(points[3*ipnt+1]-xyzbest[1])
                                +SQR(points[3*ipnt+2]-xyzbest[2]));

                    if (dtest > EPS03) {
                        okay = 1;
                        break;
                    }
                }

                /* remove the scribe if all points are withig dbest of the Edge */
                if (okay == 0) {
                    npoint = npoint_beg;
                    nscribe--;
                    nremove++;
                }
            }

            nscribe++;

            if (iseg == nseg-1) break;

            /* start the next scribe */
            facnum[  nscribe] = NINT(IFACE(iseg+1));
            begend[2*nscribe] = NINT(IBEG( iseg+1));
            npoint_beg        = npoint;
        }
    }

    /* make sure npoint was big enough */
    if (npoint >= nseg+nscribe+2) {
        SPRINT2(1, "WARNING:: npoint=%d but only %d were allocated", npoint, nseg+nscribe+2);
        status = -999;
        goto cleanup;
    }

    new_time = clock();
    SPRINT1(1, "%8.3f sec - after pruning points",
            (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC));
    old_time = new_time;

#ifdef DEBUG
    printPoints(nscribe, facnum, begend, nper, points);
#endif

#ifdef MAKE_PLOTFILES
    fp = fopen("offset.plot", "a");
    if (fp != NULL) {
        for (iloop = 0; iloop < nloop; iloop++) {
            fprintf(fp, "%5d %5d pruned_%d|m\n", iend[iloop]-ibeg[iloop]+2, 1, *NumUdp);
            for (jseg = ibeg[iloop]; jseg <= iend[iloop]; jseg++) {
                fprintf(fp, "%10.5f %10.5f %10.5f\n", XBEG(jseg), YBEG(jseg), ZBEG(jseg));
            }
            fprintf(fp, "%10.5f %10.5f %10.5f\n", XEND(iend[iloop]), YEND(iend[iloop]), ZEND(iend[iloop]));
        }
        fclose(fp);
        fp = NULL;
    }
#endif

    /* put a temporary attribute on all Faces in facelist and all
       Edges in edgelist */
    for (i = 0; i < FACELIST_SIZ(0); i++) {
        iface = FACELIST(0,i);

        status = EG_attributeAdd(efaces[iface-1], "__tempOffset__",
                                 ATTRINT, 1, &iface, NULL, NULL);
        CHECK_STATUS(EG_attributeAdd);
    }

    for (i = 0; i < EDGELIST_SIZ(0); i++) {
        iedge = EDGELIST(0,i);

        status = EG_attributeAdd(eedges[iedge-1], "__tempOffset__",
                                 ATTRINT, 1, &iedge, NULL, NULL);
        CHECK_STATUS(EG_attributeAdd);
    }

    /* scribe the Faces.  note that 222 is here so that my_scribeFace gets
       compiled in without warnings; using my_scribeFaces creates the
       offset.plot plotfile and a .egads and .txt file for debugging */
    if (METHOD(0)!= 222) {
        if (STRLEN(ATTRNAME(0)) > 0) {
            status = EG_scribeFaces(tempBody, ATTRNAME(0), nscribe, facnum, begend, nper, points, ebodyOut);
            CHECK_STATUS(EG_scribeFaces);
        } else {
            status = EG_scribeFaces(tempBody, NULL,        nscribe, facnum, begend, nper, points, ebodyOut);
            CHECK_STATUS(EG_scribeFaces);
        }
    } else {
        if (STRLEN(ATTRNAME(0)) > 0) {
            status = my_scribeFaces(tempBody, ATTRNAME(0), nscribe, facnum, begend, nper, points, ebodyOut, NumUdp);
            CHECK_STATUS(my_scribeFaces);
        } else {
            status = my_scribeFaces(tempBody, NULL,        nscribe, facnum, begend, nper, points, ebodyOut, NumUdp);
            CHECK_STATUS(my_scribeFaces);
        }
    }

    new_time = clock();
    SPRINT1(1, "%8.3f sec - after scribing Faces",
            (double)(new_time - old_time) / (double)(CLOCKS_PER_SEC));

    /* get the Edges and Faces in  the scribed Body */
    if (eedges != NULL) {
        EG_free(eedges);   eedges = NULL;
    }
    if (efaces != NULL) {
        EG_free(efaces);   efaces = NULL;
    }

    status = EG_getBodyTopos(*ebodyOut, NULL, EDGE, &nedge, &eedges);
    CHECK_STATUS(EG_getBodyTopos);

    status = EG_getBodyTopos(*ebodyOut, NULL, FACE, &nface, &efaces);
    CHECK_STATUS(EG_getBodyTopos);

    SPLINT_CHECK_FOR_NULL(eedges);
    SPLINT_CHECK_FOR_NULL(efaces);

    /* if we removed scribes, there is probably a portion of an original Edge
       that should get attributed */
    if (nremove > 0 && STRLEN(ATTRNAME(0)) > 0) {
        nremove = 0;

        status = EG_getBodyTopos(*ebodyOut, NULL, NODE, &nnode, NULL);
        CHECK_STATUS(EG_getBodyTopos);

        MALLOC(enodes,  ego, nnode);
        MALLOC(numedge, int, nnode);

        nnode = 0;

        /* loop through the Edges and make a list of the Nodes associated
           with the it if it has the ATTRNAME attribute */
        for (iedge = 0; iedge < nedge; iedge++) {
            status = EG_attributeRet(eedges[iedge], ATTRNAME(0), &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == SUCCESS) {
                status = EG_getTopology(eedges[iedge], &eref, &oclass, &mtype,
                                        data, &nchild, &echilds, &senses);
                CHECK_STATUS(EG_getTopology);

                for (i = 0; i < nchild; i++) {
                    okay = 0;
                    for (inode = 0; inode < nnode; inode++) {
                        if (echilds[i] == enodes[inode]) {
                            numedge[inode]++;
                            okay = 1;
                            break;
                        }
                    }
                    if (okay == 0) {
                        enodes[nnode] = echilds[i];
                        numedge[ nnode] = 1;
                        nnode++;
                    }
                }
            }
        }

        /* for any Edge that is not attributed, if its two Nodes are in
           the Node list (enodes) and they both have a numedge of 1, attribute it too */
        for (iedge = 0; iedge < nedge; iedge++) {
            status = EG_attributeRet(eedges[iedge], ATTRNAME(0), &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status == SUCCESS) continue;

            status = EG_getTopology(eedges[iedge], &eref, &oclass, &mtype,
                                    data, &nchild, &echilds, &senses);
            CHECK_STATUS(EG_getTopology);

            okay = 0;
            for (inode = 0; inode < nnode; inode++) {
                if        (echilds[0] == enodes[inode] && numedge[inode] == 1) {
                    okay++;
                } else if (echilds[1] == enodes[inode] && numedge[inode] == 1) {
                    okay++;
                }
            }

            if (okay == 2) {
                nremove--;
                status = EG_attributeAdd(eedges[iedge], ATTRNAME(0), ATTRINT, 1,
                                         &nremove, NULL, NULL);
                CHECK_STATUS(EG_attributeAdd);
            }
        }
    }

    /* every Face (in facelist) that is adjacent to an Edge in edgelist
       should be attributed */
    if (STRLEN(ATTRNAME(0)) > 0) {
        for (iedge = 1; iedge <= nedge; iedge++) {
            status = EG_attributeRet(eedges[iedge-1], "__tempOffset__", &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
            if (status != EGADS_SUCCESS) continue;

            status = EG_getBodyTopos(*ebodyOut, eedges[iedge-1], FACE, &nlist, &elist);
            CHECK_STATUS(EG_getBodyTopos);

            SPLINT_CHECK_FOR_NULL(elist);

            for (ilist = 0; ilist < nlist; ilist++) {
                status = EG_attributeRet(elist[ilist], "__tempOffset__", &attrType, &attrLen,
                                     &tempIlist, &tempRlist, &tempClist);
                if (status == EGADS_SUCCESS) {
                    status = EG_attributeAdd(elist[ilist], ATTRNAME(0), ATTRINT, 1,
                                             &ilist, NULL, NULL);
                    CHECK_STATUS(EG_attributeAdd);
                }
            }

            EG_free(elist);   elist = NULL;
        }

        /* propagate ATTRNAME across Edges that are not offset Edges to
           a Face in the facelist that does not have have ATTRNAME already,
           this is done in up to nface passes */
        for (ipass = 0; ipass < nface; ipass++) {
            nchange = 0;

            for (iedge = 1; iedge <= nedge; iedge++) {
                status = EG_attributeRet(eedges[iedge-1], ATTRNAME(0), &attrType, &attrLen,
                                         &tempIlist, &tempRlist, &tempClist);
                if (status == EGADS_SUCCESS) continue;

                status = EG_getBodyTopos(*ebodyOut, eedges[iedge-1], FACE, &nlist, &elist);
                CHECK_STATUS(EG_getBodyTopos);

                SPLINT_CHECK_FOR_NULL(elist);

                status = EG_attributeRet(elist[0], "__tempOffset__", &attrType, &attrLen,
                                         &tempIlist, &tempRlist, &tempClist);
                if (status != EGADS_SUCCESS) {
                    EG_free(elist);   elist = NULL;
                    continue;
                }

                status = EG_attributeRet(elist[1], "__tempOffset__", &attrType, &attrLen,
                                         &tempIlist, &tempRlist, &tempClist);
                if (status != EGADS_SUCCESS) {
                    EG_free(elist);  elist = NULL;
                    continue;
                }

                status0 = EG_attributeRet(elist[0], ATTRNAME(0), &attrType, &attrLen,
                                          &tempIlist, &tempRlist, &tempClist);
                status1 = EG_attributeRet(elist[1], ATTRNAME(0), &attrType, &attrLen,
                                          &tempIlist, &tempRlist, &tempClist);

                if        (status0 != EGADS_SUCCESS && status1 == EGADS_SUCCESS) {
                    status = EG_attributeAdd(elist[0], ATTRNAME(0), ATTRINT, 1,
                                             &iedge, NULL, NULL);
                    nchange++;
                    CHECK_STATUS(EG_attributeAdd);
                } else if (status0 == EGADS_SUCCESS && status1 != EGADS_SUCCESS) {
                    status = EG_attributeAdd(elist[1], ATTRNAME(0), ATTRINT, 1,
                                             &iedge, NULL, NULL);
                    CHECK_STATUS(EG_attributeAdd);
                    nchange++;
                }

                EG_free(elist);   elist = NULL;
            }

            if (nchange == 0) break;
        }
    }

    /* if any Edge has ATTRNAME but its two adjoining Faces do not, remove ATTRNAME
       from the Edge */
    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_attributeRet(eedges[iedge], ATTRNAME(0), &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status != EGADS_SUCCESS) continue;

        status = EG_getBodyTopos(*ebodyOut, eedges[iedge], FACE, &nchild, &echilds);
        CHECK_STATUS(EG_getBodyTopos);

        if (nchild != 2) {
            EG_free(echilds);
            continue;
        }

        SPLINT_CHECK_FOR_NULL(echilds);

        status0 = EG_attributeRet(echilds[0], ATTRNAME(0), &attrType, &attrLen,
                                  &tempIlist, &tempRlist, &tempClist);
        status1 = EG_attributeRet(echilds[1], ATTRNAME(0), &attrType, &attrLen,
                                  &tempIlist, &tempRlist, &tempClist);

        EG_free(echilds);

        if (status0 == EGADS_NOTFOUND && status1 == EGADS_NOTFOUND) {
            status = EG_attributeDel(eedges[iedge], ATTRNAME(0));
            CHECK_STATUS(EG_attributeDel);
        }
    }

    /* remove all the temporary attributes */
    for (iedge = 0; iedge < nedge; iedge++) {
        status = EG_attributeRet(eedges[iedge], "__tempOffset__", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            (void) EG_attributeDel(eedges[iedge], "__tempOffset__");
        }
    }

    for (iface = 0; iface < nface; iface++) {
        status = EG_attributeRet(efaces[iface], "__tempOffset__", &attrType, &attrLen,
                                 &tempIlist, &tempRlist, &tempClist);
        if (status == EGADS_SUCCESS) {
            (void) EG_attributeDel(efaces[iface], "__tempOffset__");
        }
    }

    /* clean up */
    status = EG_deleteObject(etess);
    CHECK_STATUS(EG_deleteObject);

    status = EG_deleteObject(tempBody);
    CHECK_STATUS(EG_deleteObject);

cleanup:

    /* printouts to help with debugging */
    if (status < EGADS_SUCCESS) {
        printf("ERROR:: in makeSolidOffset2, detected %s\n", message);
        printLoops(nloop, ibeg, iend, seg);
    }

    FREE(tris   );
    FREE(tricn  );
    FREE(xyz    );
    FREE(state  );
    FREE(dist   );
    FREE(seg    );
    FREE(ibeg   );
    FREE(iend   );
    FREE(closed );
    FREE(xyzsmth);
    FREE(points );
    FREE(facnum );
    FREE(begend );
    FREE(nper   );
    FREE(numedge);
    FREE(enodes );

    if (eedges != NULL) EG_free(eedges);
    if (efaces != NULL) EG_free(efaces);
    if (elist  != NULL) EG_free(elist );

//$$$#undef XBEG
//$$$#undef YBEG
//$$$#undef ZBEG
//$$$#undef XEND
//$$$#undef YEND
//$$$#undef ZEND
//$$$#undef IFACE
//$$$#undef IBEG
//$$$#undef IEND

    return status;
}


/*
Notes:
* ebody can be a SolidBody or a SheetBody
* aName is an integer attribute name to store the scribe index (bias-1) on the Edges generated
  can be NULL for no Edge attributes
* nscribe must be positive
* the entries in iface can be repeated for more than a single scribe on a Face
* startEnd are the starting and ending Edge/Node for a scribe
  zero for both if closed, (+) for Edge index, (-) for hitting an Node exactly
* there are 3*sum(nper) entries in xyz
* the points in xyz for a given scribe either
  * begin and end on an Edge associated with the Face, or
  * forms a closed loop within the Face
* for a closed scribe
  * xyz contains both the first and last (repeated) points
  * the Edge can be arbitrarily split to make two Edges
* an error should be returned if the scribe curves intersect each other
*/

/*@-nullpass@*/
static int
my_scribeFaces(ego    ebody,      /* (in)  the original Body */
     /*@null@*/char   aName[],    /* (in)  scribe attribute name */
               int    nscribe,    /* (in)  the number of scribes */
               int    iface[],    /* (in)  array of Faces to scribe (bias-1) */
               int    startEnd[], /* (in)  start and ending Edge (bias-1), 2*nscribe in len */
               int    nper[],     /* (in)  array of number of points per scribe */
               double xyz[],      /* (in)  array of coordinates */
               ego    *newBody,   /* (out) newly scribed Body */
               int    *NumUdp)
{
    int    status = EGADS_SUCCESS;

    int    icase, iscribe, ipnt, jpnt=0, kpnt=0;
    char   filename[80];
    FILE   *fp, *fp2;

    ROUTINE(my_scribeFaces);

    /* --------------------------------------------------------------- */

    /* find a case that does not exist */
    for (icase = 0; icase < 1000; icase++) {
        snprintf(filename, 79, "case_%03d.egads", icase);
        fp = fopen(filename, "r");
        if (fp == NULL) {
            break;
        } else {
            fclose(fp);
        }
    }

    /* dump ebody */
    remove(filename);
    status = EG_saveModel(ebody, filename);
    CHECK_STATUS(EG_saveModel);

    /* file with data */
    snprintf(filename, 79, "case_%03d.txt", icase);

    fp = fopen(filename, "w");
    if (fp != NULL) {
        fprintf(fp, "aName=%s\n",   aName);
        for (iscribe = 0; iscribe < nscribe; iscribe++) {
            fprintf(fp, "   iface[   %3d]=%5d\n",     iscribe, iface[iscribe]);
            fprintf(fp, "   startEnd[%3d]=%5d %5d\n", iscribe, startEnd[2*iscribe], startEnd[2*iscribe+1]);
            fprintf(fp, "   nper[    %3d]=%5d\n",     iscribe, nper[iscribe]);

            for (ipnt = 0; ipnt < nper[iscribe]; ipnt++) {
                fprintf(fp, "      %5d %20.12e %20.12e %20.12e\n", jpnt,
                        xyz[3*jpnt], xyz[3*jpnt+1], xyz[3*jpnt+2]);
                jpnt++;
            }

            fp2 = fopen("offset.plot", "a");
            if (fp2 != NULL) {
                fprintf(fp2, "%5d %5d output_%d|m\n", nper[iscribe], 1, *NumUdp);

                for (ipnt = 0; ipnt < nper[iscribe]; ipnt++) {
                    fprintf(fp2, "%10.5f %10.5f %10.5f\n", xyz[3*kpnt], xyz[3*kpnt+1], xyz[3*kpnt+2]);
                    kpnt++;
                }
                fclose(fp2);
                fp2 = NULL;
            }

        }

        fclose(fp);
    }

    /* increment global "icase" so that new call will generate different files */
    icase++;

    status = EG_copyObject(ebody, NULL, newBody);
    CHECK_STATUS(EG_copyObject);

cleanup:
    return status;
}
/*@+nullpass@*/


/*
 ************************************************************************
 *                                                                      *
 *   printLoops - debug printing routine                                *
 *                                                                      *
 ************************************************************************
 */

static void
printLoops(int     nloop,               /* (in)  number of loops */
           int     ibeg[],              /* (in)  first segment assoc. with each loop */
           int     iend[],              /* (in)  last  segment assoc. with each loop */
           double  seg[])               /* (in)  segments */
{
    int    iloop, iseg;
    double alen;

    /* --------------------------------------------------------------- */

    for (iloop = 0; iloop < nloop; iloop++) {
        printf("iloop %5d\n", iloop);

        alen = 0;
        for (iseg = ibeg[iloop]; iseg <= iend[iloop]; iseg++) {
            printf("iseg=%5d, beg=%10.5f %10.5f %10.5f, end=%10.5f %10.5f %10.5f, face=%3d, edge=%3d %3d\n", iseg,
                   XBEG(iseg), YBEG(iseg), ZBEG(iseg),
                   XEND(iseg), YEND(iseg), ZEND(iseg),
                   NINT(IFACE(iseg)), NINT(IBEG(iseg)), NINT(IEND(iseg)));

            alen += sqrt( (XEND(iseg)-XBEG(iseg)) * (XEND(iseg)-XBEG(iseg))
                        + (YEND(iseg)-YBEG(iseg)) * (YEND(iseg)-YBEG(iseg))
                        + (ZEND(iseg)-ZBEG(iseg)) * (ZEND(iseg)-ZBEG(iseg)));
        }
        if (fabs(XBEG(ibeg[iloop])-XEND(iend[iloop])) < EPS06 &&
            fabs(YBEG(ibeg[iloop])-YEND(iend[iloop])) < EPS06 &&
            fabs(ZBEG(ibeg[iloop])-ZEND(iend[iloop])) < EPS06   ) {
            printf("loop is closed, alen=%12.6f\n", alen);
        } else {
            printf("loop is open,   alen=%12.6f\n", alen);
        }
    }

    return;
}


/*
 ************************************************************************
 *                                                                      *
 *   printPoints - debug printing routine                               *
 *                                                                      *
 ************************************************************************
 */

#ifdef DEBUG
static void
printPoints(int     nscribe,            /* (in)  number of scribes */
            int     facnum[],           /* (in)  Face assoc. with each scribe */
            int     begend[],           /* (in)  Node at beg and end of each scribe */
            int     nper[],             /* (in)  number of points in each scribe */
            double  points[])           /* (in)  XYZ points */
{

    int     iscribe, ipoint, jpoint;
    double  alen;

    ipoint = 0;
    for (iscribe = 0; iscribe < nscribe; iscribe++) {
        printf("iscribe=%3d, facnum=%5d, begend=%5d %5d, nper=%5d\n",
               iscribe, facnum[iscribe], begend[2*iscribe], begend[2*iscribe+1], nper[iscribe]);

        alen = 0;
        for (jpoint = 0; jpoint < nper[iscribe]; jpoint++) {
            printf("     jpoint=%5d, ipoint=%5d, point=%12.6f %12.6f %12.6f\n",
                   jpoint, ipoint, points[3*ipoint], points[3*ipoint+1], points[3*ipoint+2]);

            if (jpoint > 0) {
                alen += sqrt( (points[3*ipoint  ]-points[3*ipoint-3])
                             *(points[3*ipoint  ]-points[3*ipoint-3])
                            + (points[3*ipoint+1]-points[3*ipoint-2])
                             *(points[3*ipoint+1]-points[3*ipoint-2])
                            + (points[3*ipoint+2]-points[3*ipoint-1])
                             *(points[3*ipoint+2]-points[3*ipoint-1]));
            }

            ipoint++;
        }
        printf("          alen=%12.6f\n", alen);
    }

    return;
}
#endif

#undef XBEG
#undef YBEG
#undef ZBEG
#undef XEND
#undef YEND
#undef ZEND
#undef IFACE
#undef IBEG
#undef IEND
